;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; Sample source code for HardPressed
; By Andy McFadden
; Updated: 28-Oct-93
;
; This file is in the public domain.
;
;
; This file contains two routines:
;
;   InactHP
;   ReactHP
;
; InactHP changes HardPressed to "inactive" by sending it an IPC message.
; The previous state is stored in "old_state".
;
; ReactHP reactivates HardPressed, changing it to whatever state it was
; previously in (if it was previously inactive, it won't really be
; "reactivated", but so what).
;
; Both routines should be called with a JSR.  Calling InactHP twice without
; an intervening ReactHP has no effect.  Similarly, calling ReactHP without
; an initial InactHP has no effect.
;
;
; To use these in your program, you should call InactHP before you do an
; operation that requires HP to be inactive.  When that operation completes,
; you should call ReactHP.  These routines have no nasty side effects
; should HardPressed not be loaded, so there's no reason not to use them
; unless speed is critical.
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;           AGO   ._TESTING
main       START
           BRK   0                        ;so we can watch it work
           jsr   InactHP                  ;make it inactive
           jsr   InactHP                  ;this does nothing
           jsr   ReactHP                  ;reactivate HP
           jsr   ReactHP                  ;this does nothing
           rtl
           END
._TESTING


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
HPData     PRIVDATA
;
; Equivalents and state variables for working with HardPressed.
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;
; Storage for IPC stuff.
;
data_in    anop
           ds    2                        ;room for status word
data_out   anop
           ds    2                        ;recipient count (set by system)
           ds    4                        ;room for version # or status info
           ds    16                       ;(future expansion...)

idstring   anop
           dc    I1'idend-idbeg'
idbeg      dc    C'WestCode~HardPressed~Main~'
idend      anop

; messages:
dMping      equ  $8000                    ;return HP version number (4 bytes)
dMsetStatus equ  $8001                    ;set global status (on/off/decode)
dMgetStatus equ  $8002                    ;get global status
; low byte of status word (choose exactly one):
dVpolDecode equ  $0001                    ;Expand only
dVpolOn    equ   $0002                    ;Compress and Expand
dVpolOff   equ   $0003                    ;Inactive

inact_state dc   I2'0'                    ;0 if we want an Inact, 1 for React
prev_state dc    I2'0'                    ;HP state before the "inact"

           END


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
InactHP    START
;
; Calling conventions:
;   JSR
;   Returns with carry and acc set on error
;
; Tells HP to become inactive.  Current bank and DP are irrelevant.
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
           using HPData

           phb
           phk
           plb

           lda   inact_state              ;did we already do this?
           bne   Done                     ;yup, wait for React

; retrieve status word
           pea   dMgetStatus              ;get current HP status
           pea   $8001                    ;send to one, by name
           pea   idstring|-16
           pea   idstring
           pea   $0000                    ;no data_in
           pea   $0000
           pea   data_out|-16
           pea   data_out
           ldx   #$1c01
           jsl   $e10000                  ;_SendRequest
           bcs   Fail
           lda   data_out                 ;did anybody respond?
           beq   Fail                     ;no, HP must not be loaded

           lda   data_out+$02
           sta   prev_state
           and   #$ff00
           ora   #dVpolOff

           pea   dMsetStatus              ;set current HP status
           pea   $8001                    ;send to one, by name
           pea   idstring|-16
           pea   idstring
           pea   $0000                    ;hi word must be zero
           pha                            ;lo word is new status
           pea   data_out|-16
           pea   data_out
           ldx   #$1c01
           jsl   $e10000                  ;_SendRequest
           bcs   Fail

           inc   inact_state              ;indicate we've got one

Done       ANOP
           plb
           clc
           rts
Fail       ANOP
           plb
           sec
           rts
           END

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
ReactHP    START
;
; Calling conventions:
;   JSR
;   Returns with carry and acc set on error
;
; Reactivates HP after an InactHP call.  Current bank and DP are irrelevant.
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
           using HPData

           lda   >inact_state             ;did we already do this?
           beq   Done                     ;nope, wait for Inact

           pea   dMsetStatus              ;set current HP status
           pea   $8001                    ;send to one, by name
           pea   idstring|-16
           pea   idstring
           pea   $0000                    ;hi word must be zero
           lda   >prev_state              ;grab previous state
           pha                            ;lo word is new status
           pea   data_out|-16
           pea   data_out
           ldx   #$1c01
           jsl   $e10000                  ;_SendRequest
           bcs   Fail

Done       ANOP                           ;note Acc is always 0 here
           sta   >inact_state
           clc
           rts

Fail       ANOP
           sec
           rts
           END


