/*
 * Carbon Dioxide - Open-Source 65C02 Emulation Core
 * Copyright (C) 2005 Jonathan Bettencourt (jonrelay)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>

/* 65C02 Registers */

unsigned char C02_A; /* accumulator */
unsigned char C02_X; /* X index */
unsigned char C02_Y; /* Y index */
unsigned char C02_S; /* stack pointer */
unsigned char C02_P; /* processor status */
unsigned short C02_PC; /* program counter */

/* Processor Status Flags */

#define C02_P_N 0x80 /* negative */
#define C02_P_V 0x40 /* overflow */
#define C02_P_X 0x20 /* unused */
#define C02_P_B 0x10 /* break */
#define C02_P_D 0x08 /* decimal mode */
#define C02_P_I 0x04 /* interrupt disable */
#define C02_P_Z 0x02 /* zero */
#define C02_P_C 0x01 /* carry */

/* 65C02 Addressing Modes */

#define C02_AM_IMM		0	/* LDA #00 */
#define C02_AM_ABS		1	/* LDA $0000 */
#define C02_AM_ABS_X	2	/* LDY $0000,X */
#define C02_AM_ABS_Y	3	/* LDX $0000,Y */
#define C02_AM__ABS_	4	/* JMP ($0000) */
#define C02_AM__ABS_X_	5	/* JMP ($0000,X) */
#define C02_AM_ZP		6	/* LDA $00 */
#define C02_AM_ZP_X		7	/* LDY $00,X */
#define C02_AM_ZP_Y		8	/* LDX $00,Y */
#define C02_AM__ZP_		9	/* LDA ($00) */
#define C02_AM__ZP_X_	10	/* LDA ($00,X) */
#define C02_AM__ZP__Y	11	/* LDA ($00),Y */
#define C02_AM_REL		12	/* BRA +$00 */


/* Getting and Setting Individual Bits of Processor Status */

#define __set_p(a) (C02_P |= (a))
#define __clear_p(a) (C02_P &= (~(a)))
#define __set_p_iff(c,a) (C02_P = ( (c) ? (C02_P | (a)) : (C02_P & (~(a))) ))
#define __clear_p_iff(c,a) (C02_P = ( (c) ? (C02_P & (~(a))) : (C02_P | (a)) ))
#define __get_p(a) (C02_P & (a))
#define __set_p_for(a) (C02_P = ( (C02_P & 0x7D) | ((a) & 0x80) | ((a)?0:2) ))

/* #defined as macros above
void __set_p(unsigned char mask) { C02_P |= mask; }
void __clear_p(unsigned char mask) { C02_P &= (~mask); }
unsigned char __get_p(unsigned char mask) { return (C02_P & mask); }

void __set_p_for(unsigned char a)
{
	if (a) {
		__clear_p(C02_P_Z);
	} else {
		__set_p(C02_P_Z);
	}
	if (a & 0x80) {
		__set_p(C02_P_N);
	} else {
		__clear_p(C02_P_N);
	}
}
*/

#define __uc2sc(a) (((a) & 0x80)?((a)-256):(a))

/* For BCD Addition and Subtraction */

unsigned int __bcd_add(unsigned int a, unsigned int b)
{
	unsigned int s = a + b;
	if ( ((s & 0x000F) > 0x0009) || (((a & 0x000F)+(b & 0x000F)) > 0x0009) ) { s += 0x0006; }
	if ( ((s & 0x00F0) > 0x0090) || (((a & 0x00F0)+(b & 0x00F0)) > 0x0090) ) { s += 0x0060; }
	if ( ((s & 0x0F00) > 0x0900) || (((a & 0x0F00)+(b & 0x0F00)) > 0x0900) ) { s += 0x0600; }
	if ( ((s & 0xF000) > 0x9000) || (((a & 0xF000)+(b & 0xF000)) > 0x9000) ) { s += 0x6000; }
	return s;
}

unsigned int __bcd_sub(unsigned int a, unsigned int b)
{
	unsigned int s = a - b;
	if ( ((s & 0x000F) > 0x0009) || ((a & 0x000F) < (b & 0x000F)) ) { s -= 0x0006; }
	if ( ((s & 0x00F0) > 0x0090) || ((a & 0x00F0) < (b & 0x00F0)) ) { s -= 0x0060; }
	if ( ((s & 0x0F00) > 0x0900) || ((a & 0x0F00) < (b & 0x0F00)) ) { s -= 0x0600; }
	if ( ((s & 0xF000) > 0x9000) || ((a & 0xF000) < (b & 0xF000)) ) { s -= 0x6000; }
	return s;
}

/* Operation Helper Functions */

void __adc(unsigned char a)
{
	signed int m, sm;
	if (__get_p(C02_P_D)) {
		m = __bcd_add( __bcd_add(C02_A, a), (__get_p(C02_P_C)?1:0) );
		sm = m;
	} else {
		m = C02_A + a + (__get_p(C02_P_C)?1:0);
		sm = __uc2sc(C02_A) + __uc2sc(a) + (__get_p(C02_P_C)?1:0);
	}
	C02_A = (m & 0xFF);
	__set_p_for(C02_A);
	__set_p_iff(((m<0)||(m>0xFF)), C02_P_C);
	__set_p_iff(((sm < -128)||(sm>0x7F)), C02_P_V);
}

unsigned char __asl(unsigned char a)
{
	__set_p_iff((a & 0x80), C02_P_C);
	return ((a << 1) & 0xFF);
}

void __bit(unsigned char a, int affect_v)
{
	__clear_p_iff((C02_A & a), C02_P_Z);
	if (affect_v) {
		C02_P = ((C02_P & 0x3F) | (a & 0xC0));
	} else {
		C02_P = ((C02_P & 0x7F) | (a & 0x80));
	}
}

void __cmp(unsigned char r, unsigned char m)
{
	__set_p_iff((r >= m), C02_P_C);
	__set_p_for((r-m) & 0xFF);
}

unsigned char __lsr(unsigned char a)
{
	__set_p_iff((a & 1), C02_P_C);
	return (a >> 1);
}

unsigned char __rol(unsigned char a)
{
	if (__get_p(C02_P_C)) {
		__set_p_iff((a & 0x80), C02_P_C);
		return (((a << 1) & 0xFF) | 1);
	} else {
		__set_p_iff((a & 0x80), C02_P_C);
		return ((a << 1) & 0xFF);
	}
}

unsigned char __ror(unsigned char a)
{
	if (__get_p(C02_P_C)) {
		__set_p_iff((a & 1), C02_P_C);
		return ((a >> 1) | 0x80);
	} else {
		__set_p_iff((a & 1), C02_P_C);
		return (a >> 1);
	}
}

void __sbc(unsigned char a)
{
	signed int m, mm, sm;
	mm = C02_A - a - (__get_p(C02_P_C)?0:1);
	sm = __uc2sc(C02_A) - __uc2sc(a) - (__get_p(C02_P_C)?0:1);
	if (__get_p(C02_P_D)) {
		m = __bcd_sub( __bcd_sub(C02_A, a), (__get_p(C02_P_C)?0:1) );
	} else {
		m = mm;
	}
	C02_A = (m & 0xFF);
	__set_p_for(C02_A);
	__clear_p_iff(((m<0)||(m>0xFF)), C02_P_C);
	__set_p_iff(((sm < -128)||(sm>0x7F)), C02_P_V);
}

/* 65C02 Stack */

#define c02_push(p,a) ( p( (0x100 | (C02_S--)), (a) ) )
#define c02_pull(p) ( p( (0x100 | (++C02_S)) ) )

/* #defined as macros above
void c02_push(void (*poke)(unsigned short, unsigned char), unsigned char a)
{
	* Push byte onto 65C02 stack *
	poke(0x100 | C02_S, a);
	C02_S--;
}

unsigned char c02_pull(unsigned char (*peek)(unsigned short))
{
	* Pull byte off 65C02 stack *
	C02_S++;
	return peek(0x100 | C02_S);
}
*/

/* Utility Functions for 65C02 */

unsigned short c02_operand_address(unsigned char (*peek)(unsigned short), unsigned short addr, int mode)
{
	int a;
	switch (mode) {
	case C02_AM_IMM		:	/* AND #00 */
		return addr;
		break;
	case C02_AM_ABS		:	/* AND $0000 */
		return ( peek(addr) | (peek(addr+1)<<8) );
		break;
	case C02_AM_ABS_X	:	/* AND $0000,X */
		return ( (peek(addr) | (peek(addr+1)<<8)) + C02_X );
		break;
	case C02_AM_ABS_Y	:	/* AND $0000,Y */
		return ( (peek(addr) | (peek(addr+1)<<8)) + C02_Y );
		break;
	case C02_AM__ABS_	:	/* JMP ($0000) */
		a = ( peek(addr) | (peek(addr+1)<<8) );
		return ( peek(a) | (peek(a+1)<<8) );
		break;
	case C02_AM__ABS_X_	:	/* JMP ($0000,X) */
		a = ( (peek(addr) | (peek(addr+1)<<8)) + C02_X );
		return ( peek(a) | (peek(a+1)<<8) );
		break;
	case C02_AM_ZP		:	/* AND $00 */
		return peek(addr);
		break;
	case C02_AM_ZP_X	:	/* AND $00,X */
		return ((peek(addr) + C02_X) & 0xFF);
		break;
	case C02_AM_ZP_Y	:	/* AND $00,Y */
		return ((peek(addr) + C02_Y) & 0xFF);
		break;
	case C02_AM__ZP_	:	/* AND ($00) */
		a = peek(addr);
		return ( peek(a) | (peek(a+1)<<8) );
		break;
	case C02_AM__ZP_X_	:	/* AND ($00,X) */
		a = ((peek(addr) + C02_X) & 0xFF);
		return ( peek(a) | (peek(a+1)<<8) );
		break;
	case C02_AM__ZP__Y	:	/* AND ($00),Y */
		a = peek(addr);
		return ( (peek(a) | (peek(a+1)<<8)) + C02_Y );
		break;
	case C02_AM_REL		:	/* BRA +$00 */
		a = peek(addr);
		return ( C02_PC + ((a<128)?a:(a-256)) );
		break;
	}
	return 0;
}

#define c02_operand_value(p,a,m) ( p( c02_operand_address(p,a,m) ) )

/* #defined as macro above
unsigned char c02_operand_value(unsigned char (*peek)(unsigned short), unsigned short addr, int mode)
{
	return peek(c02_operand_address(peek, addr, mode));
}
*/

/* Interrupt Generators for Emulator */

/* I'm not sure if all these interrupt generators are accurate. */

void c02_clear(void)
{
	C02_A = 0;
	C02_X = 0;
	C02_Y = 0;
	C02_S = 0xFF;
	C02_P = 0x24;
	C02_PC = 0;
}

void c02_brk(unsigned char (*peek)(unsigned short), void (*poke)(unsigned short, unsigned char))
{
	C02_PC += 2;
	c02_push(poke, (C02_PC & 0xFF00) >> 8);
	c02_push(poke, C02_PC & 0x00FF);
	__set_p(C02_P_B);
	c02_push(poke, C02_P);
	__set_p(C02_P_B);
	__set_p(C02_P_I);
	__clear_p(C02_P_D);
	C02_PC = ( peek(0xFFFE) + 256*peek(0xFFFF) );
}

void c02_irq(unsigned char (*peek)(unsigned short), void (*poke)(unsigned short, unsigned char))
{
	if (__get_p(C02_P_I)) {
		c02_push(poke, (C02_PC & 0xFF00) >> 8);
		c02_push(poke, C02_PC & 0x00FF);
		__clear_p(C02_P_B);
		c02_push(poke, C02_P);
		__set_p(C02_P_B);
		__set_p(C02_P_I);
		__clear_p(C02_P_D);
		C02_PC = ( peek(0xFFFE) + 256*peek(0xFFFF) );
	}
}

void c02_nmi(unsigned char (*peek)(unsigned short), void (*poke)(unsigned short, unsigned char))
{
	c02_push(poke, (C02_PC & 0xFF00) >> 8);
	c02_push(poke, C02_PC & 0x00FF);
	c02_push(poke, C02_P);
	__set_p(C02_P_B);
	__set_p(C02_P_I);
	__clear_p(C02_P_D);
	C02_PC = ( peek(0xFFFA) + 256*peek(0xFFFB) );
}

void c02_reset(unsigned char (*peek)(unsigned short), void (*poke)(unsigned short, unsigned char))
{
	c02_push(poke, (C02_PC & 0xFF00) >> 8);
	c02_push(poke, C02_PC & 0x00FF);
	c02_push(poke, C02_P);
	__set_p(C02_P_B);
	__set_p(C02_P_I);
	__clear_p(C02_P_D);
	C02_PC = ( peek(0xFFFC) + 256*peek(0xFFFD) );
}

void c02_init(unsigned char (*peek)(unsigned short), void (*poke)(unsigned short, unsigned char))
{
	c02_clear();
	C02_PC = ( peek(0xFFFC) + 256*peek(0xFFFD) );
}

/* Disassembler */

const char * c02_disasm_ilut[256] = {
	"BRK", "ORA", "XXX", "XXX", "TSB", "ORA", "ASL", "RMB0", "PHP", "ORA", "ASL", "XXX", "TSB", "ORA", "ASL", "BBR0",
	"BPL", "ORA", "ORA", "XXX", "TRB", "ORA", "ASL", "RMB1", "CLC", "ORA", "INC", "XXX", "TRB", "ORA", "ASL", "BBR1",
	"JSR", "AND", "XXX", "XXX", "BIT", "AND", "ROL", "RMB2", "PLP", "AND", "ROL", "XXX", "BIT", "AND", "ROL", "BBR2",
	"BMI", "AND", "AND", "XXX", "BIT", "AND", "ROL", "RMB3", "SEC", "AND", "DEC", "XXX", "BIT", "AND", "ROL", "BBR3",
	"RTI", "EOR", "XXX", "XXX", "XXX", "EOR", "LSR", "RMB4", "PHA", "EOR", "LSR", "XXX", "JMP", "EOR", "LSR", "BBR4",
	"BVC", "EOR", "EOR", "XXX", "XXX", "EOR", "LSR", "RMB5", "CLI", "EOR", "PHY", "XXX", "XXX", "EOR", "LSR", "BBR5",
	"RTS", "ADC", "XXX", "XXX", "STZ", "ADC", "ROR", "RMB6", "PLA", "ADC", "ROR", "XXX", "JMP", "ADC", "ROR", "BBR6",
	"BVS", "ADC", "ADC", "XXX", "STZ", "ADC", "ROR", "RMB7", "SEI", "ADC", "PLY", "XXX", "JMP", "ADC", "ROR", "BBR7",
	"BRA", "STA", "XXX", "XXX", "STY", "STA", "STX", "SMB0", "DEY", "BIT", "TXA", "XXX", "STY", "STA", "STX", "BBS0",
	"BCC", "STA", "STA", "XXX", "STY", "STA", "STX", "SMB1", "TYA", "STA", "TXS", "XXX", "STZ", "STA", "STZ", "BBS1",
	"LDY", "LDA", "LDX", "XXX", "LDY", "LDA", "LDX", "SMB2", "TAY", "LDA", "TAX", "XXX", "LDY", "LDA", "LDX", "BBS2",
	"BCS", "LDA", "LDA", "XXX", "LDY", "LDA", "LDX", "SMB3", "CLV", "LDA", "TSX", "XXX", "LDY", "LDA", "LDX", "BBS3",
	"CPY", "CMP", "XXX", "XXX", "CPY", "CMP", "DEC", "SMB4", "INY", "CMP", "DEX", "WAI", "CPY", "CMP", "DEC", "BBS4",
	"BNE", "CMP", "CMP", "XXX", "XXX", "CMP", "DEC", "SMB5", "CLD", "CMP", "PHX", "STP", "XXX", "CMP", "DEC", "BBS5",
	"CPX", "SBC", "XXX", "XXX", "CPX", "SBC", "INC", "SMB6", "INX", "SBC", "NOP", "XXX", "CPX", "SBC", "INC", "BBS6",
	"BEQ", "SBC", "SBC", "XXX", "XXX", "SBC", "INC", "SMB7", "SED", "SBC", "PLX", "XXX", "XXX", "SBC", "INC", "BBS7"
};

const int c02_disasm_amlut[256] = {
	C02_AM_IMM, C02_AM__ZP_X_, -2, -2, C02_AM_ZP, C02_AM_ZP, C02_AM_ZP,
	C02_AM_ZP, -1, C02_AM_IMM, -1, -2, C02_AM_ABS, C02_AM_ABS, C02_AM_ABS,
	-10, C02_AM_REL, C02_AM__ZP__Y, C02_AM__ZP_, -2, C02_AM_ZP, C02_AM_ZP_X,
	C02_AM_ZP_X, C02_AM_ZP, -1, C02_AM_ABS_Y, -1, -2, C02_AM_ABS, C02_AM_ABS_X,
	C02_AM_ABS_X, -10, C02_AM_ABS, C02_AM__ZP_X_, -2, -2, C02_AM_ZP, C02_AM_ZP,
	C02_AM_ZP, C02_AM_ZP, -1, C02_AM_IMM, -1, -2, C02_AM_ABS, C02_AM_ABS,
	C02_AM_ABS, -10, C02_AM_REL, C02_AM__ZP__Y, C02_AM__ZP_, -2, C02_AM_ZP_X,
	C02_AM_ZP_X, C02_AM_ZP_X, C02_AM_ZP, -1, C02_AM_ABS_Y, -1, -2, C02_AM_ABS_X,
	C02_AM_ABS_X, C02_AM_ABS_X, -10, -1, C02_AM__ZP_X_, -2, -2, -2, C02_AM_ZP,
	C02_AM_ZP, C02_AM_ZP, -1, C02_AM_IMM, -1, -2, C02_AM_ABS, C02_AM_ABS,
	C02_AM_ABS, -10, C02_AM_REL, C02_AM__ZP__Y, C02_AM__ZP_, -2, -2,
	C02_AM_ZP_X, C02_AM_ZP_X, C02_AM_ZP, -1, C02_AM_ABS_Y, -1, -2, -2,
	C02_AM_ABS_X, C02_AM_ABS_X, -10, -1, C02_AM__ZP_X_, -2, -2, C02_AM_ZP,
	C02_AM_ZP, C02_AM_ZP, C02_AM_ZP, -1, C02_AM_IMM, -1, -2, C02_AM__ABS_,
	C02_AM_ABS, C02_AM_ABS, -10, C02_AM_REL, C02_AM__ZP__Y, C02_AM__ZP_, -2,
	C02_AM_ZP_X, C02_AM_ZP_X, C02_AM_ZP_X, C02_AM_ZP, -1, C02_AM_ABS_Y, -1,
	-2, C02_AM__ABS_X_, C02_AM_ABS_X, C02_AM_ABS_X, -10, C02_AM_REL,
	C02_AM__ZP_X_, -2, -2, C02_AM_ZP, C02_AM_ZP, C02_AM_ZP, C02_AM_ZP, -1,
	C02_AM_IMM, -1, -2, C02_AM_ABS, C02_AM_ABS, C02_AM_ABS, -10, C02_AM_REL,
	C02_AM__ZP__Y, C02_AM__ZP_, -2, C02_AM_ZP_X, C02_AM_ZP_X, C02_AM_ZP_Y,
	C02_AM_ZP, -1, C02_AM_ABS_Y, -1, -2, C02_AM_ABS, C02_AM_ABS_X, C02_AM_ABS_X,
	-10, C02_AM_IMM, C02_AM__ZP_X_, C02_AM_IMM, -2, C02_AM_ZP, C02_AM_ZP,
	C02_AM_ZP, C02_AM_ZP, -1, C02_AM_IMM, -1, -2, C02_AM_ABS, C02_AM_ABS,
	C02_AM_ABS, -10, C02_AM_REL, C02_AM__ZP__Y, C02_AM__ZP_, -2, C02_AM_ZP_X,
	C02_AM_ZP_X, C02_AM_ZP_Y, C02_AM_ZP, -1, C02_AM_ABS_Y, -1, -2, C02_AM_ABS_X,
	C02_AM_ABS_X, C02_AM_ABS_Y, -10, C02_AM_IMM, C02_AM__ZP_X_, -2, -2,
	C02_AM_ZP, C02_AM_ZP, C02_AM_ZP, C02_AM_ZP, -1, C02_AM_IMM, -1, -1,
	C02_AM_ABS, C02_AM_ABS, C02_AM_ABS, -10, C02_AM_REL, C02_AM__ZP__Y,
	C02_AM__ZP_, -2, -2, C02_AM_ZP_X, C02_AM_ZP_X, C02_AM_ZP, -1, C02_AM_ABS_Y,
	-1, -1, -2, C02_AM_ABS_X, C02_AM_ABS_X, -10, C02_AM_IMM, C02_AM__ZP_X_,
	-2, -2, C02_AM_ZP, C02_AM_ZP, C02_AM_ZP, C02_AM_ZP, -1, C02_AM_IMM, -1,
	-2, C02_AM_ABS, C02_AM_ABS, C02_AM_ABS, -10, C02_AM_REL, C02_AM__ZP__Y,
	C02_AM__ZP_, -2, -2, C02_AM_ZP_X, C02_AM_ZP_X, C02_AM_ZP, -1, C02_AM_ABS_Y,
	-1, -2, -2, C02_AM_ABS_X, C02_AM_ABS_X, -10
};

void c02_dumpreg(FILE * fp)
{
	fprintf(fp, "A=%02X X=%02X Y=%02X S=%02X P=%02X    %04X- ", C02_A, C02_X, C02_Y, C02_S, C02_P, C02_PC);
}

void c02_dumpstack(FILE * fp, unsigned char (*peek)(unsigned short), int n)
{
	int i;
	unsigned short a;
	for (i=0; i<n; i++) {
		a = 0x1FF - i;
		if ((a & 0xFF) <= C02_S) {
			fprintf(fp, "   ");
		} else {
			fprintf(fp, "%02X ", peek(a));
		}
	}
	fprintf(fp, "   ");
}

void c02_disasm(FILE * fp, unsigned char (*peek)(unsigned short), unsigned short addr)
{
	unsigned char i = peek(addr);
	unsigned char j;
	unsigned char k;
	int x;
	unsigned short a, ad;
	char * istr = (char *)c02_disasm_ilut[i];
	int iam = c02_disasm_amlut[i];
	switch (iam) {
	case -10:
	case C02_AM_ABS:
	case C02_AM_ABS_X:
	case C02_AM_ABS_Y:
	case C02_AM__ABS_:
	case C02_AM__ABS_X_:
		j = peek(addr+1);
		k = peek(addr+2);
		fprintf(fp, "%02X %02X %02X    ", i, j, k);
		break;
	case C02_AM_IMM:
	case C02_AM_ZP:
	case C02_AM_ZP_X:
	case C02_AM_ZP_Y:
	case C02_AM__ZP_:
	case C02_AM__ZP_X_:
	case C02_AM__ZP__Y:
	case C02_AM_REL:
		j = peek(addr+1);
		k = 0;
		fprintf(fp, "%02X %02X       ", i, j);
		break;
	case -2:
	case -1:
		fprintf(fp, "%02X          ", i);
		break;
	}
	switch (iam) {
	case -2:
		fprintf(fp, "???\n");
		break;
	case -1:
		fprintf(fp, "%s\n", istr);
		break;
	case -10:
		x = peek(addr+2);
		if (x >= 128) { x -= 256; }
		fprintf(fp, "%s #$%02X,$%04X\n", istr, peek(addr+1), addr+3+x);
		break;
	case C02_AM_IMM:
		fprintf(fp, "%s #$%02X\n", istr, peek(addr+1));
		break;
	case C02_AM_REL:
		x = peek(addr+1);
		if (x >= 128) { x -= 256; }
		fprintf(fp, "%s $%04X\n", istr, addr+2+x);
		break;
	default:
		fprintf(fp, "%s ", istr);
		a = j+k*256;
		ad = c02_operand_address(peek, addr+1, iam);
		switch (iam) {
		case C02_AM_ABS:		fprintf(fp, "$%04X", a);		break;
		case C02_AM_ABS_X:		fprintf(fp, "$%04X,X    {$%04X}", a, ad);		break;
		case C02_AM_ABS_Y:		fprintf(fp, "$%04X,Y    {$%04X}", a, ad);		break;
		case C02_AM__ABS_:		fprintf(fp, "($%04X)    {$%04X}", a, ad);		break;
		case C02_AM__ABS_X_:	fprintf(fp, "($%04X,X)  {$%04X}", a, ad);	break;
		case C02_AM_ZP:			fprintf(fp, "$%02X", a);		break;
		case C02_AM_ZP_X:		fprintf(fp, "$%02X,X      {$%04X}", a, ad);		break;
		case C02_AM_ZP_Y:		fprintf(fp, "$%02X,Y      {$%04X}", a, ad);		break;
		case C02_AM__ZP_:		fprintf(fp, "($%02X)      {$%04X}", a, ad);		break;
		case C02_AM__ZP_X_:		fprintf(fp, "($%02X,X)    {$%04X}", a, ad);	break;
		case C02_AM__ZP__Y:		fprintf(fp, "($%02X),Y    {$%04X}", a, ad);	break;
		}
		fprintf(fp, "\n");
		break;
	}
}

void c02_dump(FILE * fp, unsigned char (*peek)(unsigned short))
{
	c02_dumpstack(fp, peek, 8);
	c02_dumpreg(fp);
	c02_disasm(fp, peek, C02_PC);
}

void c02_dumpjump(FILE * fp, unsigned char (*peek)(unsigned short))
{
	if ( c02_disasm_ilut[peek(C02_PC)][0] == 'J' ) c02_dump(fp, peek);
}

/* Big Main Processing Routine */

void c02_exec(unsigned char (*peek)(unsigned short), void (*poke)(unsigned short, unsigned char), int * timing)
{
	/* This takes two function pointers, one to a 'peek' function, */
	/* one to a 'poke' function, and a pointer to a variable used  */
	/* to keep track of clock cycles. */
	/* The reason peek and poke functions are used instead of a */
	/* 64Kbyte array is that we want to simulate the Apple II's */
	/* MMU and IOU. */
	unsigned short int a, a2;
	switch (peek(C02_PC)) {
	case 0x69: /* ADC #Oper */
		C02_PC += 2;	*timing += 2;
		a = c02_operand_value(peek, C02_PC-1, C02_AM_IMM);
		__adc(a);
		break;
	case 0x65: /* ADC Zpg */
		C02_PC += 2;	*timing += 3;
		a = c02_operand_value(peek, C02_PC-1, C02_AM_ZP);
		__adc(a);
		break;
	case 0x75: /* ADC Zpg,X */
		C02_PC += 2;	*timing += 4;
		a = c02_operand_value(peek, C02_PC-1, C02_AM_ZP_X);
		__adc(a);
		break;
	case 0x6D: /* ADC Abs */
		C02_PC += 3;	*timing += 4;
		a = c02_operand_value(peek, C02_PC-2, C02_AM_ABS);
		__adc(a);
		break;
	case 0x7D: /* ADC Abs,X */
		C02_PC += 3;	*timing += 4;
		a = c02_operand_value(peek, C02_PC-2, C02_AM_ABS_X);
		__adc(a);
		break;
	case 0x79: /* ADC Abs,Y */
		C02_PC += 3;	*timing += 4;
		a = c02_operand_value(peek, C02_PC-2, C02_AM_ABS_Y);
		__adc(a);
		break;
	case 0x61: /* ADC (Zpg,X) */
		C02_PC += 2;	*timing += 6;
		a = c02_operand_value(peek, C02_PC-1, C02_AM__ZP_X_);
		__adc(a);
		break;
	case 0x71: /* ADC (Zpg),Y */
		C02_PC += 2;	*timing += 5;
		a = c02_operand_value(peek, C02_PC-1, C02_AM__ZP__Y);
		__adc(a);
		break;
	case 0x72: /* ADC (Zpg) */
		C02_PC += 2;	*timing += 5;
		a = c02_operand_value(peek, C02_PC-1, C02_AM__ZP_);
		__adc(a);
		break;
	case 0x29: /* AND #Oper */
		C02_PC += 2;	*timing += 2;
		a = c02_operand_value(peek, C02_PC-1, C02_AM_IMM);
		C02_A &= a;
		__set_p_for(C02_A);
		break;
	case 0x25: /* AND Zpg */
		C02_PC += 2;	*timing += 3;
		a = c02_operand_value(peek, C02_PC-1, C02_AM_ZP);
		C02_A &= a;
		__set_p_for(C02_A);
		break;
	case 0x35: /* AND Zpg,X */
		C02_PC += 2;	*timing += 4;
		a = c02_operand_value(peek, C02_PC-1, C02_AM_ZP_X);
		C02_A &= a;
		__set_p_for(C02_A);
		break;
	case 0x2D: /* AND Abs */
		C02_PC += 3;	*timing += 4;
		a = c02_operand_value(peek, C02_PC-2, C02_AM_ABS);
		C02_A &= a;
		__set_p_for(C02_A);
		break;
	case 0x3D: /* AND Abs,X */
		C02_PC += 3;	*timing += 4;
		a = c02_operand_value(peek, C02_PC-2, C02_AM_ABS_X);
		C02_A &= a;
		__set_p_for(C02_A);
		break;
	case 0x39: /* AND Abs,Y */
		C02_PC += 3;	*timing += 4;
		a = c02_operand_value(peek, C02_PC-2, C02_AM_ABS_Y);
		C02_A &= a;
		__set_p_for(C02_A);
		break;
	case 0x21: /* AND (Zpg,X) */
		C02_PC += 2;	*timing += 6;
		a = c02_operand_value(peek, C02_PC-1, C02_AM__ZP_X_);
		C02_A &= a;
		__set_p_for(C02_A);
		break;
	case 0x31: /* AND (Zpg),Y */
		C02_PC += 2;	*timing += 5;
		a = c02_operand_value(peek, C02_PC-1, C02_AM__ZP__Y);
		C02_A &= a;
		__set_p_for(C02_A);
		break;
	case 0x32: /* AND (Zpg) */
		C02_PC += 2;	*timing += 5;
		a = c02_operand_value(peek, C02_PC-1, C02_AM__ZP_);
		C02_A &= a;
		__set_p_for(C02_A);
		break;
	case 0x0A: /* ASL A */
		C02_PC++;	*timing += 2;
		C02_A = __asl(C02_A);
		__set_p_for(C02_A);
		break;
	case 0x06: /* ASL Zpg */
		C02_PC += 2;	*timing += 5;
		a = c02_operand_address(peek, C02_PC-1, C02_AM_ZP);
		a2 = __asl(peek(a));
		poke(a, a2);
		__set_p_for(a2);
		break;
	case 0x16: /* ASL Zpg,X */
		C02_PC += 2;	*timing += 6;
		a = c02_operand_address(peek, C02_PC-1, C02_AM_ZP_X);
		a2 = __asl(peek(a));
		poke(a, a2);
		__set_p_for(a2);
		break;
	case 0x0E: /* ASL Abs */
		C02_PC += 3;	*timing += 6;
		a = c02_operand_address(peek, C02_PC-2, C02_AM_ABS);
		a2 = __asl(peek(a));
		poke(a, a2);
		__set_p_for(a2);
		break;
	case 0x1E: /* ASL Abs,X */
		C02_PC += 3;	*timing += 7;
		a = c02_operand_address(peek, C02_PC-2, C02_AM_ABS_X);
		a2 = __asl(peek(a));
		poke(a, a2);
		__set_p_for(a2);
		break;
	case 0x0F: /* BBR0 Oper */
	case 0x1F: /* BBR1 Oper */
	case 0x2F: /* BBR2 Oper */
	case 0x3F: /* BBR3 Oper */
	case 0x4F: /* BBR4 Oper */
	case 0x5F: /* BBR5 Oper */
	case 0x6F: /* BBR6 Oper */
	case 0x7F: /* BBR7 Oper */
		C02_PC += 3;	*timing += 2;
		a = peek(peek(C02_PC-2));
		/* this conditional statement is absolutely hideous */
		/* 1. get operand byte */
		/* 2. AND with 0x70 */
		/* 3. shift right 4 bits; now we have bit number */
		/* 4. shift 0x01 left that number of bits; now we have a mask */
		/* 5. AND a with the mask; now we have just that bit of a */
		/* 6. if that bit is NOT set... */
		if (!
			(a &
				(1 <<
					(
						(
							peek(C02_PC-3)
						& 0x70)
					>> 4)
				)
			)
		) {
			(*timing)++;
			a = c02_operand_address(peek, C02_PC-1, C02_AM_REL);
			if ((a & 0xFF00) != (C02_PC & 0xFF00)) { (*timing)++; }
			C02_PC = a;
		}
		break;
	case 0x8F: /* BBS0 Oper */
	case 0x9F: /* BBS1 Oper */
	case 0xAF: /* BBS2 Oper */
	case 0xBF: /* BBS3 Oper */
	case 0xCF: /* BBS4 Oper */
	case 0xDF: /* BBS5 Oper */
	case 0xEF: /* BBS6 Oper */
	case 0xFF: /* BBS7 Oper */
		C02_PC += 3;	*timing += 2;
		a = peek(peek(C02_PC-2));
		/* so is this one */
		/* 1. get operand byte */
		/* 2. AND with 0x70 */
		/* 3. shift right 4 bits; now we have bit number */
		/* 4. shift 0x01 left that number of bits; now we have a mask */
		/* 5. AND a with the mask; now we have just that bit of a */
		/* 6. if that bit IS set... */
		if (
			(a &
				(1 <<
					(
						(
							peek(C02_PC-3)
						& 0x70)
					>> 4)
				)
			)
		) {
			(*timing)++;
			a = c02_operand_address(peek, C02_PC-1, C02_AM_REL);
			if ((a & 0xFF00) != (C02_PC & 0xFF00)) { (*timing)++; }
			C02_PC = a;
		}
		break;
		break;
	case 0x90: /* BCC Oper */
		C02_PC += 2;	*timing += 2;
		if (! __get_p(C02_P_C)) {
			(*timing)++;
			a = c02_operand_address(peek, C02_PC-1, C02_AM_REL);
			if ((a & 0xFF00) != (C02_PC & 0xFF00)) { (*timing)++; }
			C02_PC = a;
		}
		break;
	case 0xB0: /* BCS Oper */
		C02_PC += 2;	*timing += 2;
		if (__get_p(C02_P_C)) {
			(*timing)++;
			a = c02_operand_address(peek, C02_PC-1, C02_AM_REL);
			if ((a & 0xFF00) != (C02_PC & 0xFF00)) { (*timing)++; }
			C02_PC = a;
		}
		break;
	case 0xF0: /* BEQ Oper */
		C02_PC += 2;	*timing += 2;
		if (__get_p(C02_P_Z)) {
			(*timing)++;
			a = c02_operand_address(peek, C02_PC-1, C02_AM_REL);
			if ((a & 0xFF00) != (C02_PC & 0xFF00)) { (*timing)++; }
			C02_PC = a;
		}
		break;
	case 0x89: /* BIT #Oper */
		C02_PC += 2;	*timing += 2;
		a = c02_operand_value(peek, C02_PC-1, C02_AM_IMM);
		__bit(a,0);
		break;
	case 0x24: /* BIT Zpg */
		C02_PC += 2;	*timing += 3;
		a = c02_operand_value(peek, C02_PC-1, C02_AM_ZP);
		__bit(a,1);
		break;
	case 0x34: /* BIT Zpg,X */
		C02_PC += 2;	*timing += 4;
		a = c02_operand_value(peek, C02_PC-1, C02_AM_ZP_X);
		__bit(a,1);
		break;
	case 0x2C: /* BIT Abs */
		C02_PC += 3;	*timing += 4;
		a = c02_operand_value(peek, C02_PC-2, C02_AM_ABS);
		__bit(a,1);
		break;
	case 0x3C: /* BIT Abs,X */
		C02_PC += 3;	*timing += 4;
		a = c02_operand_value(peek, C02_PC-2, C02_AM_ABS_X);
		__bit(a,1);
		break;
	case 0x30: /* BMI Oper */
		C02_PC += 2;	*timing += 2;
		if (__get_p(C02_P_N)) {
			(*timing)++;
			a = c02_operand_address(peek, C02_PC-1, C02_AM_REL);
			if ((a & 0xFF00) != (C02_PC & 0xFF00)) { (*timing)++; }
			C02_PC = a;
		}
		break;
	case 0xD0: /* BNE Oper */
		C02_PC += 2;	*timing += 2;
		if (! __get_p(C02_P_Z)) {
			(*timing)++;
			a = c02_operand_address(peek, C02_PC-1, C02_AM_REL);
			if ((a & 0xFF00) != (C02_PC & 0xFF00)) { (*timing)++; }
			C02_PC = a;
		}
		break;
	case 0x10: /* BPL Oper */
		C02_PC += 2;	*timing += 2;
		if (! __get_p(C02_P_N)) {
			(*timing)++;
			a = c02_operand_address(peek, C02_PC-1, C02_AM_REL);
			if ((a & 0xFF00) != (C02_PC & 0xFF00)) { (*timing)++; }
			C02_PC = a;
		}
		break;
	case 0x80: /* BRA Oper */
		C02_PC += 2;	*timing += 3;
		a = c02_operand_address(peek, C02_PC-1, C02_AM_REL);
		if ((a & 0xFF00) != (C02_PC & 0xFF00)) { (*timing)++; }
		C02_PC = a;
		break;
	case 0x00: /* BRK */
		c02_brk(peek, poke);
		break;
	case 0x50: /* BVC Oper */
		C02_PC += 2;	*timing += 2;
		if (! __get_p(C02_P_V)) {
			(*timing)++;
			a = c02_operand_address(peek, C02_PC-1, C02_AM_REL);
			if ((a & 0xFF00) != (C02_PC & 0xFF00)) { (*timing)++; }
			C02_PC = a;
		}
		break;
	case 0x70: /* BVS Oper */
		C02_PC += 2;	*timing += 2;
		if (__get_p(C02_P_V)) {
			(*timing)++;
			a = c02_operand_address(peek, C02_PC-1, C02_AM_REL);
			if ((a & 0xFF00) != (C02_PC & 0xFF00)) { (*timing)++; }
			C02_PC = a;
		}
		break;
	case 0x18: /* CLC */
		C02_PC++;	*timing += 2;
		__clear_p(C02_P_C);
		break;
	case 0xD8: /* CLD */
		C02_PC++;	*timing += 2;
		__clear_p(C02_P_D);
		break;
	case 0x58: /* CLI */
		C02_PC++;	*timing += 2;
		__clear_p(C02_P_I);
		break;
	case 0xB8: /* CLV */
		C02_PC++;	*timing += 2;
		__clear_p(C02_P_V);
		break;
	case 0xC9: /* CMP #Oper */
		C02_PC += 2;	*timing += 2;
		a = c02_operand_value(peek, C02_PC-1, C02_AM_IMM);
		__cmp(C02_A, a);
		break;
	case 0xC5: /* CMP Zpg */
		C02_PC += 2;	*timing += 3;
		a = c02_operand_value(peek, C02_PC-1, C02_AM_ZP);
		__cmp(C02_A, a);
		break;
	case 0xD5: /* CMP Zpg,X */
		C02_PC += 2;	*timing += 4;
		a = c02_operand_value(peek, C02_PC-1, C02_AM_ZP_X);
		__cmp(C02_A, a);
		break;
	case 0xCD: /* CMP Abs */
		C02_PC += 3;	*timing += 4;
		a = c02_operand_value(peek, C02_PC-2, C02_AM_ABS);
		__cmp(C02_A, a);
		break;
	case 0xDD: /* CMP Abs,X */
		C02_PC += 3;	*timing += 4;
		a = c02_operand_value(peek, C02_PC-2, C02_AM_ABS_X);
		__cmp(C02_A, a);
		break;
	case 0xD9: /* CMP Abs,Y */
		C02_PC += 3;	*timing += 4;
		a = c02_operand_value(peek, C02_PC-2, C02_AM_ABS_Y);
		__cmp(C02_A, a);
		break;
	case 0xC1: /* CMP (Zpg,X) */
		C02_PC += 2;	*timing += 6;
		a = c02_operand_value(peek, C02_PC-1, C02_AM__ZP_X_);
		__cmp(C02_A, a);
		break;
	case 0xD1: /* CMP (Zpg),Y */
		C02_PC += 2;	*timing += 5;
		a = c02_operand_value(peek, C02_PC-1, C02_AM__ZP__Y);
		__cmp(C02_A, a);
		break;
	case 0xD2: /* CMP (Zpg) */
		C02_PC += 2;	*timing += 5;
		a = c02_operand_value(peek, C02_PC-1, C02_AM__ZP_);
		__cmp(C02_A, a);
		break;
	case 0xE0: /* CPX #Oper */
		C02_PC += 2;	*timing += 2;
		a = c02_operand_value(peek, C02_PC-1, C02_AM_IMM);
		__cmp(C02_X, a);
		break;
	case 0xE4: /* CPX Zpg */
		C02_PC += 2;	*timing += 3;
		a = c02_operand_value(peek, C02_PC-1, C02_AM_ZP);
		__cmp(C02_X, a);
		break;
	case 0xEC: /* CPX Abs */
		C02_PC += 3;	*timing += 4;
		a = c02_operand_value(peek, C02_PC-2, C02_AM_ABS);
		__cmp(C02_X, a);
		break;
	case 0xC0: /* CPY #Oper */
		C02_PC += 2;	*timing += 2;
		a = c02_operand_value(peek, C02_PC-1, C02_AM_IMM);
		__cmp(C02_Y, a);
		break;
	case 0xC4: /* CPY Zpg */
		C02_PC += 2;	*timing += 3;
		a = c02_operand_value(peek, C02_PC-1, C02_AM_ZP);
		__cmp(C02_Y, a);
		break;
	case 0xCC: /* CPY Abs */
		C02_PC += 3;	*timing += 4;
		a = c02_operand_value(peek, C02_PC-2, C02_AM_ABS);
		__cmp(C02_Y, a);
		break;
	case 0x3A: /* DEA */
		C02_PC++;	*timing += 2;
		C02_A--;
		__set_p_for(C02_A);
		break;
	case 0xC6: /* DEC Zpg */
		C02_PC += 2;	*timing += 5;
		a = c02_operand_address(peek, C02_PC-1, C02_AM_ZP);
		a2 = (peek(a)-1) & 0xFF;
		poke(a, a2);
		__set_p_for(a2);
		break;
	case 0xD6: /* DEC Zpg,X */
		C02_PC += 2;	*timing += 6;
		a = c02_operand_address(peek, C02_PC-1, C02_AM_ZP_X);
		a2 = (peek(a)-1) & 0xFF;
		poke(a, a2);
		__set_p_for(a2);
		break;
	case 0xCE: /* DEC Abs */
		C02_PC += 3;	*timing += 6;
		a = c02_operand_address(peek, C02_PC-2, C02_AM_ABS);
		a2 = (peek(a)-1) & 0xFF;
		poke(a, a2);
		__set_p_for(a2);
		break;
	case 0xDE: /* DEC Abs,X */
		C02_PC += 3;	*timing += 7;
		a = c02_operand_address(peek, C02_PC-2, C02_AM_ABS_X);
		a2 = (peek(a)-1) & 0xFF;
		poke(a, a2);
		__set_p_for(a2);
		break;
	case 0xCA: /* DEX */
		C02_PC++;	*timing += 2;
		C02_X--;
		__set_p_for(C02_X);
		break;
	case 0x88: /* DEY */
		C02_PC++;	*timing += 2;
		C02_Y--;
		__set_p_for(C02_Y);
		break;
	case 0x49: /* EOR #Oper */
		C02_PC += 2;	*timing += 2;
		a = c02_operand_value(peek, C02_PC-1, C02_AM_IMM);
		C02_A ^= a;
		__set_p_for(C02_A);
		break;
	case 0x45: /* EOR Zpg */
		C02_PC += 2;	*timing += 3;
		a = c02_operand_value(peek, C02_PC-1, C02_AM_ZP);
		C02_A ^= a;
		__set_p_for(C02_A);
		break;
	case 0x55: /* EOR Zpg,X */
		C02_PC += 2;	*timing += 4;
		a = c02_operand_value(peek, C02_PC-1, C02_AM_ZP_X);
		C02_A ^= a;
		__set_p_for(C02_A);
		break;
	case 0x4D: /* EOR Abs */
		C02_PC += 3;	*timing += 4;
		a = c02_operand_value(peek, C02_PC-2, C02_AM_ABS);
		C02_A ^= a;
		__set_p_for(C02_A);
		break;
	case 0x5D: /* EOR Abs,X */
		C02_PC += 3;	*timing += 4;
		a = c02_operand_value(peek, C02_PC-2, C02_AM_ABS_X);
		C02_A ^= a;
		__set_p_for(C02_A);
		break;
	case 0x59: /* EOR Abs,Y */
		C02_PC += 3;	*timing += 4;
		a = c02_operand_value(peek, C02_PC-2, C02_AM_ABS_Y);
		C02_A ^= a;
		__set_p_for(C02_A);
		break;
	case 0x41: /* EOR (Zpg,X) */
		C02_PC += 2;	*timing += 6;
		a = c02_operand_value(peek, C02_PC-1, C02_AM__ZP_X_);
		C02_A ^= a;
		__set_p_for(C02_A);
		break;
	case 0x51: /* EOR (Zpg),Y */
		C02_PC += 2;	*timing += 5;
		a = c02_operand_value(peek, C02_PC-1, C02_AM__ZP__Y);
		C02_A ^= a;
		__set_p_for(C02_A);
		break;
	case 0x52: /* EOR (Zpg) */
		C02_PC += 2;	*timing += 5;
		a = c02_operand_value(peek, C02_PC-1, C02_AM__ZP_);
		C02_A ^= a;
		__set_p_for(C02_A);
		break;
	case 0x1A: /* INA */
		C02_PC++;	*timing += 2;
		C02_A++;
		__set_p_for(C02_A);
		break;
	case 0xE6: /* INC Zpg */
		C02_PC += 2;	*timing += 5;
		a = c02_operand_address(peek, C02_PC-1, C02_AM_ZP);
		a2 = (peek(a)+1) & 0xFF;
		poke(a, a2);
		__set_p_for(a2);
		break;
	case 0xF6: /* INC Zpg,X */
		C02_PC += 2;	*timing += 6;
		a = c02_operand_address(peek, C02_PC-1, C02_AM_ZP_X);
		a2 = (peek(a)+1) & 0xFF;
		poke(a, a2);
		__set_p_for(a2);
		break;
	case 0xEE: /* INC Abs */
		C02_PC += 3;	*timing += 6;
		a = c02_operand_address(peek, C02_PC-2, C02_AM_ABS);
		a2 = (peek(a)+1) & 0xFF;
		poke(a, a2);
		__set_p_for(a2);
		break;
	case 0xFE: /* INC Abs,X */
		C02_PC += 3;	*timing += 7;
		a = c02_operand_address(peek, C02_PC-2, C02_AM_ABS_X);
		a2 = (peek(a)+1) & 0xFF;
		poke(a, a2);
		__set_p_for(a2);
		break;
	case 0xE8: /* INX */
		C02_PC++;	*timing += 2;
		C02_X++;
		__set_p_for(C02_X);
		break;
	case 0xC8: /* INY */
		C02_PC++;	*timing += 2;
		C02_Y++;
		__set_p_for(C02_Y);
		break;
	case 0x4C: /* JMP Abs */
		C02_PC += 3;	*timing += 3;
		C02_PC = c02_operand_address(peek, C02_PC-2, C02_AM_ABS);
		break;
	case 0x6C: /* JMP (Abs) */
		C02_PC += 3;	*timing += 5;
		C02_PC = c02_operand_address(peek, C02_PC-2, C02_AM__ABS_);
		break;
	case 0x7C: /* JMP (Abs,X) */
		C02_PC += 3;	*timing += 6;
		C02_PC = c02_operand_address(peek, C02_PC-2, C02_AM__ABS_X_);
		break;
	case 0x20: /* JSR Abs */
		C02_PC += 3;	*timing += 6;
		a = c02_operand_address(peek, C02_PC-2, C02_AM_ABS);
		c02_push(poke, ((C02_PC-1)&0xFF00) >> 8);
		c02_push(poke, (C02_PC-1)&0x00FF);
		C02_PC = a;
		break;
	case 0xA9: /* LDA #Oper */
		C02_PC += 2;	*timing += 2;
		C02_A = c02_operand_value(peek, C02_PC-1, C02_AM_IMM);
		__set_p_for(C02_A);
		break;
	case 0xA5: /* LDA Zpg */
		C02_PC += 2;	*timing += 3;
		C02_A = c02_operand_value(peek, C02_PC-1, C02_AM_ZP);
		__set_p_for(C02_A);
		break;
	case 0xB5: /* LDA Zpg,X */
		C02_PC += 2;	*timing += 4;
		C02_A = c02_operand_value(peek, C02_PC-1, C02_AM_ZP_X);
		__set_p_for(C02_A);
		break;
	case 0xAD: /* LDA Abs */
		C02_PC += 3;	*timing += 4;
		C02_A = c02_operand_value(peek, C02_PC-2, C02_AM_ABS);
		__set_p_for(C02_A);
		break;
	case 0xBD: /* LDA Abs,X */
		C02_PC += 3;	*timing += 4;
		C02_A = c02_operand_value(peek, C02_PC-2, C02_AM_ABS_X);
		__set_p_for(C02_A);
		break;
	case 0xB9: /* LDA Abs,Y */
		C02_PC += 3;	*timing += 4;
		C02_A = c02_operand_value(peek, C02_PC-2, C02_AM_ABS_Y);
		__set_p_for(C02_A);
		break;
	case 0xA1: /* LDA (Zpg,X) */
		C02_PC += 2;	*timing += 6;
		C02_A = c02_operand_value(peek, C02_PC-1, C02_AM__ZP_X_);
		__set_p_for(C02_A);
		break;
	case 0xB1: /* LDA (Zpg),Y */
		C02_PC += 2;	*timing += 5;
		C02_A = c02_operand_value(peek, C02_PC-1, C02_AM__ZP__Y);
		__set_p_for(C02_A);
		break;
	case 0xB2: /* LDA (Zpg) */
		C02_PC += 2;	*timing += 5;
		C02_A = c02_operand_value(peek, C02_PC-1, C02_AM__ZP_);
		__set_p_for(C02_A);
		break;
	case 0xA2: /* LDX #Oper */
		C02_PC += 2;	*timing += 2;
		C02_X = c02_operand_value(peek, C02_PC-1, C02_AM_IMM);
		__set_p_for(C02_X);
		break;
	case 0xA6: /* LDX Zpg */
		C02_PC += 2;	*timing += 3;
		C02_X = c02_operand_value(peek, C02_PC-1, C02_AM_ZP);
		__set_p_for(C02_X);
		break;
	case 0xB6: /* LDX Zpg,Y */
		C02_PC += 2;	*timing += 4;
		C02_X = c02_operand_value(peek, C02_PC-1, C02_AM_ZP_Y);
		__set_p_for(C02_X);
		break;
	case 0xAE: /* LDX Abs */
		C02_PC += 3;	*timing += 4;
		C02_X = c02_operand_value(peek, C02_PC-2, C02_AM_ABS);
		__set_p_for(C02_X);
		break;
	case 0xBE: /* LDX Abs,Y */
		C02_PC += 3;	*timing += 4;
		C02_X = c02_operand_value(peek, C02_PC-2, C02_AM_ABS_Y);
		__set_p_for(C02_X);
		break;
	case 0xA0: /* LDY #Oper */
		C02_PC += 2;	*timing += 2;
		C02_Y = c02_operand_value(peek, C02_PC-1, C02_AM_IMM);
		__set_p_for(C02_Y);
		break;
	case 0xA4: /* LDY Zpg */
		C02_PC += 2;	*timing += 3;
		C02_Y = c02_operand_value(peek, C02_PC-1, C02_AM_ZP);
		__set_p_for(C02_Y);
		break;
	case 0xB4: /* LDY Zpg,X */
		C02_PC += 2;	*timing += 4;
		C02_Y = c02_operand_value(peek, C02_PC-1, C02_AM_ZP_X);
		__set_p_for(C02_Y);
		break;
	case 0xAC: /* LDY Abs */
		C02_PC += 3;	*timing += 4;
		C02_Y = c02_operand_value(peek, C02_PC-2, C02_AM_ABS);
		__set_p_for(C02_Y);
		break;
	case 0xBC: /* LDY Abs,X */
		C02_PC += 3;	*timing += 4;
		C02_Y = c02_operand_value(peek, C02_PC-2, C02_AM_ABS_X);
		__set_p_for(C02_Y);
		break;
	case 0x4A: /* LSR A */
		C02_PC++;	*timing += 2;
		C02_A = __lsr(C02_A);
		__set_p_for(C02_A);
		break;
	case 0x46: /* LSR Zpg */
		C02_PC += 2;	*timing += 5;
		a = c02_operand_address(peek, C02_PC-1, C02_AM_ZP);
		a2 = __lsr(peek(a));
		poke(a, a2);
		__set_p_for(a2);
		break;
	case 0x56: /* LSR Zpg,X */
		C02_PC += 2;	*timing += 6;
		a = c02_operand_address(peek, C02_PC-1, C02_AM_ZP_X);
		a2 = __lsr(peek(a));
		poke(a, a2);
		__set_p_for(a2);
		break;
	case 0x4E: /* LSR Abs */
		C02_PC += 3;	*timing += 6;
		a = c02_operand_address(peek, C02_PC-2, C02_AM_ABS);
		a2 = __lsr(peek(a));
		poke(a, a2);
		__set_p_for(a2);
		break;
	case 0x5E: /* LSR Abs,X */
		C02_PC += 3;	*timing += 7;
		a = c02_operand_address(peek, C02_PC-2, C02_AM_ABS_X);
		a2 = __lsr(peek(a));
		poke(a, a2);
		__set_p_for(a2);
		break;
	case 0xEA: /* NOP */
		C02_PC++;	*timing += 2;
		break;
	case 0x09: /* ORA #Oper */
		C02_PC += 2;	*timing += 2;
		a = c02_operand_value(peek, C02_PC-1, C02_AM_IMM);
		C02_A |= a;
		__set_p_for(C02_A);
		break;
	case 0x05: /* ORA Zpg */
		C02_PC += 2;	*timing += 3;
		a = c02_operand_value(peek, C02_PC-1, C02_AM_ZP);
		C02_A |= a;
		__set_p_for(C02_A);
		break;
	case 0x15: /* ORA Zpg,X */
		C02_PC += 2;	*timing += 4;
		a = c02_operand_value(peek, C02_PC-1, C02_AM_ZP_X);
		C02_A |= a;
		__set_p_for(C02_A);
		break;
	case 0x0D: /* ORA Abs */
		C02_PC += 3;	*timing += 4;
		a = c02_operand_value(peek, C02_PC-2, C02_AM_ABS);
		C02_A |= a;
		__set_p_for(C02_A);
		break;
	case 0x1D: /* ORA Abs,X */
		C02_PC += 3;	*timing += 4;
		a = c02_operand_value(peek, C02_PC-2, C02_AM_ABS_X);
		C02_A |= a;
		__set_p_for(C02_A);
		break;
	case 0x19: /* ORA Abs,Y */
		C02_PC += 3;	*timing += 4;
		a = c02_operand_value(peek, C02_PC-2, C02_AM_ABS_Y);
		C02_A |= a;
		__set_p_for(C02_A);
		break;
	case 0x01: /* ORA (Zpg,X) */
		C02_PC += 2;	*timing += 6;
		a = c02_operand_value(peek, C02_PC-1, C02_AM__ZP_X_);
		C02_A |= a;
		__set_p_for(C02_A);
		break;
	case 0x11: /* ORA (Zpg),Y */
		C02_PC += 2;	*timing += 5;
		a = c02_operand_value(peek, C02_PC-1, C02_AM__ZP__Y);
		C02_A |= a;
		__set_p_for(C02_A);
		break;
	case 0x12: /* ORA (Zpg) */
		C02_PC += 2;	*timing += 5;
		a = c02_operand_value(peek, C02_PC-1, C02_AM__ZP_);
		C02_A |= a;
		__set_p_for(C02_A);
		break;
	case 0x48: /* PHA */
		C02_PC++;	*timing += 3;
		c02_push(poke, C02_A);
		break;
	case 0x08: /* PHP */
		C02_PC++;	*timing += 3;
		c02_push(poke, C02_P);
		break;
	case 0xDA: /* PHX */
		C02_PC++;	*timing += 3;
		c02_push(poke, C02_X);
		break;
	case 0x5A: /* PHY */
		C02_PC++;	*timing += 3;
		c02_push(poke, C02_Y);
		break;
	case 0x68: /* PLA */
		C02_PC++;	*timing += 4;
		C02_A = c02_pull(peek);
		__set_p_for(C02_A);
		break;
	case 0x28: /* PLP */
		C02_PC++;	*timing += 4;
		C02_P = c02_pull(peek);
		break;
	case 0xFA: /* PLX */
		C02_PC++;	*timing += 4;
		C02_X = c02_pull(peek);
		__set_p_for(C02_X);
		break;
	case 0x7A: /* PLY */
		C02_PC++;	*timing += 4;
		C02_Y = c02_pull(peek);
		__set_p_for(C02_Y);
		break;
	case 0x07: /* RMB0 Zpg */
	case 0x17: /* RMB1 Zpg */
	case 0x27: /* RMB2 Zpg */
	case 0x37: /* RMB3 Zpg */
	case 0x47: /* RMB4 Zpg */
	case 0x57: /* RMB5 Zpg */
	case 0x67: /* RMB6 Zpg */
	case 0x77: /* RMB7 Zpg */
		C02_PC += 2;	*timing += 5;
		a = c02_operand_address(peek, C02_PC-1, C02_AM_ZP);
		poke(a, peek(a) & ~(1 << ((peek(C02_PC-2) & 0x70) >> 4)));
		break;
	case 0x2A: /* ROL A */
		C02_PC++;	*timing += 2;
		C02_A = __rol(C02_A);
		__set_p_for(C02_A);
		break;
	case 0x26: /* ROL Zpg */
		C02_PC += 2;	*timing += 5;
		a = c02_operand_address(peek, C02_PC-1, C02_AM_ZP);
		a2 = __rol(peek(a));
		poke(a, a2);
		__set_p_for(a2);
		break;
	case 0x36: /* ROL Zpg,X */
		C02_PC += 2;	*timing += 6;
		a = c02_operand_address(peek, C02_PC-1, C02_AM_ZP_X);
		a2 = __rol(peek(a));
		poke(a, a2);
		__set_p_for(a2);
		break;
	case 0x2E: /* ROL Abs */
		C02_PC += 3;	*timing += 6;
		a = c02_operand_address(peek, C02_PC-2, C02_AM_ABS);
		a2 = __rol(peek(a));
		poke(a, a2);
		__set_p_for(a2);
		break;
	case 0x3E: /* ROL Abs,X */
		C02_PC += 3;	*timing += 7;
		a = c02_operand_address(peek, C02_PC-2, C02_AM_ABS_X);
		a2 = __rol(peek(a));
		poke(a, a2);
		__set_p_for(a2);
		break;
	case 0x6A: /* ROR A */
		C02_PC++;	*timing += 2;
		C02_A = __ror(C02_A);
		__set_p_for(C02_A);
		break;
	case 0x66: /* ROR Zpg */
		C02_PC += 2;	*timing += 5;
		a = c02_operand_address(peek, C02_PC-1, C02_AM_ZP);
		a2 = __ror(peek(a));
		poke(a, a2);
		__set_p_for(a2);
		break;
	case 0x76: /* ROR Zpg,X */
		C02_PC += 2;	*timing += 6;
		a = c02_operand_address(peek, C02_PC-1, C02_AM_ZP_X);
		a2 = __ror(peek(a));
		poke(a, a2);
		__set_p_for(a2);
		break;
	case 0x6E: /* ROR Abs */
		C02_PC += 3;	*timing += 6;
		a = c02_operand_address(peek, C02_PC-2, C02_AM_ABS);
		a2 = __ror(peek(a));
		poke(a, a2);
		__set_p_for(a2);
		break;
	case 0x7E: /* ROR Abs,X */
		C02_PC += 3;	*timing += 7;
		a = c02_operand_address(peek, C02_PC-2, C02_AM_ABS_X);
		a2 = __ror(peek(a));
		poke(a, a2);
		__set_p_for(a2);
		break;
	case 0x40: /* RTI */
		C02_PC++;	*timing += 6;
		C02_P = c02_pull(peek);
		a = c02_pull(peek);
		a += (c02_pull(peek) * 256);
		C02_PC = a;
		break;
	case 0x60: /* RTS */
		C02_PC++;	*timing += 6;
		a = c02_pull(peek);
		a += (c02_pull(peek) * 256);
		C02_PC = a + 1;
		break;
	case 0xE9: /* SBC #Oper */
		C02_PC += 2;	*timing += 2;
		a = c02_operand_value(peek, C02_PC-1, C02_AM_IMM);
		__sbc(a);
		break;
	case 0xE5: /* SBC Zpg */
		C02_PC += 2;	*timing += 3;
		a = c02_operand_value(peek, C02_PC-1, C02_AM_ZP);
		__sbc(a);
		break;
	case 0xF5: /* SBC Zpg,X */
		C02_PC += 2;	*timing += 4;
		a = c02_operand_value(peek, C02_PC-1, C02_AM_ZP_X);
		__sbc(a);
		break;
	case 0xED: /* SBC Abs */
		C02_PC += 3;	*timing += 4;
		a = c02_operand_value(peek, C02_PC-2, C02_AM_ABS);
		__sbc(a);
		break;
	case 0xFD: /* SBC Abs,X */
		C02_PC += 3;	*timing += 4;
		a = c02_operand_value(peek, C02_PC-2, C02_AM_ABS_X);
		__sbc(a);
		break;
	case 0xF9: /* SBC Abs,Y */
		C02_PC += 3;	*timing += 4;
		a = c02_operand_value(peek, C02_PC-2, C02_AM_ABS_Y);
		__sbc(a);
		break;
	case 0xE1: /* SBC (Zpg,X) */
		C02_PC += 2;	*timing += 6;
		a = c02_operand_value(peek, C02_PC-1, C02_AM__ZP_X_);
		__sbc(a);
		break;
	case 0xF1: /* SBC (Zpg),Y */
		C02_PC += 2;	*timing += 5;
		a = c02_operand_value(peek, C02_PC-1, C02_AM__ZP__Y);
		__sbc(a);
		break;
	case 0xF2: /* SBC (Zpg) */
		C02_PC += 2;	*timing += 5;
		a = c02_operand_value(peek, C02_PC-1, C02_AM__ZP_);
		__sbc(a);
		break;
	case 0x38: /* SEC */
		C02_PC++;	*timing += 2;
		__set_p(C02_P_C);
		break;
	case 0xF8: /* SED */
		C02_PC++;	*timing += 2;
		__set_p(C02_P_D);
		break;
	case 0x78: /* SEI */
		C02_PC++;	*timing += 2;
		__set_p(C02_P_I);
		break;
	case 0x87: /* SMB0 Zpg */
	case 0x97: /* SMB1 Zpg */
	case 0xA7: /* SMB2 Zpg */
	case 0xB7: /* SMB3 Zpg */
	case 0xC7: /* SMB4 Zpg */
	case 0xD7: /* SMB5 Zpg */
	case 0xE7: /* SMB6 Zpg */
	case 0xF7: /* SMB7 Zpg */
		C02_PC += 2;	*timing += 5;
		a = c02_operand_address(peek, C02_PC-1, C02_AM_ZP);
		poke(a, peek(a) | (1 << ((peek(C02_PC-2) & 0x70) >> 4)));
		break;
	case 0x85: /* STA Zpg */
		C02_PC += 2;	*timing += 3;
		a = c02_operand_address(peek, C02_PC-1, C02_AM_ZP);
		poke(a, C02_A);
		break;
	case 0x95: /* STA Zpg,X */
		C02_PC += 2;	*timing += 4;
		a = c02_operand_address(peek, C02_PC-1, C02_AM_ZP_X);
		poke(a, C02_A);
		break;
	case 0x8D: /* STA Abs */
		C02_PC += 3;	*timing += 4;
		a = c02_operand_address(peek, C02_PC-2, C02_AM_ABS);
		poke(a, C02_A);
		break;
	case 0x9D: /* STA Abs,X */
		C02_PC += 3;	*timing += 5;
		a = c02_operand_address(peek, C02_PC-2, C02_AM_ABS_X);
		poke(a, C02_A);
		break;
	case 0x99: /* STA Abs,Y */
		C02_PC += 3;	*timing += 5;
		a = c02_operand_address(peek, C02_PC-2, C02_AM_ABS_Y);
		poke(a, C02_A);
		break;
	case 0x81: /* STA (Zpg,X) */
		C02_PC += 2;	*timing += 6;
		a = c02_operand_address(peek, C02_PC-1, C02_AM__ZP_X_);
		poke(a, C02_A);
		break;
	case 0x91: /* STA (Zpg),Y */
		C02_PC += 2;	*timing += 6;
		a = c02_operand_address(peek, C02_PC-1, C02_AM__ZP__Y);
		poke(a, C02_A);
		break;
	case 0x92: /* STA (Zpg) */
		C02_PC += 2;	*timing += 5;
		a = c02_operand_address(peek, C02_PC-1, C02_AM__ZP_);
		poke(a, C02_A);
		break;
	case 0xDB: /* STP */
		*timing += 3;
		/* does not advance the program counter, leaving the processor   */
		/* halted until an interrupt comes along to make it do something */
		break;
	case 0x86: /* STX Zpg */
		C02_PC += 2;	*timing += 3;
		a = c02_operand_address(peek, C02_PC-1, C02_AM_ZP);
		poke(a, C02_X);
		break;
	case 0x96: /* STX Zpg,Y */
		C02_PC += 2;	*timing += 4;
		a = c02_operand_address(peek, C02_PC-1, C02_AM_ZP_Y);
		poke(a, C02_X);
		break;
	case 0x8E: /* STX Abs */
		C02_PC += 3;	*timing += 4;
		a = c02_operand_address(peek, C02_PC-2, C02_AM_ABS);
		poke(a, C02_X);
		break;
	case 0x84: /* STY Zpg */
		C02_PC += 2;	*timing += 3;
		a = c02_operand_address(peek, C02_PC-1, C02_AM_ZP);
		poke(a, C02_Y);
		break;
	case 0x94: /* STY Zpg,X */
		C02_PC += 2;	*timing += 4;
		a = c02_operand_address(peek, C02_PC-1, C02_AM_ZP_X);
		poke(a, C02_Y);
		break;
	case 0x8C: /* STY Abs */
		C02_PC += 3;	*timing += 4;
		a = c02_operand_address(peek, C02_PC-2, C02_AM_ABS);
		poke(a, C02_Y);
		break;
	case 0x64: /* STZ Zpg */
		C02_PC += 2;	*timing += 3;
		a = c02_operand_address(peek, C02_PC-1, C02_AM_ZP);
		poke(a, 0);
		break;
	case 0x74: /* STZ Zpg,X */
		C02_PC += 2;	*timing += 4;
		a = c02_operand_address(peek, C02_PC-1, C02_AM_ZP_X);
		poke(a, 0);
		break;
	case 0x9C: /* STZ Abs */
		C02_PC += 3;	*timing += 4;
		a = c02_operand_address(peek, C02_PC-2, C02_AM_ABS);
		poke(a, 0);
		break;
	case 0x9E: /* STZ Abs,X */
		C02_PC += 3;	*timing += 5;
		a = c02_operand_address(peek, C02_PC-2, C02_AM_ABS_X);
		poke(a, 0);
		break;
	case 0xAA: /* TAX */
		C02_PC++;	*timing += 2;
		C02_X = C02_A;
		__set_p_for(C02_X);
		break;
	case 0xA8: /* TAY */
		C02_PC++;	*timing += 2;
		C02_Y = C02_A;
		__set_p_for(C02_Y);
		break;
	case 0x14: /* TRB Zpg */
		C02_PC += 2;	*timing += 5;
		a = c02_operand_address(peek, C02_PC-1, C02_AM_ZP);
		if (peek(a) & C02_A) {
			__set_p(C02_P_Z);
		} else {
			__clear_p(C02_P_Z);
		}
		poke(a, peek(a) & (~C02_A));
		break;
	case 0x1C: /* TRB Abs */
		C02_PC += 3;	*timing += 6;
		a = c02_operand_address(peek, C02_PC-2, C02_AM_ABS);
		if (peek(a) & C02_A) {
			__set_p(C02_P_Z);
		} else {
			__clear_p(C02_P_Z);
		}
		poke(a, peek(a) & (~C02_A));
		break;
	case 0x4: /* TSB Zpg */
		C02_PC += 2;	*timing += 5;
		a = c02_operand_address(peek, C02_PC-1, C02_AM_ZP);
		if (peek(a) & C02_A) {
			__set_p(C02_P_Z);
		} else {
			__clear_p(C02_P_Z);
		}
		poke(a, (peek(a) | C02_A));
		break;
	case 0x0C: /* TSB Abs */
		C02_PC += 3;	*timing += 6;
		a = c02_operand_address(peek, C02_PC-2, C02_AM_ABS);
		if (peek(a) & C02_A) {
			__set_p(C02_P_Z);
		} else {
			__clear_p(C02_P_Z);
		}
		poke(a, (peek(a) | C02_A));
		break;
	case 0xBA: /* TSX */
		C02_PC++;	*timing += 2;
		C02_X = C02_S;
		__set_p_for(C02_X);
		break;
	case 0x8A: /* TXA */
		C02_PC++;	*timing += 2;
		C02_A = C02_X;
		__set_p_for(C02_A);
		break;
	case 0x9A: /* TXS */
		C02_PC++;	*timing += 2;
		C02_S = C02_X;
		break;
	case 0x98: /* TYA */
		C02_PC++;	*timing += 2;
		C02_A = C02_Y;
		__set_p_for(C02_A);
		break;
	case 0xCB: /* WAI */
		*timing += 3;
		/* does not advance the program counter, leaving the processor   */
		/* halted until an interrupt comes along to make it do something */
		__set_p(C02_P_B);
		break;
	default: /* undocumented opcode; treat as NOP */
		C02_PC++;	*timing += 2;
		break;
	}
}
