**********************************************************
*                                                        *
*                    Message Server                      *
*                                                        *
*           Michael J. Mahon - April 20, 2004            *
*                 Revised May 21, 2008                   *
*                                                        *
*            Copyright (c) 2004, 2005, 2008              *
*                                                        *
*       Client Request Routines                          *
*         Put Message Request                            *
*         Get Message Request                            *
*                                                        *
*       Server Definitions                               *
*         Message Page Table                             *
*         Message Class Table                            *
*         Message Buffers (pages)                        *
*                                                        *
*       Server Routines (w/ Monitor)                     *
*         Put Message Server                             *
*         Get Message Server                             *
*                                                        *
*       Utility Routines                                 *
*         Look Up class in Message Table                 *
*                                                        *
**********************************************************

*--------------------------------------------------------*
*         Requester                     Server           *
*  ========================     =======================  *
*  PUTMSG REQ (class,leng) ====>                         *
*         (lock)             :                           *
*                         (<==== PUTMSG NAK if no space) *
*                                                        *
*                          <==== PUTMSG ACK              *
*         Data < 256 bytes ====>                         *
*                          <==== PUTMSG DACK             *
*--------------------------------------------------------*
*  GETMSG REQ (class)      ====>                         *
*         (lock)             :                           *
*                         (<==== GETMSG NAK if no msg)   *
*                                                        *
*                          <==== GETMSG ACK (class,leng) *
*                          <==== Data < 256 bytes        *
*  GETMSG DACK             ====>                         *
*--------------------------------------------------------*
         pag
         do    master.crate ; (If 'master or crate')
**********************************************************
*                                                        *
*                     P U T M R E Q                      *
*                                                        *
*           Michael J. Mahon - April 17, 2004            *
*                 Revised May 21, 2008                   *
*                                                        *
*               Copyright (c) 2004, 2008                 *
*                                                        *
*  Request message server (at 'sbuf+dest') to accept a   *
*  message of class 'sbuf+adr' and length 'sbuf+len'     *
*  at our local address 'locaddr'.                       *
*                                                        *
*  PUTMREQ will retry the request in case of timeout or  *
*  checksum errors up to 'maxreqrt' times.  If errors    *
*  persist, it returns with C set and A=0.               *
*                                                        *
*  If the server NAKs the request for lack of space,     *
*  PUTMREQ returns with C set and A=1.                   *
*                                                        *
*  PUTMREQ does the following steps:                     *
*     1. Make the PUTMSG request                         *
*     2. If server NAKs, return with C set and A=1.      *
*     3. Send 'sbuf+len'-byte message from 'locaddr'     *
*     4. Receive DATA ACK packet                         *
*     5. Retry in case of error up to 'maxreqrt' times   *
*     6. If unsuccessful, return with C set and A=0.     *
*                                                        *
**********************************************************

PUTMREQ  lda   #maxreqrt  ; Set request retry
         sta   reqretry   ;  counter.
:retry   lda   #r_PUTMSG  ; Send PUTMSG request.
         jsr   REQUEST
         bcs   :failed
         jsr   lasl=>al   ; Set up address/length
         jsr   SENDLONG   ;  and send message.
         jsr   RCVDACK    ; Receive DATA ACK packet.
         bcc   :done      ; -All OK.
         lda   #0         ; Not a NAK error
:failed  bne   :nakexit
:cksumer dec   reqretry   ; Dec request retry count
         bne   :retry     ; Try until OK or exhausted,
:nakexit sec              ;  then return with C set.
:done    rts
         pag
**********************************************************
*                                                        *
*                     G E T M R E Q                      *
*                                                        *
*            Michael J. Mahon - April 19, 2004           *
*                 Revised May 21, 2008                   *
*                                                        *
*               Copyright (c) 2004, 2008                 *
*                                                        *
*  Request message server (at 'sbuf+dst') to deliver     *
*  the first message of class 'sbuf+adr' to our address  *
*  'locaddr', actual length in 'rbuf+len' after ACK.     *
*                                                        *
*  GETMREQ will retry the request in case of timeout or  *
*  checksum errors up to 'maxreqrt' times.  If errors    *
*  persist, it returns with C set and A=0.               *
*                                                        *
*  If the server NAKs the request because the message    *
*  queue is empty, GETMREQ returns with C set and A=1.   *
*                                                        *
*  GETMREQ does the following steps:                     *
*     1. Make the GETMSG request                         *
*     2. If server NAKs, return with C set and A=1.      *
*     3. Receive 'rbuf+len'-byte message to 'locaddr'    *
*     4. If no error, send DATA ACK packet               *
*     5. Retry in case of error up to 'maxreqrt' times   *
*     6. If unsuccessful, return with C set and A=0.     *
*                                                        *
**********************************************************

GETMREQ  lda   #maxreqrt  ; Set request retry
         sta   reqretry   ;  counter.
:retry   lda   #r_GETMSG  ; Send GETMSG request.
         jsr   REQUEST
         bcs   :failed    ; Timeout or no msg.
         jsr   la=>a      ; Set up address
         mov16 rbuf+len   ;length ; and length.
         jsr   RCVLONG    ; Receive segmented message
         bcs   :err       ; Timeout or cksum err.
         delay 40         ; Kill some time...
         lda   #rm_DACK   ; -OK, send DATA ACK.
         jmp   SENDRSP    ;   and return w/ C clear.

:failed  bne   :nak       ; Server has no message.
:err     dec   reqretry   ; Cksum or timeout; dec count.
         bne   :retry     ; Try until OK or exhausted,
:nak     sec              ;  then return with C set.
         rts
         pag
         else  ('not master or crate')
************* Message Server Definitions **************

* Apple II ROM routines

BASCALC  equ   $FBC1      ; Compute BASL for line A ***
PRNTAX   equ   $F941      ; Print A,X in hex
COUT1    equ   $FDF0      ; Print char in A

BASL     equ   $28        ; Text line base address  ***
CH       equ   $24        ; Horizontal currsor position

* Page Zero variables

msgtbl   equ   $06        ; Pointer to msg table entry

* Non-page zero variables

msginit  db    0          ; Message Server initialized
                          ; (0 = not, >0 = initialized)
monbot   db    1          ; Monitor current bottom line
                          ; (0 = off, >0 = on)
monhead  asc   "==Message Server==",00  ; Monitor heading
monmax   equ   23         ; Monitor screen last line
mtblbot  da    mtbltop    ; Ptr to bottom of msg table
                          ; Grows down toward mtbllim

***********  Message Server memory layout  ***********

pagemap  equ   $C000-$100 ; Page map

mtbltop  equ   pagemap    ; Top of message table + 1
mtbllim  equ   mtbltop-$800 ; Bottom limit of table

pgmaphi  equ   mtbllim    ; Highest message page + 1
* (pgmaplo is the first page after end of code)

         dum   pgmaphi/256+pagemap ; Unused part of page map
monchlo  ds    monmax+1   ; Lo byte of last mark on line
monchhi  ds    monmax+1   ; Hi byte of last mark on line
         dend

************  Message Table Definition  *************

         dum   0          ; Message table layout
mt_clas  ds    2          ;   Class
mt_frst  ds    1          ;   First msg page
mt_last  ds    1          ;   Last msg page
                          ; ====================
mt_leng  ds    0          ; Length of table entry
         dend

************  Message Buffer Definition  *************

         dum   0          ; Message buffer layout
ms_next  ds    1          ;   Page # of next in list
ms_data  ds    255        ;   Data of message
         dend
         pag
**********************************************************
*                                                        *
*                     P U T M S R V                      *
*                                                        *
*           Michael J. Mahon - April 17, 2004            *
*                  Revised Oct 13, 2004                  *
*                                                        *
*                   Copyright (c) 2004                   *
*                                                        *
*  Service request to store a message of class 'rbuf+adr'*
*  and size 'rbuf+len' (1..255 bytes).                   *
*                                                        *
*  PUTMSRV will NAK the request if it does not have      *
*  space for the message.                                *
*                                                        *
*  PUTMSRV does the following steps:                     *
*     1. Lock the network since ACK may be delayed by    *
*        more than the minimum arbitration time (1 ms.)  *
*     2. Initialize message server if not initialized.   *
*     3. Look up message class in table, add if missing. *
*     4. If insufficient space, send NAK and return.     *
*     5. If sufficient space, send ACK                   *
*     6. Receive message data (1..255 bytes)             *
*     7. If received OK, send DATA ACK.                  *
*                                                        *
**********************************************************

PUTMSRV  sta   dsend+1    ; Lock net by sending ONE
         lda   msginit    ; Message Server initialized?
         bne   :ready     ; -Yes, go.
         inc   msginit    ; -No, mark initialized
         lda   #0         ;   and do it.
         tay              ; Free all entries
:clear   sta   pagemap,y  ;  in page map.
         iny
         bne   :clear
         lda   monbot     ; Monitor mode on?
         beq   :ready     ; -No, go.
         jsr   HOME       ; -Yes, clear text screen
         ldy   #0         ;   and print heading.
:prloop  lda   monhead,y
         beq   :ready     ; $00 terminates heading
         jsr   COUT1
         iny
         bne   :prloop    ; (always)

:ready   jsr   lookup     ; Find class in table
         beq   :getpage   ; -Found class.
         sec              ; -No, add new entry
         lda   mtblbot    ;   if possible.
         sbc   #mt_leng
         tay
         lda   mtblbot+1
         sbc   #0
         cmp   #>mtbllim  ; At lower limit?
         beq   :nak       ; -Yes, NAK the request.
         sty   mtblbot    ; -No, update 'mtblbot'
         sty   msgtbl     ;   and 'msgptr'.
         sta   mtblbot+1
         sta   msgtbl+1
         ldy   #mt_clas   ; -Space but not found,
         lda   rbuf+adr   ;   add new class to table.
         sta   (msgtbl),y
         iny
         lda   rbuf+adr+1
         sta   (msgtbl),y
         lda   #0
         ldy   #mt_frst   ; Init 'frst'
         sta   (msgtbl),y ;  and 'last' links
         ldy   #mt_last   ;   to nil.
         sta   (msgtbl),y
         lda   monbot     ; Is monitor enabled?
         beq   :getpage   ; -No.
         cmp   #monmax    ; -Yes.  Is monitor list full?
         bcs   :getpage   ; -Yes.  Don't add to list.
         inc   monbot     ; -No, add class to list.
         lda   monbot
         jsr   BASCALC    ; Compute base addr of line
         ldy   #0
         sty   CH         ; Cursor at first char.
         ldx   rbuf+adr
         lda   rbuf+adr+1
         jsr   PRNTAX     ; Print class in hex
         lda   #":"
         jsr   COUT1      ; Print ":"
         ldy   monbot
         lda   BASL+1     ; Save base address of
         sta   monchhi,y  ;  last char written.
         clc
         lda   BASL
         adc   #4
         sta   monchlo,y
:getpage ldy   #>pgmaplo  ; Find free page for message
:findfre lda   pagemap,y
         beq   :gotfree   ; Bingo!
         iny
         cpy   #>pgmaphi
         bcc   :findfre   ; Keep looking...
:nak     lda   #rm_NAK    ; No free space
         jmp   SENDRSP    ;  send NAK and return.

:gotfree lda   rbuf+len+1 ; Check message length:
         bne   :nak       ; Must be < 256 bytes
         lda   rbuf+len
         beq   :nak       ; Cannot be 0 bytes
         sta   length
         sty   locaddr+1  ; Page is msg buffer
         jsr   SENDACK    ; Send ACK.
         ldx   length     ; length = 1..255
         lda   #ms_data   ; Offset of data
         ldy   locaddr+1  ; Buffer page
         jsr   RCVPKT     ; Receive the message data.
         bcs   :nomon     ; Exit w/o alloc or DACK.
         ldx   #0         ; Link in the new msg
         stx   ptr        ; ptr -> new msg
         ldy   locaddr+1
         sty   ptr+1
         lda   length
         sta   pagemap,y  ; Allocate page (map=len>0)
         txa
         sta   (ptr,x)    ; new msg.next = nil
         ldy   #mt_last
         lda   (msgtbl),y ; Page of last msg
         beq   :frstmsg   ; nil means list empty...
         sta   ptr+1
         lda   locaddr+1  ; Page of new msg
         sta   (ptr,x)    ; last msg.next -> new
         sta   (msgtbl),y ; msgtbl.last -> new
         bne   :sendack   (Always)

:frstmsg lda   ptr+1      ; -> new
         sta   (msgtbl),y ; msgtbl.last -> new
         ldy   #mt_frst
         sta   (msgtbl),y ; msgtbl.frst -> new
:sendack lda   #rm_DACK   ; Send DATA ACK
         jsr   SENDRSP
         jsr   monlinx    ; Get monitor line in X
         bcs   :nomon     ; -Not visible.
         lda   monchhi,x  ; -Yes, set ptr
         sta   ptr+1
         inc   monchlo,x  ; Advance to next char
         lda   monchlo,x
         sta   ptr
         and   #$7F       ; First, mod 128
         sec
         sbc   #4         ; Subtract "xxxx:" offset
:again   cmp   #10        ;  and convert mod 10
         bcc   :done
         sbc   #10
         bcs   :again     ; (always)

:done    ora   #"0"       ; Convert to ASCII
         ldy   #0
         sta   (ptr),y    ;  and put it on screen.
:nomon   rts
         pag
**********************************************************
*                                                        *
*                     G E T M S R V                      *
*                                                        *
*           Michael J. Mahon - April 20, 2004            *
*                 Revised Jun 29, 2005                   *
*                                                        *
*                Copyright (c) 2004, 2005                *
*                                                        *
*  Service machine 'rbuf+frm's request for delivery of   *
*  a message of class 'rbuf+adr'.                        *
*                                                        *
*  GETMSRV does the following steps:                     *
*     1. Lock net, since ACK can be delayed by more than *
*        the minimum arbitration delay (1 ms.)           *
*     2. Look for message of class 'rbuf+adr' in table   *
*     3. If no class or no msg, send NAK and quit.       *
*     4. If msg found, send ACK with msg length.         *
*     5. Send message data (1..255 bytes)                *
*     6. Receive DATA ACK                                *
*     7. If no DATA ACK, keep message in table.          *
*     8. If DATA ACK good, deallocate message            *
*         and update monitor display if monitoring.      *
*                                                        *
**********************************************************

GETMSRV  sta   dsend+1    ; Lock net by sending ONE
         jsr   lookup     ; Look up class in msgtbl
         beq   :found     ; -Found it.
:nak     lda   #rm_NAK    ; -Not found
         jmp   SENDRSP    ;   NAK request and return.

:found   ldy   #mt_frst
         lda   (msgtbl),y ; Get page of first msg.
         beq   :nak       ; -No message.
         sta   locaddr+1  ; -Save start address
         tay
         lda   pagemap,y  ; Get actual length (1..255)
         sta   sbuf+len   ;  and put in to ACK
         lda   #0         ; (always <256 bytes)
         sta   sbuf+len+1
         jsr   SENDACK    ; Send ACK to requester.
         delay 140        ; Wait for requester...
         ldx   sbuf+len   ; Actual msg length (1..255)
         lda   #ms_data   ; Offset of data in msg
         ldy   locaddr+1  ; Page containing msg
         jsr   SENDPKT    ; Send msg to requester.
         jsr   RCVDACK    ; Receive DATA ACK
         bcs   :rts       ; -NG, don't free msg.
         lda   #0         ; -OK, pop the message
         sta   ptr        ;   from the table.
         lda   locaddr+1  ; Get msg page
         sta   ptr+1      ;  and set ptr
         ldy   #ms_next   ; Offset of 'next' link
         lda   (ptr),y    ; Get page of next msg
         ldy   #mt_frst   ; Offset of 'frst' link
         sta   (msgtbl),y ; Pop first msg off list
         tay              ; Is link nil?
         bne   :notlast   ; -No, not last msg.
         ldy   #mt_last   ; -Yes, set 'last' link
         sta   (msgtbl),y ;   to nil, too.
:notlast ldy   locaddr+1  ; Old msg page
         lda   #0         ; Free the page
         sta   pagemap,y  ;  in the page map.
         jsr   monlinx    ; -Yes, get line in X
         bcs   :rts       ; -Not on screen.
         lda   monchlo,x  ; -OK, get address of
         sta   ptr        ;   last mark on screen.
         lda   monchhi,x
         sta   ptr+1
         dec   monchlo,x  ; Point to previous char.
         lda   #$A0       ; Blank the last mark
         ldy   #0
         sta   (ptr),y
:rts     rts              ; Return...
         pag
**********************************************************
*                                                        *
*                     M O N L I N X                      *
*                                                        *
*           Michael J. Mahon - April 22, 2004            *
*                                                        *
*                   Copyright (c) 2004                   *
*                                                        *
*  Compute line on monitor screen corresponding to the   *
*  value of 'msgtbl' and put it in X.  If monitor mode   *
*  is off or line is not visible on screen, return witn  *
*  Carry set.                                            *
*                                                        *
**********************************************************

monlinx  lda   monbot     ; Monitor mode on?
         beq   :nomon     ; -No, return w/ carry.
         sec              ; -Yes, subtract msgtbl from
         lda   #<mtbltop  ;   top of table to get
         sbc   msgtbl     ;   (line-1)*4.
         tax
         lda   #>mtbltop
         sbc   msgtbl+1
         bne   :nomon     ; This class not on screen.
         txa
         lsr              ; Divide by 4
         lsr
         tax
         inx              ; Entry #1 maps to line 2
         cpx   #monmax+1  ; Beyond bottom line?
         bcs   :nomon     ; -Yes.
         rts              ; -No, return with X=line

:nomon   sec              ; Signal line not visible
         rts              ;  and return.
         pag
**********************************************************
*                                                        *
*                      L O O K U P                       *
*                                                        *
*           Michael J. Mahon - April 20, 2004            *
*                                                        *
*                   Copyright (c) 2004                   *
*                                                        *
*  Look up message class 'rbuf+adr' in message table.    *
*  Returns with "=" if found, or "not =" if not found.   *
*                                                        *
**********************************************************

lookup   lda   mtblbot    ; Start at bottom of table
         sta   msgtbl
         lda   mtblbot+1
         sta   msgtbl+1
         bne   :search    ; (always)

:loop    lda   rbuf+adr
         ldy   #mt_clas
         cmp   (msgtbl),y ; Compare class (lo)
         bne   :no
         lda   rbuf+adr+1
         iny
         cmp   (msgtbl),y ; Compare class (hi)
         bne   :no
         rts              ; Found. Return "="

:no      clc              ; Increment to next entry
         lda   msgtbl
         adc   #mt_leng
         sta   msgtbl
         lda   msgtbl+1
         adc   #0
         sta   msgtbl+1
:search  cmp   #>mtbltop  ; End of table?
         bne   :loop      ; -No, continue search.
         tay              ; -Yes, set "not ="
         rts              ;   and return.
         pag
         fin              ; ('master or crate')
