;--------------------------------------------------------------
; Driver for CompactFlash in True IDE mode on Apple // series*
; CFDRV.s  Version 1.0 - 12/19/2001
;
; Chris Schumann & Richard Dreher 2001
; cschumann@twp-llc.com & rich@dreher.net
;
; Currently this driver will only work on a 65C02 based machine
; That is an enhanced IIe, or IIe Platinum or a IIe/II+ that you have
; replaced the 6502 with a a 65C02 yourself.
;
; *NOTE: I was able to make GTEu G65SC02 P-1 work in a II+
;        but was unable to make a WDC65C02 work at all. So you may have to experiment.
;
;
; The number of 65C02 specific instruction are few, but very handy,
; mainly because of the difficulty in not having much RAM available for each
; slot PROM. So it is easier to just save temp data on the stack, and the 65C02
; makes pushing X and Y on the stack much easier.
;
;
; Tools used to make: CC65 6502 Cross C compiler & Assembler
; http://www.cc65.org/
;
; To build, use the following commands:
;
; ca65 -t apple2 --cpu 65C02 -l cfdrv.s
; ld65 -t apple2 cfdrv.o -o cfdrv.bin
;
; This will create an EPROM ready output file: sandrv.bin
; One extra step must be taken before burning the EPROM:
; The data located from file offset $700 to $7FF will have to be copied
; to offsets: $100-$1FF, $200-$2FF ... $600-$6FF. Which can be done
; easily on most EPROM programmers. This allows this interface card to
; be used in I/O slots 1 to 7.
;
; Altera CPLD design files:
; Appleideinterface.gdf
; Appleidelogic.tdf
;
;--------------------------------------------------------------------------
;
; Set DEBUG = 1 to enable debug output. This will output one line of text
; for each request ProDOS makes of this driver. This can be seen
; on another computer using serial terminal software.
; NOTE: It is assumes you have an Apple Super Serial card is slot 2,
; set up for 19200, N, 8, 1.
;
; An example debug output:
; A ProDOS write request:  Req:02 70 $DC00 B0077 W: $F2EF
; where Req stands for Request.
; 01 = read  (00 = status, 02 = write)
; 70 = slot number * 16 in this case slot 7.
; $DC00 ProDOS buffer address
; W: for Write request
; $F2EF is a simple checksum used to visually check data integrity.
;

DEBUG		=	0
				 
; ProDOS block interface definitions
pdCommandCode	=	$42
pdUnitNumber	=	$43
pdIOBuffer	    =	$44
pdIOBufferH	=	$45
pdBlockNumber	=	$46
pdBlockNumberH	=	$47

; ProDOS request Constants
PRODOS_STATUS  =   $00
PRODOS_READ    =   $01
PRODOS_WRITE   =   $02
PRODOS_FORMAT  =   $03

; Apple hardware definitions
mslot		    =   $7FC        ;Apple defined location for the last active slot

; Slot I/O space
IOBase 		=	$C080
ATADataHigh	=	IOBase+0
ATASetCSMask   =   IOBase+1    ;Special strobe bits to set and clear MASK bit
ATAClearCSMask =   IOBase+2    ;   used to disable CS0 output when doing a write cycle
                               ;   this is due to the fact that the 65x02 does a read cycle before
                               ;   every write cycle. THis would mess up some CompactFlash hardware
ATADevCtrl 	=	IOBase+6    ;when writing
ATAAltStatus   =   IOBase+6    ;when reading
ATADataLow 	=	IOBase+8
ATAError       =   IOBase+9
ATASectorCnt   =   IOBase+10
ATASector  	=	IOBase+11
ATACylinder	=	IOBase+12
ATACylinderH	=	IOBase+13
ATAHead		=	IOBase+14
ATACommand 	=	IOBase+15	; when writing
ATAStatus  	=	IOBase+15	; when reading

; Driver Zero-page usage
.IF DEBUG
CheckSumLow    =   $18         ;Checksum is used for debug only!!!
CheckSumHigh   =   $19         ;Checksum is used for debug only!!!
.ENDIF
;
; WARNING: zero page location $1A & $1B are used by Copy II plus and should not be used!
;
zpt3		    =	$1c
zpt4		    =	$1d
zpt5		    =	$1e

; I/O Scratchpad RAM. Must be accessed with the Y register containg the slot #
DriveInitDone	=   $478
SaveACC   		=	$4f8
sp2 	        =	$578        
sp3            =	$5f8        
sp4    		=	$678
sp5	    	=	$6f8
sp6		    =	$778
sp7		    =	$7f8



; Project definitions
BLOCKS	    	=	$f500	; Number of blocks on device. This is the Sandisk 32 (Marketing Meg) limit.
INITDONESIG	=	$A5     ; Init done signature

; ATA Commands
;
ATACRead	=	$20
ATACWrite	=	$30

;--------------------- Start of Peripheral Card ROM Space --------------------
; DO NOT USE ABSOLUTE ADDRESSES IN THIS SECTION
.ORG $C700  ;start here, but keep in mind it can move

	lda	#$20	;$20 is a signature for a drive to ProDOS
	ldx	#$00	;$00 "
	lda	#$03	;$03 "
	lda	#$3c	;$3c "

;----------- Load boot loader code at location $800 ------------------

; Read the PRODOS bootload off of the drive, read block 0 into location $800
   bit $CFFF               ;turn off anybody elses expansion ROM
   jsr	KnownRTS
	tsx	    	; get stack pointer
	lda	$100,x	; get return address high byte, should be $Cn, where n = slot#
	asl
	asl
	asl
	asl
	sta	pdUnitNumber
	lda	#$01
	sta	pdCommandCode
	ldx	#0
	stx	pdIOBuffer
	stx	pdBlockNumber   ;read boot block 0000
	stx	pdBlockNumberH
	lda	#$08
	sta	pdIOBufferH
	pha			        ; Push return address on stack: $801 ($800 is on stack)
	txa                  
	pha                 ; Fall into driver code, it will RTS to boot loader code

;------------------- Non-boot entry point for driver code -----------------
driver:
	lda	$cfff           ;turn off other ROMs that might be on

.IF DEBUG
;warning this debug code trashes the Acc register
; Set up serial port on the Apple Super Serial card. Always assume slot 2
	lda	#$1F
	sta	$c0ab	        ; control register
	lda	#$0b
	sta	$c0aa	        ; format
	lda	$c0a9	        ; clear status
.ENDIF

; Determine what Apple II slot we are in (1 thru 7)
; this must be calculated every time we enter the driver
;
   jsr	KnownRTS
	tsx	    	        ; get stack pointer
	lda	$100,x	        ; get return address high byte
	and	#$0f            ; mask off high bits
	sta	mslot           ; Apple defined location reserverd for last slot active
	tay		            ; Y now has $0s for accessing RAM
	asl
	asl
	asl
	asl
	tax		            ; get $s0 into X for indexing I/O, where s is the slot#

   lda ATAClearCSMask,x    ;reset MASK bit in PLD for normal CS0 signaling

.IF DEBUG
; Display request parameters
	lda	#'R'
	jsr	DSChar
	lda	#'e'
	jsr	DSChar
	lda	#'q'
	jsr	DSChar
	lda	#':'
	jsr	DSChar
	lda	pdCommandCode
	jsr	DSByte
	lda	#' '
	jsr	DSChar
	lda	pdUnitNumber
	jsr	DSByte
	lda	#' '
	jsr	DSChar
	lda	#'$'
	jsr	DSChar
	lda	pdIOBufferH
	jsr	DSByte
	lda	pdIOBuffer
	jsr	DSByte
	lda	#' '
	jsr	DSChar
	lda	#'B'
	jsr	DSChar
	lda	pdBlockNumberH
	jsr	DSByte
	lda	pdBlockNumber
	jsr	DSByte
	lda	#' '
	jsr	DSChar
.ENDIF

;  Check slot portion of requested unit number, should match slot number $s0 held in X reg
   txa
   eor pdUnitNumber
   and #$70
   beq SlotOK

; Request was for drive in different slot, return error
   lda	#$28
	sec
	rts
   
;
; Reset the drive once, the first time the driver is called
; ide_devctrl Bit 2 = Software Reset, Bit 1 = nIEN (enable assertion of INTRQ)
;
SlotOK:

	lda	DriveInitDone,Y
	cmp	#INITDONESIG
	beq	checkcommand

   lda #0
   sta ATADataHigh,x       ;Clear the high byte of the interface data latch
	lda	#$06			    ;Set reset bit, disable drive INTRQ
	sta	ATADevCtrl,x
	lda	#$02			    ;Clear reset bit, disable drive INTRQ
	sta	ATADevCtrl,x
   jsr	IDEWaitReady
   lda	#INITDONESIG
	sta	DriveInitDone,Y      ; Set the init done flag so init only happens once.

; process the command code and jump to appropiate routine. Note: a branch will
; not reach so a jmp is used.

checkcommand:

   lda	pdCommandCode
   cmp #PRODOS_READ          
   bne chk1
   jmp dRead

chk1:
	cmp #PRODOS_WRITE
   bne chk2
   jmp dWrite

chk2:
	cmp #PRODOS_STATUS
   bne chk3
   jmp dStatus

chk3:
; An invalid request has been sent to this driver, this should never happen
; but return an I/O error if it ever does
;
.IF DEBUG
;warning this debug code trashes the Acc register
   pha
	lda	#'C'
	jsr	DSChar
	lda	#'E'
	jsr	DSChar
   pla
	jsr	DSByte
   jsr DSCRLF
.ENDIF

   lda	#$27
	sec
	rts


	.RES  	$C7FC-*		    ;skip to $CsFC, where s is the slot#

; Data table for ProDOS drive scan
; $CsFC/FD = disk capacity
; $CsFE = status bits (BAP p7-14)
; $CsFF = LSB of block driver
	.word	BLOCKS
	.byte	$17
	.byte	<driver		; low-order offset to driver code

;----------------- End of Peripheral Card ROM Space ----------------------



;--------------------- Start of I/O expansion ROM Space --------------------
; Code here starts at $c800
; This code area is absolute and can use absolute addressing
.ORG   $C800           ;note: .ORG does not cause this code to be placed at c800
                       ; the .RES statement above offsets this code

KnownRTS:
	rts                 ;Used by the boot prom to determin which slot we are in

;---------------- Process PRODOS status request ------------------------
; return status of device (present? write protected?)
dStatus:
.IF DEBUG
;warning this debug code trashes the Acc register
	lda	#'S'
	jsr	DSChar
	lda	#'.'
	jsr	DSChar
	jsr	DSCRLF
.ENDIF


	ldx	#<BLOCKS
	ldy	#>BLOCKS
	lda	#0	; no errors
	clc		; no errors
	rts
;
;---------------- Process PRODOS Read request ------------------------
; Read a block from the device into memory
; Assumes X reg contains slot number in form: $s0 where s = slot 1 to 7

dRead:

.IF DEBUG
;warning this debug code trashes the Acc register
	lda	#'R'
	jsr	DSChar
	lda	#':'
	jsr	DSChar

   lda #0
   sta CheckSumLow
   sta CheckSumHigh
.ENDIF

; Copy IOBuffer pointer to zero page location
	lda	pdIOBuffer
	sta	zpt4
	lda	pdIOBufferH
	sta	zpt5

; Program the task file registers based on ProDOS address
	jsr	IDEWaitReady
	jsr	Block2LBA

   lda #0
   sta ATADataHigh,x

	lda	#ATACRead
	sta	ATACommand,x        ; Issue the read command to the drive
   jsr IDEWaitReady        ; Wait for BUSY flad to go away

   lda ATAStatus,x         ; Check for error response
   and #$09
   cmp #01                 ; if DRQ=0 and ERR=1 an error occured
   bne ReadCommandOK

.IF DEBUG
;warning this debug code trashes the Acc register
	lda	#'R'
	jsr	DSChar
	lda	#'E'
	jsr	DSChar
   lda ATAError,x
	jsr	DSByte
   jsr DSCRLF
.ENDIF
;
; The drive has returned an error code. Just return I/O error code to PRODOS
;
   lda	#$27
	sec
	rts
;
; Sector is ready to read
;
ReadCommandOK:
	ldy	#2
	sty	zpt3
	ldy	#0

drl:
   jsr	IDEWaitReady
   bcs Readmore

;-------------------------------
; Display "R:Short" 
.IF DEBUG
	lda	#' '
	jsr	DSChar
	lda	#'R'
	jsr	DSChar
	lda	#':'
	jsr	DSChar
	lda	#'S'
	jsr	DSChar
	lda	#'h'
	jsr	DSChar
	lda	#'o'
	jsr	DSChar
	lda	#'r'
	jsr	DSChar
	lda	#'t'
	jsr	DSChar
	lda	#':'
	jsr	DSChar
   lda zpt3
   jsr DSByte
   tya
   jsr DSByte
   jsr DSCRLF
.ENDIF       
;
; The Block was short, return I/O error code to PRODOS
;
   lda	#$27
	sec
	rts

Readmore:
	lda	ATADataLow,x
	sta	(zpt4),y
	iny

.IF DEBUG
   clc
   adc CheckSumLow
   sta CheckSumLow
.ENDIF

	lda	ATADataHigh,x
	sta	(zpt4),y

.IF DEBUG
   adc CheckSumHigh
   sta CheckSumHigh
.ENDIF

	iny
	bne	drl
	inc	zpt5
	dec	zpt3
	bne	drl

ReadEnd:
.IF DEBUG
	lda	#' '
	jsr	DSChar
	lda	#'$'
	jsr	DSChar

   lda CheckSumHigh
   jsr DSByte
   lda CheckSumLow
   jsr DSByte
	jsr	DSCRLF
.ENDIF

	lda	#0
	clc
	rts


;---------------- Process PRODOS Write request ------------------------
; Write a block from memory to the device
; Assumes X reg contains slot number in form: $s0 where s = slot 1 to 7
;
dWrite:

.IF DEBUG
;warning this debug code trashes the Acc register
	lda	#'W'
	jsr	DSChar
	lda	#':'
	jsr	DSChar

   lda #0
   sta CheckSumLow
   sta CheckSumHigh
.ENDIF

	lda	pdIOBuffer          ; Copy ProDOS request pointer to zero page
	sta	zpt4
	lda	pdIOBufferH
	sta	zpt5

   lda #0
   sta ATADataHigh,x       ;Clear the high byte of the 16 bit interface data latch

	jsr	IDEWaitReady
	jsr	Block2LBA           ;program IDE task file

; Write sector from RAM
   lda	#ATACWrite
	sta	ATACommand,x
   jsr IDEWaitReady

   lda ATAStatus,x         ; Check for error response from writing command
   and #$09
   cmp #01                 ; if DRQ=0 and ERR=1 an error occured
   bne WriteCommandOK

.IF DEBUG
;warning this debug code trashes the Acc register
	lda	#'W'
	jsr	DSChar
	lda	#'E'
	jsr	DSChar
   lda ATAError,x
	jsr	DSByte
   jsr DSCRLF
.ENDIF

; The drive has returned an error code. Just return I/O error code to PRODOS
;
   lda	#$27
	sec
   rts
;
; Sector is ready to write
;
WriteCommandOK:
	ldy	#2
	sty	zpt3
	ldy	#0


dwl:
 	jsr	IDEWaitReady
   bcs Writemore

;-------------------------------
; Display "W:Short" 
   .IF DEBUG
	lda	#' '
	jsr	DSChar
	lda	#'W'
	jsr	DSChar
	lda	#':'
	jsr	DSChar
	lda	#'S'
	jsr	DSChar
	lda	#'h'
	jsr	DSChar
	lda	#'o'
	jsr	DSChar
	lda	#'r'
	jsr	DSChar
	lda	#'t'
	jsr	DSChar
	lda	#':'
	jsr	DSChar
   lda zpt3
   jsr DSByte
   tya
   jsr DSByte
   jsr DSCRLF
.ENDIF
;
;
; The Block was short, return I/O error code to PRODOS
;
   lda	#$27
	sec
   rts

Writemore:
	lda	(zpt4),y
	pha

.IF DEBUG
   clc
   adc CheckSumLow
   sta CheckSumLow
.ENDIF

	iny
	lda	(zpt4),y
	sta	ATADataHigh,x

.IF DEBUG
   adc CheckSumHigh
   sta CheckSumHigh
.ENDIF

	pla
   sta ATASetCSMask,x      ;any access sets mask bit to block IDE -CS0 on I/O read to drive
	sta	ATADataLow,x        ; and remember that all write cycles are preceded by a read cycle on the 6502
   sta ATAClearCSMask,x    ;back to normal, allow CS0 on read cycles

	iny
	bne	dwl
	inc	zpt5
	dec	zpt3
	bne	dwl

.IF DEBUG
;----------------------------------
; display the Checksum
;
;warning this debug code trashes the Acc register
	lda	#' '
	jsr	DSChar
	lda	#'$'
	jsr	DSChar
   lda CheckSumHigh
   jsr DSByte
   lda CheckSumLow
   jsr DSByte
	jsr	DSCRLF
;----------------------------------
.ENDIF

	lda	#0
	clc
	rts

;------------------ Parse ProDOS Block address into LBA ----------------------
; This function translates the block number sent in the PRODOS request
; packet, into the LBA address. This is a straight 1 to 1 translation.
; This function just divides up the bits into the appropriate parts
; A ProDOS block and a IDE sector are both 512 bytes typically.
;
;Logical Block Mode, the Logical Block Address is interpreted as follows:
;LBA07-LBA00: Sector Number Register D7-D0.
;LBA15-LBA08: Cylinder Low Register D7-D0.
;LBA23-LBA16: Cylinder High Register D7-D0.
;LBA27-LBA24: Drive/Head Register bits HS3-HS0.

;Assume at most 2 32 meg drives. Actually 30.62 Meg drive, because a 32Meg
;compactFlash drives only suppors $F500 sectors, and not $FFFF sectors.
;So with a ~32meg card we will support 1 drive, with a ~64meg card or greater
;we will support 2 drives.
;
;Therefore the Drive bit from the ProDOS unit number will become
;bit 16 of the LBA (to select the second 32meg block)

Block2LBA:

   lda #$E0            ;1, (LBA), 1, (Drive), LBA 27-24, where LBA=1, Drive=0
	sta	ATAHead,x

;  Get the drive number from the requested Unit number
   lda pdUnitNumber
   and #$80
   lsr
   lsr
   lsr
   lsr
   lsr
   lsr
   lsr
	sta	ATACylinderH,x      ;store ProDOS drive# in LBA bit 16
	
   lda	pdBlockNumberH
	sta	ATACylinder,x       ;store ProDOS High block # into LBA 15-8
   
   lda	pdBlockNumber
	sta	ATASector,x
   lda #1
   sta	ATASectorCnt,x
	rts

;------------- Wait for drive to be ready, and get DRQ status ---------
; Wait for the BUSY flag to be clear. And return the DRQ bit in the carry
; Assumes X enters containing slot number in form: $s0 where s = slot 1 to 7
; 
IDEWaitReady:
	lda	ATAStatus,x
   bmi IDEWaitReady        ;Wait for BUSY (bit 7) to be zero
   asl                     ;shift DRQ status bit into the Carry bit
   asl
   asl
   asl
   asl
	rts

.IF DEBUG

DSChar:

;Send a char out the serial port assumed to be in slot 2
;Input: Character to send
;Regs changed at exit: P
;
	pha
dsc0:
   lda	$c0a9
	and	#%00010000      ;Transmit register empty?
	beq	dsc0	        ;No, wait
	pla		            ;get byte back
	sta	$c0a8	        ;send it
	rts

DSByte:
   phy
   pha                 ;Push A last, as it is pulled and re-pushed

   lsr
	lsr
	lsr
	lsr
	tay
	lda	DSBTable,y
	jsr	DSChar
   pla
   pha
   and	#$0F
	tay
	lda	DSBTable,y
	jsr	DSChar
   pla
   ply
	rts
DSBTable:
	.byte	"0123456789ABCDEF"

DSCRLF:
   pha
	lda	#$0D
	jsr	DSChar
	lda	#$0A
	jsr	DSChar
   pla
   rts
.ENDIF


