;------------------------------------------------------------------------------
; ProDOS/SmartPort driver for CompactFlash/IDE Interface for Apple II computers
; CFFA.s  Version 2.0 - 27-Apr-2008
;
; Firmware Contributors:        Email:
;  Chris Schumann                cschumann@twp-llc.com
;  Rich Dreher                   rich@dreher.net
;  Dave Lyons                    dlyons@lyons42.com
;
; This code can be built to require a 65C02 or 65816, or to work on a 6502.
;
; Tools used to build this driver: CA65: 6502 Cross Assembler
;     http://www.cc65.org/
;
; Here is the copyright from that tool using --version option
;    ca65 V2.11.0 - (C) Copyright 1998-2005 Ullrich von Bassewitz
;
; Example build instructions on an MSDOS based machine:
;------------------------------------------------------
; Assumes you have installed the CC65 package and set your path, etc.
;
; 1) c:\firmware> ca65 -t apple2 --cpu 65C02 -l CFFA.s
; 2) c:\firmware> ld65 -t apple2 CFFA.o -o CFFA.bin
;
; 3) If you have a newer, EEPROM-based CFFA card:
;
;    Use CFFA.UTIL to copy the firmware onto the card.  The Firmware Select
;    jumper allows two versions of the firmware on the card (for example, a
;    6502 version and a 65C02 version).
;
;    If you have made any changes to the "slot ROM" portion ($Cnxx), you
;    must update the slot ROM repeatedly, once for each Apple II slot where
;    you will ever use the CFFA card.  After updating the last slot, then
;    you can update the Aux ROM, just once.
;
; -- or --
;
; 3) If you have an older, EPROM-based CFFA:
;
;    Because the EPROM can hold up to 8 user selectable versions of firmware
;    you must load CFFA.bin into your EPROM programmer with one of the
;    following offsets:
;    (Note: this offset has nothing to do with the card slot offsets)
;
;  for driver #0: use offset $0000.  Selected with: A14= IN, A13= IN, A12= IN
;  for driver #1: use offset $1000.  Selected with: A14= IN, A13= IN, A12=OUT
;  for driver #2: use offset $2000.  Selected with: A14= IN, A13=OUT, A12= IN
;  for driver #3: use offset $3000.  Selected with: A14= IN, A13=OUT, A12=OUT
;  for driver #4: use offset $4000.  Selected with: A14=OUT, A13= IN, A12= IN
;  for driver #5: use offset $5000.  Selected with: A14=OUT, A13= IN, A12=OUT
;  for driver #6: use offset $6000.  Selected with: A14=OUT, A13=OUT, A12= IN
;  for driver #7: use offset $7000.  Selected with: A14=OUT, A13=OUT, A12=OUT
;                                    
;  where IN = jumper shorted, and OUT = jumper open
;  Driver #0 through #7 correspond to the user selectable
;  jumpers: J6 (A14, A13, A12) on the interface card.
;
; 4) Load as many firmware versions, up to 8, into the EPROM programmer as you
;    want. Remember that most programmers will, by default, clear all unused
;    memory to $FF when you load a binary file. You will need to disable that
;    feature after loading the first file.
;
; 5) Now you have an EPROM ready image to be programmed.
;    Using a standard 27C256 EPROM or similar, program your EPROM.
;
; Firmware Version History
;-------------------------
; Version 2.0 (February to April 2008, DAL)
;   - Started with version 1.20 source and merged in the 6502-specific changes.
;   - Merged in flow control changes from Dave Schmenk and made a further speed-up
;     in the Write loop.
;   - Added support for Dev0 and Dev1 (2 devices on the ATA bus)
;   - Added an interactive boot-time menu (press "M" during boot)
;   - Allows booting from any partition (on either device).  On an EEPROM-based
;     CFFA, you can save the setting into EEPROM, or just use it temporarily.
;   - $C800 space contains user-configurable parameters: number of partitions,
;     default startup partition, 3-byte block offset allowing the partitions
;     to reside anywhere in the first 8GB of the device.
;
; 6502-only Version 1.0:  (6502 version located at EPROM offset $4000, selection #4)
;   - Start of 6502 specific driver. Based on Version 1.2 spdrv.s driver
;   - Now works with 6502 CPU. Removed all 65C02 specific instruction from the ver 1.2 firmware
;   - NOTE: To work in 6502 based machine, logic CPLD version 1.3 or later is required.
;   - Changed error messages to upper case for Apple ]['s that don't support lower case.
;   - Set up slot specific behavior: Slot 5 has smartport enabled but will not autoboot.
;       All other slots have smartport disabled, but will autoboot in an Apple ][+ or ][e.
;   - Made 4th boot signature byte a parameter to boot code macro to ease slot specific setup.
;   - Changed GSOS_DRIVER signature to $10. This will prevent Dave Lyons' GS/OS driver from
;       loading should someone inadvertently use this firmware in a IIgs.
;   - Changed offsets for code loading with addition of .RES command
;       bin files can now be loaded in EPROM at normal 4K offsets:$0, $1000, $2000, etc,
;       instead of offsets $100, $1100, $2100, etc.
;
; Version 1.2
;   - Start of SmartPort driver. Based on Version 1.1 ProDOS driver
;
; Version 1.1
;   - dynamically calculate drive sizes in GetStatus function
;   - turn off interrupts in case boot code is called manually
;   - add cardID/firmware revision bytes: CFFA$xx
;   - added continuation of boot scan if device not present
;   - added continuation of boot scan if boot block code looks invalid
;   - reformatted this source file, removed tabs and added function headers
;
;
; PLD Logic Firmware Information
; -------------------------------
; This version of firmware assumes you are using PLD logic of at
; least CPLD logic version 1.3.  The source files for U6, the Altera PLD are:
;       Appleideinterface.gdf
;       Appleidelogic.tdf
;
; The programmer ready output file for the PLD logic is:
;       Appleideinterface.pof
;
; These files are not included with this code.
;
; Acknowledgements
; -----------------
; Thanks to:
;    Chris Schuman - for his extensive initial development work
;    David Lyons   - for technical information, many improvement ideas and
;                    SmartPort code development, and v2.0 work
;    Dave Schmenk  - for read/write speed-ups
;------------------------------------------------------------------------------

.segment "EXEHDR"            ; just to keep the linker happy
.segment "STARTUP"           ; just to keep the linker happy

.linecont +                  ; Allow line continuations

.ifpc02
   USE_65C02 = 1
.else
   USE_65C02 = 0
.endif

REQUIRES_EEPROM           = 1
IN_DEVELOPMENT            = 0          ; show a non-"v" version letter

FULL_MENU                 = REQUIRES_EEPROM

MENU_IDENTIFY_DEVS = (USE_65C02 & FULL_MENU)  ; 87 bytes (as of 15-Mar-2008)
MENU_DIGIT_INPUT          = 0          ; 12 bytes
SMARTPORT_RAW_BLOCK_MAGIC = 0          ; 30 bytes
SMARTPORT_STATUS_D0_D1    = 0          ; 14 bytes
MENU_STAR_MONITOR         = 0          ; 10 bytes
WRITE_PROTECT             = 0          ; 18 bytes (incomplete, don't use)

.if WRITE_PROTECT
   .error "WRITE_PROTECT isn't finished.  It needs to affect status calls."
.endif

;------------------------------------------------------------------------------
; Firmware Version Information
;------------------------------------------------------------------------------
FIRMWARE_VER    = $20        ; Version 2.0 (Version of this code)
SPDRIVERVERSION = $2000      ; Version of our SmartPort implementation (2.0)
;
; The GS/OS driver checks the GSOS_DRIVER byte to see if it is compatible with
; this firmware.
;
; Increment by one, when something changes that would require a change
; in the GS/OS driver.
;    $01 = ProDOS Driver supporting 2 drives.
;    $02 = SmartPort Driver supporting 4 drives
;    $03 = SmartPort Driver supporting 8 drives. NOTE: GS/OS driver won't load
;    $10 = 6502 version 1.0: ProDOS Driver supporting 2 drives. NOTE: GS/OS driver won't load
;    $11 = for CFFA firmware 2.0 (a future version of the GS/OS driver may load)
;
GSOS_DRIVER     = $11

;------------------------------------------------------------------------------
; ProDOS request Constants
PRODOS_STATUS   = $00
PRODOS_READ     = $01
PRODOS_WRITE    = $02
PRODOS_FORMAT   = $03

; ProDOS Return Codes
PRODOS_NO_ERROR      = $00    ; No error
PRODOS_BADCMD        = $01    ; Bad Command
PRODOS_IO_ERROR      = $27    ; I/O error
PRODOS_NO_DEVICE     = $28    ; No Device Connected
PRODOS_WRITE_PROTECT = $2B    ; Write Protected
PRODOS_BADBLOCK      = $2D    ; Invalid block number requested
PRODOS_OFFLINE       = $2F    ; Device off-line

; SmartPort return codes
BAD_PARM_COUNT       = $04
BAD_UNIT_NUMBER      = $11
INTERRUPT_NOT_SUPT   = $1F
BAD_STATUS_CODE      = $21

; ATA Commands Codes
ATACRead        = $20
ATACWrite       = $30
ATAIdentify     = $EC

;------------------------------------------------------------------------------
; Constants for Wait / ROMWait
;   Constant = (Delay[in uS]/2.5 + 2.09)^.5 - 2.7
;------------------------------------------------------------------------------
WAIT_100ms      = 197
WAIT_40ms       = 124
WAIT_100us      = 4

;------------------------------------------------------------------------------
; ASCII characters
;------------------------------------------------------------------------------
BELL            = $07
kLeftArrow      = $08
kDownArrow      = $0A
kUpArrow        = $0B
CR              = $0D
kRightArrow     = $15
kEscape         = $1B


;------------------------------------------------------------------------------
; Apple II I/O locations
;------------------------------------------------------------------------------
KEYBOARD        = $C000
KBDSTROBE       = $C010

;------------------------------------------------------------------------------
; Slot I/O definitions
;------------------------------------------------------------------------------
MSLOT           = $7F8          ; Apple-defined -- $Cn for slot owning Aux ROM

IOBase          = $C080         ; indexed with X = $n0
ATADataHigh     = IOBase+0

; SetCSMask / ClearCSMask - these strobe locations set and clear the MASK bit on
; the CPLD that disables the CS0 line to the CompactFlash during the CPU read
; cycles that occur before every CPU write cycle.  (These "false reads" would
; cause a device to double increment during a write.)
SetCSMask       = IOBase+1
ClearCSMask     = IOBase+2
WriteEEPROM     = IOBase+3      ; with CPLD v2.1 - force-allow write $C800..C81F
NoWriteEEPROM   = IOBase+4      ; with CPLD v2.1 - respect W/P jumper for $C800..C81F

ATADevCtrl      = IOBase+6      ; when writing
ATAAltStatus    = IOBase+6      ; when reading
ATADataLow      = IOBase+8
ATAError        = IOBase+9
ATASectorCnt    = IOBase+$0A
ATASector       = IOBase+$0B
ATACylinder     = IOBase+$0C
ATACylinderH    = IOBase+$0D
ATAHead         = IOBase+$0E
ATACommand      = IOBase+$0F    ; when writing
ATAStatus       = IOBase+$0F    ; when reading

;
; "Screen hole" Scratchpad RAM base addresses.
; Access using the Y register containg the slot number (lda DriveNumber,Y)
;
DriveResetDone  = $478          ; remember the device has been software reset
DriveNumber     = $4F8          ; 0 for the first 32MB partition, 1 for the second, ...
BootDevice      = $578          ; 0 = boot from Dev0, 1 = boot from Dev1
TempScreenHole  = $5F8
PartitionsDev0  = $678          ; number of 32MB partitions on Dev0
PartitionsDev1  = $6F8          ; number of 32MB partitions on Dev1
DrvMiscFlags    = $778          ; see kMiscXXX constants below
BootPartition   = $7F8          ; physical index of boot partition (0 for no remapping)

kMiscRaw        = $80           ; set this bit to read/write blocks on an entire device
kMiscDev1       = $40           ; set this bit in DrvMiscFlags,y to access Device 1

;------------------------------------------------------------------------------
; Zero-page locations
;------------------------------------------------------------------------------
CH              = $24
INVFLG          = $32
  kNormalText   = $FF
  kInverseText  = $3F
  kFlashingText = $7F

zpt1            = $EF           ; data at this location is saved/restored
bootError       = $FF

; Used only during the boot-time menu
menuSlot        = spSlot
menuSlot16      = spSlotX16
menuRow         = $82
menuMustSave    = $83
menuJumpVectorH = $84
menuValues      = $85           ; kMenuInteractiveRows (4) bytes
WriteOneByteToEEPROM = $90      ; 16 bytes

; Saved and restored, used during Status calls to hold block counts
blockCount      = $06           ; $06 to $08 (3 bytes)

; ProDOS block interface locations
pdCommandCode   = $42
pdUnitNumber    = $43
pdIOBuffer      = $44
pdIOBufferH     = $45
pdBlockNumber   = $46
pdBlockNumberH  = $47

; Arbitrary locations for Smartport data,
;  these locations are saved/restored before exit.
spCommandCode   = pdCommandCode

spSlot          = $40           ; menuSlot = spSlot
spSlotX16       = $41           ; menuSlot16 = spSlotX16

spParamList     = $48           ; 2 bytes
spCmdList       = $4A           ; 2 bytes
spCSCode        = $4C
spLastZP        = spCSCode

spZeroPgArea    = spSlot        ; $40
spZeroPgSize    = spLastZP-spZeroPgArea+1

StackBase       = $100

ResetVector     = $3F2

;------------------------------------------------------------------------------
; SmartPort constants
;------------------------------------------------------------------------------
; Use these in SmartPort ReadBlock/WriteBlock calls for raw LBA blocks.
.if SMARTPORT_RAW_BLOCK_MAGIC
MagicRawBlocksDev0Unit = 127
MagicRawBlocksDev1Unit = 126
.endif

SMARTPORT_STATUS     = 0
SMARTPORT_READ       = 1
SMARTPORT_WRITE      = 2
SMARTPORT_FORMAT     = 3
SMARTPORT_CONTROL    = 4
SMARTPORT_INIT       = 5
SMARTPORT_OPEN       = 6
SMARTPORT_CLOSE      = 7

SP_STATUS_IDENT_DEV0 = $49     ; 'I' for Identify
SP_STATUS_IDENT_DEV1 = $48     ; identify the other device

;------------------------------------------------------------------------------
; Apple II ROM entry points
;
; We can use these at boot time, but not while handling a call through
; our ProDOS or SmartPort entry points, as the ROM may not be available.
;------------------------------------------------------------------------------
AppleSoft       = $E000
ResumeBootScan  = $FABA     ; SLOOP
TABV            = $FB5B
SETPWRC         = $FB6F     ; fix up Reset vector
BELL1           = $FBDD     ; delay, then beep
KEYIN           = $FD0C
CROUT           = $FD8E
PRBYTE          = $FDDA
COUT            = $FDED
SetKBD          = $FE89
SetVID          = $FE93
INIT            = $FB2F
HOME            = $FC58
ROMWAIT         = $FCA8
SETNORM         = $FE84
MONITOR         = $FF69

;------------------------------------------------------------------------------
; Macros
;------------------------------------------------------------------------------
.macro BRA_OR_JMP Destination
   .IF USE_65C02
       bra Destination
   .ELSE
       jmp Destination
   .ENDIF
.endmacro

.macro INC_A_OR_ADC1
   .IF USE_65C02
      inc a
   .ELSE
      clc
      adc #1
   .ENDIF
.endmacro

.macro DEC_A_OR_SBC1
   .IF USE_65C02
      dec a
   .ELSE
      sec
      sbc #1
   .ENDIF
.endmacro

.macro STZ_OR_LDA_STA mem
   .IF USE_65C02
      stz mem
   .ELSE
      lda #0
      sta mem
   .ENDIF
.endmacro

.macro STZ_OR_STA_ATADataHigh_X
   .IF USE_65C02
      stz ATADataHigh,x           ; Clear high byte data latch
   .ELSE
      lda #0
      sta ATADataHigh,x           ; Clear high byte data latch
   .ENDIF
.endmacro

.macro PHY_OR_TYA_PHA
   .IF USE_65C02
      phy
   .ELSE
      tya
      pha                         ; save Y, it holds the $0n sratchpad RAM offset
   .ENDIF
.endmacro

.macro PLY_OR_PLA_TAY
   .IF USE_65C02
      ply
   .ELSE
      pla
      tay
   .ENDIF
.endmacro

.macro ASSERT condition, string
   .if condition
   .else
      .error string
   .endif
.endmacro

.macro WARN condition, string
   .if condition
   .else
      .warning string
   .endif
.endmacro

; String macro from <http://www.cc65.org/mailarchive/2003-05/2995.html>
.macro HiBitString str
   .repeat .strlen(str), i
      .byte .strat(str,i) | $80
   .endrepeat
.endmacro

; SentinelString sets bit 7 on the final character in the string.
.macro SentinelString str
   .repeat .strlen(str) - 1, i
      .byte .strat(str,i)
   .endrepeat
   .byte .strat(str, .strlen(str) - 1) | $80
.endmacro


;-------------------------------------------------------------------------
; BEGIN ROM CONTENT
;-------------------------------------------------------------------------

;-------------------------------------------------------------------------
; $C000
;
; The base address for this 4K of code is $C000.  The first page ($C0xx) is
; not available in the Apple II address space, so we'll just fill it with
; something you might want to see if you have a copy of the firmware on
; disk.
;-------------------------------------------------------------------------
   .ORG $C000
   .byte "CFFA Firmware",CR,CR,CR
   .byte "See <http://dreher.net/CFforAppleII/>."
   .byte CR,CR,CR,CR,CR,CR,CR,CR,CR,CR
   .byte "Version "
   .byte $30+(FIRMWARE_VER/16), ".", $30+(FIRMWARE_VER & $F)
   .byte " for "
.IF USE_65C02
   .byte "65C02"
.ELSE
   .byte "6502"
.ENDIF
   .byte " or later.",CR,CR
.IF REQUIRES_EEPROM
   .byte "Requires CFFA with EEPROM, not EPROM.",CR,CR
.ELSE
   .byte "Compatible with older EPROM-based CFFA cards.",CR,CR
.ENDIF
   .RES $C100-*, $FF            ; fill the rest of the $C0xx area (unused)

;------------------------------------------------------------------------------
; $C100
;
; Start of Peripheral Card ROM Space $Cn00 to $CnFF
;
; A macro (CnXX) is used here so that this code can be easily generated for
; each slot, with minor variations.
;
; This is done because the hardware does not overlay the C1xx address space at
; C2xx, C3xx, etc. automatically. Instead, the base address for the ROM is
; $C000, and a read from $Cnxx is mapped to $0nxx in our ROM.
;
; ROM offsets $0000 to $00FF are not used.  $0nxx is used for slot n.  From
; $0800 to $0FFE is the 2K expansion ROM.
;
; In an older EPROM-based CFFA card, the expansion ROM stops at $CEFF, and any
; access to $CFxx turns off the expansion ROM -- so $0F00 to $0FFF are not used.
;
; In a newer EEPROM-based CFFA card, the expansion ROM continues all the way to
; $CFFE, except that $CFEF and $CFFF both disable the expansion ROM.
;------------------------------------------------------------------------------
.macro CnXX SLOT, SIGNATURE_BYTE_4
   .local P8DriverEntry
   .local SmartPortEntry
   .local commonEntry
   .local Boot
   .local MenuEntry
   .local notMenu
   .local PrepareForJumpToAuxROM
   .local InvalidBootBlock
   .local Error
   .local ShowErrorMessage
   .local ShowErrorMsgY
   .local ErrorMessages
   .local msgCheckTheDevice
   .local msgNoBootBlock
   .local msgCouldNotBootPartition

   lda #$20                    ; $20 is a signature for a drive to ProDOS
   ldx #$00                    ; $00 "
   lda #$03                    ; $03 "
   ASSERT (SIGNATURE_BYTE_4 < $80), "SIGNATURE_BYTE_4 must be <$80 (BPL)"
   lda #SIGNATURE_BYTE_4       ; $00 for SmartPort, $3C to look like a Disk II
   bpl Boot                    ; NOTE: this must be a 2 byte instuction to keep the entry point offset correct

;------------------- Non-boot P8 driver entry point ---------------------
; ProDOS block device entry point (READ, WRITE, STATUS, FORMAT).
;-----------------------------------------------------------------------
   ASSERT (* = $C00A+$100*SLOT), "Should be at $Cn0A"
P8DriverEntry:                  ; At $Cn0A for best compatibility.
   clc
   bcc commonEntry
   ASSERT (* = $C00D+$100*SLOT), "Should be at $Cn0D"
SmartPortEntry:                 ; At $Cn0D: SmartPort entry must be 3 bytes later
   sec
commonEntry:
   jsr PrepareForJumpToAuxROM
   jmp P8_SmartPort_Handler

;-----------------------------------------------------------------------
; Boot
;
; Start up the Apple II from our device.  Typically we will get here
; because of a "boot scan" (the Apple II is searching for a bootable
; device), or because of a direct PR#n for our slot.
;
; If the user has pressed the "menu snag" key, we'll display an
; interactive settings menu.
;
; Otherwise, we'll try to read the boot block (block 0) from the
; configured boot partition and then jump to the boot code.
;-----------------------------------------------------------------------
Boot:
   jsr PrepareForJumpToAuxROM  ; also needed to access config params below
   stx pdUnitNumber            ; prepare to boot from "drive 1"

;
; Wait here (before the first CheckDevice) in case the CFFA RESET jumper
; is enabled, or a Delkin Devices CF card never becomes ready.
;
   ldy BootTimeDelayTenths
@wasteTime:
   lda #WAIT_100ms
   jsr ROMWAIT
   dey
   bne @wasteTime
;
; Pressing a certain key (normally "M") at boot time brings up
; the interactive menu.
;
   lda KEYBOARD
   and MenuSnagMask
   cmp MenuSnagKey
   bne notMenu

   .RES $C000+(SLOT*$100)+$30-*,$EA  ; fill with NOPs to $Cn30
;
; MenuEntry -- available from the monitor as Cn30G
;
; (Because this is a supported user entry point, we can't assume that
; we have already called PrepareForJumpToAuxROM above.)
;
MenuEntry:
   ASSERT MenuEntry=$C030+(SLOT*$100), "MenuEntry should be $Cn30"
   jsr PrepareForJumpToAuxROM
   jsr InteractiveMenu

notMenu:
   ldy #PRODOS_READ            ; Request: READ block
   sty pdCommandCode
   ASSERT PRODOS_READ=1, "PRODOS_READ must be 1"
   dey                         ; Y = 0
   sty pdBlockNumber           ; ProDOS block $0000 (the bootloader block)
   sty pdBlockNumberH
   lda #$08
   sta pdIOBufferH
   sty pdIOBuffer              ; Into Location $0800
; Read the block by calling a special AuxROM entry point.
; We could just use P8DriverEntry, but this allows future firmware
; changes to the boot process without changes to the $Cn ROM.
   jsr PrepareForJumpToAuxROM
   jsr BootROM_ProDOS_Call
   sta bootError               ; store on zero page for error display
   bcs Error
   ldy $800                    ; Check the first byte of boot loader code.
   dey                         ; If bootload code is there, this byte = $01
   bne InvalidBootBlock
   lda $801                    ; If second byte is a 0, it's invalid
   beq InvalidBootBlock        ;   (we'd JMP to a BRK)

   ldx pdUnitNumber            ; X should contain the unit number when jumping
                               ;  to the bootloader
   jmp $801                    ; Jump to the bootloader we just read.


; If any error occured, like drive not present, check to see if we are in a
; boot scan, if so re-enter scan routine, else drop to Applesoft, aborting boot.
InvalidBootBlock:
Error:
   lda $00
   bne @notScanning
   lda $01
   cmp MSLOT
   bne @notScanning
   jmp ResumeBootScan          ; Re-enter Monitor's Autoscan Routine
@notScanning:
;
; The boot code must have been called manually because we are not in a slot scan.
;
ShowErrorMessage:
   jsr PrepareToShowErrorMessage   ; clear screen and show banner

   ldy #msgCheckTheDevice-ErrorMessages
   ldx bootError
   bne @someError
   ldy #msgNoBootBlock-ErrorMessages
@someError:
   jsr ShowErrorMsgY

   txa
   beq @skipDevNumber

; show dev "0" or "1" after "Check device "
   lda DrvMiscFlags+SLOT
   ASSERT kMiscDev1=$40, "kMiscDev1 must be bit 6"
   asl a
   asl a
   lda #'0'+$80
   adc #0
   jsr COUT
@skipDevNumber:

   ldy #msgCouldNotBootPartition-ErrorMessages
   jsr ShowErrorMsgY

   jsr PrepareForJumpToAuxROM     ; set up X and Y for slot
   jmp FinishBootTimeErrorMsg

;-----------------------------------------------------------------------
; ShowErrorMsgY
;
; Display a message from the boot-ROM table (ErrorMessages).
;
; Input:  Y = offset from ErrorMessages.
;-----------------------------------------------------------------------
ShowErrorMsgY:
@loop:
   lda ErrorMessages,y
   beq @done
   jsr CaseSafeCOUT
   iny
   bne @loop
@done:
   rts

;-----------------------------------------------------------------------
; PrepareForJumpToAuxROM
;
; Output - MSLOT = $Cn, X = $n0, Y = $0n
;          A = 0
;          Carry - preserved
;
; Turns off any other card's Aux ROM.
; Resets the CFFA's CS0 mode.
;
; Verifies that the 2.0-or-later Aux ROM is present (in case the firmware
; has been incompletely upgraded, or incompletely downgraded).
;-----------------------------------------------------------------------
PrepareForJumpToAuxROM:
   sta ClearCSMask+SLOT*16     ; reset MASK bit in PLD for normal CS0 signaling

; We must set MSLOT to $Cn *before* touching $CFFF, in case an interrupt
; occurs.  The interrupt handler uses it to re-establish Aux ROM ownership.
   lda #$C0+SLOT
   sta MSLOT
   bit $cfff                   ; turn off other ROM that might be on.

   lda C820_Signature          ; must have $CF if the Aux ROM is present
   eor #$CF                    ; compare without disturbing the carry
   bne @AuxROMTrouble

   ldy #SLOT                   ; Y reg now has $0n for accessing scratchpad RAM
   ldx #SLOT*16                ; X reg now has $n0 for indexing I/O
   rts

; The 2.0-and-later AuxROM is not there.  Just BRK into the monitor.
@AuxROMTrouble:
   jsr SetVID                  ; undo a PR#n to avoid an infinite reboot loop
   brk

;-------------------------------------------------------------------------
; ErrorMessages table
;
; Each string ends with a 0 byte.
;-------------------------------------------------------------------------
ErrorMessages:
msgCheckTheDevice:
   .byte "Check device ",0
msgNoBootBlock:
   .byte "No boot block",0
msgCouldNotBootPartition:
   .byte ".",CR,CR,"Could not boot partition ",0

;-------------------------------------------------------------------------
; $CnF5 - $CnFF -- Boot ROM signature, version, and capability ID bytes
;
; $CnF5 to $CnFA were defined by CFFA, but should no longer be used.
;     Instead, see the similar bytes in the Aux ROM ($C8xx) area.
;
; $CnFB to $CnFF are defined by ProDOS and SmartPort.
;-------------------------------------------------------------------------
   .RES   $C000+(SLOT*$100)+$F5-*,$77  ; skip to $CnF5

   .byte $FF                   ; was GSOS_DRIVER value (see $C8xx version area)
   .byte "CFFA", $FF           ; $CnF6..CnFA: Card ID, old 1.x version #

; $CnFB: SmartPort status byte
   .byte $0                    ; Not Extended; not SCSI; not RAM card
                               ; Even if not supporting SmartPort, we need a
                               ; zero at $CnFB so Apple's RAMCard driver
                               ; doesn't mistake us for a "Slinky" memory
                               ; card.

; Data table for ProDOS drive scan
; $CnFC/FD = disk capacity, if zero use status command to determine
   .word $0000                 ; $CnFC-D: A zero here will cause ProDOS to
                               ; rely on the STATUS command to determine volume size

; $CnFE = status bits (BAP p7-14)
;  7 = medium is removable
;  6 = device is interruptable
;  5-4 = number of volumes (0..3 means 1..4)
;  3 = device supports Format call
;  2 = device can be written to
;  1 = device can be read from (must be 1)
;  0 = device status can be read (must be 1)
   .byte %00010111             ; $CnFE: support 2 ProDOS drives

; $CnFF = LSB of block driver
   .byte <P8DriverEntry        ; $CnFF: low-order offset to ProDOS entry point

.endmacro
;-------------------------------------------------------------------------
;=========================================================================


;-------------------------------------------------------------------------
; For $C1xx through $C7xx, expand the CnXX macro 7 times to generate the
; seven variants of the slot-ROM code.
;
; Parameters for following macro invocations:
;     CnXX SLOT, SIGNATURE_BYTE_4
;
; A signature byte of $3C is same as Disk ][, and turns SmartPort support off,
; and allows autoboot on ][+, ][e.
;
; A signature byte of $00 signals SmartPort support, allow 4 drives in slot 5,
; but disables autoboot on ][+, ][e.
;-------------------------------------------------------------------------
.IF USE_65C02
   CnXX 1, $00         ; Slot PROM code for slot 1
   CnXX 2, $00         ; Slot PROM code for slot 2
   CnXX 3, $00         ; Slot PROM code for slot 3
   CnXX 4, $00         ; Slot PROM code for slot 4
   CnXX 5, $00         ; Slot PROM code for slot 5
   CnXX 6, $00         ; Slot PROM code for slot 6
.listbytes unlimited   ; Show all of the code bytes for the 7th slot
   CnXX 7, $00         ; Slot PROM code for slot 7
.ELSE
   CnXX 1, $3C         ; Slot PROM code for slot 1. SmartPort disabled. Autoboot.
   CnXX 2, $3C         ; Slot PROM code for slot 2. SmartPort disabled. Autoboot.
   CnXX 3, $3C         ; Slot PROM code for slot 3. SmartPort disabled. Autoboot.
   CnXX 4, $3C         ; Slot PROM code for slot 4. SmartPort disabled. Autoboot.
   CnXX 5, $00         ; Slot PROM code for slot 5. SmartPort enabled. No Autoboot.
   CnXX 6, $3C         ; Slot PROM code for slot 6. SmartPort disabled. Autoboot.
.listbytes unlimited   ; Show all of the code bytes for the 7th slot
   CnXX 7, $3C         ; Slot PROM code for slot 7. SmartPort disabled. Autoboot.
.ENDIF

.listbytes      12              ; revert back to normal listing mode for the rest
;-------------------- End of Peripheral Card ROM Space ----------------------


;----------------- Start of I/O expansion ROM (AUX ROM) Space ---------------
; $C800
;
; This code area is absolute and can use absolute addressing.  But it can't
; easily call into the earlier code, because it might have been called from
; any slot number.
;---------------------------------------------------------------------------

;---------------------------------------------------------------------------
; User-configurable parameters ($C800..$C81F)
;
; If the CFFA card's CPLD is 2.1 or later, then these 32 bytes can be made
; writable (even when the write-protect jumper is installed) by touching
; location $C0x3 (WriteEEPROM,X) and put back to normal by touching $C0x4
; (NoWriteEEPROM,X).
;---------------------------------------------------------------------------
   ASSERT *=$C800, "User-config section must start at $C800"
Max32MBPartitionsDev0:
   .byte 4                  ; range 0 to 13
Max32MBPartitionsDev1:
   .byte 0                  ; range 0 to 13
DefaultBootDevice:
   .byte 0                  ; range 0 to 1
DefaultBootPartition:
   .byte 1                  ; range 1 to 13

FourReservedBytes:
   .byte 0, 0, 0, 0
WriteProtectBits:
   .byte 0                  ; $80 = protect Dev1, $40 = protect Dev0
MenuSnagMask:
   ASSERT MenuSnagMask=$C809, "Frozen $Cn ROM knows MenuSnagMask is $C809"
   .byte $DF                ; allow "m" or "M" to activate the boot-time menu
MenuSnagKey:
   ASSERT MenuSnagKey=$C80A, "Frozen $Cn ROM knows MenuSnagKey is $C80A"
   .byte 'M'+$80
BootTimeDelayTenths:
   ASSERT BootTimeDelayTenths=$C80B, "Frozen $Cn ROM knows BootTimeDelayTenths is $C80B"
   .byte 5                  ; number of tenths of a second to wait on boot before kbd check
BusResetSeconds:
   .byte 31                 ; number of seconds to wait for ready after bus reset
CheckDeviceTenths:
   .byte 100                ; number of tenths of a second to wait for device to be ready
ConfigOptionBits:
   .byte 0                  ; set high bit ($80) to skip bus reset
BlockOffsetDev0:
   .byte 0,0,0              ; range 0 to 2^24 blocks (8 GB)
BlockOffsetDev1:
   .byte 0,0,0              ; range 0 to 2^24 blocks (8 GB)

TenBytesAvailable:
   .byte 0,0,0,0,0,0,0,0,0,0

ConfigAreaVersion:
   ASSERT ConfigAreaVersion=$C81F, "ConfigAreaVersion must be at $C81F"
   .byte 1

   .RES $C820-*,0
;---------------------------------------------------------------------------
; Version & compatibility bytes
;---------------------------------------------------------------------------
C820_Signature:
   ASSERT C820_Signature=$C820, "Signature must be at $C820"
   .byte $CF,$FA
C822_VersionByte:
   ASSERT C822_VersionByte=$C822, "Version must be at $C822"
   .byte FIRMWARE_VER

; GS/OS driver compatibility byte. The GS/OS driver checks this byte to see
; if it is compatible with this version of firmware. This way, firmware
; changes that do not affect the GS/OS driver will not prevent the GS/OS
; driver from loading and running.
C823_GSOS_Compatibility:
   ASSERT C823_GSOS_Compatibility=$C823, "Must be at $C823"
   .byte GSOS_DRIVER

C824_FeatureBits:
   .byte ($80*(1-USE_65C02))+($40*REQUIRES_EEPROM)+ \
         ($20*WRITE_PROTECT)+($10*SMARTPORT_RAW_BLOCK_MAGIC)

; 2 bytes available here
   .byte 0
   .byte 0

C827_Release_Candidate_Rev:
   ASSERT C827_Release_Candidate_Rev=$C827, "Must be at $C827"
   .byte 7   ;another minor version number to distinguish release candidates

   .res $C828-*,0
;---------------------------------------------------------------------------
; JMP vectors.  These are for use by our $Cn00 ROM.
;
; These are new for firmware 2.0, and they should remain in place for every
; future firmware version.  The intent is for the $Cn00 ROM to not change
; in future versions, for ease of updating the firmware in the field.
;
; So, don't rearrange these lightly!  Add more at the end, if necessary.
;---------------------------------------------------------------------------
   ASSERT *=$C828, "Fatal error - JMP vectors must start at $C828"
P8_SmartPort_Handler:
   jmp P8_SmartPort_Handler_Impl
InteractiveMenu:
   jmp InteractiveMenu_Impl
PrepareToShowErrorMessage:
   jmp PrepareToShowErrorMessage_Impl
CaseSafeCOUT:
   jmp CaseSafeCOUT_Impl
FinishBootTimeErrorMsg:
   jmp FinishBootTimeErrorMsg_Impl
BootROM_ProDOS_Call:
   jmp BootROM_ProDOS_Call_Impl
ASSERT *=$C83A, "C83A"

;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
; FinishBootTimeErrorMsg
;
; Input: Y = SLOT, X = SLOT*16
;
; Called after the $Cn ROM displays an error message and "Could not boot
; partition ".  We should idenify the partition, beep, and jump to Applesoft.
;
; The error from reading block 0 is in bootError, in case we want to display it.
;------------------------------------------------------------------------------
FinishBootTimeErrorMsg_Impl:
   lda BootPartition,y
   INC_A_OR_ADC1
   jsr PrintSmallDecimal
; It would be nice to display " (Dev0)" or " (Dev1)" here, if the code would fit.
   jsr CROUT
   jsr BELL1

   ldx bootError
   beq @noErrorNumber
   ldy #MsgErrorNumber-Messages
   jsr Message
   txa
   jsr PRBYTE

@noErrorNumber:
   jmp AppleSoft

;------------------------------------------------------------------------------
; PrepareToShowErrorMessage_Impl
;
; This is separate from ClearScreenAndIdentifyCFFA in case we want the boot-
; time screen to act different (without changing the $Cn ROM).
;------------------------------------------------------------------------------
PrepareToShowErrorMessage_Impl:
;------------------------------------------------------------------------------
; ClearScreenAndIdentifyCFFA
;------------------------------------------------------------------------------
ClearScreenAndIdentifyCFFA:
   jsr INIT             ; text mode, full screen, page 1
   jsr SETNORM
   jsr HOME
   jsr SetVID
   jsr SetKBD

   jsr LineOfDashes
   ldy #MsgVersion-Messages
   jsr Message

   lda #34              ; room for "Slot N" at the right edge
   sta CH
   ldy #MsgSlot-Messages
   jsr Message
   lda MSLOT
   eor #$70             ; change $Cx to $Bx (slot digit)
   jsr COUT   

LineOfDashes:
   ldy #40
@loop:
   lda #'-'+$80
   jsr COUT
   dey
   bne @loop
   rts

;------------------------------------------------------------------------------
; SmartPort_Handler
;
; Called through $Cn0D entry point.
;
; Inputs:
;    Y = SLOT
;    X = SLOT*16 
;
; 1) Push a block of zero-page locations onto the stack, creating a work space.
; 2) Get the request parameters that follow the call to this driver
; 3) Using request parameters as pointers get request specific information
;    and set up the P8 driver request registers $42-$47.
; 4) Call P8Driver code
; 5) On return from P8Driver code, restore zero page work space, and return to
;    caller.
;
;------------------------------------------------------------------------------
SmartPort_Handler:
; save the zero page locations we're going to use
   ldy #spZeroPgSize-1
@save:
   lda spZeroPgArea,y
   pha
   dey
   bpl @save

   stx spSlotX16

; fetch the parameter list pointer from the code stream & adjust the returna address
   tsx
   lda $101+spZeroPgSize,x
   sta spParamList
   clc
   adc #3
   sta $101+spZeroPgSize,x
   lda $102+spZeroPgSize,x
   sta spParamList+1
   adc #0
   sta $102+spZeroPgSize,x

; set up Y and spSlot, and reload X from spSlotX16
   lda MSLOT
   and #$0f
   sta spSlot
   tay
   ldx spSlotX16

; reset the device if this is our first call
   jsr ResetBusIfFirstTime_ClearMiscFlags   ; needs X and Y

; get command code from parameter list
   ldy #1
   lda (spParamList),y
   sta spCommandCode
   iny
   lda (spParamList),y
   tax
   iny
   lda (spParamList),y
   sta spParamList+1
   stx spParamList

   lda #PRODOS_BADCMD          ; anticipate bad command error
   ldx spCommandCode
   cpx #$09+1                  ; command too large
   bcs @out

.IF USE_65C02
   lda (spParamList)           ; parameter count
.ELSE
   ldy #0
   lda (spParamList),y         ; parameter count
.ENDIF
   cmp RequiredParamCounts,x   ; command number still in X
   bne @pCountMismatch

   ldy #1
   lda (spParamList),y
   ldy spSlot
   sta DriveNumber,y           ; don't remap yet -- Status call needs unit 0

   txa                         ; X is still the command code
   asl a
   tax
   jsr JumpToSPCommand         ; Y is still spSlot

   bcs @out
   lda #0

@out:
   tax                         ; error code in X

; Restore zero page
   ldy #0
@restore:
   pla
   sta spZeroPgArea,y
   iny
   cpy #spZeroPgSize
   bcc @restore

   txa
   ldy #2                     ; high byte of # bytes transferred
                              ;   (always (1) undefined, or
                              ;   (2) #bytes transferred to host)
   ldx #0                     ; low byte of # bytes transferred
   cmp #1                     ; C=1 if error code nonzero
   rts

@pCountMismatch:
   lda #BAD_PARM_COUNT
   bne @out

JumpToSPCommand:
.IF USE_65C02
   RTS_ADJUST = 0
      jmp (spDispatch,x)
.ELSE
   RTS_ADJUST = 1
      lda spDispatch+1,x
      pha
      lda spDispatch,x
      pha
      rts
.ENDIF

RequiredParamCounts:
   .byte 3   ; 0 = status
   .byte 3   ; 1 = read
   .byte 3   ; 2 = write
   .byte 1   ; 3 = format
   .byte 3   ; 4 = control
   .byte 1   ; 5 = init
   .byte 1   ; 6 = open
   .byte 1   ; 7 = close
   .byte 4   ; 8 = read
   .byte 4   ; 9 = write

spDispatch:
   .word spStatus-RTS_ADJUST
   .word spReadBlock-RTS_ADJUST
   .word spWriteBlock-RTS_ADJUST
   .word spFormat-RTS_ADJUST
   .word spControl-RTS_ADJUST
   .word spInit-RTS_ADJUST
   .word spOpen-RTS_ADJUST
   .word spClose-RTS_ADJUST
   .word spReadChars-RTS_ADJUST
   .word spWriteChars-RTS_ADJUST

;------------------------------------------------------------------------------
; SmartPort STATUS call
;
; We support the standard calls, plus an "Identify" command
; for unit 0, which fills a buffer with the card's Identify
; data.
;------------------------------------------------------------------------------
spStatus:
   jsr SPSetUpControlOrStatus
   bcs statOut

   ldy spSlot
   lda DriveNumber,y
   bne StatusForOneUnit

; StatusCode = 0 && unit == 0: status for this entire SmartPort interface
   lda spCSCode
   beq Status00

; Check for device-specific status subcodes, only defined for the CFFA.
   cmp #SP_STATUS_IDENT_DEV0
   beq @ident
   cmp #SP_STATUS_IDENT_DEV1
   bne BadStatusCode
   lda #kMiscDev1
;  ora DrvMiscFlags,y     ; use ORA if we define new flag bits
   sta DrvMiscFlags,y
@ident:
   jmp spStatusIdentify

; Any other status code for unit 0 is an error.
BadStatusCode:
   lda #BAD_STATUS_CODE
   sec
   rts

Status00:
   ldy spSlot
   clc
   lda PartitionsDev0,y
   adc PartitionsDev1,y
   .IF USE_65C02
      sta (spCmdList)             ; byte +0 = number of drives
   .ELSE
      ldy #0
      sta (spCmdList),y           ; byte +0 = number of drives
   .ENDIF

   ldy #7
Stat00Loop:
   lda Stat00Data-1,y
   sta (spCmdList),y
   dey
   bne Stat00Loop
   clc
statOut:
   rts

Stat00Data:
; Interrupt flag = no interrupts caused by this interface
   .byte $40
; Vendor ID assigned to Rich Dreher by www.syndicomm.com 3/16/2002
   .word $CC00
   .word SPDRIVERVERSION        ; Our version number
   .byte $00                    ; Reserved byte
   .byte $00                    ; Reserved byte
   ASSERT (*-Stat00Data)=7, "There must be exactly 7 bytes of Stat00Data"

StatusForOneUnit:
   DEC_A_OR_SBC1
   jsr RemapAndStoreDriveNum
   bcs @error

   ldx spCSCode
   beq Status0or3
   dex
   beq StatusGetDCB
   dex
   dex
   beq Status0or3
   lda #BAD_STATUS_CODE
   sec
@error:
   rts
;
; We have no interesting data to return for the device control
; block, so return the shortest one allowed: a length byte of
; 1 followed by a data byte of 0.
;
StatusGetDCB:
   lda #1
.IF USE_65C02
   sta (spCmdList)             ; Returned length = 1
.ELSE
   ldy #0
   sta (spCmdList),y           ; Returned length = 1
.ENDIF
   tay
   lda #0
   sta (spCmdList),y           ; Returned data = $00
   clc
   rts

;
; Status code 0 and 3 start off the same; 3 returns extra data.
;
; 0: return device status (1 byte device status + 3-byte block count)
; 3: return Device Information Block
;
Status0or3:
   lda #$F8                    ; Block device, write, read, format, online,
                               ; not write-prot
.IF USE_65C02
   sta (spCmdList)
.ELSE
   ldy #0
   sta (spCmdList),y
.ENDIF
   ldx spSlotX16
   ldy spSlot
   jsr GetStatus               ; Returns block count in YX
   bcs statOut

   tya
   ldy #2
   sta (spCmdList),y           ; CmdList +2 = bits 8..15 of block count
   dey
   txa                         ; CmdList +1 = bits 0..7 of block count
   sta (spCmdList),y
   ldy #3
   lda #0
   sta (spCmdList),y           ; CmdList +3 = bits 16..23 of block count

   lda spCSCode
   beq statDone

; status code 3: return 21 more bytes of data
   ldy #4
stat3Loop:
   lda stat3Data-4,y
   sta (spCmdList),y
   iny
   cpy #21+4
   bcc stat3Loop

.if SMARTPORT_STATUS_D0_D1
; Doctor the "D0" to a "D1" in the status data, if we're on Device 1.
   ldy spSlot
   lda DrvMiscFlags,y
   asl a
   bpl statDone
   ldy #4+16
   lda #'1'
   sta (spCmdList),y
.endif

statDone:
   clc
   rts

stat3Data:
.if SMARTPORT_STATUS_D0_D1
   .byte 16,"COMPACT FLASH D0"  ; length byte + 16-byte ASCII, padded
.else
   .byte 16,"COMPACT FLASH   "  ; length byte + 16-byte ASCII, padded
.endif
   .byte $02                    ; device type = hard disk
   .byte $20                    ; subtype (no removable media, no extended,
                                ; no disk switched)
   .word SPDRIVERVERSION
   ASSERT (*-stat3Data)=21, "There must be exactly 21 bytes of stat3Data"

;------------------------------------------------------------------------------
; Identify (Status subcode)
;
; The status list is a 512-byte buffer that we will fill with
; the drive's IDE "Identify" data.  We post-process the data
; by byte-swapping the model name and serial number strings
; so they make sense.
;
; Input:  DrvMiscFlags,y has been set up
;         spSlot
;         spSlot16
;         spCmdList
;------------------------------------------------------------------------------
spStatusIdentify:
   ldy spSlot
   ldx spSlotX16

; Make sure that Partition 0 exists on the specified device.
   jsr IsDeviceKnownToExist
   bne @ok
   lda #PRODOS_NO_DEVICE
   sec
   rts
@ok:

   jsr WaitReady_SetATAHeadFromMiscDrvFlags

   jsr WaitReady_IssueIdentify_CheckErr
   bne iCommandOK

ReturnIOError:
   lda #PRODOS_IO_ERROR
   sec
   rts
;
; The "Identify" data is ready to read
;
pageCount = spCommandCode       ; re-use a zero-page location

iCommandOK:
   ldy #1
   sty pageCount
   dey                          ; Y = 0
@iLoop:
   lda ATADataLow,x
   sta (spCmdList),y
   iny
   lda ATADataHigh,x
   sta (spCmdList),y
   iny
   bne @iLoop
   inc spCmdList+1
   dec pageCount
   bpl @iLoop

; Swap ASCII text data of the Identify data to be readable.
; These are both on the first page of the data.

   dec spCmdList+1
   dec spCmdList+1             ; Point to beginning of buffer

   ldy #23*2                   ; Start at word 23 (firmware rev, model number)
   ldx #24                     ; 24 words
   jsr SwapBytes

   ldy #10*2                   ; Start at word 10 (serial number)
   ldx #10                     ; 10 words
; Fall into SwapBytes and return with carry clear.
; Swap X words of data starting at offset Y.
SwapBytes:
   lda (spCmdList),y
   pha                         ; Save the 1st byte
   iny
   lda (spCmdList),y           ; Get the 2nd byte
   dey
   sta (spCmdList),y           ; Store it in position 1
   iny
   pla                         ; Finally, retrieve the 1st byte
   sta (spCmdList),y           ; Put it in position 2
   iny
   dex
   bne SwapBytes
   clc
   rts

;------------------------------------------------------------------------------
; SmartPort CONTROL command
;
;------------------------------------------------------------------------------
spControl:
   jsr SPSetUpControlOrStatus
   bcs ctrlDone

   jsr SPValidateAndRemapUnitNumber
   bcs ctrlDone
; carry is clear
   ldx spCSCode
   beq ctlReset                 ; Control code 0 = Reset
   dex
   beq ctlSetDCB                ; Control code 1 = SetDCB
   dex
   beq ctlSetNewline            ; Control code 2 = SetNewline
   dex
   beq ctlServiceInterrupt      ; Control code 3 = ServiceInterrupt
   dex
   beq ctlEject                 ; Control code 4 = Eject

ctlSetNewline:
   lda #BAD_STATUS_CODE         ; Bad control code
   sec
ctlReset:
ctlSetDCB:
ctlEject:
ctrlDone:
   rts

ctlServiceInterrupt:
   lda #INTERRUPT_NOT_SUPT
   sec
   rts

;------------------------------------------------------------------------------
; SmartPort INIT command
;
; unit 0 = entire chain
;
;  SmartPort Technote #2 says you can't init an individual unit,
;  so return error if DriveNumber is nonzero.
;------------------------------------------------------------------------------
spInit:
   lda DriveNumber,y
   clc
   beq @initChain
   lda #BAD_UNIT_NUMBER
   sec
@initChain:
   rts

;------------------------------------------------------------------------------
; SmartPort: Open and Close are for character devices only
;------------------------------------------------------------------------------
spOpen:
spClose:
   lda #PRODOS_BADCMD
   sec
   rts

;------------------------------------------------------------------------------
; SmartPort: Read, Write
;
; We don't bother implementing Read and Write, although they are allowed
; for block devices.  We would only support 512-byte transfers anyway.
;------------------------------------------------------------------------------
spReadChars = ReturnIOError
spWriteChars = ReturnIOError
;   lda #PRODOS_IO_ERROR
;   sec
;   rts


;------------------------------------------------------------------------------
; SPSetUpControlOrStatus
;
; fetch from parameter block:
;    status/control list pointer (word)
;    status/control code (byte)
;------------------------------------------------------------------------------
SPSetUpControlOrStatus:
   ldy #2
   lda (spParamList),y
   sta spCmdList
   iny
   lda (spParamList),y
   sta spCmdList+1
   iny
   lda (spParamList),y
   sta spCSCode
   clc
   rts

;------------------------------------------------------------------------------
; SPSetUpReadWrite
;
; Input:
;    DriveNumber,y is already a copy of SmartPort unit number
;
; fetch from SP parameter list:
;   buffer pointer
;   block number
;
; Validate unit number:
;   127 = magic, allow any block number
;   1..N:  Validate block number: $0..FFFE
;
; Output:
;     DriveNumber,y: translated unit number in case unit was magic
;                    or BootPartition is set
;     DrvMiscFlags: bit 7 set if we should not use BlockOffset
;     SEC = error
;     CLC:
;        Y = slot
;        X = Slot*16
;------------------------------------------------------------------------------
SPSetUpReadWrite:
; copy pdIOBuffer from parameter list
   ldy #2
   lda (spParamList),y
   sta pdIOBuffer
   iny
   lda (spParamList),y
   sta pdIOBuffer+1

; copy pdBlockNumber from parameter list
   iny
   lda (spParamList),y
   sta pdBlockNumber
   iny
   lda (spParamList),y
   sta pdBlockNumber+1

; Validate the unit number and block number.
   ldy spSlot
   lda DriveNumber,y
.if SMARTPORT_RAW_BLOCK_MAGIC
   cmp #MagicRawBlocksDev0Unit
   beq magicUnitDev0
   cmp #MagicRawBlocksDev1Unit
   beq magicUnitDev1
.endif

   jsr SPValidateAndRemapUnitNumber
   bcs srwOut

   ldy #6
   lda (spParamList),y          ; Bits 16..23 of block number must be $00
   bne badBlockNum

   lda pdBlockNumber+1          ; Block $FFFF is invalid
   and pdBlockNumber
.IF USE_65C02
   inc a
.ELSE
   cmp #$FF
.ENDIF
   beq badBlockNum

loadXYandReturnNoError:
   ldy spSlot
   ldx spSlotX16
   clc
   rts

badBlockNum:
   lda #PRODOS_BADBLOCK         ; Bad block number
   sec
srwOut:
   rts
;
; For the "magic raw blocks" units, allow a full 3-byte block number.
;
.if SMARTPORT_RAW_BLOCK_MAGIC
magicUnitDev1:
   lda #kMiscRaw+kMiscDev1
   bne magicUnitCommon
magicUnitDev0:
   lda #kMiscRaw
magicUnitCommon:
;  ora DrvMiscFlags,y           ; use ORA if any other bits are defined
   sta DrvMiscFlags,y           ; Raw block access

   ldy #6
   lda (spParamList),y          ; Bits 16..23 of block number
   sta DriveNumber,y
   BRA_OR_JMP loadXYandReturnNoError
.endif

;------------------------------------------------------------------------------
; SmartPort FORMAT command
;
; We don't actually do anything beyond validating the unit number.
;------------------------------------------------------------------------------
spFormat:
;   jmp SPValidateAndRemapUnitNumber
;
; v v v   FALL INTO   v v v
;------------------------------------------------------------------------------
; SPValidateAndRemapUnitNumber
;
; Validate that DriveNumber is from 1 to N.
;
; Input: DriveNumber,Y
;
; Output: CLC = no error
;            DriveNumber,Y in range 0..N-1, remapped as needed
;            DrvMiscFlags,Y dev1 bit
;         SEC, A = error
;------------------------------------------------------------------------------
SPValidateAndRemapUnitNumber:
   ldy spSlot
   lda DriveNumber,y
   sec
   beq @badUnit
   DEC_A_OR_SBC1

   jsr RemapAndStoreDriveNum
@badUnit:
   lda #BAD_UNIT_NUMBER         ; only an error if SEC
   rts

;------------------------------------------------------------------------------
; P8_SmartPort_Handler_Impl
;
; Input: CLC if we should handle a ProDOS call
;        SEC if we should handle a SmartPort call
;        Y = SLOT
;        X = SLOT * 16
;
; Other inputs as set up by the caller (ZP for ProDOS, inline parms for
; SmartPort).
;------------------------------------------------------------------------------
P8_SmartPort_Handler_Impl:
   bcc ProDOS_Handler
   jmp SmartPort_Handler

;------------------------------------------------------------------------------
; BootROM_ProDOS_Call_Impl
;
; Same as BootROM_ProDOS_Call_Impl, but we have a chance to do extra stuff
; in the boot path, if needed, without changing the $Cn00 ROM.
;------------------------------------------------------------------------------
BootROM_ProDOS_Call_Impl:
; v v v   FALL INTO   v v v
;------------------------------------------------------------------------------
; ProDOS_Handler
;
; Input:
;     Y = SLOT
;     X = SLOT*16
;     zero page pdUnitNumber, etc.
;------------------------------------------------------------------------------
ProDOS_Handler:
   jsr ResetBusIfFirstTime_ClearMiscFlags  ; needs X and Y
;
; If the ProDOS unit number doesn't match our slot number, add 2 to
; the drive number.
;
; This actually happens: If we're in slot 5, we get calls for slot 2
; that want to access our 3rd and 4th partitions.
;
   txa                         ; A = $n0
   eor pdUnitNumber
   and #$70                    ; check only the slot-number bits
   beq @ourSlot
   lda #2                      ; Point to 3rd/4th partitions (2, 3)
@ourSlot:                      ; A is 0 or 2

   bit pdUnitNumber
   bpl @driveOne
   INC_A_OR_ADC1               ; A is 1 or 3
@driveOne:

   jsr RemapAndStoreDriveNum
   bcs @error

   lda pdCommandCode
   ASSERT PRODOS_STATUS=0, "PRODOS_STATUS must be 0"
;  cmp #PRODOS_STATUS         ; 0, so no "cmp" is needed
   beq GetStatus

   cmp #PRODOS_READ          ; [OPT] (1 byte) DEC A to check for READ (1)
   beq ReadBlock

@chk1:
   cmp #PRODOS_WRITE        ; [OPT] (1 byte) DEC A to check for WRITE (2)
   beq WriteBlock

@chk2:                      ; Invalid request number
   lda #PRODOS_IO_ERROR
@error:
   sec
   rts

;------------------------------------------------------------------------------
; GetStatus - Called by ProDOS and SmartPort to get device status and size
;
; Input:
;       DriveNumber,y
;       DrvMiscFlags,y
;       X = Slot*16 ($n0)
;       Y = slot
;
; Output:
;       A = ProDOS status return code
;       X = drive size LSB
;       Y = drive size MSB
;       Carry flag: 0 = Okay, 1 = Error
;------------------------------------------------------------------------------
GetStatus:
   lda DrvMiscFlags,y
   asl a
   ASSERT kMiscDev1=$40, "kMiscDev1 must be $40"
   bmi @dev1

   lda DriveNumber,y
   INC_A_OR_ADC1
   cmp PartitionsDev0,y
   beq @finalPartition
   bne @fullSize

@dev1:
   lda DriveNumber,y
   INC_A_OR_ADC1
   cmp PartitionsDev1,y
   beq @finalPartition

@fullSize:
   ldx #$FF                  ; X gets low byte of size
   ldy #$FF                  ; Y gets high byte of size
   lda #0
   clc
   rts

@finalPartition:
   lda blockCount+2
   pha
   lda blockCount+1
   pha
   lda blockCount
   pha

   jsr GetDeviceBlockCount   ; SEC if error (check below)
   ldy blockCount+1
   ldx blockCount

   pla
   sta blockCount
   pla
   sta blockCount+1
   pla
   sta blockCount+2

   bcs @ioErr
   lda #0
   rts

@ioErr:
   lda #PRODOS_IO_ERROR
readDone:
writeDone:
   rts

;------------------------------------------------------------------------------
; SmartPort WRITE BLOCK command
;------------------------------------------------------------------------------
spWriteBlock:
   jsr SPSetUpReadWrite
   bcs writeDone
;  jmp WriteBlock
;
; v v v   FALL INTO   v v v
;------------------------------------------------------------------------
; WriteBlock - Write a block in memory to device
;
; Input:
;       pd Command Block Data $42 - $47
;       X = slot * 16 ($n0)
;       Y = slot
;       DrvMiscFlags,y has been set up to specify Dev0 or Dev1
;       DriveNumber,y specifies a 32MB chunk (physical partition number)
;
; Output:
;       A = ProDOS write return code
;       Carry flag: 0 = Okay, 1 = Error
;
; ZeroPage Usage:
;       zpt1 is used but restored
;------------------------------------------------------------------------
WriteBlock:
   sec
   bcs ReadWriteCommon

;------------------------------------------------------------------------------
; SmartPort READ BLOCK command
;------------------------------------------------------------------------------
spReadBlock:
   jsr SPSetUpReadWrite
   bcs readDone
;  jmp ReadBlock
;
; v v v   FALL INTO   v v v
;------------------------------------------------------------------------------
; ReadBlock - Read a block from device into memory
;
; Input:
;       pd Command Block Data $42 - $47
;       X = requested slot number in form $n0 where n = slot 1 to 7
;       DrvMiscFlags,y has been set up to specify Dev0 or Dev1
;       DriveNumber,y specifies a 32MB chunk (physical partition number)
;
; Output:
;       A = ProDOS read return code
;       Carry flag: 0 = Okay, 1 = Error
;
; ZeroPage Usage:
;       zpt1 and pbBlockNumber - used but restored
;------------------------------------------------------------------------------
ReadBlock:
   clc
;
; CLC = Read
; SEC = Write
;
ReadWriteCommon:
   lda pdBlockNumberH           ; used and restored in WriteBlockCore
   pha
   lda pdBlockNumber
   pha

   lda pdIOBufferH
   pha
   lda zpt1
   pha

   php
   jsr Block2LBA               ; program IDE task file
   plp
   lda #ATACRead
   bcc @issueCmd               ; CLC=read, SEC=write
.if WRITE_PROTECT              ; 18 bytes
   lda DrvMiscFlags,y
   asl a
   bmi :+
   ora #$40                    ; assume that DrvMiscFlags bit 5 was 0
:  and WriteProtectBits        ; only bits 7 and 6 are used
   beq @writeEnabled
   lda #PRODOS_WRITE_PROTECT
   bne @outErrorA
@writeEnabled:
   sec
.endif
   lda #ATACWrite
@issueCmd:
   php
   jsr WaitReady_IssueCommand_CheckErr
   beq @PLP_ioError
   plp
   
   ldy #1
   sty zpt1
   dey                          ; Y = 0

   bcs @write

;--- ReadBlockCore:
@rLoop:                         ; rLoop is 27 cycles/word
   lda ATADataLow,x
   sta (pdIOBuffer),y
   iny
   lda ATADataHigh,x
   sta (pdIOBuffer),y
   iny
   bne @rLoop
   WARN (*/256)=(@rLoop/256), "Timing citical rLoop should not cross a page."

   inc pdIOBufferH
   dec zpt1
   bpl @rLoop
;--- end of ReadBlockCore
   BRA_OR_JMP @common

@write:
;--- WriteBlockCore:
@ioBufferShifted = pdBlockNumber   ; saved & restored
   lda pdIOBuffer
   clc
   adc #1                          ; [OPT] (carefully) for 65C02
   sta @ioBufferShifted
   lda pdIOBuffer+1
   adc #0
   sta @ioBufferShifted+1

   lda SetCSMask,x             ; block IDE -CS0 on I/O read to drive

; wLoop is 27 cycles/word (using a 2nd zero-page pointer, no PHA/PLA in loop)
@wLoop:
   lda (@ioBufferShifted),y
   sta ATADataHigh,x
   lda (pdIOBuffer),y
   sta ATADataLow,x
   iny
   iny
   bne @wLoop
   WARN (*/256)=(@wLoop/256), "Timing citical wLoop should not cross a page."

   inc pdIOBuffer+1
   inc @ioBufferShifted+1
   dec zpt1
   bpl @wLoop
; Set back to normal, allow CS0 assertions on read cycles
   lda ClearCSMask,x
;--- end of WriteBlockCore

@common:                        ; finish after a Read or Write
   jsr IDEWaitReadyCheckErr
   beq @ioError
   lda #0
@outErrorA:
   cmp #1                       ; SEC when A>0
   tax

   pla
   sta zpt1
   pla
   sta pdIOBufferH

   pla
   sta pdBlockNumber
   pla
   sta pdBlockNumberH

   txa
   rts

@PLP_ioError:
   plp
@ioError:
   lda #PRODOS_IO_ERROR
   bne @outErrorA

;------------------------------------------------------------------------------
; Block2LBA - Translates ProDOS block# into LBA and programs device's task file
;             registers.
;
; Input:
;       pd Command Block Data $42 - $47
;       DrvMiscFlags,y has been set up to specify Dev0 or Dev1
;       DriveNumber,y specifies a 32MB chunk (physical partition number)
;       X = slot * 16 ($n0)
;       Y = slot
;
; Ouput:
;       None
;
; ZeroPage Usage:
;       None
;
; CPU Registers changed:  A, P
;
; This function translates the block number sent in the ProDOS request
; packet, into an ATA Logical Block Address (LBA).
; The least significant 16 bits becomes the ProDOS block#.
; The most significant 16 becomes the ProDOS Drive #
;
; A ProDOS block and a ATA sector are both 512 bytes.
;
; Logical Block Mode, the Logical Block Address is interpreted as follows:
; LBA07-LBA00: Sector Number Register D7-D0.
; LBA15-LBA08: Cylinder Low Register D7-D0.
; LBA23-LBA16: Cylinder High Register D7-D0.
; LBA27-LBA24: Drive/Head Register bits HS3-HS0.
;------------------------------------------------------------------------------
Block2LBA:
   jsr WaitReady_SetATAHeadFromMiscDrvFlags

   lda #1
   sta ATASectorCnt,x
;
; Add BlockOffset to the ProDOS block number to offset the first drive block we
; use. This keeps the device's first BlockOffset blocks free, which usually
; includes a MBR at block 0.
;
   lda DrvMiscFlags,y          ; bit 7 = raw block access
   ASSERT kMiscRaw=$80, "kMiscRaw must be $80 so we can test it with BMI"
.if SMARTPORT_RAW_BLOCK_MAGIC
   bmi @rawBlocks
.endif
   ASSERT kMiscDev1=$40, "kMiscDev1 must be $40 for this test"
   asl a                       ; since A<$80, this clears the carry
   bmi @dev1

;  clc
   lda pdBlockNumber
   adc BlockOffsetDev0
   sta ATASector,x             ; store ProDOS Low block # into LBA 0-7

   lda pdBlockNumberH
   adc BlockOffsetDev0+1
   sta ATACylinder,x           ; store ProDOS High block # into LBA 15-8

   lda BlockOffsetDev0+2
@addDriveAndStoreCylinderH:
   adc DriveNumber,y
   sta ATACylinderH,x          ; store LBA bits 23-16
   rts

@dev1:
;  clc
   lda pdBlockNumber
   adc BlockOffsetDev1
   sta ATASector,x             ; store ProDOS Low block # into LBA 0-7

   lda pdBlockNumberH
   adc BlockOffsetDev1+1
   sta ATACylinder,x           ; store ProDOS High block # into LBA 15-8

   lda BlockOffsetDev1+2
   BRA_OR_JMP @addDriveAndStoreCylinderH

.if SMARTPORT_RAW_BLOCK_MAGIC
@rawBlocks:
   lda pdBlockNumber
   sta ATASector,x             ; store ProDOS Low block # into LBA 0-7
   lda pdBlockNumberH
   sta ATACylinder,x           ; store ProDOS High block # into LBA 15-8
   lda DriveNumber,y
   BRA_OR_JMP @storeCylinderH
.endif

;------------------------------------------------------------------------------
; ResetBusIfFirstTime_ClearMiscFlags
;
; Reset the bus (both devices) once, the first time the driver is called.
;
; ide_devctrl Bit 2 = Software Reset, Bit 1 = nIEN (enable assertion of INTRQ)
;
; Input:
;       X = slot * 16 ($n0)
;       Y = slot
;
; ZeroPage Usage:
;       None
;
; CPU Registers changed:  A, P
;------------------------------------------------------------------------------
INIT_DONE_SIG = $A5

ResetBusIfFirstTime_ClearMiscFlags:
   lda DriveResetDone,Y
   cmp #INIT_DONE_SIG
   beq noNeedToResetBus

ResetBusAlways_ClearMiscFlags:
   lda DefaultBootPartition
   DEC_A_OR_SBC1
   sta BootPartition,y
   lda DefaultBootDevice
   sta BootDevice,y

   lda Max32MBPartitionsDev0
   sta PartitionsDev0,y
   lda Max32MBPartitionsDev1
   sta PartitionsDev1,y
   ora PartitionsDev0,y
   beq @totalMaxPartitionsZero
;
; Reset the ATA bus (applies to both devices at once)
;
   lda ClearCSMask,x           ; reset MASK bit in PLD for normal CS0 signaling

   STZ_OR_STA_ATADataHigh_X

   lda ConfigOptionBits
   bmi @skipBusReset

   lda #$06                    ; SRST (software reset) = 1, Disable INTRQ=1
   sta ATADevCtrl,x
;
; Per ATA-6 sec 9.2, need to wait 5us minimum. Use a delay of 100us to
; cover accelerated Apples up to 20 MHz.
;
   lda #WAIT_100us
   jsr Wait

   lda #$02                    ; SRST=0, Disable INTRQ=1
   sta ATADevCtrl,x
;
; Per ATA-6 sec 9.2, need to wait 2ms minimum. Use a delay of 40ms to
; cover accelerated Apples up to 20 MHz.
;
   lda #WAIT_40ms
   jsr Wait

@skipBusReset:
   jsr DetermineDevPartitionCounts

; Set the init done flag so init only happens once.
@totalMaxPartitionsZero:
   lda #INIT_DONE_SIG
   sta DriveResetDone,Y

noNeedToResetBus:
   lda #0
   sta DrvMiscFlags,Y
   rts


;------------------------------------------------------------------------------
; DetermineDevPartitionCounts
;
; For each device that has a MaxPartitions > 0 and is actually present,
; find out how many partitions it contains (up to its configured max).
;
; Inputs:
;    X = slot * 16
;    Y = slot
;    PartitionsDev0,y = Max32MBPartitionsDev0
;    PartitionsDev1,y = Max32MBPartitionsDev1
;
; Outputs:
;    PartitionsDev0,y
;    PartitionsDev1,y
;------------------------------------------------------------------------------
DetermineDevPartitionCounts:
   lda ClearCSMask,x           ; reset MASK bit in PLD for normal CS0 signaling

   jsr WaitForBusReadyAfterReset
   bcc @busReady

   lda #0
   sta PartitionsDev0,y
   sta PartitionsDev1,y
   rts

@busReady:
   lda PartitionsDev0,y
   beq @skipDev0
   lda #$E0                    ; $E0 = [1, LBA=1, 1, Drive=0, LBA 27-24]
   jsr CheckOneDevice_GetPartitionCount
   cmp PartitionsDev0,y
   bcs @skipDev0                ; don't increase beyond the configured number
   sta PartitionsDev0,y
@skipDev0:

   lda PartitionsDev1,y
   beq @skipDev1
   lda #$F0                    ; $F0 = [1, LBA=1, 1, Drive=1, LBA 27-24]
   jsr CheckOneDevice_GetPartitionCount
   cmp PartitionsDev1,y
   bcs @skipDev1                ; don't increase beyond the configured number
   sta PartitionsDev1,y
@skipDev1:
   rts


;------------------------------------------------------------------------------
; WaitForBusReadyAfterReset
;
; Inputs:
;   X = slot * 16 ($n0)
;   Y = slot
;
; Output:
;   CLC if the ATA bus is available
;   SEC if the bus remained busy -- no devices present?
;------------------------------------------------------------------------------
WaitForBusReadyAfterReset:
   PHY_OR_TYA_PHA
   ldy BusResetSeconds
@check:
   lda ATAStatus,x
   bpl @ready

; Wait1Second
   lda #10
:  pha
   lda #WAIT_100ms
   jsr Wait
   pla
   DEC_A_OR_SBC1
   bne :-

   dey
   bne @check
   sec
   bcs @exit
@ready:
   clc
@exit:
   PLY_OR_PLA_TAY
   rts

;------------------------------------------------------------------------------
; CheckOneDevice_GetPartitionCount
;
; Check to see if a device is attached to the interface.
;
; Input:
;       X = slot * 16 ($n0)
;       Y = slot
;       A = value for ATAHead ($E0 for Dev0, $F0 for Dev1)
;
; Output:
;       A = number of partitions on device
;
;  Checks to see if the drive status register is readable and equal to $50
;  If so, return with the Carry clear, otherwise return with the carry set.
;  Waits up to 10sec on a standard 1 MHz Apple II for drive to become ready
;
;  If we determine (by timing out) that the specified device is not present,
;  we set ATAHead to the *other* device.
;------------------------------------------------------------------------------
CheckOneDevice_GetPartitionCount:
   sta ATAHead,x
   sta TempScreenHole,y
   PHY_OR_TYA_PHA
   ldy CheckDeviceTenths
:  lda ATAStatus,x
   and #%11010000
   cmp #%01010000              ; if BUSY=0 and RDY=1 and DSC=1
   beq @found
   lda #WAIT_100ms
   jsr Wait                    ; Wait 100ms for device to be ready
   dey
   bne :-
   PLY_OR_PLA_TAY
; We timed out - the specified device is not present, so select the other one.
   lda TempScreenHole,y
   eor #$10
   sta ATAHead,x

   lda #0
   rts

@found:
   PLY_OR_PLA_TAY
   lda ATAHead,x
   and #$10
   asl a
   asl a
   ASSERT kMiscDev1=$40, "kMiscDev1 must be bit 6 for this code"
   sta DrvMiscFlags,y

   lda blockCount+2
   pha
   lda blockCount+1
   pha
   lda blockCount
   pha

   jsr GetDeviceBlockCount
   lda #0
   bcs @exit

; Round *up* to the next whole number of 32MB partitions.
; Start with blockCount+2 (number of 32MB multiples), and
; add one if the two lower bytes are anything but $0000.
   lda blockCount
   ora blockCount+1
   cmp #1                     ; SEC if any bits set in blockCount, blockCount+1
   lda blockCount+2
   adc #0

@exit:
   sta TempScreenHole,y

   pla
   sta blockCount
   pla
   sta blockCount+1
   pla
   sta blockCount+2

   lda TempScreenHole,y
   rts

;------------------------------------------------------------------------------
; IDEWaitReady - Waits for BUSY flag to clear, and returns DRQ bit status
;
; Input:
;       X = slot*16 ($n0)
; Ouput:
;       BNE if DRQ is set
;       BEQ if DRQ is clear
;       (used to be: Carry flag = DRQ status bit)
;
; CPU Registers changed:  A, P
;------------------------------------------------------------------------------
IDEWaitReady:
:  lda ATAStatus,x
   bmi :-                       ; Wait for BUSY (bit 7) to be zero
   and #$08
   rts

;------------------------------------------------------------------------------
; WaitReady_IssueCommand_CheckErr
;
; Input:  A = ATA command
;         X = slot*16
;------------------------------------------------------------------------------
WaitReady_IssueIdentify_CheckErr:
   lda #ATAIdentify
WaitReady_IssueCommand_CheckErr:
   pha
:  lda ATAStatus,x
   bmi :-                       ; Wait for BUSY (bit 7) to be zero
   STZ_OR_STA_ATADataHigh_X
   pla
   sta ATACommand,x            ; Issue the read command to the drive
;  jmp IDEWaitReadyCheckErr
;
; v v v   FALL INTO  v v v
;------------------------------------------------------------------------------
; IDEWaitReadyCheckErr
;
; Input:
;    X = slot * 16
;
; Output:
;    BEQ error    (DRQ=0, ERR=1)
;    BNE noError
;------------------------------------------------------------------------------
IDEWaitReadyCheckErr:
:  lda ATAStatus,x
   bmi :-                      ; Wait for BUSY (bit 7) to be zero
   and #$09
.IF USE_65C02
   dec a                       ; if DRQ=0 and ERR=1 (A=$01) an error occured
.ELSE
   cmp #$01                    ; if DRQ=0 and ERR=1 an error occured
.ENDIF
   rts


;------------------------------------------------------------------------------
; WaitReady_SetATAHeadFromMiscDrvFlags
;
; Input:  X = slot*16 ($n0)
;         DrvMiscFlags,y has been set up
;
; Output: A = the value stored into the ATAHead register
;------------------------------------------------------------------------------
WaitReady_SetATAHeadFromMiscDrvFlags:
   jsr IDEWaitReady
   lda DrvMiscFlags,y
   ASSERT kMiscDev1=$40, "kMiscDev1 must be $40 for the following code"
   lsr a
   lsr a
   and #$10
   ora #$E0                    ; $E0 = [1, LBA=1, 1, Drive, LBA 27-24]
   sta ATAHead,x
   rts

;------------------------------------------------------------------------------
; GetDeviceBlockCount
;
; Inputs:
;    DrvMiscFlags,y has been set up
;    X = slot * 16
;    Y = slot
;
; Output:
;    CLC, blockCount (3 bytes) -- size from device minus BlockOffset
;         (Caller must save/restore these locations)
;    SEC if error
;------------------------------------------------------------------------------
GetDeviceBlockCount:
   jsr WaitReady_SetATAHeadFromMiscDrvFlags
   jsr WaitReady_IssueIdentify_CheckErr
   bne @ok
   lda #PRODOS_IO_ERROR
   sec
   rts

@ok:
   PHY_OR_TYA_PHA
   ldy #0                      ; zero loop counter
@tossWord:
   lda ATADataLow,x            ; Read words 0 thru 56 but throw them away
   iny
   cpy #57                     ; Number of the first word to keep
   bne @tossWord
   PLY_OR_PLA_TAY

   lda DrvMiscFlags,y
   ASSERT kMiscDev1=$40, "kMiscDev1 must be $40 for this test"
   asl a
   asl a
   PHY_OR_TYA_PHA
   ldy #0
   bcc @dev0
   ASSERT BlockOffsetDev1>BlockOffsetDev0, "BlockOffsetDev1 must come after Dev0"
   ldy #BlockOffsetDev1-BlockOffsetDev0
@dev0:

   sec
   lda ATADataLow,x            ; Read the current capacity in sectors (LBA)
   sbc BlockOffsetDev0,y
   sta blockCount

   lda ATADataHigh,x
   sbc BlockOffsetDev0+1,y
   sta blockCount+1

   lda ATADataLow,x
   sbc BlockOffsetDev0+2,y
   sta blockCount+2

   lda ATADataHigh,x
   sbc #0
   beq @lessThan8GB

   lda #$ff
   sta blockCount
   sta blockCount+1
   sta blockCount+2

@lessThan8GB:
   PLY_OR_PLA_TAY

@tossRemaining:
   jsr IDEWaitReady            ; read the rest of the words, until command ends
   beq @done
   lda ATADataLow,x
   BRA_OR_JMP @tossRemaining
@done:
   clc
   rts


;------------------------------------------------------------------------------
; IsDeviceKnownToExist
;
; Input:
;    DrvMiscFlags,y is set up
;    Y = slot
; Output:
;    BNE = device exists
;    BEQ = device does not exist
;------------------------------------------------------------------------------
IsDeviceKnownToExist:
   lda DrvMiscFlags,y
   ASSERT kMiscDev1=$40, "kMiscDev1 must be bit 6 for this ASL"
   asl a
   bmi @dev1

   lda PartitionsDev0,y
   rts

@dev1:
   lda PartitionsDev1,y
   rts


;------------------------------------------------------------------------------
;  Wait - Copy of Apple's wait routine. Can't use ROM based routine in case
;         ROM is not active when we need it.
;
; Input:
;       A = desired delay time, where Delay(us) = .5(5A^2 + 27A + 26)
;       or more usefully: A = (Delay[in uS]/2.5 + 2.09)^.5 - 2.7
;
; CPU Registers changed:  A, P
;------------------------------------------------------------------------------
Wait:
   sec
@Wait2:
   pha
@Wait3:
   sbc #1
   bne @Wait3
   pla
   sbc #1
   bne @Wait2
   rts

;------------------------------------------------------------------------------
; RemapAndStoreDriveNum
;
; Input:  A = logical drive number (0 = first partition, 1 = second, ...)
;         Y = slot
;         DrvMiscFlags,Y does not have the kMiscDev1 bit set yet
;         BootDevice,Y indicates which device's partitions come first
;         BootPartition,Y indicates the boot partition (0=first)
;
; Output: SEC = error (no such partition)
;            A = error code
;         CLC = no error
;            DriveNumber,Y = A = physical drive number
;            DrvMiscFlags,Y --> set kMiscDev1 if Dev1
;         X and Y are preserved
;
; Bring the desired partition to #0 by swapping it with the first one
;    (for example, if you boot from partition #3 then eight 32MB chunks
;    on the device appear will in the order: 3, 1, 2, 0, 4, 5, 6, 7).
;
; If you have two devices, all the partitions on the boot device come
; before all the partitions on the non-boot device.
;
; Algorithm:
;
;   If BootDevice == Dev0:
;      ; Dev0 comes first
;      swap 0 with BootPartition
;      If P >= NumPartitionsDev0:
;         MiscFlags = Dev1
;         P -= NumPartitionsDev0
;         If P >= NumPartitionsDev1, error.
;   Else
;      ; Dev1 comes first
;      swap 0 with BootPartition
;      If P >= NumPartitionsDev1:
;         P -= NumPartitionsDev1
;         if P >= NumPartitionsDev0, error.
;      else
;         MiscFlags = Dev1
;
;------------------------------------------------------------------------------
RemapAndStoreDriveNum:
   pha
   lda BootDevice,y
   bne @dev1BeforeDev0

; Dev0 partitions come first (the normal order)
;
; (1)  Swap 0 with BootPartition
;
; (2)  if P >= PartitionsDev0:
;           MiscFlags = Dev1
;           P -= PartitionsDev0
;           if P >= PartitionsDev1, error.
   pla
   beq @swapToBootPartition
   cmp BootPartition,y
   bne @haveAdjustedPartitionIndex
@physicalZero:
   lda #0      ; map the boot partition's "normal" location to physical 0
   beq @haveAdjustedPartitionIndex
@swapToBootPartition:
   lda BootPartition,y     ; map 0 to physical location of boot partition
@haveAdjustedPartitionIndex:
   sta DriveNumber,y
   sec
   sbc PartitionsDev0,y    ; SBC affects Carry just like CMP
   bcc @noError
   sta DriveNumber,y
   cmp PartitionsDev1,y
   bcs @tooLarge
   bcc @onDev1

; Dev1 partitions come first
;
; (1) Swap 0 with BootPartition
;
; (2) if P >= PartitionsDev1:
;        P -= PartitionsDev1
;        if P >= PartitionsDev0, error.
;     else
;        MiscFlags = Dev1
@dev1BeforeDev0:
   pla
   beq @swapToBootPartitionDev1
   cmp BootPartition,y
   bne @haveAdjustedPartitionIndexDev1
@physicalZeroDev1:
   lda #0
   beq @haveAdjustedPartitionIndexDev1
@swapToBootPartitionDev1:
   lda BootPartition,y
@haveAdjustedPartitionIndexDev1:
   sta DriveNumber,y
   sec
   sbc PartitionsDev1,y     ; SBC affects Carry just like CMP
   bcc @onDev1
   sta DriveNumber,y
   cmp PartitionsDev0,y
   bcc @noError
@tooLarge:                  ; carry is already set
   lda #PRODOS_NO_DEVICE
;  sec
   rts

@onDev1:                    ; carry is already clear
   lda #kMiscDev1
;  ora DrvMiscFlags,y
   sta DrvMiscFlags,y
@noError:
;  clc
   lda DriveNumber,y
   rts

;==============================================================================
;==============================================================================

.IF FULL_MENU=0
;------------------------------------------------------------------------------
; InteractiveMenu for 6502
;
; INPUT:  X = slot*16, Y = slot
;
; RTS to caller to continue booting.
;------------------------------------------------------------------------------
InteractiveMenu_Impl:
   rts
.ELSE
;------------------------------------------------------------------------------
; InteractiveMenu_Impl
;
; INPUT:  X = slot*16, Y = slot
;
; RTS to caller to continue booting.
;------------------------------------------------------------------------------
kMenuInteractiveRows = 4
kMenuTotalRows = 5

kRowPartitionsDev0 = 0
kRowPartitionsDev1 = 1
kRowBootDevice = 2
kRowBootPartition = 3
kRowBootSaveQuit = 4

   ASSERT (Max32MBPartitionsDev0-Max32MBPartitionsDev0)=kRowPartitionsDev0, "oops"
   ASSERT (Max32MBPartitionsDev1-Max32MBPartitionsDev0)=kRowPartitionsDev1, "oops"
   ASSERT (DefaultBootDevice-Max32MBPartitionsDev0)=kRowBootDevice, "oops"
   ASSERT (DefaultBootPartition-Max32MBPartitionsDev0)=kRowBootPartition, "oops"

kMenuTop = 10
kMenuBootSaveQuitLine = 21

kMenuAlignment = 20

menuVTAB:
   .byte kMenuTop
   .byte kMenuTop+1
   .byte kMenuTop+3
   .byte kMenuTop+4
   .byte kMenuBootSaveQuitLine
menuHTAB:
   .byte kMenuAlignment-15-2
   .byte kMenuAlignment-4-2
   .byte kMenuAlignment-8-2
   .byte kMenuAlignment-9-2
   .byte 8
menuRowText:
   .byte MsgMenuRow0-Messages
   .byte MsgMenuRow1-Messages
   .byte MsgMenuRow2-Messages
   .byte MsgMenuRow3-Messages
   .byte MsgBootSaveQuit-Messages
menuMinimumValues:
   .byte 0, 0, 0, 1
menuMaximumValuesPlusOne:
   .byte 13+1, 13+1, 1+1, 13+1


InteractiveMenu_Impl:
   stx menuSlot16
   sty menuSlot
; Intentionally change $01 so that monitor SLOOP won't continue the boot scan.
   sty $01                     ; store anything *other* than $Cn

; Reset goes to BASIC
.if USE_65C02
   stz ResetVector
.else
   lda #<AppleSoft
   sta ResetVector
.endif
   lda #AppleSoft/256
   sta ResetVector+1
   jsr SETPWRC

   jsr ClearScreenAndIdentifyCFFA

; Call ResetBus to set up default screen-hole parameters.
   ldx menuSlot16
   ldy menuSlot
   jsr ResetBusAlways_ClearMiscFlags   ; needs X and Y

   sta KBDSTROBE

.if MENU_IDENTIFY_DEVS
   clc                          ; ask for dev0
   jsr MenuIdentifyDevice

   lda Max32MBPartitionsDev1
   beq @skipDev1
   sec                          ; ask for dev1
   jsr MenuIdentifyDevice
@skipDev1:
.endif

   ldy menuSlot
   lda BootDevice,Y
   sta menuValues+kRowBootDevice
   lda BootPartition,Y
   INC_A_OR_ADC1
   sta menuValues+kRowBootPartition

   ldx #kRowPartitionsDev1
   bne ResetDownFromX

; Pressing Esc in the menu comes here.
; Reset the menu values to the saved defaults.
ResetMenuValues:
   ldx #kMenuInteractiveRows-1
ResetDownFromX:
:  lda Max32MBPartitionsDev0,x
   sta menuValues,x
   dex
   bpl :-

   lda #kRowBootPartition
   sta menuRow

RedisplayValues:
   STZ_OR_LDA_STA menuMustSave
   ldx #kMenuTotalRows-1
:  lda menuVTAB,x
   jsr TABV
   lda menuHTAB,x
   sta CH
   lda menuRowText,x
   tay
   jsr Message
   cpx #kRowBootSaveQuit
   beq @noValueForThisRow
   ldy #MsgColonSpace-Messages
   jsr Message
   lda menuValues,x
   cmp Max32MBPartitionsDev0,x
   beq @valueUnchanged
   cpx #kRowPartitionsDev1+1
   bcs @saveIsOptional
   inc menuMustSave
@saveIsOptional:
   ldy #kInverseText
   sty INVFLG
@valueUnchanged:
   jsr PrintSmallDecimal
   ldy #kNormalText
   sty INVFLG
   lda #' '+$80
   jsr COUT
@noValueForThisRow:
   dex
   bpl :-

; Keep the screen holes up to date with the latest settings.
   ldy menuSlot
   lda menuValues+kRowBootDevice
   sta BootDevice,Y
   lda menuValues+kRowBootPartition
   DEC_A_OR_SBC1
   sta BootPartition,Y

MenuKeyFreshRow:
   ldx menuRow
   lda menuVTAB,x
   jsr TABV
   lda #kMenuAlignment+2
   sta CH
   bne MenuKey

BeepMenuKey:
   jsr BELL1
;
; Get a key
;
MenuKey:
   jsr KEYIN
   cmp #kEscape+$80
   beq ResetMenuValues

   cmp #kLeftArrow+$80
   bne notLeft
   lda #-2
AdjustMenuValueByAPlusOne:
;  sec
   adc menuValues,x
; X = menuRow, A = new value
SetValueFromA:
   cmp menuMinimumValues,x
   bcc BeepMenuKey
   cmp menuMaximumValuesPlusOne,x
   bcs BeepMenuKey
   sta menuValues,x
   bcc RedisplayValues

notLeft:
.if MENU_DIGIT_INPUT
   cmp #'0'+$80
   bcc notDigit
   cmp #'9'+1+$80
   bcs notDigit
   and #$0F
   bpl SetValueFromA
notDigit:
.endif

   cmp #kRightArrow+$80
   bne @notRight
   lda #0
   beq AdjustMenuValueByAPlusOne

@notRight:
.if MENU_STAR_MONITOR
   cmp #'*'+$80
   bne @notMonitor
   jsr HOME
   jmp MONITOR
@notMonitor:
.endif

   cmp #CR+$80
   beq @down
   cmp #kDownArrow+$80
   bne @notDown
@down:
   txa
   INC_A_OR_ADC1
   cmp #kMenuInteractiveRows
   bcc @row
   lda #0
@row:
   sta menuRow
@ok:
@notQuit:
   jmp MenuKeyFreshRow

@notDown:
   cmp #kUpArrow+$80
   bne @notUp
   dec menuRow
   bpl @ok
   lda #kMenuInteractiveRows-1
   bne @row

@notUp:
   and #$DF    ; fold lowercase letters to uppercase

   cmp #'B'+$80
   bne @notBoot

@MenuBoot:
   lda menuMustSave
   bne BeepMenuKey
   jsr HOME
   ASSERT menuJumpVectorH = menuMustSave+1, "sorry"
   lda MSLOT
   sta menuJumpVectorH
   jmp (menuMustSave)

@notBoot:
   cmp #'S'+$80
   beq MenuSave

   cmp #'Q'+$80
   bne @notQuit
   jsr HOME
   jmp AppleSoft

;
; Save the menuValues settings to the EEPROM.
;
; Only write the changed values.
;
; If the max # of partitions changes for either device,
; force a device reset on the next boot.
;
MenuSave:
   ldx #Image_WOB_End-Image_WriteOneByteToEEPROM-1
:  lda Image_WriteOneByteToEEPROM,x
   sta WriteOneByteToEEPROM,x
   dex
   bpl :-

   ldx menuSlot16
   sta WriteEEPROM,x

   ldx #kMenuInteractiveRows-1
@nextByte:
   lda menuValues,x
   cmp Max32MBPartitionsDev0,x
   beq @noNeedToWrite
   cpx #kRowPartitionsDev1+1
   bcs @noNeedToReinit
   ldy menuSlot
   sta DriveResetDone,y   ; force device reset later
@noNeedToReinit:
   jsr WriteOneByteToEEPROM
   beq @checkWriteProt
@noNeedToWrite:
   dex
   bpl @nextByte

   ldx menuSlot16
   sta NoWriteEEPROM,x
   bne @toMenuKey

@checkWriteProt:
   jsr BELL1
@toMenuKey:
   jmp RedisplayValues
.endif

.if MENU_IDENTIFY_DEVS
;------------------------------------------------------------------------------
; MenuIdentifyDevice
;
; Input: CLC for Dev0, SEC for Dev1
;------------------------------------------------------------------------------
menuIdentifyData       = $0C00

MenuIdentifyDevice:
   php
   ldy #MsgDev-Messages
   jsr Message
   plp
   php
   lda #0
   rol a
   jsr PrintSmallDecimal
   ldy #MsgColonSpace-Messages
   jsr Message

   ASSERT menuSlot=spSlot, "oops"
   ASSERT menuSlot16=spSlotX16, "oops"

   plp
   lda #0               ; for RORs below as well as the STA
   sta spCmdList
   ror a
   ror a
   ldy menuSlot
   sta DrvMiscFlags,y

   lda #menuIdentifyData/256
   sta spCmdList+1

   jsr spStatusIdentify
;  jmp PrintIdentifyData
;
; v v v   FALL INTO   v v v
;------------------------------------------------------------------------------
; PrintIdentifyData
;
; Display a device's Identify string (the model, not the serial number).
;
; Input:  SEC if there was an error reading the Identify string
;         CLC if menuIdentifyData has been filled in
;------------------------------------------------------------------------------
PrintIdentifyData:
   bcs @error
; show model string (40 chars)
   ldy #27*2
@loop:
   lda menuIdentifyData,y
   ora #$80
   cmp #' '+$80
   bcc @skip
   jsr CaseSafeCOUT
@skip:
   iny
   cpy #(23*2)+(24*2)
   bcc @loop
   jmp CROUT

@error:
   ldy #MsgQuestionReturn-Messages
.endif
;  jmp Message
; v v v   FALL INTO   v v v
;------------------------------------------------------------------------------
; Message
;
; Input:  Y = offset from Messages table
;
; Prints a message through CaseSafeCOUT.
;------------------------------------------------------------------------------
Message:
@loop:
   lda Messages,y
   pha
   jsr CaseSafeCOUT
   iny
   pla
   bpl @loop
   rts

;------------------------------------------------------------------------------
; PrintSmallDecimal
;
; Input:  A = 0..19
;
; Prints 1 or 2 digits, or a "?" if the value is too large.
;------------------------------------------------------------------------------
MAX_DECIMAL_TO_PRINT = 19

PrintSmallDecimal:
   cmp #10
   bcc PrintDigit
   cmp #MAX_DECIMAL_TO_PRINT+1
   bcc :+
   lda #'?'
   bne CaseSafeCOUT_Impl

:  sbc #10-1      ; CLC subtract an extra 1
   pha
   lda #'1'+$80
   jsr COUT
   pla
PrintDigit:
   ora #'0'+$80
;   jmp COUT
; v v v   FALL INTO   v v v
;------------------------------------------------------------------------------
; CaseSafeCOUT
;
; Print a character, but first change it to uppercase if necessary.
;
; Check a ROM ID byte to see if we are on an Apple IIe or later, where lowercase
; is available.   $FBB3 = $06 on Apple IIe and later.
;------------------------------------------------------------------------------
CaseSafeCOUT_Impl:
   ora #$80
.IF USE_65C02
.ELSE
   cmp #'a'+$80
   bcc @notLC
   pha
   lda $FBB3
   eor #6
   cmp #1         ; SEC if $FBB3 was not $06
   pla
   bcc @lowercaseAvailable
   and #$DF
@lowercaseAvailable:
@notLC:
.ENDIF
   jmp COUT

;------------------------------------------------------------------------------
; Messages (up to 256 bytes)
;
; It's OK to use lowercase here, because the code uppercases characters at
; runtime if needed.
;------------------------------------------------------------------------------
Messages:
MsgVersion:
   .byte "CFFA "
.IF USE_65C02
.ELSE
   .byte "6502 "
.ENDIF
.if IN_DEVELOPMENT
   .byte "d"    ; indicate development versions by changing this letter
.else
   .byte "v"
.endif
   .byte $30+(FIRMWARE_VER/16), ".", $30+(FIRMWARE_VER & $F)+$80   ; "X.Y"

MsgSlot:
   SentinelString "Slot "
MsgColonSpace:
   SentinelString ": "
.if MENU_IDENTIFY_DEVS
MsgQuestionReturn:
   .byte "?",CR+$80
.endif
MsgErrorNumber:
   .byte CR,CR
   SentinelString "Err $"
.if FULL_MENU
MsgMenuRow0:
   SentinelString "Partitions Dev0"
MsgMenuRow1:
   SentinelString "Dev1"
MsgMenuRow2:
   .byte "Boot "       ; "Boot Dev" (continues in next message)
MsgDev:
   SentinelString "Dev"
MsgMenuRow3:
   SentinelString "Partition"
MsgBootSaveQuit:
   SentinelString "B)OOT S)AVE Q)UIT"
.endif
   ASSERT (*-Messages)<256

;------------------------------------------------------------------------------

ALLOW_OVERFLOWING_ROM = 0

.if REQUIRES_EEPROM

.if ALLOW_OVERFLOWING_ROM
   .warning "Allowing ROM overflow, just to see how big the code is."
   .byte $FF ; for $CFEF
   .byte $FF ; for $CFFF
.else
   .out .concat("Free bytes in EEPROM = ", .string($CFEF-*))
   .RES $CFEF-*,$77    ; 238 bytes avail EEPROM from $CF00 to $CFEE.
   CFEF_Unavailable:
      .byte $FF           ; touching $CFEF turns off the AUX ROM
.endif

;------------------------------------------------------------------------------
; This "image" of WriteOneByteToEEPROM can't be executed in place,
; because after writing to EEPROM, you can't read from it for the
; next 200 microseconds.  This get copied into RAM and called there.
;
; 15 bytes available (carefully) on EEPROM ($CFF0..CFFE).
;------------------------------------------------------------------------------
.if FULL_MENU
Image_WriteOneByteToEEPROM:
   sta Max32MBPartitionsDev0,x
   ldy #0
:  dey
   beq @fail
   cmp Max32MBPartitionsDev0,x
   bne :-
@fail:
   tya     ; out:  BEQ failure, BNE success
   rts
Image_WOB_End = *
.endif
;------------------------------------------------------------------------------
.if ALLOW_OVERFLOWING_ROM
   .out .concat("Free bytes in EEPROM = ", .string($D000-*))
.else
   .RES $CFFF-*,$33
   CFFF_Unavailable:      ; touching $CFFF turns off the AUX ROM
      .byte $FF
      ASSERT *=$D000, "Fatal error - We should be at exactly $D000 here."
.endif
;------------------------------------------------------------------------------
.else
   ; EPROM only, no EEPROM.  We have to stop before $CF00.
   .out .concat("Free bytes in EPROM = ", .string($CF00-*))
   .if ALLOW_OVERFLOWING_ROM
      .warning "Allowing ROM overflow, just to see how big the code is."
   .else
      .RES $CF00-*,$77
      ASSERT *=$CF00, "Fatal error - We should be at exactly $CF00 here."   
   .endif
.endif


