;-------------------------------------------------------------------------------------------------------\
; PVCC.asm  Version 1.1b                                                                                |
; MCU Target 68HC908QY4                                                                                 |
;                                                                                                       |
; Firmware source module for Remote Observation Station's System Control Board (PVCC)                   |
; Hardware version 1.0 RevA with rework                                                                 |
;                                                                                                       |
; External clock source assumed to be 32.7680MHz (A very slight overclock - no problems found)          |
; Bus Clk = 8.192 MHz                                                                                   |
;                                                                                                       |
; Internal clock source assumed to be 12.8MHz (trimmed)                                                 |
; Bus Clk = 3.200 MHz                                                                                   |
;                                                                                                       |
;                                                                                                       |
; Version 1.1a                                                                                          |
; - Revised LoadControl(), to make it easier to understand and modify                                   |
; - Fixed GenerateDisplayData(), Convertion to decimal correctly/display ':' bug                        |
; - Fixed ReadVoltage(), Full Scale value was off by one                                                |
;                                                                                                       |
; Version 1.1:                                                                                          |
; - Changed Hardware power supply arrangement to keep MCU logic always on.                              |
; - Auxillary 10 volt supply is now controlled via PTB2.                                                |
; - Support for LVD added.                                                                              |
; - Photo cell sensor added to control auto on/off feature                                              |
;                                                                                                       |
; Current firmware supports GEL cells.                                                                  |
;                                                                                                       |
; List of Issues: Search for "ISSUE" in source to locate:                                               |
; (1): Disabling this interrupt breaks hung semaphore check used to detect loss of video signal. This   |
;      is currently done to remove an occational jitter in the OSD. If the video signal were to be lost |
;      while IRQ_ISR is generating the overlay the system will do a COP reset, and continue operating   |
;      fine without a video signal.                                                                     |
;-------------------------------------------------------------------------------------------------------/
;
;Copyright (c) 2003, Richard Dreher d/b/a R&D Automation
;All rights reserved.
;
;Redistribution and use in source and binary forms, with or without modification, are permitted provided
;that the following conditions are met:
;
; * Redistributions of source code must retain the above copyright notice, this list of conditions and
;   the following disclaimer.
; * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
;   the following disclaimer in the documentation and/or other materials provided with the distribution.
; * Neither the name of the R&D Automation/Richard Dreher nor the names of its contributors may be used
;   to endorse or promote products derived from this software without specific prior written permission.
;
;THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
;WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
;PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
;ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
;(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
;OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
;STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
;SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
;-----------------------------------------------------------------------------------------------------------

                            XDEF      Entry,main
                            ABSENTRY  Entry
                            Include   'qtqy_registers.inc' ; For the QT1, QT2, QT4, QY1, QY2, QY4

FALSE                       EQU   $00
TRUE                        EQU   $01

CONFIG_DATA_VER             EQU   $02           ; Any time the configuration format/layout changes inc this number
GEL_TC_ELEMENT_SIZE         EQU   $04           ; Bytes/entry in Gel Cell temperature compensation data table


;-------------------- QY series built-in ROM routine entry points -----------------------
GETBYTE                     EQU   $2800
RDVRRNG                     EQU   $2803         ; Read & Verify range
ERARNGE                     EQU   $2806         ; FLASH erase range
PRGRNGE                     EQU   $2809         ; FLASH program range
DELNUS                      EQU   $280C         ; Delay
ICGTRIM                     EQU   $280F         ; Calulate Trim value
ICGTEST                     EQU   $2812
PUTBYTE                     EQU   $FEA1

;-------------------- QY series ROM function defines ------------------------------------
CtrlByt                     EQU   $88           ; Control byte for ROM subroutines
CPUSPD                      EQU   $89           ; CPU speed in units of 0.25MHz
LADDR                       EQU   $8A           ; Last FLASH address to be programmed (2 bytes)
DATA                        EQU   $8C           ; Source Data for FLASH programming route in ROM
CPUSPEEDMHZx4               EQU   13            ; CPU busclock speed in MHz / .25MHz. Rounded to nearest integer.

;----------------- Commands/Response codes received from Host PC -------------------------
CMD_GET_CONFIG              EQU   $A1           ; ConfigMode Host command code to signal sending of Config data
CMD_SEND_CONFIG             EQU   $AF           ; ConfigMode Host command code to signal the reception of ConfigData
COMM_GOOD_RESPONSE          EQU   $11
COMM_FAIL_RESPONSE          EQU   $FF

;------------------------ SysSems1 BYTE bit definitions and masks------------------------
TEMP_ERROR_BIT              EQU   0             ; Set when a error reading DS1820 occurs
TEMP_ERROR_MASK             EQU   %00000001

DATA_SAMPLE_TIM_BIT         EQU   1             ; Set when one user defined data sample period elapses
DATA_SAMPLE_TIM_MASK        EQU   %00000010

DISPLAY_ACTIVE_BIT          EQU   2             ; Set when IRQ ISR is actively overlaying text
DISPLAY_ACTIVE_MASK         EQU   %00000100

OVERLAY_DONE_SIGNAL_BIT     EQU   3             ; Set when Overlay drawing finished last line
OVERLAY_DONE_SIGNAL_MASK    EQU   %00001000

NO_VIDEO_PRESENT_TWICE_BIT  EQU   4             ; Set when no video signal is present for two sample periods
NO_VIDEO_PRESENT_TWICE_MASK EQU   %00010000

NO_VIDEO_PRESENT_ONCE_BIT   EQU   5             ; Set when no video signal is present for one sample period
NO_VIDEO_PRESENT_ONCE_MASK  EQU   %00100000

TEMP_NEG_SIGN_BIT           EQU   6             ; Set when temperature read from DS1820 is negative
TEMP_NEG_SIGN_MASK          EQU   %01000000

DISPLAY_CALLSIGN_BIT        EQU   7             ; Set when a user text message should be displayed for a sample period
DISPLAY_CALLSIGN_MASK       EQU   %10000000

;------------------------ SysSems2 BYTE bit definitions and masks------------------------
DISPLAY_FULL_SETPOINT_BIT   EQU   0             ; Set when the PV full charge Set point should be displayed
DISPLAY_FULL_SETPOINT_MASK  EQU   %00000001

AUXPOWER_REQUEST_BIT        EQU   1
AUXPOWER_REQUEST_MASK       EQU   %00000010     ; Used by LoadControl as temp memory bit. (Not enough stack space for locals)

TEMP_SENSOR_ERROR_BIT       EQU   2             ; Set when multiple consecutive temperature sensor reads occurr. Reset on good read
TEMP_SENSOR_ERROR_MASK      EQU   %00000100

;-------------------------- MCU register bit labels ------------------------------
ECGST                       EQU   0             ; OSCSTAT bit 0
ECGON                       EQU   1             ; OSCSTAT bit 1
TSTOP                       EQU   5             ; TSC bit 5
TOIE                        EQU   6             ; TSC bit 6
TOF                         EQU   7             ; TSC bit 7
COCO                        EQU   7             ; ADSCR bit 7

;--------------------------- Video Overlay defines --------------------------------
CHAR_HEIGHT                 EQU   7                     ; Number of vertical dot is character.
SCAN_LINE_COUNT             EQU   CHAR_HEIGHT * 2       ; Total scan lines in Character.
LEFT_MARGIN                 EQU   9
DISPLAY_LEN                 EQU   9
OSD_AT_BOTTOM_SLINE         EQU   225
OSD_AT_TOP_SLINE            EQU   25

;--------------------------- Port B defines ------------------------------
VERT_SYNC_PORT              EQU   PORTB
VERT_SYNC_BIT               EQU   0

EXOSC_ENABLE_BIT            EQU   1             ; Bit 1 of PORTB: Set to output and set B1=0 to disable external oscillator
EXOSC_ENABLE_PORT           EQU   PORTB

AUXPOWER_PORT               EQU   PORTB
AUXPOWER_BIT                EQU   2             ; On/Off control of 10v power supply

; JP5 berg 1-2:  Jumper for Charge mode. Remove jumper for Config mode
CONFIG_MODE_PORT            EQU   PORTB
CONFIG_MODE_BIT             EQU   3             ; Bit 3 of PORTB: 1 = Config mode; 0 = Charge Mode

LED3_PORT                   EQU   PORTB
LED3_BIT                    EQU   4

LED2_PORT                   EQU   PORTB
LED2_BIT                    EQU   5

LED1_PORT                   EQU   PORTB
LED1_BIT                    EQU   6

VIDEO_OUT_PORT              EQU   PORTB
VIDEO_OUT_BIT               EQU   7

;--------------------------- Port A defines --------------------------------
DS1820_DQ_PORT              EQU   PORTA
DS1820_DQ_BIT               EQU   3             ; 1820 DQ signal


BATTERY_CHARGE_PORT         EQU   PORTA
BATTERY_CHARGE_BIT          EQU   4

CHARGE_LED_PORT             EQU   LED2_PORT
CHARGE_LED_BIT              EQU   LED2_BIT
FULL_SCALE_VOLTAGE          EQU   $14           ; 20.0 volts is full scale for the A/D input (uses resistor divider to divide down to 5volts)

;----------------------- TIM Module related defines -------------------------
TIM_MOD_COUNT_8MHZ          EQU   512           ; For external osc = 8.1920 MHz, Interrupt every 4ms.
TIM_MOD_COUNT_3MHZ          EQU   200           ; For internal osc = 3.2MHz.  Interrupt every 4ms.
MIN_UPDT_PERIOD             EQU   500           ; Minimum update period = 4 seconds = 1000 * 4ms
INTS_PER_ONE_SECOND         EQU   250           ; number of 4ms interrupts per second
MINS_BETWEEN_HAMCALL_DISP   EQU 10              ; Period in minutes between display of HAM call sign

;----------------------- DS1820 Temperature Sensor defines -------------------------
DS1820_DQ_DDR               EQU   DDRA
SKIPROM                     EQU   $CC           ; 1820 Skip ROM command byte
CONVERT                     EQU   $44           ; 1820 Temperature Convert command byte
READRAM                     EQU   $BE           ; 1820 Read RAM command byte
POSITIVE_SIGN               EQU   $00           ; MSB of a positive temperature reading
NEGATIVE_SIGN               EQU   $FF           ; MSB of a neagtive temperature reading
POSITIVE_LIMIT              EQU   $AA           ; The highest LSB for a positive temperature.
NEGATIVE_LIMIT              EQU   $92           ; The lowest LSB for a negative temperature.
MAX_TEMP_SENSOR_ERRORS      EQU   3             ; Allow 3 consecutive bad temperature reading, before assuming sensor is bad or missing

;-------------------------------- Misc constants -----------------------------------
Delay_10uS                  EQU   2
Delay_15uS                  EQU   3
Delay_70uS                  EQU   14
Delay_80uS                  EQU   16
Delay_100uS                 EQU   20
Delay_120uS                 EQU   24
Delay_425uS                 EQU   85
Delay_480uS                 EQU   96
Delay_500uS                 EQU   100
Delay_1mS                   EQU   200


;----------------------------------- RAM Memory -----------------------------------------
                            org   $0080
SysSems1:                   DS.B  1             ; System semaphores #1 bit packed variable
                                                ; Bit 0 = Error during DS1820 access
                                                ; Bit 1 = Request Sample of Voltage and Temperature Sempaphore
                                                ; Bit 2 = Overlay Display Currently Active
                                                ; Bit 3 = Overlay Display Done Semaphore
                                                ; Bit 4 = No video sync was present for two Sample period
                                                ; Bit 5 = No video sync was present for one Sample periods
                                                ; Bit 6 = Remember temperature last read was negative
                                                ; Bit 7 = Display call sign request semaphore
                                                
SysSems2:                   DS.B  1             ; System semaphores #2 bit packed variable
                                                ; Bit 0 = Temperature sensor had too many consecutive errors
                                                ; Bit 1 = Display PV full charge Set point 
                                                ; Bit 2 = Used by LoadControl as temp memory bit.
                                                ; Bit 3 = 
                                                ; Bit 4 = 
                                                ; Bit 5 = 
                                                ; Bit 6 = 
                                                ; Bit 7 = 
                                              
CRC16Value:                 DS.W  1             ; CRC16 intermediate result storage
pBuffer:                    DS.W  1             ; Pointer to CRC16 data buffer
TIMSampleTimer              DS.B  1             ; Number of seconds remaining until next sample
CountInterrupts             DS.B  1             ; Number of Interrupts remaining until 1 second elapses
CountSeconds                DS.B  1             ; Number of Seconds remaining until 1 minutes elapses
CountMinutes                DS.B  1             ; Number of minutes remaining until the user's call-sign is displayed
Temp:                       DS.B  1             ; Temporary storage space
LastTempRead                DS.W  1             ; Last Temp Read in signed binary. $80 = last reading invalid
SavePCH:                    DS.B  1             ; Saving location for PCH. Used by IRQ ISR
SavePCL:                    DS.B  1             ; Saving location for PCL. Used by IRQ ISR
CharIndex:                  DS.B  1             
LineIndex:                  DS.B  1             ; Line number in Character cell. Typically 0 to 7
HSyncCount:                 DS.B  1             ; Horizontal sync counter per frame
DisplayData:                DS.B  DISPLAY_LEN   ; Data to be displayed
OffsetOfChar:               DS.B  1             ; Offset into Source String
OffsetOfCharSet:            DS.W  1             ; Offset into CharGen Table
Index                       DS.B  1             ; Table Index for temp comp
BatteryVoltage:             DS.W  1             ; Last battery voltage reading. High byte = integer part Low byte = fractional part
TC_MaxBattVolts             DS.W  1             ; Temperature compensated max battery voltage
ConsecutiveTempSensorErrors DS.B  1             ; Number of consecutive bad temperature readings 
VideoMemory:                DS.B  DISPLAY_LEN * CHAR_HEIGHT   ; Store 9 characters in 8 x 7 dot format
;
; Total RAM usage:
; Variables: 98 bytes
; Stack:     30 bytes
;
;--------------------------- Start of firmware in Flash -----------------------------
      org $EE00
Entry:
main:
      rsp                                       ; Set stack pointer to top of RAM
      sta   COPCTL                            ; service COP control register to prevent COP reset
      clr   SysSems1                            ; Initialize All semaphores to FALSE.
      clr   SysSems2                            ; Initialize All semaphores to FALSE.
      clr   TIMSampleTimer
      clr   TIMSampleTimer+1
      clr   ConsecutiveTempSensorErrors
      
; Initialize I/O Ports
      clr   PORTA
      clr   PORTB
      mov   #%00010000, DDRA
      mov   #%11110110, DDRB

; Turn on all LEDs for a short time to indicate PVCC has started.
; This code also provides a delay before switching to the external osc. if OSD mode is selected.
      bset  LED1_BIT, LED1_PORT
      bset  LED2_BIT, LED2_PORT
      bset  LED3_BIT, LED3_PORT
      jsr   Delay1S

; Clear all LEDs for end of startup indication
      bclr  LED1_BIT, LED1_PORT
      bclr  LED2_BIT, LED2_PORT
      bclr  LED3_BIT, LED3_PORT
      jsr   Delay1S
;
; CRC check product configuration data stored in FLASH for validity. If not valid, write new Config data from default data.
; This code should execute from the internal OSC, and the MCU CONFIG registers should be programmed
; after this code.
;
      ldhx  #CONFIG_DATA
      aix   #2                                  ; Do not include the stored CRC value in the current CRC calculation
      lda   #LOW(CONFIG_DATA_SIZE)
      deca
      deca                                      ; Subtract 2, CRC value in setup is not included in local CRC calculation
      
      jsr   CRC16                               ; Compute CRC16, returns CRC in H:X
      cpx   CONFIG_CRC+1                        ; Check high and low byte for match with stored value
      bne   LoadDefaultConfig
      pshh
      pulx
      cpx   CONFIG_CRC
      beq   CheckSystemMode

; Config data didn't pass CRC, so write new default setup data vaules to FLASH memory
LoadDefaultConfig:
      sta   COPCTL                            ; service COP control register to prevent COP reset
      clrh
      ldx   #LOW(CONFIG_DATA_SIZE)
CopyLoop:
      lda   FactoryDefaultConfigData-1,x        ; Move Default Config Data to RAM for programming into FLASH
      sta   DATA-1,x
      dbnzx CopyLoop

; Calculate the new CRC16 for the data just copied to RAM location DATA
      ldhx  #DATA+2
      lda   #LOW(CONFIG_DATA_SIZE)
      deca
      deca                                      ; Subtract 2, CRC value in setup is not included in local CRC calculation
      jsr   CRC16                               ; Compute CRC16, returns CRC in H:X
      sthx  DATA

; Erase 64 Bytes starting at lable CONFIG_DATA.
      mov   #13, CPUSPD
      bclr  6, CtrlByt                          ;Flag to erase a single page
      ldhx  #CONFIG_DATA                        ;Valid address in page
      jsr   ERARNGE

; Program Configuration data into flash
      ldhx  #(CONFIG_DATA+CONFIG_DATA_SIZE-1)
      sthx  LADDR
      mov   #13, CPUSPD                         ; 3.2MHz/0.25MHz = 13
      clr   CtrlByt                             ; Page (not mass) erase
      ldhx  #CONFIG_DATA
      jsr   PRGRNGE
      bset  LED1_BIT, LED1_PORT                 ; Turn on the Red LED1, to indicate that default setup data was written
      
CheckSystemMode:
; Initialize timing counter with constants/Config data
      mov   #INTS_PER_ONE_SECOND, CountInterrupts
      mov   #60, CountSeconds
      mov   #MINS_BETWEEN_HAMCALL_DISP, CountMinutes
      lda   CONFIG_UPDT_PERIOD
      sta   TIMSampleTimer

      sta   COPCTL                            ; service COP control register to prevent COP reset
      bset  LED3_BIT, LED3_PORT                 ; Turn On Red LED3 to indicate Setup has been read
      jsr   Delay1S
      jsr   Delay1S                             ; Delay 2 seconds to allow LEDs to be seen
      bclr  LED3_BIT, LED3_PORT                 ; Turn OFF Red LED2
      bclr  LED1_BIT, LED1_PORT                 ; Turn OFF Red LED1 (may not have been on)

; Max battery setpoint is copied to RAM even if Temp comp is not used.
      lda   CONFIG_MAX_BATT_VOLTS
      sta   TC_MaxBattVolts
      lda   CONFIG_MAX_BATT_VOLTS+1
      sta   TC_MaxBattVolts+1

; Turn on the auxillary loads. Typically the camera and transmitter
      bset  AUXPOWER_BIT, AUXPOWER_PORT
      
; Based on setting of JP5 mode select jumper block
      brset CONFIG_MODE_BIT, CONFIG_MODE_PORT, ConfigMode  ; Check for setup mode request, i.e. jumper removed.

; Check the OSD enable byte in configuration data for specific Charge Mode
      lda   CONFIG_ENABLE_OSD
      bne   JMPtoOSDChargeMode
      jmp   ChargeModeInit
JMPtoOSDChargeMode:      
      jmp   OSDChargeMode

;----------------------------------------------------------------------------------\
; Config Mode:                                                                     |
; This mode of operation is used to setup the behavior of this device.             |
; Configuration is acomplished by connecting the station controller to a PC via    |
; an RS232 serial port. Using a simple Microsoft Windows program the user can set  |
; various settings to control the behavior of the charge controller                |
; This mode is characterized by the following:                                     |
; 1) Lower clock speed, internal OSC.                                              |
; 2) No Video overlay generation                                                   |
; 3) No use of interrupts                                                          |
; 4) No Charge Controller operation                                                |
; 5) Communication with host PC via RS232 serial cable                             |
;----------------------------------------------------------------------------------/
ConfigMode:

      mov   #%01001111, CONFIG1               ; MCU config 1 register. Same config for both low and high speed operation
;             |||||||| 
;             |||||||\- COPD    : 1 = Disable COP watchdog
;             ||||||\-- STOP    : 1 = Enable STOP instruction
;             |||||\--- SSREC   : 1 = 32 cycle STOP recovery
;             ||||\---- LVI5OR3 : 1 = Set LVI for 5v system
;             |||\----- LVIPWRD : 0 = Enable power to LVI system
;             ||\------ LVIRSTD : 0 = Enable reset on LVI trip
;             |\------- LVISTOP : 1 = Enable LVI in STOP mode
;             \-------- COPRS   : 0 = Long COP timeout

      mov   #%10000000, CONFIG2               ; Internal Osc. | No IRQ | No IRQ Pull-up
;             |||||||| 
;             |||||||\- RSTEN       RST Pin Function Select: 1 = Reset function active, 0 = inactive                         
;             ||||||\-- reserved
;             |||||\--- reserved                                                                                 
;             ||||\---- OSCOPT0     0, 0 Internal Osc.    0, 1 External Osc.                                                                             
;             |||\----- OSCOPT1     1, 0 External RC Osc. 1, 1 External XTAL Osc.
;             ||\------ reserved                                                                                                                                                        
;             |\------- IRQEN       IRQ Pin Function Select Bit: 1 = IRQ Active, 0 = IRQ inactive                                                                                                                                                                     
;             \-------- IRQPUD      IRQ Pin Pullup Control Bit 1 = no pull up, 0 = internal pullup to Vdd enabled 



      lda   IOSCTRM                             ; Trim the internal Oscillator based on the value stored at $FFC0
      cmp   #$FF
      beq   SM_Port_Init                        ; If Trim value at $FFC0 is $FF, assume no trim value was programmed and skip.
      sta   OSCTRIM

; Initialize I/O Ports
SM_Port_Init:
      clr   PORTA
      clr   PORTB
      mov   #%00010001, DDRA
      mov   #%01110110, DDRB
      
      bset  LED1_BIT, LED1_PORT                 ; Turn on LED1 to indicate Config Mode was entered.
      
; Turn off external osc to save power
      bclr  EXOSC_ENABLE_BIT, EXOSC_ENABLE_PORT


ConfigModeLoop:
      bclr  0, DDRA                             ;set PTA0 to input
      jsr   GETBYTE                             ; Wait for a command byte from the host PC
      cmp   #CMD_GET_CONFIG
      bne   CheckForSend

; Process Host's Get Config Request
; This command will send data to the Host PC       
      bset  LED2_BIT, LED2_PORT                 ; Turn on LED2 to indicate HostGetConfigRequest
      ldhx  #CONFIG_DATA
      aix   #LOW(CONFIG_DATA_SIZE)
      aix   #-1
      sthx  LADDR
      ldhx  #CONFIG_DATA
      clra                                      ; Acc = 0 Signals RDVRRNG to send data to monitor port
      jsr   RDVRRNG
      bclr  LED2_BIT, LED2_PORT                 ; Turn off LED2
      bra   DumpChars

CheckForSend:
      cmp   #CMD_SEND_CONFIG                    ;[2]
      bne   DumpChars
      
; Process Host's Send Config Request
; Receive the configuration data being sent by the host PC.
      bset  LED3_BIT, LED3_PORT                 ; Turn on LED3 to indicate HostGetConfigRequest
      clrh
      ldx   #0
HS_Loop:
      bclr  0, DDRA                             ;[4] Set to input.
      jsr   GETBYTE                             ;[5] 
      sta   DATA,x                              ;[3] 
      incx                                      ;[1] 
      cpx   #LOW(CONFIG_DATA_SIZE)              ;[2] 
      bne   HS_Loop                             ;[3] 

; Calculate the CRC16 for the data just recieved. Don't include the CRC data at DATA, DATA+1
      ldhx  #DATA+2
      lda   #LOW(CONFIG_DATA_SIZE)
      deca
      deca                                      ; Subtract 2, CRC value in setup is not included in local CRC calculation
      jsr   CRC16                               ; Compute CRC16, returns CRC in H:X
      cphx  DATA
      bne   BadSendCRC

; DEBUG: CRC was good
      bset  6, PORTB                            ;DEBUG turn on LED to show CRC was good      
;
; Send response indicating good CRC: $FA, $11
      lda   #$FA
      bclr  0, DDRA                             ;set PTA0 to input. PUTBYTE checks the line before transmitting
      jsr   PUTBYTE

      lda   #COMM_GOOD_RESPONSE
      bclr  0, DDRA                             ;set PTA0 to input. PUTBYTE checks the line before transmitting
      jsr   PUTBYTE

; Erase 64 Bytes starting at lable CONFIG_DATA.
      mov   #13, CPUSPD
      bclr  6, CtrlByt                          ; Flag to erase a single page
      ldhx  #CONFIG_DATA                        ; Valid address in page
      jsr   ERARNGE


; Program Configuration data into flash
      ldhx  #(CONFIG_DATA+CONFIG_DATA_SIZE-1)
      sthx  LADDR
      mov   #CPUSPEEDMHZx4, CPUSPD              ; Flash programming routine needs to know how fast the CPU is running
      clr   CtrlByt                             ; Page (not mass) erase
      ldhx  #CONFIG_DATA
      jsr   PRGRNGE

      bclr  LED3_BIT, LED3_PORT                 ; Turn off LED3
      bra   ConfigModeLoop

BadSendCRC:
; Send response indicating failed CRC: $FA, $FF
      lda   #$FA
      bclr  0, DDRA                             ; Set PTA0 to input. PUTBYTE checks the line before transmitting
      jsr   PUTBYTE

      lda   #COMM_FAIL_RESPONSE
      bclr  0, DDRA                             ; Set PTA0 to input. PUTBYTE checks the line before transmitting
      jsr   PUTBYTE

      bclr  LED3_BIT, LED3_PORT                 ; Turn off LED3

      
; Delay 20ms. Let additional noise bytes go by unread.
      ldx   #20
DumpChars:
      lda   #Delay_1mS
      jsr   Delay
      dbnzx DumpChars      
      jmp   ConfigModeLoop
           
;----------------------------------------------------------------------------------\
; Charge Mode without OSD:                                                         |
; This mode of operation is the normal default operating mode.                     |
; This mode is characterized by the following:                                     |
; 1) Lower clock speed, internal OSC.                                              |
; 2) Lower power consumption by use of WAIT instruction and slower clock           |
; 3) No Video overlay generation                                                   |
; 4) No external interrupts                                                        |
; 5) Normal charge controller operation                                            |
;----------------------------------------------------------------------------------/
ChargeModeInit:
      mov   #%01001110, CONFIG1                 ; MCU config 1 register. Same config for both low and high speed operation
;             |||||||| 
;             |||||||\- COPD    : 0 = Enable COP watchdog
;             ||||||\-- STOP    : 1 = Enable STOP instruction
;             |||||\--- SSREC   : 1 = 32 cycle STOP recovery
;             ||||\---- LVI5OR3 : 1 = Set LVI for 5v system
;             |||\----- LVIPWRD : 0 = Enable power to LVI system
;             ||\------ LVIRSTD : 0 = Enable reset on LVI trip
;             |\------- LVISTOP : 1 = Enable LVI in STOP mode
;             \-------- COPRS   : 0 = Long COP timeout      =  2^18 - 16 BUSCLKX4 clocks
;
      mov   #%10000000, CONFIG2                 ; Internal Osc. | No IRQ | No IRQ Pull-up
;             |||||||| 
;             |||||||\- RSTEN     0 = RST Pin Function Inactive
;             ||||||\-- reserved  0
;             |||||\--- reserved  0
;             ||||\---- OSCOPT0   0
;             |||\----- OSCOPT1   0 = 0,0 Internal Osc. 0,1 External Osc. 1,0 External RC Osc. 1,1 External XTAL Osc.
;             ||\------ reserved  0
;             |\------- IRQEN     0 = IRQ Pin Function Inactive
;             \-------- IRQPUD    1 = IRQ Pin Pullup Control Inactive(no pull up),  1 = internal pullup to Vdd enabled 
;

      lda   IOSCTRM                             ; Trim the internal Oscillator based on the value stored at $FFC0
      cmp   #$FF
      beq   NM_Timer_Init                       ; If Trim value at $FFC0 is $FF, assume no trim value was programmed and skip.
      sta   OSCTRIM

; Setup Timer Interface Module (TIM) configuration registers to run at approximately .5 interrupt rate
NM_Timer_Init:
      mov   #%00000110, TSC                     ; Select divide by 64 prescaler
      bset  4, TSC                              ; set TRST, Set Write-only bit to clear TIM counter and prescaler before writing Module registers

; Setup up periodic timer to interrupt every 4ms(12800 cycles). Note: The COP will time out every 20.5ms.
; For internal osc. Assume approx 12.8Mhz / 4  = 3.2MHz
      mov   #HIGH(TIM_MOD_COUNT_3MHZ), TMODH    ; Load the Counter Modulo register High
      mov   #LOW(TIM_MOD_COUNT_3MHZ), TMODL     ; Load the Counter Modulo register Low
      
      bset  TOIE, TSC                           ; Set TOIE  = TRUE; to allow interrupts from timer
      bclr  TSTOP, TSC                          ; Set TSTOP = FALSE; to allow hardware timer to run

;; Initialize I/O Ports
;      clr   PORTA
;      clr   PORTB
;      mov   #%00010001, DDRA
;      mov   #%01110110, DDRB

; Turn off external osc to save power
      bclr  EXOSC_ENABLE_BIT, EXOSC_ENABLE_PORT

; Initialize A/D Converter to measure voltage on Pin AD1 (PTA1)
      mov   #%01100000, ADICLK                  ; Setup clock for A/D to ExternalClock/4/8 MHz
      cli
      sta   COPCTL                            ; Service cop control register to prevent reset
      
ChargeModeLoop:
      wait
      sta   COPCTL                            ; Service cop control register to prevent reset
      brclr DATA_SAMPLE_TIM_BIT, SysSems1, ChargeModeLoop

; Read battery voltage and temperature
      jsr   ReadVoltage

; Check to see if the temp should be ignored
      lda   CONFIG_USE_TEMP_SENSOR
      beq   ChargeCtrl

      bclr  TEMP_ERROR_BIT, SysSems1
      jsr   StartTempConversion
      jsr   CheckForTempErrors

      jsr   Delay1S
      jsr   ReadTemperature
      jsr   CheckForTempErrors
      jsr   GELTempComp

; Turn Charging on/off based current voltage/temp/user settings
ChargeCtrl:
      jsr   ChargeControl      
      jsr   LoadControl
      bclr  DATA_SAMPLE_TIM_BIT, SysSems1
      bra   ChargeModeLoop

;----------------------------------------------------------------------------------\
; OSD Mode:                                                                        |
; This mode of operation is used to display the battery voltage and temperature    |
; as a video overlay on the video being transmitted by the station                 |
; This mode is characterized by the following:                                     |
; 1) High clock speed, external canned OSC.                                        |
; 2) Higher power consumption, no STOP/WAIT instruction used in main loop          |
; 3) Video overlay generation for display of diagnostic information                |
; 4) Use of external interrupt                                                     |
; 5) Normal charge controller operation                                            |
;----------------------------------------------------------------------------------/
OSDChargeMode:
      mov   #%01001110, CONFIG1                 ; MCU config 1 register. Same config for both low and high speed operation
;             |||||||| 
;             |||||||\- COPD    : 0 = Enable COP watchdog
;             ||||||\-- STOP    : 1 = Enable STOP instruction
;             |||||\--- SSREC   : 1 = 32 cycle STOP recovery
;             ||||\---- LVI5OR3 : 1 = Set LVI for 5v system
;             |||\----- LVIPWRD : 0 = Enable power to LVI system
;             ||\------ LVIRSTD : 0 = Enable reset on LVI trip
;             |\------- LVISTOP : 1 = Enable LVI in STOP mode
;             \-------- COPRS   : 0 = Long COP timeout

      mov   #%01001000, CONFIG2                 ; MCU config 2 register
;             |||||||| 
;             |||||||\- RSTEN       RST Pin Function Select: 1 = Reset function active, 0 = inactive                         
;             ||||||\-- reserved
;             |||||\--- reserved                                                                                 
;             ||||\---- OSCOPT0     0, 0 Internal Osc.    0, 1 External Osc.                                                                             
;             |||\----- OSCOPT1     1, 0 External RC Osc. 1, 1 External XTAL Osc.
;             ||\------ reserved                                                                                                                                                        
;             |\------- IRQEN       IRQ Pin Function Select Bit: 1 = IRQ Active, 0 = IRQ inactive                                                                                                                                                                     
;             \-------- IRQPUD      IRQ Pin Pullup Control Bit 1 = no pull up, 0 = internal pullup to Vdd enabled 

      bset  EXOSC_ENABLE_BIT, EXOSC_ENABLE_PORT ; Enable external canned osc.
      bset  ECGON, OSCSTAT                      ; Switch MCU to external clock source
; Wait for external source to be engaged.
OscWait:
      brclr ECGST, OSCSTAT, OscWait


; Init DisplayData to show sign-on message at startup if in OSD mode.
      clrh
      ldx   #DISPLAY_LEN   
InitDispData:
      lda   SignOn-1, x
      sta   DisplayData-1, x
      dbnzx InitDispData

; Setup Timer Interface Module (TIM) configuration registers to run at approximately .5 interrupt rate
      mov   #%00000110, TSC                     ; Select divide by 64 prescaler
      bset  4, TSC                              ; Set TRST, Set Write-only bit to clear TIM counter and prescaler before writing Module registers
                                              
; Setup up periodic timer to interrupt every 8ms(64000 cycles). The COP will time out every 8.19ms.
; This leaves .19ms ( ~320 four cycle instruction) to reset the COP before it triggers a reset.      
      mov   #HIGH(TIM_MOD_COUNT_8MHZ), TMODH    ; Load high byte of TIM MOD terminal value
      mov   #LOW(TIM_MOD_COUNT_8MHZ), TMODL     ; Load low byte of TIM MOD terminal value
      bset  TOIE, TSC                           ; Set TOIE  = TRUE; to allow interrupts from timer
      bclr  TSTOP, TSC                          ; Set TSTOP = FALSE; to allow hardware timer to run

; Initialize A/D Converter to measure voltage on Pin AD1 (PTA1)
      mov   #%01100000, ADICLK                  ; Setup clock for A/D to ExternalClock/4/8 MHz
      jsr   UpdateVideoRAM
      cli                                       ; Allow interrupts

OSDChargeModeLoop:
      sta   COPCTL                            ; Service cop control register to prevent reset
      brclr DATA_SAMPLE_TIM_BIT, SysSems1, OSDChargeModeLoop

; Read battery voltage and temperature and update DisplayData
      jsr   ReadVoltage

; Don't bother waiting for video if camera is off.
      brclr AUXPOWER_BIT, AUXPOWER_PORT, OSDStartTempConv

; Communications with the DS18S20 may be corrupted if a H-video interrupt occurs during the command.
; So synchronize the issuing of commands with the Overlay done signal. To prevent this code
; from hanging if the video signal is removed, the read temperature override signal will be set if ever a
; periodic DATA_SAMPLE_TIM_BIT is set and the OVERLAY_DONE_SIGNAL_BIT is not clear.

      bclr  OVERLAY_DONE_SIGNAL_BIT, SysSems1   ; Clear semaphore flags to get the next rising edge of OVERLAY_DONE_SIGNAL
      bset  LED3_BIT, LED3_PORT
VideoWait0:      
      sta   COPCTL                              ; reset COP
      lda   SysSems1
      and   #(OVERLAY_DONE_SIGNAL_MASK | NO_VIDEO_PRESENT_TWICE_MASK)
      beq   VideoWait0
      bclr  LED3_BIT, LED3_PORT
      
OSDStartTempConv:
; Check to see if the temp should be ignored
      lda   CONFIG_USE_TEMP_SENSOR
      beq   OSDCharge

      bclr  TEMP_ERROR_BIT, SysSems1
      jsr   StartTempConversion
      jsr   CheckForTempErrors
      brset TEMP_ERROR_BIT, SysSems1, OSDCharge ; If error in temp reading skip temp read and comp
      
      jsr   Delay1S

      brclr AUXPOWER_BIT, AUXPOWER_PORT, OSDReadTemp  ; Don't bother waiting for video if camera is off.
; Again, wait for next video overlay to complete.
      bclr  OVERLAY_DONE_SIGNAL_BIT, SysSems1   ; Clear semaphore flags to get the next rising edge of OVERLAY_DONE_SIGNAL
VideoWait1:      
      sta   COPCTL                              ; Kick the dog
      lda   SysSems1
      and   #(OVERLAY_DONE_SIGNAL_MASK | NO_VIDEO_PRESENT_TWICE_MASK)
      beq   VideoWait1

OSDReadTemp:
      jsr   ReadTemperature
      jsr   CheckForTempErrors
      jsr   GELTempComp

OSDCharge:
; Turn Charging on/off based on voltage, temp, and user settings
      jsr   ChargeControl      
      jsr   LoadControl
      brclr DISPLAY_CALLSIGN_BIT, SysSems1, CheckDispSP

      bclr  DISPLAY_CALLSIGN_BIT, SysSems1      ; Clear display call sign request semaphore
      jsr   GenerateCallSignData
      bra   DoUpdate      

CheckDispSP:
      brclr DISPLAY_FULL_SETPOINT_BIT, SysSems2, DataDisp
      bclr  DISPLAY_FULL_SETPOINT_BIT, SysSems2 ; Clear display call sign request semaphore
      jsr   GenerateSetPointDisplay
      bra   DoUpdate      
    
DataDisp:
      jsr   GenerateDisplayData

; Update the video memory from DisplayData
DoUpdate:
      jsr   UpdateVideoRAM
      bclr  DATA_SAMPLE_TIM_BIT, SysSems1
      bra   OSDChargeModeLoop

;-----------------------------------------------------------------------------------\
; Function Name: ChargeControl()                                                    |
;                                                                                   |
; Inputs: Setup data area: CONFIG_, and Gobals: "LastTempRead", "BatteryVoltage"    |
;                                                                                   |
; Outputs: BATTERY_CHARGE_BIT, CHARGE_LED_BIT                                       |
;                                                                                   |
; Registers affected: Acc, H:X.                                                     |
;                                                                                   |
; Purpose: Check conditions to allow or inhibit battery charging. Turn on/off       |
; battery charging based on current conditions                                      |
;-----------------------------------------------------------------------------------/
ChargeControl:
; Check the User Configuration to see if charging has been Enabled.
      lda   CONFIG_ENABLE_CHARGING
      beq   CC_DisableCharge

; Check to see if the temp should be ignored
      lda   CONFIG_USE_TEMP_SENSOR
      beq   CC_CheckMin

;Check to see if too many consecutive temperture sensor read errors have ocurred, if so disable charging
      brset TEMP_SENSOR_ERROR_BIT, SysSems2, CC_DisableCharge

; Check the battery temperature      
      lda   CONFIG_MAX_BATT_TEMP
      psha
      pulh
      ldx   CONFIG_MAX_BATT_TEMP+1
      cphx  LastTempRead
      blo   CC_DisableCharge
      
CC_CheckMin:
; Compare sampled battery voltage against Configuration Data Min Battery Voltage level.
      lda   CONFIG_MIN_BATT_VOLTS
      psha
      pulh
      ldx   CONFIG_MIN_BATT_VOLTS+1
      cphx  BatteryVoltage
      blo   CC_CheckMax
      
; Turn ON charging from Solar Panel
      bset  BATTERY_CHARGE_BIT, BATTERY_CHARGE_PORT
      bset  CHARGE_LED_BIT, CHARGE_LED_PORT
      bra   ExitCC
      
; Compare sampled battery voltage against Configuration Data Max Battery Voltage level.
CC_CheckMax:
      ldhx  BatteryVoltage
      cphx  TC_MaxBattVolts
      blo   ExitCC
      
; Turn OFF charging from Solar Panel
CC_DisableCharge:
      bclr  BATTERY_CHARGE_BIT, BATTERY_CHARGE_PORT
      bclr  CHARGE_LED_BIT, CHARGE_LED_PORT

ExitCC:
      rts

;----------------------------------------------------------------------------------------\
; Function Name: LoadControl()                                                           |
;                                                                                        |
; Inputs: AUXPOWER_BIT, CONFIG_ setup data, AD0 (light level)                            |
;                                                                                        |
; Outputs: AUXPOWER_REQUEST_BIT, AUXPOWER_BIT                                            |
;                                                                                        |
; Registers affected: Acc, H:X                                                           |
;                                                                                        |
; Purpose:Turn on/off loads (aux power) based on battery voltage and daylight            |
;                light levels                                                            |
;                                                                                        |
;----------------------------------------------------------------------------------------/
LoadControl:
      bset  AUXPOWER_REQUEST_BIT, SysSems2

; Check Aux power supply output state.
      brset AUXPOWER_BIT, AUXPOWER_PORT, AuxPowerAlreadyON

; Check the automatic on/off enable bit
      lda   CONFIG_ENABLE_LOAD_CTRL             ; Check if "Day On/Night Off" is active
      beq   CheckLVD                            ; If not, branch

; Check the light level sensor (CdS cell). A/D reading will be low in day light and high at night
      mov   #0, ADSCR                           ; Start Conversion on Channel AD0
ADCWait0:                                 
      brclr COCO, ADSCR, ADCWait0               ; Wait for conversion complete      
      lda   ADR                                 ; Get A/D reading
      coma                                      ; Compliment reading so low values correspond to low light levels
                                          
      cmp   CONFIG_MAX_OFF_LIGHT_LEV            ; Check light level against "turn on" light level point
      bhi   CheckLVD                            ; Still too dark
      bclr  AUXPOWER_REQUEST_BIT, SysSems2
            
CheckLVD:      

; Compare the LVD enable bit
      lda   CONFIG_ENABLE_LVD                   ; Check if LVD feature is requested by user
      beq   UpdateLoads

; Check the LV reconnect, if battery voltage is >= turn on loads
      lda   CONFIG_RECONNECT_VOLTS
      psha
      pulh
      ldx   CONFIG_RECONNECT_VOLTS+1
      cphx  BatteryVoltage
      ble   UpdateLoads
      bclr  AUXPOWER_REQUEST_BIT, SysSems2
      bra   UpdateLoads

; Aux power supply is already on, check to see if it should be turned off
AuxPowerAlreadyON:
      lda   CONFIG_ENABLE_LOAD_CTRL             ; Check if "Day On/Night Off" is active
      beq   CheckLVD1                           ; If not, branch

; Check the light level sensor (CdS cell). A/D reading will be low in day light and high at night
      mov   #0, ADSCR                           ; Start Conversion on Channel AD0
ADCWait0_1:                                     
      brclr COCO, ADSCR, ADCWait0_1             ; Wait for conversion complete      
      lda   ADR                                 ; Get A/D reading
      coma                                      ; Compliment reading so low values correspond to low light levels
                                                
      cmp   CONFIG_MIN_ON_LIGHT_LEV             ; Check light level against "minimum on" light level point
      bhs   CheckLVD1                           ; Branch if still light enough
      bclr  AUXPOWER_REQUEST_BIT, SysSems2
            
CheckLVD1:      

; Compare the LVD enable bit
      lda   CONFIG_ENABLE_LVD                   ; Check if LVD feature is requested by user
      beq   UpdateLoads

; Check the LV reconnect, if battery voltage is >= turn on loads
      lda   CONFIG_LVD_VOLTS
      psha
      pulh
      ldx   CONFIG_LVD_VOLTS+1
      cphx  BatteryVoltage
      blo   UpdateLoads
      
      bclr  AUXPOWER_REQUEST_BIT, SysSems2


UpdateLoads:
      brclr AUXPOWER_REQUEST_BIT, SysSems2, DoClear
      bset  AUXPOWER_BIT, AUXPOWER_PORT
      bra   ExitLC

DoClear:
      bclr  AUXPOWER_BIT, AUXPOWER_PORT

ExitLC:
      rts

;----------------------------------------------------------------------------------------\
; Function Name: ReadVoltage()                                                           |
;                                                                                        |
; Inputs: None                                                                           |
;                                                                                        |
; Outputs: BatteryVoltage                                                                |
;                                                                                        |
; Registers affected: A, X                                                               |
;                                                                                        |
; Purpose:Read Battery voltage & scale into two byte value                               |
;                                                                                        |
; 0 to 20v input is scaled with a resistor divider to be 0 to 5v at A/D input            |
; Voltage is expressed as: 1 byte left of decimal and 1 byte right of decimal or         |
;  ... 2^2 + 2^1 + 2^0 + 2^-1 + 2^-2 + ... + 2^-8                                        |
; 00 = 00.0                                                                              |
; 80 = 10.0                                                                              |
; FF = 19.9218                                                                           |
; 100= 20.0                                                                              |
; A full scale reading in binary = $14  ($14.00 or 20.00)                                |
;                                                                                        |
; Voltage (2 bytes) =                                                                    |
;  VoltageSample (cnts) * Desired Full Scale (volts) / Max A/D input voltage (cnts)      |
;                                                                                        |
;----------------------------------------------------------------------------------------/
ReadVoltage:
      mov   #1, ADSCR                           ; Start Conversion on Channel 1

ADCWait:
      brclr COCO, ADSCR, ADCWait                ; Wait for conversion complete      
      lda   ADR
      ldx   #FULL_SCALE_VOLTAGE
      mul                                       ; X:A <-- X * A

; Since the Max A/D input is 256 ($100), no division is needed.
      sta   BatteryVoltage+1
      stx   BatteryVoltage
      rts

;----------------------------------------------------------------------------------------\
; Function Name: ReadTemperature()                                                       |
;                                                                                        |
; Inputs: None                                                                           |
;                                                                                        |
; Outputs: TEMP_ERROR_BIT, LastTempRead, TEMP_NEG_SIGN_BIT                               |
;                                                                                        |
; Registers affected:  A, H:X                                                            |
;                                                                                        |
; Purpose: Read the DS1820 temperature.                                                  |
;                                                                                        |
;----------------------------------------------------------------------------------------/
ReadTemperature:

; Read the DS1820 temperature sensor LSB & MSB(sign)
      bclr  TEMP_ERROR_BIT, SysSems1            ; Reset error flag
      jsr   DS1820ReadTemp
      brset TEMP_ERROR_BIT, SysSems1, RT_ErrorExit ; If error in temp reading skip conversion update

;Check for invalid data values
      bclr  TEMP_NEG_SIGN_BIT, SysSems1
      lda   LastTempRead
      beq   RTChkPositive

      cmp   #NEGATIVE_SIGN                      ; Check for an invalid negative data value
      bne   RT_ErrorExit

; Value is negative, check limit
      bset  TEMP_NEG_SIGN_BIT, SysSems1         ; Remember number is negative
      lda   LastTempRead+1
      cmp   #NEGATIVE_LIMIT
      blo   RT_ErrorExit
; Take 2's compliment. Negative sign is added below
      coma
      add   #1                                  ; Twos compliment if negative.
      sta   LastTempRead+1
      bra   ConvertToF
      
; Check for an invalid positive data value.
RTChkPositive:                                 
      lda   LastTempRead+1
      cmp   #POSITIVE_LIMIT
      bhi   RT_ErrorExit                        ; If LSB of temp > positive limit(AA) then exit with error.
      
; Convert Celsius to Fahrenheit. F = C * 9/5 + 32
; NOTE: TempLow's low bit is weighted at 2^-1 (the .5 digit), in effect, it has already been multiplied by two
ConvertToF: 
      lda   LastTempRead+1
      ldx   #9
      mul                                       ;[5] X:A <= X * A
      pshx
      pulh
      ldx   #10                                 ; Divide by 5 and 2 (because temperature started with b0 representing the 2^-1 binary digit (.5 digit)      
      div                                       ;[7] A <= H:A / X     H <= remainder
      brclr TEMP_NEG_SIGN_BIT, SysSems1, RT_Add32
         
; Reading was negative so take 2's compliment before addition
      coma
      add   #1                                  ; Twos compliment if negative.

RT_Add32:      
      add   #32
      sta   LastTempRead+1
      bra   RT_Exit
      
RT_ErrorExit:
      bset  TEMP_ERROR_BIT, SysSems1
RT_Exit:
      rts

;----------------------------------------------------------------------------------------\
; Function Name: CheckForTempErrors()                                                    |
;                                                                                        |
; Inputs: TEMP_ERROR_BIT, TEMP_SENSOR_ERROR_BIT                                          |
;                                                                                        |
; Outputs: TEMP_ERROR_BIT, TEMP_SENSOR_ERROR_BIT                                         |
;                                                                                        |
; Registers affected:  A                                                                 |
;                                                                                        |
; Purpose: Count the number of consecutive temperature read errors and set appropriate   |
;          flags.                                                                        |
;----------------------------------------------------------------------------------------/
CheckForTempErrors:
      brclr TEMP_ERROR_BIT, SysSems1, GoodTemp
      brset TEMP_SENSOR_ERROR_BIT, SysSems2, CFTE_Exit
      inc   ConsecutiveTempSensorErrors
      lda   ConsecutiveTempSensorErrors
      cmp   #MAX_TEMP_SENSOR_ERRORS
      ble   CFTE_Exit
      bset  TEMP_SENSOR_ERROR_BIT, SysSems2
      bset  LED1_BIT, LED1_PORT      ; DEBUG: turn on LED1 (config led) to indicate temp read error to user
      bra   CFTE_Exit
      
GoodTemp:
      bclr  TEMP_SENSOR_ERROR_BIT, SysSems2
      clr   ConsecutiveTempSensorErrors
      bclr  LED1_BIT, LED1_PORT      ; DEBUG: turn OFF LED1 (config led) to indicate temp read is okay again

CFTE_Exit:
      rts


;----------------------------------------------------------------------------------------\
; Function Name: GELTempComp()                                                           |
;                                                                                        |
; Inputs: TEMP_ERROR_BIT, GEL_TC_TABLE                                                   |
;                                                                                        |
; Outputs: TEMP_ERROR_BIT, TC_MaxBattVolts                                               |
;                                                                                        |
; Registers affected: A, H:X                                                             |
;                                                                                        |
; Purpose: Apply temperature compensation to PV full charge regulation                   |
;          set point: CONFIG_MAX_BATT_VOLTS. If last temp reading resulted in Error      |
;          this function returns having done nothing.                                    |
;----------------------------------------------------------------------------------------/
GELTempComp:
      brset TEMP_ERROR_BIT, SysSems1, GTC_ErrorExit   ; If error in temp reading skip conversion update

      ldx   #0
; Handle Special case if temp is negative. Handling this here, eliminates the need for a 2 byte sign compare below
      lda   LastTempRead
      bmi   TC_Found
      
; Search through the temperature comp table looking for the closest temperature entry. 
TC_Loop:
      clrh
      lda   GEL_TC_TABLE+1, x
      cmp   LastTempRead+1
      bhs   TC_Found                            ;NOTE: using unsigned compare (Branch if higher or same)

      FOR label = 1 TO GEL_TC_ELEMENT_SIZE      ; Repeat incx  GEL_TC_ELEMENT_SIZE times.
      incx
      ENDFOR

      bra   TC_Loop

; Grab temp comp offset value and add to Config Regulation setpoint and store in TC_MaxBattVolts  
TC_Found:
      clc
      lda   GEL_TC_TABLE+3, x
      add   CONFIG_MAX_BATT_VOLTS+1
      sta   TC_MaxBattVolts+1
      
      lda   GEL_TC_TABLE+2, x
      adc   CONFIG_MAX_BATT_VOLTS
      sta   TC_MaxBattVolts

GTC_ErrorExit:
      rts      

;----------------------------------------------------------------------------------------\
; Function Name: GenerateCallSignData()                                                  |
;                                                                                        |
; Inputs: CONFIG_CALLSIGN                                                                |
;                                                                                        |
; Outputs: DisplayData                                                                   |
;                                                                                        |
; Registers affected: A, H:X                                                             |
;                                                                                        |
; Purpose: Fill display string with call sign supplied in CONFIG_* data area             |
;                                                                                        |
;----------------------------------------------------------------------------------------/
GenerateCallSignData:
      clrh
      ldx   #DISPLAY_LEN   
CallSignLoop:
      lda   CONFIG_CALLSIGN-1, x
      sta   DisplayData-1, x
      dbnzx CallSignLoop
      rts

;----------------------------------------------------------------------------------------\
; Function Name: GenerateDisplayData()                                                   |
;                                                                                        |
; Inputs: None                                                                           |
;                                                                                        |
; Outputs: DisplayData                                                                   |
;                                                                                        |
; Registers affected: A, H:X                                                             |
;                                                                                        |
; Purpose: Convert Voltage & Temp into a display string.                                 |
;          'V' voltage units symbol is changed to Vc if charging.                        |
;----------------------------------------------------------------------------------------/
GenerateDisplayData:
      
; Clear Display data string
      clrh
      ldx   #DISPLAY_LEN   
      lda   #' '

UD_ClearLoop:
      sta   DisplayData-1, x
      dbnzx UD_ClearLoop

; Convert (binary) voltage value to characters (3 Digits)
; The first 2 are easy just div by 10 and the quotient and remainder are the digits
      clr   Index
      
      lda   BatteryVoltage
      ldx   #10
      div                                       ;[7] A <= H:A / X     H <= remainder
      pshh
      beq   OnesDigit                           ; If zero blank leading zero

      clrh      
      ldx   Index
      add   #$30
      sta   DisplayData, x                      ; 10^1 digit
      inc   Index
      
OnesDigit:      
      pula
      add   #$30
      clrh
      ldx   Index
      sta   DisplayData, x                      ; 10^0 digit
      inc   Index

      lda   #'.'
      ldx   Index      
      sta   DisplayData, x                      ; Decimal point
      inc   Index

; Extract the first digit to the right of the decimal point
      lda   BatteryVoltage+1
      ldx   #10
      mul                                       ; X:A = X * A   X = 10th digit, A = remainder
      txa      
      add   #$30                                ; Convert to a character

      clrh
      ldx   Index
      sta   DisplayData, x                      ; 10^-1 digit
      inc   Index

      lda   #'V'                                ; V for Volts (when not charging)
      brclr  BATTERY_CHARGE_BIT, BATTERY_CHARGE_PORT, ChargeIndicator
      lda   #$3C                                ; Custom char: little 'v' with Superscript 'c' to indicate charging
ChargeIndicator:
      ldx   Index
      sta   DisplayData, x                      ; Units

      ldx   #5
      stx   Index


      brclr TEMP_SENSOR_ERROR_BIT, SysSems2, NoSensorError
      lda   #'E'
      sta   DisplayData, x
      incx
      lda   #'R'
      sta   DisplayData, x
      incx
      sta   DisplayData, x
      bra   ExitGDD      
;
; Check sign(High) Byte of temp and display "-" as need be.      
NoSensorError:
      lda   LastTempRead
      cmp   #NEGATIVE_SIGN
      bne   TempHundredsDigit

      lda   #'-'
      sta   DisplayData, x

; Note: If temp is negative, there can't be a 100's digit so  don't increment Index here as you normally would

; Convert binary Temperature value to characters (3 Digits)
TempHundredsDigit:
      lda   LastTempRead+1
      ldx   #100
      div                                       ;[7] A <= H:A / X     H <= remainder
      pshh
      
      bne   NonZero
      lda   #' '
      bra   T_100s

NonZero:
      add   #$30

T_100s:
      clrh
      ldx   Index
      sta   DisplayData ,x                      ; 10^2 digit
      inc   Index
      
; tens digit
TempTensDigit:
      pula                                      ; Get remainder from div by 100
      ldx   #10
      div                                       ; Now div by 10   A <= H:A / X     H <= remainder
      pshh                                      ; Save remainder
      add   #$30
      ldx   Index
      clrh
      sta   DisplayData ,x                      ; 10^1 digit
      inc   Index

; Ones digit
      pula
      ldx   Index
      clrh
      add   #$30
      sta   DisplayData ,x                      ; 10^0 digit
      inc   Index
      
      lda   #$3B                                ; ASCII ($3B) is used to hold special char <degree>F
      ldx   Index
      sta   DisplayData, x

ExitGDD:
      rts


;----------------------------------------------------------------------------------------\
; Function Name: GenerateSetPointDisplay()                                               |
;                                                                                        |
; Inputs: TC_MaxBattVolts, CONFIG_MIN_BATT_VOLTS                                         |
;                                                                                        |
; Outputs: DisplayData                                                                   |
;                                                                                        |
; Registers affected: A, H:X                                                             |
;                                                                                        |
; Purpose: Convert Low and High setpoints into a display string.                         |
;                                                                                        |
;----------------------------------------------------------------------------------------/
GenerateSetPointDisplay:
;      
; Clear Display data string
;
      clrh
      ldx   #DISPLAY_LEN   
      lda   #' '

GSP_ClearLoop:
      sta   DisplayData-1, x
      dbnzx GSP_ClearLoop

;
; Convert (binary) voltage value to characters (3 Digits)
; The first 2 are easy just div by 10 and the quotient and remainder are the digits
      clr   Index


; Display TempCompensated Max Battery Voltage Setpoint      
      lda   TC_MaxBattVolts
      ldx   #10
      div                                       ;[7] A <= H:A / X     H <= remainder
      pshh                                    
      beq   GSP_Max_1sDigit                     ; If zero blank leading zero
                                              
      clrh                                    
      ldx   Index                             
      add   #$30                                ; Convert to a character
      sta   DisplayData, x                      ; 10^1 digit
      inc   Index                             
                                              
GSP_Max_1sDigit:                              
      pula                                    
      add   #$30                                ; Convert to a character
                                              
      clrh                                    
      ldx   Index                             
      sta   DisplayData, x                      ; 10^0 digit
      inc   Index                             
                                              
      lda   #'.'                              
      ldx   Index                             
      sta   DisplayData, x                      ; Decimal point
      inc   Index

; Extract the first digit to the right of the decimal point
      lda   TC_MaxBattVolts+1
      ldx   #10
      mul                                       ; X:A = X * A   X = 10th digit, A = remainder
      txa      
      add   #$30                                ; Convert to a character

      clrh
      ldx   Index
      sta   DisplayData, x                      ; 10^-1 digit
      inc   Index

      lda   #' '
      ldx   Index
      sta   DisplayData, x
      inc   Index
      

; Display Min Battery Voltage Reconnect Setpoint
      lda   CONFIG_MIN_BATT_VOLTS
      ldx   #10
      div                                       ;[7] A <= H:A / X     H <= remainder
      pshh                                     
      beq   GSP_Min_1sDigit                     ; If zero blank leading zero
                                               
      clrh                                     
      ldx   Index                              
      add   #$30                                ; Convert to a character
      sta   DisplayData, x                      ; 10^1 digit
      inc   Index                              
                                               
GSP_Min_1sDigit:                               
      pula                                     
      add   #$30                                ; Convert to a character
      clrh                                     
      ldx   Index                              
      sta   DisplayData, x                      ; 10^0 digit
      inc   Index                              
                                               
      lda   #'.'                               
      ldx   Index                              
      sta   DisplayData, x                      ; Decimal point
      inc   Index                              
                                               
; Extract the first digit to the right of the decimal point
      lda   CONFIG_MIN_BATT_VOLTS+1
      ldx   #10
      mul                                       ; X:A = X * A   X = 10th digit, A = remainder
      txa      
      add   #$30                                ; Convert to a character

      clrh
      ldx   Index
      sta   DisplayData, x                      ; 10^-1 digit
      inc   Index

      lda   #' '
      ldx   Index
      sta   DisplayData, x
      inc   Index
      
      rts


;----------------------------------------------------------------------------------------\
; Function Name: UpdateVideoRAM()                                                        |
;                                                                                        |
; Inputs: CharGen                                                                        |
;                                                                                        |
; Outputs: VideoMemory                                                                   |
;                                                                                        |
; Registers affected: A, H:X                                                             |
;                                                                                        |
; Purpose: Update the video RAM area with characters in: DisplayData                     |
;                                                                                        |
;----------------------------------------------------------------------------------------/
UpdateVideoRAM:

      clrx
      clrh   
; Using Character data as an index, fill video memory with charcter font data stored in CharGenData table.
 LoopTop:
      pshh
      pshx
      stx   OffsetOfChar
      lda   DisplayData,x
      cmp   #' '
      blo   InvalidChar
      
      cmp   #'Z'
      ble   FillVidMem

InvalidChar:
      lda   #' '                                ; Use a space for all characters after "Z"
      
; Calculate offset into character table.
FillVidMem:
      sub   #$20                                ; Char table doesn't include the first 32 character
      ldx   #$08                               
      mul                                       ; X:A = X * A  Multiply by 8, because 8 bytes / char in table
      pshx
      pulh
      tax      
      sthx   OffsetOfCharSet
        
; Move the bit pattern data to the characters video memory.
      lda   CharGen,x                           ; Get bit patten of character line 0
      ldx   OffsetOfChar
      clrh
      sta   VideoMemory,x

      ldhx  OffsetOfCharSet
      lda   CharGen+1,x                         ; Get bit patten of character line 1
      ldx   OffsetOfChar
      clrh      
      sta   VideoMemory+(1*DISPLAY_LEN),x

      ldhx  OffsetOfCharSet
      lda   CharGen+2,x                         ; Get bit patten of character line 2
      ldx   OffsetOfChar
      clrh      
      sta   VideoMemory+(2*DISPLAY_LEN),x

      ldhx  OffsetOfCharSet
      lda   CharGen+3,x                         ; Get bit patten of character line 3
      ldx   OffsetOfChar
      clrh
      sta   VideoMemory+(3*DISPLAY_LEN),x

      ldhx  OffsetOfCharSet
      lda   CharGen+4,x                         ; Get bit patten of character line 4
      ldx   OffsetOfChar
      clrh      
      sta   VideoMemory+(4*DISPLAY_LEN),x

      ldhx  OffsetOfCharSet
      lda   CharGen+5,x                         ; Get bit patten of character line 5
      ldx   OffsetOfChar
      clrh      
      sta   VideoMemory+(5*DISPLAY_LEN),x
      
      ldhx  OffsetOfCharSet
      lda   CharGen+6,x                         ; Get bit patten of character line 6
      ldx   OffsetOfChar
      clrh      
      sta   VideoMemory+(6*DISPLAY_LEN),x
      
      ldhx  OffsetOfCharSet
      lda   CharGen+7,x                         ; Get bit patten of character line 7
      ldx   OffsetOfChar
      clrh      
      sta   VideoMemory+(7*DISPLAY_LEN),x

      pulx
      pulh
         
NextChar:         
      incx
      cpx   #DISPLAY_LEN
      blo   LoopTop
      rts

;----------------------------------------------------------------------------------------\
; Function Name: StartTempConversion()                                                   |
;                                                                                        |
; Inputs: None                                                                           |
;                                                                                        |
; Outputs: C=0 if no device present, C=1 if device is present                            |
;                                                                                        |
; Registers affected: A                                                                  |
;                                                                                        |
; Purpose: This function sends commands to start a temperature conversion in the DS1820. |
;                                                                                        |
;----------------------------------------------------------------------------------------/
StartTempConversion:
      bclr  TEMP_ERROR_BIT, SysSems1            ; Reset error flag

      jsr   DS1820Reset                         ; Reset the 1820.
      bcc   STC_ErrorExit                       ; No device, exit with error bit set
                                              
      lda   #SKIPROM                            ; Send the 1820's SKIP ROM command.
      sta   Temp                              
      jsr   DS1820Write                       
                                              
      lda   #CONVERT                            ; Send the 1820's CONVERT T command.
      sta   Temp
      jsr   DS1820Write
      bra   STC_Exit
      
STC_ErrorExit:
      bset  TEMP_ERROR_BIT, SysSems1
      
STC_Exit:
      rts

;----------------------------------------------------------------------------------------\
; Function Name: DS1820ReadTemp()                                                        |
;                                                                                        |
; Inputs: None                                                                           |
;                                                                                        |
; Outputs: If Error occurs, Sets TEMP_ERROR_BIT in SysSems1                              |
;                                                                                        |
; Registers affected: A                                                                  |
;                                                                                        |
; Purpose:Read the temperature of the last successful temperature conversion.            |
; The temperature read is returned in the LastTempRead+1 variable.                       |
; Temperature sign is return in LastTempRead. 0 = pos, FF = Neg                          |
;                                                                                        |
;----------------------------------------------------------------------------------------/
DS1820ReadTemp:
      jsr   DS1820Reset                         ; Reset the 1820.
      bcc   RRErrorExit                         ; If the reset fails set the error bit and exit the routine.

      lda   #SKIPROM                    
      sta   Temp                                ; Send the 1820's SKIP ROM command.
      jsr   DS1820Write

; Send read ram command      
      lda   #READRAM                            ; Read the 1820's RAM to get the temperature
      sta   Temp
      jsr   DS1820Write

; Get the Temperature Low byte from DS1820 RAM
      jsr   DS1820Read
      lda   Temp
      sta   LastTempRead+1

; Get Temperature Sign byte from DS1820 RAM      
      jsr   DS1820Read
      lda   Temp
      sta   LastTempRead
      bra   RRExit

RRErrorExit:
      bset  TEMP_ERROR_BIT, SysSems1

RRExit:
      jsr   DS1820Reset                         ; Reset the 1820, this aborts reading further bytes from the DS1820
      rts
      
;----------------------------------------------------------------------------------------\
; Function Name: DS1820Reset()                                                           |
;                                                                                        |
; Inputs: None                                                                           |
;                                                                                        |
; Outputs: Carry is set if part is present, otherwise Carry is clear.                    |
;                                                                                        |
; Registers affected:  A                                                                 |
;                                                                                        |
; Purpose: This function resets the 1820. If the 1820 reset properly, it will            |
; return a response low pulse. If detected function return w/ C =1                       |
; Otherwise C = 0 for no part detected.                                                  |
;                                                                                        |
;----------------------------------------------------------------------------------------/
DS1820Reset:
      bclr  DS1820_DQ_BIT, DS1820_DQ_PORT       ; DQ = 0     
      bset  DS1820_DQ_BIT, DS1820_DQ_DDR
      lda   #Delay_480uS
      jsr   Delay                               ; Create pulse width of 480uS
      bset  DS1820_DQ_BIT, DS1820_DQ_PORT       ; DQ = 1

      bclr  DS1820_DQ_BIT, DS1820_DQ_DDR        ; Set port i/o to input to check response

      lda   #Delay_80uS
      jsr   Delay                               ; Delay 80uS to sample presence pulse
      brset DS1820_DQ_BIT, DS1820_DQ_PORT, NoPart  ; If the start of the pulse is not received, handle the error
      lda   #Delay_425uS
      jsr   Delay                               ; Delay another 480uS and check to see if pulse has ended
      sec                                       ; C = 1 part is present
      rts
      
NoPart:
      lda   #Delay_425uS
      jsr   Delay                               ; Delay another 480uS and check to see if pulse has ended
      clc                                       ; C = 0 part is not present
      rts

;----------------------------------------------------------------------------------------\
; Function Name: DS1820Write()                                                           |
;                                                                                        |
; Inputs: Temp                                                                           |
;                                                                                        |
; Outputs: None                                                                          |
;                                                                                        |
; Registers affected: A, X are used, but restored                                        |
;                                                                                        |
; Purpose: This function writes the data stored in the TEMP variable to the 1820.        |
;                                                                                        |
;----------------------------------------------------------------------------------------/
DS1820Write:
      psha
      pshx
      ldx   #8                                  ; Load X with count.
                                              
WriteLoop:                                    
      bclr  DS1820_DQ_BIT, DS1820_DQ_PORT       ; DQ = 0
      bset  DS1820_DQ_BIT, DS1820_DQ_DDR      
                                              
; Delay 15 cycles / 8MHz = 1.875uS            
      brn   *                                   ; [3] nop
      brn   *                                   ; [3] nop
      brn   *                                   ; [3] nop
      brn   *                                   ; [3] nop
      brn   *                                   ; [3] nop
                                              
      lsr   Temp                                ; Shift bit to be sent into Carry
      bcc   StayZero                          
                                              
      bset  DS1820_DQ_BIT, DS1820_DQ_PORT       ; DQ = 1
                                              
StayZero:                                     
      lda   #Delay_100uS                        ; Delay 100uS
      jsr   Delay                             
                                              
      bset  DS1820_DQ_BIT, DS1820_DQ_PORT       ; DQ = 1

      decx
      bne   WriteLoop

      pulx
      pula
      rts
      
;----------------------------------------------------------------------------------------\
; Function Name: DS1820Read()                                                            |
;                                                                                        |
; Inputs: None                                                                           |
;                                                                                        |
; Outputs: Temp                                                                          |
;                                                                                        |
; Registers affected: A, X are used, but restored                                        |
;                                                                                        |
; Purpose:                                                                               |
;                                                                                        |
;----------------------------------------------------------------------------------------/
DS1820Read:
      psha
      pshx
      
      ldx   #8                                  ; Load X registers with count
                                              
ReadBit:                                      
      bclr  DS1820_DQ_BIT, DS1820_DQ_PORT       ; DQ = 0
      bset  DS1820_DQ_BIT, DS1820_DQ_DDR      
                                              
; Delay 15 cycles / 8MHz = 1.875uS            
      brn   *                                   ; [3] nop
      brn   *                                   ; [3] nop
      brn   *                                   ; [3] nop
      brn   *                                   ; [3] nop
      brn   *                                   ; [3] nop
                                              
      bset  DS1820_DQ_BIT, DS1820_DQ_PORT       ; DQ = 1
                                              
      bclr  DS1820_DQ_BIT, DS1820_DQ_DDR        ; Set the DS1820_DQ_BIT line to receive data
      lda   #Delay_10uS
      jsr   Delay
      brset DS1820_DQ_BIT, DS1820_DQ_PORT, ReadOne           ; Read bit

      clc
      bra   ReadShift

ReadOne:
      sec

ReadShift:
      ror   Temp                                ; Rotate the bit in the TEMP variable
      lda   #Delay_120uS
      jsr   Delay                               ; Wait for rest of time slot
      decx
      bne   ReadBit
      pulx
      pula
      rts

;----------------------------------------------------------------------------------------\
; Function Name:   Delay()                                                               |
;                                                                                        |
; Inputs: A where Delay = A * 5uS                                                        |
;                                                                                        |
; Outputs: None                                                                          |
;                                                                                        |
; Registers affected: A, X                                                               |
;                                                                                        |
; Purpose: This function provides delay from Acc=1 delays 5uS to Acc= 0 1280uS           |
;          ECGST bit in OSCSTAT is checked to see if the external or                     |
;          internal osc. is currently active.                                            |
;                                                                                        |
;          External Osc is assumed to be 8.192MHz or 122ns instruction clock             |
;          Internal Osc is assumed to be 3.200Mhz or 312.5ns instruction clock           |
;                                                                                        |
;----------------------------------------------------------------------------------------/
Delay:
      brclr ECGST, OSCSTAT, IntDelay1stEntry    ; [5]
      bra   Delay1stEntry                       ; [3] Account for function call overhead, skip 12 cycles on first loop
                                              
DelayLoop:                                    
      sta   COPCTL                              ; [3] service cop control register to prevent reset
      brn   *                                   ; [3] nop
      brn   *                                   ; [3] nop
      brn   *                                   ; [3] nop
Delay1stEntry:                                
      brn   *                                   ; [3] nop
      brn   *                                   ; [3] nop
      brn   *                                   ; [3] nop
      brn   *                                   ; [3] nop
      brn   *                                   ; [3] nop
      brn   *                                   ; [3] nop
      brn   *                                   ; [3] nop      
      brn   *                                   ; [3] nop
      nop                                       ; [1] nop
      deca                                      ; [1]
      bne   DelayLoop                           ; [3]
                                              
ExitDelay:                                    
      rts                                       ; [4]

; handle delay when internal osc is in use.   
IntDelayLoop:                                 
      sta   COPCTL                              ; [3] Service COP control register to prevent reset
      brn   *                                   ; [3] nop
      brn   *                                   ; [3] nop      
IntDelay1stEntry:                             
      brn   *                                   ; [3] nop
      deca                                      ; [1]
      bne   IntDelayLoop                        ; [3]
      rts                                       ; [4]

; Delay 1 second
Delay1S:
      pshx
      ldx   #200
Delay1SLoop:      
      lda   #Delay_1mS
      jsr   Delay
      lda   #Delay_1mS
      jsr   Delay
      lda   #Delay_1mS
      jsr   Delay
      lda   #Delay_1mS
      jsr   Delay
      lda   #Delay_1mS
      jsr   Delay

      decx
      bne   Delay1SLoop
      pulx
      rts

;----------------------------------------------------------------------------------------\
; Function Name: CRC16()                                                                 |
;                                                                                        |
; Inputs: H:X pointer to data buffer, A  Number of bytes in buffer                       |
;                                                                                        |
; Outputs: H:X contains the CRC16 of buffer                                              |
;                                                                                        |
; Registers affected:A, H:X                                                              |
;                                                                                        |
; Purpose: Compute the CRC16 on 1 to 255 bytes. Initial CRC value: $FFFF                 |
;          this allows detection of all zeros in data set.                               |
;          Polynomial: $8005                                                             |
; Equivolent C code:                                                                     |
;  while(Count--)                                                                        |
;  {                                                                                     |
;    crc = (crc<<8) ^ CRC16Table[(crc>>8) ^ *DataBuffer++];                              |
;  }                                                                                     |
;  return(crc);                                                                          |
;                                                                                        |  
;----------------------------------------------------------------------------------------/
CRC16:      
      psha                                      ; Save Count parameter
      sthx  pBuffer                             ; Save Data pointer parameter
      ldhx  #$FFFF                              
      sthx  CRC16Value                          ; Init CRC to $FFFF
                                                
CRCTop:                                         
      ldhx  pBuffer                             ; Load H:X with pointer to data
      lda   CRC16Value                          ; Load high byte of CRC16  (crc>>8)
      eor   ,x                                  ; Xor with current buffer data byte
      aix   #1                                  
      sthx  pBuffer                             ; Increment Buffer pointer
      tax                                       ; Tranfer to X for table lookup
      clrh                                      ; Zero top byte of index reg
      lslx                                      ; Multiply word offset into table by 2 for byte offset
      bcc   crc1                                ; Check carry out
      lda   #1                                  
      psha                                      ; If so, put 1 in lowest bit of H reg.
      pulh                                      
crc1:                                           
      lda   CRC16Table,x                        ; Get high byte from CRCTable word
      eor   CRC16Value + 1                      ; Xor with (crc<<8)    
      sta   CRC16Value                          
      aix   #1                                  
      lda   CRC16Table,x                        ; Get low byte from CRCTable word            
      sta   CRC16Value + 1                      ; No Xor needed because Xor with 0 doesn't change value
                                                
      pula                                      ; Retrieve Count from stack
      deca                                      ; Decrement byte count
      psha                                      ; Save back on stack
      bne   CRCTop                              ; If not done, jump to top of loop
                                                
      pula                                      ; Remove count to fix stack before exit
      ldhx  CRC16Value                          ; Return CRC in H:X
      rts    

;----------------------------------------------------------------------------------------\
; Function Name: IRQ_ISR()                                                               |
;                                                                                        |
; Inputs: DISPLAY_ACTIVE_BIT, VideoMemory                                                |
;                                                                                        |
; Outputs: VIDEO_OUT_BIT, NO_VIDEO_PRESENT_ONCE_BIT, NO_VIDEO_PRESENT_TWICE_BIT          |
;                                                                                        |
; Registers affected: A, X, H all saved                                                  |
;                                                                                        |
; Purpose: IRQ_ISR - external IRQ pin Interrupt service routine.                         |
;  This service routine is called every time the falling edge of the horizontal sync     |
;  pulse occures. The start of a new frame is detected when a H-sync pulse occures       |
;  during the vertical blanking period. This will cause the H-sync counter to be reset.  |
;  Once the proper number of sync pulses have been counted, the data in VideoMemory will |
;  be queued on the stack and then shifted out PORTB bit 7.                              |
;----------------------------------------------------------------------------------------/
IRQ_ISR:
; Save h-reg, PORTB and DDRB. Set only b7 to output. Restore at end of interrupt routine.
      pshh
      lda   DDRB
      psha
      lda   PORTB
      psha

      bclr  VIDEO_OUT_BIT, VIDEO_OUT_PORT       ; Clear Dot-output to prevent any flicker when enabling output 
      mov   #%10000000, DDRB                    ; Set only Bit 7 as output. This will prevent the LEDs from blinking
      clrh
 
 
      brset DISPLAY_ACTIVE_BIT, SysSems1, DisplayText

; Check to see if we are in the vertical blanking section  i.e. VERT_SYNC_BIT = 0
      brset VERT_SYNC_BIT, VERT_SYNC_PORT, CheckForStartLine   ; 5 cyc
      
; Vertical blanking active. Clear the horizontal scan counter to zero to get ready for next frame

      clr   HSyncCount
      jmp   ExitIRQisr

CheckForStartLine:
; Currently not displaying. Looking for starting scan line
      inc   HSyncCount
      ldx   HSyncCount

      lda   CONFIG_DISPLAY_AT_BOT
      bne   DispAtBottom

; Display at the top
      cpx   #OSD_AT_TOP_SLINE                   ; Looking for the starting scan line to begin displaying
      beq   FoundStart
      jmp   ExitIRQisr
      
; Display at the bottom      
DispAtBottom:
      cpx   #OSD_AT_BOTTOM_SLINE                ; Looking for the starting scan line to begin displaying
      beq   FoundStart
      jmp   ExitIRQisr

FoundStart:      
      bset  DISPLAY_ACTIVE_BIT, SysSems1
      bclr  TOIE, TSC                           ; Disable interrupts from the TIM Modulo Counter before overlay starts
                                                ; ISSUE (1): Disabling this interrupt breaks the "hung semaphore check"
                                                ;            used to detect loss of video signal. This is currently done to
                                                ;            remove perodic jitter in the OSD.
      clr   LineIndex
      
; Save the return address and modify stack to return to low latency wait instruction
      lda   7, SP
      sta   SavePCH
      lda   8, SP
      sta   SavePCL
     
      lda   #LOW(LowLatency)
      sta   8, SP
      lda   #HIGH(LowLatency)
      sta   7, SP
      jmp   ExitIRQisr

; Displaying text is active. Output single scan line of characters.
DisplayText:

; Short delay until beam is on viewable portion of the screen
      ldx   #LEFT_MARGIN
HDelay:
      decx                                      ;[1]
      bne   HDelay                              ;[3]
;
; Calculate the offset into video memory for first character on line n = LineIndex.
; The same offset is calculated for each two consecutive lines so each dot is two scan lines high.
;
      lda   LineIndex                           ;[3] LineIndex counts from 0 to (2 * Char Height) - 1
      lsra                                      ;[1] Divide by 2 because each vertical dot is two scan lines tall.

      ldx   #DISPLAY_LEN                        ;[2] Calculate the offset of the video memory for the start of a scan line.
      mul                                       ;[6] X:A <-- X * A
      tax
      
; method #4 - Queueing Data on the stack
; Prepare one scan line of data to be displayed and push it on the stack for fast retrieval.
; This will minimize the size of the inter-character space.
;
      lda   VideoMemory+8,x                     ;[3]
      asla                                      ;[1]
      psha                                      ;[2]
      lda   VideoMemory+7,x                     ;[3]
      asla                                      ;[1]
      psha                                      ;[2]
      lda   VideoMemory+6,x                     ;[3]
      asla                                      ;[1]
      psha                                      ;[2]
      lda   VideoMemory+5,x                     ;[3]
      asla                                      ;[1]
      psha                                      ;[2]
      lda   VideoMemory+4,x                     ;[3]
      asla                                      ;[1]
      psha                                      ;[2]
      lda   VideoMemory+3,x                     ;[3]
      asla                                      ;[1]
      psha                                      ;[2]
      lda   VideoMemory+2,x                     ;[3]
      asla                                      ;[1]
      psha                                      ;[2]
      lda   VideoMemory+1,x                     ;[3]
      asla                                      ;[1]
      psha                                      ;[2]
                                     
      lda   VideoMemory,x                       ;[3]
      asla                                      ;[1]
      ldx   #VIDEO_OUT_PORT                     ;[2]
      psha                  
;
; The following code generates the bit stream to create the video overlay
; A loop is not used, to minimize the inter-character time (spacing).
;
; Generate character 0 to DISPLAY_LEN-1

      FOR lable = 1 TO DISPLAY_LEN              ; Repeate code sequence DISPLAY_LEN times!
                                                ;----------
      pula                                      ;[2]     
      sta   ,x                                  ;[2]     
      asla                                      ;[1]     
      sta   ,x                                  ;[2]     
      asla                                      ;[1]     
      sta   ,x                                  ;[2]     
      asla                                      ;[1]     
      sta   ,x                                  ;[2]     
      asla                                      ;[1]     
      sta   ,x                                  ;[2]     
      asla                                      ;[1]     
      sta   ,x                                  ;[2]     
      asla                                      ;[1]     
      sta   ,x                                  ;[2]     
                                                ;-----------
                                                ;[22] cycles  
      ENDFOR                              

      bclr   VIDEO_OUT_BIT, VIDEO_OUT_PORT      

; Increment to next scan line. Check if last line of char has been displayed
      lda   LineIndex                           ;[3]
      inca                                      ;[1]
      sta   LineIndex                           ;[3]
      cmp   #SCAN_LINE_COUNT                    ;[2]
      blo   ExitIRQisr                          ;[3] More lines to process on next interrupt, so get out.

; Last line has been displayed so clear active flag.
      bclr  DISPLAY_ACTIVE_BIT, SysSems1
      bset  OVERLAY_DONE_SIGNAL_BIT, SysSems1
      bset  TOIE, TSC                           ; Re-enable interrupts from the TIM Modulo Counter after overlay is done

; Fix the return address on stack to return to orginally interrupted code.
      lda   SavePCH
      sta   7, SP
      lda   SavePCL
      sta   8, SP

ExitIRQisr:
      bclr  NO_VIDEO_PRESENT_ONCE_BIT, SysSems1   ; Video is present so clear flag
      bclr  NO_VIDEO_PRESENT_TWICE_BIT, SysSems1  ; Video is present so clear flag

      pula
      sta   PORTB                               ; Restore LEDs state
      pula
      sta   DDRB
      pulh
      rti

; Execute "wait" so the interrupt latency is low AND repeatable during video updates.
; without this, too much jitter occures in the video overlay.
; This code is accessed by the IRQ_ISR, by "returning" to it via RTI instruction after fixing up the stack
LowLatency:
      wait
      jmp    LowLatency                         ; This instruction should never be executed, and is here just incase.

;----------------------------------------------------------------------------------------\
; Function Name: TOF_ISR() - Periodic Timer Overflow Interrupt Service Routine           |
;                                                                                        |
; Inputs:  CountInterrupts, TIMSampleTimer, CountMinutes                                 |
;                                                                                        |
; Outputs: DATA_SAMPLE_TIM_BIT, NO_VIDEO_PRESENT_ONCE_BIT, NO_VIDEO_PRESENT_TWICE_BIT    |
;          DISPLAY_CALLSIGN_BIT, DISPLAY_FULL_SETPOINT_BIT                               |
;                                                                                        |
; Registers affected: A, H:X all saved                                                   |
;                                                                                        |
; Purpose: This interrupt service routine assumes that the TIM modulo counter interrupt  |
;           has been programmed at startup to call this function every 4ms.              |
;                                                                                        |
;----------------------------------------------------------------------------------------/
TOF_ISR:
      pshh                                      ;[2] Stack the H register
      lda   TSC                                 ;[3] Clears interrupt Step 1 of 2
      bclr  TOF, TSC                            ;[4] Clears interrupt Step 2 of 2
      dec   CountInterrupts
      bne   ExitTOF
      
; One second has elapsed
      mov   #INTS_PER_ONE_SECOND, CountInterrupts
      dec   TIMSampleTimer
      dec   CountSeconds
      bne   ClockDone

; One Minute has elapsed      
      bset  DISPLAY_FULL_SETPOINT_BIT, SysSems2 ; Display the Setpoint data once per minute
      mov   #60, CountSeconds
      dec   CountMinutes
      bne   ClockDone
      
      mov   #MINS_BETWEEN_HAMCALL_DISP, CountMinutes
      bset  DISPLAY_CALLSIGN_BIT, SysSems1      ;[4] Request to display Amateur Call-sign

ClockDone:
      lda   TIMSampleTimer
      bne   ExitTOF

      lda   CONFIG_UPDT_PERIOD
      sta   TIMSampleTimer
      bset  DATA_SAMPLE_TIM_BIT, SysSems1       ;[4] Set bit one of system status to indicate that a temp and display update requested

; Safety check for video signal, if lost, set flag to prevent video sem from deadlock
      brclr NO_VIDEO_PRESENT_ONCE_BIT, SysSems1, SetOnce  ;[5]
      bset  NO_VIDEO_PRESENT_TWICE_BIT, SysSems1          ;[4] No video signal was present for two periodic system interrupts
SetOnce:
      bset  NO_VIDEO_PRESENT_ONCE_BIT, SysSems1           ;[4] Set flag to test if video has been unexpectedly lost.


      
ExitTOF:
      pulh                                      ;[2]
      rti                                       ;[7]

;----------------------------------------------------------------------------------------\
; Function Name: CatchAll()                                                              |
;                                                                                        |
; Inputs: None                                                                           |
;                                                                                        |
; Outputs: None                                                                          |
;                                                                                        |
; Registers affected: None                                                               |
;                                                                                        |
; Purpose: Spurious Interrupts catch routine                                             |
;                                                                                        |
;----------------------------------------------------------------------------------------/
CatchAll:
      rti


; Temp comp is currenly done with a table to simplify and reduce RAM usage
; A two entry table, with temp in degrees fahrenheit and regulation set point offset
; Both entries are signed.
; Table Usage: If sampled temperature is less than table entry use corresponding offset.
; If not repeat check with next table entry. Offset are stored in binary and can be 
; added directly to the binary setpoint values using signed arithmetic.

GEL_TC_TABLE:
; Temperature (unsigned word), Voltage Offset (signed)

      DC.W  40,   $00B3                         ; < 40    offset = +0.7
      DC.W  50,   $0066                         ; < 50    offset = +0.4
      DC.W  60,   $0033                         ; < 60    offset = +0.2
      DC.W  70,   $0000                         ; < 70    offset =  0.0
      DC.W  80,   $FFE7                         ; < 80    offset = -0.1
      DC.W  90,   $FFB4                         ; < 90    offset = -0.3
      DC.W  100,  $FF9A                         ; < 100   offset = -0.4
      DC.W  110,  $FF80                         ; < 110   offset = -0.5
      DC.W  120,  $FF67                         ; < 120   offset = -0.6
      DC.W  $FF,  $FF34                         ; < MAX   offset = -0.8
      
            
SignOn:
      DC.B 'PVCCV1.1B'      
      
CRC16Table:
      DC.W   $0000, $C0C1, $C181, $0140, $C301, $03C0, $0280, $C241
      DC.W   $C601, $06C0, $0780, $C741, $0500, $C5C1, $C481, $0440
      DC.W   $CC01, $0CC0, $0D80, $CD41, $0F00, $CFC1, $CE81, $0E40
      DC.W   $0A00, $CAC1, $CB81, $0B40, $C901, $09C0, $0880, $C841
      DC.W   $D801, $18C0, $1980, $D941, $1B00, $DBC1, $DA81, $1A40
      DC.W   $1E00, $DEC1, $DF81, $1F40, $DD01, $1DC0, $1C80, $DC41
      DC.W   $1400, $D4C1, $D581, $1540, $D701, $17C0, $1680, $D641
      DC.W   $D201, $12C0, $1380, $D341, $1100, $D1C1, $D081, $1040
      DC.W   $F001, $30C0, $3180, $F141, $3300, $F3C1, $F281, $3240
      DC.W   $3600, $F6C1, $F781, $3740, $F501, $35C0, $3480, $F441
      DC.W   $3C00, $FCC1, $FD81, $3D40, $FF01, $3FC0, $3E80, $FE41
      DC.W   $FA01, $3AC0, $3B80, $FB41, $3900, $F9C1, $F881, $3840
      DC.W   $2800, $E8C1, $E981, $2940, $EB01, $2BC0, $2A80, $EA41
      DC.W   $EE01, $2EC0, $2F80, $EF41, $2D00, $EDC1, $EC81, $2C40
      DC.W   $E401, $24C0, $2580, $E541, $2700, $E7C1, $E681, $2640
      DC.W   $2200, $E2C1, $E381, $2340, $E101, $21C0, $2080, $E041
      DC.W   $A001, $60C0, $6180, $A141, $6300, $A3C1, $A281, $6240
      DC.W   $6600, $A6C1, $A781, $6740, $A501, $65C0, $6480, $A441
      DC.W   $6C00, $ACC1, $AD81, $6D40, $AF01, $6FC0, $6E80, $AE41
      DC.W   $AA01, $6AC0, $6B80, $AB41, $6900, $A9C1, $A881, $6840
      DC.W   $7800, $B8C1, $B981, $7940, $BB01, $7BC0, $7A80, $BA41
      DC.W   $BE01, $7EC0, $7F80, $BF41, $7D00, $BDC1, $BC81, $7C40
      DC.W   $B401, $74C0, $7580, $B541, $7700, $B7C1, $B681, $7640
      DC.W   $7200, $B2C1, $B381, $7340, $B101, $71C0, $7080, $B041
      DC.W   $5000, $90C1, $9181, $5140, $9301, $53C0, $5280, $9241
      DC.W   $9601, $56C0, $5780, $9741, $5500, $95C1, $9481, $5440
      DC.W   $9C01, $5CC0, $5D80, $9D41, $5F00, $9FC1, $9E81, $5E40
      DC.W   $5A00, $9AC1, $9B81, $5B40, $9901, $59C0, $5880, $9841
      DC.W   $8801, $48C0, $4980, $8941, $4B00, $8BC1, $8A81, $4A40
      DC.W   $4E00, $8EC1, $8F81, $4F40, $8D01, $4DC0, $4C80, $8C41
      DC.W   $4400, $84C1, $8581, $4540, $8701, $47C0, $4680, $8641
      DC.W   $8201, $42C0, $4380, $8341, $4100, $81C1, $8081, $4040
  
    Include 'CharGenData.inc'  


;-----------------------------------------------------------------------------------\
; GEL Cell Battery default configuration values                                     |
;                                                                                   |
; Voltage levels are stored as two byte binary values. Their format is MSB.LSB      |
; where the LSB is right of the decimal point.                                      |
; Table of decimal to binary equivolents for LSB                                    |
;  .1 = $19                                                                         |
;  .2 = $33                                                                         |
;  .3 = $4C                                                                         |
;  .4 = $66                                                                         |
;  .5 = $80                                                                         |
;  .6 = $99                                                                         |
;  .7 = $B3                                                                         |
;  .8 = $CC                                                                         |
;  .9 = $E6                                                                         |
;-----------------------------------------------------------------------------------/    
FactoryDefaultConfigData:
DF_CONFIG_CRC:              DC.W    $0000
DF_CONFIG_VERSION:          DC.B    CONFIG_DATA_VER
DF_CONFIG_UPDT_PERIOD:      DC.B    5                ; Seconds      Number of seconds between data readings
DF_CONFIG_MAX_BATT_VOLTS:   DC.W    $0E00            ; 14.0 volts   (GEL) Stop charging when voltage is at or above this value
DF_CONFIG_MIN_BATT_VOLTS:   DC.W    $0C99            ; 12.6 volts   (GEL) Start charging when voltage falls below this value
DF_CONFIG_LVD_VOLTS         DC.W    $0B66            ; 11.4 volts   Low Voltage Disconnect point
DF_CONFIG_RECONNECT_VOLTS   DC.W    $0C99            ; 12.6 volts   Reconnect voltage after LVD has been reached
DF_CONFIG_MAX_BATT_TEMP     DC.W    120              ; Degrees F    Stop charging battery when temp goes above this value
DF_CONFIG_MIN_ON_LIGHT_LEV  DC.B    $18              ; 0 to 255     Minimum light level for loads to be on
DF_CONFIG_MAX_OFF_LIGHT_LEV DC.B    $25              ; 0 to 255     Maximum light level that loads can still be off at
DF_CONFIG_ENABLE_LVD        DC.B    TRUE             ; Boolean      Enable LVD feature
DF_CONFIG_ENABLE_LOAD_CTRL  DC.B    FALSE            ; Boolean      Use photo cell light sensor for automatic load control
DF_CONFIG_ENABLE_CHARGING   DC.B    TRUE             ; Boolean      Enable charging
DF_CONFIG_USE_TEMP_SENSOR   DC.B    TRUE             ; Boolean      Check temperature while charging
DF_CONFIG_ENABLE_OSD        DC.B    TRUE             ; Boolean      Enable the OSD feature.
DF_CONFIG_DISPLAY_AT_BOT    DC.B    TRUE             ; Boolean      OSD on bottom of screen by default, FALSE = OSD at top
DF_CONFIG_ENABLE_CALLSIGN   DC.B    TRUE             ; Boolean      Dislay call sign for one CONFIG_UPDT_PERIOD every 10 min.
DF_CONFIG_CALLSIGN          DC.B    "CALLSIGN ", $00 ; xxxxxxxxx    String, 10 chars total = 9 + null terminator

;-----------------------------------------------------------------------------------\
; Product Configuration data area                                                   |
;                                                                                   |
; This area is used to store product configuration data, which can be uploaded      |
; by the user and written to this flash area. For simplicity the entire page is     |
; reserved for this purpose. The RAM buffer for writing this flash area will be     |
; VideoMemory area, which is not used during config updates                         |
;                                                                                   |
;-----------------------------------------------------------------------------------/
      org   $FD00

CONFIG_DATA:                                       
CONFIG_CRC:                 DC.W    $0000
CONFIG_VERSION:             DC.B    $00
CONFIG_UPDT_PERIOD          DC.B    $00                ; Number of seconds between data readings                                 
CONFIG_MAX_BATT_VOLTS       DC.W    $0000              ; (GEL) Stop charging when voltage is at or above this value
CONFIG_MIN_BATT_VOLTS       DC.W    $0000              ; (GEL) Start charging when voltage falls below this value  
CONFIG_LVD_VOLTS            DC.W    $0000              ; Low Voltage Disconnect point                              
CONFIG_RECONNECT_VOLTS      DC.W    $0000              ; Reconnect voltage after LVD has been reached              
CONFIG_MAX_BATT_TEMP        DC.W    $0000              ; Stop charging battery when temp goes above this value     
CONFIG_MIN_ON_LIGHT_LEV     DC.B    $00                ; Minimum light level for loads to be on                    
CONFIG_MAX_OFF_LIGHT_LEV    DC.B    $00                ; Maximum light level that loads can still be off at        
CONFIG_ENABLE_LVD           DC.B    $00                ; Enable LVD feature                                        
CONFIG_ENABLE_LOAD_CTRL     DC.B    $00                ; Use photo cell light sensor for automatic load control    
CONFIG_ENABLE_CHARGING      DC.B    $00                ; Enable charging                                           
CONFIG_USE_TEMP_SENSOR      DC.B    $00                ; Check temperature while charging                          
CONFIG_ENABLE_OSD           DC.B    $00                ; Enable the OSD feature.                                   
CONFIG_DISPLAY_AT_BOT       DC.B    $00                ; OSD on bottom of screen by default, FALSE = OSD at top    
CONFIG_ENABLE_CALLSIGN      DC.B    $00                ; Dislay call sign for one CONFIG_UPDT_PERIOD every 10 min. 
CONFIG_CALLSIGN             DCB.B   DISPLAY_LEN+1, $00 ; String, 10 chars total = 9 + null terminator               

CONFIG_DATA_SIZE:           EQU     * - CONFIG_DATA

;-----------------------------------------------------------------------------------\
; Internal Oscillator Trim Value                                                    |
;                                                                                   |
;-----------------------------------------------------------------------------------/
      org   $FFC0
 TrimValue:
      DC.B  $70                                 ;Internal OSC trim value

;-----------------------------------------------------------------------------------\
; Vector Table                                                                      |
;                                                                                   |
;-----------------------------------------------------------------------------------/
      org   $FFDE                               ; ADC Conversion Complete Vector (NOT USED)
      DC.W  CatchAll                          
                                              
      org   $FFE0                               ; Keyboard Vector (NOT USED)
      DC.W  CatchAll                          
                                              
      org   $FFF2                               ; TIM Overflow Interrupt Vector
      DC.W  TOF_ISR                           
                                              
      org   $FFF4                               ; TIM Channel 1 Vector (NOT USED)
      DC.W  CatchAll                          
                                              
      org   $FFF6                               ; TIM Channel 0 Vector (NOT USED)
      DC.W  CatchAll                          
                                              
      org   $FFFA                               ; IRQ Pin Interrupt Vector
      DC.W  IRQ_ISR                           
                                              
      org   $FFFC                               ; SWI Vector (NOT USED)
      DC.W  CatchAll                          
                                              
      org   $FFFE                               ; Reset Vector
      DC.W  main
       
END  


; Test configuration for operation with SLI Battery
;DF_CONFIG_CRC:               DC.W    $0000
;DF_CONFIG_VERSION:           DC.B    CONFIG_DATA_VER
;DF_CONFIG_UPDT_PERIOD:       DC.B    5                 ; Seconds      Number of seconds between data readings                   
;DF_CONFIG_MAX_BATT_VOLTS:    DC.W    $0E99             ; 14.6 volts   (SLI Battery) Stop charging when voltage is at or above this value
;DF_CONFIG_MIN_BATT_VOLTS:    DC.W    $0C99             ; 12.6 volts   (SLI Battery) Start charging when voltage falls below this value  
;DF_CONFIG_LVD_VOLTS          DC.W    $0BCC             ; 11.4 volts   Low Voltage Disconnect point                              
;DF_CONFIG_RECONNECT_VOLTS    DC.W    $0C66             ; 12.6 volts   Reconnect voltage after LVD has been reached              
;DF_CONFIG_MAX_BATT_TEMP      DC.W    120               ; Degrees F    Stop charging battery when temp goes above this value     
;DF_CONFIG_MIN_ON_LIGHT_LEV   DC.B    $18               ; 0 to 255     Minimum light level for loads to be on                    
;DF_CONFIG_MAX_OFF_LIGHT_LEV  DC.B    $25               ; 0 to 255     Maximum light level that loads can still be off at        
;DF_CONFIG_ENABLE_LVD         DC.B    $01               ; Boolean      Enable LVD feature                                        
;DF_CONFIG_ENABLE_LOAD_CTRL   DC.B    $01               ; Boolean      Use photo cell light sensor for automatic load control    
;DF_CONFIG_ENABLE_CHARGING    DC.B    $01               ; Boolean      Enable charging                                           
;DF_CONFIG_USE_TEMP_SENSOR    DC.B    $01               ; Boolean      Check temperature while charging                          
;DF_CONFIG_ENABLE_OSD         DC.B    $01               ; Boolean      Enable the OSD feature.                                   
;DF_CONFIG_DISPLAY_AT_BOT     DC.B    $01               ; Boolean      OSD on bottom of screen by default, FALSE = OSD at top    
;DF_CONFIG_ENABLE_CALLSIGN    DC.B    $01               ; Boolean      Dislay call sign for one CONFIG_UPDT_PERIOD every 10 min. 
;DF_CONFIG_CALLSIGN           DC.B    "N9PIX    ", $00  ; xxxxxxxxx    String, 10 chars total = 9 + null terminator              

;----------------------------------------------------------------------------------------\
; Function Name:                                                                         |
;                                                                                        |
; Inputs:                                                                                |
;                                                                                        |
; Outputs:                                                                               |
;                                                                                        |
; Registers affected:                                                                    |
;                                                                                        |
; Purpose:                                                                               |
;                                                                                        |
;----------------------------------------------------------------------------------------/
