/*
** CLEP.c
**
** This file is part of CLEP - the Command Line and Environment Processor
** Copyright 1997, William Sheldon Simms III
**
** CLEP is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2, or (at your option)
** any later version.
**
** CLEP is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with GNU Emacs; see the file COPYING.  If not, write to
** the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
** Boston, MA 02111-1307, USA.
*/


#include <assert.h>
#include <ctype.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include "CLEP.h"


/********************************************************************
**
**  CLEP module function prototypes
**
*********************************************************************/


/* CLEP's default error handler - should be replaced by user program */
static void CLEP_Error( int error, int index, char **which );

/* CLEP environment processing functions */
static int  CLEP_Match_Var_Value ( int valid_vars_index, char *value );
static void CLEP_Environment     ( void );

/* CLEP command line argument processing functions */
static int  CLEP_Match_Arg_Value          ( int valid_index );
static void CLEP_Match_Top_Level_Argument ( void );
static void CLEP_Command_Line             ( int num_args, char **arg_array );


/********************************************************************
**
**  CLEP module variable definitions
**
*********************************************************************/


/* Used for debugging purposes - ensures that CLEP_Environment() */
/* is called before CLEP_Command_Line()                          */
static short CLEP_Processed_Environment = 0;

/* CLEP_Arg_Index keeps track of which command line argument, */
/* out of the total number, have been processed.              */         
static int     CLEP_Arg_Index;

/* CLEP_Arg_Array points to the next argument string to be processed */
static char ** CLEP_Arg_Array;

/* CLEP_Top_Level_Arg points to the current top level argument */
static char ** CLEP_Top_Level_Arg;

/* CLEP_Valid_Vars is an array of environment variables that the user     */
/* program wants CLEP to match. See the description of CLEP_Var in CLEP.h */
static CLEP_Var *CLEP_Valid_Vars;

/* CLEP_Valid_Args is an array of top-level arguments that the user       */
/* program wants CLEP to match. See the description of CLEP_Arg in CLEP.h */
static CLEP_Arg *CLEP_Valid_Args;

/* CLEP_Error_Hook is a pointer to a function that CLEP calls when it cannot */
/* successfully process an environment variable or command line argument     */
static CLEP_Err_Handler CLEP_Error_Hook = CLEP_Error;


/********************************************************************
**
**  CLEP exported variable definitions
**
*********************************************************************/


/* CLEP_Invocation_Name is set by CLEP to point at a string representing */
/* the name of the program being run, as it was invoked from the shell.  */
char *CLEP_Invocation_Name;


/********************************************************************
**
**  CLEP module functions
**
*********************************************************************/


/*
** void CLEP_Error( int error, int index, char **which )
**
** CLEP_Error() is a default error handler that prints a short error
** message based on which error occurs and then exits. Most programs
** using CLEP should install their own error handler rather than
** use the default.
*/

static char *CLEP_Error_Strings[] = {
"CLEP: You should never see this error. (oops!)\n",
"The environment variable \"%s\" has a bad value (\"%s\").\n",
"The argument \"%s\" is not valid.\n",
"The argument \"%s\" cannot be followed by \"%s\".\n"
};

static void CLEP_Error( int error, int index, char **which )
{
    switch( error )
    {
    default:
    case CLEP_NO_ERROR:
	fputs( CLEP_Error_Strings[error], stderr );
	break;

    case CLEP_ENV_BAD_VALUE:
	fprintf( stderr, CLEP_Error_Strings[error],
		 CLEP_Valid_Vars[index].valid_variable, *which );
	return;

    case CLEP_ARG_UNKNOWN_ARG:
	fprintf( stderr, CLEP_Error_Strings[error], *which );
	break;

    case CLEP_ARG_BAD_VALUE:
	fprintf( stderr, CLEP_Error_Strings[error], *which,
		 *(which+1) );
	break;
    }

    exit( 1 );
}

/*
** int CLEP_Match_Var_Value ( int valid_vars_index, char *value )
**
** CLEP_Match_Var_Value() attempts to match the value of an already matched
** environment variable (indexed in CLEP_Valid_Vars[] by valid_vars_index)
** with the array of valid values supplied in CLEP_Valid_Vars[] for the
** variable.
*/

static int CLEP_Match_Var_Value ( int valid_vars_index, char *value )
{
    int  i;
    char *this_value;
    char *valid_value;

    for( i = 0; CLEP_Valid_Vars[valid_vars_index].valid_values[i]; i++ )
    {
	this_value  = value;
	valid_value = CLEP_Valid_Vars[valid_vars_index].valid_values[i];

	/* compare the strings case-insensitve */

	while( *valid_value && *this_value &&
	       toupper( *valid_value ) == toupper( *this_value ) )
	{
	    this_value++;
	    valid_value++;
	}

	/* found a match */

	if( *valid_value == 0 && *this_value == 0 )
	    break;

	/* no match - try the next one*/
    }

    /* no match */

    if ( !CLEP_Valid_Vars[valid_vars_index].valid_values[i] )
	CLEP_Error_Hook( CLEP_ENV_BAD_VALUE, valid_vars_index, &value );

    /* match, return the index of the choice that matched */

    return i;
}

/*
** void CLEP_Environment( void )
**
** CLEP_Environment() is the environment processing main function. It loops
** through the supplied valid environment variables and, for each, calls
** the supplied handler, after matching the value if an array of valid values
** was supplied.
*/

static void CLEP_Environment( void )
{
    int  i;
    int  value_index;
    char *value;

    for( i = 0; CLEP_Valid_Vars[i].valid_variable; i++ )
    {
	/* get the value of the next valid environment variable */
	value = getenv( CLEP_Valid_Vars[i].valid_variable );

	/* if there was a value, and it wasn't an empty string */
	if ( value && *value )
	{
	    if ( CLEP_Valid_Vars[i].valid_values )
		value_index = CLEP_Match_Var_Value( i, value );
	    else
		value_index = 0;

	    CLEP_Valid_Vars[i].handler( CLEP_Valid_Vars[i].user_int,
					value_index, value );
	}
    }

    /* remember that environment processing has been done */
    CLEP_Processed_Environment = 1;
}

/*
** int CLEP_Match_Arg_Value( int valid_arg_index )
**
** CLEP_Match_Arg_Value() assumes that the next argument as returned by
** CLEP_Next_Argument is supposed to be one of the values in
** CLEP_Valid_Args[valid_arg_index].valid_values[] and seeks to make a match.
** If it finds a match, it returns the index of the matching choice in
** CLEP_Valid_Args[valid_arg_index].valid_values[].
*/

static int CLEP_Match_Arg_Value( int valid_arg_index )
{
    int  i;
    char *value_ptr;
    char *this_value;
    char *valid_value;

    /* Get a pointer to the next arg (which should be a valid value */
    /* of the previous arg)                                         */

    value_ptr = CLEP_Next_Argument();

    /* check the string pointed to by value_ptr against valid values */
    /* in CLEP_Valid_Args[valid_arg_index] until there's a match or  */
    /* until the valid values have been exhausted                    */

    for( i = 0; CLEP_Valid_Args[valid_arg_index].valid_values[i]; i++ )
    {
	/* get the next valid value */
	valid_value = CLEP_Valid_Args[valid_arg_index].valid_values[i];

	/* re-get the value we're checking */
	this_value = value_ptr;

	/* compare strings case-insensitve */
	while( *valid_value && *this_value &&
	       toupper( *valid_value ) == toupper( *this_value ) )
	{
	    valid_value++;
	    this_value++;
	}

	/* a match */
	if( *valid_value == 0 && *this_value == 0 )
	    break;

	/* no match - try the next one*/
    }

    /* no match */
    if ( !CLEP_Valid_Args[valid_arg_index].valid_values[i] )
    {
	CLEP_Error_Hook( CLEP_ARG_BAD_VALUE, CLEP_Arg_Index-1,
			 CLEP_Top_Level_Arg );
    }

    /* match, return the index of the choice that matched */
    return i;
}

/*
** void CLEP_Match_Top_Level_Argument ( void )
**
** CLEP_Match_Top_Level_Argument() attempts to match the next command line
** argument with one of the valid top level arguments supplied in
** CLEP_Valid_Args[]. If it finds a match, it calls the handler specified in
** CLEP_Valid_Args[] after matching the argument's value, if an array of
** valid values was supplied in CLEP_Valid_Args[].
*/

static void CLEP_Match_Top_Level_Argument ( void )
{
    int  i;
    int  value_index;
    char *arg_ptr;
    char *this_arg;
    char *valid_arg;

    /* Get the argument we need to match */
    arg_ptr = CLEP_Next_Argument();

    /* check this_arg against CLEP_Valid_Args[] until there's a match */
    /* or until CLEP_Valid_Args[] has been exhausted                  */

    for( i = 0; CLEP_Valid_Args[i].valid_argument; i++ )
    {
	/* get the next valid argument */
	valid_arg = CLEP_Valid_Args[i].valid_argument;

	/* re-get the arg we're checking */
	this_arg = arg_ptr;

	/* compare strings case-insensitve */
	while( *valid_arg && *this_arg &&
	       toupper( *valid_arg ) == toupper( *this_arg ) )
	{
	    this_arg++;
	    valid_arg++;
	}

	/* a full match */
	if( *valid_arg == 0 && *this_arg == 0 )
	    break;

	/* no match - try the next one*/
    }

    /* no match */
    if ( !CLEP_Valid_Args[i].valid_argument )
	CLEP_Error_Hook( CLEP_ARG_UNKNOWN_ARG,
			 CLEP_Arg_Index-1, CLEP_Top_Level_Arg );

    /* do we need to try to match the next argument as a value for this one? */
    if ( CLEP_Valid_Args[i].valid_values )
    {
	/* if so, match the choice */
	value_index = CLEP_Match_Arg_Value( i );

	/* and call the handler */
	CLEP_Valid_Args[i].handler( CLEP_Valid_Args[i].user_int, value_index );
    }
    else
    {
	/* if not, just call the handler */
	CLEP_Valid_Args[i].handler( CLEP_Valid_Args[i].user_int, 0 );
    }
}

/*
** void CLEP_Command_Line( int num_args, char **arg_array )
**
** CLEP_Command_Line() is the top-level command line argument
** processor. It remembers the invocation name of the program
** and loops through the arguments, calling CLEP_Match_Argument()
** for each one.
*/

static void CLEP_Command_Line( int num_args, char **arg_array )
{
    /* Debug - ensure proper call order */
    assert( CLEP_Processed_Environment );

    /* set up counter and pointer to current arg */
    CLEP_Arg_Index = 0;
    CLEP_Arg_Array = arg_array;

    /* Get name by which the program was invoked */
    CLEP_Invocation_Name = CLEP_Next_Argument();

    /* loop through the top level arguments */
    while( CLEP_Arg_Index < num_args )
    {
	CLEP_Top_Level_Arg = CLEP_Arg_Array;
	CLEP_Match_Top_Level_Argument();
    }
}


/********************************************************************
**
**  CLEP exported functions
**
*********************************************************************/


/*
** char *CLEP_Next_Argument( void )
**
** CLEP_Next_Argument() returns the current argument string and then
** advances the argument counter and pointer to the next argument
*/

char *CLEP_Next_Argument( void )
{
    /* Get a ptr to the argument */
    char *arg = *CLEP_Arg_Array;

    /* advance to the next argument */
    CLEP_Arg_Index++;
    CLEP_Arg_Array++;

    /* return the argument */
    return arg;
}

/*
** void CLEP_Main( CLEP_Var *valid_vars, CLEP_Arg *valid_args,
**	           CLEP_Err_Handler user_error_handler,
**	           int argc, char **argv )
**
** CLEP_Main() accepts from the user program an array of CLEP_Var structures,
** an array of CLEP_Arg structures, the address of an error handling function,
** and argc and argv arguments as passed to main(). It installs the error
** handling function and then calls functions to process the environment and
** command line.
*/

void CLEP_Main( CLEP_Var *valid_vars, CLEP_Arg *valid_args,
		CLEP_Err_Handler user_error_handler,
		int argc, char **argv )
{
    CLEP_Valid_Vars = valid_vars;
    CLEP_Valid_Args = valid_args;

    if ( user_error_handler )
	CLEP_Error_Hook = user_error_handler;

    CLEP_Environment();
    CLEP_Command_Line( argc, argv );
}

/*
** End of CLEP.c
*/
