/*
** CLEP.h
**
** This file is part of CLEP - the Command Line and Environment Processor
** Copyright 1997, William Sheldon Simms III
**
** CLEP is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2, or (at your option)
** any later version.
**
** CLEP is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with GNU Emacs; see the file COPYING.  If not, write to
** the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
** Boston, MA 02111-1307, USA.
*/

#ifndef INCLUDED_CLEP_H
#define INCLUDED_CLEP_H

/*
** CLEP Errors:
**
** One of these enumeration constants will be passed as the second argument
** to the function referenced by CLEP_Error_Hook. Constants with a prefix
** of CLEP_ENV refer to errors in environment processing and constants with
** a prefix of CLEP_ARG refer to errors in command line argumenr processing.
**
** CLEP_NO_ERROR is just a placeholder and is never passed to an error
** handler through CLEP_Error_Hook.
*/

enum { CLEP_NO_ERROR, CLEP_ENV_BAD_VALUE,
       CLEP_ARG_UNKNOWN_ARG, CLEP_ARG_BAD_VALUE };

/*
** CLEP function pointer types
**
** CLEP_Var_Handler is a pointer to a function that handles matched
** environment variables. When CLEP matches an environment variable with one
** of the valid variables supplied in CLEP_Valid_Vars[], CLEP calls the
** associated handler through the supplied hook of type CLEP_Var_Handler,
** passing it the supplied user_int, the index of the valid value matched by
** the environment variable's value (if a list of valid values was supplied
** and if the variable's value matches one of them), and a pointer to the
** value string.
**
** CLEP_Arg_Handler is a pointer to a function that handles matched top-level
** arguments. When CLEP matches a top level argument with one of the valid
** arguments supplied in CLEP_Valid_Args[], CLEP calls the associated handler
** through the supplied hook of type CLEP_Arg_Handler, passing it the supplied
** user_int and the index of the valid value matched by the top-level
** argument's value (if a list of valid values was supplied, if the top-
** level argument has a value, and if the argument's value matches one of the
** valid values)
**
** CLEP_Err_Handler is a pointer to a function that CLEP calls when it cannot
** successfully process an environment variable or command line argument. If
** the failure was in processing an environment variable, the error handler
** is passed the error number, the index of the environment variable in
** CLEP_Valid_Vars, and the address of a pointer to the value of the
** environment variable. If the failure was in processing a command line
** argument, the error handler is passed the error number, the index of the
** failing argument in the argv[] array and the address of a pointer to the
** current top-level argument string
*/

typedef void (*CLEP_Var_Handler)( int    user_int,
				  int    value_index,
				  char * value_string );

typedef void (*CLEP_Arg_Handler)( int user_int,
				  int value_index );

typedef void (*CLEP_Err_Handler)( int     error,
				  int     index,
				  char ** which );

/*
** CLEP valid variable and valid argument types
**
** Programs using CLEP must pass arrays of these types to CLEP_Main() so that
** CLEP knows what environment variables and command line arguments to look
** for. Each contains: a pointer to a string representing one valid variable
** or argument, a user_int that CLEP will pass back to the program's handler
** for matches of the variable or argument, a pointer to the handler, and a
** pointer to an array of valid values for the variable or argument, if the
** calling program wants CLEP to attempt to match a valid value when it
** matches a valid variable or argument.
*/

typedef struct
{
    char *            valid_variable;
    int               user_int;
    CLEP_Var_Handler  handler;
    char **           valid_values;

} CLEP_Var;

typedef struct
{
    char *            valid_argument;
    int               user_int;
    CLEP_Arg_Handler  handler;
    char **           valid_values;

} CLEP_Arg;

/*
** CLEP_Invocation_Name is set by CLEP to point at a string representing the
** name of the program being run, as it was invoked from the shell.
*/

extern char *CLEP_Invocation_Name;


/*
** CLEP function prototypes
*/

char *CLEP_Next_Argument( void );

void CLEP_Main( CLEP_Var *valid_vars, CLEP_Arg *valid_args,
		CLEP_Err_Handler user_error_handler,
		int argc, char **argv );

#endif /* #ifndef INCLUDED_CLEP_H */

/*
** End of CLEP.h
*/

