
#include <ctype.h>
#include <fcntl.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "CLEP.h"
#include "cmdline.h"
#include "em65.h"       /* char *em65_version_string
			   char *em65_copyright_string */
#include "em65types.h"  /* TRUE, FALSE */

static void handle_env_disk( int, int, char * );
static void handle_env_debug( int, int, char * );
static void handle_env_reset( int, int, char * );
static void handle_env_rom_file( int, int, char * );
static void handle_env_clockrate( int, int, char * );
static void handle_env_clocktime( int, int, char * );
static void handle_env_rom_address( int, int, char * );

static void handle_arg_rom( int, int );
static void handle_arg_disk( int, int );
static void handle_arg_help( int, int );
static void handle_arg_debug( int, int );
static void handle_arg_reset( int, int );
static void handle_arg_clockrate( int, int );
static void handle_arg_clocktime( int, int );

static void bad_option_handler( int error, int index, char **which );

short em65_debug           = TRUE;  /* -debug on|off           */
short em65_realtime_clock  = TRUE;  /* -clocktime real|virtual */
short em65_clock_rate      = 30;    /* -clockrate <number>     */

unsigned short em65_rom_address   = 0x0000;  /* -rom <address> <filename> */
unsigned short em65_reset_address = 0xf000;  /* -reset <address>          */

char *em65_rom_filename =  NULL;         /* -rom <address> <filename> */
char *em65_disk_filename = "em65_disk";  /* -disk <filename>          */

char *debug_choices[] = { "off", "on", NULL };
char *clocktime_choices[] = { "virtual", "real", NULL };

CLEP_Arg arg_array[] =
{
    { "-debug",     0, handle_arg_debug,     debug_choices },
    { "-clocktime", 0, handle_arg_clocktime, clocktime_choices },
    { "-clockrate", 0, handle_arg_clockrate, NULL },
    { "-rom",       0, handle_arg_rom,       NULL },
    { "-reset",     0, handle_arg_reset,     NULL },
    { "-disk",      0, handle_arg_disk,      NULL },
    { "-help",      0, handle_arg_help,      NULL },
    { "-h",         0, handle_arg_help,      NULL },
    { "-?",         0, handle_arg_help,      NULL },
    { NULL,         0, NULL,                 NULL }
};

CLEP_Var var_array[] =
{
    { "EM65_DEBUG",         0, handle_env_debug,         debug_choices },
    { "EM65_CLOCKTIME",     0, handle_env_clocktime,     clocktime_choices },
    { "EM65_CLOCKRATE",     0, handle_env_clockrate,     NULL },
    { "EM65_ROM_FILE",      0, handle_env_rom_file,      NULL },
    { "EM65_ROM_ADDRESS",   0, handle_env_rom_address,   NULL },
    { "EM65_RESET_ADDRESS", 0, handle_env_reset,         NULL },
    { "EM65_DISK_FILE",     0, handle_env_disk,          NULL },
    { NULL,                 0, NULL,                     NULL }
};

static void handle_env_debug( int ignore1, int which, char *ignore2 )
{
    if ( which == 0 )
	em65_debug = FALSE;
    else
	em65_debug = TRUE;
}

static void handle_env_clocktime( int ignore1, int which, char *ignore2 )
{
    if ( which == 0 )
	em65_realtime_clock = FALSE;
    else
	em65_realtime_clock = TRUE;
}

static void handle_env_clockrate( int ignore1, int ignore2, char *value )
{
    long rate;

    /* what rate has been requested? */
    rate = strtol( value, NULL, 10 );
    
    /* remember the requested rate */
    if ( rate >= 1 && rate <= 100 )
	em65_clock_rate = (short)rate;
}

static void handle_env_rom_file( int ignore1, int ignore2, char *value )
{
    /* remember the requested disk filename */
    if ( value && *value )
	em65_rom_filename = value;
}

static void handle_env_disk( int ignore1, int ignore2, char *value )
{
    /* remember the requested disk filename */
    if ( value && *value )
	em65_disk_filename = value;
}

static void handle_env_rom_address( int ignore1, int ignore2, char *value )
{
    long address;
    char *sptr;

    /* accept a leading '$' or "0x" */
    if ( '$' == *value )
    {
	value++;
    }
    else if ( '0' == *value && 'X' == toupper( *(value+1) ))
    {
	value += 2;
    }

    /* handle case of address 0 (because idiots designed strtol) */
    for( sptr = value; *sptr != 0; sptr++ )
	if ( *sptr != '0' ) break;

    /* NUL, not the character '0' */
    if ( *sptr == 0 )
	address = 0;
    else
	address = strtol( value, NULL, 16 );
    
    /* remember the requested rom load address */
    if ( address >= 0 && address <= 0xffff )
	em65_rom_address = (unsigned short)address;
}

static void handle_env_reset( int ignore1, int ignore2, char *value )
{
    long address;
    char *sptr;

    /* accept a leading '$' or "0x" */
    if ( '$' == *value )
    {
	value++;
    }
    else if ( '0' == *value && 'X' == toupper( *(value+1) ))
    {
	value += 2;
    }

    /* handle case of address 0 (because idiots designed strtol) */
    for( sptr = value; *sptr != 0; sptr++ )
	if ( *sptr != '0' ) break;

    /* NUL, not the character '0' */
    if ( *sptr == 0 )
	address = 0;
    else
	address = strtol( value, NULL, 16 );
    
    /* remember the requested reset vector */
    if ( address >= 0 && address <= 0xffff )
	em65_reset_address = (unsigned short)address;
}

static void handle_arg_debug( int ignore1, int which )
{
    if ( which == 0 )
	em65_debug = FALSE;
    else
	em65_debug = TRUE;
}

static void handle_arg_clocktime( int ignore1, int which )
{
    if ( which == 0 )
	em65_realtime_clock = FALSE;
    else
	em65_realtime_clock = TRUE;
}

static void handle_arg_disk( int ignore1, int ignore2 )
{
    /* remember the requested disk filename */
    em65_disk_filename = CLEP_Next_Argument();
}

static void handle_arg_clockrate( int ignore1, int ignore2 )
{
    long rate;
    char *rate_str;

    /* get the rate str */
    rate_str = CLEP_Next_Argument();

    if ( rate_str && *rate_str )
    {
	/* what rate has been requested? */
	rate = strtol( rate_str, NULL, 10 );
    }
    else
    {
	/* cause failure */
	rate = 0;
    }

    /* don't allow ridiculous rates */
    if ( rate < 1 || rate > 100 )
    {
	fputs( "Bad clock rate specified.\n", stderr );
	fprintf( stderr, "Type '%s -help' for more information about allowable "
		 "clock rates\n", CLEP_Invocation_Name );

	exit( 1 );
    }

    /* remember the requested rate */
    em65_clock_rate = (short)rate;
}

static void handle_arg_reset( int ignore1, int ignore2 )
{
    long address;
    char *sptr;
    char *addr_str;

    /* get the reset address string */
    addr_str = CLEP_Next_Argument();

    if ( addr_str && *addr_str )
    {
	/* accept leading 0x */
	if ( '0' == *addr_str && 'X' == toupper( *(addr_str+1) ))
	{
	    addr_str += 2;
	}

	/* handle case of address 0 (because idiots designed strtol) */
	for( sptr = addr_str; *sptr != 0; sptr++ )
	    if ( *sptr != '0' ) break;

	/* NUL, not the character '0' */
	if ( *sptr == 0 )
	    address = 0;
	else
	    address = strtol( addr_str, NULL, 16 );
    }
    else
    {
	/* cause failure */
	address = -1;
    }

    /* don't allow impossible addresses */
    if ( address < 0 || address > 0xffff )
    {
	fputs( "Bad reset vector specified. The reset vector must be specified\n",
	       stderr );
	fputs( "as a hexadecimal number of 4 or fewer digits\n", stderr );
	exit( 1 );
    }

    /* remember the requested reset vector */
    em65_reset_address = (unsigned short)address;    
}

static void handle_arg_rom( int ignore1, int ignore2 )
{
    int  f;
    long address;
    char *sptr;
    char *addr_str;

    /* get my own pointer to the address argument string */
    addr_str = CLEP_Next_Argument();

    if ( addr_str && *addr_str )
    {
	/* accept leading 0x */
	if ( '0' == *addr_str && 'X' == toupper( *(addr_str+1) ))
	{
	    addr_str += 2;
	}

	/* handle case of address 0 (because idiots designed strtol) */
	for( sptr = addr_str; *sptr != 0; sptr++ )
	    if ( *sptr != '0' ) break;

	/* NUL, not the character '0' */
	if ( *sptr == 0 )
	    address = 0;
	else
	    address = strtol( addr_str, NULL, 16 );
    }
    else
    {
	/* cause failure */
	address = -1;
    }
    
    /* don't allow impossible addresses */
    if ( address < 0 || address > 0xffff )
    {
	fputs( "Bad ROM load address specified. The rom load address must be\n",
	       stderr );
	fputs( "specified as a hexadecimal number of 4 or fewer digits\n", stderr );
	exit( 1 );
    }

    /* remember the requested rom load address */
    em65_rom_address = (unsigned short)address;
    
    /* remember the requested rom filename */
    em65_rom_filename = CLEP_Next_Argument();

    /* make sure the file exists */
    f = open( em65_rom_filename, O_RDONLY );
    if ( f == -1 )
    {
	fputs( "The specified ROM file does not exist.\n", stderr );
	exit( 1 );
    }

    close( f );
}

static void handle_arg_help( int ignore1, int ignore2 )
{
    /* print some brief help information */

    printf( "\n%s - %s %s (%s)\n\n", em65_version_string, em65_copyright_string,
	    em65_author_string, em65_email_string );
    printf( "Usage: %s [ options ]\n\n", CLEP_Invocation_Name );
    puts(   "Where [ options ] is one or more of:" );
    puts(   " -debug on|off              Enable or disable the built-in debugger" );
    puts(   " -clocktime real|virtual    Set clock interrupt to measure real time," );
    puts(   "                              or process virtual time." );
    puts(   " -clockrate <number>        Set clock interrupt rate to <number> Hz (1-100)" );
    puts(   "                              NOTE: although rates of up to 100 Hz are" );
    puts(   "                              accepted on the command line, rates higher" );
    puts(   "                              than that of the system clock will not work." );
    puts(   "                              System clock rates are often only 50 or 60 Hz." );
    puts(   " -rom <address> <filename>  Load the ROM file <filename> at <address> (hex)" );
    puts(   " -reset <address>           Load the reset vector with <address> (hex)" );
    puts(   " -disk <filename>           Use the file <filename> as the virtual disk\n" );
    puts(   "For example, option default can be explicitly selected by:\n" );
    printf( "%s -debug on -clocktime real -clockrate 30 -reset F000 -disk em65_disk\n\n",
	    CLEP_Invocation_Name );

    /* exit the program */
    exit( 0 );
}

static void bad_option_handler( int error, int index, char **which )
{
    if ( error == CLEP_ENV_BAD_VALUE )
	return;

    if ( error == CLEP_ARG_UNKNOWN_ARG )
	fprintf( stderr, "Bad argument \"%s\"\n", *which );
    else if ( error == CLEP_ARG_BAD_VALUE )
	fprintf( stderr, "Bad argument value \"%s\"\n", *(which+1) );

    fprintf( stderr, "Type '%s -help' for more information about valid "
	     "command line arguments\n", CLEP_Invocation_Name );

    exit( 1 );
}

void process_options( int argc, char **argv )
{
    CLEP_Main( var_array, arg_array, bad_option_handler, argc, argv );
}






