
#include <stdio.h>
#include <stdlib.h>
#include "cmdline.h"
#include "cpu.h"
#include "em65.h"
#include "em65types.h"
#include "debugger.h"
#include "tables.h"
#include "terminal.h"

/* 65c02 registers */

unsigned char A,         /* Accumulator     */
              X, Y,      /* Index Registers */
              P = 0x20,  /* Status Byte     */
              S;         /* Stack Pointer   */

unsigned short emPC;

short C, Z, I, D, B, V, N;

/* variables */

unsigned char  arg;
unsigned char  opcode;
unsigned short effective_address;

/* 65c02 memory space */
unsigned char *memory;

/* Macros to read and write. Addresses in the range IO_BOTTOM to */
/* IO_TOP are used for memory mapped I/O.                        */

#define WRITE( addr, byte )                         \
{                                                   \
    if ( addr >= IO_BOTTOM && addr <= IO_TOP )      \
	io_write[(unsigned char)(addr)]( (byte) );  \
    else                                            \
	memory[(addr)] = (byte);                    \
}

#define READ( addr )                                                       \
( addr >= IO_BOTTOM && addr <= IO_TOP ? io_read[(unsigned char)(addr)]() : \
  memory[(addr)] )

/* Macros that define the simpler 65c02 addressing modes. They evaluate */
/* to the instruction's effective address.                              */

#define IMMEDIATE (emPC++)

#define ZERO_PAGE (emPC++, READ(emPC-1))

#define ABSOLUTE  (emPC+=2, 256*READ(emPC-1) + READ(emPC-2))

#define INDEXED_X (emPC+=2, 256*READ(emPC-1) + READ(emPC-2) + X)

#define INDEXED_Y (emPC+=2, 256*READ(emPC-1) + READ(emPC-2) + Y)

#define ZERO_PAGE_INDEXED_X (emPC++, (unsigned char)(READ(emPC-1) + X))

#define ZERO_PAGE_INDEXED_Y (emPC++, (unsigned char)(READ(emPC-1) + Y))

#define RELATIVE ((unsigned short)((short)emPC + 1 + (signed char)READ(emPC)))

/* Macros to push bytes to and pull bytes from the 65c02 stack */

#define PUSH( b ) (memory[ 0x100 + S-- ] = (unsigned char)(b))
#define PULL()    ((unsigned char)memory[ 0x100 + (++S) ])

/* Macro to set N and Z flags */

#define SET_FLAGS_NZ( a )  \
{                          \
    N = (a) & 0x80;        \
    Z = (a) == 0;          \
}

/* addressing mode functions */

unsigned short ZERO_PAGE_INDIRECT( void )
{
    unsigned char temp_byte = READ(emPC); emPC++;
    return 256 * READ( (unsigned char)( temp_byte + 1 ) ) +
	   READ( temp_byte );
}

unsigned short INDEXED_INDIRECT( void )
{
    unsigned char temp_byte = READ(emPC) + X; emPC++;
    return 256 * READ( (unsigned char)( temp_byte + 1 ) ) +
	   READ( temp_byte );
}

unsigned short INDIRECT_INDEXED( void )
{
    unsigned char temp_byte = READ(emPC); emPC++;
    return 256 * READ( (unsigned char)( temp_byte + 1 ) ) +
	   READ( temp_byte ) + Y;
}

unsigned short ABSOLUTE_INDIRECT( void )
{
    unsigned short temp_address;
    
    temp_address = 256*READ( emPC + 1 ) + READ( emPC );
    return 256*READ( temp_address + 1 ) + READ( temp_address );
}

unsigned short ABSOLUTE_INDEXED_INDIRECT( void )
{
    unsigned short temp_address;
    
    temp_address = 256*READ( emPC + 1 ) + READ( emPC ) + X;
    return 256*READ( temp_address + 1 ) + READ( temp_address );
}

/* helper functions */

void add_binary( unsigned char arg )
{
    unsigned char old_A = A;

    A = A + C + arg;

    N = A & 0x80;
    Z = !A;

    if ( A < old_A )
	C = 1;
    else
	C = 0;

    if ( (signed char)arg < 0 && (signed char)A > (signed char)old_A )
	V = 1;
    else if ( (signed char)A < (signed char)old_A )
	V = 1;
    else
	V = 0;
}

void add_decimal( unsigned char arg )
{
    unsigned char old_A = A;

    /* add 'ones' digits */
    A = (old_A & 0xf) + C + (arg & 0xf);

    /* carry if needed */
    if ( A > 9 )
	A = 0x10 + ( A - 10 );

    /* add 'tens' digits */
    A = A + (old_A & 0xf0) + (arg & 0xf0);

    /* set flags */
    Z = !A;

    if ( A < old_A )
	C = 1;
    else
	C = 0;
}

void subtract_decimal( unsigned char arg )
{
    unsigned char old_A = A;

    /* subtract 'ones' digits */
    A = A - (1-C) - (arg & 0xf);

    /* if borrow is needed */
    if ( (A & 0x0f) > (old_A & 0xf) )
    {
	/* change 0x0f -> 0x09, etc. */
	A -= 6;
	
	/* borrow from 'tens' digit */
	A -= 0x10;
    }

    /* subtract 'tens' digits */
    A -= (arg & 0xf0);

    /* set flags */
    Z = !A;

    if ( A > old_A )
	C = 0;
    else
	C = 1;
}

void subtract_binary( unsigned char arg )
{
    unsigned char old_A = A;

    A = A - (1-C) - arg;

    N = A & 0x80;
    Z = !A;

    if ( A > old_A )
	C = 0;
    else
	C = 1;

    if ( (signed char)arg < 0 && (signed char)A < (signed char)old_A )
	V = 1;
    else if ( (signed char)A > (signed char)old_A )
	V = 1;
    else
	V = 0;
}

void compare( unsigned char cpureg, unsigned char arg )
{
    unsigned char result;

    result = cpureg - arg;

    N = result & 0x80;
    Z = !result;

    if ( result > cpureg )
	C = 0;
    else
	C = 1;
}

unsigned char build_P( void )
{
    P = 0x20;
    if ( N ) P |= 0x80;
    if ( V ) P |= 0x40;
    if ( B ) P |= 0x10;
    if ( D ) P |= 0x08;
    if ( I ) P |= 0x04;
    if ( Z ) P |= 0x02;
    if ( C ) P |= 0x01;

    return P;
}

void unbuild_P( unsigned char P )
{
    N = P & 0x80;
    V = P & 0x40;
    B = P & 0x10;
    D = P & 0x08;
    I = P & 0x04;
    Z = P & 0x02;

    if ( P & 0x01 )
	C = 1;
    else
	C = 0;
}

void interrupt( unsigned short vector )
{
    PUSH( emPC / 256 );
    PUSH( emPC & 0xff );
    PUSH( build_P() );

    I = 1;
    D = 0;
    emPC = (unsigned short)256 * (unsigned short)memory[ vector + 1 ] +
	                         (unsigned short)memory[ vector ];
}

/* Main CPU fetch/decode/execute loop */

int cnts[ 1000 ];
int cntdx = 0;
int cnt;

void execute_loop( void )
{
    int ch;
    int icount = 10000;
    int *device_interrupt_flags;

    while( !quitting )
    {
	/* check for interrupts */

	for( device_interrupt_flags = interrupt_ptr;
	     device_interrupt_flags < last_interrupt;
	     device_interrupt_flags++ )
	{
	    if ( *device_interrupt_flags )
	    {
		if ( I == 0 && *device_interrupt_flags & F_IRQ )
		{
		    interrupt( 0xfffe );
		    (*device_interrupt_flags) &= ~F_IRQ;
		}
		else if ( *device_interrupt_flags & F_NMI )
		{
		    interrupt( 0xfffa );
		    (*device_interrupt_flags) &= ~F_NMI;
		}
		else if ( *device_interrupt_flags & F_RESET )
		{
		    interrupt( 0xfffc );
		    (*device_interrupt_flags) &= ~F_RESET;
		}

		cnts[ cntdx ] = cnt;
		if ( ++cntdx == 1000 ) cntdx--;
		cnt = 0;
	    }
	}

	/* debugger, if option is on */
	if ( em65_debug )
	{
	    debugger_step();
	}

	/* fetch, decode, and execute and instruction */
	opcode = READ( emPC );
	emPC++;
	instruction_table[ opcode ]();

	/* if ( I == 0 ) */
	    cnt++;

	if ( running )
	{
	    num_instructions++;

	    /* see if there are any keypresses for terminal (occasionally) */
	    if ( --icount == 0 )
	    {
		icount = 10000;
		ch = nb_getchar();
		if ( ch != -1 )
		    terminal_tell( TERMINAL_KEYPRESS | ch );
	    }
	}
    }

    for( ch = 0; ch < 200; ch++ )
	printf( "%d ", cnts[ch] );

    fflush( stdout );
    getchar();
}

/* memory mapped I/O read and write nop functions           */
/* (other functions are defined in their respective modules */
/*  such as terminal, disk, etc.)                           */

void iow_nop( unsigned char byte )
{
    /* Do Nothing */
}

unsigned char ior_nop( void )
{
    return 0;
}

/* 65c02 instructions */

void i00_BRK( void )
{
    emPC++;
    PUSH( emPC / 256 );
    PUSH( emPC & 0xff );
    
    B = 1;
    PUSH( build_P() );

    I = 1;
    D = 0;
    emPC = (unsigned short)256 * (unsigned short)memory[ 0xffff ] +
                                 (unsigned short)memory[ 0xfffe ];
}

void i01_ORA( void )
{
    effective_address = INDEXED_INDIRECT();
    arg = READ( effective_address );

    A |= arg;
    SET_FLAGS_NZ( A );
}

void i04_TSB( void )
{
    effective_address = ZERO_PAGE;
    arg = READ( effective_address );
   
    WRITE( effective_address, A | arg );

    Z = !(A & arg);
}

void i05_ORA( void )
{
    effective_address = ZERO_PAGE;
    arg = READ( effective_address );

    A |= arg;
    SET_FLAGS_NZ( A );
}

void i06_ASL( void )
{
    effective_address = ZERO_PAGE;
    arg = READ( effective_address );

    if ( arg & 0x80 )
	C = 1;
    else
	C = 0;

    arg <<= 1;
    
    SET_FLAGS_NZ( arg );

    WRITE( effective_address, arg );
}

void i08_PHP( void )
{
    PUSH( build_P() );
}

void i09_ORA( void )
{
    effective_address = IMMEDIATE;
    arg = READ( effective_address );

    A |= arg;
    SET_FLAGS_NZ( A );
}

void i0A_ASL( void )
{
    if ( A & 0x80 )
	C = 1;
    else
	C = 0;

    A <<= 1;
    
    SET_FLAGS_NZ( A );
}

void i0C_TSB( void )
{
    effective_address = ABSOLUTE;
    arg = READ( effective_address );

    WRITE( effective_address, A | arg );

    Z = !(A & arg);
}

void i0D_ORA( void )
{
    effective_address = ABSOLUTE;
    arg = READ( effective_address );

    A |= arg;
    SET_FLAGS_NZ( A );
}

void i0E_ASL( void )
{
    effective_address = ABSOLUTE;
    arg = READ( effective_address );

    if ( arg & 0x80 )
	C = 1;
    else
	C = 0;

    arg <<= 1;
    
    SET_FLAGS_NZ( arg );

    WRITE( effective_address, arg );
}

void i10_BPL( void )
{
    if ( N )
	emPC++;
    else
	emPC = RELATIVE;
}

void i11_ORA( void )
{
    effective_address = INDIRECT_INDEXED();
    arg = READ( effective_address );

    A |= arg;
    SET_FLAGS_NZ( A );
}

void i12_ORA( void )
{
    effective_address = ZERO_PAGE_INDIRECT();
    arg = READ( effective_address );

    A |= arg;
    SET_FLAGS_NZ( A );
}

void i14_TRB( void )
{
    effective_address = ZERO_PAGE;
    arg = READ( effective_address );

    WRITE( effective_address, (~A) & arg );

    Z = !(A & arg);
}

void i15_ORA( void )
{
    effective_address = ZERO_PAGE_INDEXED_X;
    arg = READ( effective_address );

    A |= arg;
    SET_FLAGS_NZ( A );
}

void i16_ASL( void )
{
    effective_address = ZERO_PAGE_INDEXED_X;
    arg = READ( effective_address );

    if ( arg & 0x80 )
	C = 1;
    else
	C = 0;

    arg <<= 1;
    
    SET_FLAGS_NZ( arg );

    WRITE( effective_address, arg );
}

void i18_CLC( void )
{
    C = 0;
}

void i19_ORA( void )
{
    effective_address = INDEXED_Y;
    arg = READ( effective_address );

    A |= arg;
    SET_FLAGS_NZ( A );
}

void i1A_INC( void )
{
    A++;
    SET_FLAGS_NZ( A );
}

void i1C_TRB( void )
{
    effective_address = ABSOLUTE;
    arg = READ( effective_address );

    WRITE( effective_address, (~A) & arg );

    Z = !(A & arg);
}

void i1D_ORA( void )
{
    effective_address = INDEXED_X;
    arg = READ( effective_address );

    A |= arg;
    SET_FLAGS_NZ( A );
}

void i1E_ASL( void )
{
    effective_address = INDEXED_X;
    arg = READ( effective_address );

    if ( arg & 0x80 )
	C = 1;
    else
	C = 0;

    arg <<= 1;
    
    SET_FLAGS_NZ( arg );

    WRITE( effective_address, arg );
}

void i20_JSR( void )
{
    PUSH( (emPC+1) / 256 );
    PUSH( (emPC+1) & 0xff );
    emPC = ABSOLUTE;
}

void i21_AND( void )
{
    effective_address = INDEXED_INDIRECT();
    arg = READ( effective_address );

    A &= arg;
    SET_FLAGS_NZ( A );
}

void i24_BIT( void )
{
    effective_address = ZERO_PAGE;
    arg = READ( effective_address );

    N = arg & 0x80;
    V = arg & 0x40;
    Z = !(arg & A);
}

void i25_AND( void )
{
    effective_address = ZERO_PAGE;
    arg = READ( effective_address );

    A &= arg;
    SET_FLAGS_NZ( A );
}

void i26_ROL( void )
{
    unsigned char result;

    effective_address = ZERO_PAGE;
    arg = READ( effective_address );

    result = ( arg << 1 ) | C;

    if ( arg & 0x80 )
	C = 1;
    else
	C = 0;
    
    SET_FLAGS_NZ( result );

    WRITE( effective_address, result );
}

void i28_PLP( void )
{
    unbuild_P( PULL() );
}

void i29_AND( void )
{
    effective_address = IMMEDIATE;
    arg = READ( effective_address );

    A &= arg;
    SET_FLAGS_NZ( A );
}

void i2A_ROL( void )
{
    unsigned char result;

    result = ( A << 1 ) | C;

    if ( A & 0x80 )
	C = 1;
    else
	C = 0;
    
    SET_FLAGS_NZ( result );

    A = result;
}

void i2C_BIT( void )
{
    effective_address = ABSOLUTE;
    arg = READ( effective_address );

    N = arg & 0x80;
    V = arg & 0x40;
    Z = !(arg & A);
}

void i2D_AND( void )
{
    effective_address = ABSOLUTE;
    arg = READ( effective_address );

    A &= arg;
    SET_FLAGS_NZ( A );
}

void i2E_ROL( void )
{
    unsigned char result;

    effective_address = ABSOLUTE;
    arg = READ( effective_address );

    result = ( arg << 1 ) | C;

    if ( arg & 0x80 )
	C = 1;
    else
	C = 0;
    
    SET_FLAGS_NZ( result );

    WRITE( effective_address, result );
}

void i30_BMI( void )
{
    if ( N )
	emPC = RELATIVE;
    else
	emPC++;
}

void i31_AND( void )
{
    effective_address = INDIRECT_INDEXED();
    arg = READ( effective_address );

    A &= arg;
    SET_FLAGS_NZ( A );
}

void i32_AND( void )
{
    effective_address = ZERO_PAGE_INDIRECT();
    arg = READ( effective_address );

    A &= arg;
    SET_FLAGS_NZ( A );
}

void i34_BIT( void )
{
    effective_address = ZERO_PAGE_INDEXED_X;
    arg = READ( effective_address );

    N = arg & 0x80;
    V = arg & 0x40;
    Z = !(arg & A);
}

void i35_AND( void )
{
    effective_address = ZERO_PAGE_INDEXED_X;
    arg = READ( effective_address );

    A &= arg;
    SET_FLAGS_NZ( A );
}

void i36_ROL( void )
{
    unsigned char result;

    effective_address = ZERO_PAGE_INDEXED_X;
    arg = READ( effective_address );

    result = ( arg << 1 ) | C;

    if ( arg & 0x80 )
	C = 1;
    else
	C = 0;
    
    SET_FLAGS_NZ( result );

    WRITE( effective_address, result );
}

void i38_SEC( void )
{
    C = 1;
}

void i39_AND( void )
{
    effective_address = INDEXED_Y;
    arg = READ( effective_address );

    A &= arg;
    SET_FLAGS_NZ( A );
}

void i3A_DEC( void )
{
    A--;
    SET_FLAGS_NZ( A );
}

void i3C_BIT( void )
{
    effective_address = INDEXED_X;
    arg = READ( effective_address );

    N = arg & 0x80;
    V = arg & 0x40;
    Z = !(arg & A);
}

void i3D_AND( void )
{
    effective_address = INDEXED_X;
    arg = READ( effective_address );

    A &= arg;
    SET_FLAGS_NZ( A );
}

void i3E_ROL( void )
{
    unsigned char result;

    effective_address = INDEXED_X;
    arg = READ( effective_address );

    result = ( arg << 1 ) | C;

    if ( arg & 0x80 )
	C = 1;
    else
	C = 0;
    
    SET_FLAGS_NZ( result );

    WRITE( effective_address, result );
}

void i40_RTI( void )
{
    unbuild_P( PULL() );
    emPC = (unsigned short)(PULL());
    emPC += (unsigned short)256 * (unsigned short)(PULL());
}

void i41_EOR( void )
{
    effective_address = INDEXED_INDIRECT();
    arg = READ( effective_address );

    A ^= arg;
    SET_FLAGS_NZ( A );
}

void i45_EOR( void )
{
    effective_address = ZERO_PAGE;
    arg = READ( effective_address );

    A ^= arg;
    SET_FLAGS_NZ( A );
}

void i46_LSR( void )
{
    effective_address = ZERO_PAGE;
    arg = READ( effective_address );

    if ( arg & 0x01 )
	C = 1;
    else
	C = 0;

    arg = (arg >> 1) & 0x7f;
    
    SET_FLAGS_NZ( arg );

    WRITE( effective_address, arg );
}

void i48_PHA( void )
{
    PUSH( A );
}

void i49_EOR( void )
{
    effective_address = IMMEDIATE;
    arg = READ( effective_address );

    A ^= arg;
    SET_FLAGS_NZ( A );
}

void i4A_LSR( void )
{
    if ( A & 0x01 )
	C = 1;
    else
	C = 0;

    A = (A >> 1) & 0x7f;
    
    SET_FLAGS_NZ( A );
}

void i4C_JMP( void )
{
    emPC = ABSOLUTE;
}

void i4D_EOR( void )
{
    effective_address = ABSOLUTE;
    arg = READ( effective_address );

    A ^= arg;
    SET_FLAGS_NZ( A );
}

void i4E_LSR( void )
{
    effective_address = ABSOLUTE;
    arg = READ( effective_address );

    if ( arg & 0x01 )
	C = 1;
    else
	C = 0;

    arg = (arg >> 1) & 0x7f;
    
    SET_FLAGS_NZ( arg );

    WRITE( effective_address, arg );
}

void i50_BVC( void )
{
    if ( V )
	emPC++;
    else
	emPC = RELATIVE;
}

void i51_EOR( void )
{
    effective_address = INDIRECT_INDEXED();
    arg = READ( effective_address );

    A ^= arg;
    SET_FLAGS_NZ( A );
}

void i52_EOR( void )
{
    effective_address = ZERO_PAGE_INDIRECT();
    arg = READ( effective_address );

    A ^= arg;
    SET_FLAGS_NZ( A );
}

void i55_EOR( void )
{
    effective_address = ZERO_PAGE_INDEXED_X;
    arg = READ( effective_address );

    A ^= arg;
    SET_FLAGS_NZ( A );
}

void i56_LSR( void )
{
    effective_address = ZERO_PAGE_INDEXED_X;
    arg = READ( effective_address );

    if ( arg & 0x01 )
	C = 1;
    else
	C = 0;

    arg = (arg >> 1) & 0x7f;
    
    SET_FLAGS_NZ( arg );

    WRITE( effective_address, arg );
}

void i58_CLI( void )
{
    I = 0;
}

void i59_EOR( void )
{
    effective_address = INDEXED_Y;
    arg = READ( effective_address );

    A ^= arg;
    SET_FLAGS_NZ( A );
}

void i5A_PHY( void )
{
    PUSH( Y );
}

void i5D_EOR( void )
{
    effective_address = INDEXED_X;
    arg = READ( effective_address );

    A ^= arg;
    SET_FLAGS_NZ( A );
}

void i5E_LSR( void )
{
    effective_address = INDEXED_X;
    arg = READ( effective_address );

    if ( arg & 0x01 )
	C = 1;
    else
	C = 0;

    arg = (arg >> 1) & 0x7f;
    
    SET_FLAGS_NZ( arg );

    WRITE( effective_address, arg );
}

void i60_RTS( void )
{
    emPC = (unsigned short)(PULL());
    emPC += (unsigned short)256 * (unsigned short)(PULL());
    emPC++;
}

void i61_ADC_binary( void )
{
    effective_address = INDEXED_INDIRECT();
    arg = READ( effective_address );
    add_binary( arg );
}

void i61_ADC_decimal( void )
{
    effective_address = INDEXED_INDIRECT();
    arg = READ( effective_address );
    add_decimal( arg );
}

void i64_STZ( void )
{
    effective_address = ZERO_PAGE;
    WRITE( effective_address, 0 );
}

void i65_ADC_binary( void )
{
    effective_address = ZERO_PAGE;
    arg = READ( effective_address );
    add_binary( arg );
}

void i65_ADC_decimal( void )
{
    effective_address = ZERO_PAGE;
    arg = READ( effective_address );
    add_decimal( arg );
}

void i66_ROR( void )
{
    unsigned char result;

    effective_address = ZERO_PAGE;
    arg = READ( effective_address );

    result = arg >> 1;
    if ( C )
	result |= 0x80;

    if ( arg & 0x01 )
	C = 1;
    else
	C = 0;
    
    SET_FLAGS_NZ( result );

    WRITE( effective_address, result );
}

void i68_PLA( void )
{
    A = PULL();
    SET_FLAGS_NZ( A );
}

void i69_ADC_binary( void )
{
    effective_address = IMMEDIATE;
    arg = READ( effective_address );
    add_binary( arg );
}

void i69_ADC_decimal( void )
{
    effective_address = IMMEDIATE;
    arg = READ( effective_address );
    add_decimal( arg );
}

void i6A_ROR( void )
{
    unsigned char result;

    result = A >> 1;
    if ( C )
	result |= 0x80;

    if ( A & 0x01 )
	C = 1;
    else
	C = 0;
    
    SET_FLAGS_NZ( result );

    A = result;
}

void i6C_JMP( void )
{
    emPC = ABSOLUTE_INDIRECT();
}

void i6D_ADC_binary( void )
{
    effective_address = ABSOLUTE;
    arg = READ( effective_address );
    add_binary( arg );
}

void i6D_ADC_decimal( void )
{
    effective_address = ABSOLUTE;
    arg = READ( effective_address );
    add_decimal( arg );
}

void i6E_ROR( void )
{
    unsigned char result;

    effective_address = ABSOLUTE;
    arg = READ( effective_address );

    result = arg >> 1;
    if ( C )
	result |= 0x80;

    if ( arg & 0x01 )
	C = 1;
    else
	C = 0;
    
    SET_FLAGS_NZ( result );

    WRITE( effective_address, result );
}

void i70_BVS( void )
{
    if ( V )
	emPC = RELATIVE;
    else
	emPC++;
}

void i71_ADC_binary( void )
{
    effective_address = INDIRECT_INDEXED();
    arg = READ( effective_address );
    add_binary( arg );
}

void i71_ADC_decimal( void )
{
    effective_address = INDIRECT_INDEXED();
    arg = READ( effective_address );
    add_decimal( arg );
}

void i72_ADC_binary( void )
{
    effective_address = ZERO_PAGE_INDIRECT();
    arg = READ( effective_address );
    add_binary( arg );
}

void i72_ADC_decimal( void )
{
    effective_address = ZERO_PAGE_INDIRECT();
    arg = READ( effective_address );
    add_decimal( arg );
}

void i74_STZ( void )
{
    effective_address = ZERO_PAGE_INDEXED_X;
    WRITE( effective_address, 0 );
}

void i75_ADC_binary( void )
{
    effective_address = ZERO_PAGE_INDEXED_X;
    arg = READ( effective_address );
    add_binary( arg );
}

void i75_ADC_decimal( void )
{
    effective_address = ZERO_PAGE_INDEXED_X;
    arg = READ( effective_address );
    add_decimal( arg );
}

void i76_ROR( void )
{
    unsigned char result;

    effective_address = ZERO_PAGE_INDEXED_X;
    arg = READ( effective_address );

    result = arg >> 1;
    if ( C )
	result |= 0x80;

    if ( arg & 0x01 )
	C = 1;
    else
	C = 0;
    
    SET_FLAGS_NZ( result );

    WRITE( effective_address, result );
}

void i78_SEI( void )
{
    I = 1;
}

void i79_ADC_binary( void )
{
    effective_address = INDEXED_Y;
    arg = READ( effective_address );
    add_binary( arg );
}

void i79_ADC_decimal( void )
{
    effective_address = INDEXED_Y;
    arg = READ( effective_address );
    add_decimal( arg );
}

void i7A_PLY( void )
{
    Y = PULL();
    SET_FLAGS_NZ( Y );
}

void i7C_JMP( void )
{
    emPC = ABSOLUTE_INDEXED_INDIRECT();
}

void i7D_ADC_binary( void )
{
    effective_address = INDEXED_X;
    arg = READ( effective_address );
    add_binary( arg );
}

void i7D_ADC_decimal( void )
{
    effective_address = INDEXED_X;
    arg = READ( effective_address );
    add_decimal( arg );
}

void i7E_ROR( void )
{
    unsigned char result;

    effective_address = INDEXED_X;
    arg = READ( effective_address );

    result = arg >> 1;
    if ( C )
	result |= 0x80;

    if ( arg & 0x01 )
	C = 1;
    else
	C = 0;
    
    SET_FLAGS_NZ( result );

    WRITE( effective_address, result );
}

void i80_BRA( void )
{
    emPC = RELATIVE;
}

void i81_STA( void )
{
    effective_address = INDEXED_INDIRECT();
    WRITE( effective_address, A );
}

void i84_STY( void )
{
    effective_address = ZERO_PAGE;
    WRITE( effective_address, Y );
}

void i85_STA( void )
{
    effective_address = ZERO_PAGE;
    WRITE( effective_address, A );
}

void i86_STX( void )
{
    effective_address = ZERO_PAGE;
    WRITE( effective_address, X );
}

void i88_DEC( void )
{
    Y--;
    SET_FLAGS_NZ( Y );
}

void i89_BIT( void )
{
    effective_address = IMMEDIATE;
    arg = READ( effective_address );

/* The N and V flags aren't set when BIT is used with IMMEDIATE addressing */

    Z = !(arg & A);
}

void i8A_TXA( void )
{
    A = X;
    SET_FLAGS_NZ( A );
}

void i8C_STY( void )
{
    effective_address = ABSOLUTE;
    WRITE( effective_address, Y );
}

void i8D_STA( void )
{
    effective_address = ABSOLUTE;
    WRITE( effective_address, A );
}

void i8E_STX( void )
{
    effective_address = ABSOLUTE;
    WRITE( effective_address, X );
}

void i90_BCC( void )
{
    if ( C )
	emPC++;
    else
	emPC = RELATIVE;
}

void i91_STA( void )
{
    effective_address = INDIRECT_INDEXED();
    WRITE( effective_address, A );
}

void i92_STA( void )
{
    effective_address = ZERO_PAGE_INDIRECT();
    WRITE( effective_address, A );
}

void i94_STY( void )
{
    effective_address = ZERO_PAGE_INDEXED_X;
    WRITE( effective_address, Y );
}

void i95_STA( void )
{
    effective_address = ZERO_PAGE_INDEXED_X;
    WRITE( effective_address, A );
}

void i96_STX( void )
{
    effective_address = ZERO_PAGE_INDEXED_Y;
    WRITE( effective_address, X );
}

void i98_TYA( void )
{
    A = Y;
    SET_FLAGS_NZ( A );
}

void i99_STA( void )
{
    effective_address = INDEXED_Y;
    WRITE( effective_address, A );
}

void i9A_TXS( void )
{
    S = X;
}

void i9C_STZ( void )
{
    effective_address = ABSOLUTE;
    WRITE( effective_address, 0 );
}

void i9D_STA( void )
{
    effective_address = INDEXED_X;
    WRITE( effective_address, A );
}

void i9E_STZ( void )
{
    effective_address = INDEXED_X;
    WRITE( effective_address, 0 );
}

void iA0_LDY( void )
{
    effective_address = IMMEDIATE;
    Y = READ( effective_address );
    SET_FLAGS_NZ( Y );
}

void iA1_LDA( void )
{
    effective_address = INDEXED_INDIRECT();
    A = READ( effective_address );
    SET_FLAGS_NZ( A );
}

void iA2_LDX( void )
{
    effective_address = IMMEDIATE;
    X = READ( effective_address );
    SET_FLAGS_NZ( X );
}

void iA4_LDY( void )
{
    effective_address = ZERO_PAGE;
    Y = READ( effective_address );
    SET_FLAGS_NZ( Y );
}

void iA5_LDA( void )
{
    effective_address = ZERO_PAGE;
    A = READ( effective_address );
    SET_FLAGS_NZ( A );
}

void iA6_LDX( void )
{
    effective_address = ZERO_PAGE;
    X = READ( effective_address );
    SET_FLAGS_NZ( X );
}

void iA8_TAY( void )
{
    Y = A;
    SET_FLAGS_NZ( Y );
}

void iA9_LDA( void )
{
    effective_address = IMMEDIATE;
    A = READ( effective_address );
    SET_FLAGS_NZ( A );
}

void iAA_TAX( void )
{
    X = A;
    SET_FLAGS_NZ( X );
}

void iAC_LDY( void )
{
    effective_address = ABSOLUTE;
    Y = READ( effective_address );
    SET_FLAGS_NZ( Y );
}

void iAD_LDA( void )
{
    effective_address = ABSOLUTE;
    A = READ( effective_address );
    SET_FLAGS_NZ( A );
}

void iAE_LDX( void )
{
    effective_address = ABSOLUTE;
    X = READ( effective_address );
    SET_FLAGS_NZ( X );
}

void iB0_BCS( void )
{
    if ( C )
	emPC = RELATIVE;
    else
	emPC++;
}

void iB1_LDA( void )
{
    effective_address = INDIRECT_INDEXED();
    A = READ( effective_address );
    SET_FLAGS_NZ( A );
}

void iB2_LDA( void )
{
    effective_address = ZERO_PAGE_INDIRECT();
    A = READ( effective_address );
    SET_FLAGS_NZ( A );
}

void iB4_LDY( void )
{
    effective_address = ZERO_PAGE_INDEXED_X;
    Y = READ( effective_address );
    SET_FLAGS_NZ( Y );
}

void iB5_LDA( void )
{
    effective_address = ZERO_PAGE_INDEXED_X;
    A = READ( effective_address );
    SET_FLAGS_NZ( A );
}

void iB6_LDX( void )
{
    effective_address = ZERO_PAGE_INDEXED_Y;
    X = READ( effective_address );
    SET_FLAGS_NZ( X );
}

void iB8_CLV( void )
{
    V = 0;
}

void iB9_LDA( void )
{
    effective_address = INDEXED_Y;
    A = READ( effective_address );
    SET_FLAGS_NZ( A );
}

void iBA_TSX( void )
{
    X = S;
    SET_FLAGS_NZ( X );
}

void iBC_LDY( void )
{
    effective_address = INDEXED_X;
    Y = READ( effective_address );
    SET_FLAGS_NZ( Y );
}

void iBD_LDA( void )
{
    effective_address = INDEXED_X;
    A = READ( effective_address );
    SET_FLAGS_NZ( A );
}

void iBE_LDX( void )
{
    effective_address = INDEXED_Y;
    X = READ( effective_address );
    SET_FLAGS_NZ( X );
}

void iC0_CPY( void )
{
    effective_address = IMMEDIATE;
    arg = READ( effective_address );
    compare( Y, arg );
}

void iC1_CMP( void )
{
    effective_address = INDEXED_INDIRECT();
    arg = READ( effective_address );
    compare( A, arg );
}

void iC4_CPY( void )
{
    effective_address = ZERO_PAGE;
    arg = READ( effective_address );
    compare( Y, arg );
}

void iC5_CMP( void )
{
    effective_address = ZERO_PAGE;
    arg = READ( effective_address );
    compare( A, arg );
}

void iC6_DEC( void )
{
    effective_address = ZERO_PAGE;
    arg = READ( effective_address ) - 1;
    WRITE( effective_address, arg );
    SET_FLAGS_NZ( arg );
}

void iC8_INC( void )
{
    Y++;
    SET_FLAGS_NZ( Y );
}

void iC9_CMP( void )
{
    effective_address = IMMEDIATE;
    arg = READ( effective_address );
    compare( A, arg );
}

void iCA_DEC( void )
{
    X--;
    SET_FLAGS_NZ( X );
}

void iCC_CPY( void )
{
    effective_address = ABSOLUTE;
    arg = READ( effective_address );
    compare( Y, arg );
}

void iCD_CMP( void )
{
    effective_address = ABSOLUTE;
    arg = READ( effective_address );
    compare( A, arg );
}

void iCE_DEC( void )
{
    effective_address = ABSOLUTE;
    arg = READ( effective_address ) - 1;
    WRITE( effective_address, arg );
    SET_FLAGS_NZ( arg );
}

void iD0_BNE( void )
{
    if ( Z )
	emPC++;
    else
	emPC = RELATIVE;
}

void iD1_CMP( void )
{
    effective_address = INDIRECT_INDEXED();
    arg = READ( effective_address );
    compare( A, arg );
}

void iD2_CMP( void )
{
    effective_address = ZERO_PAGE_INDIRECT();
    arg = READ( effective_address );
    compare( A, arg );
}

void iD5_CMP( void )
{
    effective_address = ZERO_PAGE_INDEXED_X;
    arg = READ( effective_address );
    compare( A, arg );
}

void iD6_DEC( void )
{
    effective_address = ZERO_PAGE_INDEXED_X;
    arg = READ( effective_address ) - 1;
    WRITE( effective_address, arg );
    SET_FLAGS_NZ( arg );
}

void iD8_CLD( void )
{
    D = 0;
    instruction_table = binary_instruction_table;
}

void iD9_CMP( void )
{
    effective_address = INDEXED_Y;
    arg = READ( effective_address );
    compare( A, arg );
}

void iDA_PHX( void )
{
    PUSH( X );
}

void iDD_CMP( void )
{
    effective_address = INDEXED_X;
    arg = READ( effective_address );
    compare( A, arg );
}

void iDE_DEC( void )
{
    effective_address = INDEXED_X;
    arg = READ( effective_address ) - 1;
    WRITE( effective_address, arg );
    SET_FLAGS_NZ( arg );
}

void iE0_CPX( void )
{
    effective_address = IMMEDIATE;
    arg = READ( effective_address );
    compare( X, arg );
}

void iE1_SBC_binary( void )
{
    effective_address = INDEXED_INDIRECT();
    arg = READ( effective_address );
    subtract_binary( arg );
}

void iE1_SBC_decimal( void )
{
    effective_address = INDEXED_INDIRECT();
    arg = READ( effective_address );
    subtract_decimal( arg );
}

void iE4_CPX( void )
{
    effective_address = ZERO_PAGE;
    arg = READ( effective_address );
    compare( X, arg );
}

void iE5_SBC_binary( void )
{
    effective_address = ZERO_PAGE;
    arg = READ( effective_address );
    subtract_binary( arg );
}

void iE5_SBC_decimal( void )
{
    effective_address = ZERO_PAGE;
    arg = READ( effective_address );
    subtract_decimal( arg );
}

void iE6_INC( void )
{
    effective_address = ZERO_PAGE;
    arg = READ( effective_address ) + 1;
    WRITE( effective_address, arg );
    SET_FLAGS_NZ( arg );
}

void iE8_INC( void )
{
    X++;
    SET_FLAGS_NZ( X );
}

void iE9_SBC_binary( void )
{
    effective_address = IMMEDIATE;
    arg = READ( effective_address );
    subtract_binary( arg );
}

void iE9_SBC_decimal( void )
{
    effective_address = IMMEDIATE;
    arg = READ( effective_address );
    subtract_decimal( arg );
}

void iEA_NOP( void )
{
    /* Do nothing! */
}

void iEC_CPX( void )
{
    effective_address = ABSOLUTE;
    arg = READ( effective_address );
    compare( X, arg );
}

void iED_SBC_binary( void )
{
    effective_address = ABSOLUTE;
    arg = READ( effective_address );
    subtract_binary( arg );
}

void iED_SBC_decimal( void )
{
    effective_address = ABSOLUTE;
    arg = READ( effective_address );
    subtract_decimal( arg );
}

void iEE_INC( void )
{
    effective_address = ABSOLUTE;
    arg = READ( effective_address ) + 1;
    WRITE( effective_address, arg );
    SET_FLAGS_NZ( arg );
}

void iF0_BEQ( void )
{
    if ( Z )
	emPC = RELATIVE;
    else
	emPC++;
}

void iF1_SBC_binary( void )
{
    effective_address = INDIRECT_INDEXED();
    arg = READ( effective_address );
    subtract_binary( arg );
}

void iF1_SBC_decimal( void )
{
    effective_address = INDIRECT_INDEXED();
    arg = READ( effective_address );
    subtract_decimal( arg );
}

void iF2_SBC_binary( void )
{
    effective_address = ZERO_PAGE_INDIRECT();
    arg = READ( effective_address );
    subtract_binary( arg );
}

void iF2_SBC_decimal( void )
{
    effective_address = ZERO_PAGE_INDIRECT();
    arg = READ( effective_address );
    subtract_decimal( arg );
}

void iF5_SBC_binary( void )
{
    effective_address = ZERO_PAGE_INDEXED_X;
    arg = READ( effective_address );
    subtract_binary( arg );
}

void iF5_SBC_decimal( void )
{
    effective_address = ZERO_PAGE_INDEXED_X;
    arg = READ( effective_address );
    subtract_decimal( arg );
}

void iF6_INC( void )
{
    effective_address = ZERO_PAGE_INDEXED_X;
    arg = READ( effective_address ) + 1;
    WRITE( effective_address, arg );
    SET_FLAGS_NZ( arg );
}

void iF8_SED( void )
{
    D = 1;
    instruction_table = decimal_instruction_table;
}

void iF9_SBC_binary( void )
{
    effective_address = INDEXED_Y;
    arg = READ( effective_address );
    subtract_binary( arg );
}

void iF9_SBC_decimal( void )
{
    effective_address = INDEXED_Y;
    arg = READ( effective_address );
    subtract_decimal( arg );
}

void iFA_PLX( void )
{
    X = PULL();
    SET_FLAGS_NZ( X );
}

void iFD_SBC_binary( void )
{
    effective_address = INDEXED_X;
    arg = READ( effective_address );
    subtract_binary( arg );
}

void iFD_SBC_decimal( void )
{
    effective_address = INDEXED_X;
    arg = READ( effective_address );
    subtract_decimal( arg );
}

void iFE_INC( void )
{
    effective_address = INDEXED_X;
    arg = READ( effective_address ) + 1;
    WRITE( effective_address, arg );
    SET_FLAGS_NZ( arg );
}

