/*
** debugger.c
**
** Part of Em65 - a 65c02 computer system emulator
**
** Copyright 1997, W. Sheldon Simms III
**
** This file implements Em65's 65c02 machine level debugger. It is
** executed in the "cpu" process, and controls the user's physical
** terminal. The debugger communicates with the "terminal" process
** to tell it when it can and can't use the physical terminal.
*/

/****** Standard Includes ******/

#include <assert.h>
#include <ctype.h>
#include <curses.h>
#include <string.h>

/****** Project Includes ******/

#include "cmdline.h"    /* short em65_debug */

#include "cpu.h"        /* #define IO_TOP
                           #define IO_BOTTOM
                           #define IO_BOTTOM_STR
			   short C, Z, I, D, B, V, N
			   unsigned char  A, X, Y, P, S
			   unsigned short emPC
			   unsigned char *memory        */

#include "em65.h"       /* short quitting
                           int nb_getchar( void ) */

#include "em65types.h"  /* TRUE, FALSE */

#include "tables.h"     /* char instruction_size[],
                           char instruction_is_branch[],
                           char instruction_is_immediate[],
                           char *standard_instruction_strings[],
                           char *orthogonal_instruction_strings[],
                           char **instruction_strings              */

#include "terminal.h"   /* #define TERMINAL_WRITE,
                           #define TERMINAL_NOWRITE
                           int terminal_tell( int )   */

#include "timer.h"      /* unsigned long num_instructions */


/************************************************************************

                             local #defines

************************************************************************/


/* symbolic representations of ASCII values */

#ifdef KEY_ESCAPE
#undef KEY_ESCAPE
#endif

#ifdef KEY_DELETE
#undef KEY_DELETE
#endif

#define KEY_CONTROL_F  0x06
#define KEY_CONTROL_B  0x02
#define KEY_ESCAPE     0x1b
#define KEY_DELETE     0x7f

/* bit set in long  returned by edit_hex() and edit_dec() */
/* to indicate that the function was aborted without any  */
/* digits being typed by the user                         */

#define NO_DIGITS_ENTERED  0x1000

/* sizes of the various windows */

#define WREGISTERS_LINES    8
#define WREGISTERS_COLS     10

#define WBREAKPOINTS_LINES  12
#define WBREAKPOINTS_COLS   11

#define WWATCHES_LINES      12
#define WWATCHES_COLS       13

#define WDISASSEMBLY_LINES  12
#define WDISASSEMBLY_COLS   43

#define WSTACK_LINES        15
#define WSTACK_COLS         10

#define WMEMORY_LINES       11
#define WMEMORY_COLS        42

#define WMENU_LINES         11
#define WMENU_COLS          26

/* the number of bytes to display on each line of the memory window */

#define WMEMORY_BYTES_PER_LINE   8


/************************************************************************

                       exported global variables

************************************************************************/


/***** Flags *****/


short running  = 0;
/* execute continuously and don't return to the debugger */

short stepping = 1;
/* single-step 65c02 instructions, returning to debugger after each */

short stepover = 0;
/* execute continuously until RTS is executed, then return to debugger */

/*
** Note that if none of the above (including running) are set, the
** debugger is "tracing" - instructions are executed continuously but
** the debugger screen is updated after each instruction is executed.
*/

short starting = 1;
/* next call to debugger_step will be the first */

short debug_controls_screen = 1;
/* debugger is writing to screen, terminal is not allowed to */


/************************************************************************

                     non-exported global variables

************************************************************************/


/***** Curses Definitions *****/


static WINDOW *wdebug;
/* curses WINDOW - the entire debugger screen (24x80) */

static WINDOW *wregisters, *wbreakpoints, *wwatches, *wdisassembly;
/* curses WINDOWs - the top row of debugger windows */

static WINDOW *wstack, *wmemory, *wmenu;
/* curses WINDOWs - the bottom row of debugger windows */


/***** Stack Window *****/


static unsigned char stack_first;
/* the page 1 address of first byte displayed in stack window */

static int stack_line;
/* the line of the stack window occupied by the stack pointer */


/***** Memory Window *****/


static unsigned short memory_first;
/* the address of the first location displayed in the memory window */


/***** Disassembly Window *****/


static unsigned short disassembly_first;
/* the address of first instruction displayed in the disassembly window */

static unsigned short disassembly_last;
/* the address of the last byte displayed in the disassembly window + 1 */

static unsigned short disassembly_len[ WDISASSEMBLY_LINES ];
/* the number of bytes displayed on each line in the disassembly window */

static int pc_line;
/* the line of the disassembly window occupied by the program counter */


/***** Breakpoints Window *****/


static struct {

    short           used;     /* is this entry used?            */
    short           enabled;  /* is this breakpoint enabled?    */
    unsigned short  address;  /* the address of this breakpoint */

} breakpoints[ WBREAKPOINTS_LINES ];
/* array of breakpoints - kept sorted by address */

static int num_breakpoints;
/* the number of breakpoints currently used (out of WBREAKPOINTS_LINES) */


/***** Watches Window *****/


static struct {

    short           used;     /* is this entry used?       */
    unsigned short  address;  /* the address being watched */

} watches[ WWATCHES_LINES ];
/* array of watches - kept sorted by address */

static int num_watches;
/* the number of watches currently used (out of WWATCHES_LINES) */


/************************************************************************

                      local function prototypes

************************************************************************/


/* utility functions */

static void my_mvwaddhex_char( WINDOW *w, int y, int x, unsigned char c );
static void my_mvwaddhex_short( WINDOW *w, int y, int x, unsigned short s );
static int edit_hex( WINDOW *w, int y, int x,
		     unsigned long *val, int max_xdigits );
static int edit_dec( WINDOW *w, int y, int x,
		     long *val, int max_digits, short sign_ok );
static int edit_string( WINDOW *w, int y, int x,
			char *s, int max_display, int max_length );
static void page_file( char *fname );

/* register window functions */

static void display_registers( void );
static void update_registers( void );
static int edit_status( void );
static void edit_registers( void );

/* breakpoint window functions */

static void display_breakpoints( void );
static int insert_breakpoint( unsigned short address );
static void add_breakpoint( int prompt_line );
static void delete_breakpoint( int index );
static void kill_breakpoint( int prompt_line );
static void toggle_breakpoint( int prompt_line );
static void edit_breakpoint( void );

/* stack window functions */

static void display_stack( void );
static void update_stack( void );

/* memory window functions */

static void display_memory_line( int display_line, unsigned short address );
static void display_memory( void );
static void display_edit_memory_menu( void );
static unsigned short goto_memory_address( void );
static void fill_memory_range( void );
static void edit_memory( void );

/* watch window functions */

static void display_watches( void );
static int insert_watch( unsigned short address );
static void add_watch( int prompt_line );
static void delete_watch( int index );
static void kill_watch( int prompt_line );
static void edit_watch( void );

/* disassembly window functions */

static void disassemble_data_byte( int y, int x1, int x2,
				   unsigned short address );
static void disassemble_data( int y, int x,
			      unsigned short address,
			      int min_bytes, int max_bytes );
static void disassemble_three_bytes( int y, int x, unsigned short address );
static void disassemble_two_bytes( int y, int x, unsigned short address );
static void disassemble_one_byte( int y, int x, unsigned short address );
static void disassemble_line( int y, int x, unsigned short address );
static void display_disassembly( void );
static void update_disassembly( void );

/* main menu selections */

static void edit_register_memory( void );
static void load_file_interactive( void );
static void display_help( void );
static void display_info( void );
static void display_menu( void );
static void ensure_debugger_screen( void );
static void debug_command_loop( void );

/* debugger display functions */

static void display_windows( void );
static void draw_vline( int y1, int y2, int x, char c );
static void init_display( void );

/* execution control functions */

static void test_breakpoints( void );
static void sigint_respond( void );


/************************************************************************

                          Utility Functions

************************************************************************/


/*
** void my_mvwaddhex_char( WINDOW *w, int y, int y, unsigned char c )
** void my_mvwaddhex_short( WINDOW *w, int y, int y, unsigned short c )
**
** Both assume an ASCII-compatible character set. Specifically, they assume
** that 'A' - 10 == '7' and that both numerals and letters are in order and
** have successive values.
**
** my_mvwaddhex_char() prints the unsigned char c as two hexadecimal digits
** at the specified position y,x in the specified window w
**
** my_mvwaddhex_short() prints the unsigned short s as four hexadecimal digits
** at the specified position y,x in the specified window w
*/

#define NIBBLE_TO_XDIGIT( n ) ( (n) < 10 ? (n) + '0' : (n) + '7' )

static void my_mvwaddhex_char( WINDOW *w, int y, int x, unsigned char c )
{
    int xdigit;

    /* Debug - check arguments */
    assert( w );
    assert( y >= 0 && x >= 0 );
    /* don't know how big y and x can be - check for absurdity */
    assert( y < LINES && x < COLS );
    /* every possible value of c is ok */

    /* print high nibble */

    xdigit = c / 16;
    xdigit = NIBBLE_TO_XDIGIT( xdigit );
    mvwaddch( w, y, x++, xdigit );

    /* print low nibble */

    xdigit = c & 0x0f;
    xdigit = NIBBLE_TO_XDIGIT( xdigit );
    mvwaddch( w, y, x, xdigit );
}

static void my_mvwaddhex_short( WINDOW *w, int y, int x, unsigned short s )
{
    /* Debug - my_mvwaddhex_char assert()s the arguments */

    my_mvwaddhex_char( w, y, x,   s / 256  );
    my_mvwaddhex_char( w, y, x+2, s & 0xff );
}

/*
** int edit_hex( WINDOW *w, int y, int x,
**               unsigned long *val, int max_xdigits )
**
** edit_hex() allows interactive editing and entry of a hexadecimal number.
** It waits for edit keys to be entered and then responds by performing an
** appropriate editing action.
**
** If some other key is hit (one that has no editing action) edit_hex()
** copies the number being edited, as it stands at that point, into the
** provided unsigned long, and returns the non-editing key that was hit.
** If the user hits a non-editing key before any hexadecimal digit has
** been typed, edit_hex() returns the special value NO_DIGITS_ENTERED.
**
** The argument max_xdigits allows the caller to tell edit_hex() to accept
** no hexadecimal number more than max_xdigits hexadecimal digits long.
*/

static int edit_hex( WINDOW *w, int y, int x,
		     unsigned long *val, int max_xdigits )
{
    int c;                 /* character read from keyboard    */
    int num_xdigits;       /* the length of the number so far */

    unsigned long result;  /* the value of the number so far */

    /* Debug - check arguments */
    assert( w );
    assert( val );
    assert( y >= 0 && x >= 0 );
    /* don't know how big y and x can be - check for absurdity */
    assert( y < LINES && x < COLS );
    assert( max_xdigits > 0 && max_xdigits <= 8 );

    /* initialization */
    result = 0; num_xdigits = 0;
    wmove( w, y, x );
    wrefresh( w );

    while( 1 )
    {
	c = wgetch( w );

	if ( isxdigit( c ) )
	{
	    if ( num_xdigits < max_xdigits )
	    {
		/* print the xdigit */
		mvwaddch( w, y, x + num_xdigits, c );

 		/* modify the result value accordingly */
		result *= 16;
		if ( isdigit( c ) )
		    result += ( c - '0' );
		else if ( islower( c ) )
		    result += ( c - 'a' + 10 );
		else
		    result += ( c - 'A' + 10 );

		/* increment the count of xdigits */
		num_xdigits++;
	    }
	}
	else if ( c == '\b' || c == KEY_DELETE )
	{
	    if ( num_xdigits > 0 )
	    {
		/* delete the last xdigit entered */
		num_xdigits--;
		mvwaddch( w, y, x + num_xdigits, ' ' );

		result /= 16;
	    }
	}
	else
	{
	    /* don't change the value if no editing characters were */
	    /* typed or if user hit ESC                             */

	    if ( num_xdigits && c != KEY_ESCAPE )
		*val = result;

	    if ( !num_xdigits )
		return NO_DIGITS_ENTERED | c;

	    return c;
	}

	wmove( w, y, x + num_xdigits );
	wrefresh( w );
    }
}

/*
** int edit_dec( WINDOW *w, int y, int x,
**               long *val, int max_digits, short sign_ok )
**
** edit_dec() allows interactive editing and entry of a decimal number.
** It waits for edit keys to be entered and then responds by performing an
** appropriate editing action.
**
** If some other key is hit (one that has no editing action) edit_dec()
** copies the number being edited, as it stands at that point, into the
** provided unsigned long, and returns the non-editing key that was hit.
** If the user hits a non-editing key before any digit has been typed
** edit_dec() returns the special value NO_DIGITS_ENTERED.
**
** The argument max_digits allows the caller to tell edit_dec() to accept
** no decimal number more than max_digits decimal digits long.
**
** The argument sign_ok allows the caller to tell edit_dec() whether or not
** a signed number is allowed. If a signed number is allowed, edit_dec() will
** accept a '+' or '-' followed by up to max_digits digits and will return
** a signed value in val. If a signed number is not allowed, then edit_dec()
** will not allow the characters '+' or '-' to be entered and the result
** returned in val can be considered to be cast to an unsigned long
*/

static int edit_dec( WINDOW *w, int y, int x,
		     long *val, int max_digits, short sign_ok )
{
    int c;                 /* character read from keyboard               */
    int sign;              /* +1 or -1, with the same sign as the number */
    int got_sign;          /* 1 if a sign was entered, 0 otherwise       */
    int num_digits;        /* the number of digits in the number so far  */

    unsigned long result;  /* the value of the number so far */

    /* Debug - check arguments */
    assert( w );
    assert( val );
    assert( y >= 0 && x >= 0 );
    /* don't know how big y and x can be - check for absurdity */
    assert( y < LINES && x < COLS );
    assert( max_digits > 0 && max_digits <= 10 );

    /* initialization */

    sign = 1; got_sign = 0; result = 0; num_digits = 0;
    wmove( w, y, x ); wrefresh( w );

    while( 1 )
    {
	c = wgetch( w );

	if ( isdigit( c ) )
	{
	    if ( num_digits < max_digits )
	    {
		/* check for and prevent overflow */
		if ( num_digits == 9 )
		{
		    if ( sign_ok )
		    {
			/* prevent signed overflow */
			if ( sign == 1 )
			{
			    if ( result > 214748364 )
				continue;
			    else if ( result == 214748364 && c > '7' )
				continue;
			}
			else if ( sign == -1 )
			{
			    if ( result > 214748364 )
				continue;
			    else if ( result == 214748364 && c > '8' )
				continue;
			}
		    }
		    else
		    {
			/* prevent unsigned overflow */
			if ( result > 429496729 )
			    continue;
			else if ( result == 429496729 && c > '6' )
			    continue;
		    }
		}

		mvwaddch( w, y, x + num_digits + got_sign, c );

		result *= 10;
		result += ( c - '0' );

		num_digits++;
	    }
	}
	else if (( c == '+' || c == '-' ) &&
		 sign_ok && !got_sign && num_digits == 0 )
	{
	    if ( c == '-' ) sign = -1;
	    got_sign++;
	}
	else if ( c == '\b' || c == KEY_DELETE )
	{
	    if ( num_digits > 0 )
	    {
		num_digits--;
		result /= 10;
	    }
	    else if ( got_sign )
	    {
		sign = 1;
		got_sign--;
	    }

	    mvwaddch( w, y, x + num_digits + got_sign, ' ' );
	}
	else
	{
	    /* don't change the value if no editing characters were */
	    /* typed or if user hit ESC                             */

	    if ( num_digits && c != KEY_ESCAPE )
	    {
		if ( sign_ok )
		    *val = (long)result * sign;
		else
		    *val = result;
	    }

	    if ( !num_digits )
		return NO_DIGITS_ENTERED | c;

	    return c;
	}

	wmove( w, y, x + num_digits + got_sign );
	wrefresh( w );
    }
}

/*
** int edit_string( WINDOW *w, int y, int x,
**                  char *s, int max_display, int max_len )
**
** edit_string() allows the user to type a string, which it stores in
** a buffer provided by the caller. If the user types any non-printing
** character, edit_string terminates the string and returns the non-
** printing character that was typed.
**
** edit string allows the entry of a string up to max_length characters
** long. It will display up to max_display characters of the string at
** a time, scrolling as necessary.
*/

static int edit_string( WINDOW *w, int y, int x,
			char *s, int max_display, int max_length )
{
    int c;           /* character read from keyboard              */
    int num_chars;   /* number of characters in the string so far */
    char *sptr;      /* ptr to the next position in the string    */

    int curs_x;             /* the cursor offset from argument x       */
    char *first_displayed;  /* ptr to the first displayed character    */

    /* Debug - check arguments */

    assert( w );
    assert( s );
    assert( max_length > 1 );
    assert( max_display > 1 );
    assert( y >= 0 && x >= 0 );

    /* don't know how big max_display can be - check for absurdity */
    assert( max_display < COLS );

    /* don't know how big y and x can be - check for absurdity */
    assert( y < LINES && x < COLS );

    /* initialize */

    curs_x = 0;
    num_chars = 0;
    first_displayed = sptr = s;
    memset( s, 0, max_length );

    while( 1 )
    {
	wmove( w, y, x + curs_x );
	wrefresh( w );

	c = wgetch( w );

	if ( isprint( c ) )
	{
	    if ( num_chars < max_length )
	    {
		*sptr++ = c;
		num_chars++;
		if ( (curs_x + 1) > max_display )
		{
		    wmove( w, y, x );
		    wclrtoeol( w );

		    first_displayed += max_display;
		    mvwaddstr( w, y, x, first_displayed );
		    
		    curs_x = 1;
		}
		else
		{
		    mvwaddch( w, y, x + curs_x, c );
		    curs_x++;
		}
	    }
	}
	else if ( c == '\b' || c == KEY_DELETE )
	{
	    if ( num_chars > 0 )
	    {
		sptr--;
		num_chars--;
		if ( 0 == (curs_x - 1) && num_chars > 0 )
		{
		    wmove( w, y, x );
		    wclrtoeol( w );

		    *sptr = 0;
		    first_displayed -= max_display;

		    if ( first_displayed < s )
			first_displayed = s;

		    mvwaddstr( w, y, x, first_displayed );
		    
		    curs_x = max_display;
		}
		else
		{
		    *sptr = 0;
		    curs_x--;
		    mvwaddch( w, y, x + curs_x, ' ' );
		}
	    }
	}
	else
	{
	    *sptr = 0;
	    return c;
	}
    }
}

/*
** void page_file( char *fname )
*/

static void page_file( char *fname )
{
}


/************************************************************************

                      Register Window Functions

************************************************************************/


/*
** void display_registers( void );
**
** display_registers() draws everything in the registers window.
*/

static void display_registers( void )
{
    int flag;

    /* display registers */

    mvwprintw( wregisters, 0, 2,  "A: %2.2X", A    );
    mvwprintw( wregisters, 1, 2,  "X: %2.2X", X    );
    mvwprintw( wregisters, 2, 2,  "Y: %2.2X", Y    );
    mvwprintw( wregisters, 3, 2,  "S: %2.2X", S    );
    mvwprintw( wregisters, 4, 1, "PC: %4.4X", emPC );

    /* display status flags */

    mvwaddstr( wregisters, 6, 1, "NV-BDIZC" );

    if ( N ) flag = '+'; else flag = ' '; mvwaddch( wregisters, 7, 1, flag );
    if ( V ) flag = '+'; else flag = ' '; mvwaddch( wregisters, 7, 2, flag );
    
    mvwaddch( wregisters, 7, 3, '+' );  /* unused flag - always set */
    
    if ( B ) flag = '+'; else flag = ' '; mvwaddch( wregisters, 7, 4, flag );
    if ( D ) flag = '+'; else flag = ' '; mvwaddch( wregisters, 7, 5, flag );
    if ( I ) flag = '+'; else flag = ' '; mvwaddch( wregisters, 7, 6, flag );
    if ( Z ) flag = '+'; else flag = ' '; mvwaddch( wregisters, 7, 7, flag );
    if ( C ) flag = '+'; else flag = ' '; mvwaddch( wregisters, 7, 8, flag );

    wrefresh( wregisters );
}

/*
** void update_registers( void )
**
** update_registers() draws the values of registers in the registers window
*/

static void update_registers( void )
{
    int flag;

    /* display register values */

    my_mvwaddhex_char( wregisters, 0, 5, A );
    my_mvwaddhex_char( wregisters, 1, 5, X );
    my_mvwaddhex_char( wregisters, 2, 5, Y );
    my_mvwaddhex_char( wregisters, 3, 5, S );

    my_mvwaddhex_short( wregisters, 4, 5, emPC );

    /* display status flags */

    if ( N ) flag = '+'; else flag = ' '; mvwaddch( wregisters, 7, 1, flag );
    if ( V ) flag = '+'; else flag = ' '; mvwaddch( wregisters, 7, 2, flag );
    if ( B ) flag = '+'; else flag = ' '; mvwaddch( wregisters, 7, 4, flag );
    if ( D ) flag = '+'; else flag = ' '; mvwaddch( wregisters, 7, 5, flag );
    if ( I ) flag = '+'; else flag = ' '; mvwaddch( wregisters, 7, 6, flag );
    if ( Z ) flag = '+'; else flag = ' '; mvwaddch( wregisters, 7, 7, flag );
    if ( C ) flag = '+'; else flag = ' '; mvwaddch( wregisters, 7, 8, flag );

    wrefresh( wregisters );
}

/*
** void edit_status( void )
**
** edit_status() allows the interactive editing of the 65c02 status
** flags in the registers window.
*/

static int edit_status( void )
{

#define STATUS_Y 7
#define STATUS_X 1

    short flag; /* the updated value of a flag */

    int c;    /* character read from keyboard        */
    int i;    /* which bit is being edited (0-1,3-7) */

    flag = 0;

    i = 0;
    while( i < 8 )
    {
	wmove( wregisters, STATUS_Y, STATUS_X + i );
	wrefresh( wregisters );

	/* save the value of the current flag */
	switch( i )
	{
	case 0: flag = N; break;
	case 1: flag = V; break;
	case 3: flag = B; break;
	case 4: flag = D; break;
	case 5: flag = I; break;
	case 6: flag = Z; break;
	case 7: flag = C; break;
	}

	c = wgetch( wregisters );

	switch( c & 0x7f )
	{
	case '\r': case '\n':
	    if ( i == 1 )
		i+=2;
	    else
		i++;

	    break;

	case KEY_ESCAPE:
	    /* restore the value of the current flag */
	    switch( i )
	    {
	    case 0: N = flag; break;
	    case 1: V = flag; break;
	    case 3: B = flag; break;
	    case 4: D = flag; break;
	    case 5: I = flag; break;
	    case 6: Z = flag; break;
	    case 7: C = flag; break;
	    }
	    return -1;

	case ' ':
	    switch( i )
	    {
	    case 0: flag = N = !N; break;
	    case 1: flag = V = !V; break;
	    case 3: flag = B = !B; break;
	    case 4: flag = D = !D; break;
	    case 5: flag = I = !I; break;
	    case 6: flag = Z = !Z; break;
	    case 7: flag = C = !C; break;
	    }

	    if ( flag )
		mvwaddch( wregisters, STATUS_Y, STATUS_X + i, '+' );
	    else
		mvwaddch( wregisters, STATUS_Y, STATUS_X + i, ' ' );

	    break;

	default:
	    return c;
	}
    }
    
    return '\n';
}

/*
** void edit_registers( void )
**
** edit_registers() allows the interactive editing of the 65c02 machine
** registers in the registers window.
*/

static void edit_registers( void )
{
    int c = 0;  /* character read from keyboard */
    int index;  /* which register is being edited - a state variable (0-5) */

    unsigned long edited_value; /* passed by reference to edit_hex() */

    /* save register values */
    unsigned short sPC = emPC;
    unsigned char  sA = A, sX = X, sY = Y, sS = S;
    short          sN = N, sV = V, sB = B, sD = D, sI = I, sZ = Z, sC = C;

    /* print menu */
    wclear( wmenu );
    mvwaddstr( wmenu, 0, 1, "RET - next register/flag" );
    mvwaddstr( wmenu, 1, 1, "ESC - abort changes made" );
    mvwaddstr( wmenu, 2, 1, "      to a register" );
    mvwaddstr( wmenu, 3, 1, "SPC - toggle status" );
    mvwaddstr( wmenu, 4, 1, "      flags" );
    mvwaddstr( wmenu, 6, 1, "Q - accept changes and" );
    mvwaddstr( wmenu, 7, 1, "    quit editing" );
    mvwaddstr( wmenu, 8, 1, "X - reject changes and" );
    mvwaddstr( wmenu, 9, 1, "    quit editing" );
    wrefresh( wmenu );

    /* initialize */
    index = 0;

    while( 1 )
    {
	/* Debug - catch index errors */
	assert( index >= 0 && index <= 5 );

	switch( index )
	{
	case 0:
	    edited_value = (unsigned long)A;
	    c = edit_hex( wregisters, 0, 5, &edited_value, 2 );
	    if ( c != KEY_ESCAPE && !( C & NO_DIGITS_ENTERED ) )
		A = (unsigned char)edited_value;
	    break;

	case 1:
	    edited_value = (unsigned long)X;
	    c = edit_hex( wregisters, 1, 5, &edited_value, 2 );
	    if ( c != KEY_ESCAPE && !( C & NO_DIGITS_ENTERED ) )
		X = (unsigned char)edited_value;
	    break;

	case 2:
	    edited_value = (unsigned long)Y;
	    c = edit_hex( wregisters, 2, 5, &edited_value, 2 );
	    if ( c != KEY_ESCAPE && !( C & NO_DIGITS_ENTERED ) )
		Y = (unsigned char)edited_value;
	    break;

	case 3:
	    edited_value = (unsigned long)S;
	    c = edit_hex( wregisters, 3, 5, &edited_value, 2 );
	    if ( c != KEY_ESCAPE && !( C & NO_DIGITS_ENTERED ) )
		S = (unsigned char)edited_value;
            /* redraw the stack window */
	    update_stack();
	    break;

	case 4:
	    edited_value = (unsigned long)emPC;
	    c = edit_hex( wregisters, 4, 5, &edited_value, 4 );
	    if ( c != KEY_ESCAPE && !( C & NO_DIGITS_ENTERED ) )
		emPC = (unsigned short)edited_value;
            /* redraw the disassembly window, if necessary */
	    update_disassembly();
	    break;

	case 5:
	    c = edit_status();
	    break;
	}

	/* redraw the registers to reflect any changes */
	update_registers();

	/* ignore everything but the ASCII character */
	c &= 0x7f;

	if ( c == '\n' || c == '\r' )
	    index = ( index + 1 ) % 6;
	else if ( c == 'q' || c == 'Q' )
	    break;
	else if ( c == 'x' || c == 'X' )
	{
	    /* restore all registers */
	    emPC = sPC;
	    A = sA; X = sX; Y = sY; S = sS;
	    N = sN; V = sV; B = sB; D = sD;
	    I = sI; Z = sZ; C = sC;

	    /* redraw affected windows */
	    update_stack();
	    update_registers();
	    update_disassembly();

	    break;
	}
    }
}


/************************************************************************

                     Breakpoint Window Functions

************************************************************************/


/*
** void display_breakpoints( void )
**
** display_breakpoints() displays all breakpoints in the breakpoints
** window.
*/

static void display_breakpoints( void )
{
    char enabled_mark; /* '+' for enabled, '-' for disabled breakpoints */
    int  i;            /* loop index */

    wclear( wbreakpoints );

    /* the breakpoints[] array is kept sorted by address so this loop */
    /* prints the breakpoints in order.                               */

    for ( i = 0; i < num_breakpoints; i++ )
    {
	if ( breakpoints[i].enabled )
	    enabled_mark = '+';
	else
	    enabled_mark = '-';

	mvwprintw( wbreakpoints, i, 1, "%2d %c %4.4X ", i+1, enabled_mark,
		   breakpoints[i].address );
    }

    wrefresh( wbreakpoints );
}

/*
** int insert_breakpoint( unsigned short address )
**
** insert_breakpoint() inserts a breakpoint into the breakpoints[] array
** which is kept in order sorted by address. insert_breakpoint() returns
** FALSE (0) if the new breakpoint cannot be inserted into the breakpoints[]
** array. Currently, this only happens if there is already a breakpoint at
** the specified address. Otherwise, insert_breakpoint() returns TRUE (1)
*/

static int insert_breakpoint( unsigned short address )
{
    int i, j; /* loop indices */

    for ( i = 0; i < WBREAKPOINTS_LINES; i++ )
    {
	/* find place for this breakpoint */
	if ( breakpoints[i].used == 0 || address < breakpoints[i].address )
	    break;

	/* only one breakpoint allowed at any address */
	if ( address == breakpoints[i].address )
	    return FALSE;
    }

    if ( breakpoints[i].used )
    {
	/* if we're inserting in the middle, move the     */
	/* breakpoints at higher addresses up one element */

	for ( j = WBREAKPOINTS_LINES - 1; j > i; j-- )
	    breakpoints[j] = breakpoints[j-1];
    }

    /* breakpoints[i] is now free */

    breakpoints[i].used = 1;
    breakpoints[i].enabled = 1;
    breakpoints[i].address = address;

    num_breakpoints++;

    return TRUE;
}

/*
** void add_breakpoint( int )
**
** add_breakpoint() allows the user to interactively set a new breakpoint
** by specifying the address as which the new breakpoint is to be set.
** add_breakpoint() will fail to add a new breakpoint if there is a breakpoint
** already at the specified address or if there is not room for another
** breakpoint in the breakpoints[] array.
**
** prompt and character echoing is done on the line in the menu window
** specified by prompt_line
*/

static void add_breakpoint( int prompt_line )
{
    int c;                       /* character read from keyboard */
    unsigned long break_address; /* the address of the new breakpoint */

    if ( num_breakpoints == WBREAKPOINTS_LINES )
    {
	mvwaddstr( wmenu, prompt_line, 1, "Too many breakpoints" );
	return;
    }

    /* get the address of the new breakpoint */

    wmove( wmenu, prompt_line, 1 );
    wclrtoeol( wmenu );
    mvwaddstr( wmenu, prompt_line, 1, "Address: " );
    wrefresh( wmenu );

    c = edit_hex( wmenu, prompt_line, 10, &break_address, 4 );
    if ( c == KEY_ESCAPE || c & NO_DIGITS_ENTERED )
    {
	wmove( wmenu, prompt_line, 1 );
	wclrtoeol( wmenu );
	return;
    }

    /* insert the new breakpoint in the breakpoints[] array */

    if ( !insert_breakpoint( (unsigned short)break_address ) )
    {
	wmove( wmenu, prompt_line, 1 );
	wclrtoeol( wmenu );
	mvwaddstr( wmenu, prompt_line, 1, "Breakpoint already set" );
    }
    
    wmove( wmenu, prompt_line, 1 );
    wclrtoeol( wmenu );
}

/*
** void delete_breakpoint( int index )
**
** delete_breakpoint() removes a breakpoint from the breakpoints[] array
** as specified by the index into the array of the breakpoint to remove
*/

static void delete_breakpoint( int index )
{
    int i; /* loop index */

    /* Debug - check argument */

    assert( index >= 0 && index < WBREAKPOINTS_LINES );

    /* check for the easy case - the last breakpoint */

    if ( index == ( num_breakpoints - 1 ) )
	breakpoints[index].used = 0;

    /* move the ones after index down, obliterating index */

    for ( i = index; i < ( num_breakpoints - 1 ); i++ )
	breakpoints[i] = breakpoints[i+1];

    num_breakpoints--;
}

/*
** void kill_breakpoint( int )
**
** kill_breakpoint() allows the user to interactively remove a breakpoint
** from the breakpoints array by specifying (by index into the array) which
** breakpoint is to be removed.
**
** prompt and character echoing is done on the line in the menu window
** specified by prompt_line
*/

static void kill_breakpoint( int prompt_line )
{
    int c;            /* character read from keyboard */
    long break_index; /* the breakpoint to remove     */

    /* select the breakpoint to remove */

    wmove( wmenu, prompt_line, 1 );
    wclrtoeol( wmenu );
    mvwaddstr( wmenu, prompt_line, 1, "Which: " );
    wrefresh( wmenu );

    c = edit_dec( wmenu, prompt_line, 8, &break_index, 2, FALSE );
    if ( c == KEY_ESCAPE || c & NO_DIGITS_ENTERED )
    {
	wmove( wmenu, prompt_line, 1 );
	wclrtoeol( wmenu );
	return;
    }

    /* don't remove a non-existant breakpoint */

    if ( break_index > num_breakpoints )
    {
	wmove( wmenu, prompt_line, 1 );
	wclrtoeol( wmenu );
	mvwaddstr( wmenu, prompt_line, 1, "No such breakpoint" );
	return;
    }

    /* remove it */

    delete_breakpoint( break_index-1 );

    wmove( wmenu, prompt_line, 1 );
    wclrtoeol( wmenu );
}

/*
** void toggle_breakpoint( int )
**
** toggle_breakpoint() allows the user to interactively select a breakpoint
** by entering the index in the breakpoints array[] of the desired breakpoint.
** If that breakpoint is enabled, it become disabled and vice-versa
**
** prompt and character echoing is done on the line in the menu window
** specified by prompt_line
*/

static void toggle_breakpoint( int prompt_line )
{
    int c;            /* character read from keyboard */
    long break_index; /* the breakpoint to toggle     */

    /* select a breakpoint */

    wmove( wmenu, prompt_line, 1 );
    wclrtoeol( wmenu );
    mvwaddstr( wmenu, prompt_line, 1, "Which: " );
    wrefresh( wmenu );

    c = edit_dec( wmenu, prompt_line, 8, &break_index, 2, FALSE );
    if ( c == KEY_ESCAPE || c & NO_DIGITS_ENTERED )
    {
	wmove( wmenu, prompt_line, 1 );
	wclrtoeol( wmenu );
	return;
    }

    /* can't toggle a non-existant breakpoint */

    if ( break_index > num_breakpoints )
    {
	wmove( wmenu, prompt_line, 1 );
	wclrtoeol( wmenu );
	mvwaddstr( wmenu, prompt_line, 1, "No such breakpoint" );
	return;
    }

    /* toggle it */

    breakpoints[break_index-1].enabled = 1 - breakpoints[break_index-1].enabled;

    wmove( wmenu, prompt_line, 1 );
    wclrtoeol( wmenu );
}

/*
** void edit_breakpoint( void )
**
** edit_breakpoint() presents a menu of breakpoint editing options to the
** user and allows him to select one.
*/

static void edit_breakpoint( void )
{
    int c;     /* character read from the keyboard */
    int done;  /* loop control variable            */

    /* present the menu */

    wclear( wmenu );
    mvwaddstr( wmenu, 0, 1, "A - add a breakpoint" );
    mvwaddstr( wmenu, 1, 1, "K - kill a breakpoint" );
    mvwaddstr( wmenu, 2, 1, "T - toggle a breakpoint" );
    mvwaddstr( wmenu, 4, 1, "Q - quit editing" );
    mvwaddstr( wmenu, 5, 1, "    breakpoints" );
    wrefresh( wmenu );

    done = FALSE;

    /* loop, getting a selection and act upon it until user quits */

    while ( done == FALSE )
    {
	wmove( wmenu, WMENU_LINES-1, WMENU_COLS-2 );
	wrefresh( wmenu );

	c = wgetch( wmenu );

	switch( c & 0x7f )
	{
	case 'a': case 'A':
	    add_breakpoint( 7 );
	    break;

	case 'k': case 'K':
	    kill_breakpoint( 7 );
	    break;

	case 't': case 'T':
	    toggle_breakpoint( 7 );
	    break;

	case 'q': case 'Q':
	    done = TRUE;
	    break;

	default:
	    break;
	}

	display_breakpoints();
    }

    display_menu();
}


/************************************************************************

                        Stack Window Functions

************************************************************************/


/*
** void display_stack( void )
**
** display_stack() draws everything in the stack window - 15 bytes of the
** stack and the current stack pointer
*/

static void display_stack( void )
{
    unsigned char  i;        /* loop index                               */
    unsigned short address;  /* address of each line of the stack window */

    for ( i = 0; i < WSTACK_LINES; i++ )
    {
	address = 0x0100 + (unsigned char)( stack_first - i );

	my_mvwaddhex_short( wstack, i, 0, address );
	waddstr( wstack, ":     " );
	my_mvwaddhex_char( wstack, i, 6, memory[address] );

	if ( address == ( 0x100 + S ) )
	{
	    stack_line = i;
	    mvwaddstr( wstack, i, 8, "<<" );
	}
    }

    wrefresh( wstack );
}

/*
** void update_stack( void )
**
** update_stack() checks to see if the area of the stack being displayed
** needs to change and then redraws the stack window
*/

static void update_stack( void )
{
    int i;                      /* loop index */
    unsigned char line_address; /* page 1 address of each line */

    line_address = stack_first;

    /* see if the address of the current stack pointer is already */
    /* being displayed in the window                              */

    for ( i = 0; i < WSTACK_LINES; i++ )
    {
	if ( S == line_address )
	    break;
	
	line_address--;
    }

    /* if not, change the display so it is */
	
    if ( i == WSTACK_LINES )
	stack_first = S;

    /* and then redraw the stack */

    display_stack();
}


/************************************************************************

                        Memory Window Functions

************************************************************************/


/*
** void display_memory_line( int, unsigned short )
**
** display_memory_line displays 8 bytes of memory in the memory window
** on the specified line of the window starting with the specified address
*/

static void display_memory_line( int display_line, unsigned short address )
{
    int i;   /* loop index         */
    int col; /* column to print at */

    col = 1;

    /* don't try to display I/O space */

    if ( address >= IO_BOTTOM && address <= IO_TOP )
    {
	wmove( wmemory, display_line, col );
	wclrtoeol( wmemory );
	mvwprintw( wmemory, display_line, col, "%4.4X:", address );
	wstandout( wmemory );
	mvwaddstr( wmemory, display_line, col + 8, "*** I/O space ***" );
	wstandend( wmemory );
	return;
    }

    /* display the address of the first byte on this line */

    mvwprintw( wmemory, display_line, col, "%4.4X:", address );

    /* display the bytes as hexadecimal numbers */

    col += 6;
    for( i = 0; i < WMEMORY_BYTES_PER_LINE; i++ )
    {
	my_mvwaddhex_char( wmemory, display_line, col, memory[address+i] );
	col += 3;
    }

    /* display the bytes as ASCII characters */

    col++;
    for( i = 0; i < WMEMORY_BYTES_PER_LINE; i++ )
    {
	if ( isprint( 0x7f & memory[address+i] ) )
	{
	    /* make high-bit-set printables inverse, bold, etc. */

	    if ( 0x80 & memory[address+i] ) wstandout( wmemory );
	    mvwaddch( wmemory, display_line, col, 0x7f & memory[address+i] );
	    wstandend( wmemory );
	}
	else
	{
	    mvwaddch( wmemory, display_line, col, '.' );
	}

	col++;
	if ( i == 3 ) col++;
    }
}

/*
** void display_memory( void )
**
** display_memory() displays everything in the memory window - WMEMORY_LINES
** containing WMEMORY_BYTES_PER_LINE each.
*/

static void display_memory( void )
{
    int i;                  /* loop index */
    unsigned short address; /* address of first byte of each line displayed */

    address = memory_first;
    for( i = 0; i < WMEMORY_LINES; i++ )
    {
	display_memory_line( i, address );
	address += WMEMORY_BYTES_PER_LINE;
    }

    wrefresh( wmemory );
}

/*
** void display_edit_memory_menu( void )
**
** display_edit_memory_menu() displays the menu for edit_memory() in the
** menu window.
*/

static void display_edit_memory_menu( void )
{
    wclear( wmenu );
    mvwaddstr( wmenu, 0, 1, "hjkl - left,down,up,right" );
    mvwaddstr( wmenu, 1, 1, " C-f - page down" );
    mvwaddstr( wmenu, 2, 1, " C-b - page up" );
    mvwaddstr( wmenu, 3, 1, "   G - goto address" );
    mvwaddstr( wmenu, 5, 1, "   S - fill memory range" );
    mvwaddstr( wmenu, 6, 1, "       with a value" );
    mvwaddstr( wmenu, 8, 1, "   Q - quit editing" );
    mvwaddstr( wmenu, 9, 1, "       memory" );
    wrefresh( wmenu );
}

/*
** unsigned short goto_memory_address( void )
**
** goto_memory_address() allows the user to enter a memory address, which
** then becomes the first address displayed in the memory window.
*/

static unsigned short goto_memory_address( void )
{
    int c;                  /* character read from keyboard */
    unsigned long address;  /* address to go to */

    /* prompt for the address */

    wclear( wmenu );
    mvwaddstr( wmenu, 1, 1, "Address: " );
    wrefresh( wmenu );

    /* read the address */

    address = memory_first;
    c = edit_hex( wmenu, 1, 10 , &address, 4 );

    display_edit_memory_menu();

    /* return the new address, or the current one if user aborted */

    return address;
}

/*
** void fill_memory_range( void )
**
** fill_memory_range() allows the user to specify a range of memory and
** a byte value. The specified range is filled with the specified value
** and the memory window is redrawn.
*/

static void fill_memory_range( void )
{
    int c;  /* character read from keyboard */

    unsigned short address;        /* loop index incrementing through range */
    unsigned short stack_address;  /* full address of top line of stack     */

    short stack_needs_update;        /* flag - need to redraw stack?       */
    short disassembly_needs_update;  /* flag - need to redraw disassembly? */

    unsigned long fill;           /* value to fill range with */
    unsigned long address_first;  /* first address in range   */
    unsigned long address_last;   /* last address in range    */

    /* initialize */

    stack_needs_update = FALSE;
    disassembly_needs_update = FALSE;

    /* prompt for the address range boundaries */

    wclear( wmenu );
    mvwaddstr( wmenu, 1, 1, "First address: " );
    wrefresh( wmenu );

    /* read the first address */

    c = edit_hex( wmenu, 1, 16 , &address_first, 4 );
    if ( c == KEY_ESCAPE || c & NO_DIGITS_ENTERED )
    {
	display_edit_memory_menu();
	return;
    }

    /* prompt for the address range boundaries */

    mvwaddstr( wmenu, 2, 1, "Last address: " );
    wrefresh( wmenu );

    /* read the last address */

    c = edit_hex( wmenu, 2, 15 , &address_last, 4 );
    if ( c == KEY_ESCAPE || c & NO_DIGITS_ENTERED )
    {
	display_edit_memory_menu();
	return;
    }

    /* prompt for the fill value */

    mvwaddstr( wmenu, 4, 1, "Fill with: " );
    wrefresh( wmenu );

    /* read the fill value */

    c = edit_hex( wmenu, 4, 12 , &fill, 2 );
    if ( c == KEY_ESCAPE || c & NO_DIGITS_ENTERED )
    {
	display_edit_memory_menu();
	return;
    }

    /* fill the range, wrapping to beginning of memory */

    address = (unsigned short)address_first;
    do {

	memory[address] = (unsigned char)fill;

    } while( address++ != address_last );

    /*
    ** for the next two if-statements. Look at the diagrams above
    ** each part of the if condition to see the condition being
    ** tested. The symbols used are:
    **
    **  [ ] - beginning and end of 65c02 address space
    **  { } - beginning and end of displayed stack/disassembly
    **  < > - beginning and end of filled range
    **   .  - filled area
    */

    /* see if we wrote on the displayed part of the stack */

    stack_address = 0x100 + stack_first;

    /*
    ** for the stack if-statement, it is important to remember that,
    ** since the stack grows down in memory, stack_address is the
    ** HIGHEST address displayed in the stack window, and that
    ** therefore the lowest address displayed in the stack window
    ** is stack_address - WSTACK_LINES + 1.
    */

    if  (

	/* [        { <...> }        ] */
	/* [        {   <...}...>    ] */
	/* [..>     {   <...}........] */
	/* [........{..> <..}........] */

	( address_first <= stack_address &&
	  address_first >  ( stack_address - WSTACK_LINES )    ) ||

	/* [    <...{...>   }        ] */
	/* [........{...>   }    <...] */

	( address_last <= stack_address &&
	  address_last >  ( stack_address - WSTACK_LINES )     ) ||

	/* [    <...{.......}...>    ] */

	( address_last  >  stack_address &&
	  address_first <= ( stack_address - WSTACK_LINES )    ) ||

	/* [........{.......}..>  <..] */

	( address_last < address_first &&
	  address_last > stack_address                         ) ||

	/* [..>  <..{.......}........] */

	( address_first <= ( stack_address - WSTACK_LINES ) &&
	  address_last  <  address_first                       )

	)
    {
	stack_needs_update = TRUE;
    }

    /* see if we wrote on the disassembled memory */

    if  (

	/* [        { <...> }        ] */
	/* [        {   <...}...>    ] */
	/* [..>     {   <...}........] */
	/* [........{..> <..}........] */

	( address_first >= disassembly_first &&
	  address_first <  disassembly_last     ) ||

	/* [    <...{...>   }        ] */
	/* [........{...>   }    <...] */

	( address_last >= disassembly_first &&
	  address_last <  disassembly_last      ) ||

	/* [    <...{.......}...>    ] */

	( address_first <  disassembly_first &&
	  address_last  >= disassembly_last     ) ||

	/* [........{.......}..>  <..] */

	( address_last <  address_first &&
	  address_last >= disassembly_last      ) ||

	/* [..>  <..{.......}........] */

	( address_first < disassembly_first &&
	  address_last  < address_first         )
	)
    {
	disassembly_needs_update = TRUE;
    }

    /* redraw affected windows */

    if ( stack_needs_update == TRUE )
    {
	display_stack();
    }

    if ( disassembly_needs_update == TRUE )
    {
	display_disassembly();
    }

    display_edit_memory_menu();
}

/*
** void edit_memory( void )
**
** edit_memory() allows the user to interactively edit the contents of
** any part of the 65c02 address space that is mapped to RAM (currently
** everything but the I/O space).
**
** The user can change which part of the 65c02 address space is displayed
** in the memory window and can edit the displayed values, changing the
** contents of memory.
*/

static void edit_memory( void )
{
    int c;     /* character read from keyboard                       */
    int line;  /* current line cursor is on                          */
    int col;   /* current column (0 <= col < WMEMORY_BYTES_PER_LINE) */

    short done;                      /* flag - is used finished?           */
    short stack_needs_update;        /* flag - need to redraw stack?       */
    short disassembly_needs_update;  /* flag - need to redraw disassembly? */

    unsigned short address;      /* address currently being edited       */
    unsigned long edited_value;  /* temporary value filled by edit_hex() */

    /* display the menu */

    display_edit_memory_menu();

    /* initialization */

    line = 0;
    col  = 0;

    done = FALSE;
    stack_needs_update = FALSE;
    disassembly_needs_update = FALSE;

    address = memory_first;

    /* act upon user selection until done */

    while( done == FALSE )
    {
	/* allow user to edit current byte, or hit another key */

	edited_value = (unsigned long)memory[address];
	c = edit_hex( wmemory, line, 7+(col*3), &edited_value, 2 );

	if ( c != KEY_ESCAPE || c & NO_DIGITS_ENTERED )
	{
	    /* if user edited byte, update the byte and redraw */
	    /* other windows as necessary                      */

	    if ( memory[address] != (unsigned char)edited_value )
	    {
		unsigned short stack_address = 0x100 + stack_first;

		if ( address >= disassembly_first &&
		     address < disassembly_last )
		{
		    disassembly_needs_update = TRUE;
		}
	    
		if ( address <= stack_address &&
		     address > ( stack_address - WSTACK_LINES ) )
		{
		    stack_needs_update = TRUE;
		}

		memory[address] = (unsigned char)edited_value;
	    }
	}

	/* see what user wants to do, based on character returned */
	/* from edit_hex()                                        */

	switch( c & 0x7f )
	{
	case 'q': case 'Q':
	    done = TRUE;
	    break;

	case 'g': case 'G':
	    address = memory_first = goto_memory_address();
	    line = col = 0;
	    break;

	case 's': case 'S':
	    fill_memory_range();
	    break;

	case 'h': case 'H':
	    address--;
	    if ( col > 0 )
		col--;
	    else
	    {
		col = WMEMORY_BYTES_PER_LINE - 1;
		if ( line > 0 )
		    line--;
		else
		    memory_first -= WMEMORY_BYTES_PER_LINE;
	    }
	    break;

	case 'l': case 'L':
	case '\r': case '\n': case ' ':
	    address++;
	    if ( col < ( WMEMORY_BYTES_PER_LINE - 1 ) )
		col++;
	    else
	    {
		col = 0;
		if ( line < ( WMEMORY_LINES - 1 ) )
		    line++;
		else
		    memory_first += WMEMORY_BYTES_PER_LINE;
	    }
	    break;

	case 'k': case 'K':
	    address -= WMEMORY_BYTES_PER_LINE;
	    if ( line > 0 )
		line--;
	    else
		memory_first -= WMEMORY_BYTES_PER_LINE;
	    break;

	case 'j': case 'J':
	    address += WMEMORY_BYTES_PER_LINE;
	    if ( line < ( WMEMORY_LINES - 1 ) )
		line++;
	    else
		memory_first += WMEMORY_BYTES_PER_LINE;
	    break;

	case KEY_CONTROL_F:
	    address += ( WMEMORY_LINES * WMEMORY_BYTES_PER_LINE );
	    memory_first += ( WMEMORY_LINES * WMEMORY_BYTES_PER_LINE );
	    break;

	case KEY_CONTROL_B:
	    address -= ( WMEMORY_LINES * WMEMORY_BYTES_PER_LINE );
	    memory_first -= ( WMEMORY_LINES * WMEMORY_BYTES_PER_LINE );
	    break;
	}
	
	display_memory();

	if ( stack_needs_update )
	{
	    display_stack();
	    stack_needs_update = FALSE;
	}

	if ( disassembly_needs_update )
	{
	    display_disassembly();
	    disassembly_needs_update = FALSE;
	}
    }
}


/************************************************************************

                         Watch Window Functions

************************************************************************/


/*
** void display_watches( void )
**
** display_watches displays all watches in the watches window
*/

static void display_watches( void )
{
    int i;

    wclear( wwatches );

    for ( i = 0; i < num_watches; i++ )
    {
	mvwprintw( wwatches, i, 1, "%2d %4.4X: %2.2X", i+1, watches[i].address,
		   memory[ watches[i].address ] );
    }

    wrefresh( wwatches );
}

/*
** int insert_watch( unsigned short address )
**
** insert_watch() inserts a watch into the watches[] array which is kept
** in order sorted by address. insert_watch() returns FALSE (0) if the new
** watch cannot be inserted into the watches[] array. Currently, this only
** happens if there is already a watch at the specified address. Otherwise,
** insert_watch() returns TRUE (1)
*/

static int insert_watch( unsigned short address )
{
    int i, j;  /* loop indices */

    for ( i = 0; i < WWATCHES_LINES; i++ )
    {
	/* find place for this watch */
	if ( watches[i].used == 0 || address < watches[i].address )
	    break;

	/* only one watch allowed at any address */
	if ( address == watches[i].address )
	    return FALSE;
    }

    if ( watches[i].used )
    {
	/* if we're inserting in the middle, move the   */
	/* watches at higher addresses up one space */

	for ( j = WWATCHES_LINES - 1; j > i; j-- )
	    watches[j] = watches[j-1];
    }

    /* watches[i] is now free */

    watches[i].used = 1;
    watches[i].address = address;

    num_watches++;

    return TRUE;
}

/*
** void add_watch( void )
**
** add_watch() allows the user to interactively add a new watch by specifying
** address to watch. add_watch() will fail to add a new watch if there is
** already a watch at the specified address or if there is not room for
** another watch in the watches[] array.
**
** prompt and character echoing is done on the line in the menu window
** specified by prompt_line
*/

static void add_watch( int prompt_line )
{
    int c;                       /* character read from keyboard */
    unsigned long watch_address; /* address of the new watch */

    if ( num_watches == WWATCHES_LINES )
    {
	mvwaddstr( wmenu, prompt_line, 1, "Too many watches" );
	return;
    }

    wmove( wmenu, prompt_line, 1 );
    wclrtoeol( wmenu );
    mvwaddstr( wmenu, prompt_line, 1, "Address: " );
    wrefresh( wmenu );

    c = edit_hex( wmenu, prompt_line, 10, &watch_address, 4 );
    if ( c == KEY_ESCAPE || c & NO_DIGITS_ENTERED )
    {
	wmove( wmenu, prompt_line, 1 );
	wclrtoeol( wmenu );
	return;
    }

    if ( !insert_watch( (unsigned short)watch_address ) )
    {
	wmove( wmenu, prompt_line, 1 );
	wclrtoeol( wmenu );
	mvwaddstr( wmenu, prompt_line, 1, "Watch already set" );
    }
    
    wmove( wmenu, prompt_line, 1 );
    wclrtoeol( wmenu );
}

/*
** void delete_watch( int index )
**
** delete_watch() removes a breakpoint from the watches[] array
** as specified by an index into the watches[] array
*/

static void delete_watch( int index )
{
    int i;  /* loop index */

    /* check for the easy case - the last watch */
    if ( index == ( num_watches - 1 ) )
	watches[index].used = 0;

    /* move the ones after index down, obliterating index */
    for ( i = index; i < ( num_watches - 1 ); i++ )
	watches[i] = watches[i+1];

    num_watches--;
}

/*
** void kill_watch( int prompt_line )
**
** kill_watch() allows the user to interactively remove a watch from the
** watches[] array by specifying (by index into the array) which watch
** is to be removed.
**
** prompt and character echoing is done on the line in the menu window
** specified by prompt_line
*/

static void kill_watch( int prompt_line )
{
    int c;             /* character read from keyboard */
    long watch_index;  /* the watch to remove */

    /* prompt for a watch to remove */

    wmove( wmenu, prompt_line, 1 );
    wclrtoeol( wmenu );
    mvwaddstr( wmenu, prompt_line, 1, "Which: " );
    wrefresh( wmenu );

    /* get the watch to remove */

    c = edit_dec( wmenu, prompt_line, 8, &watch_index, 2, FALSE );
    if ( c == KEY_ESCAPE || c & NO_DIGITS_ENTERED )
    {
	wmove( wmenu, prompt_line, 1 );
	wclrtoeol( wmenu );
	return;
    }

    /* don't try remove a non-existant watch */

    if ( watch_index > num_watches )
    {
	wmove( wmenu, prompt_line, 1 );
	wclrtoeol( wmenu );
	mvwaddstr( wmenu, prompt_line, 1, "No such watch" );
	return;
    }

    /* remove the watch */

    delete_watch( watch_index - 1 );

    wmove( wmenu, prompt_line, 1 );
    wclrtoeol( wmenu );
}

/*
** void edit_watch( void )
**
** edit_watch() presents a menu of watch editing options to the user
** and allows him to select one.
*/

static void edit_watch( void )
{
    int   c;    /* character read from keyboard             */
    short done; /* flag - is user finished editing watches? */

    /* initialize */

    done = FALSE;

    /* display the menu */

    wclear( wmenu );
    mvwaddstr( wmenu, 0, 1, "A - add watch" );
    mvwaddstr( wmenu, 1, 1, "K - kill watch" );
    mvwaddstr( wmenu, 3, 1, "Q - quit editing" );
    mvwaddstr( wmenu, 4, 1, "    watches" );
    wrefresh( wmenu );

    /* act upon selections until done */

    while ( done == FALSE )
    {
	wmove( wmenu, WMENU_LINES-1, WMENU_COLS-2 );
	wrefresh( wmenu );

	c = wgetch( wmenu );

	switch( c )
	{
	case 'a': case 'A':
	    add_watch( 6 );
	    break;

	case 'k': case 'K':
	    kill_watch( 6 );
	    break;

	case 'q': case 'Q':
	    done = TRUE;
	    break;

	default:
	    break;
	}

	display_watches();
    }

    display_menu();
}


/************************************************************************

                     Disassembly Window Functions

************************************************************************/


/*
** void disassemble_data_byte( int y, int x1, int x2,
**                             unsigned short address )
**
** disassemble_data_byte prints a byte of memory in the disassembly
** window as a hexadecimal number at coordinates y,x1 and as an
** ASCII character at coordinates y,x2.
*/

static void disassemble_data_byte( int y, int x1, int x2,
				   unsigned short address )
{
    /* print as a hexadecminal number */

    my_mvwaddhex_char( wdisassembly, y, x1, memory[address] );

    /* print as an ASCII character */

    if ( isprint( 0x7f & memory[address] ) )
    {
	if ( 0x80 & memory[address] ) wstandout( wdisassembly );
	mvwaddch( wdisassembly, y, x2, 0x7f & memory[address] );
	wstandend( wdisassembly );
    }
    else
    {
	mvwaddch( wdisassembly, y, x2, '.' );
    }
}

/*
** static void disassemble_data( int y, int x,
**                               unsigned short address,
**                               int min_bytes, int max_bytes )
**
** disassemble_data prints not less than min_bytes, nor more than max_bytes
** of memory in the disassembly window as data, starting with the byte at
** the specified address.
*/

static void disassemble_data( int y, int x,
			      unsigned short address,
			      int min_bytes, int max_bytes )
{
    int i;       /* loop index */
    int x1, x2;  /* x-coordinates for printing bytes as hex and ASCII */

    /* initialize */

    x1 = x;
    x2 = x + 13;

    disassembly_len[ y ] = 0;
    mvwaddch( wdisassembly, y, x2++, '[' );

    /* do min_bytes bytes without checking anything */

    for( i = 0; i < min_bytes; i++ )
    {
	disassemble_data_byte( y, x1, x2++, address++ );
	disassembly_len[ y ]++;
	x1 += 3;
    }

    /* do the rest (if any) only if the byte is not a valid opcode */

    for( ; i < max_bytes; i++ )
    {
	if ( 0 == instruction_size[ memory[ address ] ] )
	{
	    disassemble_data_byte( y, x1, x2++, address++ );
	    disassembly_len[ y ]++;
	    x1 += 3;
	}
    }

    mvwaddch( wdisassembly, y, x2, ']' );
}

/*
** void disassemble_three_bytes( int y, int x, unsigned short address )
**
** disassemble_three_bytes() disassembles a three-byte 65c02 instruction.
*/

static void disassemble_three_bytes( int y, int x, unsigned short address )
{
    unsigned short operand =
	memory[ address+1 ] + 256*memory[ address+2 ];

    mvwprintw( wdisassembly, y, x,
	       instruction_strings[ memory[ address ] ], operand );
}

/*
** void disassemble_two_bytes( int y, int x, unsigned short address )
**
** disassemble_two_bytes() disassembles a two-byte 65c02 instruction. For
** a branch instruction it also prints the branch target. For an immediate
** addressing mode instruction, it prints the ASCII value of the operand.
*/

static void disassemble_two_bytes( int y, int x, unsigned short address )
{
    unsigned char opcode  = memory[ address ];
    unsigned char operand = memory[ address+1 ];

    if ( instruction_is_branch[ opcode ] )
    {
	mvwprintw( wdisassembly, y, x,
		   instruction_strings[ opcode ], (signed char)operand );
	mvwprintw( wdisassembly, y, x+13, "[$%4.4X]",
		   address + 2 + (signed char)operand );
    }
    else
    {
	mvwprintw( wdisassembly, y, x,
		   instruction_strings[ opcode ], operand );

	if ( instruction_is_immediate[ opcode ] )
	{
	    if ( isprint( 0x7f & operand ) )
	    {
		mvwaddstr( wdisassembly, y, x+15, "[' ']" );
		mvwaddch( wdisassembly, y, x+17, operand );
	    }
	}
    }
}

/*
** void disassemble_one_byte( int y, int x, unsigned short address )
**
** disassemble_one_byte() disassembles a one-byte 65c02 instruction.
*/

static void disassemble_one_byte( int y, int x, unsigned short address )
{
    mvwaddstr( wdisassembly, y, x,
	       instruction_strings[ memory[ address ] ] );
}

/*
** void disassemble_line( int y, int x, unsigned short address )
**
** disassemble_line() disassembles one line at the specified address.
** one line is one 65c02 instruction, one to four bytes of data,
** or the entire range of I/O space.
*/

static void disassemble_line( int y, int x, unsigned short address )
{
    wmove( wdisassembly, y, x );
    wclrtoeol( wdisassembly );

    /* Don't try to disassemble the I/O space */

    if ( address == IO_BOTTOM )
    {
	mvwaddstr( wdisassembly, y, x+2, IO_BOTTOM_STR );
	wstandout( wdisassembly );
	mvwaddstr( wdisassembly, y, x+10, "*** I/O space ***" );
	wstandend( wdisassembly );
	disassembly_len[ y ] = 256;
	return;
    }

    mvwprintw( wdisassembly, y, x+2, "%4.4X:", address );

    if ( 3 == instruction_size[ memory[ address ] ] &&
	 ( address > IO_TOP || (address+2) < IO_BOTTOM ) )
    {
	/* Handle three byte instructions that don't cross */
	/* into the I/O space                              */

	mvwprintw( wdisassembly, y, x+8, "%2.2X %2.2X %2.2X",
		   memory[address+0], memory[address+1], memory[address+2] );

	disassemble_three_bytes( y, x+21, address );
	disassembly_len[ y ] = 3;
    }
    else if ( 2 == instruction_size[ memory[ address ] ] &&
	      ( address > IO_TOP || (address+1) < IO_BOTTOM ) )
    {
	/* Handle two byte instructions that don't cross into the I/O space */

	mvwprintw( wdisassembly, y, x+8, "%2.2X %2.2X",
		   memory[address+0], memory[address+1] );

	disassemble_two_bytes( y, x+21, address );
	disassembly_len[ y ] = 2;
    }
    else if ( 1 == instruction_size[ memory[ address ] ] )
    {
	/* Handle one byte instructions */

	mvwprintw( wdisassembly, y, x+8, "%2.2X", memory[address+0] );

	disassemble_one_byte( y, x+21, address );
	disassembly_len[ y ] = 1;
    }
    else
    {
	/* disassemble a line of data */

	if ( 0 == instruction_size[ memory[ address ] ] )
	{
	    /* treat non-opcodes as data, print up to four on a line */

	    if ( address > IO_TOP || (address+3) < IO_BOTTOM )
		disassemble_data( y, x+8, address, 1, 4 );
	    else
		disassemble_data( y, x+8, address, 1, IO_BOTTOM - address );
	}
	else
	{
	    /* for apparent instructions that cross into I/O space */

	    disassemble_data( y, x+8, address,
			      IO_BOTTOM - address, IO_BOTTOM - address );
	}
    }

    /* indicate that this line is the next to be executed */

    if ( emPC == address )
    {
	pc_line = y;
	mvwaddstr( wdisassembly, y, x, ">>" );
    }
}

/*
** void display_disassembly( void )
**
** display_disassembly() disassembles enough of the contents of memory,
** starting at disassembly_first, to fill the disassembly window
*/

static void display_disassembly( void )
{
    int i;                   /* loop index                            */
    unsigned short address;  /* the address of each disassembled line */

    address = disassembly_first;
    for( i = 0; i < WDISASSEMBLY_LINES; i++ )
    {
	disassemble_line( i, 1, address );
	address += disassembly_len[ i ];
    }

    disassembly_last = address;
    wrefresh( wdisassembly );
}

/*
** void update_disassembly( void )
*/

static void update_disassembly( void )
{
    int i;                        /* loop index */
    unsigned short line_address;  /* address of lines in the window */

    /* see if the PC is currently in the disassembly window */

    line_address = disassembly_first;
    for ( i = 0; i < WDISASSEMBLY_LINES; i++ )
    {
	if ( emPC == line_address )
	    break;
	
	line_address += disassembly_len[ i ];
    }

    if ( i >= WDISASSEMBLY_LINES )
	disassembly_first = emPC;

    display_disassembly();

#if 0
    if ( i < WDISASSEMBLY_LINES )
    {
	/* if it is, just move the PC indicator */

	mvwaddstr( wdisassembly, pc_line, 1, "  " );
	pc_line = i;
	mvwaddstr( wdisassembly, pc_line, 1, ">>" );

	wrefresh( wdisassembly );
    }
    else
    {
	/* otherwise redraw the entire window */

	disassembly_first = emPC;
	display_disassembly();
    }
#endif
}


/************************************************************************

                     Main Menu Selection Functions

************************************************************************/


/*
** void edit_register_memory( void )
**
** edit_register_memory() allows the user to select whether to edit
** registers or memory.
*/

static void edit_register_memory( void )
{
    int c; /* character read from keyboard */

    wclear( wmenu );
    mvwaddstr( wmenu, 0, 1, "R - edit registers" );
    mvwaddstr( wmenu, 1, 1, "M - edit memory" );
    mvwaddstr( wmenu, 3, 1, "Q - don't edit anything" );
    wmove( wmenu, WMENU_LINES-1, WMENU_COLS-2 );
    wrefresh( wmenu );

    c = wgetch( wmenu );

    switch( c )
    {
    case 'r': case 'R':
	edit_registers();
	break;

    case 'm': case 'M':
	edit_memory();
	break;

    default:
	break;
    }

    display_menu();
}

/*
** void load_file( char *fname, unsigned short address )
**
** load file reads the specified file into memory at the specified
** address.
*/

void load_file( char *fname, unsigned short address )
{
    FILE *f;
    size_t bytes_read;  /* # of bytes read by read()    */

    f = fopen( fname, "r" );

    emPC = (unsigned short)address;
    disassembly_first = emPC;
    memory_first = emPC;

    do {

	if ( address & 0xff )
	{
	    bytes_read =
		fread( &(memory[ (unsigned short)address ]), 1,
		       256 - (unsigned short)( address & 0xff ), f );
	}
	else
	{
	    bytes_read =
		fread( &(memory[ (unsigned short)address ]), 1, 256, f );
	}

	address = (unsigned short)( address +  bytes_read );

    } while( bytes_read );

    fclose( f );
}

/*
** void load_file_interactive( void )
**
** load_file_interactive() allows the user to interactively select a file
** to load and an address as which to to it, and then the file is loaded.
*/

static void load_file_interactive( void )
{
    int c;              /* character read from keyboard */
    FILE *f;            /* the file to load             */
    long load_address;  /* where to load it             */
    char fname[256];    /* the name of the file to load */

    /* get the name of the file to load */

    wclear( wmenu );
    mvwaddstr( wmenu, 0, 1, "File (ESC to abort):" );
    wrefresh( wmenu );

    while( 1 )
    {
	c = edit_string( wmenu, 1, 1, fname, 24, 255 );
	if ( c == KEY_ESCAPE )
	{
	    display_menu();
	    return;
	}

	f = fopen( fname, "r" );
	if ( f )
	{
	    fclose( f );
	    break;
	}
	else
	{
	    wmove( wmenu, 3, 1 );
	    wclrtoeol( wmenu );
	    wmove( wmenu, 1, 1 );
	    wclrtoeol( wmenu );
	    mvwaddstr( wmenu, 3, 1, "No such file" );
	}
    }

    /* get the address at which to load it */

    wmove( wmenu, 3, 1 );
    wclrtoeol( wmenu );
    mvwaddstr( wmenu, 3, 1, "Address: " );
    wrefresh( wmenu );

    c = edit_hex( wmenu, 3, 10, &load_address, 4 );
    if ( c == KEY_ESCAPE || c & NO_DIGITS_ENTERED )
    {
	wmove( wmenu, 3, 1 );
	wclrtoeol( wmenu );
	return;
    }

    /* load it */
    
    load_file( fname, load_address );

    display_stack();
    display_memory();
    display_disassembly();

    display_menu();
}

/*
** void display_help( void )
**
** display_help() allows the user to read the em65 help files
*/

static void display_help( void )
{
    int c;     /* character read from keyboard */
    int done;  /* loop control variable        */

    wclear( wmenu );
    mvwprintw( wmenu, 0, 1, "%s Help", em65_version_string );
    mvwaddstr( wmenu, 2, 1, "D - em65 debugger" );
    mvwaddstr( wmenu, 3, 1, "P - em65 programming" );
    mvwaddstr( wmenu, 4, 1, "E - em65 environment" );
    mvwaddstr( wmenu, 5, 1, "I - em65 implementation" );
    mvwaddstr( wmenu, 7, 1, "Q - quit help" );
    wmove( wmenu, WMENU_LINES-1, WMENU_COLS-2 );
    wrefresh( wmenu );

    done = FALSE;
    while ( done == FALSE )
    {
	c = wgetch( wmenu );
	switch( c )
	{
	case 'd': case 'D':
	    page_file( "debugger.txt" );
	    break;
	    
	case 'p': case 'P':
	    page_file( "65c02.txt" );
	    break;
	    
	case 'e': case 'E':
	    page_file( "environment.txt" );
	    break;
	    
	case 'i': case 'I':
	    page_file( "implementation.txt" );
	    break;
	    
	case 'q': case 'Q':
	    done = TRUE;
	    break;
	    
	default:
	    break;
	}
    }

    display_menu();
}

/*
** void display_info( void )
**
** display_info() displays the copyright information about em65
** and allows the user to read the GNU General Public License
*/

static void display_info( void )
{
    int c;     /* character read from keyboard */
    int done;  /* loop control variable        */

    wclear( wmenu );
    mvwprintw( wmenu, 0, 1, "%s - 65c02 emulator", em65_version_string );
    mvwprintw( wmenu, 2, 1, "%s", em65_copyright_string );
    mvwprintw( wmenu, 3, 1, "by %s", em65_author_string );
    mvwprintw( wmenu, 4, 1, "   %s", em65_email_string );
    mvwaddstr( wmenu, 6, 1, "G - gnu license" );
    mvwaddstr( wmenu, 8, 1, "Q - quit info" );
    wmove( wmenu, WMENU_LINES-1, WMENU_COLS-2 );
    wrefresh( wmenu );

    done = FALSE;
    while ( done == FALSE )
    {
	c = wgetch( wmenu );
	switch( c )
	{
	case 'g': case 'G':
	    page_file( "COPYING" );
	    break;
	    
	case 'q': case 'Q':
	    done = TRUE;
	    break;
	    
	default:
	    break;
	}
    }

    display_menu();
}

/*
** void display_menu( void )
**
** display_menu() prints the main menu in the menu window.
*/

static void display_menu( void )
{
    wclear( wmenu );
    mvwprintw( wmenu, 0, 1, "%s <> 9/17/1997", em65_version_string );
    mvwaddstr( wmenu, 2, 1, "N - next    O - over" );
    mvwaddstr( wmenu, 3, 1, "T - trace   R - run" );
    mvwaddstr( wmenu, 4, 1, "B - break   W - watch" );
    mvwaddstr( wmenu, 5, 1, "E - edit    L - load" );
    mvwaddstr( wmenu, 6, 1, "S - screen  M - mnemonics" );
    mvwaddstr( wmenu, 8, 1, "H - help    I - info" );
    mvwaddstr( wmenu, 9, 1, "Q - quit" );
    wrefresh( wmenu );
}

/*
** void ensure_debugger_screen( void )
**
** ensure_debugger_screen() checks to see if the debugger is currently
** controlling the screen. If not, it takes it back and refreshes
** the debugger display.
*/

static void ensure_debugger_screen( void )
{
    /* if the debugger doesn't currently control the screen */
    if ( !debug_controls_screen )
    {
	/* tell the virtual terminal to give it up */
	(void)terminal_talk( TERMINAL_NOWRITE );

	/* mark our possession */
	debug_controls_screen = 1;

	/* redraw the debugger display */
	/*
	touchwin( wregisters );
	touchwin( wbreakpoints );
	touchwin( wwatches );
	touchwin( wdisassembly );
	touchwin( wstack );
	touchwin( wmemory );
	touchwin( wmenu );
	*/
	touchwin( wdebug );
	init_display();
    }
}

/*
** void debug_command_loop( void )
**
** debug_command_loop() allows the user to select commands from the
** main menu repeatedly until one of the selections causes 65c02
** execution to resume or cause this program to terminate.
*/

static void debug_command_loop( void )
{
    int c;     /* character read from keyboard */
    int done;  /* loop control variable        */

    done = FALSE;
    while( done == FALSE )
    {
	if ( debug_controls_screen )
	{
	    wmove( wmenu, WMENU_LINES-1, WMENU_COLS-2 );
	    wrefresh( wmenu );
	}

	c = wgetch( wmenu );

	switch( c & 0x7f )
	{
	case 'n': case 'N':
	    done = TRUE;
	    break;

	case 'o': case 'O':
	    if ( memory[emPC] == 0x20 ) /* JSR */
		stepover = 1;
	    done = TRUE;
	    break;

	case 't': case 'T':
	    stepping = 0;
	    done = TRUE;
	    break;

	case 'r': case 'R':
	    terminal_tell( TERMINAL_WRITE );
	    running = 1;
	    stepping = 0;
	    debug_controls_screen = 0;
	    done = TRUE;
	    break;

	case 'b': case 'B':
	    ensure_debugger_screen();
	    edit_breakpoint();
	    done = FALSE;
	    break;

	case 'w': case 'W':
	    ensure_debugger_screen();
	    edit_watch();
	    done = FALSE;
	    break;

	case 'e': case 'E':
	    ensure_debugger_screen();
	    edit_register_memory();
	    done = FALSE;
	    break;

	case 'l': case 'L':
	    ensure_debugger_screen();
	    load_file_interactive();
	    done = FALSE;
	    break;

	case 's': case 'S':
	    clear();
	    refresh();
	    if ( debug_controls_screen )
	    {
		terminal_tell( TERMINAL_WRITE );
		debug_controls_screen = 0;
	    }
	    else
	    {
		(void)terminal_talk( TERMINAL_NOWRITE );
		debug_controls_screen = 1;
		init_display();
	    }
	    done = FALSE;
	    break;

	case 'm': case 'M':
	    ensure_debugger_screen();
	    if ( instruction_strings == standard_instruction_strings )
		instruction_strings = orthogonal_instruction_strings;
	    else
		instruction_strings = standard_instruction_strings;

	    display_disassembly();
	    done = FALSE;
	    break;

	case 'h': case 'H':
	    ensure_debugger_screen();
	    display_help();
	    done = FALSE;
	    break;

	case 'i': case 'I':
	    ensure_debugger_screen();
	    display_info();
	    done = FALSE;
	    break;

	case 'q': case 'Q':
	    quitting = TRUE;
	    done = TRUE;
	    break;

	default:
	    done = FALSE;
	    break;
	}
    }
}


/************************************************************************

                      Debugger Display Functions

************************************************************************/


/*
** void display_windows( void )
**
** display_windows() draws all debugger windows
*/

static void display_windows( void )
{
    display_registers();
    display_breakpoints();
    display_stack();
    display_memory();
    display_watches();
    display_disassembly();
    display_menu();
}

/*
** void draw_vline( int y1, int y2, int x, char c )
**
** draw_vline draws a vertical line from y1,x to y2,x made of
** the specified characters.
*/

void draw_vline( int y1, int y2, int x, char c )
{
    assert( y1 >= 0 && y1 < LINES );
    assert( y2 >= 0 && y2 < LINES && y2 >= y1 );
    assert( x >= 0 && x < COLS );

    while( y1 <= y2 )
    {
	mvwaddch( wdebug, y1++, x, c );
    }
}

/*
** void init_display( void )
**
** init_display() draws the entire debugger screen.
*/

static void init_display( void )
{
    draw_vline( 0,  23, 10, '|' );
    draw_vline( 0,  11, 22, '|' );
    draw_vline( 0,  11, 36, '|' );
    draw_vline( 13, 23, 53, '|' );

    mvwaddstr( wdebug, 8,  0,  "----------+" );
    mvwaddstr( wdebug, 12, 10, "+-----------+-------------+"
                               "----------------+--------------------------" );
    wrefresh( wdebug );

    display_windows();
}


/************************************************************************

                      Execution Control Functions

************************************************************************/


/*
** void test_breakpoints( void )
**
** test_breakpoints() tests to see whether any of the currently set
** breakpoints are triggered by the current 65c02 machine state. If
** any are, test_breakpoints() sets the flag 'stepping' so that the
** 65c02 emulation will break and enter the debugger.
*/

static void test_breakpoints( void )
{
    int i;  /* loop index */

    for( i = 0; i < num_breakpoints; i++ )
    {
	if ( emPC == breakpoints[i].address && breakpoints[i].enabled == 1 )
	{
	    running = 0;
	    stepping = 1;
	    stepover = 0;

	    ensure_debugger_screen();

	    return;
	}
    }
}

/*
** void sigint_respond( void )
**
** sigint_respond() responds to a SIGINT by redrawing the screen and
** entering into single-step mode.
*/

static void sigint_respond( void )
{
    /* own the screen */

    if ( !debug_controls_screen )
    {
	(void)terminal_talk( TERMINAL_NOWRITE );
	debug_controls_screen =1;
    }

    /* touch everything because of possible screen corruption */

    touchwin( wdebug );
    wrefresh( wdebug );

    touchwin( wstack );
    touchwin( wregisters );
    touchwin( wbreakpoints );
    touchwin( wwatches );
    touchwin( wmemory );
    touchwin( wdisassembly);
    touchwin( wmenu );

    /* redraw these because they aren't ordinarily updated every step */

    display_breakpoints();
    display_menu();

    /* go into single-step mode */

    running = 0;
    stepover = 0;
    stepping = 1;

    /* clear sigint flag */

    got_sigint = 0;
}

/*
** void debugger_step( void )
**
** debugger_step() is the em65 debugger's main function. debugger_step()
** updates the debugger screen to reflect the current 65c02 machine state
** and allows the user to control 65c02 execution by using the debugger
** functionality.
**
** debugger_step() also handles keyboard input for the virtual terminal
** since the debugger owns the physical terminal.
*/

void debugger_step( void )
{
    if ( got_sigint )
    {
	/* respond to SIGINT (usually Control-C on the keyboard) */
	sigint_respond();
    }

    /* if executing (semi-)normally, check for a breakpoint */

    if ( !stepping )
	test_breakpoints();

    /* update debugger screen if terminal isn't using it */

    if ( debug_controls_screen )
    {
	if ( starting )
	{
	    stack_first = S;
	    memory_first = emPC;
	    disassembly_first = emPC;
	    init_display();
	    starting = 0;
	}
	else if ( !running && !stepover )
	{
	    display_registers();
	    update_stack();
	    display_memory();
	    display_watches();
	    update_disassembly();
	}
    }

    /* if stepping over a subroutine, see if the instruction about */
    /* to be executed is an RTS. If it is, end stepover so we will */
    /* stop at first instruction after RTS                         */

    if ( stepover )
    {
	if ( memory[emPC] == 0x60 ) /* RTS */
	    stepover = 0;
    }

    /* if stepping, then halt and let user enter commands   */

    else if ( stepping )
    {
	/* allow user to select as many commands as he wishes */

	debug_command_loop();
    }
}

/*
** int debugger_init( void )
**
** debugger_init() allocates the debugger windows
*/

int debugger_init( void )
{
    wdebug       = newwin( 24, 80, 0,  0  );
    wregisters   = newwin( 8,  10, 0,  0  );
    wbreakpoints = newwin( 12, 11, 0,  11 );
    wstack       = newwin( 15, 10, 9,  0  );
    wmemory      = newwin( 11, 42, 13, 11 );
    wwatches     = newwin( 12, 13, 0,  23 );
    wdisassembly = newwin( 12, 43, 0,  37 );
    wmenu        = newwin( 11, 26, 13, 54 );

    if( wregisters && wbreakpoints && wstack && wmemory &&
	wwatches   && wdisassembly && wmenu  && wdebug )
    {
	return 1;
    }
    else
    {
	return 0;
    }
}

/*
** End of debugger.c
*/

