
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include "cmdline.h"  /* em65_disk_filename */
#include "cpu.h"
#include "disk.h"
#include "em65.h"

/* file descriptors for pipes used for IPC */
int to_disk[2];
int from_disk[2];

/* file descriptor of file acting as "disk" */
int disk_fd;

/*
** This fictitious disk has a peculiar quirk.
** To make disks > 32mb possible and yet not require 4 byte block numbers
** for all reads and writes (which is a pain on the 65c02), I pretend the
** disk controller divides the disk into "physical partitions" of 32mb
** in size. Each read or write accesses one of 65536 blocks on the "current"
** physical partition. A separate command switches physical partitions.
** Naturally, this makes the 65c02 disk controller code more complex if
** it is to handle a disk > 32mb. But since that seems unlikely anyway...
*/

/* Fictitous geometry - corresponds to a disk one 1 32mb "partition"   */
/* This is just for fun. The virtual disk controller provides a higher */
/* level interface */

unsigned char   num_heads = 4;
unsigned char   num_sectors_per_track = 16;
unsigned short  num_bytes_per_sector = 512;
unsigned short  num_cylinders = 1024;

/* This is somewhat real. It needs to be in sync with the above    */
/* information and is the size of the virtual disk divided by 32mb */

unsigned short  num_phys_parts = 1;

/* Real information, but kept just for fun */

unsigned long   num_reads;
unsigned long   num_writes;

/* functions called in "CPU" process to communicate with "disk" process */

void setup_disk_pipes( void )
{
    /* int pipe_flags; */

    /* open the to_disk pipe */
    if ( pipe( to_disk ) )
    {
	perror( "In setup_disk_pipes (pipe)<1>" );
	exit( 1 );
    }

    /* open the from_disk pipe */
    if ( pipe( from_disk ) )
    {
	perror( "In setup_disk_pipes (pipe)<2>" );
	exit( 1 );
    }

#if 0
    /* Make to_disk pipe reads nonblocking */
    pipe_flags = fcntl( to_disk[ PIPE_READ ], F_GETFL, 0 );
    if ( pipe_flags == -1 )
    {
	perror( "In setup_disk_pipes (F_GETFL)" );
	exit( 1 );
    }

    pipe_flags |= O_NONBLOCK;
    if ( -1 == fcntl( to_disk[ PIPE_READ ], F_SETFL, pipe_flags ) )
    {
	perror( "In setup_disk_pipes (F_SETFL)" );
	exit( 1 );
    }
#endif
}

void disk_init( void )
{
    setup_disk_pipes();
    disk_fd = open( em65_disk_filename, O_CREAT | O_RDWR, 0644 );
    if ( disk_fd == -1 )
    {
	perror( "In disk_init (open)" );
	exit( 1 );
    }

    read( disk_fd, &num_reads, sizeof( long ) );
    read( disk_fd, &num_writes, sizeof( long ) );
}

void disk_shutdown( void )
{
    close( disk_fd );
}

int disk_talk( int request )
{
    int result, response;

    result = write( to_disk[ PIPE_WRITE ], &request, sizeof( int ) );
    if ( result == -1 )
    {
	perror( "In disk_talk (write)" );
	exit( 1 );
    }

    result = read( from_disk[ PIPE_READ ], &response, sizeof( int ) );
    if ( result == -1 )
    {
	perror( "In disk_talk (read)" );
	exit( 1 );
    }

    return response;
}

unsigned char disk_flag( void )
{
    return (unsigned char)(IPC_vars->disk_status);
}

unsigned char disk_read( void )
{
    return (unsigned char)disk_talk( DISK_RESET );
}

void disk_write( unsigned char byte )
{
    disk_talk( DISK_COMMAND | (unsigned int)byte );
}

/*
** +----------------------------------------------------------+
**
** Everything above this is executed in the "CPU" process.
** Everything below this is executed in the "disk" process.
**
** +----------------------------------------------------------+
*/

#define DCMD_INFO   0x01
#define DCMD_READ   0x02
#define DCMD_WRITE  0x03

int do_disk_command;
int disk_command;
int disk_block_req;
int disk_memory_addr;

/* See if "CPU" has sent a message */
int disk_read_message( void )
{
    int result, message;

    result = read( to_disk[ PIPE_READ ], &message, sizeof( int ) );
    if ( result == -1 )
    {
	if ( errno == EAGAIN )
	    return 0;
	else
	{
	    perror( "In disk_read_message (read)" );
	    exit( 1 );
	}
    }

    /* got a message, return it */
    return message;
}

void disk_respond( int response )
{
    int result;

    result = write( from_disk[ PIPE_WRITE ], &response, sizeof( int ) );
    if ( result == -1 )
    {
	perror( "In disk_respond (write)" );
	exit( 1 );
    }
}

/* keeps track of what bytes the disk controller is looking for */
int disk_cmd_state = 0;

void disk_handle_command( int cmd_byte )
{
    switch( disk_cmd_state )
    {
    case 0:
	switch( cmd_byte )
	{
	case DCMD_INFO:
	    disk_command = cmd_byte;
	    disk_cmd_state = 3;
	    break;

	case DCMD_READ:
	case DCMD_WRITE:
	    disk_command = cmd_byte;
	    disk_cmd_state = 1;
	    break;

	default:
	    break;
	}
	break;

    case 1:
	disk_block_req = cmd_byte;
	disk_cmd_state = 2;
	break;

    case 2:
	disk_block_req += ( 256 * cmd_byte );
	disk_cmd_state = 3;
	break;

    case 3:
	disk_memory_addr = cmd_byte;
	disk_cmd_state = 4;
	break;

    case 4:
	disk_memory_addr += ( 256 * cmd_byte );
	disk_cmd_state = 0;
	do_disk_command = 1;
	break;
    }
}

void disk_handle_message( int message )
{
    switch( message & 0xff00 )
    {
    case DISK_RESET:
	disk_respond( 1 );
	disk_cmd_state = 0;
	break;

    case DISK_COMMAND:
        disk_respond( 1 );
	disk_handle_command( message & 0xff );
 	break;

    default:
	break;
    }
}

void disk_info( void )
{
    unsigned char *mptr = memory + disk_memory_addr;

    *mptr++ = num_heads;

    *mptr++ = num_sectors_per_track;

    *mptr++ = (unsigned char)( num_bytes_per_sector );
    *mptr++ = (unsigned char)( num_bytes_per_sector / 256 );

    *mptr++ = (unsigned char)( num_cylinders );
    *mptr++ = (unsigned char)( num_cylinders / 256 );

    *mptr++ = (unsigned char)( num_phys_parts );
    *mptr++ = (unsigned char)( num_phys_parts / 256 );

    *mptr++ = (unsigned char)( num_reads );
    *mptr++ = (unsigned char)( num_reads / 256 );
    *mptr++ = (unsigned char)( num_reads / 65536 );
    *mptr++ = (unsigned char)( num_reads / 16777216 );

    *mptr++ = (unsigned char)( num_writes );
    *mptr++ = (unsigned char)( num_writes / 256 );
    *mptr++ = (unsigned char)( num_writes / 65536 );
    *mptr++ = (unsigned char)( num_writes / 16777216 );
}

/*
** CPU can request the disk controller to DMA disk blocks to and
** from memory. Because disk is a separate process, CPU doesn't
** have to wait for it to finish - disk will write to the interrupt
** pipe when it is done.
*/

void disk_read_block( void )
{
    int   bytes_read;
    off_t disk_seek;

    unsigned char *mptr;

    /* figure out where to read from and where to put the data */
    mptr = memory + disk_memory_addr;
    disk_seek = 2*sizeof( long ) + 512*disk_block_req;

    /* increment the read counter */
    num_reads++;
    lseek( disk_fd, 0, SEEK_SET );
    write( disk_fd, &num_reads, sizeof( long ) );

    /* read the block */
    lseek( disk_fd, disk_seek, SEEK_SET );
    bytes_read = read( disk_fd, mptr, 512 );
}

void disk_write_block( void )
{
    int   bytes_written;
    off_t disk_seek;

    unsigned char *mptr;

    /* figure out where to write and where the data comes from */
    mptr = memory + disk_memory_addr;
    disk_seek = 2*sizeof( long ) + 512*disk_block_req;

    /* increment the write counter */
    num_writes++;
    lseek( disk_fd, sizeof( long ), SEEK_SET );
    write( disk_fd, &num_writes, sizeof( long ) );

    /* write the block */
    lseek( disk_fd, disk_seek, SEEK_SET );
    bytes_written = write( disk_fd, mptr, 512 );
}

void do_disk( void )
{
    IPC_vars->disk_status = 0;

    switch( disk_command )
    {
    case DCMD_INFO:
	disk_info();
	break;

    case DCMD_READ:
	disk_read_block();
	break;

    case DCMD_WRITE:
	disk_write_block();
	break;
    }

    usleep( 50000 );
    IPC_vars->disk_status = 0x80;
    assert_interrupt( DISK_INDEX, F_IRQ );
}

void disk_main( void )
{
    int message;

    /* to start out: */

    /* disk has no command */
    disk_command = 0;

    while( 1 )
    {
	message = disk_read_message();
	if ( message )
	    disk_handle_message( message );

	if ( do_disk_command )
	    do_disk();

	do_disk_command = 0;
    }
}




