
#include <curses.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/shm.h>
#include <unistd.h>
#include "cmdline.h"
#include "cpu.h"
#include "em65.h"
#include "em65types.h"  /* TRUE, FALSE */
#include "debugger.h"
#include "disk.h"
#include "tables.h"
#include "terminal.h"
#include "timer.h"

char *em65_version_string   = "Em65 0.6";
char *em65_copyright_string = "Copyright 1997";
char *em65_author_string    = "W. Sheldon Simms III";
char *em65_email_string     = "sheldon@atlcom.net";

/* set to quit the program (break the 65c02 execution loop) */
short quitting = FALSE;

short num_devices;

short got_sigint = FALSE;
/* SIGINT has been caught, stop running or tracing and begin stepping */

int shared_memory_id;

IPC_space *IPC_vars;
int *interrupt_ptr;
int *last_interrupt;

unsigned long num_instructions = 0;
unsigned long num_ticks = 0;

pid_t cpu_pid, term_pid, disk_pid;

unsigned char builtin_program[] = {

0xA2, 0xFF,       /* ldx #$FF    */
0x9A,             /* txs         */
0x80, 0x0A,       /* bra +10     */

0x20, 0x53, 0x68, 0x65, 0x6C, 0x64, 0x6F, 0x6E, 0x20, 0x00, /* " Sheldon " */

0xA9, 0x05,       /* lda #$05    */
0x85, 0x00,       /* sta $00     */
0xA9, 0xF0,       /* lda #$F0    */
0x85, 0x01,       /* sta $01     */
0xA0, 0x00,       /* ldy #$00    */
0xB1, 0x00,       /* lda ($00),y */
0xF0, 0x0C,       /* beq +12     */
0x48,             /* pha         */
0x8D, 0x00, 0xEC, /* sta $EC00   */
0x2c, 0x01, 0xEC, /* bit $EC01   */
0x50, 0xfb,       /* bvc -5      */
0xC8,             /* iny         */
0x80, 0xF0,       /* bra -16     */
0x68,             /* pla         */
0x8D, 0x00, 0xEC, /* sta $EC00   */
0x2c, 0x01, 0xEC, /* bit $EC01   */
0x50, 0xfb,       /* bvc -5      */
0x88,             /* dey         */
0xF0, 0xE4,       /* beq -28     */
0x80, 0xF2,       /* bra -14     */
0x00, 0x00, 0x00

};

void catch_sigint()
{
    if ( em65_debug )
	got_sigint = TRUE;
    else
	quitting = TRUE;
}

void load_builtin_program( unsigned short addr, unsigned char *prog )
{
    int numz = 0;

    while( numz < 3 )
    {
	if ( *prog ) numz = 0;
	else numz++;

	memory[ addr++ ] = *prog++;
    }
}

void test_limits( void )
{
    if ( CHAR_BIT != 8 )
    {
	fputs( "Fatal Error - this program will only run on computers on\n"
               "              which type 'char' is 8 bits wide.\n", stderr );
	exit( 1 );
    }

    if ( sizeof( unsigned char ) != sizeof( char ) ||
	 sizeof( signed char )   != sizeof( char ) )
    {
	fputs( "Fatal Error - this program will only run on computers on\n"
               "              which types 'char', 'unsigned char', and\n"
	       "              'signed char' are the same size.", stderr );
	exit( 1 );
    }

    if ( sizeof( short ) != 2 * sizeof( char ) )
    {
	fputs( "Fatal Error - this program will only run on computers on\n"
               "              which type 'short' is 16 bits wide.\n", stderr );
	exit( 1 );
    }

    if ( sizeof( unsigned short ) != sizeof( short ) ||
	 sizeof( signed short )   != sizeof( short ) )
    {
	fputs( "Fatal Error - this program will only run on computers on\n"
               "              which types 'short', 'unsigned short', and\n"
	       "              'signed short' are the same size.", stderr );
	exit( 1 );
    }
}

void device_init( void )
{
    /* put terminal I/O port at IO_BOTTOM */
    install_ior( terminal_read, 0 );
    install_iow( terminal_write, 0 );

    /* put terminal flag at IO_BOTTOM+1 */
    install_ior( terminal_flag, 1 );

    /* put terminal flag clear at IO_BOTTOM+1 */
    install_iow( terminal_clear_iflag, 1 );

    /* put terminal character buffer at IO_BOTTOM+2 */
    install_iow( terminal_save, 2 );

    /* put disk I/O port at IO_BOTTOM+16 */
    install_ior( disk_read, 16 );
    install_iow( disk_write, 16 );

    /* put disk flag at IO_BOTTOM+17 */
    install_ior( disk_flag, 17 );

    /* put timer control at IO_BOTTOM+32 */
    install_iow( timer_write, 32 );

    /* put timer flag at IO_BOTTOM+33 */
    install_ior( timer_flag, 33 );

    num_devices = 3;
}

void memory_init( void )
{
    /* hack - set reset vector to point at reset address */
    memory[ 0xfffc ] = (unsigned char)( em65_reset_address &  0xff );
    memory[ 0xfffd ] = (unsigned char)( em65_reset_address >> 8 );

    if ( em65_rom_filename )
    {
	load_file( em65_rom_filename, em65_rom_address );
    }
    else
    {
	/* hack - enter a built in program */
	load_builtin_program( 0xf000, builtin_program );
    }
}

void set_nonblocking_stdin( void )
{
    int stdin_flags;

    /* save stdin's flags */
    stdin_flags = fcntl( STDIN_FILENO, F_GETFL, 0 );
    if ( stdin_flags == -1 )
    {
	perror( "In nonblocking_stdin (F_GETFL)" );
	exit( 1 );
    }

    /* change to non-blocking */
    stdin_flags |= O_NONBLOCK;
    if ( -1 == fcntl( STDIN_FILENO, F_SETFL, stdin_flags ) )
    {
	perror( "In nonblocking_stdin (F_SETFL)" );
	exit( 1 );
    }
}

void set_blocking_stdin( void )
{
    int stdin_flags;

    /* save stdin's flags */
    stdin_flags = fcntl( STDIN_FILENO, F_GETFL, 0 );
    if ( stdin_flags == -1 )
    {
	perror( "In blocking_stdin (F_GETFL)" );
	exit( 1 );
    }

    /* switch back to blocking */
    stdin_flags &= (~O_NONBLOCK);
    if ( -1 == fcntl( STDIN_FILENO, F_SETFL, stdin_flags ) )
    {
	perror( "In blocking_stdin (F_SETFL)" );
	exit( 1 );
    }
}

/* get a char from stdin, or return -1 if no char is available */
int nb_getchar( void )
{
    char buf;
    int result;

    set_nonblocking_stdin();

    result = read( STDIN_FILENO, &buf, 1 );

    set_blocking_stdin();

    if ( result == -1 )
	return result;
    else
	return (int)buf;
}

pid_t fork_terminal( void )
{
    pid_t pid = fork();

    if ( pid == -1 )
    {
	perror( "fork_terminal (fork)" );
	exit( 1 );
    }
    else if ( pid == 0 )
    {    
	/* background process - ignore SIGINT */
	if ( -1 == (int)signal( SIGINT, SIG_IGN ) )
	{
	    perror( "fork_terminal (signal)" );
	    exit( 1 );
	}

	/* the child process becomes the virtual terminal */
	setproctitle( "terminal" );

	/* map in 65c02 memory */
	memory = shmat( shared_memory_id, 0, 0 );
	if ( -1 == (int)memory || 0 == memory )
	{
	    perror( "fork_terminal [terminal] (shmat)" );
	    exit(1);
	}

	IPC_vars = (IPC_space *)memory;
	memory += sizeof( IPC_space );

	/* be a terminal */
	terminal_main();
    }

    return pid;
}

pid_t fork_disk( void )
{
    pid_t pid = fork();

    if ( pid == -1 )
    {
	perror( "fork_disk (fork)" );
	exit( 1 );
    }
    else if ( pid == 0 )
    {    
	/* background process - ignore SIGINT */
	if ( -1 == (int)signal( SIGINT, SIG_IGN ) )
	{
	    perror( "fork_disk (signal)" );
	    exit( 1 );
	}

	/* the child process becomes the virtual disk */
	setproctitle( "disk" );

	/* map in 65c02 memory */
	memory = shmat( shared_memory_id, 0, 0 );
	if ( -1 == (int)memory || 0 == memory )
	{
	    perror( "fork_disk [disk] (shmat)" );
	    exit(1);
	}

	IPC_vars = (IPC_space *)memory;
	memory += sizeof( IPC_space );

	/* be a disk */
	disk_main();
    }

    return pid;
}

void main_init( void )
{
    /* make sure this computer has appropriate types */
    test_limits();

    /* setup curses */
    initscr();
    cbreak();
    noecho();

    /* intialize other stuff */
    device_init();
    disk_init();
    terminal_init();
    debugger_init();

    /* allocate shared memory to act as 65c02 memory */
    shared_memory_id = shmget( (key_t)cpu_pid,
			       65536 + sizeof( IPC_space ), IPC_CREAT | 0666 );
    if ( shared_memory_id == -1 )
    {
	perror( "main_init (shmget)" );
	exit( 1 );
    }
}

void main_cleanup( void )
{
    /* warn disk to close it's file */
    disk_shutdown();

    /* kill our children */
    kill( term_pid, SIGTERM );
    kill( disk_pid, SIGTERM );

    /* release the shared memory */
    if ( -1 == shmctl( shared_memory_id, IPC_RMID, NULL ) )
    {
	perror( "main_cleanup (shmctl)" );
        exit( 1 );
    }

    /* end curses */
    endwin();
}

void print_statistics( void )
{
    if ( num_ticks && num_instructions > 10000 )
    {
	printf( "\n\nThe total number of instructions executed at full speed was %ld\n",
		num_instructions );
	printf( "The time required was %ld ticks at a rate of %d ticks per second.\n",
		num_ticks, em65_clock_rate );
	printf( "The average number of instructions per second was therefore %.2f.\n",
		(double)num_instructions / ( (double)num_ticks / (double)em65_clock_rate ) );
    }
    else
    {
	puts( "\n\nInsufficient data for execution statistics to be printed." );
    }
}

int main( int argc, char **argv )
{
    /* get command line and environment options */
    process_options( argc, argv );

#if 0
    /* Debug - print results of option processing */
    printf( "\ndebug is %s\n", em65_debug ? "on" : "off" );
    printf( "clock is %s\n", em65_realtime_clock ? "real time" : "process virtual time" );
    printf( "clock rate is %d Hz\n", em65_clock_rate );
    printf( "reset vector is $%4.4X\n", em65_reset_address );
    printf( "disk file is %s\n", em65_disk_filename );
    printf( "rom file is %s loaded at $%4.4X\n", em65_rom_filename, em65_rom_address );
    exit( 0 );
#endif

    /* know myself */
    cpu_pid = getpid();

    /* initialize everything */
    main_init();

    /* fork a virtual terminal process */
    term_pid = fork_terminal();

    /* fork a virtual disk process */
    disk_pid = fork_disk();
    
    /* catch SIGINT */
    if ( -1 == (int)signal( SIGINT, catch_sigint ) )
    {
	perror( "main (signal)" );
	exit( 1 );
    }

    /* parent process becomes "CPU" */
    setproctitle( "cpu" );

    /* map in 65c02 memory */
    memory = shmat( shared_memory_id, 0, 0 );
    if ( -1 == (int)memory || 0 == memory )
    {
	perror( "main (shmat)" );
	exit(1);
    }

    IPC_vars = (IPC_space *)memory;
    memory += sizeof( IPC_space );
    interrupt_ptr = &(IPC_vars->device_interrupt[0]);
    last_interrupt = interrupt_ptr + NUM_DEVICES;
    *interrupt_ptr = F_RESET;
    IPC_vars->terminal_status = 0x40;
    
    /* hack - intialize some memory */
    memory_init();

    /* start interval timer (65c02 interrupt source) */
    timer_init();

    /* if debugger is off tell terminal to use screen */
    if ( !em65_debug )
    {
	running = TRUE;
	terminal_tell( TERMINAL_WRITE );
    }

    /* execute 65c02 code */
    execute_loop();

    /* cleanup after ourself */
    main_cleanup();

    /* print emulation statistics */
    print_statistics();

    /* and quit */
    exit( 0 );
}

