
#include <curses.h>    /* WINDOW                  */
#include <unistd.h>    /* pipe(), read(), write() */
#include "cpu.h"       /* PIPE_READ, PIPE_WRITE   */
#include "em65.h"      /* F_IRQ                   */
#include "terminal.h"

int to_terminal[2];
int from_terminal[2];

WINDOW *wterminal;

void setup_terminal_pipes( void )
{
    /* open the to_terminal pipe */
    if ( pipe( to_terminal ) )
    {
	perror( "In setup_terminal_pipes (pipe)<1>" );
	exit( 1 );
    }

    /* open the from_terminal pipe */
    if ( pipe( from_terminal ) )
    {
	perror( "In setup_terminal_pipes (pipe)<2>" );
	exit( 1 );
    }
}

void terminal_init( void )
{
    setup_terminal_pipes();

    wterminal = newwin( 24, 80, 0, 0 );
    scrollok( wterminal, 1 );
}

int terminal_talk( int message )
{
    int result, response;

    result = write( to_terminal[ PIPE_WRITE ], &message, sizeof( int ) );
    if ( result == -1 )
    {
	perror( "In terminal_talk (write)" );
	exit( 1 );
    }

    result = read( from_terminal[ PIPE_READ ], &response, sizeof( int ) );
    if ( result == -1 )
    {
	perror( "In terminal_talk (read)" );
	exit( 1 );
    }

    return response;
}

void terminal_tell( int message )
{
    int result = write( to_terminal[ PIPE_WRITE ], &message, sizeof( int ) );
    if ( result == -1 )
    {
	perror( "In terminal_tell (write)" );
	exit( 1 );
    }
}

unsigned char terminal_flag( void )
{
    return (unsigned char)(IPC_vars->terminal_status);
}

unsigned char terminal_read( void )
{
    return (unsigned char)terminal_talk( TERMINAL_GET_CHAR );
}

void terminal_write( unsigned char byte )
{
    (void)terminal_talk( TERMINAL_PUT_CHAR | (unsigned int)byte );
}

void terminal_save( unsigned char byte )
{
    terminal_tell( TERMINAL_SAVE_CHAR | (unsigned int)byte );
}

void terminal_clear_iflag( unsigned char byte )
{
    terminal_tell( TERMINAL_CLEAR_IFLAG );
}

/*
** +----------------------------------------------------------+
**
** Everything above this is executed in the "CPU" process.
** Everything below this is executed in the "terminal" process.
**
** +----------------------------------------------------------+
*/

#define TERMINAL_IN_IRQ   0x80
#define TERMINAL_OUT_IRQ  0x40

int terminal_char;
int terminal_write_ok;

#define SAVE_CHAR_BUF_SIZE 80
unsigned char save_char_buf[ SAVE_CHAR_BUF_SIZE ];
int save_char_buf_index = 0;

#define MESSAGE_BLOCK_SIZE 8
int message_array[ MESSAGE_BLOCK_SIZE ];
int num_messages = 0;
int next_message = 0;

/* See if "CPU" has sent a message */
int terminal_read_message( void )
{
    int result;

    if ( next_message == num_messages )
    {
	result = read( to_terminal[ PIPE_READ ], &(message_array[0]),
		       MESSAGE_BLOCK_SIZE * sizeof( int ) );
	if ( result == -1 )
	{
	    perror( "In terminal_read_message (read)" );
	    exit( 1 );
	}

	num_messages = result / sizeof( int );
	next_message = 0;
    }

    return message_array[ next_message++ ];
}

void terminal_respond( int response )
{
    int result;

    result = write( from_terminal[ PIPE_WRITE ], &response, sizeof( int ) );
    if ( result == -1 )
    {
	perror( "In terminal_respond (write)" );
	exit( 1 );
    }
}

void terminal_print( int printchar )
{
    waddch( wterminal, printchar );
    if ( terminal_write_ok )
    {
	wrefresh( wterminal );
    }
}

void terminal_handle_message( int message )
{
    switch( message & 0xff00 )
    {
    case TERMINAL_GET_CHAR:
	IPC_vars->terminal_status &= ~TERMINAL_IN_IRQ;
	terminal_respond( terminal_char );
	break;

    case TERMINAL_SAVE_CHAR:
    {
	int i;
	
	if ( save_char_buf_index == SAVE_CHAR_BUF_SIZE )
	{
	    save_char_buf_index--;
	    for( i = 0; i < (SAVE_CHAR_BUF_SIZE-1); i++ )
		save_char_buf[i] = save_char_buf[i+1];
	}
	save_char_buf[ save_char_buf_index++ ] =
	    (unsigned char)(message & 0xff);
	break;
    }

    case TERMINAL_PUT_CHAR:
    {
	int i;

	IPC_vars->terminal_status &= ~TERMINAL_OUT_IRQ;
	terminal_respond( 1 );
	if ( save_char_buf_index > 0 )
	{
	    for( i = 0; i < save_char_buf_index; i++ )
	    {
		terminal_print( save_char_buf[i] );
	    }
	    save_char_buf_index = 0;
	}
	else
	{
	    terminal_print( message & 0xff );
	}
	usleep( 33333 ); 
	IPC_vars->terminal_status |= TERMINAL_OUT_IRQ;
	assert_interrupt( TERMINAL_OUT_INDEX, F_IRQ );
 	break;
    }

    case TERMINAL_CLEAR_IFLAG:
	IPC_vars->terminal_status = 0;
	break;

    case TERMINAL_NOWRITE:
	terminal_write_ok = 0;
	terminal_respond( 1 );
	break;

    case TERMINAL_WRITE:
	if ( 0 == terminal_write_ok )
	{
	    terminal_write_ok = 1;
	    touchwin( wterminal );
	    wrefresh( wterminal );
	}
	break;

    case TERMINAL_KEYPRESS:
	terminal_char = message & 0xff;
	IPC_vars->terminal_status |= TERMINAL_IN_IRQ;
	assert_interrupt( TERMINAL_IN_INDEX, F_IRQ );
	break;

    default:
	break;
    }
}

void terminal_main( void )
{
    int message;

    /* to start out: */

    /* terminal cannot update screen */
    terminal_write_ok = 0;

    /* no key has been pressed */
    terminal_char = 0;

    while( 1 )
    {
	message = terminal_read_message();
	terminal_handle_message( message );
    }
}



