/***********************************************************************
 *
 *	umdss.c		UNIX Mail Delivery Subsystem (main)
 *			Part of the UMDSS Project
 *			(C)opyright 1993 Morgan Davis Group
 *
 **********************************************************************/

#include <sys/types.h>
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <sys/ioctl.h>
#include <sys/stat.h>

#include "umdss.h"	/* SCOPE undefined so global data is created */

	/* Local functions */

void cleanup();			/* Restore tty settings upon exit */
int getid();			/* Get the remote site's id and password */
int locksite();			/* Tries to create a site's LCK file */

	/* External functions */
	
extern int slave();		/* Go into Slave mode */
extern void master();		/* Go into Master mode */
extern char get_timed_char();	/* Wait for a command character */
extern void trap_hangup();	/* Trap for hangup signal */
extern void logentry();		/* Make an entry to the log file */


int
main(argc, argv)
	int argc;
	char *argv[];
{
	TERMSTRUCT	mystty;
	FILE		*fp;
	char		c, *cp;
	int		errflag = 0;
	extern int	optind;
	extern char	*optarg;

	while ((c = getopt(argc, argv, "qv")) != EOF) {
		switch (c) {
		case 'q':
			exit(tally());
		case 'v':
			printf("%s: version %s, %s\n", *argv, VERSION, COPR);
		default:
			errflag = 1;
		}
	}

	if (errflag || argc > 2) {
		fprintf(stderr, "Usage:\t%s [site]\n\t%s [-v] [-q]\n", *argv, *argv);
		return(ERROR);
	}

#ifndef SYSTEMID
	if ((fp = fopen("/etc/systemid", "r")) == NULL) {
		fprintf(stderr, "%s: can't get hostname\n", *argv);
		return(ERROR);
	}
	fgets(hostname, SITELEN_MAX, fp);
	fclose(fp);
	if ((cp = strchr(hostname, '\n')))
		*cp = '\0';
#endif /* SYSTEMID */
	
	if ((logref = fopen(logfile, "a")) == NULL) {
		fprintf(stderr, "%s: can't write to log\n", *argv);
	}

	oldcmask = umask(0177);

	ioctl(fileno(stdin), TERMGET, &mystty);
	default_stty = mystty;

#if SGTTY
	mystty.sg_flags &= ~ECHO;                 /* No echo */
	mystty.sg_flags |= HUPCL;                 /* Hangup friendly */
	mystty.sg_flags |= RAW;                   /* RAW mode */
#else
	mystty.c_oflag = mystty.c_iflag = mystty.c_lflag = 0;	/* disabled */
	mystty.c_cflag |= (CS8|CREAD|HUPCL);
	mystty.c_cc[VMIN] = 1;
	mystty.c_cc[VTIME] = 0;
#endif

	ioctl(fileno(stdin), TERMSET, &mystty);

	signal(SIGINT, SIG_IGN);      /* Ignore interrupts */
	signal(SIGQUIT, SIG_IGN);     /* Ignore Quit signal */
	signal(SIGHUP, trap_hangup);  /* If they hang up, record it! */

	if (argc == 1 || **argv == '-') {    /* We're being polled! */
		if (getid())
			cleanup(ERROR);
		if (slave() == SEND_CMD)
			master(HANGUP_CMD);
	} else {
		/* We've called somebody else */
		strncpy(sitename, argv[1], SITELEN_MAX);
		if (locksite() == NO_ERROR) {
			do {
				master(SEND_CMD);
			} while(slave() == SEND_CMD);
		}
	}
	cleanup(NO_ERROR);
	return(NO_ERROR);
}

/*
 *	cleanup()	Clean up tty modes, close log file, etc.
 */

void cleanup(exitflag)
	int exitflag;
{
	signal(SIGHUP, SIG_IGN);		/* Ignore this now */

	if (*lockfile) {			/* delete lock file */
		unlink(lockfile);
		lockfile[0] = '\0';
	}

	if (!exitflag || (jobsSent | jobsRcvd)) {
		sprintf(buf, "S#%d,%lu R#%d,%lu %s %d",
			jobsSent, bytesSent,
			jobsRcvd, bytesRcvd,
			basetty(0), speeds[TERMSPD]);
		logentry("H", buf);
	}
	fclose(logref);	/* close log file */
	logstatus((exitflag) ? "FAILED!" : "success");
	logstatus(NULL);	/* close status file */

#ifdef BOSCO
	fflush(stdout);
	while(get_timed_char(2) != TIMEOUT);	/* pause before hangup */
	TERMHUP;
	ioctl(fileno(stdin), TERMSET, &default_stty);
	while(get_timed_char(2) != TIMEOUT);	/* pause after hangup */
#endif /* BOSCO */
#ifndef BOSCO
	ioctl(fileno(stdin), TERMSET, &default_stty);
	while(get_timed_char(4) != TIMEOUT);	/* pause after hangup */
#endif

	umask(oldcmask);

	if (exitflag)
		exit(exitflag);
}


/*
 * getid()	Prompt calling site for id;password
 */

int getid()
{
	FILE	*idref;
	char	idfile_sitename[SITELEN_MAX],
		idfile_passwd[PASSWD_MAX],
		passwd[PASSWD_MAX],
		*bufptr, *p;

	while(get_timed_char(2) != TIMEOUT);	/* pause and flush input */

	fputs("id: ", stdout);			/* Display prompt */
	if (scanf(IDINPUT_TMPL, sitename, passwd) != 2) {
		logentry("*", "<ALERT: LOGIN FAILED>");
		return(ERROR);
	}

	if ((idref = fopen(idfile, "r")) == NULL) {
		logentry("*","<ERROR: CAN'T READ IDFILE>");
		cleanup(ERROR);
	}

	while (fgets(buf, BUFLEN, idref) != NULL) {
		/* strip leading space */
		for (bufptr = buf; *bufptr && (*bufptr <= ' '); bufptr++);
		/* find pound sign */
		if (p = strchr(bufptr, '#')) {
			/* walk backwards to eat white space */
			while (--p >= bufptr && *p <= ' ');
			*++p = '\0';
		}
		if ((sscanf(bufptr, IDENTRY_TMPL, idfile_sitename,
			idfile_passwd, sndcmd, rcvcmd) != EOF) &&
		    !strcmp(sitename, idfile_sitename)) {
			/* Got a match */
			if (strcmp(passwd, idfile_passwd)) {
				logentry("*", "<ALERT: PASSWORD FAILED>");
				fclose(idref);
				return(ERROR);
			}

			if (locksite() != NO_ERROR)
				return(ERROR);

			/* Log site entry */
			sprintf(buf, "%s %d", basetty(0), speeds[TERMSPD]);
			logentry("C", buf);
			logstatus("");
			fclose(idref);
			return(NO_ERROR);
		}
	}

	/* Can't find sitename in idref */

	logentry("*", "<ALERT: UNKNOWN SITE>");
	fclose(idref);
	return(ERROR);
}


/*
 * locksite()	returns 0 if new LCK file created OK
 *		returns -1 if LCK exists and is active (or error)
 */

int locksite()
{
	FILE *fp;
	int pid;

	if (chklock(sitename) != 1) {
		if (fp = fopen(lockfile, "w")) {
			fprintf(fp, PID_TMPL, getpid());
			fclose(fp);
			chmod(lockfile, 0664);
			return (NO_ERROR);
		}
	}
	logentry("*", "<ERROR: LOCK FAILED>");
	lockfile[0] = '\0';
	return(ERROR);
}


/*
 * chklock()	returns: 0=not active 1=active -1=bad lock
 */

int
chklock(site)
	char *site;
{
	FILE *fp;
	pid_t pid;

	sprintf(lockfile, LCK_TMPL, lockdir, site);

	if (stat(lockfile, &stbuf) == NO_ERROR) {	/* It might exist...*/
		if ((fp = fopen(lockfile, "r")) == NULL)
			return(ERROR);			/* couldn't read it */
		fscanf(fp, PID_TMPL, &pid);
		fclose(fp);
		if ((kill(pid, 0) == NO_ERROR) || errno == EPERM)
			return(1);		/* active */
		return(ERROR);		/* bogus pid */
	}
	return(0);		/* inactive */
}
