/*---------------------------
 * Apple 13-Sector Disk Transfer 1.22
 *
 * Original Program:
 * By Paul Guertin
 * pg@sff.net
 * October 13th, 1994 - 1999
 * Distribute freely
 *
 * 13-Sector Support:
 * By Linards Ticmanis
 * ticmanis@gmx.net
 * November 17th, 2005
 * Distribute freely
 *--------------------------*/

/*
 * Compile with Turbo C 2.0:
 * TCC ADT13.C COMM.C
 *
 * Compile with gcc (or others) for Unix:
 * cc -DUNIX adt13.c ucomm.c -o adt13 -l[n]curses
 */

#if !defined(UNIX) && !defined(DOS)
#  define DOS
#endif

#include <stdio.h>
#ifdef DOS
#  include <conio.h>
#  include <dir.h>
#  include <io.h>
#endif
#ifdef UNIX
#  include <curses.h>
#  include <stdlib.h>
#  include <string.h>
#  include <sys/stat.h>
#  include <sys/types.h>
#  include <sys/time.h>
#  include <unistd.h>
#  include <signal.h>
#  include <errno.h>
#  include <termios.h>
#  include <dirent.h>
#endif
#include <ctype.h>
#include <setjmp.h>
#include <time.h>
#include "comm.h"

#define ACK 0x06
#define NAK 0x15

int main( int argc, char **argv );
void send_directory( void );
void send_disk( void );
void send_sector( unsigned char *buffer, int part, int track, int sector );
void receive_disk( void );
void receive_sector( unsigned char *buffer, int part, int track, int sector );
void getfname( char *fname );
void comm_puts( char *p );
int set_port( void );
void clear_display( void );
void putsat( int x, int y, char *s );
void give_help( void );
void make_crctable( void );
unsigned short do_crc( unsigned char *ptr, int count );
int kbhit( void );

#ifdef UNIX
    void sighand( int num );
    off_t filelength( int fd);
#   define gotoxy(x,y) move(y,x)
#   define clreol() clrtoeol()
#   define putch(c) addch(c)
#   define cputs(s) addstr(s)
#   define cprintf printw
#   define clrscr() clear()
    void makeraw(void);
    void makeunraw(void);
    void cleanup_for_exit(void);
    static int flag = 0;
    static struct termios tty,otty;
    extern int comm_fd;
#   define INIT_DISP "xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx"
#   define DONE_DISP 'o'
#endif /* #ifdef UNIX */

#ifdef DOS
    void refresh(void) {}
    void makeunraw(void) {}
    void cleanup_for_exit(void) {}
#   define INIT_DISP ""
#   define DONE_DISP ''
#endif /* #ifdef DOS */


int comport, comspeed;          /* serial port details */
jmp_buf beginning;              /* where to abort when ESC pushed */
unsigned short crctable[256];   /* CRC-16 lookup table for speed */


int main( int argc, char **argv )
{
    int command;                            /* character from Apple */
#ifdef UNIX
    int retval;
    fd_set fds;
#endif

    comport = comspeed = 0;
    argc = argc;                            /* prevent compiler warning */

    while( *++argv ) {
        if( !isdigit( **argv ) ) {          /* argument is not a digit, */
            give_help();                    /* give help and quit */
            return 0;
        }
        if( !(*argv)[1] )                   /* kluge: a number is a port */
            comport = **argv - '0';         /* iff it has only one digit */
        else
            comspeed = atoi( *argv );
    }

    make_crctable();

#ifdef UNIX
    initscr();
    signal(SIGINT,sighand);
    signal(SIGTERM,sighand);
    signal(SIGHUP,sighand);
#endif

refresh:
    makeunraw();
    setjmp( beginning );                    /* return here when ESC pushed */

    clrscr();
#ifdef DOS
    textattr( LIGHTGRAY + (BLACK<<4) );
#endif
    putsat( 23, 2, "Apple 13-Sector Disk Transfer 1.22" );
    refresh();

    if( set_port() ) {
        cleanup_for_exit();
        return -1;
    }

    clear_display();
    putsat( 1, 24, "C to change port/speed, ^L to refresh screen, "
                   "Q to quit program." );

    refresh();
    for( ;; ) {
        putsat( 1, 9, "READY." );
        refresh();
#ifdef DOS
        while( !comm_avail() ) {
            if( kbhit() ) {
#endif
#ifdef UNIX
        {
          again:
            makeraw();
            FD_ZERO(&fds);
            FD_SET(0,&fds);
            FD_SET(comm_fd,&fds);
            retval = select(comm_fd + 1,&fds,NULL,NULL,NULL);
            if (retval == -1) {
                if (errno == EINTR) goto again;
                if (flag) {
                    tcsetattr(0,TCSANOW,&otty);
                }
                endwin();
                fprintf(stderr,"select: %s\n",strerror(errno));
                exit(1);
            }
            if (FD_ISSET(0,&fds)) {
#endif
                switch( getch() ) {
                    case 'C': case 'c':
                        comm_close();
                        comport = comspeed = 0;
                        if( set_port() ) {
                            cleanup_for_exit();
                            return -1;
                        }
                        putsat( 1, 9, "READY." );
                        refresh();
                        break;
                    case 'Q': case 'q':
                        putsat( 1, 9, "" );
                        putsat( 1, 24, "" );
                        putsat( 1, 20, "Goodbye!\n" );
                        cleanup_for_exit();
                        return 0;
                    case 0x0C:              /* ctrl-L = refresh screen */
                        goto refresh;
                }
            }
        }
#ifdef UNIX
        if (! FD_ISSET(comm_fd,&fds)) continue;
#endif
        putsat( 1, 20, "" );
        command = comm_getc() & 0x7F;
        refresh();
        switch( command ) {
            case 'D': send_directory();  break;
            case 'R': send_disk(); break;
            case 'S': receive_disk(); break;
            default:
                putsat( 1, 20, "ERROR: unknown command received: " );
                cprintf( "%02X", command );
                refresh();
        }
    }
}

#ifdef DOS

void send_directory()
/* Format and send the contents of current directory */
{
    int i;
    int tab, line;
    char *dirname = getcwd( NULL, 256 );
    struct ffblk ff;

    putsat( 1, 9, "Sending directory listing   (ESC to abort)" );
    comm_puts( "DIRECTORY OF " );
    comm_puts( dirname );
    if( (strlen( dirname ) + 13) % 40 )
        comm_putc(  '\r' );
    for( i=0; i<40; i++ )
        comm_putc( '-' );

    if( findfirst( "*.*", &ff, 0 ) == 0 ) {
        tab = 0;
        line = (strlen( dirname ) + 13) / 40 + 2;
        for( ;; ) {
            if( kbhit() && getch() == 0x1B )
                break;
            comm_puts( ff.ff_name );
            for( i=strlen( ff.ff_name ); i<12; i++ )
                comm_putc( ' ' );
            if( tab < 2 ) {
                comm_putc( ' ' );
                comm_putc( ' ' );
            }
            if( comm_avail() && comm_getc() == '\0' )
                break;
            if( findnext( &ff ) == -1 )
                break;
            if( ++tab > 2 ) {
                tab = 0;
                if( ++line > 20 ) {
                    line = 0;
                    comm_putc( '\0' );
                    comm_putc( '\1' );
                    if( comm_getc() == '\0' )
                        break;
                }
            }
        }
    }
    else
        comm_puts( "NO FILES" );

    comm_putc( '\0' );
    comm_putc( '\0' );
    comm_flush();
    free( dirname );
}

#endif /* #ifdef DOS */
#ifdef UNIX

void send_directory()
/* Format and send the current of current directory */
{
    DIR *curdir = NULL;
    char *dirname = getcwd( NULL, 256 );
    char buf[128];
    struct dirent *entry;

    int i;
    int tab, line;

    putsat( 1, 9, "Sending directory listing   (ESC to abort)" );
    comm_puts( "DIRECTORY OF " );
    comm_puts( dirname );
    if( (strlen( dirname ) + 13) % 40 )
        comm_putc(  '\r' );
    for( i=0; i<40; i++ )
        comm_putc( '-' );

    curdir = opendir(".");
    if (curdir) {

        tab = 0;
        line = (strlen( dirname ) + 13) / 40 + 2;
        entry = readdir(curdir);
        if (! entry) goto end;
        for( ;; ) {
            if( kbhit() && getch() == 0x1B )
                break;
            comm_puts( entry->d_name );
            for( i=strlen( entry->d_name ); i<12; i++ )
                comm_putc( ' ' );
            if( tab < 2 ) {
                comm_putc( ' ' );
                comm_putc( ' ' );
            }
            if( comm_avail() && comm_getc() == '\0' )
                break;
            if( !(entry = readdir(curdir)) )
                break;
            if( ++tab > 2 ) {
                tab = 0;
                if( ++line > 20 ) {
                    line = 0;
                    comm_putc( '\0' );
                    comm_putc( '\1' );
                    if( comm_getc() == '\0' )
                        break;
                }
            }
        }
    }
    else {
        sprintf(buf,"cannot open: %s\n",strerror(errno));
        comm_puts( buf );
    }

  end:
    if (curdir) closedir(curdir);
    comm_putc( '\0' );
    comm_putc( '\0' );
    comm_flush();
    free( dirname );
}

#endif /* #ifdef UNIX */

void send_disk()
/* Main send routine */
{
    char fname[13], error[80];
    unsigned char buffer[23296];
    FILE *f;
    int myerrno;
    int part, track, sector, report;
    time_t tstart, tend;
    int minutes, seconds, cps;

    getfname( fname );
    putsat( 1, 9, "Sending file " );
    putsat( 14, 9, fname );
    cputs( "   (ESC to abort)" );
    refresh();

    f = fopen( fname, "rb" );
    if( !f ) {
#ifdef UNIX
        myerrno = errno;
#endif
        comm_putc( 26 );        /* can't open */
#ifdef UNIX
        sprintf( error, "ERROR: File %s can't be opened for input. %s", fname, strerror(myerrno) );
#else
        sprintf( error, "ERROR: File %s can't be opened for input.", fname );
#endif
        putsat( 1, 20, error );
        refresh();
        return;
    }
    if( filelength( fileno( f ) ) != 116480L ) {
        comm_putc( 30 );        /* not a 114k image */
        sprintf( error, "ERROR: File %s is not a 114K disk image.", fname );
        putsat( 1, 20, error );
        refresh();
        return;
    }
    comm_putc( 0 );     /* File is now ready */

    time( &tstart );
    clear_display();
    refresh();
    while( comm_getc() != ACK ) {
        putsat( 1, 20, "ERROR: Bad header byte received." );
        refresh();
    }
    for( part=0; part<5; part++ ) {
        putsat( 60, 12+part, "Reading" );
        refresh();
        fread( buffer, 1, 23296, f );
        putsat( 60, 12+part, "" );
        gotoxy( 12, 12+part );
        refresh();
        for( track=0; track<7; track++ ) {
            for( sector=12; sector>=0; sector-- ) {
                send_sector( buffer, part, track, sector );
                if( sector%2 == 1 ) {
                    putch( DONE_DISP );
                    refresh();
                }
            }
        }
        putsat( 60, 12+part, "OK" );
        refresh();
    }
    report = comm_getc();
    fclose( f );
    time( &tend );

    seconds = (int)difftime( tend, tstart );
    cps = (int)(116480L / seconds);
    minutes = seconds/60;
    seconds = seconds%60;
    if( report )
        sprintf( error, "File %s sent WITH ERRORS in %d:%02d (%d cps).",
            fname, minutes, seconds, cps );
    else
        sprintf( error, "File %s sent successfully in %d:%02d (%d cps).",
            fname, minutes, seconds, cps );
    putsat( 1, 20, error );
    refresh();
}


void send_sector( unsigned char *buffer, int part, int track, int sector )
/* Send a sector with RLE compression */
{
    int byte, ok;
    short crc;
    unsigned char data, prev, newprev;
    unsigned char *thissector = buffer+(track*3328)+(sector<<8);
#ifdef DOS
    struct text_info textinfo;
#endif
#ifdef UNIX
    int cur_x,cur_y;
#endif

    do {
        prev = 0;
        for( byte=0; byte<256; ) {
            newprev = thissector[byte];
            data = newprev - prev;
            prev = newprev;
            comm_putc( data );
            if( data )
                byte++;
            else {
                while( byte<256 && thissector[byte] == newprev )
                    byte++;
                comm_putc( byte & 0xFF );       /* 256 becomes 0 */
            }
        }
        crc = do_crc( thissector, 256 );
        comm_putc( crc & 0xFF );
        comm_putc( crc >> 8 );
        ok = comm_getc();
        if( ok != ACK ) {
#ifdef DOS
            gettextinfo( &textinfo );           /* save cursor position */
#endif
#ifdef UNIX
            getyx(stdscr,cur_y,cur_x);
#endif
            gotoxy( 1, 20 );
            clreol();
            cprintf( "Track $%02X, sector $%02X: bad CRC",
                7*part+track, sector );
#ifdef DOS
            gotoxy( textinfo.curx, textinfo.cury );
#endif
#ifdef UNIX
            move(cur_y,cur_x);
            refresh();
#endif
        }
    } while( ok != ACK );
}


void receive_disk()
/* Main receive routine */
{
    char fname[13], error[80];
    unsigned char buffer[23296];
    FILE *f;
    int part, track, sector, report;
    time_t tstart, tend;
    int minutes, seconds, cps;

    getfname( fname );
    putsat( 1, 9, "Receiving file " );
    putsat( 16, 9, fname );
    cputs( "   (ESC to abort)" );
    refresh();

    if( !access( fname, 0 ) ) {
        comm_putc( 28 );        /* File exists */
        sprintf( error, "ERROR: File %s already exists.", fname );
        putsat( 1, 20, error );
        refresh();
        return;
    }
    if( (f = fopen( fname, "wb" )) == NULL ) {
        comm_putc( 26 );        /* Can't open */
        sprintf( error, "ERROR: File %s can't be opened for output.", fname );
        putsat( 1, 20, error );
        refresh();
        return;
    }
    for( track=0; track<35; track++ ) {
        if( fwrite( buffer, 1, 3328, f ) != 3328 ) {
            comm_putc( 30 );    /* Disk full */
            sprintf( error, "ERROR: File %s doesn't fit on disk.", fname );
            putsat( 1, 20, error );
            refresh();
            fclose( f );
            return;
        }
    }
    rewind( f );
    comm_putc( 0 );         /* File is now ready */

    time( &tstart );
    clear_display();
    while( comm_getc() != ACK ) {
        putsat( 1, 20, "ERROR: Bad header byte received." );
        refresh();
    }
    for( part=0; part<5; part++ ) {
        gotoxy( 12, 12+part );
        for( track=0; track<7; track++ ) {
            for( sector=12; sector>=0; sector-- ) {
                receive_sector( buffer, part, track, sector );
                if( sector%2 == 1 ) {
                    putch( DONE_DISP );
                    refresh();
                }
            }
        }
        fwrite( buffer, 1, 23296, f );
        putsat( 60, 12+part, "OK" );
        refresh();
    }
    report = comm_getc();
    fclose( f );
    time( &tend );

    seconds = (int)difftime( tend, tstart );
    cps = (int)(116480L / seconds);
    minutes = seconds/60;
    seconds = seconds%60;
    if( report )
        sprintf( error, "File %s received WITH ERRORS in %d:%02d (%d cps).",
            fname, minutes, seconds, cps );
    else
        sprintf( error, "File %s received successfully in %d:%02d (%d cps).",
            fname, minutes, seconds, cps );
    putsat( 1, 20, error );
    refresh();
}


void receive_sector( unsigned char *buffer, int part, int track, int sector )
/* Receive a sector with RLE compression */
{
    int byte;
    unsigned short received_crc, computed_crc;
    unsigned char data, prev, crc1, crc2;
    unsigned char *thissector = buffer+(track*3328)+(sector<<8);
#ifdef DOS
    struct text_info textinfo;
#endif
#ifdef UNIX
    int cur_x,cur_y;
#endif

    do {
        prev = 0;
        for( byte=0; byte<256; ) {
            data = comm_getc();
            if( data ) {
                prev += data;
                thissector[byte++] = prev;
            }
            else {
                data = comm_getc();
                do {
                    thissector[byte++] = prev;
                } while( byte<256 && byte!=data );
            }
        }
        crc1 = comm_getc();
        crc2 = comm_getc();

        received_crc = crc1 + 256*crc2;
        computed_crc = do_crc( thissector, 256 );
        if( received_crc != computed_crc ) {
#ifdef DOS
            gettextinfo( &textinfo );           /* save cursor position */
#endif
#ifdef UNIX
            getyx(stdscr,cur_y,cur_x);
#endif
            gotoxy( 1, 20 );
            clreol();
            cprintf( "Track %02X, sector %02X: bad CRC"
                " (expected %04X, got %04X)",
                7*part+track, sector, computed_crc, received_crc );
#ifdef DOS
            gotoxy( textinfo.curx, textinfo.cury );
#endif
#ifdef UNIX
            move(cur_y,cur_x);
            refresh();
#endif
            comm_putc( NAK );
        }
    } while( received_crc != computed_crc );
    comm_putc( ACK );
}


void getfname( char *fname )
/* Receive a null-terminated MS-DOS filename from the Apple */
{
    int i;

    for( i=0; i<=13; i++ ) {
        fname[i] = comm_getc() & 0x7F;
        if( !fname[i] )
            break;
    }
}


void comm_puts( char *p )
/* Like puts() but through the comm port */
{
    while( *p )
        comm_putc( *p++ );
}


int set_port()
/* Show comport and comspeed values, asking for them if they're zero */
/* Return -1 on failure, 0 on success */
{
#ifdef UNIX
    echo();
    nl();
    makeunraw();
#endif
retry:
    putsat( 1, 5, "Serial port: " );
    refresh();
    if( comport ) {
        putch( comport + '0' );
        refresh();
    }
    else {
#ifdef DOS
        scanf( "%d", &comport );
#endif
#ifdef UNIX
        scanw( "%d", &comport );
#endif
        if( !comport )              /* return & quit if port = 0 */
            return -1;
    }

    putsat( 1, 6, "Speed: " );
    refresh();
    if( comspeed ) {
        cprintf( "%d", comspeed );
        refresh();
    }
    else
#ifdef DOS
        scanf( "%d", &comspeed );
#endif
#ifdef UNIX
        scanw( "%d", &comspeed );
#endif
    refresh();

    if( !comm_open( comport, comspeed ) ) {
        cprintf( "\n\nUnable to open port %d at %d bps.\n\n",
                 comport, comspeed );
        refresh();
        comport = comspeed = 0;
        goto retry;                 /* so sue me */
    }
#ifdef UNIX
    refresh();
    noecho();
    nonl();
#endif
    return 0;
}


void clear_display()
/* Clear the disk transfer status */
{
    int part;

    for( part=1; part<=5; part++ ) {
        putsat( 4, 11+part, "Part " );
        putch( '0'+part );
        cputs( ": " INIT_DISP );
    }
}


void putsat( int x, int y, char *s )
/* Show a string at x,y, also erasing to end of line */
{
    gotoxy( x, y );
    clreol();
    cputs( s );
}


void give_help()
/* Text printed when called with illegal arguments */
{
    puts( "\n"
          "Apple 13-Sector Disk Transfer 1.22 -- November 17th, 2005\n"
          "Original program by Paul Guertin (pg@sff.net)\n"
	  "13-Sector support added by Linards Ticmanis (ticmanis@gmx.de)\n"
          "Distribute freely\n"
          "\n"
          "This program transfers a 13-sector Apple II disk\n"
          "to a 114k MS-DOS file and back.  The file format\n"
          "is not yet compatible with any known emulator.\n"
	  "I suggest using the file name extension .d13\n"
          "\n"
          "Syntax: ADT13 [comport] [comspeed]\n"
          "See readme.txt for details.\n" );
}


void make_crctable()
/* Fill the crctable[] array needed by do_crc */
{
    int byte, bit;
    unsigned short crc;

    for( byte=0; byte<256; byte++ ) {
        crc = byte<<8;
        for( bit=0; bit<8; bit++ )
            crc = (crc & 0x8000 ? (crc<<1)^0x1021 : crc<<1);
        crctable[byte] = crc;
    }
}


unsigned short do_crc( unsigned char *ptr, int count )
/* Return the CRC of ptr[0]..ptr[count-1] */
{
    unsigned short crc = 0;

    while( count-- )
        crc = (crc<<8) ^ crctable[(crc>>8) ^ *ptr++];
    return crc;
}


#ifdef UNIX
void sighand( int num )
{
    signal(SIGINT,SIG_IGN);
    signal(SIGTERM,SIG_IGN);
    signal(SIGHUP,SIG_IGN);
    if (flag) {
        tcsetattr(0,TCSANOW,&otty);
    }
    endwin();
    printf("terminated\n");
    exit(0);
}

off_t filelength( int fd)
{
    struct stat statbuf;
    int s;

    s = fstat(fd,&statbuf);
    if (s || !S_ISREG(statbuf.st_mode)) return(0);
    return(statbuf.st_size);
}

int kbhit(void)
{
    int retval;
    fd_set fds;
    struct timeval tv;

 again:
    FD_ZERO(&fds);
    FD_SET(0,&fds);
    tv.tv_sec = tv.tv_usec = 0;
    retval = select(1,&fds,NULL,NULL,&tv);
    if (retval == -1) {
        if (errno == EINTR) goto again;
        if (flag) {
            tcsetattr(0,TCSANOW,&otty);
        }
        endwin();
        fprintf(stderr,"select: %s\n",strerror(errno));
        exit(1);
    }
    if (FD_ISSET(0,&fds)) {
        return(1);
    }
    return(0);
}

void cleanup_for_exit(void)
{
    /*refresh();*/
    if (flag) {
        tcsetattr(0,TCSANOW,&otty);
    }
    endwin();
}

void makeraw(void)
{
    static int first_call = 1;

    if (first_call) {
        first_call = 0;
        if (tcgetattr(0,&tty)) {  /* input terminal */
            endwin();
            fprintf(stderr,"cannot get terminal attributes: %s\n",strerror(errno));
            exit(1);
        }
        otty=tty;  /* save it */
        cfmakeraw(&tty);
        tty.c_lflag |= ISIG;  /* enable signals from ^C etc. */
    }
    if (flag == 0) {
        tcsetattr(0,TCSANOW,&tty);
        flag = 1;
    }
}

void makeunraw(void)
{
    if (flag == 1) {
        tcsetattr(0,TCSANOW,&otty);
        flag = 0;
    }
}

#endif
