/*
 * simple Unix serial comm routines
 * written by Christian Groessler (chris@groessler.org) 2001
 * distribute freely
 */

#include <stdio.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/types.h>
#include <termios.h>
#include <fcntl.h>
#include <curses.h>
#include <string.h>
#ifdef UNIX
#  include <unistd.h>
#endif

#ifndef COMM_NAME  /* be able to override it from the command line / makefile */
#ifdef LINUX
#define COMM_NAME "/dev/ttyS0"
#else
#define COMM_NAME "/dev/tty00"
#endif
#endif

static struct termios tio;

int comm_fd = -1;

int  comm_open  ( int port, unsigned baud )
{
    char fn[] = COMM_NAME;
    tcflag_t baudval = B19200;
    int ret;

    if (port > 4 || baud < 50) return(0);
    fn[strlen(fn)-1] += port - 1;

    comm_fd = open(fn,O_RDWR | O_EXCL | O_NDELAY);
    if (comm_fd == -1) {
        printw("cannot open %s: %s\n",fn,strerror(errno));
        refresh();
        return(0);
    }

    /* get existing settings */
    ret = tcgetattr(comm_fd,&tio);
    if (ret == -1) {
        printw("cannot get line settings!\n");
        refresh();
        close(comm_fd);
        comm_fd = -1;
        return(0);
    }
    if (baud >= 50) baudval = B50;
    if (baud >= 75) baudval = B75;
    if (baud >= 110) baudval = B110;
    if (baud >= 134) baudval = B134;
    if (baud >= 150) baudval = B150;
    if (baud >= 200) baudval = B200;
    if (baud >= 300) baudval = B300;
    if (baud >= 600) baudval = B600;
    if (baud >= 1200) baudval = B1200;
    if (baud >= 1800) baudval = B1800;
    if (baud >= 2400) baudval = B2400;
    if (baud >= 4800) baudval = B4800;
    if (baud >= 9600) baudval = B9600;
    if (baud >= 19200) baudval = B19200;
    if (baud >= 38400) baudval = B38400;

    tio.c_cflag = CS8 | CREAD | CLOCAL;
    cfsetspeed(&tio,baudval);
    cfmakeraw(&tio);

    ret = tcsetattr(comm_fd,TCSANOW,&tio);
    if (ret == -1) {
        close(comm_fd);
        comm_fd = -1;
        printw("cannot set line settings!\n");
        refresh();
        return(0);
    }

    if (fcntl(comm_fd,F_SETFL, fcntl(comm_fd, F_GETFL, 0) & ~O_NDELAY) < 0) {
        close(comm_fd);
        comm_fd = -1;
        printw("cannot remove O_NDELAY!\n");
        refresh();
        return(0);
    }
    return(1);
}

void comm_close ( void )
{
    close(comm_fd);
    comm_fd = -1;
}

void comm_flush ( void )
{
    tcdrain(comm_fd);
}

int  comm_avail ( void )
{
    int retval;
    fd_set fds;
    struct timeval tv;

 again:
    FD_ZERO(&fds);
    FD_SET(comm_fd,&fds);
    tv.tv_sec = tv.tv_usec = 0;
    retval = select(comm_fd+1,&fds,NULL,NULL,&tv); /* no timeout */
    if (retval == -1) {
        if (errno == EINTR) goto again;
        fprintf(stderr,"errorloop in comm_avail!\n");
        goto again;
    }
    if (FD_ISSET(comm_fd,&fds)) {
        return(1);
    }
    return(0);
}

void comm_putc  ( unsigned char c )
{
    write(comm_fd,&c,1);
}

int  comm_getc  ( void )
{
    int res;
    unsigned char c;
    int first = 1;

  retry:
    res = read(comm_fd,&c,1);
    if (res == 1) return(c & 0xff);
    if (res == -1) {
        if (errno == EINTR || errno == EAGAIN) {
            fprintf(stderr,"retrying in comm_getc!\n");
            goto retry;
        }
    }
    if (first == 1) {
        first = 0;
        fprintf(stderr,"errorloop in comm_getc!\n");
    }
    goto retry;
}
