/* Nessus
 * Copyright (C) 1998 Renaud Deraison
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
 
#include <includes.h>
#ifndef NESSUSNT
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#else  
extern int optind;
extern char * optarg;
#endif /* not HAVE_GETOPT_H */   

#else/* not def NESSUSNT */
#include "wstuff.h"
#endif

#ifdef ENABLE_CRYPTO_LAYER
#include "crypto-layer.h"
#endif


#include "pluginload.h"
#include "preferences.h"
#include "auth.h"
#include "rules.h"
#include "comm.h"
#include "attack.h"
#include "sighand.h"
#include "log.h"
#include "threads.h"
#include "users.h"
#include "ntp.h"
#include "utils.h"
#include "corevers.h"

#ifdef DEBUG
int single_shot = 0 ;
#endif

#ifndef HAVE_SETSID
#define setsid() setpgrp()
#endif

/*
 * Globals that should not be touched
 */
int g_max_threads = 15;
struct arglist * g_options = NULL;
int g_serv_socket;
struct arglist * g_plugins;
struct arglist * g_preferences;
struct nessus_rules * g_rules;
struct users * g_users;

/*
 * Functions prototypes
 */
static void main_loop();
static int init_nessusd(struct arglist *, int);
static int init_network(int, int *, struct in_addr);
static void server_thread(struct arglist *);


static void
start_daemon_mode
  (void)
{
  char *s;
  int fd, n;

# ifdef DEBUG
  if (single_shot)
    return ;
# endif

  /* do not block the listener port for sub sequent servers */
  close (g_serv_socket);

  /* become process group leader */
  if (setsid () < 0) {
    log_write 
      ("Cannot set process group leader (%s) -- aborting", strerror (errno));
    exit (0);
  }

  if ((fd = open ("/dev/tty", O_RDWR)) >= 0) {
    /* detach from any controlling terminal */
    ioctl (fd, TIOCNOTTY) ;
    close (fd);
  }
  
  /* no input, anypmore: provide an empty-file substitute */
  if ((fd = open ("/dev/null", O_RDONLY)) < 0) {
    log_write ("Cannot open /dev/null -- aborting",strerror (errno));
    exit (0);
  }
  dup2 (fd, 0);
  close (fd);

  /* provide a dump file to collect stdout and stderr */
  if ((s = arg_get_value (g_preferences, "dumpfile")) == 0)
    s = NESSUSD_DEBUGMSG ;
  /* setting "-" denotes terminal mode */
  if (s [0] != '-' || s [1]) {
#   ifdef ENABLE_CRYPTO_LAYER
    if ((fd = open (s, O_WRONLY|O_CREAT, 0600)) < 0) {
      log_write 
	("Cannot open dumpfile %s (%s)-- aborting", s, strerror (errno));
      exit (2);
    }
    if (peks_private_access (s, 2)) {
      log_write ("Access rights problem with %s (%s)-- aborting", 
		 s, peks_strerr (errno));
      exit (2);
    }
#   else
    unlink (s);
    if ((fd = open (s, O_WRONLY|O_CREAT|O_EXCL, 0600)) < 0) {
      log_write ("Cannot create a new dumpfile %s (%s)-- aborting",
		 s, strerror (errno));
      exit (2);
    }
#   endif /* ENABLE_CRYPTO_LAYER */
    dup2 (fd, 1);
    dup2 (fd, 2);
    close (fd);
    log_write ("Redirecting debugging output to %s", s);
  }
}


static void
end_daemon_mode
  (void)
{
# ifdef DEBUG
  if (single_shot)
    return ;
# endif

  /* clean up all processes the process group */
  make_em_die (SIGTERM);
}


static void
sighup(i)
 int i;
{
  struct arglist * options = g_options;
  log_write("Caught HUP signal - reconfiguring nessusd\n");
  init_nessusd(options,0);
  g_plugins = arg_get_value(options, "plugins");
  g_preferences = arg_get_value(options, "preferences");
  g_rules = arg_get_value(options, "rules");
  g_users = arg_get_value(options, "users");
}



static void 
server_thread(globals)
 struct arglist * globals;
{
 struct sockaddr_in * address = arg_get_value(globals, "client_address");
 struct arglist * plugins = arg_get_value(globals, "plugins");
#ifdef ENABLE_CRYPTO_LAYER
 struct arglist * prefs = arg_get_value (globals, "preferences") ;
 char * opt = NULL;
 int fd, n ;
#endif
 int soc = (int)arg_get_value(globals, "global_socket");
 struct nessus_rules* perms;
 char * asciiaddr;
 struct nessus_rules * rules = arg_get_value(globals, "rules");
 ntp_caps* caps;
 int sz = 4096;
#ifdef USE_PTHREADS
 int off = 0;
 ioctl(soc, FIONBIO, &off);
#ifdef ENABLE_CRYPTO_LAYER
 io_ptinit (0); /* initialize iostreams */
#endif
#endif
	
#ifdef HAVE_ADDR2ASCII
 asciiaddr = emalloc(20);
 addr2ascii(AF_INET, &address->sin_addr, sizeof(struct in_addr), asciiaddr);
#elif defined(HAVE_INET_NETA)
 asciiaddr = emalloc(20);
 inet_neta(ntohl(address->sin_addr.s_addr), asciiaddr, 20);
#else
 asciiaddr = estrdup(inet_ntoa(address->sin_addr));
#endif
 caps = comm_init(soc);
 if(!caps)
 {
  log_write("New connection timeout -- closing the socket\n");
  shutdown(soc, 2);
  close(soc);
  EXIT(0);
 }

#ifdef ENABLE_CRYPTO_LAYER
 opt = arg_get_value(prefs, "force_pubkey_auth");
 if(opt && !strncmp(opt, "yes", 3) && !caps->pubkey_auth) {
    char *error = "This server only accepts public-key authenticated users";
    log_write("%s wants to connect without pubkey auth\n", (char*)asciiaddr);
    send(soc, error, strlen(error), 0);
    goto shutdown_and_exit;
  }
 if (caps->ciphered) {
   /* don't replace s by asciiaddr, weird things may happen ... */
   char *s = inet_ntoa (address->sin_addr) ;
   int n;
   if (server_negotiate_session_key 
       (arg_get_value					/* default peks key */
	(prefs, caps->rpc_client ? "peks_rpckey" : "peks_key"),
	soc,						/* tcp session */
	s,						/* client address */
	0) < 0) {	/* cipher/key list */
     log_write ("While negotiating key with %s: %s\n", s, peks_strerr (errno));
     goto shutdown_and_exit;
   }
   init_process_tracker (globals) ;
   io_ctrl (soc, IO_RESIZE_BUF, (n=4096, &n), 1) ; /* set write buffer */
   io_ctrl (soc, IO_RESIZE_BUF, (n=4096, &n), 0) ; /* limit read buffer */
 }
#endif /* ENABLE_CRYPTO_LAYER */

 arg_add_value(globals, "ntp_caps", ARG_STRUCT, sizeof(*caps), caps);
 plugins_set_ntp_caps(plugins, caps);
 plugins_set_socket(plugins, soc);
 if(((perms = auth_check_user(globals, asciiaddr))==BAD_LOGIN_ATTEMPT)||
   !perms)
 {
   auth_printf(globals, "Bad login attempt !\n"); 
   log_write("bad login attempt from %s\n", 
			asciiaddr);
   efree(&asciiaddr);			
   goto shutdown_and_exit;
 }
  else {
   efree(&asciiaddr);
#ifdef ENABLE_CRYPTO_LAYER
   /* offer rpc services and return */
   if (arg_get_value (prefs, "ena_rpcsrv") != 0 && caps->rpc_client) {
     if (offer_rpc_services (soc, globals) >= 0) {
       char buf [4098];
       int n ;
       /* handle request implicitely */
       io_ctrl (soc, IO_STOPONEMPTY_STATE, (n=1, &n), 0) ;
       io_recv (soc, buf, 4098, 0) ;
     }
     goto shutdown_and_exit;
     /*NOTREACHED*/
   }
#endif /* ENABLE_CRYPTO_LAYER */
   
   if(perms){
     	rules_add(&rules, &perms);
#ifdef DEBUG_RULES
	printf("Rules have been added : \n");
	rules_dump(rules);
#endif	
	arg_set_value(globals, "rules", -1, rules);
   }
  
   comm_send_pluginlist(globals);
   if(caps->ntp_11){
       comm_send_preferences(globals);
       comm_send_rules(globals);
       }
#ifdef ENABLE_CRYPTO_LAYER_currently_disabled
   log_user_cookie (globals, "login", 0, 0) ;
#endif /* ENABLE_CRYPTO_LAYER */

   comm_wait_order(globals);
   rules = arg_get_value(globals, "rules");

   /* become process group leader and the like ... */
   start_daemon_mode ();

#ifdef ENABLE_CRYPTO_LAYER
   io_ctrl (soc, IO_RESIZE_BUF, (n=1024, &n), 0) ; /* limit read buffer */
#endif
   attack_host(globals);
   comm_terminate(globals);
  
#ifdef ENABLE_CRYPTO_LAYER_currently_disabled
   log_user_cookie (globals, "terminated", 0, 0) ;
#endif /* ENABLE_CRYPTO_LAYER */
  }

 shutdown_and_exit:
 shutdown(soc, 2);
 close(soc);
 /* kill left overs */
 end_daemon_mode ();
#ifndef NESSUSNT
 /*arg_free(globals); */
#endif
 EXIT(0);
}
     
static void 
main_loop()
{
  char *old_addr = 0, *asciiaddr = 0;
  time_t last = 0;
  int count ;

#ifdef ENABLE_CRYPTO_LAYER
  int lid = 0 ;
  char * logp = arg_get_value (g_preferences,"cookie_logpipe");
#ifdef USE_PTHREADS
  /* pre-initialize iostream data */
  io_ptinit (0);
  io_ptdestroy ();
#endif
#endif /* ENABLE_CRYPTO_LAYER */

  /* catch dead children */
  nessus_signal(SIGCHLD, sighand_chld);

  while(1)
    {
      int soc;
      unsigned int lg_address = sizeof(struct sockaddr_in);
      struct sockaddr_in address;
      struct arglist * globals;
      struct arglist * my_plugins, * my_preferences;
      struct nessus_rules * my_rules;
      struct sockaddr_in * p_addr;
     
      /* prevent from an io table overflow attack against nessus */
      if (asciiaddr != 0) {
	time_t now = time (0);

	/* did we reach the max nums of connect/secs ? */
	if (last == now) {
	  if (++ count > NESSUSD_CONNECT_RATE) {
	    log_write 
	      ("connection rate/sec exceeded - blocking for a while\n");
	    sleep (NESSUSD_CONNECT_BLOCKER);
	    last = 0 ;
	  }
	} else {
	  count = 0 ;
	  last = now ;
	}
	
	if (old_addr != 0) {
	  /* detect whether sombody logs in more than once in a row */
	  if (strcmp (old_addr, asciiaddr) == 0 &&
	      now < last + NESSUSD_CONNECT_RATE) {
	    log_write
	      ("same client %s has connected twice - blocking for a while\n", 
	       asciiaddr);
	    sleep (1);
	  }
	  efree (&old_addr);
	  old_addr = 0 ; /* currently done by efree, as well */
	}
      }
      old_addr = asciiaddr ;
      asciiaddr = 0 ;
      
      soc = accept(g_serv_socket, (struct sockaddr *)(&address), &lg_address);
      if(soc == -1)continue;
      asciiaddr = estrdup(inet_ntoa(address.sin_addr));
      log_write("connection from %s\n", (char *)asciiaddr);
      /* efree(&asciiaddr); */
      /* 
       * duplicate everything so that the threads don't share the
       * same variables.
       *
       * Useless when fork is used, necessary for the pthreads and
       * the NT threads
       */
      globals = emalloc(sizeof(struct arglist));
      arg_add_value(globals, "global_socket", ARG_INT, -1, (void *)soc);

#ifdef ENABLE_CRYPTO_LAYER_currently_disabled
      /* check, whether we need to (re)start the pipe handler */
      if (logp != 0) {
	if ((lid = install_pipe_logger (logp, lid, globals)) < 0) {
	  print_error ("Canot start logpipe server on %s (%s) -- aborting\n",
		       logp, peks_strerr (errno));
	  EXIT (1);
	}
      }
#endif
      
#ifdef USE_FORK_THREADS
      my_plugins = g_plugins;
#else      
      my_plugins = emalloc(sizeof(struct arglist));
      arg_dup(my_plugins, g_plugins);
#endif    
      arg_add_value(globals, "plugins", ARG_ARGLIST, -1, my_plugins);
      
     
#ifdef USE_FORK_THREADS     
      my_preferences = g_preferences;
#else      
      my_preferences = emalloc(sizeof(struct arglist));
      arg_dup(my_preferences, g_preferences);
#endif      
      arg_add_value(globals, "preferences", ARG_ARGLIST, -1, my_preferences);
          
      my_rules = /*rules_dup*/(g_rules);
      
      arg_add_value(globals, "users", ARG_PTR, -1, g_users);
       
      p_addr = emalloc(sizeof(struct sockaddr_in));
      *p_addr = address;
      arg_add_value(globals, "client_address", ARG_PTR, -1, p_addr);
      
      arg_add_value(globals, "rules", ARG_PTR, -1, my_rules);

      /* we do not want to create an io thread, yet so the last argument is -1 */
      create_thread(server_thread, globals, (struct arglist *)(-1));
#ifdef USE_FORK_THREADS
      arg_free(globals);
#endif    
    }
}


/*
 * Initialization of the network : 
 * we setup the socket that will listen for
 * incoming connections on port <port> on address
 * <addr> (which are set to 3001 and INADDR_ANY by
 * default)
 */ 
static int 
init_network(port, sock, addr)
     int port;
     int * sock;
     struct in_addr addr;
{

#ifndef NESSUSNT
  int option = 1;
#endif

#ifdef USE_AF_INET
  struct sockaddr_in address;
#else
  struct sockaddr_un address;
  char * name = AF_UNIX_PATH;
#endif


#ifdef NESSUSNT
    WORD wVersionRequested;
    WSADATA wsaData;
    int err;
    
    wVersionRequested = MAKEWORD( 2, 0 );
    err = WSAStartup( wVersionRequested, &wsaData );
    if(err){print_error("Could not initialize WinSock !");DO_EXIT(0);}
#endif

#ifdef USE_AF_INET
  if((*sock = socket(AF_INET, SOCK_STREAM, 0))==-1)
    {
      perror("socket ");
      DO_EXIT(1);
    }
  bzero(&address, sizeof(struct sockaddr_in));
  address.sin_family = AF_INET;
  address.sin_addr = addr;
  address.sin_port = htons((unsigned short)port);
#else
  if((*sock = socket(AF_UNIX, SOCK_STREAM,0))==-1)
    {
     perror("socket ");
     DO_EXIT(1);
   }
  bzero(&address, sizeof(struct sockaddr_un));
  address.sun_family = AF_UNIX;
  bcopy(name, address.sun_path, strlen(name));
#endif

#ifndef NESSUSNT
  setsockopt(*sock, SOL_SOCKET, SO_REUSEADDR, &option, sizeof(int));
#endif
  if(bind(*sock, (struct sockaddr *)(&address), sizeof(address))==-1)
    {
      perror("bind ");
      DO_EXIT(1);
    }
  if(listen(*sock, 10)==-1)
    {
      perror("listen ");
      shutdown(*sock, 2);
      close(*sock);
      DO_EXIT(1);
    }
  return(0);
}

/*
 * Initialize everything
 */
static int 
init_nessusd(options, first_pass)
     struct arglist * options;    
     int first_pass;

{
  int  sock;
  struct arglist * plugins;
  struct arglist * preferences;
  struct nessus_rules * rules = NULL;
  struct users * users;
  int serv_port = (int)arg_get_value(options, "serv_port");
  char * config_file = arg_get_value(options, "config_file");
  struct in_addr * addr = arg_get_value(options, "addr");
  preferences_init(config_file, &preferences);
  /*
  if(arg_get_value(preferences, "max_threads"))
   g_max_threads = atoi(arg_get_value(preferences, "max_threads"));
 */
#ifdef ENABLE_CRYPTO_LAYER
  cipher_stuff_init        (options, preferences) ;
  do_local_cipher_commands (options, preferences) ;
  /* connect to rpc server */
  if (arg_get_value (preferences, "dis_rpcfwd") == 0 &&
      connect_to_rpc_server (options, preferences) < 0)
    DO_EXIT (1);
  /* set global I-am-preks-rpc-server flag */
  if (arg_get_value (options, "ena_rpcsrv"))
    arg_add_value (preferences, "ena_rpcsrv", ARG_INT, sizeof (int), (void*)1);
#endif

  arg_add_value(preferences, "config_file", ARG_STRING, strlen(config_file), 
                config_file);
  log_init(arg_get_value(preferences, "logfile"));
  rules_init(&rules, preferences);
#ifdef DEBUG_RULES
  rules_dump(rules);
#endif
  users_init(&users, preferences);
  plugins = plugins_init(preferences);
  if(first_pass)init_network(serv_port, &sock, *addr);

  
#ifndef USE_PTHREADS
  if(first_pass)
  {
   nessus_signal(SIGCHLD, sighand_chld);
   nessus_signal(SIGTERM, sighand_term);
   nessus_signal(SIGINT, sighand_int);
   nessus_signal(SIGKILL, sighand_kill);
   nessus_signal(SIGHUP, sighup);
#ifndef DEBUG
   nessus_signal(SIGSEGV, sighand_segv);
#endif
   nessus_signal(SIGPIPE, SIG_IGN);
  }
#endif  
   arg_add_value(options, "sock", ARG_INT, sizeof(int),(void *)sock);
   arg_add_value(options, "plugins", ARG_ARGLIST, -1, plugins);
   arg_add_value(options, "rules", ARG_PTR, -1, rules);
   arg_add_value(options, "users", ARG_PTR, -1, users);
   arg_add_value(options, "preferences", ARG_ARGLIST, -1, preferences);
   return(0);
}


static void 
display_help()
{
  printf("nessusd, version %s\n", NESSUS_VERSION);
#ifdef USE_AF_INET
#ifdef ENABLE_CRYPTO_LAYER
  printf("\nusage : nessusd [options]\n\noptions:\n");
#else
  printf("\nusage : nessusd [-vcphdDLC] [-a address]\n\n");
#endif /* ENABLE_CRYPTO_LAYER */
#else
  printf("\nusage : nessusd [-vchdD]\n\n");
#endif /*def USE_AF_INET */

  printf("\ta <address>    : listen on <address>\n");
  printf("\tv              : shows version number\n");
  printf("\th              : shows this help\n");

#ifdef USE_AF_INET
  printf("\tp <number>     : use port number <number>\n");
#endif /* USE_AF_INET */
#ifdef ENABLE_CRYPTO_LAYER
  printf("\tC              : change passphrase for private key\n");
  printf("\tP <user>       : list temporary user passwd\n");
  printf("\tP <user>,<pwd> : replace temporary user passwd\n");
  printf("\tP <user>,      : delete temporary user passwd\n");
  printf("\tK <user>       : kill registered user auth key\n");
  printf("\tL              : list user auth key entries\n");
 #endif /* ENABLE_CRYPTO_LAYER */
  printf("\tc <filename>   : alternate configuration file to use\n");
  printf("\t\t\t (default : %s)\n", NESSUSD_CONF);
  printf("\tD              : runs in daemon mode\n");
  printf("\td              : dumps the nessusd compilation options\n");
}

#ifdef NESSUSNT
int WINAPI WinMain(HINSTANCE hThisInst, HINSTANCE hPrevInst,
                                 LPSTR lpszArgs, int nWinMode)
#else
int 
main(int argc, char * argv[])
#endif
{
  int serv_port = -1;
  char * config_file = NULL;
  int do_fork = 0;
  struct in_addr addr; 
  struct arglist * options = emalloc(sizeof(struct arglist));
#ifndef NESSUSNT
  int i;
#endif
#ifdef RLIMIT_CORE
  struct rlimit rl = {0,0};
  setrlimit(RLIMIT_CORE, &rl);
#endif
  if(getuid())
  {
    printf("Only root should start nessusd.\n");
     exit(1);
  }
  addr.s_addr = htonl(INADDR_ANY);
#ifdef USE_PTHREADS
  /* pull in library symbols - otherwise abort */
  nessuslib_pthreads_enabled ();
#ifdef ENABLE_CRYPTO_LAYER
  io_ptavail ();
#endif
#endif
#ifndef NESSUSNT

  while((i=getopt(argc, argv, "CDhvc:d0123a:p:K:P:LSFQ:e"))!=-1)
      switch(i)
	{
	case 'e' :
	  break;
	  
        case 'a' :
          if(!optarg){display_help();DO_EXIT(0);}
          if(!inet_aton(optarg, &addr)){display_help();DO_EXIT(0);}
          break;
          
	case 'p' :
	  if(!optarg){display_help();DO_EXIT(0);}
	  serv_port = atoi(optarg);
	  if((serv_port<=0)||(serv_port>=65536))
	    {
	      printf("Invalid port specification\n");
	      display_help();
	      DO_EXIT(1);
	    }
	  break;

	case 'S' :
#ifdef ENABLE_CRYPTO_LAYER
	  arg_add_value (options, "ena_rpcsrv", ARG_INT, sizeof(int), (void*)1);
 	  break;
#else
 	  goto disp_help ;
#endif /* ENABLE_CRYPTO_LAYER */

 	case 'F' :
#ifdef ENABLE_CRYPTO_LAYER
	  arg_add_value(options, "dis_rpcfwd", ARG_INT, sizeof(int), (void*)1);
 	  break;
#else
 	  goto disp_help ;
#endif /* ENABLE_CRYPTO_LAYER */

 	case 'Q' :
#ifdef ENABLE_CRYPTO_LAYER
 	  /* syntax: passwd */
	  arg_add_value(options, "rpc_passwd", ARG_PTR, -1,  estrdup (optarg));
 	  break;
#else
 	  goto disp_help ;
#endif /* ENABLE_CRYPTO_LAYER */

 	case 'P' :
#ifdef ENABLE_CRYPTO_LAYER
 	  /* syntax: user[,passwd] */
	  if (strchr (optarg, ',') != 0) {
	    char *usr = estrdup (optarg) ;
	    char *pwd  = strchr (usr, ',') ;
	    * pwd ++ = '\0' ;
	    arg_add_value(options, "pwd_passwd", ARG_PTR, -1, pwd);
	    arg_add_value(options, "pwd_user",   ARG_PTR, -1, usr);
	    break;
	  }
	  arg_add_value(options, "pwd_user", ARG_PTR, -1, estrdup (optarg));
 	  break;
#else
 	  goto disp_help ;
#endif /* ENABLE_CRYPTO_LAYER */
 
 	case 'K' :
#ifdef ENABLE_CRYPTO_LAYER
 	  /* syntax: user */
	  arg_add_value(options, "kill_userkey", ARG_PTR, -1, estrdup (optarg));
 	  break;
#else
 	  goto disp_help ;
#endif /* ENABLE_CRYPTO_LAYER */
 
 	case 'L' :
#ifdef ENABLE_CRYPTO_LAYER
	  arg_add_value(options, "list_udb", ARG_INT, sizeof(int), (void*)1);
	  break;
#else
 	  goto disp_help ;
#endif /* ENABLE_CRYPTO_LAYER */

 	case 'C' :
#ifdef ENABLE_CRYPTO_LAYER
	  arg_add_value(options, "chng_pph", ARG_INT, sizeof(int), (void*)1);
	  break;
#else
 	  goto disp_help ;
#endif /* ENABLE_CRYPTO_LAYER */
 
	case 'D' : 
	  do_fork++;
	  break;
	  
	case 'v' :
	  print_error("nessusd (%s) %s for %s\n(C) 1998-1999 Renaud Deraison <deraison@nessus.org>\n\n", 
		 PROGNAME,NESSUS_VERSION, NESS_OS_NAME);
	  DO_EXIT(0);
	  break;
#ifdef ENABLE_CRYPTO_LAYER
	case '0' : dump_recv = 1; break;
	case '1' : dump_send = 1; break;
	case '2' : dump_iopt = 1; break;
 	case '3' :
# ifdef DEBUG
	  single_shot = 1; 
	  break;
# endif
#else
	case '0' :
	case '1' :
	case '2' :
 	case '3' :
#endif
	case 'h' :
 	case '?' : /* getopt: error */
 	case ':' : /* getopt: missing parameter */
 	disp_help:
	  display_help();
	  DO_EXIT(0);
	  break;
	case 'c' : 
	  if(!optarg){display_help();DO_EXIT(1);}
	  config_file = emalloc(strlen(optarg)+1);
	  strncpy(config_file, optarg, strlen(optarg));
	  break;
       case 'd' :
           printf("This is Nessus %s for %s %s\n", NESSUS_VERSION, NESS_OS_NAME, NESS_OS_VERSION);
           printf("compiled with %s\n", NESS_COMPILER);
           printf("Current setup :\n");
           printf("\tThread manager : ");
#ifdef USE_PTHREADS
	   printf("pthreads\n");
#else
	   printf("fork\n");
#endif
#ifdef ENABLE_CRYPTO_LAYER
	   printf("\tCrypto layer : %s\n", peks_version ());
#endif
	   printf("\tRunning as euid : %d\n", geteuid());
           printf("\n\nInclude these infos in your bug reports\n");
	   DO_EXIT(0);
	   break;
	}
#endif

  if(serv_port == -1)serv_port = DEFAULT_PORT;
  if(!config_file)
    {
      config_file = emalloc(strlen(NESSUSD_CONF)+1);
      strncpy(config_file, NESSUSD_CONF, strlen(NESSUSD_CONF));
    }
  arg_add_value(options, "serv_port", ARG_INT, sizeof(int), (void *)serv_port);
  arg_add_value(options, "config_file", ARG_STRING, strlen(config_file), 
  		config_file);
  arg_add_value(options, "addr", ARG_PTR, -1, &addr);
  init_nessusd(options, 1);
  g_options = options;
  g_serv_socket = (int)arg_get_value(options, "sock");
  g_plugins = arg_get_value(options, "plugins");
  g_preferences = arg_get_value(options, "preferences");
  g_rules = arg_get_value(options, "rules");
  g_users = arg_get_value(options, "users");

#if 0
  /* this is my code to test the pty suff -- jordan */
  arg_add_value (g_preferences, "dumpfile", ARG_STRING, 2, "-");
  start_daemon_mode ();
  { 
    FILE *fp ;
    char *cmd, **argv, buf [1000], *s ;

    /* nessus_signal (SIGHUP, SIG_IGN); */
    cmd = "/bin/cat";
    argv = append_argv      (0, "cat");
    fp = ptyexecvp (cmd, argv, 0);
    destroy_argv (argv);
    fputs ("how much wood ...\n", fp);
    while ((s = fgets (buf, sizeof (buf), fp)) == 0 && errno == EAGAIN)
      ;
    while (s) {
      printf (s);
      s = fgets (buf, sizeof (buf), fp) ;
    }
    fclose (fp);

    /* new game */

    cmd  = "/bin/ls";
    argv = append_argv      (0, "ls");
    argv = append_argv (argv, "/dev");
    fp = ptyexecvp (cmd, argv, 0);
    destroy_argv (argv);
    while ((s = fgets (buf, sizeof (buf), fp)) == 0 && errno == EAGAIN)
      ;
    while (s) {
      printf (s);
      s = fgets (buf, sizeof (buf), fp) ;
    }
    fclose (fp);
  }
  exit (1);
#endif

#ifdef NESSUSNT
  /*
   * Just tell to the user that nessusd is running in background
   */
  create_thread(print_error, "nessusd is now running in background", 0);
  main_loop();
#else
  if(do_fork)
  {  	
   if(!fork()){
        setsid();
   	main_loop();
	}
  }
  else main_loop();
#endif
  DO_EXIT(0);
  return(0);
}
