/* 
   cadaver, command-line DAV client
   Copyright (C) 1999-2001, Joe Orton <joe@orton.demon.co.uk>, 
   except where otherwise indicated.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

/* Some UI guidelines:
 *  1. Use dispatch, or out_* to do UI. This makes it CONSISTENT.
 *  2. Get some feedback on the screen before making any requests
 *     to the server. Tell them what is going on: remember, on a slow
 *     link or a loaded server, a request can take AGES to return.
 */

#include <config.h>

#include <sys/types.h>

#include <sys/stat.h>
#include <sys/wait.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include <fcntl.h>
#include <errno.h>

/* readline requires FILE *, silly thing */
#include <stdio.h>

#ifdef HAVE_READLINE_H
#include <readline.h>
#elif HAVE_READLINE_READLINE_H
#include <readline/readline.h>
#endif

#ifdef NEED_SNPRINTF_H
#include "snprintf.h"
#endif

#include <ne_request.h>
#include <ne_basic.h>
#include <ne_auth.h> /* for http_forget_auth */
#include <ne_redirect.h>
#include <ne_props.h>
#include <ne_string.h>
#include <ne_uri.h>
#include <ne_locks.h>
#include <ne_alloc.h>
#include <ne_dates.h>

#include "i18n.h"
#include "basename.h"
#include "dirname.h"
#include "cadaver.h"
#include "commands.h"
#include "options.h"

int yesno(void);

/* Local variables */
int child_running; /* true when we have a child running */

/* Command alias mappings */
const static struct {
    enum command_id id;
    const char *name;
} command_names[] = {
    /* The direct mappings */
#define C(n) { cmd_##n, #n }
    C(ls), C(cd), C(quit), C(open), C(logout), C(close), C(set), C(unset), 
    C(pwd), C(help), C(put), C(get), C(mkcol), C(delete), C(move), C(copy),
    C(less), C(cat), C(lpwd), C(lcd), C(lls), C(echo), C(quit),
    C(mget), C(mput), C(rmcol), C(lock), C(unlock), C(discover), C(steal),
    C(chexec), C(showlocks), C(version), C(propget), C(propset),
#if 0
C(propedit), 
#endif
C(propnames), C(edit),
#undef C
    /* And now the real aliases */
    { cmd_less, "more" }, { cmd_mkcol, "mkdir" }, 
    { cmd_delete, "rm" }, { cmd_copy, "cp"}, { cmd_move, "mv" }, 
    { cmd_help, "h" }, { cmd_help, "?" },
    { cmd_quit, "exit" }, { cmd_quit, "bye" },
    { cmd_unknown, NULL }
};    

/* Tell them we are doing 'VERB' to 'NOUN'.
 * (really 'METHOD' to 'RESOURCE' but hey.) */
void out_start(const char *verb, const char *noun)
{
    output(o_start, "%s `%s':", verb, noun);
}

void out_success(void)
{
    output(o_finish, _("succeeded.\n"));
}

void out_result(int ret)
{
    switch (ret) {
    case NE_OK:
	out_success();
	break;
    case NE_AUTH:
    case NE_AUTHPROXY:
	output(o_finish, _("authentication failed.\n"));
	break;
    case NE_CONNECT:
	output(o_finish, _("could not connect to server.\n"));
	break;
    case NE_TIMEOUT:
	output(o_finish, _("connection timed out.\n"));
	break;
    case NE_REDIRECT:
	output(o_finish, _("redirected to a different server:\n%s\n"),
	       ne_redirect_location(session));
	break;
    default:
	output(o_finish, _("failed:\n%s\n"), ne_get_error(session));
	break;
    }
}

int out_handle(int ret)
{
    out_result(ret);
    return (ret == NE_OK);
}

/* The actual commands */

void init_locking(const char *lockstore)
{
    lock_session = ne_lock_register(session);
    /* TODO: read in lock list from ~/.davlocks */
}

void finish_locking(const char *lockstore)
{
    /* TODO: write out lock list to ~/.davlocks */
}

/* Auxiliaries */

static void 
map_multi(void (*func)(const char *), int argc, const char *argv[]);

static void simple_copy(const char *from, const char *to);
static void simple_put(const char *local, const char *remote);
static void simple_move(const char *src, const char *dest);

static void do_copymove(int argc, const char *argv[],
			const char *v1, const char *v2,
			void (*cb)(const char *, const char *));

static char *clever_path(const char *p, const char *src, 
			 const char *dest);

static const char *choose_pager(void);

static void dispatch(const char *verb, const char *filename, 
		     int (*func)(ne_session *, const char *), 
		     const char *pass);

static void display_help_message(void);

static void print_lock(const struct ne_lock *lock);

#ifdef HAVE_LIBREADLINE

/* Command name generator for readline.
 * Copied almost verbatim from the info doc */
char *command_generator(char *text, int state)
{
    static int i, len;
    const char *name;

    if (!state) {
	i = 0;
	len = strlen(text);
    }

    while ((name = command_names[i].name) != NULL) {
	i++;
	if (strncmp(name, text, len) == 0) {
	    return ne_strdup(name);
	}
    }
    return NULL;
}

#endif

void execute_logout(void)
{
    ne_forget_auth(session);
}

const struct command *get_command(const char *name)
{
    int n, m;
    for(n = 0; command_names[n].name != NULL; n++) {
	if (strcasecmp(command_names[n].name, name) == 0) {
	    for(m = 0; commands[m].id != cmd_unknown; m++) {
		if (commands[m].id == command_names[n].id)
		    return &commands[m];
	    }
	    return NULL;
	}
    }
    return NULL;
}


/* A bit like Haskell's map function... applies func to each
 * of the first argc items in argv */
static void 
map_multi(void (*func)(const char *), int argc, const char *argv[])
{
    int n;
    for(n = 0; n < argc; n++) {
	(*func)(argv[n]);
    }
}

void multi_mkcol(int argc, const char *argv[])
{
    map_multi(execute_mkcol, argc, argv);
}

void multi_delete(int argc, const char *argv[])
{
    map_multi(execute_delete, argc, argv);
}

void multi_rmcol(int argc, const char *argv[])
{
    map_multi(execute_rmcol, argc, argv);
}

void multi_less(int argc, const char *argv[])
{
    map_multi(execute_less, argc, argv);
}

void multi_cat(int argc, const char *argv[])
{
    map_multi(execute_cat, argc, argv);
}

static void dispatch(const char *verb, const char *filename, 
		     int (*func)(ne_session *, const char *), const char *arg)
{
    out_start(verb, filename);
    out_result((*func)(session, arg));
}

static char * getowner(void)
{
#if 0
    /* mod_dav breaks if you submit an owner */
    return ne_strdup("");
#else
    char *ret, *owner = get_option(opt_lockowner);
    if (owner) {
	CONCAT3(ret, "<href>", owner, "</href>");
	return ret;
    } else {
	return NULL;
    }
#endif
}

/* If in UTF-8 mode, simply duplicates 'str'.
 * When not in UTF-8 mode, presume 'str' is ISO-8859-1,
 * and UTF-8 encode it.
 *
 * TODO: I'm not 100% sure this is right. At least, the strdup
 * doesn't cope with embedded NUL's which it should, right?
 */
static char *utf8_encode(const char *str)
{
    if (get_bool_option(opt_utf8)) {
	return ne_strdup(str);
    } else {
	return ne_utf8_encode(str);
    }
}

/* If in UTF-8 mode, simply duplicates 'str'.
 * When not in UTF-8 mode, does a UTF-8 decode on 'str',
 * or at least, the least significant 8-bits of the Unicode
 * characters in 'str'.  Returns NULL if 'str' contains
 * >8-bit characters.
 *
 * TODO: yes, is this sensible? embedded NUL's?
 */
static char *utf8_decode(const char *str)
{
    /* decoded version can be at most as long as encoded version. */
    if (get_bool_option(opt_utf8)) {
	return ne_strdup(str);
    } else {
	return ne_utf8_decode(str);
    }
}

/* Resolve path, appending trailing slash if resource is a
 * collection. This function MAY make a request to the server.
 * TODO: caching, so it MAY not, too.
 */
static char *true_path(const char *res)
{
    char *full;
    full = resolve_path(path, res, false);
    if (is_collection(full)) {
	if (!uri_has_trailing_slash(full)) {
	    char *tmp;
	    CONCAT2(tmp, full, "/");
	    free(full);
	    full = tmp;
	}
    }
    return full;
}

static const char *get_lockscope(enum ne_lock_scope s)
{
    switch (s) {
    case ne_lockscope_exclusive: return _("exclusive");
    case ne_lockscope_shared: return _("shared");
    default: return _("unknown");
    }
}

static const char *get_locktype(enum ne_lock_type t)
{
    if (t == ne_locktype_write) {
	return _("write");
    } else {
	return _("unknown");
    }
}

static const char *get_timeout(int t)
{
    static char buf[128];
    switch (t) {
    case NE_TIMEOUT_INFINITE: return _("infinite");
    case NE_TIMEOUT_INVALID: return _("invalid");
    default:
	sprintf(buf, "%d", t);
	return buf;
    }
}

static const char *get_depth(int d)
{
    switch (d) {
    case NE_DEPTH_INFINITE:
	return _("infinity");
    case 0:
	/* TODO: errr... do I need to i18n'ize numeric strings??? */
	return "0";
    case 1:
	return "1";
    default:
	return _("invalid");
    }
}

static void print_lock(const struct ne_lock *lock)
{
    printf(_("Lock <%s> on %s:\n"
	     "  Scope: %s  Type: %s  Timeout: %s\n"
	     "  Owner: %s  Depth: %s\n"), 
	       lock->token, lock->uri,
	   get_lockscope(lock->scope),
	   get_locktype(lock->type), get_timeout(lock->timeout),
	   lock->owner?lock->owner:_("(none)"), get_depth(lock->depth));
}

static void discover_result(void *userdata, const struct ne_lock *lock, 
			    const char *uri, const ne_status *status)
{
    if (lock) {
	print_lock(lock);
    } else {
	printf(_("Failed on %s: %d %s\n"), uri, 
	       status->code, status->reason_phrase);
    }
}

void execute_discover(const char *res)
{
    char *real_remote;
    int ret;
    real_remote = resolve_path(path, res, false);
    out_start(_("Discovering locks on"), res);
    ret = ne_lock_discover(session, real_remote, 
			    discover_result, NULL);
    switch (ret) {
    case NE_OK:
	out_success();
	break;
    default:
	out_result(ret);
	break;
    }
    free(real_remote);
}

static void steal_result(void *userdata, const struct ne_lock *lock, 
			 const char *uri, const ne_status *status)
{
    if (lock != NULL) {
	printf("%s: <%s>\n", lock->uri, lock->token);
	ne_lock_add(lock_session, ne_lock_copy(lock));
    } else {
	printf(_("Failed on %s: %d %s\n"), uri,
	       status->code, status->reason_phrase);
    }
}

void execute_steal(const char *res)
{
    char *real_remote;
    int ret;
    real_remote = resolve_path(path, res, false);
    out_start(_("Stealing locks on"), res);
    ret = ne_lock_discover(session, real_remote, 
			    steal_result, NULL);
    switch (ret) {
    case NE_OK:
	out_success();
	break;
    default:
	out_result(ret);
	break;
    }
    
    free(real_remote);
}

static void showlocks_walker(struct ne_lock *lock, void *userdata)
{
    print_lock(lock);
}

void execute_showlocks(void)
{
    printf(_("List of owned locks:\n"));
    if (ne_lock_iterate(lock_session, showlocks_walker, NULL) == 0) {
	printf(_("No owned locks.\n"));
    }
}

void execute_lock(const char *res)
{
    char *real_remote;
    struct ne_lock *lock;
    lock = ne_calloc(sizeof *lock);
    out_start(_("Locking"), res);
    real_remote = true_path(res);
    lock->type = ne_locktype_write;
    lock->scope = ne_lockscope_exclusive;
    lock->owner = getowner();
    if (uri_has_trailing_slash(real_remote)) {
	lock->depth = NE_DEPTH_INFINITE;
    } else {
	lock->depth = 0;
    }
    lock->uri = real_remote;
    if (out_handle(ne_lock(session,lock))) {
	/* Locktoken: <%s>\n", lock->token); */
	ne_lock_add(lock_session, lock);
    } else {
	ne_lock_free(lock);
    }
}

void execute_unlock(const char *res)
{
    struct ne_lock *lock;
    char *real_remote;
    int in_list = 1;
    out_start(_("Unlocking"), res);
    real_remote = true_path(res);
    lock = ne_lock_find(lock_session, real_remote);
    if (!lock) {
	lock = ne_calloc(sizeof *lock);
	lock->token = readline(_("Enter locktoken: "));
	if (!lock->token || strlen(lock->token) == 0) {
	    free(lock);
	    return;
	}
	lock->uri = real_remote;
	in_list = 0;
    }
    out_result(ne_unlock(session, lock));

    if (in_list) {
	ne_lock_remove(lock_session, lock);
	free(real_remote);
    } 
    /* else real_remote will get free'd here... */
    ne_lock_free(lock);
}

void execute_mkcol(const char *filename)
{
    char *remote;
    remote = resolve_path(path, filename, true);
    dispatch("Creating", filename, ne_mkcol, remote);
    free(remote);
}

static int all_iterator(void *userdata, const ne_propname *pname,
			 const char *value, const ne_status *status)
{
    char *dname = utf8_decode(pname->name);
    if (value != NULL) {
	char *dval = utf8_decode(value);
	printf("%s = %s\n", dname, dval);
	free(dval);
    } else if (status != NULL) {
	printf("%s failed: %s\n", dname, status->reason_phrase);
    }
    free(dname);
    return 0;
}

static void pget_results(void *userdata, const char *uri,
			 const ne_prop_result_set *set)
{
    ne_propname *pname = userdata;
    const char *value;
    char *dname;

    printf("\n");

    if (userdata == NULL) {
	/* allprop */
	ne_propset_iterate(set, all_iterator, NULL);
	return;
    }
    
    dname = utf8_decode(pname->name);

    value = ne_propset_value(set, pname);
    if (value != NULL) {
	char *dval = utf8_decode(value);
	printf(_("Value of %s is: %s\n"), dname, dval);
	free(dval);
    } else {
	const ne_status *status = ne_propset_status(set, pname);
	
	if (status) {
	    printf(_("Could not fetch property: %d %s\n"), 
		   status->code, status->reason_phrase);
	} else {
	    printf(_("Server did not return result for %s\n"),
		   dname);
	}
    } 

    free(dname);
}

void execute_propset(const char *res, const char *name, const char *value)
{
    ne_proppatch_operation ops[2];
    ne_propname pname;
    char *uri = resolve_path(path, res, false);
    char *val, *encname = utf8_encode(name);

    ops[0].name = &pname;
    ops[0].type = ne_propset;
    ops[0].value = val = utf8_encode(value);
    ops[1].name = NULL;
    
    pname.nspace = (const char *)get_option(opt_namespace);
    pname.name = encname;

    out_start(_("Setting property on "), res);
    out_handle(ne_proppatch(session, uri, ops));
    
    free(val);
    free(encname);
    free(uri);
}    

void execute_propget(const char *res, const char *name)
{
    ne_propname pnames[2] = {{NULL}, {NULL}};
    char *remote = resolve_path(path, res, false), *encname = NULL;
    ne_propname *props;
    int ret;
    
    if (name == NULL) {
	props = NULL;
    } else {
	pnames[0].nspace = (const char *)get_option(opt_namespace);
	pnames[0].name = encname = utf8_encode(name);
	props = pnames;
    }

    out_start(_("Fetching properties for"), res);
    ret = ne_simple_propfind(session, remote, NE_DEPTH_ZERO, props,
			      pget_results, props);

    if (ret != NE_OK) {
	out_result(ret);
    }

    if (encname) {
	free(encname);
    }
    free(remote);
}

static int propname_iterator(void *userdata, const ne_propname *pname,
			     const char *value, const ne_status *st)
{
    printf(" %s %s\n", pname->nspace, pname->name);
    return 0;
}

static void propname_results(void *userdata, const char *href,
			     const ne_prop_result_set *pset)
{
    ne_propset_iterate(pset, propname_iterator, NULL);
}

void execute_propnames(const char *res)
{
    char *remote;
    remote = resolve_path(path, res, false);
    out_start("Fetching property names", res);
    if (out_handle(ne_propnames(session, remote, NE_DEPTH_ZERO,
				 propname_results, NULL))) { 
    }
    free(remote);
}

static void remove_locks(const char *uri, int depth)
{
    struct ne_lock *lck;
    
    lck = ne_lock_find(lock_session, uri);
    if (lck != NULL) {
	ne_lock_remove(lock_session, lck);
    }

}

void execute_delete(const char *filename)
{
    char *remote = resolve_path(path, filename, false);
    out_start(_("Deleting"), filename);
    if (is_collection(remote)) {
	output(o_finish, 
_("is a collection resource.\n"
"The `rm' command cannot be used to delete a collection.\n"
"Use `rmcol %s' to delete this collection and ALL its contents.\n"), 
filename);
    } else {
	if (out_handle(ne_delete(session, remote))) {
	    remove_locks(remote, 0);
	}
    }
    free(remote);
}

void execute_rmcol(const char *filename)
{
    char *remote;
    remote = resolve_path(path, filename, true);
    out_start(_("Deleting collection"), filename);
    if (!is_collection(remote)) {
	output(o_finish, 
	       _("is not a collection.\n"
		 "The `rmcol' command can only be used to delete collections.\n"
		 "Use `rm %s' to delete this resource.\n"), filename);
    } else {
	out_result(ne_delete(session, remote));
    }
    remove_locks(remote, NE_DEPTH_INFINITE);
    free(remote);
}

/* Like resolve_path except more intelligent. */
static char *clever_path(const char *p, const char *src, 
			 const char *dest)
{
    char *ret;
    int src_is_coll, dest_is_coll;
    dest_is_coll = (dest[strlen(dest)-1] == '/');
    src_is_coll = (src[strlen(src)-1] == '/');
    if (strcmp(dest, ".") == 0) {
	ret = resolve_path(p, base_name(src), false);
    } else if (strcmp(dest, "..") == 0) {
	char *parent;
	parent = uri_parent(p);
	ret = resolve_path(parent, base_name(src), false);
	free(parent);
    } else if (!src_is_coll && dest_is_coll) {
	/* Moving a file to a collection... the destination should
	 * be the basename of file concated with the collection. */
	char *tmp;
	tmp = resolve_path(p, dest, true);
	CONCAT2(ret, tmp, base_name(src));
	free(tmp);
    } else {
	ret = resolve_path(p, dest, false);
    }
    return ret;
}

static void cat_callback(void *userdata, const char *buf, size_t len)
{
    FILE *f = userdata;
    if (fwrite(buf, len, 1, f) < 0) {
	perror("fwrite");
    }
}

static const char *choose_pager(void)
{
    struct stat st;
    char *tmp;
    /* TODO: add an autoconf for less/more? */
    tmp = getenv("PAGER");
    if (tmp != NULL) {
	return tmp;
    } else if (stat("/usr/bin/less", &st) == 0) {
	return "/usr/bin/less";
    } else if (stat("/bin/less", &st) == 0) {
	return "/bin/less";
    } else {
	return "/bin/more";
    }
}

static FILE *spawn_pager(const char *pager)
{
    /* Create a pipe */
    return popen(pager, "w");
}

static void kill_pager(FILE *p)
{
    /* This blocks until the pager quits. */
    pclose(p);
}

void execute_less(const char *resource)
{
    char *real_res;
    const char *pager;
    FILE *p;
    real_res = resolve_path(path, resource, false);
    pager = choose_pager();
    printf(_("Displaying `%s':\n"), real_res);
    p = spawn_pager(pager);
    if (p == NULL) {
	printf(_("Error! Could not spawn pager `%s':\n%s\n"), pager,
		 strerror(errno));
    } else {
	child_running = true;
	ne_read_file(session, real_res, cat_callback, p);
	kill_pager(p); /* Blocks until the pager quits */
	child_running = false;
    }
}

void execute_cat(const char *resource)
{
    char *real_res = resolve_path(path, resource, false);
    printf(_("Displaying `%s':\n"), real_res);
    if (ne_read_file(session, real_res, cat_callback, stdout) != NE_OK) {
	printf(_("Failed: %s\n"), ne_get_error(session));
    }
}

/* Returns non-zero if given resource is not a collection resource.
 * This function MAY make a request to the server. */
int is_collection(const char *uri)
{
    struct resource *res = NULL;
    int ret = 0;
    /* TODO: just request resourcetype here. */
    ret = fetch_resource_list(session, uri, NE_DEPTH_ZERO, 1, &res);
    if (ret == NE_OK) {
	if (res != NULL && uri_compare(uri, res->uri) == 0) {
	    if (res->type == resr_collection) {
		ret = 1;
	    } else {
		ne_set_error(session, _("Not a collection resource."));
	    }
	} else {
	    /* FIXME: this error occurs when you do open /foo and get
	     * the response for /foo/ back */
	    ne_set_error(session, _("Did not find a collection resource."));
	}
    } else {
	ret = 0;
    }
    free_resource_list(res);
    return ret;
}

void multi_copy(int argc, const char *argv[])
{
    do_copymove(argc, argv, _("copying"), _("copy"), simple_copy);
}

void multi_move(int argc, const char *argv[])
{
    do_copymove(argc, argv, _("moving"), _("move"), simple_move);
}

static void do_copymove(int argc, const char *argv[],
			const char *v1, const char *v2,
			void (*cb)(const char *, const char *))
{
    /* We are guaranteed that argc > 2... */
    char *dest;

    dest = resolve_path(path, argv[argc-1], true);
    if (is_collection(dest)) {
	int n;
	char *real_src, *real_dest;
	for(n = 0; n < argc-1; n++) {
	    real_src = resolve_path(path, argv[n], false);
	    real_dest = clever_path(path, argv[n], dest);
	    if (strcmp(real_src, real_dest) == 0) {
		printf("%s: %s and %s are the same resource.\n", v2,
			real_src, real_dest);
	    } else {
		(*cb)(real_src, real_dest);
	    }
	    free(real_src);
	    free(real_dest);
	}
    } else if (argc > 2) {
	printf(
	    _("When %s multiple resources, the last argument must be a collection.\n"), v1);
    } else {
	char *rsrc, *rdest;
	rsrc = resolve_path(path, argv[0], false);
	rdest = resolve_path(path, argv[1], false);
	/* Simple */
	(*cb) (rsrc, rdest);
	free(rsrc);
	free(rdest);
    }
    free(dest);
}

static void simple_move(const char *src, const char *dest) 
{
    output(o_start, _("Moving `%s' to `%s': "), src, dest);
    out_result(ne_move(session, 1, src, dest));
}

static void simple_copy(const char *src, const char *dest) 
{
    output(o_start, _("Copying `%s' to `%s': "), src, dest);
    out_result(ne_copy(session, 1, NE_DEPTH_INFINITE, src, dest));
}

/* TODO: can do utf-8 handling here. */
static char *escape_path(const char *p)
{
    return uri_abspath_escape(p);
}

/* Returns absolute filename which is 'filename' relative to 
 * (absolute filename) 'path'. e.g.
 *    resolve_path("/dav/", "asda") == "/dav/asda"
 *    resolve_path("/dav/", "/asda") == "/asda"
 * Also removes '..' segments, e.g.
 *    resolve_path("/dav/foobar/", "../fish") == "/dav/fish"
 * If isdir is true, ensures the return value has a trailing slash.
 */
char *resolve_path(const char *p, const char *filename, int isdir)
{
    char *ret, *pnt;
    if (*filename == '/') {
	/* It's absolute */
	ret = escape_path(filename);
    } else if (strcmp(filename, ".") == 0) {
	ret = escape_path(p);
    } else {
	pnt = escape_path(filename);
	CONCAT2(ret, p, pnt);
	free(pnt);
    }
    if (isdir && ret[strlen(ret)-1] != '/') {
	char *newret;
	CONCAT2(newret, ret, "/");
	free(ret);
	ret = newret;
    }
    /* Sort out '..', etc... */
    do {
	pnt = strstr(ret, "/../");
	if (pnt != NULL) {
	    char *last;
	    /* Find the *previous* path segment, to overwrite...
	     *    /foo/../
	     *    ^- last points here */
	    if (pnt > ret) {
		for(last = pnt-1; (last > ret) && (*last != '/'); last--);
	    } else {
		last = ret;
	    }
	    memmove(last, pnt + 3, strlen(pnt+2));
	} else {
	    pnt = strstr(ret, "/./");
	    if (pnt != NULL) {
		memmove(pnt, pnt+2, strlen(pnt+1));
	    }
	}
    } while (pnt != NULL);
    return ret;    
}

void execute_get(const char *remote, const char *local)
{
    char *filename, *real_remote;
    real_remote = resolve_path(path, remote, false);
    if (local == NULL) {
	struct stat st;
	/* Choose an appropriate local filename */
	if (stat(base_name(remote), &st) == 0) {
	    /* File already exists... don't overwrite */
	    printf(_("Enter local filename for `%s': "), real_remote);
	    filename = readline(NULL);
	    if (filename == NULL) {
		free(real_remote);
		printf(_("cancelled.\n"));
		return;
	    }
	} else {
	    filename = ne_strdup(base_name(remote));
	}
    } else {
	filename = ne_strdup(local);
    }
    {
	int fd = open(filename, O_CREAT|O_WRONLY);
	output(o_transfer, _("Downloading `%s' to %s:"), real_remote, filename);
	if (fd < 0) {
	    output(o_finish, _("failed:\n%s\n"), strerror(errno));
	} else {
	    out_result(ne_get(session, real_remote, fd));
	    /* FIXME */ close(fd);
	}
    }
    free(real_remote);
    free(filename);
}

static int run_editor(const char *filename)
{
    char editcmd[BUFSIZ];
    const char *editor;
    struct stat before_st, after_st;
    editor = get_option(opt_editor);
    if (editor == NULL) {
	editor = getenv("EDITOR");
	if (editor == NULL) {
	    editor = "vi";
	}
    }
    snprintf(editcmd, BUFSIZ, "%s %s", editor, filename);
    if (stat(filename, &before_st)) {
	printf(_("Could not stat file: %s\n"), strerror(errno));
	return -1;
    }
    printf("Running editor: `%s'...\n", editcmd);
    system(editcmd);
    if (stat(filename, &after_st)) {
	printf(_("Error! Could not examine temporary file: %s\n"), 
		 strerror(errno));
	return -1;
    }
    if (before_st.st_mtime == after_st.st_mtime) {
	/* File not changed. */
	printf(_("No changes were made.\n"));
	return -1;
    } else {
	printf(_("Changes were made.\n"));
	return 0;
    }	
}

/* Returns true if resource at URI is lockable. */
static int is_lockable(const char *uri)
{
    ne_server_capabilities caps = {0};

    /* TODO: for the mo, just check we're on a Class 2 server.
     * Should check supportedlock property really. */

    if (ne_options(session, uri, &caps) != NE_OK) {
	return 0;
    }

    return caps.dav_class2;
}


/* TODO: this is great big heap of steaming trout.  */
/* TODO: does this work under cygwin? mkstemp() may or may not open
   the file using O_BINARY, and then we *do* upload it using O_BINARY,
   so maybe this will screw things up. Maybe we should fcntl it and
   set O_BINARY, if that is allowed under cygwin? */
void execute_edit(const char *remote)
{
    char *real_remote;
    unsigned int can_lock; /* can we LOCK it? */
    struct ne_lock *lock = NULL;
    char fname[] = "/tmp/cadaver-edit-XXXXXX";
    int fd;
    
    real_remote = resolve_path(path, remote, false);

    /* Don't let them edit a collection, since PUT to a collection is
     * bogus. Really we want to be able to fetch a "DefaultDocument"
     * property, and edit on that instead: IIS does expose such a
     * property. Would be a nice hack to add the support to mod_dav
     * too. */
    if (is_collection(real_remote)) {
	printf(_("You cannot edit a collection resource (%s).\n"),
	       real_remote);
	goto edit_bail;
    }

    can_lock = is_lockable(real_remote);

    fd = mkstemp(fname);
    if (fd == -1) {
	printf(_("Could not create temporary file %s:\n%s\n"), fname,
	       strerror(errno));
	goto edit_bail;
    }

    /* Sanity check on the file perms. */
    if (fchmod(fd, 0600) == -1) {
	printf(_("Could not set file permissions for %s:\n%s\n"), fname,
	       strerror(errno));
	goto edit_bail;
    }
   
    if (can_lock) {
	/* FIXME: copy'n'friggin'paste */
	lock = ne_calloc(sizeof *lock);
	lock->type = ne_locktype_write;
	lock->scope = ne_lockscope_exclusive;
	lock->owner = getowner();
	lock->depth = 0;
	lock->uri = ne_strdup(real_remote);
	out_start("Locking", remote);
	if (out_handle(ne_lock(session, lock))) {
	    ne_lock_add(lock_session, lock);
	} else {
	    ne_lock_free(lock);
	    goto edit_close;
	    return;
	}
    } else {
	/* TODO: HEAD and get the Etag/modtime */
    }
    /* FIXME: copy'n'friggin'paste. */
    output(o_transfer, _("Downloading `%s' to %s"), real_remote, fname);

    /* Don't puke if get fails -- perhaps we are creating a new one? */
    out_result(ne_get(session, real_remote, fd));
    
    if (close(fd)) {
	output(o_finish, _("Error writing to temporary file: %s\n"), 
	       strerror(errno));
    } 
    else if (!run_editor(fname)) {
	int upload_okay = 0;

	fd = open(fname, O_RDONLY | OPEN_BINARY_FLAGS);
	if (fd < 0) {
	    output(o_finish, 
		   _("Could not re-open temporary file: %s\n"),
		   strerror(errno));
	} else {
	    do {
		output(o_transfer, _("Uploading changes to `%s'"), 
		       real_remote);
		/* FIXME: conditional PUT using fetched Etag/modtime if
		 * !can_lock */
		if (out_handle(ne_put(session, real_remote, fd))) {
		    upload_okay = 1;
		} else {
		    /* TODO: offer to save locally instead */
		    printf(_("Try uploading again (y/n)? "));
		    if (!yesno()) {
			upload_okay = 1;
		    }
		}
	    } while (!upload_okay);
	    close(fd);
	}
    }
    
    if (unlink(fname)) {
	printf(_("Could not delete temporary file %s:\n%s\n"), fname,
	       strerror(errno));
    }	       

    /* UNLOCK it again whether we succeed or failed in our mission */
    if (can_lock) {
	output(o_start, "Unlocking `%s':", remote);
	out_result(ne_unlock(session,lock));
	ne_lock_remove(lock_session, lock);
	ne_lock_free(lock);
    }

    goto edit_bail;
edit_close:
    close(fd);
edit_bail:
    free(real_remote);
    return;
}

static void simple_put(const char *local, const char *remote)
{
    int fd = open(local, O_RDONLY | OPEN_BINARY_FLAGS);
    output(o_transfer, _("Uploading %s to `%s':"), local, remote);
    if (fd < 0) {
	output(o_finish, _("Could not open file: %s\n"), strerror(errno));
    } else {
	out_result(ne_put(session, remote, fd));
	(void) close(fd);
    }
}

void execute_put(const char *local, const char *remote)
{
    char *real_remote;
    if (remote == NULL) {
	real_remote = resolve_path(path, base_name(local), false);
    } else {
	real_remote = resolve_path(path, remote, false);
    }
    simple_put(local, real_remote);
    free(real_remote);
}

/* this is getting too easy */
void multi_mput(int argc, const char *argv[])
{
    for(; argv[0] != NULL; argv++) {
	char *remote;
	remote = clever_path(path, argv[0], path);
	simple_put(argv[0], remote);
	free(remote);
    }    
}

void multi_mget(int argc, const char *argv[])
{
    for(; argv[0] != NULL; argv++) {
	execute_get(argv[0], NULL);
    }
}

void execute_chexec(const char *val, const char *remote)
{
    static const ne_propname execprop = 
    { "http://apache.org/dav/props/", "executable" };
    /* Use a single operation; set the executable property to... */
    ne_proppatch_operation ops[] = { { &execprop, ne_propset, NULL }, { NULL } };
    char *uri;
    ne_server_capabilities caps = {0};
    int ret;

    /* True or false, depending... */
    if (strcmp(val, "+") == 0) {
	ops[0].value = "T";
    } else if (strcmp(val, "-") == 0) {
	ops[0].value = "F";
    } else {
	printf(_("Use:\n"
		 "   chexec + %s   to make the resource executable\n"
		 "or chexec - %s   to make the resource unexecutable\n"),
		 remote, remote);
	return;
    }
    
    uri = resolve_path(path, remote, false);

    out_start(_("Setting isexecutable"), remote);
    ret = ne_options(session, uri, &caps);
    if (ret != NE_OK) {
	out_result(ret);
    } else if (!caps.dav_executable) {
	ne_set_error(session, 
		       _("The server does not support the 'isexecutable' property."));
	out_result(NE_ERROR);
    } else {    
	out_result(ne_proppatch(session, uri, ops));
    }
    free(uri);

}

void execute_lpwd(void)
{
    char pwd[BUFSIZ];
    if (getcwd(pwd, BUFSIZ) == NULL) {
	perror("pwd");
    } else {
	printf(_("Local directory: %s\n"), pwd);
    }
}

/* Using a hack, we get zero-effort multiple-argument 'lls' */
void execute_lls(int argc, char * const argv[])
{
    int pid;
    pid = fork();
    switch (pid) {
    case 0: /* child...  */
	/* The implicit typecast munging might screw things up in
	 * weird and wonderful ways, here? */
	execvp("ls", &argv[-1]);
	printf(_("Error executing ls: %s\n"), strerror(errno));
	exit(-1);
	break;
    case -1: /* Error */
	printf(_("Error forking ls: %s\n"), strerror(errno));
	break;
    default: /* Parent */
	wait(NULL);
	break;
    }
    return;
}

void execute_lcd(const char *p)
{
    const char *real_path;
    if (p) {
	real_path = p;
    } else {
	real_path = getenv("HOME");
	if (!real_path) {
	    printf(_("Could not determine home directory from environment.\n"));
	    return;
	}
    }
    if (chdir(real_path)) {
	printf(_("Could not change local directory:\nchdir: %s\n"),
	       strerror(errno));
    }
}

void execute_pwd(void)
{
    printf(_("Current collection is `%s', on server `%s'\n"), path, 
	    server_hostname);
}

void execute_cd(const char *newpath)
{
    char *real_path;
    int is_swap = false;
    if (strcmp(newpath, "-") == 0) {
	is_swap = true;
	real_path = old_path;
    } else if (path == NULL) {
	/* The first time we are executing a CD. */
	if (1 || uri_has_trailing_slash(newpath)) {
	    real_path = ne_strdup(newpath);
	} else {
	    CONCAT2(real_path, newpath, "/");
	}
    } else {
	real_path = resolve_path(path, newpath, true);
    }
    if (set_path(real_path)) {
	/* Error */
	if (!is_swap) free(real_path);
    } else {
	/* Success */
	if (!is_swap && old_path) free(old_path);
	old_path = path;
	path = real_path;
    }
}

static void display_help_message(void)
{
    int n;
    printf("Commands: \n");
    
    for(n = 0; commands[n].id != cmd_unknown; n++) {
	if (commands[n].call && commands[n].short_help) {
	    printf(" %-26s %s\n", commands[n].call, _(commands[n].short_help));
	}
    }

    printf(_("Aliases: rm=delete, mkdir=mkcol, mv=move, cp=copy, "
	     "more=less, quit=exit=bye\n"));
}

void execute_help(const char *arg)
{
    if (!arg) {
	display_help_message();
    } else {
	const struct command *cmd = get_command(arg);
	if (cmd) {
	    printf(" `%s'   %s\n", cmd->call, cmd->short_help);
	    if (cmd->needs_connection) {
		printf(_("This command can only be used when connected to a server.\n"));
	    }
	} else {
	    printf(_("Command name not known: %s\n"), arg);
	}
    }
}

void execute_echo(int count, const char **args)
{
    const char **pnt;
    for(pnt = args; *pnt != NULL; pnt++) {
	printf("%s ", *pnt);
    }
    putchar('\n');
}

