# preferences.tcl --
#
# This file controls the preferences window. There is just one preferences
# window, if the user requests a new preferences window when the window
# already exists, it is just unmapped and raised (if possible).
#
#
#  TkRat software and its included text is Copyright 1996-2000 by
#  Martin Forssn
#
#  The full text of the legal notice is contained in the file called
#  COPYRIGHT, included with this distribution.

# The last wisited window
set lastPref general

# Preferences --
#
# Make sure the user sees an preferences window.
#
# Arguments:

proc Preferences {} {
    if ![winfo exists .pref] {
	BuildPreferences
    } else {
	wm deiconify .pref
    }
}


# BuildPreferences --
#
# Builds the preferences window.
#
# Arguments:

proc BuildPreferences {} {
    global t b pref

    # Initialize data table
    foreach l [GetLanguages] {
	lappend lang [lrange $l 0 1]
    }

    # variable label type type_args
    #		      entry unit_label
    #		      bool true_label false_label
    #                 option options
    #		      special procedure
    set pref(general,misc) "\
	{language language option [list $lang]} \
	{charset charset entry} \
	{main_window_name window_name entry} \
	{icon_name icon_name entry} \
	{icon icon_bitmap option \
		{{normal {$t(normal_bitmap)}} \
		 {small {$t(small_bitmap)}} \
		 {none {$t(none)}}}} \
	{iconic startup_mode bool \
		{{$t(iconic)} {$t(normal)}}} \
	{info_changes show_changes bool \
		{{$t(show)} {$t(dont_show)}}} \
	{mail_steal check_stolen_mail bool \
		{{$t(check)} {$t(dont_check)}}} \
	{print_header print_headers option \
		{{all {$t(all)}} {selected {$t(selected)}} {none {$t(none)}}}} \
	{expunge_on_close expunge_on_close bool \
		{{$t(do)} {$t(do_not)}}} \
	{useinputmethods useinputmethods bool \
		{{$t(true)} {$t(false)}}} \
    "
    set pref(general,appearance) "\
	{override_color override_color bool \
		{{$t(true)} {$t(false)}}} \
	{color_set color_scheme special SetupColor} \
	{override_fonts override_fonts bool \
		{{$t(true)} {$t(false)}}} \
	{prop_norm prop_norm special {SelectFont prop_norm}} \
	{prop_light prop_light special {SelectFont prop_light}} \
	{fixed_norm fixed_norm special {SelectFont fixed_norm}} \
	{fixed_bold fixed_bold special {SelectFont fixed_bold}} \
	{watcher_font watcher_font special {SelectFont watcher_font}} \
	{keep_pos remember_pos bool \
		{{$t(do_remember)} {$t(dont_remember)}}} \
	{tearoff menu_tearoff bool \
		{{$t(true)} {$t(false)}}} \
    "
    set pref(files) "\
	{tmp tmp_dir entry} \
	{permissions file_permissions entry} \
	{userproc userproc_file entry} \
	{hold_dir hold_dir entry} \
	{mailcap_path mailcap_path entry} \
	{print_command print_command entry} \
	{terminal terminal_command entry} \
	{ispell_path ispell_cmd entry} \
	{ispell_dictionary ispell_wordlist entry} \
    "
    set pref(folder,window) "\
	{log_timeout log_timeout entry_unit {$t(seconds)}} \
	{cache_passwd cache_passwd bool \
		{{$t(do_cache)} {$t(do_not_cache)}}} \
	{cache_passwd_timeout cache_timeout entry_unit {$t(seconds)}} \
	{cache_conn cache_conn bool \
		{{$t(do_cache)} {$t(do_not_cache)}}} \
	{cache_conn_timeout cache_timeout entry_unit {$t(seconds)}} \
	{checkpoint_on_unmap checkpoint_on_unmap bool
		{{$t(true)} {$t(false)}}} \
	{checkpoint_interval checkpoint_interval entry_unit {$t(seconds)}} \
    "
    set pref(folder,msglist) "\
	{list_format list_format entry} \
	{show_header show_headers option \
		{{all {$t(show_all_headers)}} \
		 {selected {$t(show_selected_headers)}} \
		 {none {$t(show_no_headers)}}}} \
	{show_header_selection show_header_selection entry} \
	{browse default_browse_mode option \
		{{normal {$t(no_browse)}} \
		 {browse {$t(do_browse)}} \
		 {folder {$t(use_folder_default)}}}} \
	{folder_sort sort_order option \
		{{threaded {$t(sort_threaded)}} \
		 {subject {$t(sort_subject)}} \
		 {subjectonly {$t(sort_subjectonly)}} \
		 {sender {$t(sort_sender)}} \
		 {folder {$t(sort_folder)}} \
		 {reverseFolder {$t(sort_reverseFolder)}} \
		 {date {$t(sort_date)}} \
		 {reverseDate {$t(sort_reverseDate)}} \
		 {size {$t(sort_size)}}\
		 {reverseSize {$t(sort_reverseSize)}}}} \
	{start_selection start_selection option \
		{{first {$t(first_message)}} \
		 {last {$t(last_message)}} \
		 {first_new {$t(first_new_message)}} \
		 {before_new {$t(before_first_new_message)}}}} \
    "
    set pref(folder,folder_dynamic) "\
	{dynamic_behaviour dynamic_behaviour option \
		{{expanded {$t(dyn_expanded)}} \
		 {closed {$t(dyn_closed)}}}} \
    "
    set pref(folder,url) "\
	{urlprot url_protocols entry} \
	{url_color url_color entry} \
	{url_viewer url_viewer option \
		{{RatUP {$t(userproc): RatUP_ShowURL}} \
		 {netscape Netscape} \
		 {lynx Lynx} {other $t(other)}}} \
	{netscape netscape_cmd entry} \
	{lynx lynx_cmd entry} \
	{other_browser other_browser_cmd entry} \
    "
    set pref(folder,defaults) "\
	{remote_user remote_user entry} \
	{remote_host remote_host entry} \
	{imap_port imap_port entry} \
	{pop3_port pop3_port entry} \
	{debug_file debug_file entry} \
	{debug_cclient debug_cclient bool \
		{{$t(true)} {$t(false)}}} \
    "
    set pref(dbase) "\
	{dbase_dir dbase_dir entry} \
	{def_extype extype option \
		{{none {$t(none)}} \
		 {remove {$t(remove)}} \
		 {incoming {$t(incoming)}} \
		 {backup {$t(backup)}}}} \
	{def_exdate exdate entry_unit {$t(days)}} \
	{dbase_backup dbase_backup entry} \
	{chunksize chunksize entry} \
	{expire_interval expire_interval entry_unit {$t(days)}} \
    "
    set pref(watcher) "\
	{watcher_enable watcher_enable bool \
		{{$t(enabled)} {$t(disabled)}}} \
	{watcher_time intervals special SetupWatcherTime} \
	{watcher_name window_name entry} \
	{watcher_max_height max_height entry_unit {$t(lines)}} \
	{watcher_show show_messages option \
		{{new {$t(new)}} {all {$t(all)}}}} \
	{watcher_format list_format entry} \
	{watcher_bell bell_ringings entry} \
    "
    set pref(compose,basic) "\
	{compose_headers headers entry} \
	{wrap_length wrap_length entry} \
	{signature signature_file entry} \
	{sigdelimit sigdelimit bool \
		{{$t(true)} {$t(false)}}} \
	{lookup_name lookup_name bool \
		{{$t(do_lookup)} {$t(dont_lookup)}}} \
	{copy_attached copy_attached_files bool \
		{{$t(true)} {$t(false)}}} \
	{alias_expand alias_expansion option \
		{{0 {$t(alias_0)}} {1 {$t(alias_1)}} {2 {$t(alias_2)}}}} \
	{always_editor always_use_external_editor bool \
		{{$t(true)} {$t(false)}}} \
    "
    set pref(compose,replies) "\
	{re_regexp reply_regexp entry} \
	{attribution attribution entry} \
	{forwarded_message forwarded_label entry} \
	{no_subject no_subject entry} \
	{reply_lead reply_lead entry} \
	{citexp citexp entry} \
	{wrap_cited wrap_cited bool \
		{{$t(true)} {$t(false)}}} \
	{skip_sig on_reply bool \
		{{$t(skip_sig)} {$t(keep_sig)}}} \
        {append_sig append_sig bool \
                {{$t(true)} {$t(false)}}} \
        {reply_bottom reply_bottom bool \
                {{$t(at_bottom)} {$t(at_top)}}} \
    "
    set pref(sending,address) "\
	{masquerade_as masquerade_host_as entry} \
	{domain domain entry} \
	{from use_from_address entry} \
	{bcc default_bcc entry} \
	{reply_to default_reply_to entry} \
	{return_path return_path entry} \
	{create_sender create_sender bool \
		{{$t(true)} {$t(false)}}} \
	{tip tip special ShowTip} \
    "
    set pref(sending,method) "\
	{delivery_mode delivery_mode option \
		{{direct {$t(direct)}} {deferred {$t(deferred)}}}} \
	{sendprot sendprot option \
		{{smtp {$t(smtp)}} {prog {$t(user_program)}}}} \
	{smtp_hosts smtp_hosts entry} \
	{smtp_timeout smtp_timeout entry_unit {$t(seconds)}} \
	{smtp_reuse smtp_reuse bool \
		{{$t(true)} {$t(false)}}} \
	{smtp_verbose smtpv option \
		{{0 {$t(none)}} \
		 {1 {$t(terse)}} \
		 {2 {$t(normal)}} \
		 {3 {$t(verbose)}}}} \
	{sendprog sendprog entry} \
	{sendprog_8bit progin bool \
		{{$t(eightbit)} {$t(sevenbit)}}} \
	{force_send force_send bool \
		{{$t(force)} {$t(no_force)}}} \
    "
    set pref(notification) "\
	{dsn_directory dsn_directory entry} \
	{dsn_request default_action special SetupDSNRequest} \
	{dsn_snarf_reports folderwindow bool \
		{{$t(snarf_dsn)} {$t(not_snarf_dsn)}}} \
	{dsn_expiration dsn_expiration entry_unit {$t(days)}} \
	{dsn_verbose report_level special SetupDSNVerbose} \
    "
    set pref(pgp) "\
	{pgp_version pgp_version option \
		{{0 $t(none)} {gpg-1 {GPG-1}} {2 {PGP-2}} \
		 {5 {PGP-5}} {6 {PGP-6}} {auto {$t(auto)}}}} \
	{pgp_path pgp_path entry} \
	{pgp_args pgp_extra_args entry} \
	{pgp_keyring pgp_keyring entry} \
	{cache_pgp cache_passwd bool \
		{{$t(do_cache)} {$t(do_not_cache)}}} \
	{cache_pgp_timeout cache_timeout entry_unit {$t(seconds)}} \
	{pgp_sign sign_outgoing bool \
		{{$t(true)} {$t(false)}}} \
	{pgp_encrypt encrypt_outgoing bool \
		{{$t(true)} {$t(false)}}} \
    "

    # Create toplevel
    set w .pref
    toplevel $w -class TkRat
    wm title $w $t(preferences)

    # Setup panes
    rat_tabbed::create $w.t
    foreach n {{general {misc appearance}}
	       files
	       {folder {window msglist url folder_dynamic defaults}}
	       dbase watcher {compose {basic replies}} 
	       {sending {address method}} notification pgp} {
	set n0 [lindex $n 0]
	set name $t($n0)
	if {1 != [llength $n]} {
	    set p [rat_tabbed::page $w.t $name \
		    "set lastPref [lindex $n 0],[lindex [lindex $n 1] 0]"]
	    rat_tabbed::create $p.f
	    foreach n1 [lindex $n 1] {
		set name $t($n1)
		set sp [rat_tabbed::page $p.f $name \
			"set lastPref [lindex $n 0],$n1"]
		PrefBuild $n0,$n1 $sp
	    }
	    pack $p.f -side top
	} else {
	    set p [rat_tabbed::page $w.t $name "set lastPref [lindex $n 0]"]
	    PrefBuild $n $p
	}
    }

    # The buttons
    frame $w.buttons
    button $w.buttons.ok -text $t(apply) -command "PrefApply $w" -default active
    button $w.buttons.reset -text $t(reset) \
	    -command {
			global pref lastPref
			foreach v $pref(vars,$lastPref) {
			    set pref(opt,$v) $pref(old,$v)
			}
		     }
    button $w.buttons.dismiss -text $t(dismiss) \
	    -command "PrefCheck $w; \
		      RecordPos $w preferences;
		      wm withdraw $w"
    pack $w.buttons.ok \
	 $w.buttons.reset \
         $w.buttons.dismiss -side left -expand 1
    bind $w <Return> "PrefApply $w"
    set b($w.buttons.ok) apply_prefs
    set b($w.buttons.reset) reset_changes
    set b($w.buttons.dismiss) dismiss

    pack $w.t -side top -fill x
    pack $w.buttons -side bottom -fill x -pady 5

    Place $w preferences
}

# PrefBuild --
#
# Build a preferences pane
#
# Arguments:
# pane  - Which pane of preferences to build
# w	- Name of frame to build it into

proc PrefBuild {pane w} {
    global t b option pref

    set row 0
    set pref(vars,$pane) {}
    foreach p $pref($pane) {
	set var [lindex $p 0]
	lappend pref(vars,$pane) $var
	set pref(opt,$var) $option($var)

	label $w.r${row}_lab -text $t([lindex $p 1]):
	grid $w.r${row}_lab -row $row -sticky ne -pady 5
	switch [lindex $p 2] {
	    entry {
		    entry $w.r${row}_item -textvariable pref(opt,$var) \
			    -width 60
		    grid $w.r${row}_item - -row $row -column 1 -sticky we
		    set b($w.r${row}_item) pref_$var
		}
	    entry_unit {
		    entry $w.r${row}_item -textvariable pref(opt,$var)
		    label $w.r${row}_unit -text ([lindex $p 3])
		    grid $w.r${row}_item -row $row -column 1 -sticky we
		    grid $w.r${row}_unit -row $row -column 2 -sticky we
		    set b($w.r${row}_item) pref_$var
		    set b($w.r${row}_unit) unit_pref
		}
	    bool {
		    if {$pref(opt,$var)} {
			set pref(opt,$var) 1
		    } else {
			set pref(opt,$var) 0
		    }
		    set v [lindex $p 3]
		    set v [list [list 1 [lindex $v 0]] [list 0 [lindex $v 1]]]
    		    OptionMenu $w.r${row}_item pref(opt,$var) pref(text,$var) $v
		    grid $w.r${row}_item - -row $row -column 1 -sticky w
		    set b($w.r${row}_item) pref_$var
		}
	    option {
    		    OptionMenu $w.r${row}_item pref(opt,$var) \
			    pref(text,$var) [lindex $p 3]
		    grid $w.r${row}_item - -row $row -column 1 -sticky w
		    set b($w.r${row}_item) pref_$var
		}
	    special {
		    eval "[lindex $p 3] $w.r${row}_item"
		    grid $w.r${row}_item - -row $row -column 1 -sticky we
		}
	    default {puts "Internal error <$p>"}
	}
	set pref(old,$var) $pref(opt,$var)
	if [RatIsLocked option([lindex $p 0])] {
	    $w.r${row}_item configure -state disabled -relief flat
	}

	incr row
    }
    grid columnconfigure $w 1 -weight 1
}

# PrefApply --
#
# Applies any changes to the preferences made in the current window.
#
# Arguments:
# parent -	Parent window

proc PrefApply {parent} {
    global option pref t folderWindowList lastPref tk_version

    switch $lastPref {
    general,misc {
	    if {[string compare $option(useinputmethods) \
		    $pref(opt,useinputmethods)] && 8.3 <= $tk_version} {
		tk useinputmethods $pref(opt,useinputmethods)
	    }
	}
    files {
	    if {![regexp %p $pref(opt,print_command)]} {
		Popup $t(no_pp_in_print_command) $parent
	    }
	    if [string compare $option(mailcap_path) $pref(opt,mailcap_path)] {
		set option(mailcap_path) $pref(opt,mailcap_path)
		RatMailcapReload
	    }
	}
    watcher {
	    set pref(opt,watcher_time) [list [list std $pref(opt,finterval)] \
					     [list dbase $pref(opt,dinterval)]]
	}
    notification {
	    set pref(opt,dsn_verbose) [list [list failed $pref(opt,failed)] \
					[list delayed $pref(opt,delayed)] \
					[list delivered $pref(opt,delivered)] \
					[list relayed $pref(opt,relayed)] \
					[list expanded $pref(opt,expanded)]]
	}
    compose,replies {
	    if [catch {regexp -nocase $pref(opt,re_regexp) ""} e] {
		Popup "$t(re_regexp_error): $e" $parent
		return
	    }
	    if [catch {regexp -nocase $pref(opt,citexp) ""} e] {
		Popup [format $t(illegal_regexp): $e] $parent
		return
	    }
	}
    compose,basic {
	    if {[string compare $option(signature) $pref(opt,signature)]
		    && 1 == [llength [info commands RatUP_Signature]]} {
		Popup $t(sig_cmd_takes_precedence) $parent
	    }
	}
    sending,method {
	    if {[string compare $option(sendprog) $pref(opt,sendprog)]
		    && ![file executable [lindex $pref(opt,sendprog) 0]]} {
		Popup $t(warning_sendprog) $parent
	    }
	}
    folder,window {
	    if [string compare $option(icon) $pref(opt,icon)] {
		SetIcon . $pref(opt,icon)
	    }
	}
    folder,url {
	    if [string compare $option(url_color) $pref(opt,url_color)] {
		foreach fw [array names folderWindowList] {
		    upvar #0 $fw fh
		    $fh(text) tag configure URL -foreground $pref(opt,url_color)
		}
	    }
	}
    }
    set hasChanged 0
    set needRestart 0
    set newSendArgs 0
    foreach prefs [array names pref opt,*] {
	set opt [string range $prefs 4 end]
	if ![info exists option($opt)] {
	    continue
	}
	if [string compare $option($opt) $pref(opt,$opt)] {
	    set option($opt) $pref(opt,$opt)
	    set hasChanged 1
	    if { -1 != [lsearch -exact {language charset fontsize
		    			main_window_name icon_name
					default_folder watcher_name pgp_enable
					override_fonts prop_norm prop_light
					fixed_norm fixed_bold watcher_font}\
				$opt]} {
		set needRestart 1
	    }
	    if { -1 != [lsearch -exact {charset masquerade_as domain sendprot
					smtp_hosts sendprog sendprog_8bit
					charset from smtp_verbose
					delivery_mode smtp_timeout force_send
					pgp_version pgp_path pgp_args
					pgp_keyring} $opt]} {
		set newSendArgs 1
	    }
	}
    }

    if $hasChanged {
	switch -glob $lastPref {
	folder,* {
		foreach f [array names folderWindowList] {
		    Sync $f update
		}
	    }
	pgp {
		InitPgp
	    }
	}

	SaveOptions
    }
    if $needRestart {
	Popup $t(need_restart) $parent
    }
    if $newSendArgs {
	RatSend kill
    }
}


# PrefCheck --
#
# Checks if there are any unapplied changes and if there is the user is
# queried if he wants to apply them.
#
# Arguments:
# parent - Parent of window

proc PrefCheck {parent} {
    global option pref t lastPref

    switch $lastPref {
    watcher {
	    set pref(opt,watcher_time) [list [list std $pref(opt,finterval)] \
					     [list dbase $pref(opt,dinterval)]]
	}
    notification {
	    set pref(opt,dsn_verbose) [list [list failed $pref(opt,failed)] \
					[list delayed $pref(opt,delayed)] \
					[list delivered $pref(opt,delivered)] \
					[list relayed $pref(opt,relayed)] \
					[list expanded $pref(opt,expanded)]]
	}
    }
    foreach prefs [array names pref opt,*] {
	set opt [string range $prefs 4 end]
	if ![info exists option($opt)] {
	    continue
	}
	if [string compare $option($opt) $pref(opt,$opt)] {
	    set value [RatDialog $parent $t(unapplied_changes_title) \
		    $t(unapplied_changes) {} 0 $t(apply) $t(discard)]
	    if { 0 == $value } {
		PrefApply $parent
	    }
	    return
	}
    }
}


# OptionMenu --
#
# Generates an option menu. The generated menu will have window name "w"
# and will set the "varName" variable. The different options are
# controlled by the value arguments. Each value argument is a list of
# two elemnts. The first is the value to set "varName" to and the second
# is the text to show. The menubutton will use "textVar" as the textvariable.
#
# Arguments:
# w	  -	Name of menubutton to create
# varName -	Variable to set to value
# textVar -	Variable to use for the text we show
# values  -	A list of lists which describes the values of this button

proc OptionMenu {w varName textVar values} {
    upvar #0 $varName var
    upvar #0 $textVar text

    set width 20
    menubutton $w -textvariable $textVar -indicatoron 1 \
		  -relief raised -menu $w.m
    menu $w.m -tearoff 0
    foreach elem $values {
	if ![string compare [lindex $elem 0] $var] {
	    set text [lindex $elem 1]
	}
	$w.m add command -label [lindex $elem 1] \
		-command "set $varName [list [lindex $elem 0]]"
	if { $width < [string length [lindex $elem 1]]} {
	    set width [string length [lindex $elem 1]]
	}
    }
    $w configure -width $width

    trace variable var w "PrefTraceProc $varName $textVar {$values}"
}

proc PrefTraceProc {varName textVar values args} {
    upvar #0 $varName var
    upvar #0 $textVar text

    foreach v $values {
	if ![string compare $var [lindex $v 0]] {
	    set text [lindex $v 1]
	    return
	}
    }

}


proc SetupColor {w} {
    global t option pref tk_version

    frame $w
    menubutton $w.mb -textvariable pref(text,color_set) \
	    -indicatoron 1 -relief raised -menu $w.mb.m
    menu $w.mb.m -tearoff 0
    set width 20
    foreach c { {gray85 black} {PeachPuff2 black} {bisque black}
		{SlateBlue1 black} {SteelBlue4 white} {SkyBlue1 black}
		{aquamarine2 black} {SpringGreen4 black}} {
	set name $t([lindex $c 0])
	if ![string compare $c $option(color_set)] {
	    set pref(text,color_set) $name
	}
	$w.mb.m add command -label $name \
		-command "set pref(opt,color_set) [list $c]; \
			  set pref(text,color_set) [list $name]; \
			  SetColor $c" \
		-background [lindex $c 0] -foreground [lindex $c 1]
	if { $width < [string length $name]} {
	    set width [string length $name]
	}
    }
    $w.mb configure -width $width
    pack $w.mb -side left
}

proc SetupWatcherTime {w} {
    global t option pref

    frame $w
    label $w.flabel -text $t(stdfolders):
    entry $w.fentry -textvariable pref(opt,finterval)
    label $w.funit -text ($t(seconds))
    if [RatIsLocked option(watcher_time)] {
	$w.fentry configure -state disabled -relief flat
    }
    label $w.dlabel -text $t(dbase_folders):
    entry $w.dentry -textvariable pref(opt,dinterval)
    label $w.dunit -text ($t(seconds))
    if [RatIsLocked option(watcher_time)] {
	$w.dentry configure -state disabled -relief flat
    }
    foreach elem $option(watcher_time) {
	if ![string compare [lindex $elem 0] std] {
	    set pref(opt,finterval) [lindex $elem 1]
	}
	if ![string compare [lindex $elem 0] dbase] {
	    set pref(opt,dinterval) [lindex $elem 1]
	}
    }
    grid $w.flabel -row 0 -column 0 -sticky e
    grid $w.fentry -row 0 -column 1 -sticky we
    grid $w.funit -row 0 -column 2 -sticky we
    grid $w.dlabel -row 1 -column 0 -sticky e
    grid $w.dentry -row 1 -column 1 -sticky we
    grid $w.dunit -row 1 -column 2 -sticky we
    grid columnconfigure $w 1 -weight 1
}

proc SetupDSNRequest {w} {
    global option t pref b
    frame $w

    OptionMenu $w.menu pref(opt,dsn_request) pref(text,dsn_request) \
	     [list [list 0 $t(not_request_dsn)]\
		   [list 1 $t(request_dsn)]]
    button $w.but -text $t(probe) -command PrefProbeDSN
    pack $w.menu -side left
    pack $w.but -side right -fill x
    set b($w.menu) pref_dsn_request
    set b($w.but) pref_dsn_probe
}

proc SetupDSNVerbose {w} {
    global option t pref b

    foreach elem $option(dsn_verbose) {
	set pref(opt,[lindex $elem 0]) [lindex $elem 1]
    }
    frame $w
    set irow 0
    foreach cat {failed delayed delivered relayed expanded} {
	set sf $w.$cat
	label ${sf}_l -text $t($cat): -width 12 -anchor e
	OptionMenu ${sf}_mbut pref(opt,$cat) pref(text,$cat) \
		[list [list none $t(rl_none)] \
		      [list status $t(rl_status)] \
		      [list notify $t(rl_notify)]]
	if [RatIsLocked option(dsn_verbose)] {
	    ${sf}_mbut configure -state disabled
	}
	grid ${sf}_l -row $irow -column 0 -sticky e
	grid ${sf}_mbut -row $irow -column 1 -sticky w
	incr irow
	set b(${sf}_mbut) pref_dsn_verbose
    }
    grid columnconfigure $w 2 -weight 1
}

# PrefProbeDSN --
#
# Probe the current SMTP servers for DSN support.
#
# Arguments:

proc PrefProbeDSN {} {
    global idCnt option t fixedNormFont

    # Create identifier
    set id probeWin[incr idCnt]
    upvar #0 $id hd
    set w .$id
    set hd(w) $w

    # Create toplevel
    toplevel $w -class TkRat
    wm title $w $t(probe)

    if [string compare $option(sendprot) smtp] {
	message $w.message -aspect 600 -text $t(dsn_need_smtp)
	button $w.button -text $t(dismiss) \
		-command "RecordPos $w prefProbeDSN; destroy $w; unset $id"
	pack $w.message \
	     $w.button -side top -padx 5 -pady 5
	return
    }

    set row 0
    foreach h $option(smtp_hosts) {
	label $w.l$row -text $h -width 32 -anchor e
	label $w.a$row -textvariable ${id}($h) -font $fixedNormFont -width 32 \
		-anchor w
	grid $w.l$row -row $row -column 0 -sticky e
	grid $w.a$row -row $row -column 1 -sticky w
	incr row
    }
    button $w.button -text $t(dismiss) -command "destroy $w; unset $id" \
	    -state disabled
    grid $w.button -row $row -column 0 -columnspan 2
    Place $w prefProbeDSN
    wm protocol $w WM_DELETE_WINDOW "destroy $w; unset $id"

    foreach h $option(smtp_hosts) {
        set hd($h) $t(probing)...
	update idletasks
	if [RatSMTPSupportDSN $h] {
	    set hd($h) $t(supports_dsn)
	} else {
	    set hd($h) $t(no_dsn_support)
	}
    }
    $w.button configure -state normal
}

# ShowTip --
#
# show the tip string
#
# Arguments:
# w	- message to create

proc ShowTip {w} {
    global t

    frame $w
    message $w.m -text $t(test_by) -aspect 700
    pack $w.m -side left -pady 5
}


# SetupNetworkSync --
#
# Setup the network synchronization
#
# Arguments:

proc SetupNetworkSync {} {
    global t option setupNS

    # Check if window already exists
    set w .setns
    if [winfo exists $w] {
	wm deiconify $w
	raise $w
	return
    }

    # Initialize variables
    set setupNS(w) $w
    set setupNS(deferred) [lindex $option(network_sync) 0]
    set setupNS(disconnected) [lindex $option(network_sync) 1]
    set setupNS(runcmd) [lindex $option(network_sync) 2]
    set setupNS(cmd) [lindex $option(network_sync) 3]

    # Create window
    toplevel $w -bd 5 -class TkRat
    wm title $w $t(setup_netsync)

    checkbutton $w.cmd -variable setupNS(runcmd) -text $t(run_command)
    grid $w.cmd - -sticky w
    entry $w.cmdentry -textvariable setupNS(cmd)
    grid x $w.cmdentry -sticky ew

    checkbutton $w.def -variable setupNS(deferred) -text $t(send_deferred)
    grid $w.def - -sticky w

    checkbutton $w.dis -variable setupNS(disconnected) -text $t(sync_dis)
    grid $w.dis -  -sticky w

    frame $w.f
    button $w.f.ok -text $t(ok) -default active -command {
	destroy $setupNS(w)
	set option(network_sync) [list $setupNS(deferred) \
				       $setupNS(disconnected) \
				       $setupNS(runcmd) \
				       $setupNS(cmd)]
	unset setupNS
	SaveOptions
    }
    button $w.f.cancel -text $t(cancel) \
	    -command {destroy $setupNS(w); unset setupNS}
    pack $w.f.ok $w.f.cancel -side left -expand 1
    grid $w.f - -sticky nsew -pady 5
}

# SelectFont --
#
# Show font selection
#
# Arguments:
# f - font to select
# w - window to build

proc SelectFont {f w} {
    global pref fixedNormFont t b

    set d [ConvertFontToText $pref(opt,$f)]

    frame $w
    label $w.l -text $d -font [RatCreateFont $pref(opt,$f)] -width 45 -anchor w
    pack $w.l -side left
    set b($w.l) pref_$f

    button $w.e -text $t(edit)... -padx 2 -pady 2 \
	    -command "DoEditFont $f $w.l $w"
    pack $w.e -side right -fill x
}

# ConvertFontToText --
#
# COnvert a font specification to text
#
# Arguments:
# s - A font specification

proc ConvertFontToText {s} {
    global t

    if {"components" == [lindex $s 0]} {
	set d [concat [lindex $s 1] [lindex $s 2]]
	if {"bold" == [lindex $s 3]} {
	    set d "$d $t(bold)"
	}
	if {"roman" != [lindex $s 4]} {
	    set d "$d $t(italic)"
	}
	if [lindex $s 5] {
	    set d "$d $t(underline)"
	}
	if [lindex $s 6] {
	    set d "$d $t(overstrike)"
	}
	return $d
    } else {
	return [lindex $s 1]
    }

}

# DoEditFont --
#
# Show the edit font window
#
# Arguments:
# font	- Font setting to edit
# l	- Label to update afterwards

proc DoEditFont {font l parent} {
    global t idCnt pref

    set id doEditFont[incr idCnt]
    upvar #0 $id hd

    # Initialization
    set hd(done) 0
    set hd(new_spec) $pref(opt,$font)
    set hd(old_spec) ""
    set hd(font_name) ""

    if {"components" == [lindex $hd(new_spec) 0]} {
	set hd(family) [lindex $hd(new_spec) 1]
	set hd(size) [lindex $hd(new_spec) 2]
	set hd(weight) [lindex $hd(new_spec) 3]
	set hd(slant) [lindex $hd(new_spec) 4]
	set hd(underline) [lindex $hd(new_spec) 5]
	set hd(overstrike) [lindex $hd(new_spec) 6]
	set hd(method) components
    } else {
	set hd(fontname) [lindex $hd(new_spec) 1]
	set hd(method) name
	set hd(family) Helvetica
	set hd(size) 12
    }

    # Create toplevel
    set w .fontedit
    toplevel $w -class TkRat
    wm title $w $t(edit_font)
    wm transient $w $parent

    # Top label
    label $w.topl -text $t(use_one_method)

    # Specification method frame
    frame $w.s -bd 1 -relief raised
    radiobutton $w.s.select -variable ${id}(method) -value components \
	    -command "UpdateFontSpec $id components"
    label $w.s.fl -text $t(family):
    set m $w.s.family.m
    menubutton $w.s.family -bd 1 -relief raised -indicatoron 1 -menu $m \
	    -textvariable ${id}(family) -width 15
    menu $m -tearoff 0
    if ![info exists pref(families)] {
	set pref(families) [lsort -dictionary [font families]]
    }
    foreach f $pref(families) {
	$m add command -label $f -command \
		"set ${id}(family) [list $f]; UpdateFontSpec $id components"
    }
    FixMenu $m
    label $w.s.sl -text "  $t(size):"
    set m $w.s.size.m
    menubutton $w.s.size -bd 1 -relief raised -indicatoron 1 -menu $m \
	    -textvariable ${id}(size) -width 3
    menu $m -tearoff 0
    foreach s {4 5 6 7 8 9 10 11 12 13 14 15 16 18 20 22 24 26 30 36} {
	$m add command -label $s -command \
		"set ${id}(size) $s; UpdateFontSpec $id components"
    }
    checkbutton $w.s.weight -text "$t(bold) " -onvalue bold -offvalue normal \
	    -variable ${id}(weight) -command "UpdateFontSpec $id components"
    checkbutton $w.s.italic -text "$t(italic) " -onvalue italic -offvalue roman\
	    -variable ${id}(slant) -command "UpdateFontSpec $id components"
    checkbutton $w.s.underline -text "$t(underline) " \
	    -variable ${id}(underline) -command "UpdateFontSpec $id components"
    checkbutton $w.s.overstrike -text $t(overstrike) \
	    -variable ${id}(overstrike) -command "UpdateFontSpec $id components"

    pack $w.s.select \
	 $w.s.fl $w.s.family \
	 $w.s.sl $w.s.size \
	 $w.s.weight \
	 $w.s.italic \
	 $w.s.underline \
	 $w.s.overstrike -side left -pady 2

    # Name method frame
    frame $w.n -bd 1 -relief raised 
    radiobutton $w.n.select -variable ${id}(method) -value name \
	    -command "UpdateFontSpec $id name"
    label $w.n.l -text $t(name):
    entry $w.n.e -width 20 -textvariable ${id}(name)
    button $w.n.set -text $t(update) -command "UpdateFontSpec $id name" -bd 1
    pack $w.n.select \
	 $w.n.l \
	 $w.n.e \
	 $w.n.set -side left -pady 2

    # Sample text
    message $w.sample -text $t(ratatosk) -aspect 200 -justify left
    set hd(sample) $w.sample

    # Buttons
    OkButtons $w $t(ok) $t(cancel) "set ${id}(done)"

    # Pack things
    pack $w.topl \
	 $w.s \
	 $w.n \
	 $w.sample \
	 $w.buttons -side top -fill x -pady 2 -padx 2

    # Bindings
    bind $w.n.e <Tab> "UpdateFontSpec $id name"
    bind $w.n.e <Return> "UpdateFontSpec $id name; break"

    # Update sample font
    UpdateFont $id

    # Show window and wait for completion
    Place $w editFont
    ModalGrab $w
    tkwait variable ${id}(done)

    # Finalization
    RecordPos $w editFont
    destroy $w
    set pref(opt,$font) $hd(old_spec)
    if {"" != $hd(font_name)} {
	font delete $hd(font_name)
    }
    unset hd
    $l configure -text [ConvertFontToText $pref(opt,$font)] \
	    -font [RatCreateFont $pref(opt,$font)]
}

# UpdateFontSpec --
#
# Update the shown font
#
# Arguments:
# handler - Handler of font window
# method  - which method to use

proc UpdateFontSpec {handler method} {
    upvar #0 $handler hd

    if {"components" == $method} {
	set hd(new_spec) [list components $hd(family) $hd(size) $hd(weight) \
				    $hd(slant) $hd(underline) $hd(overstrike)]
	set hd(method) components
    } else {
	set hd(new_spec) [list name $hd(name)]
	set hd(method) name
    }
    UpdateFont $handler
}

# UpdateFont --
#
# Update the sample text
#
# Arguments:
# handler - Handler of font window

proc UpdateFont {handler} {
    upvar #0 $handler hd

    if {"$hd(new_spec)" == "$hd(old_spec)"} {
	return
    }
    if {[lindex $hd(new_spec) 0] == "components"} {
	if {"" == $hd(font_name)} {
	    set op create
	    set hd(font_name) fontedit
	} else {
	    set op configure
	}
	font $op $hd(font_name) \
		-family [lindex $hd(new_spec) 1] \
		-size -[lindex $hd(new_spec) 2] \
		-weight [lindex $hd(new_spec) 3] \
		-slant [lindex $hd(new_spec) 4] \
		-underline [lindex $hd(new_spec) 5] \
		-overstrike [lindex $hd(new_spec) 6]
	set fn $hd(font_name)
    } else {
	set fn [lindex $hd(new_spec) 1]
    }

    set hd(old_spec) $hd(new_spec)

    $hd(sample) configure -font $fn
}
