# -*- coding: utf-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
#  Fonctions de calcul et validation des variables Creole
#
###########################################################################
import sys
import json
import socket
import subprocess
import stat
import warnings
from pyeole.process import system_out
from pyeole.deprecation import deprecated
from . import eoleversion
from .config import func_dir, containers_default_network, \
 INSTANCE_LOCKFILE
from .error import TypeEoleError, FileNotFound, ConfigError
from .utils import classes, gen_random
from .wpkg_secrets import wcrypt, wdecrypt
if sys.version_info[0] >= 3:
    import importlib.util
else:
    import imp
from tiramisu.error import ValueWarning
from IPy import IP
try:
    with warnings.catch_warnings():
        warnings.filterwarnings("ignore",category=DeprecationWarning)
        import pyudev
except:
    pass
    # librairie non disponible (Zéphir 2.3)

from .i18n import _

# modules utilitaires à rendre accessible dans les fonctions
import os, sys, re, glob, time, hashlib, random, socket, pwd, grp

if sys.version_info[0] >= 3:
    def unicode2(val, *args, **kwargs):
        return str(val)
else:
    unicode2 = unicode


# type paramètre
class Param:
    """paramètre pour les fonctions de contrainte"""
    def __init__(self, value, name=None, typ=None):
        self.value = value
        self.name = name
        self.type = typ


#############################
## Fonctions de vérification
#############################
@deprecated
def calc_container(mode_container, container_info, mode_zephir='non'):
    """ retourne une variable de conteneur """
    #container_info contient l'ip du conteneur directement (#6240)
    return container_info


def get_version(name):
    try:
        return getattr(eoleversion, name)
    except AttributeError as err:
        raise ValueError(_(u'Unknown variable {0}').format(name))

def valid_ip(data):
    """fonction de validation de la saisie d'une adresse ip"""

    # variable non obligatoire
    if data == "":
        return True

    # découpage de l'ip
    ip = data.split('.')
    if len(ip) != 4:
        raise ValueError(_(u"Wrong IP address format"))

    # vérification des nombres
    for part in ip:
        try:
            num = int(part)
        except ValueError:
            raise ValueError(_(u"An IP address is a sequence of four numbers separated by dots."))
        else:
            if not (0 <= num < 256):
                raise ValueError(_(u"Each number is in range [0, 255]."))

    # on n'a pas rencontré d'erreur
    return True

def valid_intervalle(data):
    """ fonction de validation d'une plage d'ip """
    if data == "":
        return True

    liste = data.split()
    if len(liste) != 2:
        raise ValueError(_(u"Two IP addresses are required."))
    else:
        # on vérifie la syntaxe des ips
        for ip in liste:
            valid_ip(ip)
    return True

def valid_network(data, ip=None):
    """fonction de validation de la saisie d'un réseau"""
    if data == "":
        return True

    # on utilise la fonction de test d'adresse ip pour la syntaxe
    valid_ip(data)
    if ip is not None:
        # on effectue les tests spécifiques au réseau (fonction de l'ip ?)
        ip_parts = ip.split(".")
        network_parts = data.split(".")
        for i in range(4):
            if (int(network_parts[i]) != 0) and (network_parts[i] != ip_parts[i]):
                raise ValueError(_(u"Specified network does not match IP address {0}").format(ip))

    # aucune erreur rencontrée
    return True

def valid_netmask(data):
    """fonction de validation de la saisie d'un masque réseau"""
    if data == "":
        return True

    # on utilise la fonction de test d'adresse ip pour la syntaxe
    valid_ip(data)
    # on effectue les tests spécifiques aux masques réseaux
    netmask = []
    for part in data.split("."):
        netmask.append(int(part))
    error = 0
    mask_len = 0
    done = 0
    for part in netmask:
        bitmask = 0x80
        for bit in range(8):
            if not done:
                if part & bitmask:
                    mask_len = mask_len + 1
                else:
                    done = 1
            else:
                if part & bitmask:
                    error = 1
            bitmask = bitmask >> 1

    if error == 1:
        raise ValueError(_(u"Subnet mask format invalid"))
    # pas d'erreur rencontrée
    return True

def valid_booleen(data, type_bool="alpha"):
    """fonction de validation d'une réponse booléenne
       type_bool : alpha ou num (O/N ou 0/1)
    """
    if data == "":
        return True

    if (data[0].upper() != 'O') and (data[0].upper() != 'N'):
        raise ValueError(_(u"You must answer 'O' or 'N'"))
    # pas d'erreurs détectées
    return True

def valid_upper(data):
    """
    validation d'une valeur en majuscules
    """
    if data != data.upper():
        raise ValueError(_(u"Value must be uppercase."))
    return True

def valid_lower(data):
    """
    validation d'une valeur en minuscules
    """
    if data != data.lower():
        raise ValueError(_(u"Value must be lowercase."))
    return True

def valid_enum(data, liste=['oui','non','Oui','Non','OUI','NON','o','n','O','N'], checkval="True"):
    """fonction de validation d'une réponse en fonction
    d'une liste de valeur possibles
    """
    if data == "":
        return True

    try:
        if type(liste) != list:
            liste = eval(liste)
    except:
        raise ValueError(_(u"Invalid potential values list"))
    if data not in liste and checkval != "False":
        raise ValueError(_(u"Choose one of these options: {0}").format(liste))
    return True

def valid_regexp(data, exp_reg, err_msg=_(u"Invalid syntax")):
    """fonction de validation d'une saisie par expression régulière"""
    if data == "":
        return True
    match = re.match(exp_reg, data)
    if match is None:
        raise ValueError(err_msg)
    else:
        return True

def valid_entier(data, mini=None, maxi=None):
    """ fonction de validation d'un nombre (intervalle optionnel)"""
    if data == "":
        return True
    try:
        value = int(data)
    except ValueError:
        raise ValueError(_(u"A number is required."))
    else:
        if mini is not None and value < int(mini):
            raise ValueError(_(u"Give an integer greater than or egal {0}").format(mini))
        if maxi is not None and value > int(maxi):
            raise ValueError(_(u"Give an integer lesser than or egal {0}").format(maxi))
        return True

def valid_len(data, length=None, max_len=None, min_len=None):
    """valide la longueur d'une variable
    * length : longueur fixe de la variable
    * max_len : taille maximum
       et/ou
      min_len : taille minimum

    Attention il faut utiliser soit length soit max_len/min_len mais pas les deux groupes ensemble
    """
    if length == max_len == min_len == None:
        raise ValueError(_(u'valid_len : a length must be given, either an exact length or a maximum or minimum length.'))
    if length is not None and (max_len is not None or min_len is not None):
        raise ValueError(_(u'valid_len : an exact length can not be used with a minimum or a maximum length.'))
    if length is not None:
        if len(data) != int(length):
            raise ValueError(_(u'The exact length of the given variable must be {0}.').format(length))
    else:
        if max_len is not None and len(data) > int(max_len):
            raise ValueError(_(u'The length of the given variable must be lesser than {0}').format(max_len))
        if min_len is not None and len(data) < int(min_len):
            raise ValueError(_(u'The length of the given variable must be greater than {0}').format(min_len))

def valid_yesno(data):
    """Yes ou No pour les fichiers de config"""
    if data not in ["Yes", "No"]:
        raise ValueError(_(u"Value must be either Yes or No."))
    return True

def valid_system(data, cmd, result, *args):
    """ test d'une commande système """
    if data == "":
        return True

    cmd_string = cmd
    for arg in args:
        cmd_string += " "+arg

    # exécution de la fonction
    code_retour = os.system(cmd_string)

    if code_retour == result:
        return True
    else:
        raise ValueError(_(u"Command {0} execution has failed").format(cmd_string))

def valid_differ(data, value):
    """ test d'une valeur indésirable """
    if data == value:
        raise ValueError(_(u"Value must be different from {0}").format(value))
    else:
        return True

def valid_chaine(data, forbidden_chars="[' ']"):
    """
    Recherche de caractères interdits dans une chaine
    """
    forbidden_chars = eval(forbidden_chars)
    for char in forbidden_chars:
        if char in data:
            raise ValueError(_(u"\"{0}\" character is not allowed.").format(char))
    return True

def valid_name(data):
    """
    nom valide
    """
    valid_lower(data)
    valid_chaine(data, forbidden_chars="['@', ' ', '_']")
    return True

def valid_domaine(data):
    """
    nom de domaine valide
    """
    valid_name(data)
    valid_regexp(data, '.+\..+', _(u"Domain name is missing a top-level domain (TLD)."))
    return True

def valid_alias(data):
    """
    alias apache valide
    """
    valid_name(data)
    valid_regexp(data, '^/', _(u"Aliases must start with a slash (\"/\")."))
    return True

def valid_phpmyadmin(data, apache):
    """ phpmyadmin dépend d'apache """
    if data == 'oui' and apache == 'non':
        raise ValueError(_(u"Activating Apache is required in order to activate phpMyAdmin."))
    return True

def valid_addr_sso(data, sso_actif, revprox_actif="non", revprox_sso="", revprox_domainname="", *args):
    """vérifie les conflits entre sso local et reverse proxy"""
    if data in ['localhost', '127.0.0.1']:
        raise ValueError(_(u"SSO service must listen on an interface other than localhost."))
    if revprox_sso and revprox_actif == 'oui':
        if data in args and sso_actif == 'oui':
            raise ValueError(_(u"Local SSO service is in conflict with reverse proxy, use IP address {0} instead.").format(revprox_domainname or revprox_sso))
    return True


def valid_country(data):
    try:
        error_val = not data.isalpha()
    except:
        raise ValueError(_(u'Value must be of type string.'))
    if error_val:
        raise ValueError(_(u'Value must be uppercase letters.'))
    valid_len(data, 2)
    try:
        valid_upper(data)
    except ValueError as err:
        #uniquement warning pour compatibilité #8585
        raise ValueWarning(unicode2(str(err), 'utf-8'), None)
    return True


def valid_float(data):
    """
    Raise ValueError if data string is castable as a float
    :param data: data to be tested
    :type data: string
    """
    if data is None:
        return True

    try:
        fv = float(data)
    except ValueError:
        raise ValueError("Cette valeur doit être un nombre a virgule flotante avec un séparateur '.'")
    return True


def verif_cas_active(data, var_name, full_cas):
    """ vérifie l'activation du full_cas
    """
    if data == 'oui' and full_cas == 'non':
        raise ValueError(_(u"Please activate \"Utilisation du service sso pour les applications de votre serveur Scribe\" before activating {0}.").format(var_name))
    return True


def obligatoire(data):
    """ fonction qui vérifie que les données ne sont pas vides
    """
    if data == [] or data == "" or data[0].strip() == "":
        raise TypeEoleError(_(u"This parameter is mandatory."))
    else:
        return True

def is_empty(data):
    if str(data) in ['', '""', "''", "[]", "['']", '[""]', "None"]:
        return True
    return False

def check_name_uniq(value, values, index):
    values.pop(index)
    if value in values:
        raise ValueError('le nom {} est déjà attribué à une autre plage'.format(value))


def valid_krb_name(name, auth=None):
    """
    Vérifie la longeur du nom de machine en mode Kerberos #25885
    """
    if auth == u'NTLM/KERBEROS' and len(name) > 15:
        raise ValueError(_(u"Serveur name should have less than 15 characters for Kerberos proxy authentication"))
    return True


################################################################
## Fonctions de calcul de dépendance entre variables et groupes
################################################################

def hidden_if_not_in(val, *args):
    """ on cache si la valeur n'est pas présente dans celles proposées """
    if val not in args:
        return None
        #raise NoValueReturned("""%s n'est pas dans %s""" % (val, args))

def hidden_if_in(val, *args):
    """ on cache si la valeur est présente dans celles proposées """
    if val in args:
        return None
        #raise NoValueReturned("""%s est dans %s""" % (val, args))

#############################################
## Fonctions de saisie et calcul des valeurs
#############################################

def get_devices():
    cmd = ['/sbin/ip', '-j', 'address', 'show', 'up', 'primary']
    code, stdout, stderr = system_out(cmd)
    if code != 0:
        raise Exception(_(u'Error running command {0} : {1}').format(' '.join(cmd), stderr))

    devs = {}
    for addr in json.loads(stdout):
        if 'addr_info' not in addr:
            continue
        addr_infos = {addr_info['family']: addr_info
                      for addr_info in addr['addr_info']
                      if 'family' in addr_info}
        if 'inet' not in addr_infos:
            continue
        devname = addr['ifname']
        ip = addr_infos['inet']['local']
        prefixlen = addr_infos['inet']['prefixlen']
        broadcast = addr_infos['inet'].get('broadcast')
        if devname not in devs:
            devs[devname] = (ip, calc_netmask(str(prefixlen)), broadcast)
    return devs


def get_net_devices(mode_zephir='non'):
    if mode_zephir == 'oui':
        return []
    devices = []
    PERSISTENT = '/var/lib/eole/config/persistent-net.cfg'
    if os.path.isfile(PERSISTENT):
        fh = open(PERSISTENT, 'r')
        for name in fh.readlines():
            devices.append(name.strip())
    new_devices = []
    for dirname in glob.glob('/sys/class/net/*'):
        if os.path.islink(os.path.join(dirname, 'device', 'driver')):
            name = os.path.split(dirname)[1]
            new_devices.append(name)
    new_devices.sort()
    for device in new_devices:
        if device not in devices:
            devices.append(device)
    return devices



def get_net_device(index, mode_zephir='non'):
    if mode_zephir == 'oui':
        return None
    devices = get_net_devices()
    if not index < len(devices):
        return None
    return unicode2(devices[index])


def calc_broadcast(ip, netmask):
    """ calcule l'adresse de broadcast par défaut à partir d'un masque et d'une ip """
    # si les ip et netmask viennent du dictionnaire, on les récupère
    if is_empty(ip) or is_empty(netmask):
        return None
    broadcast, network = calc_adresses(ip, netmask)
    return broadcast

def create_ip(ip_list):
    """ création d'une adresse ip à partir d'une liste"""
    template = """%i.%i.%i.%i"""
    return template % tuple(ip_list)

def calc_adresses(ip, mask):
    """calcul l'adresse de broadcast à partir de l'ip et du netmask
    """
    if is_empty(mask) or is_empty(ip):
        return None, None
    try:
        ipy = IP('{0}/{1}'.format(ip, mask), make_net=True)
    except ValueError:
        raise ValueError(_(u'IP or subnet mask invalid ({0}, {1})').format(ip, mask))
    network = unicode2(str(ipy.net()), 'utf8')
    broadcast = unicode2(str(ipy.broadcast()), 'utf8')
    return broadcast, network

def calc_classe(netmask):
    """calcule la classe du réseau à partir du netmask"""

    if netmask in classes:
        return(classes[netmask])
    else:
        return(netmask)

def calc_netmask(plage):
    """calcule le reseau a partir de sa classe"""
    if plage in list(classes.values()):
        for mask, plaj in list(classes.items()):
            if plaj == plage:
                return mask
    else:
        return plage

def calc_pg_cache_size():
    """
    Calcule la taille du cache postgreql en fonction de la RAM
    """
    procmem = open('/proc/meminfo')
    meminfo = procmem.read().split('\n')
    procmem.close()
    for line in meminfo:
        if line.startswith('MemTotal'):
            mem_tot = float(line.split()[1])
            break
    return int( (mem_tot * 0.50) / 8 )

def calc_mysql_innodb_size():
    """
    Calcule la taille du cache mysql en fonction de la RAM
    """
    procmem = open('/proc/meminfo')
    meminfo = procmem.read().split('\n')
    procmem.close()
    for line in meminfo:
        if line.startswith('MemTotal'):
            mem_tot = float(line.split()[1])
            break
    mem_cache = int( (mem_tot * 0.72) /1024)
    return unicode2(mem_cache, 'utf8')

def calc_val(valeur=None):
    """ valeur par défaut "en dur" """
    return(valeur)

def calc_val_first_value(*args):
    if args == tuple():
        return None
    return args[0]

def calc_ssl_country_name(valeur):
    """Fonction spécifique à ssl_country_name
    Si la longueur de la valeur copier n'est pas de 2 force la valeur 'FR'
    """
    if valeur == None or len(valeur) != 2 or not valeur.isalpha():
        return u"FR"
    return valeur.upper()

def concat(*args, **kwargs):
    """ concatène deux valeurs """
    sortedkeys = list(kwargs.keys())
    sortedkeys.sort()
    sortedkwvalues = []
    for key in sortedkeys:
        if kwargs[key] == '' or kwargs[key] is None:
            return None
        sortedkwvalues.append(kwargs[key])
    if None in args:
        return None
    return "".join(args)+''.join(sortedkwvalues)


def gen_container_paths(prefix, lst=[]):
    ret = []
    for elt in lst:
        ret.append(prefix + elt)
    return ret


def concat_path(*args):
    """ concaténation de chemins """
    if None in args:
        return None
    values = []
    for val in args:
        if val.startswith('/'):
            values.append(val[1:])
        else:
            values.append(val)
    return '/' + os.path.join(*values)

def calc_multi_val(*args, **kwargs):
    """ valeur par défaut "en dur" avec un nombre non défini de paramètres
    Attention, une valeur vide passée en paramètre à cette fonction force
    le renvoi d'une liste vide ([]) sauf si allow_none vaut True
    Les doublons sont supprimés
    """
    res = []
    for arg in args:
        if arg is None:
            if kwargs.get('allow_none', u'False') == u'True':
                continue
            else:
                return []
        # gestion des valeurs multi
        if isinstance(arg, list):
            for ev_arg in arg:
                if ev_arg is not None and ev_arg not in res:
                    res.append(ev_arg)
        else:
            res.append(arg)
    return res

def calc_multi_condition(param, match=None, mismatch=None, operator='AND',
                         default_match=None, default_mismatch=None,
                         eval_match='False', eval_mismatch='False',
                         **conditions):
    """param: réponse testée sur toutes les conditions. Si les réponses des
    conditions sont différentes mettre une liste avec chacune des conditions
    a tester
    operator: doit être 'AND' ou 'OR'
    match: valeur de retour si toutes les conditions sont validées (si
           operator est à 'AND') ou si une des conditions sont validée
           (si operator est à 'OR')
    mismatch: valeur de retour si au moins une des conditions n'est pas
              validée (si operator est à 'AND') ou toutes les conditions
              ne sont pas validées (si operator est à 'OR')
    si les variables match ou mismatch sont optional ou hidden, recupère le
    contenu de la variable default_match ou default_mismatch
    conditions: liste des conditions a tester.
    exemple:
        <auto name='calc_multi_condition' target='test_activer_bareos_dir'>
            <param>['oui', 'non']</param>
            <param type='eole' name='condition_1'>activer_bareos</param>
            <param type='eole' name='condition_2'>activer_bareos_dir</param>
            <param name='match'>oui</param>
            <param name='mismatch'>non</param>
        </auto>
    """
    if operator not in ('AND', 'OR'):
        raise ValueError(_(u'Operator must be either "AND" or "OR"'))
    if eval_match not in ('True', 'False'):
        raise ValueError(_(u'eval_match must be either "True" or "False"'))
    if eval_mismatch not in ('True', 'False'):
        raise ValueError(_(u'eval_mismatch must be either "True" or "False"'))
    if match is None:
        if default_match is None:
            match = u'oui'
        else:
            if default_match == 'None':
                match = None
            else:
                match = default_match
    if mismatch is None:
        if default_mismatch is None:
            mismatch = u'non'
        else:
            if default_mismatch == 'None':
                mismatch = None
            else:
                mismatch = default_mismatch
    conditions_keys = list(conditions.keys())
    conditions_keys.sort()
    for condition in conditions_keys:
        if not condition.startswith('condition_'):
            raise ValueError(_(u'Condition keys must start with "condition".'))
    # si le paramètre est une liste...
    if sys.version_info[0] >= 3:
        str_obj = str
    else:
        str_obj = (str, unicode)
    if isinstance(param, str_obj) and param.startswith('['):
        param = eval(param)
    # si il faut évaluer le résultat à retourner...
    if eval_match == 'True':
        match = eval(match)
    if eval_mismatch == 'True':
        mismatch = eval(mismatch)
    if isinstance(param, list) and len(param) != len(conditions):
        raise ValueError(_(u'Conditions and parameters counts do not match in calc_multi_condition.'))
    for num in range(0, len(conditions)):
        key = conditions_keys[num]
        value = conditions[key]
        if not isinstance(param, list):
            if operator == 'AND' and value != param:
                return mismatch
            if operator == 'OR' and value == param:
                return match
        else:
            if operator == 'AND' and value != param[num]:
                return mismatch
            if operator == 'OR' and value == param[num]:
                return match
    if operator == 'AND':
        return match
    else:
        return mismatch


def calc_multi_domains(*args, **kwargs):
    """
    agrégation de noms de domaines depuis des variables
    mono ou multi valuées pouvant être vides
    avec suppression des adresses IP et des doublons
    """
    res = set()
    for arg in args:
        if arg is None:
            continue
        # gestion des valeurs multi
        if isinstance(arg, list):
            for ev_arg in arg:
                if ev_arg is not None and not is_ip(ev_arg):
                    res.add(ev_arg)
        elif not is_ip(arg):
            res.add(arg)
    return list(res)


def list_len_gt(param, max_len=None, match=None, mismatch=None,
                default_match=None, default_mismatch=None,
                eval_match='False', eval_mismatch='False',):
    """retourne la valeur 'match' si la longueur de la liste est supérieure à max_len, 'mismatch' sinon
    * max_len : longueur maximum de la liste à tester
    """
    if match is None:
        if default_match is None:
            match = u'oui'
        else:
            if default_match == 'None':
                match = None
            else:
                match = default_match
    if mismatch is None:
        if default_mismatch is None:
            mismatch = u'non'
        else:
            if default_mismatch == 'None':
                mismatch = None
            else:
                mismatch = default_mismatch
    # si il faut évaluer le résultat à retourner...
    if eval_match == 'True':
        match = eval(match)
    if eval_mismatch == 'True':
        mismatch = eval(mismatch)
    if isinstance(param, list):
        if len(param) > int(max_len):
            return match
        else:
            return mismatch
    else:
        return mismatch

def calc_binary_and(valeur1, valeur2):
    """
        Calcule un 'et' binaire :
        valeur1 & valeur2
    """
    if valeur1 == 'oui' and valeur2 == 'oui':
        return('oui')
    else:
        return('non')

def calc_libelle_annuaire(ldap_host, nom_machine, nom_domaine_local):
    """
    calcule un libellé pour l'annuaire local
    """
    if ldap_host is None:
        return None
    if ldap_host in ['127.0.0.1', 'localhost']:
        return _(u"LDAP directory of {0}.{1}").format(nom_machine, nom_domaine_local)
    try:
        hostname = socket.gethostbyaddr(ldap_host)[0]
    except:
        hostname = ldap_host
    return _(u"LDAP directory of {0}").format(hostname)

def calc_eolesso_adresse_local(activer_sso, eolesso_adresse=None):
    """
    retourne eolesso_adresse uniquement si le SSO est local
    """
    if activer_sso == 'local':
        return eolesso_adresse
    return None

def calc_last_interface(nombre_int):
    return unicode2('eth'+str(int(nombre_int)-1), 'utf8')

def extract_ad_domain_from_realm(realm):
    """Return the domain name of a REALM

    Split the realm accros the dots and return the first element.

    :param realm: realm FQDN
    :type realm: `str`
    :return: domain part of the realm
    :rtype: `str`

    """
    if realm is None:
        return None
    return realm.split('.')[0]


def proxy_domain_reader(activate):
    filename = '/var/lib/eole/config/domain_exceptions'
    if activate == 'non' or not is_file(filename):
        return []
    try:
        with open(filename) as reader:
            return [r.strip() for r in reader.readlines() if r]
    except Exception as err:
        raise ValueError("erreur lors de la lecture du fichier {}".format(err))


#################################################
## Fonctions de calcul des valeurs automatiques
#################################################

def server_mem():
    """renvoie la mémoire du serveur
    """
    procmem = open('/proc/meminfo')
    meminfo = procmem.read().split('\n')
    procmem.close()
    for line in meminfo:
        if line.startswith('MemTotal'):
            mem_tot = float(line.split()[1])
            break
    return unicode2(mem_tot, 'utf8')

def kernel_version():
    """ renvoie la version du kernel
    """
    return_code, output, stderr = system_out(['uname', '-'])
    if return_code == 0:
        return unicode2(output, 'utf8')
    else:
        return ''

def auto_copy_val(value):
    """
        Renvoie la valeur value
        utilisé pour forcer la valeur d'une variable
    """
    return(value)

def auto_eth(nom_carte, **kw):
    """ renvoie l'adresse de la carte nom_carte
    """
    if kw['parametre'].lower() == 'pppoe':
        nom_carte = 'ppp0'
    #if dhcp or pppoe
    if kw['parametre'].lower() in [u'dhcp', u'non géré'] or nom_carte == u'ppp0':
        eths = get_devices()
        # sur Zéphir, nom_carte est None (non calculable). #19233
        if nom_carte and 'br'+nom_carte in list(eths.keys()):
            nom_carte = 'br'+nom_carte
        #6106
        return unicode2(eths.get(nom_carte, ('169.254.0.1', None, None))[0], 'utf8')
    else:
        return None

def auto_netmask(nom_carte, parametre, condition):
    """ renvoie le netmask pour la carte nom_carte
    """
    if parametre.lower() == 'pppoe':
        nom_carte = 'ppp0'
    if parametre.lower() in [u'dhcp', u'non géré'] or nom_carte == u'ppp0':
        eths = get_devices()
        #6106
        val = eths.get(nom_carte, (None, '255.255.0.0', None))[1]
        try:
            return(unicode2(val, 'utf8'))
        except TypeError:
            return val
    else:
        return u"255.255.255.0"

def auto_ip_ldap(activer_client_ldap, **kw):
    """ renvoie l'adresse du serveur ldap local
    pas de valeur renvoyée si on choisit un ldap distant
    """
    if activer_client_ldap != 'local':
        return None
        #raise NoValueReturned
    else:
        # récupération de l'adresse du service local
        return kw['ip_annuaire']

def auto_broadcast(nom_carte, parametre, condition):
    """ renvoie le broadcast pour la carte nom_carte
    """
    if parametre.lower() == 'pppoe':
        nom_carte = 'ppp0'
    if parametre.lower() in [u'dhcp', u'non géré'] or nom_carte == u'ppp0':
        eths = get_devices()
        broadcast = str(eths.get(nom_carte, ('', '', '169.254.255.255'))[2])
        return broadcast
    else:
        return None

def auto_defaultroute(**kw):
    """ renvoie la route par défaut
    """
    cmd = "ip route list scope global"
    cmd = cmd.split()
    process = subprocess.Popen(args=cmd, stdout=subprocess.PIPE,
                               stderr=subprocess.PIPE, shell=False)
    code = process.wait()
    if code != 0:
        return False
    defaultroute = process.stdout.read()
    if sys.version_info[0] >= 3:
        defaultroute = defaultroute.decode()
    defaultroute = re.findall('default\s+via\s+([\d\.]+)\s+dev\s+(\w+)\s+', defaultroute)
    if len(defaultroute) == 0:
        cmd = "ip route list"
        cmd = cmd.split()
        process = subprocess.Popen(args=cmd, stdout=subprocess.PIPE,
                                   stderr=subprocess.PIPE, shell=False)
        code = process.wait()
        if code != 0:
            return False
        defaultroute = process.stdout.read()
        if sys.version_info[0] >= 3:
            defaultroute = defaultroute.decode()
        defaultroute = re.findall('default\s+dev\s+(\w+)\s+', defaultroute)
        if len(defaultroute) == 0:
            return [['0.0.0.0', kw.get('nom_zone_eth0', 'eth0')]]
        else:
            defaultroute = [['0.0.0.0', defaultroute[0]]]
    return defaultroute

def auto_defaultgw_ip(method, nom_zone_eth0='eth0'):
    """ renvoie la route par défaut
    """
    if method == "statique":
        return None
        #raise NoValueReturned
    #DHCP
    ip = auto_defaultroute(nom_zone_eth0=nom_zone_eth0)[0][0]
    if ip == u'0.0.0.0':
        return None
    return unicode2(ip, 'utf8')

@deprecated
def auto_defaultgw_int(method):
    """ renvoie la route par défaut
    """
    if method == "statique":
        return u'eth0'
    elif method == "pppoe":
        return u'ppp0'
    else:
        #DHCP
        rint = auto_defaultroute()
        return unicode2(rint[0][1], 'utf8')

def auto_defaultroute_ip(method, calc_ip_gw):
    """ renvoie la route par défaut
    """
    if method == "statique" or calc_ip_gw == "non":
        return None
        #raise NoValueReturned
    #DHCP
    ip = auto_defaultroute()
    return ip[0][0]

def auto_defaultroute_int(method, calc_ip_gw):
    """ renvoie la route par défaut
    """

    if method == "statique" or calc_ip_gw == "non":
        return None
        #raise NoValueReturned
    if method == "pppoe":
        return 'ppp0'
    #DHCP
    rint = auto_defaultroute()
    return rint[0][1]

def get_interface_from_ip(ip):
    try:
        ip = IP(ip).strNormal(0)
        a = system_out(['ip', 'route', 'get', ip])
        dev = a[1].split('dev')[1].split()[0]
        if dev == 'lo':
            return 'eth0'
        return dev
    except:
        return 'eth0'


def get_ips_from_interface(ips, interface):
    ret = []
    for ip in ips:
        try:
            IP(ip)
        except ValueError:
            try:
                # domain => adresse IP
                ip = socket.gethostbyname(ip)
            except:
                pass
        if interface == get_interface_from_ip(ip):
            ret.append(ip)
    return ret


def get_zone_name_bridge(nom_carte, context):
    #si une interface conteneur est en mode bridge => l'interface est en mode bridge
    is_bridge = u'non'
    try:
        if isinstance(nom_carte, list):
            if len(nom_carte) <= 1:
                nom_zone = nom_carte[0]
            else:
                return is_bridge
        else:
            nom_zone = nom_carte
        for path in context.containers.interfaces.find(byname='linkto', byvalue=nom_zone, type_='path'):
            #remove .linkto to path
            path = path[:-7]
            if getattr(context, path + '.method') == 'bridge':
                is_bridge = u'oui'
                break
    except:
        pass
    return is_bridge


def get_zone_name(nom_carte, method, zone_is_bridge='non', interface_no=None):
    #seulement pour eth0/pppoe (pas de support de bridge sur cette interface)
    if method == "pppoe":
        return u'ppp0'
    if zone_is_bridge == 'oui':
        if nom_carte is not None:
            if isinstance(nom_carte, list):
                return 'br' + nom_carte[0]
            else:
                return 'br' + nom_carte
        else:
            nom_carte = None
    elif isinstance(nom_carte, list):
        if len(nom_carte) > 1:
            return u'bond{0}'.format(str(interface_no))
        elif len(nom_carte) == 1:
            nom_carte = nom_carte[0]
        else:
            nom_carte = None
    return nom_carte


@deprecated
def auto_module():
    return "module"


def valid_fingerprint(value, type):
    if type == 'sha1':
        length = 20
    else:
        length = 32
    splitted_value = value.split(':')
    for val in splitted_value:
        if not val.isnumeric() and not val.isupper():
            raise ValueError(_(u"Value must be uppercase."))
        if len(val) != 2:
            raise ValueError(_('a {} fingerprint is something like {}XX').format(type, 'XX:'*(length-1)))
    if len(splitted_value) != length:
        raise ValueError(_('a {} fingerprint is something like {}XX').format(type, 'XX:'*(length-1)))

####################################################
# fonctions diverses accessibles dans les templates
####################################################

def calc_or_auto_network(ip, netmask, nom_carte, parametre, condition):
    """ renvoie le network pour la carte nom_carte
    """
    # condition is no longer used
    if parametre.lower() == 'pppoe':
        nom_carte = 'ppp0'
    network = None
    if parametre.lower() in [u'dhcp', u'non géré'] or nom_carte == u'ppp0':
        #calcul à partir de l'IP réel
        eths = get_devices()
        #6106
        ip, netmask, broadcast = eths.get(nom_carte, ('169.254.0.1', '255.255.0.0', '169.254.255.255'))
    if is_empty(ip) or is_empty(netmask):
        return None
    broadcast, network = calc_adresses(ip, netmask)
    return network

def calc_if_in_network(ip_addr1, netmask1, ip_addr2, netmask2, ):
    """ renvoie ip_addr1 si ip_addr1 et ip_addr2 sont dans le même réseau
        et "255.255.255.255" sinon
        ip_addr2 et netmask2 peuvent être une liste
    """
    if ip_addr1 is not None and netmask1 is not None:
        try:
            ip1 = IP("{0}/{1}".format(ip_addr1, netmask1), make_net=True)
        except ValueError:
            raise ValueError(_(u'IP or subnet mask invalid ({0}, {1})').format(ip_addr1, netmask1))
        if isinstance(ip_addr2, list):
            res = []
            for ip_addr, netmask in ip_addr2, netmask2:
                if ip_addr is not None and netmask is not None:
                    try:
                        ip = IP("{0}/{1}".format(ip_addr, netmask), make_net=True)
                    except ValueError:
                        raise ValueError(_(u'IP or subnet mask invalid ({0}, {1})').format(ip_addr, netmask))
                    if ip1 == ip:
                        res.append(unicode2(str(ip_addr1), 'utf8'))
                    else:
                        res.append(u'255.255.255.255')
                else:
                    res.append(None)
        else:
            if ip_addr2 is not None and netmask2 is not None:
                try:
                    ip2 = IP("{0}/{1}".format(ip_addr2, netmask2), make_net=True)
                except ValueError:
                    raise ValueError(_(u'IP or subnet mask invalid ({0}, {1})').format(ip_addr2, netmask2))
                if ip1 == ip2:
                    res = unicode2(str(ip_addr1), 'utf8')
                else:
                    res = u'255.255.255.255'
            else:
                res = None
    else:
        res = None
    return res

def calc_network(ip, netmask):
    """ calcule le réseau par défaut à partir d'un masque et d'une ip """
    # si les ip et netmask viennent du dictionnaire, on les récupère
    if is_empty(ip) or is_empty(netmask):
        return None
    broadcast, network = calc_adresses(ip, netmask)
    return network

def calc_or_auto_broadcast(ip, netmask, nom_carte, parametre, condition):
    #try:
    #    return auto_broadcast(nom_carte, parametre, condition)
    #except NoValueReturned:
    #    return calc_broadcast(ip, netmask)
    ret = auto_broadcast(nom_carte, parametre, condition)
    if ret == None:
        ret = calc_broadcast(ip, netmask)
    return ret


def auto_dns(container_ip_dns=None, nom_zone_eth0=None):
    """ Return list of DNS of interface_0 configured by DHCP
    """
    if nom_zone_eth0 is None:
        return []

    dns_servers = set()
    if os.access('/usr/bin/systemd-resolve', os.X_OK):
        cmd = ['/usr/bin/systemd-resolve', '--status']
        process = subprocess.Popen(args=cmd, stdout=subprocess.PIPE,
                                   stderr=subprocess.PIPE, shell=False)
        code = process.wait()
        if code != 0:
            return []

        resolve_output = process.stdout.readlines()
        store_dns = False
        for output_line in resolve_output:
            line = output_line.decode('utf-8')
            if '({0})'.format(nom_zone_eth0) in line:
                store_dns = True
                continue

            if not store_dns:
                continue

            if store_dns and not line.strip():
                break

            if not dns_servers and 'DNS Servers:' not in line:
                continue

            if 'DNS Servers' in line:
                ip_dns = line.split(':')[1].strip()
                try:
                    # Add only IPv4 #26475
                    valid_ip(ip_dns)
                    dns_servers.add(ip_dns)
                except:
                    pass
            elif dns_servers and ':' not in line:
                ip_dns = line.strip()
                try:
                    # Add only IPv4 #26475
                    valid_ip(ip_dns)
                    dns_servers.add(ip_dns)
                except:
                    pass

        if container_ip_dns is not None and  container_ip_dns in dns_servers:
            dns_servers.remove(container_ip_dns)
        return list(dns_servers)

    return []


def list_files(rep, wildcards='*', stripext=False, basename=True, default=[], exception='', substring=True):
    """
    Liste des fichiers dans un répertoire
    @rep: répertoire de recherche
    @wildcards: expression régulière sur les fichiers à rechercher
    @stripext: supprimer l'extension des fichiers
    @basename: ne conserver que le nom court des fichiers
    @default: liste à renvoyer dans le cas où aucun fichier n'est trouvé
    @exception: les fichiers contenant cette chaîne sont ignorés
    @substring: l'exception peut-être une sous-chaîne du nom du fichier
    """
    res = []
    if os.path.isdir(rep):
        for fic in glob.glob(rep+'/%s' % wildcards):
            if exception:
                found = False
                if type(exception) != list:
                    exception = [exception]
                for exc in exception:
                    if substring:
                        if exc in os.path.basename(fic):
                            found = True
                            break
                    elif exc == os.path.basename(fic):
                        found = True
                        break
                if found:
                    continue
            if basename:
                fic = os.path.basename(fic)
            if stripext:
                fic = os.path.splitext(fic)[0]
            res.append(fic)
    if res == []:
        res = default
    res.sort()
    return res

def is_file(filename):
    """ filtre permettant de verifier l'existence
    d'un fichier avant de l'inclure par exemple """
    return os.path.isfile(filename)

def to_list(lst, sep=' '):
    """retourne une liste concatenee de donnees de la liste"""
    return sep.join(lst)

def purge_list(lst):
    """supprime les doublons dans une liste"""
    res = []
    for elt in lst:
        if elt not in res:
            res.append(elt)
    return res

def to_iso(data):
    """encode une chaine en iso"""
    try:
        return unicode2(data, "UTF-8").encode("ISO-8859-1")
    except:
        return data

def to_sql(data):
    """echape les caractères ' et " d'une chaine"""
    try:
        return data.replace("'", "\\\'").replace("\"", "\\\"")
    except:
        return data

def is_installed(package, chroot=None):
    """ vérifie si un paquet est installé ou pas"""
    """FIXME: obsolete, utiliser plutot pyeole.process.is_installed"""
    cmd = ["/usr/bin/dpkg", "-l", package.strip()]
    if chroot != None:
        cmd = ['chroot', chroot] + cmd
    process = subprocess.Popen(args=cmd,
              stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=False)
    code = process.wait()
    if code != 0:
        # paquet non installé
        return False
    dpkg_info = process.stdout.read()
    pkg_status = dpkg_info.strip().split('\n')[-1].split()
    if pkg_status[0].lower() in ['pn', 'un']:
        return False
    return True

def installed_version(package):
    """renvoie la version d'un paquet"""
    process = subprocess.Popen(args=["dpkg", "-l", package.strip()],
              stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=False)
    code = process.wait()
    if code != 0:
        # paquet non installé
        return ''
    dpkg_info = process.stdout.read()
    pkg_status = dpkg_info.strip().split('\n')[-1].split()
    if pkg_status[0].lower() == 'pn':
        # ancien paquet purgé (status = pn)
        return ''
    return pkg_status[2]

def custom_join(multivar, separator=' '):
    """ mise à plat d'une variable multiple"""
    return separator.join([unicode2(i) for i in multivar])

def is_ip(data):
    """vérifie si data décrit une adresse IP"""
    if data == '':
        return False
    try:
        return valid_ip(data)
    except:
        return False


def is_instanciate():
    """
    teste la présence du fichier spécial .instance (#7051)
    """
    return {True: 'oui', False: 'non'}.get(os.path.isfile(INSTANCE_LOCKFILE))


def certs_nuauth():
    """ liste des certificats pour nuauth """
    namecert = ''
    nuauth_tls_key = ''
    rep = '/etc/ipsec.d'
    repdefault = '/etc/nufw/certs/'
    wildcards = '*.pem'
    if os.path.isdir(rep):
        if len(glob.glob(rep+'/%s' % wildcards)) > 0:
            namecert = os.path.basename(glob.glob(rep+'/%s' % wildcards)[0])
            nuauth_tls_key = os.path.join(rep, 'private', 'priv%s' % namecert)
            if not os.path.isfile(nuauth_tls_key):
                nuauth_tls_key = ''
            else:
                nuauth_tls_cert = os.path.join(rep, namecert)
                if not os.path.isfile(nuauth_tls_cert):
                    nuauth_tls_key = ''
                else:
                    nuauth_tls_cacert = os.path.join(rep, 'cacerts', 'CertifCa.pem')
                    if not os.path.isfile(nuauth_tls_cacert):
                        nuauth_tls_key = 'none'
    if nuauth_tls_key == '':
        nuauth_tls_key = os.path.join(repdefault,'nuauth-eole.key')
        nuauth_tls_cert = os.path.join(repdefault,'nuauth-eole.crt')
        nuauth_tls_cacert = '/etc/ssl/certs/ca.crt'
    res = [nuauth_tls_key, nuauth_tls_cert, nuauth_tls_cacert]
    return res

def key_nuauth():
    """retourne la clef privee pour nuauth"""
    return certs_nuauth()[0]

def cert_nuauth():
    """retourne le certificat pour nuauth"""
    return certs_nuauth()[1]

def cacert_nuauth():
    """retourne l'authorité de certification pour nuauth"""
    return certs_nuauth()[2]

def upper(string):
    """ texte en majuscules """
    return string.upper()

def lower(string):
    """ texte en minuscules """
    return string.lower()

def wpkg_crypt(string):
    if string is None:
        return ''
    return wcrypt(string)

def wpkg_decrypt(string):
    if string is None:
        return ''
    return wdecrypt(string)

def get_file_list(string):
    return glob.glob(string)

def escape_regexp(string):
    return string.replace('.', '\.')


DAY_TO_BAREOS = {1: 'mon', 2: 'tue', 3: 'wed', 4: 'thu', 5: 'fri',
                 6: 'sat', 7: 'sun'}
def gen_bareos_schedule(jobs, schedule, extra_jobs=None, suffix=''):
    ret = []
    def max_hour(day, hour):
        if hour <= 12:
            max_hours[day] = max(max_hours[day], hour)

    def min_hour(day, hour):
        if min_hours[day] > 12:
            if hour > 12:
                min_hours[day] = min(min_hours[day], hour)
        else:
            if hour <= 12:
                min_hours[day] = min(min_hours[day], hour)
            else:
                min_hours[day] = hour

    def write_job(day, level, hour, monthly):
        #weeknumber must be 1st for monthly jobs
        #weeknumber must be 2nd-5th if monthly job set same day
        if monthly:
            weeknumber = '1st '
        else:
            if str(day) in monthjobs:
                weeknumber = '2nd-5th '
            else:
                weeknumber = ''
        #if 12 < day < 24: day = yesterday
        #if 0 < day 12: day = today

        schehour = max_hours[day]
        if extra_jobs is not None:
            if (hour > 12 and min_hours[day] > 12) or \
                    (hour <= 12 and min_hours[day] <= 12):
                hour = min(hour, min_hours[day])
            elif min_hours[day] > 12:
                hour = min_hours[day]

        if hour > 12:
            if day == 1:
                newday = 7
            else:
                newday = day-1
        else:
            newday = day
            if schehour < hour:
                schehour = hour

        #could be day after (if hour in [12-24])
        tday = DAY_TO_BAREOS[newday]
        #always value "day"
        scheday = DAY_TO_BAREOS[day]
        schemin = str(schedule['minute'])
        #time must be 4:01 not 4:1
        if len(schemin) == 1:
            schemin = '0{0}'.format(schemin)
        if not monthly:
            parsed_day.append(day)
        ret.append((level, weeknumber, tday, hour, scheday, schehour, schemin))

    #get all monthly job
    monthjobs = []
    for index in range(0, len(jobs.get('job_type' + suffix, []))):
        if jobs['job_type' + suffix][index] == 'monthly':
            monthjobs.append(str(jobs['day' + suffix][index]))

    max_hours = {}
    min_hours = {}
    for day in DAY_TO_BAREOS.keys():
        max_hours[day] = schedule['hour']
        min_hours[day] = schedule['hour']

    if extra_jobs is not None:
        for key, values in extra_jobs.items():
            root, varname = key.rsplit('.', 1)
            if varname.startswith("job_type_"):
                cur_suffix = varname[9:]
                hours = extra_jobs[root + '.hour_' + cur_suffix]
                for index, value in enumerate(values):
                    hour = int(hours[index])
                    day_name = root + '.day_' + cur_suffix
                    if value == 'daily':
                        end_day_name = root + '.end_day_' + cur_suffix
                        for i in range(int(extra_jobs[day_name][index]),
                                       int(extra_jobs[end_day_name][index])+1):
                            max_hour(i, hour)
                            min_hour(i, hour)
                    else:
                        max_hour(int(extra_jobs[day_name][index]), hour)
                        min_hour(int(extra_jobs[day_name][index]), hour)

    parsed_day = []
    for index in range(0, len(jobs.get('job_type' + suffix, []))):
        level = jobs['level' + suffix][index]
        hour = int(jobs['hour' + suffix][index])
        if jobs['job_type' + suffix][index] == 'daily':
            for i in range(int(jobs['day' + suffix][index]),
                    int(jobs['end_day' + suffix][index])+1):
                write_job(i, level, hour, False)
        elif jobs['job_type' + suffix][index] == 'weekly':
            write_job(int(jobs['day' + suffix][index]), level, hour, False)
        elif jobs['job_type' + suffix][index] == 'monthly':
            write_job(int(jobs['day' + suffix][index]), level, hour, True)
        else:
            raise Exception(_(u'Unknown job type'))

    if extra_jobs is not None:
        for day in max_hours.keys():
            if day not in parsed_day:
                hour = min_hours[day]
                write_job(day, 'Full', hour, False)
    return ret


def random_int(vmin, vmax, exclude=None):
    values = list(range(int(vmin), int(vmax)+1))
    if exclude != None and exclude in values:
        values.remove(exclude)
    return random.choice(values)


def list_cdrom_devices(mode_zephir='non'):
    if mode_zephir == 'oui':
        return []
    context = pyudev.Context()
    devices = []
    try:
        cdroms = context.list_devices(ID_CDROM=1)
    except:
        return []
    for cdrom in cdroms:
        devices.append(cdrom['DEVNAME'])
    return devices


def cdrom_minormajor(typ, device):
    stats = os.stat(device)
    if stat.S_ISBLK(stats.st_mode):
        device = stats.st_rdev
#    elif stat.S_ISCHR(stats.st_mode):
#        device = stats.st_rdev
    elif stat.S_ISDIR(stats.st_mode):
        device = stats.st_dev
    else:
        raise ValueError(_(u"Disknod '{0}' is not a directory or"
                         u" a device.").format(device))
    if typ == 'minor':
        return os.minor(device)
    else:
        return os.major(device)


def get_mount_point_device(mount_point):
    """Lookup device informations for a mount point

    :param mount_point: absolute path of the mount point
    :type mount_point: `str`
    :return: device informations: `name`, `uid`, `gid` and `mode`
    :rtype: `dict`
    :raise: `Exception` if :data:`mount_point` is not a directory or
            not mounted.

    """
    if not os.path.isdir(mount_point):
        msg = u'{0} must be a directory.'
        raise Exception(msg.format(mount_point))

    mounts = open(u'/proc/mounts', 'r')
    for line in mounts:
        items = line.split()
        if items[1] == mount_point:
            device = {'name' : items[0]}
            stats = os.stat(os.path.realpath(device['name']))
            device['uid'] = pwd.getpwuid(stats.st_uid).pw_name
            device['gid'] = grp.getgrgid(stats.st_gid).gr_name
            device['mode'] = oct(stats.st_mode & 0o777)
            return device

    msg = u'No device found for mount point {0}.'
    raise Exception(msg.format(mount_point))


def device_type(device):
    stats = os.stat(device)
    if stat.S_ISBLK(stats.st_mode):
        dev_type = u'b'
    elif stat.S_ISCHR(stats.st_mode):
        dev_type = u'c'
    elif stat.S_ISDIR(stats.st_mode):
        dev_type = u'b'
    return dev_type


def calc_webredirection(roundcube="non", force_envole="non"):
    if force_envole=="oui":
        return "/envole"
    elif roundcube=="oui":
        return "/roundcube"
    else:
        return "/"


def calc_smtp_port(tls_smtp='non'):
    if tls_smtp == 'non':
        return 25
    elif tls_smtp == 'port personnalisé':
        return None
    elif tls_smtp.startswith('port '):
        return int(tls_smtp.replace('port ',''))
    raise Exception(_(f'unexpected tls_smtp value "{tls_smtp}"'))


def dn_from_realm(ad_realm):
    if ad_realm is None:
        return ad_realm
    parts = ad_realm.split('.')
    return 'DC={}'.format(',DC='.join(parts))


def activate_master_only_web_app(mode_conteneur_actif=None, activer_nginx_web=None, activer_apache=None):
    """Check if a master only web application can be enabled

    Check if service can be enabled on master only:
    - When container is enabled, activate if `activer_nginx_web` is `oui`
    - All other cases must activate if either `activer_nginx_web` or `apache` is `oui`.

    :param str mode_conteneur_actif: is container mode enabled (`oui` or `non`)
    :param str activer_nginx_web: is publication by Nginx enabled (`oui` or `non`)
    :param str activer_apache: is apache enabled (`oui` or `non`)
    :return str: `oui` if the service can be activated, `non` otherwise

    """
    without_container = (mode_conteneur_actif == 'non' and (activer_apache == 'oui' or activer_nginx_web == 'oui'))
    with_container = (mode_conteneur_actif == 'oui' and activer_nginx_web == 'oui')
    if without_container or with_container:
        return 'oui'
    else:
        return 'non'

def check_file_size_limit(data=None, ctx=None):
    if None in [data, ctx]:
        return
    if int(data) > int(ctx):
        raise ValueError(u"La valeur est supérieure à celle de php_upload_max_filesize. Elle ne sera pas prise en compte et php_upload_max_filesize sera prise en compte par le webmail lors de l'envoi.")


def convert_debit_carte(value, type):
    if type == 'autoneg':
        if value == 'autoneg on':
            return 'oui'
        return 'non'
    elif type == 'speed':
        try:
            return value.split()[1]
        except:
            return None
    try:
        return value.split()[3]
    except:
        return None


def update_repository_source(source, eole_release, current_eole_release):
    if eole_release != current_eole_release and os.path.isdir('/usr/share/eole/upgrade/source.list.d'):
        isbreak = False
        for filename in glob.glob('/usr/share/eole/upgrade/source.list.d/*.txt'):
            with open(filename, 'r') as fh:
                for line in fh.read().split('\n'):
                    if ':' not in line:
                        continue
                    filerelease, old, new = line.split(';', 2)
                    if str(filerelease) == str(eole_release) and str(old) == str(source):
                        source = new
                        isbreak = True
                        break
            if isbreak:
                break

    return source


#########################################################################
# Chargement de fonctions supplémentaires depuis /usr/share/creole/funcs
#########################################################################

def load_funcs(force_reload=False):
    dict_locals = globals()
    for mod_file in glob.glob('%s/*.py' % func_dir):
        try:
            if os.path.basename(mod_file) != '__init__.py':
                name = os.path.splitext(os.path.basename(mod_file))[0]
                if sys.version_info[0] >= 3:
                    spec = importlib.util.spec_from_file_location(name, mod_file)
                    mod_perso = importlib.util.module_from_spec(spec)
                    spec.loader.exec_module(mod_perso)
                else:
                    mod_perso = imp.load_source(name, mod_file)
                for attr in dir(mod_perso):
                    if not attr.startswith('__'):
                        if force_reload or attr not in dict_locals:
                            dict_locals[attr] = getattr(mod_perso, attr)
                        #else:
                        #    # on n'écrase pas les fonctions déjà présentes !
                        #    print "%s - déjà défini : %s" % (mod_file, attr)

        except Exception as err:
            print(_(u"Error loading custom functions in {0}.").format(mod_file))
            if sys.version_info[0] >= 3:
                print(str(err))
            else:
                try:
                    print(str(err).decode('utf8'))
                except UnicodeEncodeError:
                    print(unicode2(err))

def main():
    """fonction de test  de la bibliothèque"""
    try:
        valid_netmask(sys.argv[1])
        print("ok")
    except Exception as msg:
        print("erreur : %s" % str(msg))

load_funcs()

if __name__ == "__main__":
    main()
