#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import sys, getpass, socket
import termios
from pyeole.ihm import print_line
from pyeole.ansiprint import print_orange, print_red
from creole.client import CreoleClient
from collections import OrderedDict


try:
    from zephir.zephir_conf.zephir_conf import adresse_zephir
except:
    client = CreoleClient()
    adresse_zephir = client.get_creole('ssl_server_name')
from zephir.lib_zephir import convert
from zephir.lib_zephir import xmlrpclib
from zephir.lib_zephir import EoleProxy
from zephir.lib_zephir import TransportEole
from zephir.lib_zephir import flushed_input


def display_help():
    print("{} [numero du groupe] [--variable_master \"valeur1|value2\" --variable_slave_1 \"valeur1|value2\" --variable_slave_2 \"valeur1|value2\"]".format(sys.argv[0]))


def argparser():
    # récupère l'ID du groupe + la liste des variables éventuelles
    arglen = len(sys.argv)
    values = OrderedDict()
    if arglen == 1:
        return None, values
    if sys.argv[1] in ['-h', '--help']:
        display_help()
        sys.exit(0)
    try:
        group_id = int(sys.argv[1])
    except:
        print_red("\"{}\" n'est pas un ID de groupe valide".format(sys.argv[1]))
        display_help()
        sys.exit(1)
    if arglen > 2:
        nb_couple = float(arglen - 2) / 2.0
        if nb_couple != float(int(nb_couple)):
            print_red("nombre impaire d'argument il manque un nom de variable ou une valeur")
            display_help()
            sys.exit(1)
        nb_couple = int(nb_couple)
        if nb_couple == 1:
            print_red("une variable maitre uniquement a été déclaré, veuillez spécifier des variables esclaves")
            display_help()
            sys.exit(1)
        len_master = None
        for idx in range(0, nb_couple * 2, 2):
            variable_name = sys.argv[2 + idx]
            if not variable_name.startswith('--'):
                display_help()
                print_red('argument "{}" non valide'.format(variable_name))
            variable_name = variable_name[2:]
            value = sys.argv[3 + idx]
            if len_master is None:
                len_master = len(value.split('|'))
            elif len(value) != len_master:
                print_red('longueur de la variable esclave {} différente de la maitre'.format(variable_name))
                display_help()
                sys.exit(1)
            values[variable_name] = value
    return group_id, values


def main():
    # import des fonctions communes de Zéphir client
    group_id, values = argparser()

    authentified, proxy = get_pwd(adresse_zephir, 7080)
    if authentified == False:
        sys.exit(1)
    # on récupère la liste des groupes
    try:
        groups = {grp_id: (group_name, liste_serveurs) for grp_id, group_name, liste_serveurs in convert(proxy.serveurs.get_groups())[1]}
    except xmlrpclib.ProtocolError:
        print_red("""Erreur de permissions ou Zéphir non disponible""")
        sys.exit(1)
    except socket.error as e:
        print_red("""Erreur de connexion au serveur Zéphir (%s)""" % str(e))
        sys.exit(1)
    # on récupère l'ID du groupe concerné
    if group_id is None:
        print('Liste des groupes :')
        for grp_id, group_infos in groups.items():
            print(' - ' + str(grp_id) + ' : ' + group_infos[0])
        while group_id is None:
            tmp_group_id = flushed_input("Numéro du groupe : ")
            try:
                group_id = int(tmp_group_id)
            except Exception as err:
                print(err)
                continue
            if not group_id in groups:
                print_red('Groupe inconnu')
                group_id = None
    elif not group_id in groups:
        print_red('Le groupe "{}" est inconnu'.format(group_id))
        sys.exit(1)
    # on récupère toutes les master/slaves du groupe de serveurs
    liste_serveurs = groups[group_id][1]
    params = convert(proxy.serveurs.get_groupe_vars(liste_serveurs, True))
    # on demande, si nécessaire, la liste des variables
    if not values:
        print()
        print('Liste des variables dans un groupe :')
        list_vars = params[1][3]
        for var in list_vars:
            print(' - ' + var)
        master_name = None
        while master_name is None:
            master_name = flushed_input("Nom de la variable maitre : ")
            if not master_name in list_vars:
                print('Variable inconnu')
                master_name = None
        values[master_name] = []
        slave_idx = 1
        while True:
            slave_name = flushed_input("Nom de la variable esclave {} (rien pour terminer) : ".format(slave_idx))
            if not slave_name:
                break
            if not slave_name in list_vars:
                print('Variable inconnu')
                continue
            values[slave_name] = ''
            slave_idx += 1
    # on récupère, si nécessaire, les valeurs
    master_name = list(values)[0]
    if not values[master_name]:
        master_value = flushed_input('Valeur de {} (séparateur |) : '.format(master_name))
        if not master_value:
            print_red('un valeur est obligatoire pour la variable maitre')
        values[master_name] = master_value
        for slave_name in list(values)[1:]:
            slave_value = flushed_input('Valeur de {} : '.format(slave_name))
            values[slave_name] = slave_value
    # on envoi les valeurs
    master_value = values.pop(master_name)
    slaves_value = {}
    lenmaster = len(master_value.split('|'))
    for slave_name, slave_value in values.items():
        if lenmaster != len(slave_value.split('|')):
            print_red('longueur de la variable esclave {} différente de la maitre'.format(slave_name))
            exit(1)
        slaves_value[slave_name] = slave_value
    ret = proxy.serveurs.set_groupe_var(liste_serveurs, master_name, master_value, False, slaves_value)
    # on affiche les erreurs si nécessaire
    if ret[1]:
        print_orange('Erreur : ' + str(ret[1]))


def get_pwd(addr, port):
    """lecture d'un login/passwd pour l'application zephir
    """
    login_ok = 0
    user = "toto"
    while login_ok == 0 and user != "":
        try:
            # flush de l'entrée standard au cas où l'utilisateur aurait
            # tapé <entrée> pendant l'Upgrade
            termios.tcflush(sys.stdin, termios.TCIOFLUSH)
        except:
            pass
        user = flushed_input("Entrez votre login zephir (rien pour sortir) : ")
        if user != "":
            passwd = getpass.getpass("Mot de passe zephir pour %s : " % user)
            # création du proxy avec zephir
            proxy = EoleProxy("https://%s:%s@%s:%s" % (user, passwd, addr, port), transport=TransportEole())
            login_ok = 1
            try:
                res = convert(proxy.get_permissions(user))
            except xmlrpclib.ProtocolError:
                login_ok = 0
                print_line("\n Erreur d'authentification \n")
        else:
            return False, "! Abandon de la procédure !"
    return True, proxy


if __name__ == '__main__':
    main()

