#-*-coding:utf-8-*-
"""
    Proxy ldap pour les authentifications simple :

    proxy = BaseAuthProxy(DN('ou=education,o=gouv,c=fr'),
                           'localhost',
                           389,
                           match_attributes=['mail', 'uid'])
    deferred_auth = proxy.authenticate('monlogin_ou_monmail')
"""
from twisted.python import log
from twisted.python.failure import Failure

from ldaptor.protocols.ldap.distinguishedname import DistinguishedName as DN
from ldaptor.protocols.ldap.ldaperrors import LDAPInvalidCredentials

from eoleldaptor.utils import unbind_proto, EOLELDAPEntry
from eoleldaptor.baseproxy import BaseProxy

class BaseAuthProxy(BaseProxy):
    """
        Base ldap authentication proxy
    """
    _default_resp = False

    @staticmethod
    def _connected(proto, userdn, password):
        """
            Bind the to the user's dn with the given password
            @proto : client used to bind
            @userdn : the dn of the given user
            @password : user's password

            @return : deferred binding
        """
        baseentry = EOLELDAPEntry(client=proto,
                              dn=DN(userdn))
        deferred = baseentry.bind(password)
        deferred.addErrback(unbind_proto, proto=proto)
        return deferred

    def bind(self, userdn, password, search_base=None):
        """
            bind a user to his dn
            @userdn: user's dn
            @password: user's password
            @search_base : ldap base for searching user entries
        """
        if userdn:
            def_conn = self.connector.connect(search_base or self.config['base'],
                                              self.config['server'])
            def_conn.addCallback(self._connected,
                                 userdn=userdn,
                                 password=password)
            return def_conn
        else:
            Failure(LDAPInvalidCredentials).raiseException()

    def _bind_eb(self, failure):
        """
            return False if the failure is due to an invalid credential
        """
        if failure.trap(LDAPInvalidCredentials):
            return self._default_resp
        else:
            Failure(failure).raiseException()

    def _clean_errors(self, err, message=None):
        """
            Clean and log errors before responding
            @err : the raised Error
            @message : optionnal loging message

            @return : default response
        """
        if message:
            log.err(message)
        log.err(err.getErrorMessage())
        return self._default_resp

    def is_auth(self, result):
        """
            Test if the result comes from successfull binding
            @return : True/False
        """
        if isinstance(result, EOLELDAPEntry):
            ret_val = unbind_proto(True, result.client)
        else:
            ret_val = self._default_resp
        return ret_val

    def authenticate(self, login, password):
        """
            Authenticate with login/password
            @login: user's login (related to the match attribute)
            return True/False
        """
        d_dn = self.get_user_dn(login)
        d_dn.addCallback(self.bind, password=password)
        d_dn.addErrback(self._bind_eb)
        d_dn.addCallback(self.is_auth)
        d_dn.addErrback(self._clean_errors)
        return d_dn
