#-*-coding:utf-8-*-
"""
    BaseProxy utilisant ldaptor
    Permet de récupérer le dn d'un utilisateur
    proxy = BaseProxy(DN('ou=education,o=gouv,c=fr'),
                           'localhost',
                           389,
                           match_attributes=['mail', 'uid'])
    deferred_dn = proxy.get_user_dn('monlogin_ou_monmail')
"""
from twisted.python import log
from twisted.python.failure import Failure
from twisted.internet.defer import Deferred

from ldaptor.protocols.ldap.distinguishedname import DistinguishedName as DN
from ldaptor.protocols.pureldap import LDAP_SCOPE_wholeSubtree
from ldaptor.ldapfilter import parseFilter, InvalidLDAPFilter

from eoleldaptor.utils import make_ldapconnector, unbind_proto
from eoleldaptor.errors import (MultipleUserFound, NoUserFound,
                                                        ConnectionError)
from utils import EOLELDAPEntry

class BaseProxy(object):
    """
        Base Ldap Proxy
        @base:Ldap base's DN
        @host: the host where the ldap is available
        @port: the port used to connect to the ldap
        @match_attributes: attributes used to match user's account
    """
    def __init__(self, base, host, port, match_attributes="uid", reader_dn=None, reader_pass=None, use_tls=False):
        self.config = dict(base=DN(base),
                           server={DN(base):(host, port,)},
                           host=host,
                           reader_dn=reader_dn,
                           reader_pass=reader_pass,
                           use_tls=use_tls)
        # On construit le filtre ldap de recherche utilisateur
        # on utilise un string-formatting avec des clés de dico
        # pour pouvoir faire :
        # filtre % dict(login=login)
        if not hasattr(match_attributes, '__iter__'):
            match_attributes = [match_attributes]
        self.user_filter = "(|(%s=%%(login)s))" % (
                    "=%(login)s)(".join(match_attributes),)
        self.connector = make_ldapconnector()
        self.connector.use_tls = use_tls

    @staticmethod
    def unicity(ldap_search_result):
        """
            Verify ldap_search_result's unicity and raise if not
            @ldap_search_result : the result

            @return : nothing
         """
        nb_entries = len(ldap_search_result)
        if nb_entries == 0:
            msg = "Cet utilisateur n'existe pas"
            Failure(NoUserFound(msg)).raiseException()
        elif nb_entries  > 1:
            msg = "%d search results for single entry"
            Failure(MultipleUserFound(msg % (nb_entries,))).raiseException()

    def _filter_dn(self, dn_search_result, unicity):
        """
            Extract the user's dn from the ldap dn search
            @dn_search_result : the ldap_search_result

            @return : the user's dn
        """
        if unicity:
            self.unicity(dn_search_result)
            return dn_search_result[0].dn
        else:
            return [user.dn for user in dn_search_result]

    def _entry_search_dn_(self, result):
        search_filter, base = result
        return base.search(filterObject=search_filter,
                           attributes=('dn'),
                           scope=LDAP_SCOPE_wholeSubtree,
                           sizeLimit=2000,
                           sizeLimitIsNonFatal=True)

    def _search_dn(self, proto, login, search_base, unicity):
        """
            Make a search with the given proto for the given login
            @proto : prototype for the ldapclient connection
            @login : user's login (related to the match attribute)
            @return : a dn
        """
        dn = Deferred()
        dn.callback(login)
        dn.addCallback(lambda l:parseFilter(self.user_filter %
                                                      dict(login=l)))
        dn.addErrback(self._build_filter_eb)
        dn.addCallback(lambda f, proto: (f,
                                         EOLELDAPEntry(client=proto,
                                                   dn=search_base)),
                                         proto=proto)
        dn.addCallback(self._entry_search_dn_)
        dn.addBoth(unbind_proto, proto=proto)
        dn.addCallback(self._filter_dn, unicity)
        return dn

    def _build_filter(self, proto=None, login=None):
        """
            build the user search filter
            @proto : the ldap client used to connect
            @login : the user's login

            @return : (proto, the LDAPFilter object)
        """
        searchfilter = parseFilter(self.user_filter % dict(
                                                    login=login))
        if proto:
            return proto, searchfilter
        else:
            return searchfilter

    @staticmethod
    def _build_filter_eb(failure):
        """
            Errback
            @failure : the raised error
        """
        if failure.check(InvalidLDAPFilter):
            Failure(NoUserFound("Cet utilisateur n'existe pas")
                                             ).raiseException()
        else:
            Failure(failure).raiseException()

    @staticmethod
    def _last_eb(failure):
        """
            Clean errors to wrongdn retrieving
            @failure : the raised error
        """
        log.err(failure.getErrorMessage())
        if failure.check(NoUserFound) or failure.check(MultipleUserFound) \
                                         or failure.check(ConnectionError):
            Failure(failure).raiseException()
        else:
            Failure(NoUserFound("Cet utilisateur n'existe pas")
                                             ).raiseException()

    @staticmethod
    def _conn_eb(failure):
        """
            Format Connection Errors
            @failure : the raised error
        """
        log.err(failure.getErrorMessage())
        Failure(ConnectionError("Erreur à la connexion au serveur ldap"))

    def connect(self, search_base=None):
        """
           initiates a connector in anonymous mode
        """
        return self.connector.connectAnonymously(search_base or self.config['search_base'],
                                                 self.config['server'])

    def get_user_dn(self, login, search_base=None, unicity=True):
        """
            Returns user's dn
            @login : the user's login (cf the proxy's match_attributes)
        """
        search_base = search_base or self.config['base']
        # si disponible, on utilise l'utilisateur de lecture
        d_dn = self.connect(search_base)
        d_dn.addErrback(self._conn_eb)
        d_dn.addCallback(self._search_dn, login=login, search_base=search_base, unicity=unicity)
        d_dn.addErrback(self._last_eb)
        return d_dn

    def get_user_data(self, attributes, search_base=None):
        """
            Returns user's datas
            @attributes : the attributes to macth the user
        """
        d_dn = self.connect(search_base)
        d_dn.addErrback(self._conn_eb)
        d_dn.addCallback(self._search_with_filter, attributes=attributes)
        d_dn.addErrback(self._last_eb)
        return d_dn
