#!/usr/bin/python
#-*-coding:utf-8-*-
"""
    Test de la librairie ldaptor
"""
from twisted.python import log
from twisted.internet import reactor
from twisted.python.failure import Failure

from ldaptor.protocols.ldap.ldapconnector import LDAPClientCreator
from ldaptor.protocols.ldap.ldapclient import LDAPClient

from ldaptor.protocols.ldap.distinguishedname import DistinguishedName as DN
from ldaptor.protocols.pureldap import LDAP_SCOPE_wholeSubtree
from ldaptor.ldapfilter import parseFilter, InvalidLDAPFilter
from ldaptor.protocols.ldap.ldaperrors import LDAPInvalidCredentials
from ldaptor.protocols.ldap.ldapsyntax import LDAPEntryWithClient, LDAPEntry

CONFIG = dict(base=DN('ou=education,o=gouv,c=fr'),
              server={DN('ou=education,o=gouv,c=fr'):('127.0.0.1', 389)})

USERGROUPS_FILTER = "(&(objectclass=sambaGroupMapping)\
(objectClass=posixGroup)(memberUid=%s))"
CLASSES_FILTER = "(&(objectclass=sambaGroupMapping)\
(objectClass=posixGroup)(|(%s)))"


class EoleLdapProxy(LdapProxy):
    """
        Proxy to contact a eole ldap Server via twisted-tor
        Authenticate and return user_infos containing
        user's groups and user's groups descriptions
    """

    def __init__(self, base, host, port, match_attributes="uid", \
                                   group_filter=USERGROUPS_FILTER):
        super(EoleLdapProxy, self).__init__(base, host, port, match_attributes)
        self.group_filter = group_filter

    @staticmethod
    def add_groupinfos_inlist(groupinfos, user_groups, infos_groups):
        """
            Rajoute les informations sur les classes de l'utilisateur
        """
        for classe in groupinfos:
            infos_groups[list(classe['cn'])[0].lower()] = classe
        return (user_groups, infos_groups)

    @staticmethod
    def handle_group_list(ldapgroupentries):
        """
            Renvoie la liste des noms de groupes de l'utilisateur
        """
        user_groups = []
        infos_groups = {}
        classes = []
        for group in ldapgroupentries:
            gr_name = list(group['cn'])[0].lower()
            user_groups.append(gr_name)
            infos_groups[gr_name] = dict(group)
            description = list(group['description'])[0]
            if description.startswith('Equipe ') and \
                            gr_name.startswith('profs-'):
                classes.append(gr_name[6:])
        return user_groups, infos_groups, classes

    def add_classes_infos(self, (user_groups, infos_groups, classes), proto):
        """
            Rajoute les informations des classes pour les profs
        """
        if classes:
            print("On a des classes on récupère les informations")
            str_filter = CLASSES_FILTER % (")(".join(
                                            ("cn=%s" % cl for cl in classes)
                                                    ),)
            searchfilter = parseFilter(str_filter)
            baseentry = LDAPEntry(client=proto, dn=self.config['base'])
            deferred = baseentry.search(filterObject=searchfilter,
                                        scope=LDAP_SCOPE_wholeSubtree,
                                        sizeLimit=2000,
                                        sizeLimitIsNonFatal=True)
            deferred.addCallback(self.add_groupinfos_inlist,
                                 user_groups=user_groups,
                                 infos_groups=infos_groups)
            return deferred
        else:
            return (user_groups, infos_groups)

    def search_usergroups(self, proto, login):
        """
            Cherche les groupes de l'utilisateur 'login'
        """
        searchfilter = parseFilter(self.group_filter % (login,))
        baseentry = LDAPEntry(client=proto, dn=self.config['base'])
        deferred = baseentry.search(filterObject=searchfilter,
                                    scope=LDAP_SCOPE_wholeSubtree,
                                    sizeLimit=2000,
                                    sizeLimitIsNonFatal=True)
        deferred.addCallback(self.handle_group_list)
        deferred.addCallback(self.add_classes_infos, proto=proto)
        return deferred

    def collect_usergroups(self, (auth_ok, result), login):
        """
            Renvoie les groupes de l'utilisateur
        """
        if auth_ok:
            def_conn = self.connector.connectAnonymously(self.config['base'],
                                                         self.config['server'])
            def_conn.addCallback(self.search_usergroups, login=login)
            def_conn.addCallback(
                lambda (grs, infos_grs):(auth_ok, result, grs, infos_grs))
            return def_conn
        else:
            return  (auth_ok, result, [])

    @staticmethod
    def build_ret_dict((auth_ok, result, groups, infos_groups)):
        """
            Construit le dictionnaire à renvoyer suite
            à l'authentification de l'utilisateur
        """
        if auth_ok:
            result['user_groups'] = groups
            result['infos_groups'] = infos_groups
            return auth_ok, result
        return False, {}, []

    def authenticate(self, login, password):
        """
            Authenticate
        """
        #step1 get the user dn
        #step2 bind the user on the given dn
        #step3 collect informations
        d_dn = self.get_user_dn(login)
        d_dn.addCallback(self.bind, password=password)
        d_dn.addBoth(self._cred_valid)
        d_dn.addCallback(self.collect_informations, login=login)
        d_dn.addCallback(self.collect_usergroups, login=login)
        d_dn.addCallback(self.build_ret_dict)
        d_dn.addErrback(self._clean_errors, message="Une erreur est survenue")
        return d_dn

def main():
    """
        Fonction utilisée pour les test
    """
    myproxy1 = BaseAuthProxy(CONFIG['base'], 'localhost', 389, 'uid')
    myproxy2 = LdapProxy(CONFIG['base'], 'localhost', 389, 'uid')
    myproxy3 = EoleLdapProxy(CONFIG['base'], 'localhost', 389, 'uid')
    def _show(result):
        """
            Affiche un résultat
        """
        if result[0]:
            print("L'utilisateur a bien été authentifié")
    #        print("Les informations utilisateurs renvoyées : %s" % result[1])
        else:
            print("Erreur à l'authentification de l'utilisateur")

    for i in range(1,2):
        myproxy1.simple_authentication('pprof%d' % i, 'pprof%dlab12;' % i).addCallback(_show)

    for i in range(1, 2):
        defered = myproxy2.authenticate('pprof%d' % i, 'pprof%dlab12;' % i)
        defered.addCallback(_show)

    reactor.callLater(2, reactor.stop)
    reactor.run()

if __name__ == '__main__':
    main()
