#-*-coding:utf-8-*-
"""
    Ldap Proxy to authenticate and retrieve informations onto eole's ldap server
    >>> proxy = EoleLdapProxy(DN('ou=education,o=gouv,c=fr'),
                              'localhost',
                              389,
                              match_attributes=['mail', 'uid'])
    >>> deferred_auth = proxy.authenticate('uid_or_mail', 'password')
    deferred_auth will return
    (True/False, {user_datas_dict})
"""
from twisted.python import log
from twisted.python.failure import Failure
from twisted.internet.defer import Deferred

from ldaptor.protocols.pureldap import LDAP_SCOPE_wholeSubtree
from ldaptor.ldapfilter import parseFilter
from ldaptor.protocols.ldap import fetchschema

from eoleldaptor.utils import unbind_proto
from eoleldaptor.ldapproxy import LdapProxy


from utils import EOLELDAPEntry


USERGROUPS_FILTER = "(&(objectclass=sambaGroupMapping)\
(objectClass=posixGroup)(memberUid=%s))"
CLASSES_FILTER = "(&(objectclass=sambaGroupMapping)\
(objectClass=posixGroup)(|(%s)))"

class EoleLdapProxy(LdapProxy):
    """
        Proxy to contact a eole ldap Server via twisted-tor
        Authenticate and return user_infos containing
        user's groups and user's groups descriptions
    """

    _default_resp = False, {}

    def __init__(self, base, host, port, reader_dn=None, reader_pass=None, \
                 match_attributes="uid", group_filter=USERGROUPS_FILTER, use_tls=False):
        super(EoleLdapProxy, self).__init__(base, host, port, match_attributes, use_tls=use_tls)
        self.group_filter = group_filter
        self.config['reader_dn'] = reader_dn
        self.config['reader_pass'] = reader_pass

    def _anon_connect(self, callb_data=None, search_base=None):
        if isinstance(callb_data, Failure):
            # la configuration de l'utilisateur est invalide, on log et on continue en anonyme
            log.err("Error binding with user {0} (ldap server: {1}), using anonymous mode"\
            .format(self.config['reader_dn'], self.config['host']))
        return self.connector.connectAnonymously(search_base or self.config['search_base'],
                                                 self.config['server'])

    def _reader_connected(self, proto):
        return proto.client

    def connect(self, search_base=None):
        """
           overrides standard connect function to use reader user if available
        """
        if self.config['reader_dn'] and self.config['reader_pass']:
            conn = self.bind(self.config['reader_dn'], self.config['reader_pass'], search_base)
            return conn.addCallbacks(self._reader_connected, self._anon_connect, errbackArgs=[search_base])
        return self._anon_connect(search_base=search_base)

    def add_groupinfos_inlist(self, groupinfos, user_groups, infos_groups, proto):
        """
            Rajoute les informations sur les classes de l'utilisateur
        """
        for classe in groupinfos:
            infos_groups[list(classe['cn'])[0].lower()] = self.convert_ldap_infos(classe)
        return (user_groups, infos_groups, proto)

    def handle_group_list(self, ldapgroupentries):
        """
            Renvoie la liste des noms de groupes de l'utilisateur
        """
        user_groups = []
        infos_groups = {}
        classes = []
        for group in ldapgroupentries:
            gr_name = list(group['cn'])[0].lower()
            user_groups.append(gr_name)
            infos_groups[gr_name] = self.convert_ldap_infos(group)
            if 'description' in group:
                description = list(group['description'])[0]
                if description.startswith('Equipe ') and \
                                gr_name.startswith('profs-'):
                    classes.append(gr_name[6:])
        return user_groups, infos_groups, classes

    def add_classes_infos(self, (user_groups, infos_groups, classes), proto, search_base):
        """
            Rajoute les informations des classes pour les profs
        """
        d = Deferred()
        if classes:
            str_filter = CLASSES_FILTER % (
                    ")(".join( ("cn=%s" % (cl,) for cl in classes)),
                                          )
            d.callback(str_filter)
            d.addCallback(lambda str_f: parseFilter(str_filter))
            d.addErrback(self._build_filter_eb)
            d.addCallback(lambda f_obj, proto: (f_obj,
                            EOLELDAPEntry(client=proto, dn=search_base)),
                            proto=proto
                         )
            d.addCallback(lambda (f_obj, baseentry):baseentry.search(
                                                   filterObject=f_obj,
                                                   scope=LDAP_SCOPE_wholeSubtree,
                                                   sizeLimit=2000,
                                                   sizeLimitIsNonFatal=True))
            d.addCallback(self.add_groupinfos_inlist,
                                 user_groups=user_groups,
                                 infos_groups=infos_groups,
                                 proto=proto)
        else:
            d.callback((user_groups, infos_groups, proto))
        return d

    def collect_usergroups(self, (auth_ok, result), login, search_base):
        """
            Renvoie les groupes de l'utilisateur
        """
        if search_base is None:
            search_base = self.config['base']
        try:
            uid = list(result.get('uid', [None]))[0]
        except:
            uid = None
        # si l'attribut uid n'est pas présent, on ne sait
        # pas récupérer les groupes de l'utilisateur
        if uid is None:
            def_conn = Deferred()
            def_conn.callback((auth_ok, result, [], {}))
        else:
            def_conn = self.connect(search_base)
            def_conn.addCallback(self.search_usergroups,
                                 uid=uid,
                                 search_base=search_base)
            def_conn.addCallback(
                   lambda (grs, infos_grs, proto):((auth_ok, result, grs, infos_grs), proto)
                    )
            def_conn.addBoth(lambda (result, proto):unbind_proto(result, proto=proto))
        return def_conn

    def search_usergroups(self, binding, uid, search_base):
        """
            Search for the user's group
        """
        deferred = Deferred()
        deferred.callback(uid)
        deferred.addCallback(lambda proto:parseFilter(self.group_filter %
                                                             (uid,)))
        deferred.addErrback(self._build_filter_eb)
        deferred.addCallback(lambda f_obj, proto: (f_obj,
                        EOLELDAPEntry(client=proto, dn=search_base)),
                        proto=binding
                     )
        deferred.addCallback(lambda (f, base):base.search(
                                    filterObject=f,
                                    scope=LDAP_SCOPE_wholeSubtree,
                                    sizeLimit=2000,
                                    sizeLimitIsNonFatal=True))

        deferred.addCallback(self.handle_group_list)
        deferred.addCallback(self.add_classes_infos,
                             proto=binding,
                             search_base=search_base)
        return deferred

    def build_ret_dict(self, (auth_ok, result, groups, infos_groups)):
        """
            Construit le dictionnaire à renvoyer suite
            à l'authentification de l'utilisateur
        """
        resp = self._default_resp
        if auth_ok:
            user_dn = str(result.dn).lower()
            result = self.convert_ldap_infos(result)
            result['user_dn'] = user_dn
            result['user_groups'] = groups
            result['infos_groups'] = infos_groups
            resp = (True, result)
        return resp

    def authenticate(self, login, password, search_base=None):
        """
            Authenticate with login and return user attributes
        """
        return self.bind_and_collect(login, password, search_base=search_base)

    def get_user_data(self, search_attrs, search_base=None):
        """
            Authenticate with a dedicated 'reader' user and return
            attributes of the entry matching a filter built with search_atts
        """
        # construction d'un filtre de recherche pour un autre utilisateur (fédération)
        filter_parts = []
        for attr_name, attr_value in search_attrs.items():
            filter_parts.append("(%s=%s)" % (attr_name, attr_value))
        search_filter = "(&%s)" % "".join(filter_parts)
        login = self.config['reader_dn']
        password = self.config['reader_pass']
        return self.bind_and_collect(login, password, search_filter, default_bind=True, search_base=search_base)

    def bind_and_collect(self, login, password, search_filter=None, default_bind=False, search_base=None):
        #step1 get the user dn
        #step2 bind the user on the given dn
        #step3 collect informations
        if default_bind:
            # bind sur la branche de base du proxy
            d_dn = self.bind(login, password=password)
        else:
            # bind sur la branche de recherche de l'utilisateur
            if search_base is None:
                search_base = self.config['base']
            d_dn = self.get_user_dn(login, search_base)
            d_dn.addCallback(self.bind, password=password)
        d_dn.addErrback(self._bind_eb)
        d_dn.addCallback(self._filter_auth)
        d_dn.addCallback(self.collect_informations,
                         login=login,
                         search_filter=search_filter,
                         search_base=search_base)
        d_dn.addCallback(self.collect_usergroups,
                         login=login,
                         search_base=search_base)
        d_dn.addCallback(self.build_ret_dict)
        d_dn.addErrback(self._clean_errors)
        return d_dn
