#-*-coding:utf-8-*-
"""
    LdapProxy able to authenticate and return additional infos
    >>> proxy = LdapProxy(DN('ou=education,o=gouv,c=fr'),
                          'localhost',
                          389,
                          match_attributes=['mail', 'uid'])
    >>> deferred_auth = proxy.authenticate('uid_or_mail', 'password')
    deferred_auth will return
    (True/False, {user_datas_dict})
"""
from twisted.python.failure import Failure
from twisted.internet.defer import Deferred

from ldaptor.protocols.pureldap import LDAP_SCOPE_wholeSubtree, LDAP_SCOPE_baseObject
from ldaptor.protocols.ldap.ldapsyntax import LDAPEntryWithClient
from ldaptor.protocols.ldap.ldaperrors import LDAPInvalidCredentials, LDAPSizeLimitExceeded
from ldaptor.ldapfilter import parseFilter
from ldaptor.protocols.ldap.distinguishedname import DistinguishedName as DN

from eoleldaptor.utils import unbind_proto
from eoleldaptor.baseauthproxy import BaseAuthProxy

class LdapProxy(BaseAuthProxy):
    """
        Proxy to contact a ldap Server via twisted-tor
        Authenticate and return user_infos
    """
    _default_resp = (False, {})

    def _filter_auth(self, result):
        """
            Test authentification
            @result : binding try result
        """
        if isinstance(result, LDAPEntryWithClient):
            return result
        else:
            Failure(LDAPInvalidCredentials()).raiseException()

    def collect_informations(self, binding, login, search_filter=None, search_base=None):
        """
            Returns the user's informations
            @binding : user's account binding
            @login : associated user
        """
        def get_first_res(result):
            """récupère la première entrée de la recherche
            Pour éviter les problèmes de comptes en double, on renvoie
            une erreur si il y a plus d'une entrée
            """
            if len(result) == 1:
                return True, result[0]
            elif len(result) > 1:
                raise LDAPSizeLimitExceeded('More than one user found with filter : %s' % search_filter)
            return False, {}
        if search_filter is None:
            user_filter = self.user_filter % dict(login=login)
        else:
            user_filter = search_filter
        dn = Deferred()
        dn.callback(login)
        dn.addCallback(lambda l:parseFilter(user_filter))
        dn.addErrback(self._build_filter_eb)
        if search_filter:
            # recherche d'un utilisateur différent de celui connecté
            # HACK : ldaptor ne sait que chercher dans le sous arbre de$
            # l'entrée utilisée pour se binder
            if search_base:
                binding.dn = DN(search_base)
            else:
                binding.dn = self.config['base']
        dn.addCallback(lambda f, base:base.search(filterObject=f,
                                    scope=LDAP_SCOPE_wholeSubtree,
                                    sizeLimit=2000,
                                    sizeLimitIsNonFatal=True),
                                    base=binding)
        dn.addCallback(get_first_res)
        dn.addBoth(unbind_proto, proto=binding.client)
        return dn


    @staticmethod
    def convert_ldap_infos(result):
        """
            Convert the returned JournaledLDAPAttributeSet
            attributes to simple lists for compatibility
            and convenient handling
            @res_dict: {'sn':JournaledLDAPAttributeSet(['__Nom_de_famille'])

            @return : {'sn':['__Nom_de_famille']}
        """
        return dict([(key, list(value))for key, value in dict(result).items()])

    def authenticate(self, login, password, search_base=None):
        """
            Authenticate
            @login : user's login (cf match_attributes to know how we match)
            @password : user's password

            @return : (True/False, dictionnary with the user's ldap info)
        """
        #step1 get the user dn
        #step2 bind the user on the given dn
        #step3 collect informations
        d_dn = self.get_user_dn(login)
        d_dn.addCallback(self.bind, password=password)
        d_dn.addErrback(self._bind_eb)
        d_dn.addCallback(self._filter_auth)
        d_dn.addCallback(self.collect_informations, login=login, search_base=search_base)
        d_dn.addErrback(self._clean_errors)
        d_dn.addCallback(lambda (auth_ok, ldap_infos):(auth_ok, self.convert_ldap_infos(ldap_infos)))
        return d_dn
