# -*- coding: utf-8 -*-
'''
Instantiate a file with creole.template.CreoleTemplateEngine
for example::

    eole.file:
        - name: /etc/hosts
        - container: web

'''

from __future__ import absolute_import

import os
import difflib

# Import salt libs
from salt.exceptions import SaltRenderError

import creole.template
import creole.error

def file(name, container='root'):
    '''instanciates a file with Creole, the EOLE templating system

    :param str name: filename
    :param str container: the lxc container target where the file belongs
    :return dict: - diff between the old and the new content if there are any changes
                  - the `result` key contains `False` if there is an error
                    in the templating process, the exception message is stored
                    under the `comment` key
                  - the `result` key contains `None` if the tempalte is disabled
    '''
    ret = {
        'name': name,
        'changes': {},
        'result': False,
        'comment': '',
        'pchanges': {},
        }

    if container != 'root':
        container_comment = ' in container {0}'.format(container)
    else:
        container_comment = ''

    if not os.path.isfile(name):
        old_lines = []
    else:
        old_lines = open(name, 'r').readlines()

    try:
        engine = creole.template.CreoleTemplateEngine()
        engine.instance_file(filename=name, container=container)
        new_lines = open(name, 'r').readlines()

        diff_iter = difflib.unified_diff(old_lines,
                                         new_lines,
                                         fromfile='{0}.orig'.format(name),
                                         tofile=name)

        diff = ''.join(diff_iter)

        if diff:
            ret['result'] = True
            ret['comment'] = 'The file {0} is instantiated{1}'.format(name, container_comment)
            ret['changes'] = {'diff': ''.join(diff)}
        else:
            ret['result'] = True
            msg = "The file {0}{1} is already instantiated"
            ret['comment'] = msg.format(name, container_comment)

    except creole.error.TemplateError as err:
        ret['result'] = False
        ret['comment'] = err.msg
    except creole.error.TemplateDisabled as err:
        ret['result'] = None
        ret['comment'] = err.msg

    return ret

