# -*- coding: utf-8 -*-

import warnings
from collections import OrderedDict
from copy import copy
from tiramisu.option import ChoiceOption, \
                            FloatOption, \
                            PortOption, \
                            DateOption, \
                            DomainnameOption, \
                            OptionDescription
from tiramisu.option.option import _RegexpOption
from tiramisu.setting import undefined
from tiramisu.value import Multi
#, NetworkOption, NetmaskOption, BroadcastOption, DomainnameOption, URLOption,
from tiramisu.error import ValueWarning, PropertiesOptionError
from tiramisu.setting import groups

MODES_LEVEL = ('basic', 'normal', 'expert')
TYPES = {'UnicodeOption': 'string',
         'IntOption': 'number',
         'FloatOption': 'number',
         'IPOption': 'ip',
         'ChoiceOption': 'string',
         'BoolOption': 'boolean',
         'PasswordOption': 'password',
         'EmailOption': 'email',
         'UsernameOption': 'string',
         'FilenameOption': 'string',
         'PortOption': 'number',
         'DateOption': 'date',
         'DomainnameOption': 'domain'
        }
INPUTS = ['UnicodeOption', 'IntOption', 'FloatOption', 'PasswordOption', 'EmailOption',
          'UsernameOption', 'FilenameOption', 'PortOption', 'DomainnameOption']

ACTION_HIDE = ['hidden', 'disabled']


#return always warning (even if same warning is already returned)
warnings.simplefilter("always", ValueWarning)


def tiramisu_copy(val):
    return val


class Callbacks(object):
    def __init__(self, clearable, remotable):
        self.clearable = clearable
        self.remotable = remotable

    def add(self, id_, child, options, form, force_store_value):
        callback, callback_params = child.impl_get_callback()
        if callback is not None:
            if force_store_value and self.clearable != 'all':
                return
            if callback_params == {}:
                if self.remotable == 'none':
                    raise Exception('not remotable')
                form.setdefault(options[child], {})[u'remote'] = True
            else:
                for callbacks in callback_params.values():
                    for callbk in callbacks:
                        if isinstance(callbk, tuple):
                            if callbk[0] is None:  # pragma: optional cover
                                raise Exception('unsupported from now')
                            elif callback.__name__ == 'tiramisu_copy' and \
                                    form[options[callbk[0]]].get(u'remote', False) != True:
                                form.setdefault(options[callbk[0]], {})
                                form[options[callbk[0]]].setdefault(u'copy', []).append(format(options[child]))
                                if self.clearable == 'minimum':
                                    form.setdefault(options[child], {})[u'clearable'] = True
                            elif options.get(callbk[0]) is not None:
                                if self.remotable == 'none':
                                    raise Exception('not remotable')
                                form.setdefault(options[callbk[0]], {})
                                form[options[callbk[0]]][u'remote'] = True


class Requires(object):
    def __init__(self, context, config, remotable):
        self.requires = {}
        self.not_equal = []
        self.options = {}
        self.context = context
        self.config = config
        self.remotable = remotable

    def add(self, id_, child, form):
        #collect id of all options
        self.options[child] = id_
        current_action = None
        for consistency in child._get_consistencies():
            if consistency[0] == '_cons_not_equal':
                options = []
                for option in consistency[1]:
                    options.append(self.options[option])
                self.not_equal.append(options)

        for requires in child.impl_getrequires():
            for require in requires:
                option, expected, action, inverse, \
                    transitive, same_action = require
                option_id = self.options.get(option)
                if option_id is not None and action in ACTION_HIDE:
                    if current_action is None:
                        current_action = action
                    elif current_action != action:
                        if self.remotable == 'none':
                            raise Exception('not remotable')
                        form.setdefault(option_id, {})[u'remote'] = True
                    for exp in expected:
                        if inverse:
                            act = u'show'
                            inv_act = u'hide'
                        else:
                            act = u'hide'
                            inv_act = u'show'
                        self.requires.setdefault(id_, {u'expected': {}})[u'expected'].setdefault(exp, {}).setdefault(act, []).append(option_id)
                        if isinstance(option, ChoiceOption):
                            for value in option.impl_get_values(self.context):
                                if value not in expected:
                                    self.requires.setdefault(id_, {u'expected': {}})[u'expected'].setdefault(value, {}).setdefault(inv_act, []).append(option_id)
                        else:
                            self.requires[id_].setdefault(u'default', {}).setdefault(inv_act, []).append(option_id)

    def is_remote(self, idx, form):
        if self.remotable == 'all':
            return True
        else:
            return form.get(idx) and form[idx].get(u'remote', False)

    def process(self, form, hidden_options):
        dependencies = {}
        notequal = {}
        for idx, values in self.requires.items():
            if u'default' in values:
                for option in values[u'default'].get(u'show', []):
                    if not self.is_remote(option, form):
                        hidden_options[idx] = True
                        dependencies.setdefault(option, {u'default': {}, u'expected': {}})[u'default'].setdefault(u'show', [])
                        if not idx in dependencies[option][u'default'][u'show']:
                            dependencies[option][u'default'][u'show'].append(idx)
                for option in values[u'default'].get(u'hide', []):
                    if u'default' in dependencies.get(option, []) and idx in dependencies[option][u'default'].get(u'show', []):
                        continue
                    if not self.is_remote(option, form):
                        hidden_options[idx] = True
                        dependencies.setdefault(option, {u'default': {}, u'expected': {}})[u'default'].setdefault(u'hide', [])
                        if not idx in dependencies[option][u'default'][u'hide']:
                            dependencies[option][u'default'][u'hide'].append(idx)
            for expected, actions in values[u'expected'].items():
                for option in actions.get(u'show', []):
                    if u'default' in dependencies.get(option, []) and idx in dependencies[option][u'default'].get(u'show', []):
                        continue
                    if not self.is_remote(option, form):
                        hidden_options[idx] = True
                        dependencies.setdefault(option, {u'expected': {}})[u'expected'].setdefault(expected, {}).setdefault(u'show', [])
                        if idx not in dependencies[option][u'expected'][expected][u'show']:
                            dependencies[option][u'expected'][expected][u'show'].append(idx)
                for option in actions.get(u'hide', []):
                    if u'default' in dependencies.get(option, []) and idx in dependencies[option][u'default'].get(u'hide', []):
                        continue
                    if not self.is_remote(option, form):
                        hidden_options[idx] = True
                        dependencies.setdefault(option, {u'expected': {}})[u'expected'].setdefault(expected, {}).setdefault(u'hide', [])
                        if idx not in dependencies[option][u'expected'][expected][u'hide']:
                            dependencies[option][u'expected'][expected][u'hide'].append(idx)
        for path, dependency in dependencies.items():
            form[path][u'dependencies'] = dependency
        for not_equal in self.not_equal:
            for idx in not_equal:
                options = copy(not_equal)
                options.remove(idx)
                form[idx][u'not_equal'] = options


class TiramisuWeb(object):
    def __init__(self, config, clearable="all", remotable="minimum"):
        self.config = config
        self.context = config.cfgimpl_get_context()
        self.form = {}
        self.requires = None
        self.callbacks = None
        self.context.read_write()
        self.hidden_options = {}
        #all, minimum, none
        self.clearable = clearable
        #all, minimum, none
        self.remotable = remotable

    def get_schema(self, descr=None, root=None, id_=-1):
        def get_mode(props):
            for mode in MODES_LEVEL:
                if mode in props:
                    return mode
            return MODES_LEVEL[0]
        def add_help(obj, child):
            hlp = child.impl_get_information(u'help', None)
            if hlp is not None:
                obj[u'help'] = hlp

        schema = OrderedDict()
        if descr is None:
            init = True
            self.form = {}
            descr = self.config.cfgimpl_get_description()
            self.requires = Requires(self.context, self.config, self.remotable)
            self.callbacks = Callbacks(self.clearable, self.remotable)
        else:
            init = False
        for child in descr.impl_getchildren():
            id_ += 1
            name = child.impl_getname()
            if root is None:
                path = name
            else:
                path = root + '.' + name
            props = self.config.cfgimpl_get_settings().getproperties(child, path)
            self.requires.add(path, child, self.form)
            self.callbacks.add(path, child, self.requires.options, self.form, 'force_store_value' in props)
            if isinstance(child, OptionDescription):
                id_, properties = self.get_schema(child, path, id_)
                obj = {u'name': path,
                       u'properties': properties}
                if child.impl_get_group_type() == groups.master:
                    obj[u'type'] = u'array'
                else:
                    obj[u'type'] = u'object'
                obj[u'title'] = child.impl_getdoc()
                mode = get_mode(props)
                if mode:
                    obj[u'mode'] = mode
                add_help(obj, child)
                schema[path] = obj
            else:
                childtype = child.__class__.__name__
                obj = {u'name': path,
                       u'title': child.impl_getdoc(),
                       u'type': TYPES.get(childtype, u'string')}
                obj[u'value'] = self.context.cfgimpl_get_values()._getdefaultvalue(child,
                                                                                   path,
                                                                                   True,
                                                                                   None,
                                                                                   undefined,
                                                                                   True)
                mode = get_mode(props)
                if mode:
                    obj[u'mode'] = mode
                add_help(obj, child)

                is_multi = child.impl_is_multi()
                if is_multi:
                    obj[u'isMulti'] = is_multi

                if u'auto_freeze' in props:
                    obj[u'autoFreeze'] = True

                # apply Option differencies
                if self.clearable == 'all':
                    self.form.setdefault(path, {})[u'clearable'] = True
                if self.remotable == 'all' or child.impl_has_dependency():
                    if self.remotable == 'none':
                        raise Exception('{} is not remotable'.format(child.impl_get_display_name()))
                    self.form.setdefault(path, {})[u'remote'] = True
                if isinstance(child, ChoiceOption):
                    obj[u'enum'] = child.impl_get_values(self.context)
                    self.form.setdefault(path, {})[u'type'] = u'choice'
                elif childtype in INPUTS:
                    self.form.setdefault(path, {})[u'type'] = u'input'
                if isinstance(child, _RegexpOption):
                    self.form.setdefault(path, {})[u'pattern'] = child._regexp.pattern
                if isinstance(child, DomainnameOption):
                    self.form.setdefault(path, {})[u'pattern'] = child._get_extra('_domain_re').pattern
                if isinstance(child, FloatOption):
                    self.form.setdefault(path, {})[u'step'] = u'any'
                if isinstance(child, PortOption):
                    self.form.setdefault(path, {})[u'min'] = child._get_extra(u'_min_value')
                    self.form.setdefault(path, {})[u'max'] = child._get_extra(u'_max_value')
                #FIXME:
                #'properties': child.impl_getproperties()
                #FIXME pour master/slave voir : http://ulion.github.io/jsonform/playground/?example=schema-array
                schema[path] = obj
        if init:
            self.requires.process(self.form, self.hidden_options)
            del self.requires
            return schema
        else:
            return id_, schema

    # propriete:
    #   hidden
    #   mandatory
    #   editable

    # FIXME model:
    # #optionnel mais qui bouge
    # choices/suggests
    # warning
    #
    # #bouge
    # owner
    # properties



    def get_model(self, path=None, force_error_msg=None, force_value=None, root=None):
        settings = self.config.cfgimpl_get_settings()
        setting_properties = settings._getproperties(read_write=False)
        values = self.config.cfgimpl_get_values()
        descr = self.config.cfgimpl_get_description()
        return self._gen_model(descr, settings, values, setting_properties,
                               path, force_error_msg, force_value, root=root)[1]

    def _gen_model(self, descr, settings, values, setting_properties,
                   var_path, force_error_msg, force_value, root=None, id_=-1,
                   hide=False):
        model = []
        for child in descr.impl_getchildren():
            id_ += 1
            #props = settings[child]
            name = child.impl_getname()
            if root is None:
                path = name
            else:
                path = root + '.' + name
            id_ = self._gen_variable(id_, model, child, path, settings, values,
                                     setting_properties, var_path, force_error_msg,
                                     force_value, hide)
        return id_, model

    def _gen_variable(self, id_, model, child, path, settings, values,
                      setting_properties, var_path, force_error_msg, force_value,
                      hide):
        props = settings._getproperties(child, path, read_write=False,
                                        setting_properties=setting_properties)
        if isinstance(child, OptionDescription):
            obj = {u'key': unicode(path)}
            if props != set():
                obj[u'properties'] = list(props)
            if u'hidden' in props or u'disabled' in props:
                obj[u'hidden'] = True
            model.append(obj)
            if not hide:
                hide = self.hidden_options.get(path, False)
            id_, model_ = self._gen_model(child, settings, values, setting_properties,
                                          var_path, force_error_msg, force_value, path, id_,
                                          hide)
            model.extend(model_)
        else:
            obj = {u'key': unicode(path)}
            obj[u'invalid'] = False
            # Should be in form?
            if self.hidden_options.get(path, False):
                obj[u'hide'] = True
            hidden = self.hidden_options.get(path, False) or hide
            self._get_value(child, force_error_msg, var_path, path, force_value, setting_properties, props, hidden, obj)

            if u'mandatory' in props or u'empty' in props:
                obj[u'required'] = True
            if not hide and (u'frozen' in props or u'hidden' in props or u'disabled' in props):
                if not self.hidden_options.get(path, False):
                    obj[u'readOnly'] = True
                obj[u'hidden'] = True
            if props != set():
                obj[u'properties'] = list(props)
            if child.impl_is_master_slaves('slave'):
                owner = []
                if obj[u'value'] is not None:
                    for idx in xrange(len(obj[u'value'])):
                        own = values._getowner(child, path, None, self_properties=props,
                                               validate_properties=False, index=idx)
                        if not isinstance(own, unicode):
                            own = unicode(own)
                        owner.append(own)
            else:
                owner = values._getowner(child, path, None, self_properties=props,
                                         validate_properties=False)
                if not isinstance(owner, unicode):
                    owner = unicode(owner)
            obj[u'owner'] = owner
            model.append(obj)
        return id_

    def _get_value(self, child, force_error_msg, var_path, path, force_value, setting_properties, props, hidden, obj):
        with warnings.catch_warnings(record=True) as warns:
            value = self.context.getattr(path,
                                         returns_raise=True,
                                         _setting_properties=setting_properties,
                                         _self_properties=props)
            if isinstance(value, Multi):
                value = list(value)
        if not child.impl_is_master_slaves('slave'):
            obj.update(self._manage_value(child, value, force_error_msg, var_path, path,
                                          force_value, props, hidden, warns))
        else:
            for key in [u'value', u'error', u'invalid', u'warnings', u'hasWarnings']:
                obj[key] = []
            if not isinstance(value, Exception):
                for val in value:
                    ret = self._manage_value(child, val, force_error_msg, var_path, path, force_value,
                                             props, hidden, warns)
                    for key in [u'value', u'error', u'invalid', u'warnings', u'hasWarnings']:
                        obj[key].append(ret.get(key))

    def _manage_value(self, child, value, force_error_msg, var_path, path, force_value, props,
                      hidden, warns):
        ret = {}
        if (force_error_msg is not None and var_path == path) or isinstance(value, Exception):
            if isinstance(value, Exception):
                ret = self._get_value_with_exception(child, value, props, hidden)
            else:
                #do not reset value
                ret[u'value'] = force_value
                ret[u'error'] = force_error_msg
                ret[u'invalid'] = True
        else:
            ret[u'value'] = value
            if warns != []:
                ret[u'warnings'] = "\n".join([str(warn.message) for warn in warns])
                ret[u'hasWarnings'] = True
        return ret

    def _get_value_with_exception(self, child, value, props, hidden):
        ret = {}
        if isinstance(value, PropertiesOptionError):
            props |= set(value.proptype)
            if not hidden:
                ret[u'error'] = str(value).decode('utf8')
                ret[u'invalid'] = True
                if child.impl_is_multi():
                    ret[u'value'] = []
                else:
                    ret[u'value'] = None
            else:
                ret[u'value'] = child.impl_getdefault()
        else:
            ret[u'error'] = str(value).decode('utf8')
            ret[u'invalid'] = True
            if child.impl_is_multi():
                ret[u'value'] = []
            else:
                ret[u'value'] = None
        return ret

    def get_form(self, form):
        ret = []
        buttons = []
        dict_form = {}
        for form_ in form:
            if u'path' in form_:
                dict_form.setdefault(u'key')[form_[u'path']] = form_
            elif form_.get(u'type') == u'submit':
                buttons.append(form_)
            else:
                raise Exception('unknown form {}'.format(form_))

        for key, form_ in self.form.items():
            form_[u'key'] = key
            if key in dict_form:
                form_[u'key'].update(dict_form[key])
            ret.append(form_)
        ret.extend(buttons)
        return ret

    def get_values(self):
        return self.config.cfgimpl_get_values()._p_.exportation(None)

    def set_values(self, values):
        return self.config.cfgimpl_get_values()._p_.importation(values)

    def apply_updates(self, body):
        updates = body.get(u'updates')
        if u'model' in body:
            model_ori = body[u'model']
        else:
            model_ori = self.get_model(root=self.config.cfgimpl_get_path())
        self.hidden_options = {}
        for option in model_ori:
            if option.get(u'hide', False):
                self.hidden_options[option[u'key']] = True
        settings = self.config.cfgimpl_get_settings()
        setting_properties = settings._getproperties(read_write=False)
        values = self.config.cfgimpl_get_values()
        if updates:
            for update in updates:
                path = update[u'name']
                oripath = self.config.cfgimpl_get_path()
                if oripath is not None and not path.startswith(self.config.cfgimpl_get_path()):
                    raise Exception('not in current area')
                for idx, mod in enumerate(model_ori):
                    if mod[u'key'] == path:
                        break
                else:
                    raise Exception('unknown model')
                child = self.context.unwrap_from_path(path)
                if update[u'action'] == 'modify':
                    with warnings.catch_warnings():
                        warnings.simplefilter("ignore")
                        if child.impl_is_multi():
                            multi = getattr(self.context, path)
                            if len(multi) == 0 and idx == 0:
                                multi.append(update[u'value'])
                            else:
                                multi[int(update[u'index'])] = update[u'value']
                        else:
                            setattr(self.context, path, update[u'value'])
                elif update[u'action'] == 'delete':
                    if child.impl_is_multi() and u'index' in update:
                        multi = getattr(self.context, path)
                        multi.pop(update[u'index'])
                    else:
                        delattr(self.context, path)
                elif update[u'action'] == 'add':
                    if child.impl_is_multi():
                        getattr(self.context, path).append(update[u'value'])
                    else:
                        raise Exception('unknown action')
                else:
                    raise Exception('unknown action')
                model = []
                self._gen_variable(path, model, child, path, settings, values,
                                   setting_properties, path, None, update.get('value'), False)
                model = model[0]
                model[u'owner'] = u'temporary'
                model_ori[idx] = model
        return model_ori

    def get_jsonform(self, form):
        ret = {u'schema': self.get_schema(root=self.config.cfgimpl_get_path()),
               u'form': self.get_form(form),
               u'model': self.get_model(root=self.config.cfgimpl_get_path())}
        if self.remotable != 'none':
            ret[u'tiramisu'] = self.get_values()
        #import pprint
        #pp = pprint.PrettyPrinter(indent=1)
        #pp.pprint(ret)

        return ret

    def set_updates(self, body, path=None, value=None, index=None):
        if u'tiramisu' in body:
            self.set_values(body[u'tiramisu'])
        old_model = self.apply_updates(body)
        if path is not None:
            if value == '':
                value = None
            with warnings.catch_warnings():
                warnings.simplefilter("ignore")
                try:
                    if index is None:
                        setattr(self.context, path, value)
                    else:
                        getattr(self.context, path)[index] = value
                except ValueError as err:
                    import traceback
                    traceback.print_exc()
                    error_msg = str(err)
                else:
                    error_msg = None
        else:
            error_msg = None
            value = None

        new_model = self.get_model(root=self.config.cfgimpl_get_path(),
                                   path=path,
                                   force_error_msg=error_msg,
                                   force_value=value)
        diff = list(list_diff(old_model, new_model))
        values = {u'updates': diff}
        if self.remotable != 'none':
            values[u'model'] = new_model
            values[u'tiramisu'] = self.get_values()
        return values

    def append_value(self, body, path):
        if u'tiramisu' in body:
            self.set_values(body[u'tiramisu'])
        old_model = self.apply_updates(body)

        multi = getattr(self.context, path)
        multi.append()
        new_model = self.get_model(root=self.config.cfgimpl_get_path())
        diff = list(list_diff(old_model, new_model))

        ret = {u'updates': diff}
        if self.remotable != 'none':
            ret[u'model'] = new_model
            ret[u'tiramisu'] = self.get_values()
        return ret

    def del_value(self, body, path, index=None):
        if u'tiramisu' in body:
            self.set_values(body[u'tiramisu'])
        old_model = self.apply_updates(body)

        if index is not None:
            multi = getattr(self.context, path)
            multi.pop(index)
        else:
            homeconfig, name = self.context.cfgimpl_get_home_by_path(path)
            homeconfig.__delattr__(name)
        new_model = self.get_model(root=self.config.cfgimpl_get_path())
        diff = list(list_diff(old_model, new_model))

        ret = {u'updates': diff}
        if self.remotable != 'none':
            ret[u'model'] = new_model
            ret[u'tiramisu'] = self.get_values()
        return ret


def list_diff(lst_a, lst_b):
    for idx, val in enumerate(lst_a):
        ret = dict_diff(val, lst_b[idx])
        if ret != {}:
            ret[u'key'] = val[u'key']
            yield ret

def dict_diff(dict_a, dict_b):
    if dict_a[u'key'] != dict_b[u'key']:
        raise Exception('hu?')
    dico = {}
    keys = set(list(dict_a.keys()) + list(dict_b.keys()))
    for key in keys:
        if key in dict_a:
            if not key in dict_b:
                if dict_a[key] is True:
                    dico[key] = False
                elif dict_a[key] is False:
                    dico[key] = True
                else:
                    dico[key] = None
            elif dict_a[key] != dict_b[key]:
                dico[key] = dict_b[key]
        elif key in dict_b and (not key in dict_a or dict_a[key] != dict_b[key]):
            dico[key] = dict_b[key]
    return dico
