/***************************************************************************

  a7800.c

  Driver file to handle emulation of the Atari 7800.

  Dan Boris

    2002/05/13 kubecj   added more banks for bankswitching
                            added PAL machine description
                            changed clock to be precise

    2012/10/25 Robert Tuccitto  NTSC Color Generator utilized for
                color palette with hue shift/start
                based on observation of several
                systems across multiple displays

    2012/11/09 Robert Tuccitto  Fixed 3 degree hue begin point
                miscalculation of color palette

    2012/12/05 Robert Tuccitto  Implemented proper IRE and phase
               value to the palette

    2012/12/14 Robert Tuccitto  Adjusted colorburst/tint/hue of entire
               palette to closer reflect default hardware configuration
               setting of ~180 degrees.  Palette settings now correspond
               documented and calculated settings as follows:

               Contrast = 0.0526 --> 0.05
               Brightness = 0.0 --> 0.00
               Color = 0.2162 --> 0.22
               Phase = 25.714 --> 25.7
               Colorburst/Hue = 180 degrees

    2013/02/27 Robert Tuccitto  Palette rebuild due to misaligned
               data references.  Corrected PAL color sequence order.

    2013/03/19 Robert Tuccitto  Stripped palette to raw video output
               values removing YIQ/YUV infer data.

    2013/04/02 Robert Tuccitto  Corrected rotation values and errors
               including duplicate entries for palette.

    2013/04/07 Robert Tuccitto  Address map locations for the XBOARD
               added.

    2013/05/01 Robert Tuccitto  Red and Blue miscalculated proportions
               fixed.

    2013/08/04 Robert Tuccitto  Green miscalculated proportions fixed.

    2013/08/13 Robert Tuccitto  Normalized contrast and brightness,
               providing a standardize grayscale and adjusted color values.

    2013/09/02 Robert Tuccitto  Stored data for 26.7 & 27.7 phase shifts
               with corrections and label for 25.7 values. Made 26.7
               (medium) default. Phase shifting falls outside the realm of
               video controls and hope to implement a selectable toggle
               hardware option similar to Donkey Kong TKG02/TKG04.

    2013/09/19 Robert Tuccitto  Cleanup of Address Maps, high score maps
               added.

    2013/10/16 Robert Tuccitto  Added Phase Shifts 24.7, 25.2, 26.2, 27.2.
               Phase Shifts 24.7 through 27.7 degrees with 0.5 degree
               increments documented. Phase Shift 26.2 degrees made active.
               Fixed typo under 26.7 7$.

    2013/10/27 Robert Tuccitto  Modernized screen parameters for NTSC & PAL.

    2013/11/03 Robert Tuccitto Fixed correctly typo under 26.7 7$.

    2014/11/23 Robert Tuccitto Added NTSC Palette Notes
***************************************************************************/

#include "emu.h"
#include "cpu/m6502/m6502.h"
#include "sound/pokey.h"
#include "sound/tiaintf.h"
#include "imagedev/cartslot.h"
#include "machine/6532riot.h"
#include "includes/a7800.h"


#define A7800_NTSC_Y1   XTAL_14_31818MHz
#define CLK_PAL 1773447


/***************************************************************************
    ADDRESS MAPS
***************************************************************************/

static ADDRESS_MAP_START( a7800_mem, AS_PROGRAM, 8, a7800_state )
	AM_RANGE(0x0000, 0x001f) AM_MIRROR(0x300) AM_READWRITE(a7800_TIA_r, a7800_TIA_w)
	AM_RANGE(0x0020, 0x003f) AM_MIRROR(0x300) AM_READWRITE(a7800_MARIA_r, a7800_MARIA_w)
	AM_RANGE(0x0040, 0x00ff) AM_READ_BANK("bank5") AM_WRITE(a7800_RAM0_w)   /* RAM (6116 block 0) */
	AM_RANGE(0x0140, 0x01ff) AM_RAMBANK("bank6")    /* RAM (6116 block 1) */
	AM_RANGE(0x0280, 0x02ff) AM_DEVREADWRITE("riot", riot6532_device, read, write)
	AM_RANGE(0x0450, 0x045f) /* XBOARD POKEY1 */
	AM_RANGE(0x0460, 0x046f) /* XBOARD POKEY2 */
	AM_RANGE(0x0470, 0x047f) /* XBOARD CTRL */
	AM_RANGE(0x0480, 0x04ff) AM_MIRROR(0x100) AM_RAM    /* RIOT RAM */
	AM_RANGE(0x1000, 0x17ff) AM_RAM /* hs SRAM */
	AM_RANGE(0x1800, 0x27ff) AM_RAM
	AM_RANGE(0x2800, 0x2fff) AM_RAMBANK("bank7")    /* MAINRAM */
	AM_RANGE(0x3000, 0x37ff) AM_RAMBANK("bank7")    /* MAINRAM */
	AM_RANGE(0x3800, 0x3fff) AM_RAMBANK("bank7")    /* MAINRAM */
	AM_RANGE(0x3000, 0x3fff) AM_ROM  /* hs ROM space */
	AM_RANGE(0x4000, 0x7fff) AM_ROMBANK("bank1")                        /* f18 hornet */
	AM_RANGE(0x4000, 0xffff) AM_WRITE(a7800_cart_w) /* XBOARD SRAM */
	AM_RANGE(0x8000, 0x9fff) AM_ROMBANK("bank2")                        /* sc */
	AM_RANGE(0xa000, 0xbfff) AM_ROMBANK("bank3")                        /* sc + ac */
	AM_RANGE(0xc000, 0xdfff) AM_ROMBANK("bank4")                        /* ac */
	AM_RANGE(0xe000, 0xffff) AM_ROM
ADDRESS_MAP_END


/***************************************************************************
    INPUT PORTS
***************************************************************************/

static INPUT_PORTS_START( a7800 )
	PORT_START("joysticks")            /* IN0 */
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP)    PORT_PLAYER(2) PORT_8WAY
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN)  PORT_PLAYER(2) PORT_8WAY
	PORT_BIT(0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT)  PORT_PLAYER(2) PORT_8WAY
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT) PORT_PLAYER(2) PORT_8WAY
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_UP)    PORT_PLAYER(1) PORT_8WAY
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN)  PORT_PLAYER(1) PORT_8WAY
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT)  PORT_PLAYER(1) PORT_8WAY
	PORT_BIT(0x80, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT) PORT_PLAYER(1) PORT_8WAY

	PORT_START("buttons")              /* IN1 */
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_BUTTON1)       PORT_PLAYER(2)
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_BUTTON1)       PORT_PLAYER(1)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_BUTTON2)       PORT_PLAYER(2)
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_BUTTON2)       PORT_PLAYER(1)
	PORT_BIT(0xF0, IP_ACTIVE_LOW, IPT_UNUSED)

	PORT_START("vblank")               /* IN2 */
	PORT_BIT(0x7F, IP_ACTIVE_LOW, IPT_UNUSED)
	PORT_BIT(0x80, IP_ACTIVE_HIGH, IPT_CUSTOM) PORT_VBLANK("screen")

	PORT_START("console_buttons")      /* IN3 */
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_OTHER)  PORT_NAME("Reset")         PORT_CODE(KEYCODE_R)
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_OTHER)  PORT_NAME("Select")        PORT_CODE(KEYCODE_S)
	PORT_BIT(0x04, IP_ACTIVE_LOW, IPT_UNUSED)
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_OTHER)  PORT_NAME(DEF_STR(Pause))  PORT_CODE(KEYCODE_O)
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_UNUSED)
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_UNUSED)
	PORT_DIPNAME(0x40, 0x40, "Left Difficulty Switch")
	PORT_DIPSETTING(0x40, "A" )
	PORT_DIPSETTING(0x00, "B" )
	PORT_DIPNAME(0x80, 0x80, "Right Difficulty Switch")
	PORT_DIPSETTING(0x80, "A" )
	PORT_DIPSETTING(0x00, "B" )
INPUT_PORTS_END

/***************************************************************************
Atari 7800 NTSC Palette Notes:

Palette on a modern flat panel display (LCD, LED, Plasma, etc.) appears
different from a traditional CRT. The most outstanding difference is Hue 1x,
the hue begin point. Hue 1x looks very 'green' (~-60 to -45 degrees -
depending on how poor or well it handles the signal conversion and its
calibration) on a modern flat panel display, as opposed to 'gold' (~-33
degrees) on a CRT.

The system's pot adjustment manually manipulates the ratio of blue to
green/blue to red, while the system 'warming-up' causes the palette phase
shift to go higher in degrees.

At power on, the system's phase shift appears as low as ~23 degrees and
after a considerable consistent runtime, can be as high as ~28 degrees.

In general, the low end of ~23 degrees lasts for maybe several seconds,
whereas higher values such as ~25-27 degrees is the most dominant during
system run time.  180 degrees colorburst takes place at ~25.7 degrees (A
near exact match of Hue 1x and 15x - To the naked eye they appear to be
the same).

However, if the system is adjusted within the first several minutes of
running, the warm up, consistent system run time, causes Hue 15x (F$) to
become stronger/darker gold (More brown then ultimately red-brown); as well
as leans Hue 14x (E$) more brown than green.  Once achieving a phase shift
of 27.7, Hue 14x (E$) and Hue 15x (F$) near-exact match Hue 1x and 2x
respectively.

Therefore, an ideal phase shift while accounting for the reality of
shifting while warming up, as well as maintaining differences between 1x,
2x and 14x, 15x, would likely fall between a 25.7 and 27.7. Phase shifts
26.2 degrees and 26.7 degrees places Hue 15x (F$) between Hue 1x and
Hue 2x, having 26.2 degrees leaning closer to Hue 1x and 26.7 degrees
leaning closer to Hue 2x.

The above notion would also harmonize with what has been documented for
the colors of 1x, 2x, 14x, 15x on the 7800.  1x = Gold, 2x = Orange,
14x (E$) = Orange-Green. 15x (F$) = Light Orange.  Color descriptions are
best measured in the middle of the brightness scale.  It should be
mentioned that Green-Yellow is referenced at Hue 13x (D$), nowhere near
Hue 1x.  A Green-Yellow Hue 1x is how the palette is manipulated and
modified (in part) under a modern flat panel display.

Additionally, the blue to red (And consequently blue to green) ratio
proportions may appear different on a modern flat panel display than a CRT
in some instances for the Atari 7800 system.  Furthermore, you may have
some variation of proportions even within the same display type.

One side effect of this on the console's palette is that some values of
red may appear too pinkish - Too much blue to red.  This is not the same
as a traditional tint-hue control adjustment; rather, can be demonstrated
by changing the blue ratio values via MESS HLSL settings.

Lastly, the Atari 2600 & 5200 NTSC color palettes hold the same hue
structure order and have similar appearance differences that are dependent
upon display type.
***************************************************************************/
/***************************************************************************
    PALETTE - 26.2 PHASE SHIFT
***************************************************************************/

#define NTSC_GREY \
	MAKE_RGB(0x00,0x00,0x00), MAKE_RGB(0x11,0x11,0x11), MAKE_RGB(0x22,0x22,0x22), MAKE_RGB(0x33,0x33,0x33), \
	MAKE_RGB(0x44,0x44,0x44), MAKE_RGB(0x55,0x55,0x55), MAKE_RGB(0x66,0x66,0x66), MAKE_RGB(0x77,0x77,0x77), \
	MAKE_RGB(0x88,0x88,0x88), MAKE_RGB(0x99,0x99,0x99), MAKE_RGB(0xAA,0xAA,0xAA), MAKE_RGB(0xBB,0xBB,0xBB), \
	MAKE_RGB(0xCC,0xCC,0xCC), MAKE_RGB(0xDD,0xDD,0xDD), MAKE_RGB(0xEE,0xEE,0xEE), MAKE_RGB(0xFF,0xFF,0xFF   )

#define NTSC_GOLD \
	MAKE_RGB(0x1A,0x07,0x00), MAKE_RGB(0x2B,0x18,0x00), MAKE_RGB(0x3C,0x29,0x00), MAKE_RGB(0x4D,0x3A,0x00), \
	MAKE_RGB(0x5E,0x4B,0x00), MAKE_RGB(0x6F,0x5C,0x00), MAKE_RGB(0x80,0x6D,0x00), MAKE_RGB(0x91,0x7E,0x09), \
	MAKE_RGB(0xA2,0x8F,0x1A), MAKE_RGB(0xB3,0xA0,0x2B), MAKE_RGB(0xC4,0xB1,0x3C), MAKE_RGB(0xD5,0xC2,0x4D), \
	MAKE_RGB(0xE6,0xD3,0x5E), MAKE_RGB(0xF7,0xE4,0x6F), MAKE_RGB(0xFF,0xF5,0x83), MAKE_RGB(0xFF,0xF7,0x97   )

#define NTSC_ORANGE \
	MAKE_RGB(0x31,0x00,0x00), MAKE_RGB(0x42,0x06,0x00), MAKE_RGB(0x53,0x17,0x00), MAKE_RGB(0x64,0x28,0x00), \
	MAKE_RGB(0x75,0x39,0x00), MAKE_RGB(0x86,0X4A,0x00), MAKE_RGB(0x97,0x5B,0x0B), MAKE_RGB(0xA8,0x6C,0x1C), \
	MAKE_RGB(0xB9,0x7D,0x2D), MAKE_RGB(0xCA,0x8E,0x3E), MAKE_RGB(0xDB,0x9F,0x4F), MAKE_RGB(0xEC,0xB0,0x60), \
	MAKE_RGB(0xFD,0xC1,0x71), MAKE_RGB(0xFF,0xD2,0x86), MAKE_RGB(0xFF,0xE3,0x9D), MAKE_RGB(0xFF,0xF4,0xB3   )

#define NTSC_RED_ORANGE \
	MAKE_RGB(0x3E,0x00,0x00), MAKE_RGB(0x4F,0x00,0x00), MAKE_RGB(0x60,0x08,0x00), MAKE_RGB(0x71,0x19,0x00), \
	MAKE_RGB(0x82,0x2A,0x0F), MAKE_RGB(0x93,0x3B,0x20), MAKE_RGB(0xA4,0x4C,0x31), MAKE_RGB(0xB5,0x5D,0x42), \
	MAKE_RGB(0xC6,0x6E,0x53), MAKE_RGB(0xD7,0x7F,0x64), MAKE_RGB(0xE8,0x90,0x75), MAKE_RGB(0xF9,0xA1,0x86), \
	MAKE_RGB(0xFF,0xB2,0x9A), MAKE_RGB(0xFF,0xC3,0xB0), MAKE_RGB(0xFF,0xD4,0xC6), MAKE_RGB(0xFF,0xE5,0xDC   )

#define NTSC_PINK \
	MAKE_RGB(0x3E,0x00,0x06), MAKE_RGB(0x4F,0x00,0x12), MAKE_RGB(0x60,0x00,0x1E), MAKE_RGB(0x71,0x0E,0x2E), \
	MAKE_RGB(0x82,0x1F,0x3F), MAKE_RGB(0x93,0x30,0x50), MAKE_RGB(0xA4,0x41,0x61), MAKE_RGB(0xB5,0x52,0x72), \
	MAKE_RGB(0xC6,0x63,0x83), MAKE_RGB(0xD7,0x74,0x94), MAKE_RGB(0xE8,0x85,0xA5), MAKE_RGB(0xF9,0x96,0xB6), \
	MAKE_RGB(0xFF,0xA7,0xCB), MAKE_RGB(0xFF,0xB8,0xE1), MAKE_RGB(0xFF,0xC9,0xEF), MAKE_RGB(0xFF,0xDA,0xF4   )

#define NTSC_PURPLE \
	MAKE_RGB(0x32,0x00,0x38), MAKE_RGB(0x43,0x00,0x44), MAKE_RGB(0x54,0x00,0x50), MAKE_RGB(0x65,0x0C,0x5F), \
	MAKE_RGB(0x76,0x1D,0x70), MAKE_RGB(0x87,0x2E,0x81), MAKE_RGB(0x98,0x3F,0x92), MAKE_RGB(0xA9,0x50,0xA3), \
	MAKE_RGB(0xBA,0x61,0xB4), MAKE_RGB(0xCB,0x72,0xC5), MAKE_RGB(0xDC,0x83,0xD6), MAKE_RGB(0xED,0x94,0xE4), \
	MAKE_RGB(0xFE,0xA5,0xE4), MAKE_RGB(0xFF,0xB6,0xE9), MAKE_RGB(0xFF,0xC7,0xEE), MAKE_RGB(0xFF,0xD8,0xF3   )

#define NTSC_PURPLE_BLUE \
	MAKE_RGB(0x1B,0x00,0x5F), MAKE_RGB(0x2C,0x00,0x6B), MAKE_RGB(0x3D,0x00,0x77), MAKE_RGB(0x4E,0x11,0x88), \
	MAKE_RGB(0x5F,0x22,0x99), MAKE_RGB(0x70,0x33,0xAA), MAKE_RGB(0x81,0x44,0xBB), MAKE_RGB(0x92,0x55,0xCC), \
	MAKE_RGB(0xA3,0x66,0xDD), MAKE_RGB(0xB4,0x77,0xED), MAKE_RGB(0xC5,0x88,0xED), MAKE_RGB(0xD6,0x99,0xED), \
	MAKE_RGB(0xE7,0xAA,0xED), MAKE_RGB(0xF8,0xBB,0xED), MAKE_RGB(0xFF,0xCC,0xF0), MAKE_RGB(0xFF,0xDD,0xF5   )

#define NTSC_BLUE1 \
	MAKE_RGB(0x00,0x00,0x72), MAKE_RGB(0x10,0x00,0x7E), MAKE_RGB(0x21,0x0D,0x8E), MAKE_RGB(0x32,0x1E,0x9F), \
	MAKE_RGB(0x43,0x2F,0xB0), MAKE_RGB(0x54,0x40,0xC1), MAKE_RGB(0x65,0x51,0xD2), MAKE_RGB(0x76,0x62,0xE3), \
	MAKE_RGB(0x87,0x73,0xF4), MAKE_RGB(0x98,0x84,0xF9), MAKE_RGB(0xA9,0x95,0xF9), MAKE_RGB(0xBA,0xA6,0xF9), \
	MAKE_RGB(0xCB,0xB7,0xF9), MAKE_RGB(0xDC,0xC8,0xF9), MAKE_RGB(0xED,0xD9,0xF9), MAKE_RGB(0xFE,0xEA,0xF9   )

#define NTSC_BLUE2 \
	MAKE_RGB(0x00,0x00,0x65), MAKE_RGB(0x00,0x0C,0x7A), MAKE_RGB(0x05,0x1D,0x8E), MAKE_RGB(0x16,0x2E,0x9F), \
	MAKE_RGB(0x27,0x3F,0xB0), MAKE_RGB(0x38,0x50,0xC1), MAKE_RGB(0x49,0x61,0xD2), MAKE_RGB(0x5A,0x72,0xE3), \
	MAKE_RGB(0x6B,0x83,0xF4), MAKE_RGB(0x7C,0x94,0xFF), MAKE_RGB(0x8D,0xA5,0xFF), MAKE_RGB(0x9E,0xB6,0xFF), \
	MAKE_RGB(0xAF,0xC7,0xFF), MAKE_RGB(0xC0,0xD8,0xFF), MAKE_RGB(0xD1,0xE9,0xFF), MAKE_RGB(0xE2,0xFA,0xFF   )

#define NTSC_LIGHT_BLUE \
	MAKE_RGB(0x00,0x0D,0x48), MAKE_RGB(0x00,0x1E,0x5E), MAKE_RGB(0x00,0x2F,0x74), MAKE_RGB(0x00,0x40,0x8A), \
	MAKE_RGB(0x11,0x51,0x9B), MAKE_RGB(0x22,0x62,0xAC), MAKE_RGB(0x33,0x73,0xBD), MAKE_RGB(0x44,0x84,0xCE), \
	MAKE_RGB(0x55,0x95,0xDF), MAKE_RGB(0x66,0xA6,0xF0), MAKE_RGB(0x77,0xB7,0xFF), MAKE_RGB(0x88,0xC8,0xFF), \
	MAKE_RGB(0x99,0xD9,0xFF), MAKE_RGB(0xAA,0xEA,0xFF), MAKE_RGB(0xBB,0xFB,0xFF), MAKE_RGB(0xCC,0xFF,0xFF   )

#define NTSC_TURQUOISE \
	MAKE_RGB(0x00,0x1C,0x1C), MAKE_RGB(0x00,0x2D,0x32), MAKE_RGB(0x00,0x3E,0x49), MAKE_RGB(0x00,0x4F,0x5F), \
	MAKE_RGB(0x05,0x60,0x73), MAKE_RGB(0x16,0x71,0x84), MAKE_RGB(0x27,0x82,0x95), MAKE_RGB(0x38,0x93,0xA6), \
	MAKE_RGB(0x49,0xA4,0xB7), MAKE_RGB(0x5A,0xB5,0xC8), MAKE_RGB(0x6B,0xC6,0xD9), MAKE_RGB(0x7C,0xD7,0xEA), \
	MAKE_RGB(0x8D,0xE8,0xFB), MAKE_RGB(0x9E,0xF9,0xFF), MAKE_RGB(0xAF,0xFF,0xFF), MAKE_RGB(0xC0,0xFF,0xFF   )

#define NTSC_GREEN_BLUE \
	MAKE_RGB(0x00,0x25,0x0B), MAKE_RGB(0x00,0x36,0x10), MAKE_RGB(0x00,0x47,0x18), MAKE_RGB(0x00,0x58,0x2E), \
	MAKE_RGB(0x07,0x69,0x42), MAKE_RGB(0x18,0x7A,0x53), MAKE_RGB(0x29,0x8B,0x64), MAKE_RGB(0x3A,0x9C,0x75), \
	MAKE_RGB(0x4B,0xAD,0x86), MAKE_RGB(0x5C,0xBE,0x97), MAKE_RGB(0x6D,0xCF,0xA8), MAKE_RGB(0x7E,0xE0,0xB9), \
	MAKE_RGB(0x8F,0xF1,0xCA), MAKE_RGB(0xA0,0xFF,0xDA), MAKE_RGB(0xB1,0xFF,0xE6), MAKE_RGB(0xC2,0xFF,0xF2   )

#define NTSC_GREEN \
	MAKE_RGB(0x00,0x27,0x0C), MAKE_RGB(0x00,0x38,0x11), MAKE_RGB(0x00,0x49,0x16), MAKE_RGB(0x04,0x5A,0x1A), \
	MAKE_RGB(0x15,0x6B,0x1A), MAKE_RGB(0x26,0x7C,0x22), MAKE_RGB(0x37,0x8D,0x33), MAKE_RGB(0x48,0x9E,0x44), \
	MAKE_RGB(0x59,0xAF,0x55), MAKE_RGB(0x6A,0xC0,0x66), MAKE_RGB(0x7B,0xD1,0x77), MAKE_RGB(0x8C,0xE2,0x88), \
	MAKE_RGB(0x9D,0xF3,0x99), MAKE_RGB(0xAE,0xFF,0xA8), MAKE_RGB(0xBF,0xFF,0xB4), MAKE_RGB(0xD0,0xFF,0xC0   )

#define NTSC_YELLOW_GREEN \
	MAKE_RGB(0x00,0x21,0x0A), MAKE_RGB(0x00,0x32,0x0F), MAKE_RGB(0x0A,0x43,0x11), MAKE_RGB(0x1B,0x54,0x11), \
	MAKE_RGB(0x2C,0x65,0x11), MAKE_RGB(0x3D,0x76,0x11), MAKE_RGB(0x4E,0x87,0x11), MAKE_RGB(0x5F,0x98,0x1E), \
	MAKE_RGB(0x70,0xA9,0x2F), MAKE_RGB(0x81,0xBA,0x40), MAKE_RGB(0x92,0xCB,0x51), MAKE_RGB(0xA3,0xDC,0x62), \
	MAKE_RGB(0xB4,0xED,0x73), MAKE_RGB(0xC5,0xFE,0x84), MAKE_RGB(0xD6,0xFF,0x90), MAKE_RGB(0xE7,0xFF,0x9C   )

#define NTSC_ORANGE_GREEN \
	MAKE_RGB(0x05,0x13,0x04), MAKE_RGB(0x16,0x24,0x04), MAKE_RGB(0x27,0x35,0x04), MAKE_RGB(0x38,0x46,0x04), \
	MAKE_RGB(0x49,0x57,0x04), MAKE_RGB(0x5A,0x68,0x04), MAKE_RGB(0x6B,0x79,0x04), MAKE_RGB(0x7C,0x8A,0x09), \
	MAKE_RGB(0x8D,0x9B,0x1A), MAKE_RGB(0x9E,0xAC,0x2B), MAKE_RGB(0xAF,0xBD,0x3C), MAKE_RGB(0xC0,0xCE,0x4D), \
	MAKE_RGB(0xD1,0xDF,0x5E), MAKE_RGB(0xE2,0xF0,0x6F), MAKE_RGB(0xF3,0xFF,0x80), MAKE_RGB(0xFF,0xFF,0x8D   )

#define NTSC_LIGHT_ORANGE \
	MAKE_RGB(0x21,0x02,0x00), MAKE_RGB(0x32,0x13,0x00), MAKE_RGB(0x43,0x24,0x00), MAKE_RGB(0x54,0x35,0x00), \
	MAKE_RGB(0x65,0x46,0x00), MAKE_RGB(0x76,0x57,0x00), MAKE_RGB(0x87,0x68,0x00), MAKE_RGB(0x98,0x79,0x0C), \
	MAKE_RGB(0xA9,0x8A,0x1D), MAKE_RGB(0xBA,0x9B,0x2E), MAKE_RGB(0xCB,0xAC,0x3F), MAKE_RGB(0xDC,0xBD,0x50), \
	MAKE_RGB(0xED,0xCE,0x61), MAKE_RGB(0xFE,0xDF,0x72), MAKE_RGB(0xFF,0xF0,0x87), MAKE_RGB(0xFF,0xFF,0x9D   )

static const rgb_t a7800_palette[256*3] =
{
	NTSC_GREY,
	NTSC_GOLD,
	NTSC_ORANGE,
	NTSC_RED_ORANGE,
	NTSC_PINK,
	NTSC_PURPLE,
	NTSC_PURPLE_BLUE,
	NTSC_BLUE1,
	NTSC_BLUE2,
	NTSC_LIGHT_BLUE,
	NTSC_TURQUOISE,
	NTSC_GREEN_BLUE,
	NTSC_GREEN,
	NTSC_YELLOW_GREEN,
	NTSC_ORANGE_GREEN,
	NTSC_LIGHT_ORANGE
};

static const rgb_t a7800p_palette[256*3] =
{
	NTSC_GREY,
	NTSC_ORANGE_GREEN,
	NTSC_GOLD,
	NTSC_ORANGE,
	NTSC_RED_ORANGE,
	NTSC_PINK,
	NTSC_PURPLE,
	NTSC_PURPLE_BLUE,
	NTSC_BLUE1,
	NTSC_BLUE2,
	NTSC_LIGHT_BLUE,
	NTSC_TURQUOISE,
	NTSC_GREEN_BLUE,
	NTSC_GREEN,
	NTSC_YELLOW_GREEN,
	NTSC_ORANGE_GREEN
};

/***************************************************************************
    PALETTE - PHASE 24.7 SHIFT


define NTSC_GREY
    MAKE_RGB(0x00,0x00,0x00), MAKE_RGB(0x11,0x11,0x11), MAKE_RGB(0x22,0x22,0x22), MAKE_RGB(0x33,0x33,0x33), \
    MAKE_RGB(0x44,0x44,0x44), MAKE_RGB(0x55,0x55,0x55), MAKE_RGB(0x66,0x66,0x66), MAKE_RGB(0x77,0x77,0x77), \
    MAKE_RGB(0x88,0x88,0x88), MAKE_RGB(0x99,0x99,0x99), MAKE_RGB(0xAA,0xAA,0xAA), MAKE_RGB(0xBB,0xBB,0xBB), \
    MAKE_RGB(0xCC,0xCC,0xCC), MAKE_RGB(0xDD,0xDD,0xDD), MAKE_RGB(0xEE,0xEE,0xEE), MAKE_RGB(0xFF,0xFF,0xFF   )

define NTSC_GOLD
    MAKE_RGB(0x1A,0x07,0x00), MAKE_RGB(0x2B,0x18,0x00), MAKE_RGB(0x3C,0x29,0x00), MAKE_RGB(0x4D,0x3A,0x00), \
    MAKE_RGB(0x5E,0x4B,0x00), MAKE_RGB(0x6F,0x5C,0x00), MAKE_RGB(0x80,0x6D,0x00), MAKE_RGB(0x91,0x7E,0x09), \
    MAKE_RGB(0xA2,0x8F,0x1A), MAKE_RGB(0xB3,0xA0,0x2B), MAKE_RGB(0xC4,0xB1,0x3C), MAKE_RGB(0xD5,0xC2,0x4D), \
    MAKE_RGB(0xE6,0xD3,0x5E), MAKE_RGB(0xF7,0xE4,0x6F), MAKE_RGB(0xFF,0xF5,0x83), MAKE_RGB(0xFF,0xF7,0x97   )

define NTSC_ORANGE
    MAKE_RGB(0x30,0x00,0x00), MAKE_RGB(0x41,0x07,0x00), MAKE_RGB(0x52,0x18,0x00), MAKE_RGB(0x63,0x29,0x00), \
    MAKE_RGB(0x74,0x3A,0x00), MAKE_RGB(0x85,0x4B,0x00), MAKE_RGB(0x96,0x5C,0x0A), MAKE_RGB(0xA7,0x6D,0x1B), \
    MAKE_RGB(0xB8,0x7E,0x2C), MAKE_RGB(0xC9,0x8F,0x3D), MAKE_RGB(0xDA,0xA0,0x4E), MAKE_RGB(0xEB,0xB1,0x5F), \
    MAKE_RGB(0xFC,0xC2,0x70), MAKE_RGB(0xFF,0xD3,0x85), MAKE_RGB(0xFF,0xE4,0x9B), MAKE_RGB(0xFF,0xF5,0xB1   )

define NTSC_RED_ORANGE
    MAKE_RGB(0x3D,0x00,0x00), MAKE_RGB(0x4E,0x00,0x00), MAKE_RGB(0x5F,0x09,0x00), MAKE_RGB(0x70,0x1A,0x00), \
    MAKE_RGB(0x81,0x2B,0x09), MAKE_RGB(0x92,0x3C,0x1A), MAKE_RGB(0xA3,0x4D,0x2B), MAKE_RGB(0xB4,0x5E,0x3C), \
    MAKE_RGB(0xC5,0x6F,0x4D), MAKE_RGB(0xD6,0x80,0x5E), MAKE_RGB(0xE7,0x91,0x6F), MAKE_RGB(0xF8,0xA2,0x80), \
    MAKE_RGB(0xFF,0xB3,0x94), MAKE_RGB(0xFF,0xC4,0xAA), MAKE_RGB(0xFF,0xD5,0xC0), MAKE_RGB(0xFF,0xE6,0xD6   )

define NTSC_PINK
    MAKE_RGB(0x3F,0x00,0x00), MAKE_RGB(0x50,0x00,0x09), MAKE_RGB(0x61,0x00,0x15), MAKE_RGB(0x72,0x10,0x26), \
    MAKE_RGB(0x83,0x21,0x37), MAKE_RGB(0x94,0x32,0x48), MAKE_RGB(0xA5,0x43,0x59), MAKE_RGB(0xB6,0x54,0x6A), \
    MAKE_RGB(0xC7,0x65,0x7B), MAKE_RGB(0xD8,0x76,0x8C), MAKE_RGB(0xE9,0x87,0x9D), MAKE_RGB(0xFA,0x98,0xAE), \
    MAKE_RGB(0xFF,0xA9,0xC2), MAKE_RGB(0xFF,0xBA,0xD8), MAKE_RGB(0xFF,0xCB,0xEE), MAKE_RGB(0xFF,0xDC,0xF4   )

define NTSC_PURPLE
    MAKE_RGB(0x36,0x00,0x2E), MAKE_RGB(0x47,0x00,0x3A), MAKE_RGB(0x58,0x00,0x46), MAKE_RGB(0x69,0x0C,0x55), \
    MAKE_RGB(0x7A,0x1D,0x66), MAKE_RGB(0x8B,0x2E,0x77), MAKE_RGB(0x9C,0x3F,0x88), MAKE_RGB(0xAD,0x50,0x99), \
    MAKE_RGB(0xBE,0x61,0xAA), MAKE_RGB(0xCF,0x72,0xBB), MAKE_RGB(0xE0,0x83,0xCC), MAKE_RGB(0xF1,0x94,0xDD), \
    MAKE_RGB(0xFF,0xA5,0xE4), MAKE_RGB(0xFF,0xB6,0xE9), MAKE_RGB(0xFF,0xC7,0xEE), MAKE_RGB(0xFF,0xD8,0xF3   )

define NTSC_PURPLE_BLUE
    MAKE_RGB(0x23,0x00,0x55), MAKE_RGB(0x34,0x00,0x61), MAKE_RGB(0x45,0x00,0x6D), MAKE_RGB(0x56,0x0F,0x7E), \
    MAKE_RGB(0x67,0x20,0x8F), MAKE_RGB(0x78,0x31,0xA0), MAKE_RGB(0x89,0x42,0xB1), MAKE_RGB(0x9A,0x53,0xC2), \
    MAKE_RGB(0xAB,0x64,0xD3), MAKE_RGB(0xBC,0x75,0xE4), MAKE_RGB(0xCD,0x86,0xEA), MAKE_RGB(0xDE,0x97,0xEA), \
    MAKE_RGB(0xEF,0xA8,0xEA), MAKE_RGB(0xFF,0xB9,0xEA), MAKE_RGB(0xFF,0xCA,0xEF), MAKE_RGB(0xFF,0xDB,0xF4   )

define NTSC_BLUE1
    MAKE_RGB(0x09,0x00,0x6E), MAKE_RGB(0x1A,0x00,0x7A), MAKE_RGB(0x2B,0x08,0x88), MAKE_RGB(0x3C,0x19,0x99), \
    MAKE_RGB(0x4D,0x2A,0xAA), MAKE_RGB(0x5E,0x3B,0xBB), MAKE_RGB(0x6F,0x4C,0xCC), MAKE_RGB(0x80,0x5D,0xDD), \
    MAKE_RGB(0x91,0x6E,0xEE), MAKE_RGB(0xA2,0x7F,0xF4), MAKE_RGB(0xB3,0x90,0xF4), MAKE_RGB(0xC4,0xA1,0xF4), \
    MAKE_RGB(0xD5,0xB2,0xF4), MAKE_RGB(0xE6,0xC3,0xF4), MAKE_RGB(0xF7,0xD4,0xF4), MAKE_RGB(0xFF,0xE5,0xF7   )

define NTSC_BLUE2
    MAKE_RGB(0x00,0x00,0x6D), MAKE_RGB(0x00,0x05,0x80), MAKE_RGB(0x10,0x16,0x91), MAKE_RGB(0x21,0x27,0xA2), \
    MAKE_RGB(0x32,0x38,0xB3), MAKE_RGB(0x43,0x49,0xC4), MAKE_RGB(0x54,0x5A,0xD5), MAKE_RGB(0x65,0x6B,0xE6), \
    MAKE_RGB(0x76,0x7C,0xF7), MAKE_RGB(0x87,0x8D,0xFF), MAKE_RGB(0x98,0x9E,0xFF), MAKE_RGB(0xA9,0xAF,0xFF), \
    MAKE_RGB(0xBA,0xC0,0xFF), MAKE_RGB(0xCB,0xD1,0xFF), MAKE_RGB(0xDC,0xE2,0xFF), MAKE_RGB(0xED,0xF3,0xFF   )

define NTSC_LIGHT_BLUE
    MAKE_RGB(0x00,0x05,0x57), MAKE_RGB(0x00,0x16,0x6E), MAKE_RGB(0x00,0x27,0x84), MAKE_RGB(0x09,0x38,0x97), \
    MAKE_RGB(0x1A,0x49,0xA8), MAKE_RGB(0x2B,0x5A,0xB9), MAKE_RGB(0x3C,0x6B,0xCA), MAKE_RGB(0x4D,0x7C,0xDB), \
    MAKE_RGB(0x5E,0x8D,0xEC), MAKE_RGB(0x6F,0x9E,0xFD), MAKE_RGB(0x80,0xAF,0xFF), MAKE_RGB(0x91,0xC0,0xFF), \
    MAKE_RGB(0xA2,0xD1,0xFF), MAKE_RGB(0xB3,0xE2,0xFF), MAKE_RGB(0xC4,0xF3,0xFF), MAKE_RGB(0xD5,0xFF,0xFF   )

define NTSC_TURQUOISE
    MAKE_RGB(0x00,0x15,0x34), MAKE_RGB(0x00,0x26,0x4A), MAKE_RGB(0x00,0x37,0x60), MAKE_RGB(0x00,0x48,0x77), \
    MAKE_RGB(0x0A,0x59,0x8A), MAKE_RGB(0x1B,0x6A,0x9B), MAKE_RGB(0x2C,0x7B,0xAC), MAKE_RGB(0x3D,0x8C,0xBD), \
    MAKE_RGB(0x4E,0x9D,0xCE), MAKE_RGB(0x5F,0xAE,0xDF), MAKE_RGB(0x70,0xBF,0xF0), MAKE_RGB(0x81,0xD0,0xFF), \
    MAKE_RGB(0x92,0xE1,0xFF), MAKE_RGB(0xA3,0xF2,0xFF), MAKE_RGB(0xB4,0xFF,0xFF), MAKE_RGB(0xC5,0xFF,0xFF   )

define NTSC_GREEN_BLUE
    MAKE_RGB(0x00,0x21,0x0A), MAKE_RGB(0x00,0x32,0x1F), MAKE_RGB(0x00,0x43,0x35), MAKE_RGB(0x00,0x54,0x4B), \
    MAKE_RGB(0x04,0x65,0x60), MAKE_RGB(0x15,0x76,0x71), MAKE_RGB(0x26,0x87,0x82), MAKE_RGB(0x37,0x98,0x93), \
    MAKE_RGB(0x48,0xA9,0xA4), MAKE_RGB(0x59,0xBA,0xB5), MAKE_RGB(0x6A,0xCB,0xC6), MAKE_RGB(0x7B,0xDC,0xD7), \
    MAKE_RGB(0x8C,0xED,0xE8), MAKE_RGB(0x9D,0xFE,0xF9), MAKE_RGB(0xAE,0xFF,0xFF), MAKE_RGB(0xBF,0xFF,0xFF   )

define NTSC_GREEN
    MAKE_RGB(0x00,0x27,0x0C), MAKE_RGB(0x00,0x38,0x11), MAKE_RGB(0x00,0x49,0x16), MAKE_RGB(0x00,0x5A,0x1D), \
    MAKE_RGB(0x0A,0x6B,0x30), MAKE_RGB(0x1B,0x7C,0x41), MAKE_RGB(0x2C,0x8D,0x52), MAKE_RGB(0x3D,0x9E,0x63), \
    MAKE_RGB(0x4E,0xAF,0x74), MAKE_RGB(0x5F,0xC0,0x85), MAKE_RGB(0x70,0xD1,0x96), MAKE_RGB(0x81,0xE2,0xA7), \
    MAKE_RGB(0x92,0xF3,0xB8), MAKE_RGB(0xA3,0xFF,0xC8), MAKE_RGB(0xB4,0xFF,0xD3), MAKE_RGB(0xC5,0xFF,0xDF   )

define NTSC_YELLOW_GREEN
    MAKE_RGB(0x00,0x26,0x0B), MAKE_RGB(0x00,0x37,0x10), MAKE_RGB(0x00,0x48,0x16), MAKE_RGB(0x0A,0x59,0x18), \
    MAKE_RGB(0x1B,0x6A,0x18), MAKE_RGB(0x2C,0x7B,0x18), MAKE_RGB(0x3D,0x8C,0x27), MAKE_RGB(0x4E,0x9D,0x38), \
    MAKE_RGB(0x5F,0xAE,0x49), MAKE_RGB(0x70,0xBF,0x5A), MAKE_RGB(0x81,0xD0,0x6B), MAKE_RGB(0x92,0xE1,0x7C), \
    MAKE_RGB(0xA3,0xF2,0x8D), MAKE_RGB(0xB4,0xFF,0x9C), MAKE_RGB(0xC5,0xFF,0xA8), MAKE_RGB(0xD6,0xFF,0xB4   )

define NTSC_ORANGE_GREEN
    MAKE_RGB(0x00,0x1E,0x09), MAKE_RGB(0x00,0x2F,0x0E), MAKE_RGB(0x11,0x40,0x0E), MAKE_RGB(0x22,0x51,0x0E), \
    MAKE_RGB(0x33,0x62,0x0E), MAKE_RGB(0x44,0x73,0x0E), MAKE_RGB(0x55,0x84,0x0E), MAKE_RGB(0x66,0x95,0x17), \
    MAKE_RGB(0x77,0xA6,0x28), MAKE_RGB(0x88,0xB7,0x39), MAKE_RGB(0x99,0xC8,0x4A), MAKE_RGB(0xAA,0xD9,0x5B), \
    MAKE_RGB(0xBB,0xEA,0x6C), MAKE_RGB(0xCC,0xFB,0x7D), MAKE_RGB(0xDD,0xFF,0x8A), MAKE_RGB(0xEE,0xFF,0x96   )

define NTSC_LIGHT_ORANGE
    MAKE_RGB(0x0A,0x11,0x02), MAKE_RGB(0x1B,0x22,0x02), MAKE_RGB(0x2C,0x33,0x02), MAKE_RGB(0x3D,0x44,0x02), \
    MAKE_RGB(0x4E,0x55,0x02), MAKE_RGB(0x5F,0x66,0x02), MAKE_RGB(0x70,0x77,0x02), MAKE_RGB(0x81,0x88,0x09), \
    MAKE_RGB(0x92,0x99,0x1A), MAKE_RGB(0xA3,0xAA,0x2B), MAKE_RGB(0xB4,0xBB,0x3C), MAKE_RGB(0xC5,0xCC,0x4D), \
    MAKE_RGB(0xD6,0xDD,0x5E), MAKE_RGB(0xE7,0xEE,0x6F), MAKE_RGB(0xF8,0xFF,0x80), MAKE_RGB(0xFF,0xFF,0x8F   )
***************************************************************************/

/***************************************************************************
    PALETTE - PHASE 25.2 SHIFT


define NTSC_GREY
    MAKE_RGB(0x00,0x00,0x00), MAKE_RGB(0x11,0x11,0x11), MAKE_RGB(0x22,0x22,0x22), MAKE_RGB(0x33,0x33,0x33), \
    MAKE_RGB(0x44,0x44,0x44), MAKE_RGB(0x55,0x55,0x55), MAKE_RGB(0x66,0x66,0x66), MAKE_RGB(0x77,0x77,0x77), \
    MAKE_RGB(0x88,0x88,0x88), MAKE_RGB(0x99,0x99,0x99), MAKE_RGB(0xAA,0xAA,0xAA), MAKE_RGB(0xBB,0xBB,0xBB), \
    MAKE_RGB(0xCC,0xCC,0xCC), MAKE_RGB(0xDD,0xDD,0xDD), MAKE_RGB(0xEE,0xEE,0xEE), MAKE_RGB(0xFF,0xFF,0xFF   )

define NTSC_GOLD
    MAKE_RGB(0x1A,0x07,0x00), MAKE_RGB(0x2B,0x18,0x00), MAKE_RGB(0x3C,0x29,0x00), MAKE_RGB(0x4D,0x3A,0x00), \
    MAKE_RGB(0x5E,0x4B,0x00), MAKE_RGB(0x6F,0x5C,0x00), MAKE_RGB(0x80,0x6D,0x00), MAKE_RGB(0x91,0x7E,0x09), \
    MAKE_RGB(0xA2,0x8F,0x1A), MAKE_RGB(0xB3,0xA0,0x2B), MAKE_RGB(0xC4,0xB1,0x3C), MAKE_RGB(0xD5,0xC2,0x4D), \
    MAKE_RGB(0xE6,0xD3,0x5E), MAKE_RGB(0xF7,0xE4,0x6F), MAKE_RGB(0xFF,0xF5,0x83), MAKE_RGB(0xFF,0xF7,0x97   )

define NTSC_ORANGE
    MAKE_RGB(0x30,0x00,0x00), MAKE_RGB(0x41,0x07,0x00), MAKE_RGB(0x52,0x18,0x00), MAKE_RGB(0x63,0x29,0x00), \
    MAKE_RGB(0x74,0x3A,0x00), MAKE_RGB(0x85,0x4B,0x00), MAKE_RGB(0x96,0x5C,0x0A), MAKE_RGB(0xA7,0x6D,0x1B), \
    MAKE_RGB(0xB8,0x7E,0x2C), MAKE_RGB(0xC9,0x8F,0x3D), MAKE_RGB(0xDA,0xA0,0x4E), MAKE_RGB(0xEB,0xB1,0x5F), \
    MAKE_RGB(0xFC,0xC2,0x70), MAKE_RGB(0xFF,0xD3,0x85), MAKE_RGB(0xFF,0xE4,0x9B), MAKE_RGB(0xFF,0xF5,0xB1   )

define NTSC_RED_ORANGE
    MAKE_RGB(0x3E,0x00,0x00), MAKE_RGB(0x4F,0x00,0x00), MAKE_RGB(0x60,0x09,0x00), MAKE_RGB(0x71,0x1A,0x00), \
    MAKE_RGB(0x82,0x2B,0x0B), MAKE_RGB(0x93,0x3C,0x1C), MAKE_RGB(0xA4,0x4D,0x2D), MAKE_RGB(0xB5,0x5E,0x3E), \
    MAKE_RGB(0xC6,0x6F,0x4F), MAKE_RGB(0xD7,0x80,0x60), MAKE_RGB(0xE8,0x91,0x71), MAKE_RGB(0xF9,0xA2,0x82), \
    MAKE_RGB(0xFF,0xB3,0x96), MAKE_RGB(0xFF,0xC4,0xAC), MAKE_RGB(0xFF,0xD5,0xC2), MAKE_RGB(0xFF,0xE6,0xD8   )

define NTSC_PINK
    MAKE_RGB(0x3F,0x00,0x00), MAKE_RGB(0x50,0x00,0x0C), MAKE_RGB(0x61,0x00,0x18), MAKE_RGB(0x72,0x0F,0x28), \
    MAKE_RGB(0x83,0x20,0x39), MAKE_RGB(0x94,0x31,0x4A), MAKE_RGB(0xA5,0x42,0x5B), MAKE_RGB(0xB6,0x53,0x6C), \
    MAKE_RGB(0xC7,0x64,0x7D), MAKE_RGB(0xD8,0x75,0x8E), MAKE_RGB(0xE9,0x86,0x9F), MAKE_RGB(0xFA,0x97,0xB0), \
    MAKE_RGB(0xFF,0xA8,0xC5), MAKE_RGB(0xFF,0xB9,0xDB), MAKE_RGB(0xFF,0xCA,0xEF), MAKE_RGB(0xFF,0xDB,0xF4   )

define NTSC_PURPLE
    MAKE_RGB(0x35,0x00,0x31), MAKE_RGB(0x46,0x00,0x3D), MAKE_RGB(0x57,0x00,0x49), MAKE_RGB(0x68,0x0C,0x58), \
    MAKE_RGB(0x79,0x1D,0x69), MAKE_RGB(0x8A,0x2E,0x7A), MAKE_RGB(0x9B,0x3F,0x8B), MAKE_RGB(0xAC,0x50,0x9C), \
    MAKE_RGB(0xBD,0x61,0xAD), MAKE_RGB(0xCE,0x72,0xBE), MAKE_RGB(0xDF,0x83,0xCF), MAKE_RGB(0xF0,0x94,0xE0), \
    MAKE_RGB(0xFF,0xA5,0xE4), MAKE_RGB(0xFF,0xB6,0xE9), MAKE_RGB(0xFF,0xC7,0xEE), MAKE_RGB(0xFF,0xD8,0xF3   )

define NTSC_PURPLE_BLUE
    MAKE_RGB(0x20,0x00,0x59), MAKE_RGB(0x31,0x00,0x65), MAKE_RGB(0x42,0x00,0x71), MAKE_RGB(0x53,0x10,0x82), \
    MAKE_RGB(0x64,0x21,0x93), MAKE_RGB(0x75,0x32,0xA4), MAKE_RGB(0x86,0x43,0xB5), MAKE_RGB(0x97,0x54,0xC6), \
    MAKE_RGB(0xA8,0x65,0xD7), MAKE_RGB(0xB9,0x76,0xE8), MAKE_RGB(0xCA,0x87,0xEB), MAKE_RGB(0xDB,0x98,0xEB), \
    MAKE_RGB(0xEC,0xA9,0xEB), MAKE_RGB(0xFD,0xBA,0xEB), MAKE_RGB(0xFF,0xCB,0xEF), MAKE_RGB(0xFF,0xDC,0xF4   )

define NTSC_BLUE1
    MAKE_RGB(0x05,0x00,0x70), MAKE_RGB(0x16,0x00,0x7C), MAKE_RGB(0x27,0x09,0x8B), MAKE_RGB(0x38,0x1A,0x9C), \
    MAKE_RGB(0x49,0x2B,0xAD), MAKE_RGB(0x5A,0x3C,0xBE), MAKE_RGB(0x6B,0x4D,0xCF), MAKE_RGB(0x7C,0x5E,0xE0), \
    MAKE_RGB(0X8D,0x6F,0xF1), MAKE_RGB(0x9E,0x80,0xF6), MAKE_RGB(0xAF,0x91,0xF6), MAKE_RGB(0xC0,0xA2,0xF6), \
    MAKE_RGB(0xD1,0xB3,0xF6), MAKE_RGB(0xE2,0xC4,0xF6), MAKE_RGB(0xF3,0xD5,0xF6), MAKE_RGB(0xFF,0xE6,0xF7   )

define NTSC_BLUE2
    MAKE_RGB(0x00,0x00,0x6B), MAKE_RGB(0x00,0x08,0x7E), MAKE_RGB(0x0C,0x19,0x91), MAKE_RGB(0x1D,0x2A,0xA2), \
    MAKE_RGB(0x2E,0x3B,0xB3), MAKE_RGB(0x3F,0x4C,0xC4), MAKE_RGB(0x50,0x5D,0xD5), MAKE_RGB(0x61,0x6E,0xE6), \
    MAKE_RGB(0x72,0x7F,0xF7), MAKE_RGB(0x83,0x90,0xFF), MAKE_RGB(0x94,0xA1,0xFF), MAKE_RGB(0xA5,0xB2,0xFF), \
    MAKE_RGB(0xB6,0xC3,0xFF), MAKE_RGB(0xC7,0xD4,0xFF), MAKE_RGB(0xD8,0xE5,0xFF), MAKE_RGB(0xE9,0xF6,0xFF   )

define NTSC_LIGHT_BLUE
    MAKE_RGB(0x00,0x08,0x52), MAKE_RGB(0x00,0x19,0x68), MAKE_RGB(0x00,0x2A,0x7F), MAKE_RGB(0x05,0x3B,0x93), \
    MAKE_RGB(0x16,0x4C,0xA4), MAKE_RGB(0x27,0x5D,0xB5), MAKE_RGB(0x38,0x6E,0xC6), MAKE_RGB(0x49,0x7F,0xD7), \
    MAKE_RGB(0x5A,0x90,0xE8), MAKE_RGB(0x6B,0xA1,0xF9), MAKE_RGB(0x7C,0xB2,0xFF), MAKE_RGB(0x8D,0xC3,0xFF), \
    MAKE_RGB(0x9E,0xD4,0xFF), MAKE_RGB(0xAF,0xE5,0xFF), MAKE_RGB(0xC0,0xF6,0xFF), MAKE_RGB(0xD1,0xFF,0xFF   )

define NTSC_TURQUOISE
    MAKE_RGB(0x00,0x17,0x2D), MAKE_RGB(0x00,0x28,0x43), MAKE_RGB(0x00,0x39,0x59), MAKE_RGB(0x00,0x4A,0x6F), \
    MAKE_RGB(0x08,0x5B,0x83), MAKE_RGB(0x19,0x6C,0x94), MAKE_RGB(0x2A,0x7D,0xA5), MAKE_RGB(0x3B,0x8E,0xB6), \
    MAKE_RGB(0x4C,0x9F,0xC7), MAKE_RGB(0x5D,0xB0,0xD8), MAKE_RGB(0x6E,0xC1,0xE9), MAKE_RGB(0x7F,0xD2,0xFA), \
    MAKE_RGB(0x90,0xE3,0xFF), MAKE_RGB(0xA1,0xF4,0xFF), MAKE_RGB(0xB2,0xFF,0xFF), MAKE_RGB(0xC3,0xFF,0xFF   )

define NTSC_GREEN_BLUE
    MAKE_RGB(0x00,0x23,0x0A), MAKE_RGB(0x00,0x34,0x15), MAKE_RGB(0x00,0x45,0x2B), MAKE_RGB(0x00,0x56,0x41), \
    MAKE_RGB(0x04,0x67,0x56), MAKE_RGB(0x15,0x78,0x67), MAKE_RGB(0x26,0x89,0x78), MAKE_RGB(0x37,0x9A,0x89), \
    MAKE_RGB(0x48,0xAB,0x9A), MAKE_RGB(0x59,0xBC,0xAB), MAKE_RGB(0x6A,0xCD,0xBC), MAKE_RGB(0x7B,0xDE,0xCD), \
    MAKE_RGB(0x8C,0xEF,0xDE), MAKE_RGB(0x9D,0xFF,0xEE), MAKE_RGB(0xAE,0xFF,0xFA), MAKE_RGB(0xBF,0xFF,0xFF   )

define NTSC_GREEN
    MAKE_RGB(0x00,0x27,0x0C), MAKE_RGB(0x00,0x38,0x11), MAKE_RGB(0x00,0x49,0x16), MAKE_RGB(0x00,0x5A,0x1B), \
    MAKE_RGB(0x0D,0x6B,0x25), MAKE_RGB(0x1E,0x7C,0x36), MAKE_RGB(0x2F,0x8D,0x47), MAKE_RGB(0x40,0x9E,0x58), \
    MAKE_RGB(0x51,0xAF,0x69), MAKE_RGB(0x62,0xC0,0x7A), MAKE_RGB(0x73,0xD1,0x8B), MAKE_RGB(0x84,0xE2,0x9C), \
    MAKE_RGB(0x95,0xF3,0xAD), MAKE_RGB(0xA6,0xFF,0xBD), MAKE_RGB(0xB7,0xFF,0xC9), MAKE_RGB(0xC8,0xFF,0xD4   )

define NTSC_YELLOW_GREEN
    MAKE_RGB(0x00,0x24,0x0B), MAKE_RGB(0x00,0x35,0x10), MAKE_RGB(0x00,0x46,0x15), MAKE_RGB(0x10,0x57,0x15), \
    MAKE_RGB(0x21,0x68,0x15), MAKE_RGB(0x32,0x79,0x15), MAKE_RGB(0x43,0x8A,0x1C), MAKE_RGB(0x54,0x9B,0x2D), \
    MAKE_RGB(0x65,0xAC,0x3E), MAKE_RGB(0x76,0xBD,0x4F), MAKE_RGB(0x87,0xCE,0x60), MAKE_RGB(0x98,0xDF,0x71), \
    MAKE_RGB(0xA9,0xF0,0x82), MAKE_RGB(0xBA,0xFF,0x93), MAKE_RGB(0xCB,0xFF,0x9F), MAKE_RGB(0xDC,0xFF,0xAA   )

define NTSC_ORANGE_GREEN
    MAKE_RGB(0x00,0x1B,0x08), MAKE_RGB(0x08,0x2C,0x0B), MAKE_RGB(0x19,0x3D,0x0B), MAKE_RGB(0x2A,0x4E,0x0B), \
    MAKE_RGB(0x3B,0x5F,0x0B), MAKE_RGB(0x4C,0x70,0x0B), MAKE_RGB(0x5D,0x81,0x0B), MAKE_RGB(0x6E,0x92,0x11), \
    MAKE_RGB(0x7F,0xA3,0x22), MAKE_RGB(0x90,0xB4,0x33), MAKE_RGB(0xA1,0xC5,0x44), MAKE_RGB(0xB2,0xD6,0x55), \
    MAKE_RGB(0xC3,0xE7,0x66), MAKE_RGB(0xD4,0xF8,0x77), MAKE_RGB(0xE5,0xFF,0x85), MAKE_RGB(0xF6,0xFF,0x91   )

define NTSC_LIGHT_ORANGE
    MAKE_RGB(0x12,0x0C,0x00), MAKE_RGB(0x23,0x1D,0x00), MAKE_RGB(0x34,0x2E,0x00), MAKE_RGB(0x45,0x3F,0x00), \
    MAKE_RGB(0x56,0x50,0x00), MAKE_RGB(0x67,0x61,0x00), MAKE_RGB(0x78,0x72,0x00), MAKE_RGB(0x89,0x83,0x08), \
    MAKE_RGB(0x9A,0x94,0x19), MAKE_RGB(0xAB,0xA5,0x2A), MAKE_RGB(0xBC,0xB6,0x3B), MAKE_RGB(0xCD,0xC7,0x4C), \
    MAKE_RGB(0xDE,0xD8,0x5D), MAKE_RGB(0xEF,0xE9,0x6E), MAKE_RGB(0xFF,0xFA,0x80), MAKE_RGB(0xFF,0xFF,0x92   )
***************************************************************************/

/***************************************************************************
    PALETTE - PHASE 25.7 SHIFT


define NTSC_GREY
    MAKE_RGB(0x00,0x00,0x00), MAKE_RGB(0x11,0x11,0x11), MAKE_RGB(0x22,0x22,0x22), MAKE_RGB(0x33,0x33,0x33), \
    MAKE_RGB(0x44,0x44,0x44), MAKE_RGB(0x55,0x55,0x55), MAKE_RGB(0x66,0x66,0x66), MAKE_RGB(0x77,0x77,0x77), \
    MAKE_RGB(0x88,0x88,0x88), MAKE_RGB(0x99,0x99,0x99), MAKE_RGB(0xAA,0xAA,0xAA), MAKE_RGB(0xBB,0xBB,0xBB), \
    MAKE_RGB(0xCC,0xCC,0xCC), MAKE_RGB(0xDD,0xDD,0xDD), MAKE_RGB(0xEE,0xEE,0xEE), MAKE_RGB(0xFF,0xFF,0xFF   )

define NTSC_GOLD
    MAKE_RGB(0x1A,0x07,0x00), MAKE_RGB(0x2B,0x18,0x00), MAKE_RGB(0x3C,0x29,0x00), MAKE_RGB(0x4D,0x3A,0x00), \
    MAKE_RGB(0x5E,0x4B,0x00), MAKE_RGB(0x6F,0x5C,0x00), MAKE_RGB(0x80,0x6D,0x00), MAKE_RGB(0x91,0x7E,0x09), \
    MAKE_RGB(0xA2,0x8F,0x1A), MAKE_RGB(0xB3,0xA0,0x2B), MAKE_RGB(0xC4,0xB1,0x3C), MAKE_RGB(0xD5,0xC2,0x4D), \
    MAKE_RGB(0xE6,0xD3,0x5E), MAKE_RGB(0xF7,0xE4,0x6F), MAKE_RGB(0xFF,0xF5,0x83), MAKE_RGB(0xFF,0xF7,0x97   )

define NTSC_ORANGE
    MAKE_RGB(0x31,0x00,0x00), MAKE_RGB(0x42,0x06,0x00), MAKE_RGB(0x53,0x17,0x00), MAKE_RGB(0x64,0x28,0x00), \
    MAKE_RGB(0x75,0x39,0x00), MAKE_RGB(0x86,0X4A,0x00), MAKE_RGB(0x97,0x5B,0x0A), MAKE_RGB(0xA8,0x6C,0x1B), \
    MAKE_RGB(0xB9,0x7D,0x2C), MAKE_RGB(0xCA,0x8E,0x3D), MAKE_RGB(0xDB,0x9F,0x4E), MAKE_RGB(0xEC,0xB0,0x5F), \
    MAKE_RGB(0xFD,0xC1,0x70), MAKE_RGB(0xFF,0xD2,0x85), MAKE_RGB(0xFF,0xE3,0x9C), MAKE_RGB(0xFF,0xF4,0xB2   )

define NTSC_RED_ORANGE
    MAKE_RGB(0x3E,0x00,0x00), MAKE_RGB(0x4F,0x00,0x00), MAKE_RGB(0x60,0x08,0x00), MAKE_RGB(0x71,0x19,0x00), \
    MAKE_RGB(0x82,0x2A,0x0D), MAKE_RGB(0x93,0x3B,0x1E), MAKE_RGB(0xA4,0x4C,0x2F), MAKE_RGB(0xB5,0x5D,0x40), \
    MAKE_RGB(0xC6,0x6E,0x51), MAKE_RGB(0xD7,0x7F,0x62), MAKE_RGB(0xE8,0x90,0x73), MAKE_RGB(0xF9,0xA1,0x83), \
    MAKE_RGB(0xFF,0xB2,0x98), MAKE_RGB(0xFF,0xC3,0xAE), MAKE_RGB(0xFF,0xD4,0xC4), MAKE_RGB(0xFF,0xE5,0xDA   )

define NTSC_PINK
    MAKE_RGB(0x3F,0x00,0x03), MAKE_RGB(0x50,0x00,0x0F), MAKE_RGB(0x61,0x00,0x1B), MAKE_RGB(0x72,0x0F,0x2B), \
    MAKE_RGB(0x83,0x20,0x3C), MAKE_RGB(0x94,0x31,0x4D), MAKE_RGB(0xA5,0x42,0x5E), MAKE_RGB(0xB6,0x53,0x6F), \
    MAKE_RGB(0xC7,0x64,0x80), MAKE_RGB(0xD8,0x75,0x91), MAKE_RGB(0xE9,0x86,0xA2), MAKE_RGB(0xFA,0x97,0xB3), \
    MAKE_RGB(0xFF,0xA8,0xC8), MAKE_RGB(0xFF,0xB9,0xDE), MAKE_RGB(0xFF,0xCA,0xEF), MAKE_RGB(0xFF,0xDB,0xF4   )

define NTSC_PURPLE
    MAKE_RGB(0x33,0x00,0x35), MAKE_RGB(0x44,0x00,0x41), MAKE_RGB(0x55,0x00,0x4C), MAKE_RGB(0x66,0x0C,0x5C), \
    MAKE_RGB(0x77,0x1D,0x6D), MAKE_RGB(0x88,0x2E,0x7E), MAKE_RGB(0x99,0x3F,0x8F), MAKE_RGB(0xAA,0x50,0xA0), \
    MAKE_RGB(0xBB,0x61,0xB1), MAKE_RGB(0xCC,0x72,0xC2), MAKE_RGB(0xDD,0x83,0xD3), MAKE_RGB(0xEE,0x94,0xE4), \
    MAKE_RGB(0xFF,0xA5,0xE4), MAKE_RGB(0xFF,0xB6,0xE9), MAKE_RGB(0xFF,0xC7,0xEE), MAKE_RGB(0xFF,0xD8,0xF3   )

define NTSC_PURPLE_BLUE
    MAKE_RGB(0x1D,0x00,0x5C), MAKE_RGB(0x2E,0x00,0x68), MAKE_RGB(0x40,0x00,0x74), MAKE_RGB(0x51,0x10,0x84), \
    MAKE_RGB(0x62,0x21,0x95), MAKE_RGB(0x73,0x32,0xA6), MAKE_RGB(0x84,0x43,0xB7), MAKE_RGB(0x95,0x54,0xC8), \
    MAKE_RGB(0xA6,0x65,0xD9), MAKE_RGB(0xB7,0x76,0xEA), MAKE_RGB(0xC8,0x87,0xEB), MAKE_RGB(0xD9,0x98,0xEB), \
    MAKE_RGB(0xE9,0xA9,0xEC), MAKE_RGB(0xFB,0xBA,0xEB), MAKE_RGB(0xFF,0xCB,0xEF), MAKE_RGB(0xFF,0xDC,0xF4   )

define NTSC_BLUE1
    MAKE_RGB(0x02,0x00,0x71), MAKE_RGB(0x13,0x00,0x7D), MAKE_RGB(0x24,0x0B,0x8C), MAKE_RGB(0x35,0x1C,0x9D), \
    MAKE_RGB(0x46,0x2D,0xAE), MAKE_RGB(0x57,0x3E,0xBF), MAKE_RGB(0x68,0x4F,0xD0), MAKE_RGB(0x79,0x60,0xE1), \
    MAKE_RGB(0x8A,0x71,0xF2), MAKE_RGB(0x9B,0x82,0xF7), MAKE_RGB(0xAC,0x93,0xF7), MAKE_RGB(0xBD,0xA4,0xF7), \
    MAKE_RGB(0xCE,0xB5,0xF7), MAKE_RGB(0xDF,0xC6,0xF7), MAKE_RGB(0xF0,0xD7,0xF7), MAKE_RGB(0xFF,0xE8,0xF8   )

define NTSC_BLUE2
    MAKE_RGB(0x00,0x00,0x68), MAKE_RGB(0x00,0x0A,0x7C), MAKE_RGB(0x08,0x1B,0x90), MAKE_RGB(0x19,0x2C,0xA1), \
    MAKE_RGB(0x2A,0x3D,0xB2), MAKE_RGB(0x3B,0x4E,0xC3), MAKE_RGB(0x4C,0x5F,0xD4), MAKE_RGB(0x5D,0x70,0xE5), \
    MAKE_RGB(0x6E,0x81,0xF6), MAKE_RGB(0x7F,0x92,0xFF), MAKE_RGB(0x90,0xA3,0xFF), MAKE_RGB(0xA1,0xB4,0xFF), \
    MAKE_RGB(0xB2,0xC5,0xFF), MAKE_RGB(0xC3,0xD6,0xFF), MAKE_RGB(0xD4,0xE7,0xFF), MAKE_RGB(0xE5,0xF8,0xFF   )

define NTSC_LIGHT_BLUE
    MAKE_RGB(0x00,0x0A,0x4D), MAKE_RGB(0x00,0x1B,0x63), MAKE_RGB(0x00,0x2C,0x79), MAKE_RGB(0x02,0x3D,0x8F), \
    MAKE_RGB(0x13,0x4E,0xA0), MAKE_RGB(0x24,0x5F,0xB1), MAKE_RGB(0x35,0x70,0xC2), MAKE_RGB(0x46,0x81,0xD3), \
    MAKE_RGB(0x57,0x92,0xE4), MAKE_RGB(0x68,0xA3,0xF5), MAKE_RGB(0x79,0xB4,0xFF), MAKE_RGB(0x8A,0xC5,0xFF), \
    MAKE_RGB(0x9B,0xD6,0xFF), MAKE_RGB(0xAC,0xE7,0xFF), MAKE_RGB(0xBD,0xF8,0xFF), MAKE_RGB(0xCE,0xFF,0xFF   )

define NTSC_TURQUOISE
    MAKE_RGB(0x00,0x1A,0x26), MAKE_RGB(0x00,0x2B,0x3C), MAKE_RGB(0x00,0x3C,0x52), MAKE_RGB(0x00,0x4D,0x68), \
    MAKE_RGB(0x06,0x5E,0x7C), MAKE_RGB(0x17,0x6F,0x8D), MAKE_RGB(0x28,0x80,0x9E), MAKE_RGB(0x39,0x91,0xAF), \
    MAKE_RGB(0x4A,0xA2,0xC0), MAKE_RGB(0x5B,0xB3,0xD1), MAKE_RGB(0x6C,0xC4,0xE2), MAKE_RGB(0x7D,0xD5,0xF3), \
    MAKE_RGB(0x8E,0xE6,0xFF), MAKE_RGB(0x9F,0xF7,0xFF), MAKE_RGB(0xB0,0xFF,0xFF), MAKE_RGB(0xC1,0xFF,0xFF   )

define NTSC_GREEN_BLUE
    MAKE_RGB(0x00,0x24,0x0B), MAKE_RGB(0x00,0x35,0x10), MAKE_RGB(0x00,0x46,0x22), MAKE_RGB(0x00,0x57,0x38), \
    MAKE_RGB(0x05,0x68,0x4D), MAKE_RGB(0x16,0x79,0x5E), MAKE_RGB(0x27,0x8A,0x6F), MAKE_RGB(0x38,0x9B,0x80), \
    MAKE_RGB(0x49,0xAC,0x91), MAKE_RGB(0x5A,0xBD,0xA2), MAKE_RGB(0x6B,0xCE,0xB3), MAKE_RGB(0x7C,0xDF,0xC4), \
    MAKE_RGB(0x8D,0xF0,0xD5), MAKE_RGB(0x9E,0xFF,0xE5), MAKE_RGB(0xAF,0xFF,0xF1), MAKE_RGB(0xC0,0xFF,0xFD   )

define NTSC_GREEN
    MAKE_RGB(0x00,0x27,0x0C), MAKE_RGB(0x00,0x38,0x11), MAKE_RGB(0x00,0x49,0x16), MAKE_RGB(0x00,0x5A,0x1B), \
    MAKE_RGB(0x10,0x6B,0x1B), MAKE_RGB(0x21,0x7C,0x2C), MAKE_RGB(0x32,0x8D,0x3D), MAKE_RGB(0x43,0x9E,0x4E), \
    MAKE_RGB(0x54,0xAF,0x5F), MAKE_RGB(0x65,0xC0,0x70), MAKE_RGB(0x76,0xD1,0x81), MAKE_RGB(0x87,0xE2,0x92), \
    MAKE_RGB(0x98,0xF3,0xA3), MAKE_RGB(0xA9,0xFF,0xB3), MAKE_RGB(0xBA,0xFF,0xBF), MAKE_RGB(0xCB,0xFF,0xCB   )

define NTSC_YELLOW_GREEN
    MAKE_RGB(0x00,0x23,0x0A), MAKE_RGB(0x00,0x34,0x10), MAKE_RGB(0x04,0x45,0x13), MAKE_RGB(0x15,0x56,0x13), \
    MAKE_RGB(0x26,0x67,0x13), MAKE_RGB(0x37,0x78,0x13), MAKE_RGB(0x48,0x89,0x14), MAKE_RGB(0x59,0x9A,0x25), \
    MAKE_RGB(0x6A,0xAB,0x36), MAKE_RGB(0x7B,0xBC,0x47), MAKE_RGB(0x8C,0xCD,0x58), MAKE_RGB(0x9D,0xDE,0x69), \
    MAKE_RGB(0xAE,0xEF,0x7A), MAKE_RGB(0xBF,0xFF,0x8B), MAKE_RGB(0xD0,0xFF,0x97), MAKE_RGB(0xE1,0xFF,0xA3   )

define NTSC_ORANGE_GREEN
    MAKE_RGB(0x00,0x17,0x07), MAKE_RGB(0x0E,0x28,0x08), MAKE_RGB(0x1F,0x39,0x08), MAKE_RGB(0x30,0x4A,0x08), \
    MAKE_RGB(0x41,0x5B,0x08), MAKE_RGB(0x52,0x6C,0x08), MAKE_RGB(0x63,0x7D,0x08), MAKE_RGB(0x74,0x8E,0x0D), \
    MAKE_RGB(0x85,0x9F,0x1E), MAKE_RGB(0x96,0xB0,0x2F), MAKE_RGB(0xA7,0xC1,0x40), MAKE_RGB(0xB8,0xD2,0x51), \
    MAKE_RGB(0xC9,0xE3,0x62), MAKE_RGB(0xDA,0xF4,0x73), MAKE_RGB(0xEB,0xFF,0x82), MAKE_RGB(0xFC,0xFF,0x8E   )

define NTSC_LIGHT_ORANGE
    MAKE_RGB(0x19,0x07,0x00), MAKE_RGB(0x2A,0x18,0x00), MAKE_RGB(0x3B,0x29,0x00), MAKE_RGB(0x4C,0x3A,0x00), \
    MAKE_RGB(0x5D,0x4B,0x00), MAKE_RGB(0x6E,0x5C,0x00), MAKE_RGB(0x7F,0x6D,0x00), MAKE_RGB(0x90,0x7E,0x09), \
    MAKE_RGB(0xA1,0x8F,0x1A), MAKE_RGB(0xB2,0xA0,0x2B), MAKE_RGB(0xC3,0xB1,0x3C), MAKE_RGB(0xD4,0xC2,0x4D), \
    MAKE_RGB(0xE5,0xD3,0x5E), MAKE_RGB(0xF6,0xE4,0x6F), MAKE_RGB(0xFF,0xF5,0x82), MAKE_RGB(0xFF,0xFF,0x96   )
***************************************************************************/

/***************************************************************************
    PALETTE - PHASE 26.7 SHIFT


define NTSC_GREY
    MAKE_RGB(0x00,0x00,0x00), MAKE_RGB(0x11,0x11,0x11), MAKE_RGB(0x22,0x22,0x22), MAKE_RGB(0x33,0x33,0x33), \
    MAKE_RGB(0x44,0x44,0x44), MAKE_RGB(0x55,0x55,0x55), MAKE_RGB(0x66,0x66,0x66), MAKE_RGB(0x77,0x77,0x77), \
    MAKE_RGB(0x88,0x88,0x88), MAKE_RGB(0x99,0x99,0x99), MAKE_RGB(0xAA,0xAA,0xAA), MAKE_RGB(0xBB,0xBB,0xBB), \
    MAKE_RGB(0xCC,0xCC,0xCC), MAKE_RGB(0xDD,0xDD,0xDD), MAKE_RGB(0xEE,0xEE,0xEE), MAKE_RGB(0xFF,0xFF,0xFF   )

define NTSC_GOLD
    MAKE_RGB(0x1A,0x07,0x00), MAKE_RGB(0x2B,0x18,0x00), MAKE_RGB(0x3C,0x29,0x00), MAKE_RGB(0x4D,0x3A,0x00), \
    MAKE_RGB(0x5E,0x4B,0x00), MAKE_RGB(0x6F,0x5C,0x00), MAKE_RGB(0x80,0x6D,0x00), MAKE_RGB(0x91,0x7E,0x09), \
    MAKE_RGB(0xA2,0x8F,0x1A), MAKE_RGB(0xB3,0xA0,0x2B), MAKE_RGB(0xC4,0xB1,0x3C), MAKE_RGB(0xD5,0xC2,0x4D), \
    MAKE_RGB(0xE6,0xD3,0x5E), MAKE_RGB(0xF7,0xE4,0x6F), MAKE_RGB(0xFF,0xF5,0x83), MAKE_RGB(0xFF,0xFF,0x97   )

define NTSC_ORANGE
    MAKE_RGB(0x32,0x00,0x00), MAKE_RGB(0x43,0x06,0x00), MAKE_RGB(0x54,0x17,0x00), MAKE_RGB(0x65,0x28,0x00), \
    MAKE_RGB(0x79,0x39,0x00), MAKE_RGB(0x87,0x4A,0x00), MAKE_RGB(0x98,0x5B,0x0C), MAKE_RGB(0xA9,0x6C,0x1D), \
    MAKE_RGB(0xBA,0x7D,0x2E), MAKE_RGB(0xCB,0x8E,0x3F), MAKE_RGB(0xDC,0x9F,0x50), MAKE_RGB(0xED,0xB0,0x61), \
    MAKE_RGB(0xFE,0xC1,0x72), MAKE_RGB(0xFF,0xD2,0x87), MAKE_RGB(0xFF,0xE3,0x9E), MAKE_RGB(0xFF,0xF4,0xB4   )

define NTSC_RED_ORANGE
    MAKE_RGB(0x3E,0x00,0x00), MAKE_RGB(0x4F,0x00,0x00), MAKE_RGB(0x60,0x07,0x00), MAKE_RGB(0x71,0x18,0x00), \
    MAKE_RGB(0x82,0x29,0x10), MAKE_RGB(0x93,0x3A,0x21), MAKE_RGB(0xA4,0x4B,0x32), MAKE_RGB(0xB5,0x5C,0x43), \
    MAKE_RGB(0xC6,0x6D,0x54), MAKE_RGB(0xD7,0x7E,0x65), MAKE_RGB(0xE8,0x8F,0x76), MAKE_RGB(0xF9,0xA0,0x87), \
    MAKE_RGB(0xFF,0xB1,0x9C), MAKE_RGB(0xFF,0xC2,0xB2), MAKE_RGB(0xFF,0xD3,0xC8), MAKE_RGB(0xFF,0xE4,0xDE   )

define NTSC_PINK
    MAKE_RGB(0x3E,0x00,0x09), MAKE_RGB(0x4F,0x00,0x15), MAKE_RGB(0x60,0x00,0x21), MAKE_RGB(0x71,0x0E,0x31), \
    MAKE_RGB(0x82,0x1F,0x42), MAKE_RGB(0x93,0x30,0x53), MAKE_RGB(0xA4,0x41,0x64), MAKE_RGB(0xB5,0x52,0x75), \
    MAKE_RGB(0xC6,0x63,0x86), MAKE_RGB(0xD7,0x74,0x97), MAKE_RGB(0xE8,0x85,0xA8), MAKE_RGB(0xF9,0x96,0xB9), \
    MAKE_RGB(0xFF,0xA7,0xCE), MAKE_RGB(0xFF,0xB8,0xE4), MAKE_RGB(0xFF,0xC9,0xEF), MAKE_RGB(0xFF,0xDA,0xF4   )

define NTSC_PURPLE
    MAKE_RGB(0x30,0x00,0x3D), MAKE_RGB(0x41,0x00,0x48), MAKE_RGB(0x52,0x00,0x54), MAKE_RGB(0x63,0x0C,0x64), \
    MAKE_RGB(0x74,0x1D,0x75), MAKE_RGB(0x85,0x2E,0x86), MAKE_RGB(0x96,0x3F,0x97), MAKE_RGB(0xA7,0x50,0xA8), \
    MAKE_RGB(0xB8,0x61,0xB9), MAKE_RGB(0xC9,0x72,0xCA), MAKE_RGB(0xDA,0x83,0xDB), MAKE_RGB(0xEB,0x94,0xE5), \
    MAKE_RGB(0xFC,0xA5,0xE5), MAKE_RGB(0xFF,0xB6,0xE9), MAKE_RGB(0xFF,0xC7,0xEE), MAKE_RGB(0xFF,0xD8,0xF3   )

define NTSC_PURPLE_BLUE
    MAKE_RGB(0x18,0x00,0x62), MAKE_RGB(0x29,0x00,0x6E), MAKE_RGB(0x3A,0x01,0x7A), MAKE_RGB(0x4B,0x12,0x8B), \
    MAKE_RGB(0x5C,0x23,0x9C), MAKE_RGB(0x6D,0x34,0xAD), MAKE_RGB(0x7E,0x45,0xBE), MAKE_RGB(0x8F,0x56,0xCF), \
    MAKE_RGB(0xA0,0x67,0xE0), MAKE_RGB(0xB1,0x78,0xEE), MAKE_RGB(0xC2,0x89,0xEE), MAKE_RGB(0xD3,0x9A,0xEE), \
    MAKE_RGB(0xE4,0xAB,0xEE), MAKE_RGB(0xF5,0xBC,0xEE), MAKE_RGB(0xFF,0xCD,0xE0), MAKE_RGB(0xFF,0xDE,0xF5   )

define NTSC_BLUE1
    MAKE_RGB(0x00,0x00,0x72), MAKE_RGB(0x0C,0x00,0x7F), MAKE_RGB(0x1D,0x0E,0x8F), MAKE_RGB(0x2E,0x1F,0xA0), \
    MAKE_RGB(0x3F,0x30,0xB1), MAKE_RGB(0x50,0x41,0xC2), MAKE_RGB(0x61,0x52,0xD3), MAKE_RGB(0x72,0x63,0xE4), \
    MAKE_RGB(0x83,0x74,0xF5), MAKE_RGB(0x94,0x85,0xFA), MAKE_RGB(0xA5,0x96,0xFA), MAKE_RGB(0xB6,0xA7,0xFA), \
    MAKE_RGB(0xC7,0xB8,0xFA), MAKE_RGB(0xD8,0xC9,0xFA), MAKE_RGB(0xE9,0xDA,0xFA), MAKE_RGB(0xFA,0xE8,0xFA   )

define NTSC_BLUE2
    MAKE_RGB(0x00,0x00,0x62), MAKE_RGB(0x00,0x0F,0x77), MAKE_RGB(0x01,0x20,0x8D), MAKE_RGB(0x12,0x31,0x9E), \
    MAKE_RGB(0x23,0x42,0xAF), MAKE_RGB(0x34,0x53,0xC0), MAKE_RGB(0x45,0x64,0xD1), MAKE_RGB(0x56,0x75,0xE2), \
    MAKE_RGB(0x67,0x86,0xF3), MAKE_RGB(0x78,0x97,0xFF), MAKE_RGB(0x89,0xA8,0xFF), MAKE_RGB(0x9A,0xB9,0xFF), \
    MAKE_RGB(0xAB,0xCA,0xFF), MAKE_RGB(0xBC,0xDB,0xFF), MAKE_RGB(0xCD,0xEC,0xFF), MAKE_RGB(0xDE,0xFD,0xFF   )

define NTSC_LIGHT_BLUE
    MAKE_RGB(0x00,0x10,0x42), MAKE_RGB(0x00,0x21,0x58), MAKE_RGB(0x00,0x32,0x6E), MAKE_RGB(0x00,0x43,0x84), \
    MAKE_RGB(0x0E,0x54,0x96), MAKE_RGB(0x1F,0x65,0xA7), MAKE_RGB(0x30,0x76,0xB8), MAKE_RGB(0x41,0x87,0xC9), \
    MAKE_RGB(0x52,0x98,0xDA), MAKE_RGB(0x63,0xA9,0xEB), MAKE_RGB(0x74,0xBA,0xFC), MAKE_RGB(0x85,0xCB,0xFF), \
    MAKE_RGB(0x96,0xDC,0xFF), MAKE_RGB(0xA7,0xED,0xFF), MAKE_RGB(0xB8,0xFE,0xFF), MAKE_RGB(0xC9,0xFF,0xFF   )

define NTSC_TURQUOISE
    MAKE_RGB(0x00,0x1E,0x14), MAKE_RGB(0x00,0x2F,0x2A), MAKE_RGB(0x00,0x40,0x40), MAKE_RGB(0x00,0x51,0x56), \
    MAKE_RGB(0x04,0x62,0x6B), MAKE_RGB(0x15,0x73,0x7C), MAKE_RGB(0x26,0x84,0x8D), MAKE_RGB(0x37,0x95,0x9E), \
    MAKE_RGB(0x48,0xA6,0xAF), MAKE_RGB(0x59,0xB7,0xC0), MAKE_RGB(0x6A,0xC8,0xD1), MAKE_RGB(0x7B,0xD9,0xE2), \
    MAKE_RGB(0x8C,0xEA,0xF3), MAKE_RGB(0x9D,0xFB,0xFF), MAKE_RGB(0xAE,0xFF,0xFF), MAKE_RGB(0xBF,0xFF,0xFF   )

define NTSC_GREEN_BLUE
    MAKE_RGB(0x00,0x26,0x0B), MAKE_RGB(0x00,0x37,0x10), MAKE_RGB(0x00,0x48,0x16), MAKE_RGB(0x00,0x59,0x25), \
    MAKE_RGB(0x08,0x6A,0x38), MAKE_RGB(0x19,0x7B,0x49), MAKE_RGB(0x2A,0x8C,0x5A), MAKE_RGB(0x3B,0x9D,0x6B), \
    MAKE_RGB(0x4C,0xAE,0x7C), MAKE_RGB(0x5D,0xBF,0x8D), MAKE_RGB(0x6E,0xD0,0x9E), MAKE_RGB(0x7F,0xE1,0xAF), \
    MAKE_RGB(0x90,0xF2,0xC0), MAKE_RGB(0xA1,0xFF,0xD0), MAKE_RGB(0xB2,0xFF,0xDC), MAKE_RGB(0xC3,0xFF,0xE8   )

define NTSC_GREEN
    MAKE_RGB(0x00,0x26,0x0B), MAKE_RGB(0x00,0x37,0x10), MAKE_RGB(0x00,0x48,0x16), MAKE_RGB(0x08,0x59,0x18), \
    MAKE_RGB(0x19,0x6A,0x18), MAKE_RGB(0x2A,0x7B,0x18), MAKE_RGB(0x3B,0x8C,0x29), MAKE_RGB(0x4C,0x9D,0x3A), \
    MAKE_RGB(0x5D,0xAE,0x4B), MAKE_RGB(0x6E,0xBF,0x5C), MAKE_RGB(0x7F,0xD0,0x6D), MAKE_RGB(0x90,0xE1,0x7E), \
    MAKE_RGB(0xA1,0xF2,0x8F), MAKE_RGB(0xB2,0xFF,0x9F), MAKE_RGB(0xC3,0xFF,0xAB), MAKE_RGB(0xD4,0xFF,0xB7   )

define NTSC_YELLOW_GREEN
    MAKE_RGB(0x00,0x1E,0x09), MAKE_RGB(0x00,0x2F,0x0E), MAKE_RGB(0x11,0x40,0x0E), MAKE_RGB(0x22,0x51,0x0E), \
    MAKE_RGB(0x33,0x62,0x0E), MAKE_RGB(0x44,0x73,0x0E), MAKE_RGB(0x55,0x84,0x0E), MAKE_RGB(0x66,0x95,0x17), \
    MAKE_RGB(0x77,0xA6,0x28), MAKE_RGB(0x88,0xB7,0x39), MAKE_RGB(0x99,0xC8,0x4A), MAKE_RGB(0xAA,0xD9,0x5B), \
    MAKE_RGB(0xBB,0xEA,0x6C), MAKE_RGB(0xCC,0xFB,0x7D), MAKE_RGB(0xDD,0xFF,0x8A), MAKE_RGB(0xEE,0xFF,0x96   )

define NTSC_ORANGE_GREEN
    MAKE_RGB(0x0D,0x0F,0x01), MAKE_RGB(0x1E,0x20,0x01), MAKE_RGB(0x2F,0x31,0x01), MAKE_RGB(0x40,0x42,0x01), \
    MAKE_RGB(0x51,0x53,0x01), MAKE_RGB(0x62,0x64,0x01), MAKE_RGB(0x73,0x75,0x01), MAKE_RGB(0x84,0x86,0x08), \
    MAKE_RGB(0x95,0x97,0x19), MAKE_RGB(0xA6,0xA8,0x2A), MAKE_RGB(0xB7,0xB9,0x3B), MAKE_RGB(0xC8,0xCA,0x4C), \
    MAKE_RGB(0xD9,0xDB,0x5D), MAKE_RGB(0xEA,0xEC,0x6E), MAKE_RGB(0xFB,0xFD,0x7F), MAKE_RGB(0xFF,0xFF,0x8F   )

define NTSC_LIGHT_ORANGE
    MAKE_RGB(0x28,0x00,0x00), MAKE_RGB(0x39,0x0E,0x00), MAKE_RGB(0x4A,0x1F,0x00), MAKE_RGB(0x5B,0x30,0x00), \
    MAKE_RGB(0x6C,0x41,0x00), MAKE_RGB(0x7D,0x52,0x00), MAKE_RGB(0x8E,0x63,0x00), MAKE_RGB(0x9F,0x74,0x10), \
    MAKE_RGB(0xB0,0x85,0x21), MAKE_RGB(0xC1,0x96,0x32), MAKE_RGB(0xD2,0xA7,0x43), MAKE_RGB(0xE3,0xB8,0x54), \
    MAKE_RGB(0xF4,0xC9,0x65), MAKE_RGB(0xFF,0xDA,0x78), MAKE_RGB(0xFF,0xEB,0x8E), MAKE_RGB(0xFF,0xFC,0xA4   )
***************************************************************************/


/***************************************************************************
    PALETTE - PHASE 27.2 SHIFT


define NTSC_GREY
    MAKE_RGB(0x00,0x00,0x00), MAKE_RGB(0x11,0x11,0x11), MAKE_RGB(0x22,0x22,0x22), MAKE_RGB(0x33,0x33,0x33), \
    MAKE_RGB(0x44,0x44,0x44), MAKE_RGB(0x55,0x55,0x55), MAKE_RGB(0x66,0x66,0x66), MAKE_RGB(0x77,0x77,0x77), \
    MAKE_RGB(0x88,0x88,0x88), MAKE_RGB(0x99,0x99,0x99), MAKE_RGB(0xAA,0xAA,0xAA), MAKE_RGB(0xBB,0xBB,0xBB), \
    MAKE_RGB(0xCC,0xCC,0xCC), MAKE_RGB(0xDD,0xDD,0xDD), MAKE_RGB(0xEE,0xEE,0xEE), MAKE_RGB(0xFF,0xFF,0xFF   )

define NTSC_GOLD
    MAKE_RGB(0x1A,0x07,0x00), MAKE_RGB(0x2B,0x18,0x00), MAKE_RGB(0x3C,0x29,0x00), MAKE_RGB(0x4D,0x3A,0x00), \
    MAKE_RGB(0x5E,0x4B,0x00), MAKE_RGB(0x6F,0x5C,0x00), MAKE_RGB(0x80,0x6D,0x00), MAKE_RGB(0x91,0x7E,0x09), \
    MAKE_RGB(0xA2,0x8F,0x1A), MAKE_RGB(0xB3,0xA0,0x2B), MAKE_RGB(0xC4,0xB1,0x3C), MAKE_RGB(0xD5,0xC2,0x4D), \
    MAKE_RGB(0xE6,0xD3,0x5E), MAKE_RGB(0xF7,0xE4,0x6F), MAKE_RGB(0xFF,0xF5,0x83), MAKE_RGB(0xFF,0xF7,0x97   )

define NTSC_ORANGE
    MAKE_RGB(0x32,0x00,0x00), MAKE_RGB(0x43,0x05,0x00), MAKE_RGB(0x54,0x16,0x00), MAKE_RGB(0x65,0x27,0x00), \
    MAKE_RGB(0x76,0x38,0x00), MAKE_RGB(0x87,0X49,0x00), MAKE_RGB(0x98,0x5A,0x0C), MAKE_RGB(0xA9,0x6B,0x1D), \
    MAKE_RGB(0xBA,0x7C,0x2E), MAKE_RGB(0xCB,0x8D,0x3F), MAKE_RGB(0xDC,0x9E,0x50), MAKE_RGB(0xED,0xAF,0x61), \
    MAKE_RGB(0xFE,0xC0,0x72), MAKE_RGB(0xFF,0xD1,0x88), MAKE_RGB(0xFF,0xE2,0x9E), MAKE_RGB(0xFF,0xF3,0xB4   )

define NTSC_RED_ORANGE
    MAKE_RGB(0x3F,0x00,0x00), MAKE_RGB(0x50,0x00,0x00), MAKE_RGB(0x61,0x07,0x00), MAKE_RGB(0x72,0x18,0x01), \
    MAKE_RGB(0x83,0x29,0x12), MAKE_RGB(0x94,0x3A,0x23), MAKE_RGB(0xA5,0x4B,0x34), MAKE_RGB(0xB6,0x5C,0x45), \
    MAKE_RGB(0xC7,0x6D,0x56), MAKE_RGB(0xD8,0x7E,0x67), MAKE_RGB(0xE9,0x8F,0x78), MAKE_RGB(0xFA,0xA0,0x89), \
    MAKE_RGB(0xFF,0xB1,0x9E), MAKE_RGB(0xFF,0xC2,0xB4), MAKE_RGB(0xFF,0xD3,0xCA), MAKE_RGB(0xFF,0xE4,0xE0   )

define NTSC_PINK
    MAKE_RGB(0x3E,0x00,0x0C), MAKE_RGB(0x4F,0x00,0x18), MAKE_RGB(0x60,0x00,0x24), MAKE_RGB(0x71,0x0E,0x34), \
    MAKE_RGB(0x82,0x1F,0x45), MAKE_RGB(0x93,0x30,0x56), MAKE_RGB(0xA4,0x41,0x67), MAKE_RGB(0xB5,0x52,0x78), \
    MAKE_RGB(0xC6,0x63,0x89), MAKE_RGB(0xD7,0x74,0x9A), MAKE_RGB(0xE8,0x85,0xAB), MAKE_RGB(0xF9,0x96,0xB6), \
    MAKE_RGB(0xFF,0xA7,0xD1), MAKE_RGB(0xFF,0xB8,0xE7), MAKE_RGB(0xFF,0xC9,0xEF), MAKE_RGB(0xFF,0xDA,0xF4   )

define NTSC_PURPLE
    MAKE_RGB(0x2F,0x00,0x3F), MAKE_RGB(0x40,0x00,0x4B), MAKE_RGB(0x51,0x00,0x57), MAKE_RGB(0x62,0x0C,0x66), \
    MAKE_RGB(0x73,0x1D,0x77), MAKE_RGB(0x84,0x2E,0x88), MAKE_RGB(0x95,0x3F,0x99), MAKE_RGB(0xA6,0x50,0xAA), \
    MAKE_RGB(0xB7,0x61,0xBB), MAKE_RGB(0xC8,0x72,0xCC), MAKE_RGB(0xD9,0x83,0xDD), MAKE_RGB(0xEA,0x94,0xE5), \
    MAKE_RGB(0xFB,0xA5,0xE5), MAKE_RGB(0xFF,0xB6,0xE9), MAKE_RGB(0xFF,0xC7,0xEE), MAKE_RGB(0xFF,0xD8,0xF3   )

define NTSC_PURPLE_BLUE
    MAKE_RGB(0x16,0x00,0x64), MAKE_RGB(0x27,0x00,0x70), MAKE_RGB(0x38,0x02,0x7D), MAKE_RGB(0x49,0x13,0x8E), \
    MAKE_RGB(0x5A,0x24,0x9F), MAKE_RGB(0x6B,0x35,0xB0), MAKE_RGB(0x7C,0x46,0xC1), MAKE_RGB(0x8D,0x57,0xD2), \
    MAKE_RGB(0x9E,0x68,0xE3), MAKE_RGB(0xAF,0x79,0xEF), MAKE_RGB(0xC0,0x8A,0xEF), MAKE_RGB(0xD1,0x9D,0xEF), \
    MAKE_RGB(0xE2,0xAC,0xEF), MAKE_RGB(0xF3,0xBD,0xEF), MAKE_RGB(0xFF,0xCE,0xF0), MAKE_RGB(0xFF,0xDF,0xF5   )

define NTSC_BLUE1
    MAKE_RGB(0x00,0x00,0x71), MAKE_RGB(0x09,0x00,0x7F), MAKE_RGB(0x1A,0x10,0x90), MAKE_RGB(0x2B,0x21,0xA1), \
    MAKE_RGB(0x3C,0x32,0xB2), MAKE_RGB(0x4D,0x43,0xC3), MAKE_RGB(0x5E,0x54,0xD4), MAKE_RGB(0x6F,0x65,0xE5), \
    MAKE_RGB(0x80,0x76,0xF6), MAKE_RGB(0x91,0x87,0xFC), MAKE_RGB(0xA2,0x98,0xFC), MAKE_RGB(0xB3,0xA9,0xFC), \
    MAKE_RGB(0xC4,0xBA,0xFC), MAKE_RGB(0xD5,0xCB,0xFC), MAKE_RGB(0xE6,0xDC,0xFC), MAKE_RGB(0xF7,0xED,0xFC   )

define NTSC_BLUE2
    MAKE_RGB(0x00,0x00,0x5E), MAKE_RGB(0x00,0x11,0x74), MAKE_RGB(0x00,0x22,0x8A), MAKE_RGB(0x0F,0x33,0x9C), \
    MAKE_RGB(0x20,0x44,0xAD), MAKE_RGB(0x31,0x55,0xBE), MAKE_RGB(0x42,0x66,0xCF), MAKE_RGB(0x53,0x77,0xE0), \
    MAKE_RGB(0x64,0x88,0xF1), MAKE_RGB(0x75,0x99,0xFF), MAKE_RGB(0x86,0xAA,0xFF), MAKE_RGB(0x97,0xBB,0xFF), \
    MAKE_RGB(0xA8,0xCC,0xFF), MAKE_RGB(0xB9,0xDD,0xFF), MAKE_RGB(0xCA,0xEE,0xFF), MAKE_RGB(0xDB,0xFF,0xFF   )

define NTSC_LIGHT_BLUE
    MAKE_RGB(0x00,0x12,0x3B), MAKE_RGB(0x00,0x23,0x51), MAKE_RGB(0x00,0x34,0x68), MAKE_RGB(0x00,0x45,0x7E), \
    MAKE_RGB(0x0C,0x56,0x90), MAKE_RGB(0x1D,0x67,0xA1), MAKE_RGB(0x2E,0x78,0xB2), MAKE_RGB(0x3F,0x89,0xC3), \
    MAKE_RGB(0x50,0x9A,0xD4), MAKE_RGB(0x61,0xAB,0xE5), MAKE_RGB(0x72,0xBC,0xF6), MAKE_RGB(0x83,0xCD,0xFF), \
    MAKE_RGB(0x94,0xDE,0xFF), MAKE_RGB(0xA5,0xEF,0xFF), MAKE_RGB(0xB6,0xFF,0xFF), MAKE_RGB(0xC7,0xFF,0xFF   )

define NTSC_TURQUOISE
    MAKE_RGB(0x00,0x20,0x0C), MAKE_RGB(0x00,0x31,0x22), MAKE_RGB(0x00,0x42,0x38), MAKE_RGB(0x00,0x53,0x4E), \
    MAKE_RGB(0x04,0x64,0x63), MAKE_RGB(0x15,0x75,0x74), MAKE_RGB(0x26,0x86,0x85), MAKE_RGB(0x37,0x97,0x96), \
    MAKE_RGB(0x48,0xA8,0xA7), MAKE_RGB(0x59,0xB9,0xB8), MAKE_RGB(0x6A,0xCA,0xC9), MAKE_RGB(0x7B,0xDB,0xDA), \
    MAKE_RGB(0x8C,0xEC,0xEB), MAKE_RGB(0x9D,0xFD,0xFC), MAKE_RGB(0xAE,0xFF,0xFF), MAKE_RGB(0xBF,0xFF,0xFF   )

define NTSC_GREEN_BLUE
    MAKE_RGB(0x00,0x27,0x0C), MAKE_RGB(0x00,0x38,0x11), MAKE_RGB(0x00,0x49,0x16), MAKE_RGB(0x00,0x5A,0x1C), \
    MAKE_RGB(0x0B,0x6B,0x2F), MAKE_RGB(0x1C,0x7C,0x40), MAKE_RGB(0x2D,0x8D,0x51), MAKE_RGB(0x3E,0x9E,0x62), \
    MAKE_RGB(0x4F,0xAF,0x73), MAKE_RGB(0x60,0xC0,0x84), MAKE_RGB(0x71,0xD1,0x95), MAKE_RGB(0x82,0xE2,0xA6), \
    MAKE_RGB(0x93,0xF3,0xB7), MAKE_RGB(0xA4,0xFF,0xC6), MAKE_RGB(0xB5,0xFF,0xD2), MAKE_RGB(0xC6,0xFF,0xDE   )

define NTSC_GREEN
    MAKE_RGB(0x00,0x25,0x0B), MAKE_RGB(0x00,0x36,0x10), MAKE_RGB(0x00,0x47,0x15), MAKE_RGB(0x0D,0x58,0x16), \
    MAKE_RGB(0x1E,0x69,0x16), MAKE_RGB(0x2F,0x7A,0x16), MAKE_RGB(0x40,0x8B,0x21), MAKE_RGB(0x51,0x9C,0x32), \
    MAKE_RGB(0x62,0xAD,0x43), MAKE_RGB(0x73,0xBE,0x54), MAKE_RGB(0x84,0xCF,0x65), MAKE_RGB(0x95,0xE0,0x76), \
    MAKE_RGB(0xA6,0xF1,0x87), MAKE_RGB(0xB7,0xFF,0x98), MAKE_RGB(0xC8,0xFF,0xA3), MAKE_RGB(0xD9,0xFF,0xAF   )

define NTSC_YELLOW_GREEN
    MAKE_RGB(0x00,0x1B,0x08), MAKE_RGB(0x06,0x2C,0x0B), MAKE_RGB(0x17,0x3D,0x0B), MAKE_RGB(0x28,0x4E,0x0B), \
    MAKE_RGB(0x39,0x5F,0x0B), MAKE_RGB(0x4A,0x70,0x0B), MAKE_RGB(0x5B,0x81,0x0B), MAKE_RGB(0x6C,0x92,0x12), \
    MAKE_RGB(0x7D,0xA3,0x23), MAKE_RGB(0x8E,0xB4,0x34), MAKE_RGB(0x9F,0xC5,0x45), MAKE_RGB(0xB0,0xD6,0x56), \
    MAKE_RGB(0xC1,0xE7,0x67), MAKE_RGB(0xD2,0xF8,0x78), MAKE_RGB(0xE3,0xFF,0x86), MAKE_RGB(0xF4,0xFF,0x92   )

define NTSC_ORANGE_GREEN
    MAKE_RGB(0x13,0x0B,0x00), MAKE_RGB(0x24,0x1C,0x00), MAKE_RGB(0x35,0x2D,0x00), MAKE_RGB(0x46,0x3E,0x00), \
    MAKE_RGB(0x57,0x4F,0x00), MAKE_RGB(0x68,0x60,0x00), MAKE_RGB(0x79,0x71,0x00), MAKE_RGB(0x8A,0x82,0x08), \
    MAKE_RGB(0x9B,0x93,0x19), MAKE_RGB(0xAC,0xA4,0x2A), MAKE_RGB(0xBD,0xB5,0x3B), MAKE_RGB(0xCE,0xC6,0x4C), \
    MAKE_RGB(0xDF,0xD7,0x5D), MAKE_RGB(0xF0,0xE8,0x6E), MAKE_RGB(0xFF,0xF9,0x7F), MAKE_RGB(0xFF,0xFF,0x92   )

define NTSC_LIGHT_ORANGE
    MAKE_RGB(0x2D,0x00,0x00), MAKE_RGB(0x3E,0x0A,0x00), MAKE_RGB(0x4F,0x1B,0x00), MAKE_RGB(0x60,0x2C,0x00), \
    MAKE_RGB(0x71,0x3D,0x00), MAKE_RGB(0x82,0x4E,0x00), MAKE_RGB(0x93,0x5F,0x05), MAKE_RGB(0xA4,0x70,0x16), \
    MAKE_RGB(0xB5,0x81,0x27), MAKE_RGB(0xC4,0x90,0x37), MAKE_RGB(0xD7,0xA3,0x49), MAKE_RGB(0xE8,0xB4,0x5A), \
    MAKE_RGB(0xF9,0xC5,0x6B), MAKE_RGB(0xFF,0xD6,0x80), MAKE_RGB(0xFF,0xE7,0x96), MAKE_RGB(0xFF,0xF8,0xAC   )
***************************************************************************/


/***************************************************************************
    PALETTE - 27.7 PHASE SHIFT


define NTSC_GREY
    MAKE_RGB(0x00,0x00,0x00), MAKE_RGB(0x11,0x11,0x11), MAKE_RGB(0x22,0x22,0x22), MAKE_RGB(0x33,0x33,0x33), \
    MAKE_RGB(0x44,0x44,0x44), MAKE_RGB(0x55,0x55,0x55), MAKE_RGB(0x66,0x66,0x66), MAKE_RGB(0x77,0x77,0x77), \
    MAKE_RGB(0x88,0x88,0x88), MAKE_RGB(0x99,0x99,0x99), MAKE_RGB(0xAA,0xAA,0xAA), MAKE_RGB(0xBB,0xBB,0xBB), \
    MAKE_RGB(0xCC,0xCC,0xCC), MAKE_RGB(0xDD,0xDD,0xDD), MAKE_RGB(0xEE,0xEE,0xEE), MAKE_RGB(0xFF,0xFF,0xFF   )

define NTSC_GOLD
    MAKE_RGB(0x1A,0x07,0x00), MAKE_RGB(0x2B,0x18,0x00), MAKE_RGB(0x3C,0x29,0x00), MAKE_RGB(0x4D,0x3A,0x00), \
    MAKE_RGB(0x5E,0x4B,0x00), MAKE_RGB(0x6F,0x5C,0x00), MAKE_RGB(0x80,0x6D,0x00), MAKE_RGB(0x91,0x7E,0x09), \
    MAKE_RGB(0xA2,0x8F,0x1A), MAKE_RGB(0xB3,0xA0,0x2B), MAKE_RGB(0xC4,0xB1,0x3C), MAKE_RGB(0xD5,0xC2,0x4D), \
    MAKE_RGB(0xE6,0xD3,0x5E), MAKE_RGB(0xF7,0xE4,0x6F), MAKE_RGB(0xFF,0xF5,0x83), MAKE_RGB(0xFF,0xFF,0x97   )

define NTSC_ORANGE
    MAKE_RGB(0x32,0x00,0x00), MAKE_RGB(0x43,0x05,0x00), MAKE_RGB(0x54,0x16,0x00), MAKE_RGB(0x65,0x27,0x00), \
    MAKE_RGB(0x76,0x38,0x00), MAKE_RGB(0x87,0x49,0x00), MAKE_RGB(0x98,0x5A,0x0C), MAKE_RGB(0xA9,0x6B,0x1D), \
    MAKE_RGB(0xBA,0x7C,0x2E), MAKE_RGB(0xCB,0x8D,0x3F), MAKE_RGB(0xDC,0x9E,0x50), MAKE_RGB(0xED,0xAF,0x61), \
    MAKE_RGB(0xFE,0xC0,0x72), MAKE_RGB(0xFF,0xD1,0x88), MAKE_RGB(0xFF,0xE2,0x9E), MAKE_RGB(0xFF,0xF3,0xB4   )

define NTSC_RED_ORANGE
    MAKE_RGB(0x3F,0x00,0x00), MAKE_RGB(0x50,0x00,0x00), MAKE_RGB(0x61,0x06,0x00), MAKE_RGB(0x72,0x17,0x03), \
    MAKE_RGB(0x83,0x28,0x14), MAKE_RGB(0x94,0x39,0x25), MAKE_RGB(0xA5,0x4A,0x36), MAKE_RGB(0xB6,0x5B,0x47), \
    MAKE_RGB(0xC7,0x6C,0x58), MAKE_RGB(0xD8,0x7D,0x69), MAKE_RGB(0xE9,0x8E,0x7A), MAKE_RGB(0xFA,0x9F,0x8B), \
    MAKE_RGB(0xFF,0xB0,0x9F), MAKE_RGB(0xFF,0xC1,0xB5), MAKE_RGB(0xFF,0xD2,0xCB), MAKE_RGB(0xFF,0xE3,0xE1   )

define NTSC_PINK
    MAKE_RGB(0x3D,0x00,0x10), MAKE_RGB(0x4E,0x00,0x1C), MAKE_RGB(0x5F,0x00,0x27), MAKE_RGB(0x70,0x0D,0x37), \
    MAKE_RGB(0x81,0x1E,0x48), MAKE_RGB(0x92,0x2F,0x59), MAKE_RGB(0xA3,0x40,0x6A), MAKE_RGB(0xB4,0x51,0x7B), \
    MAKE_RGB(0xC5,0x62,0x8C), MAKE_RGB(0xD6,0x73,0x9D), MAKE_RGB(0xE7,0x84,0xAE), MAKE_RGB(0xF8,0x95,0xBF), \
    MAKE_RGB(0xFF,0xA6,0xD3), MAKE_RGB(0xFF,0xB7,0xE9), MAKE_RGB(0xFF,0xC8,0xEE), MAKE_RGB(0xFF,0xD9,0xF4   )

define NTSC_PURPLE
    MAKE_RGB(0x2D,0x00,0x42), MAKE_RGB(0x3E,0x00,0x4E), MAKE_RGB(0x4F,0x00,0x5A), MAKE_RGB(0x60,0x0C,0x6A), \
    MAKE_RGB(0x71,0x1D,0x7B), MAKE_RGB(0x82,0x2E,0x8C), MAKE_RGB(0x93,0x3F,0x9D), MAKE_RGB(0xA4,0x50,0xAE), \
    MAKE_RGB(0xB5,0x61,0xBF), MAKE_RGB(0xC6,0x72,0xD0), MAKE_RGB(0xD7,0x83,0xE1), MAKE_RGB(0xE8,0x94,0xE6), \
    MAKE_RGB(0xF9,0xA5,0xE6), MAKE_RGB(0xFF,0xB6,0xE9), MAKE_RGB(0xFF,0xC7,0xEE), MAKE_RGB(0xFF,0xD8,0xF3   )

define NTSC_PURPLE_BLUE
    MAKE_RGB(0x13,0x00,0x67), MAKE_RGB(0x24,0x00,0x73), MAKE_RGB(0x35,0x03,0x80), MAKE_RGB(0x46,0x14,0x91), \
    MAKE_RGB(0x57,0x25,0xA2), MAKE_RGB(0x68,0x36,0xB3), MAKE_RGB(0x79,0x47,0xC4), MAKE_RGB(0x8A,0x58,0xD5), \
    MAKE_RGB(0x9B,0x69,0xE6), MAKE_RGB(0xAC,0x7A,0xF0), MAKE_RGB(0xBD,0x8B,0xF0), MAKE_RGB(0xCE,0x9C,0xF0), \
    MAKE_RGB(0xDF,0xAD,0xF0), MAKE_RGB(0xF0,0xBE,0xF0), MAKE_RGB(0xFF,0xCF,0xF1), MAKE_RGB(0xFF,0xE0,0xF6   )

define NTSC_BLUE1
    MAKE_RGB(0x00,0x00,0x70), MAKE_RGB(0x05,0x01,0x80), MAKE_RGB(0x16,0x12,0x91), MAKE_RGB(0x27,0x23,0xA2), \
    MAKE_RGB(0x38,0x34,0xB3), MAKE_RGB(0x49,0x45,0xC4), MAKE_RGB(0x5A,0x56,0xD5), MAKE_RGB(0x6B,0x67,0xE6), \
    MAKE_RGB(0x7C,0x78,0xF7), MAKE_RGB(0x8D,0x89,0xFE), MAKE_RGB(0x9E,0x9A,0xFE), MAKE_RGB(0xAF,0xAB,0xFE), \
    MAKE_RGB(0xC0,0xBC,0xFE), MAKE_RGB(0xD1,0xCD,0xFE), MAKE_RGB(0xE2,0xDE,0xFE), MAKE_RGB(0xF3,0xEF,0xFE   )

define NTSC_BLUE2
    MAKE_RGB(0x00,0x03,0x5B), MAKE_RGB(0x00,0x14,0x71), MAKE_RGB(0x00,0x25,0x87), MAKE_RGB(0x0C,0x36,0x9A), \
    MAKE_RGB(0x1D,0x47,0xAB), MAKE_RGB(0x2E,0x58,0xBC), MAKE_RGB(0x3F,0x69,0xCD), MAKE_RGB(0x50,0x7A,0xDE), \
    MAKE_RGB(0x61,0x8B,0xEF), MAKE_RGB(0x72,0x9C,0xFF), MAKE_RGB(0x83,0xAD,0xFF), MAKE_RGB(0x94,0xBE,0xFF), \
    MAKE_RGB(0xA5,0xCF,0xFF), MAKE_RGB(0xB6,0xE0,0xFF), MAKE_RGB(0xC7,0xF1,0xFF), MAKE_RGB(0xD8,0xFF,0xFF   )

define NTSC_LIGHT_BLUE
    MAKE_RGB(0x00,0x15,0x35), MAKE_RGB(0x00,0x26,0x4B), MAKE_RGB(0x00,0x37,0x61), MAKE_RGB(0x00,0x48,0x78), \
    MAKE_RGB(0x0A,0x59,0x8B), MAKE_RGB(0x1B,0x6A,0x9C), MAKE_RGB(0x2C,0x7B,0xAD), MAKE_RGB(0x3D,0x8C,0xBE), \
    MAKE_RGB(0x4E,0x9D,0xCF), MAKE_RGB(0x5F,0xAE,0xE0), MAKE_RGB(0x70,0xBF,0xF1), MAKE_RGB(0x81,0xD0,0xFF), \
    MAKE_RGB(0x92,0xE1,0xFF), MAKE_RGB(0xA3,0xF2,0xFF), MAKE_RGB(0xB4,0xFF,0xFF), MAKE_RGB(0xC5,0xFF,0xFF   )

define NTSC_TURQUOISE
    MAKE_RGB(0x00,0x22,0x0A), MAKE_RGB(0x00,0x33,0x19), MAKE_RGB(0x00,0x44,0x2F), MAKE_RGB(0x00,0x55,0x45), \
    MAKE_RGB(0x04,0x66,0x5A), MAKE_RGB(0x15,0x77,0x6B), MAKE_RGB(0x26,0x88,0x7C), MAKE_RGB(0x37,0x99,0x8D), \
    MAKE_RGB(0x48,0xAA,0x9E), MAKE_RGB(0x59,0xBB,0xAF), MAKE_RGB(0x6A,0xCC,0xC0), MAKE_RGB(0x7B,0xDD,0xD1), \
    MAKE_RGB(0x8C,0xEE,0xE2), MAKE_RGB(0x9D,0xFF,0xF3), MAKE_RGB(0xAE,0xFF,0xFF), MAKE_RGB(0xBF,0xFF,0xFF   )

define NTSC_GREEN_BLUE
    MAKE_RGB(0x00,0x27,0x0C), MAKE_RGB(0x00,0x38,0x11), MAKE_RGB(0x00,0x49,0x16), MAKE_RGB(0x00,0x5A,0x1B), \
    MAKE_RGB(0x0D,0x6B,0x25), MAKE_RGB(0x1E,0x7C,0x36), MAKE_RGB(0x2F,0x8D,0x47), MAKE_RGB(0x40,0x9E,0x58), \
    MAKE_RGB(0x51,0xAF,0x69), MAKE_RGB(0x62,0xC0,0x7A), MAKE_RGB(0x73,0xD1,0x8B), MAKE_RGB(0x84,0xE2,0x9C), \
    MAKE_RGB(0x95,0xF3,0xAD), MAKE_RGB(0xA6,0xFF,0xBD), MAKE_RGB(0xB7,0xFF,0xC9), MAKE_RGB(0xC8,0xFF,0xD4   )

define NTSC_GREEN
    MAKE_RGB(0x00,0x24,0x0B), MAKE_RGB(0x00,0x35,0x10), MAKE_RGB(0x01,0x46,0x15), MAKE_RGB(0x12,0x57,0x15), \
    MAKE_RGB(0x23,0x68,0x15), MAKE_RGB(0x34,0x79,0x15), MAKE_RGB(0x45,0x8A,0x19), MAKE_RGB(0x56,0x9B,0x2A), \
    MAKE_RGB(0x67,0xAC,0x3B), MAKE_RGB(0x78,0xBD,0x4C), MAKE_RGB(0x89,0xCE,0x5D), MAKE_RGB(0x9A,0xDF,0x6E), \
    MAKE_RGB(0xAB,0xF0,0x7F), MAKE_RGB(0xBC,0xFF,0x8F), MAKE_RGB(0xCD,0xFF,0x9B), MAKE_RGB(0xDE,0xFF,0xA7   )

define NTSC_YELLOW_GREEN
    MAKE_RGB(0x00,0x18,0x07), MAKE_RGB(0x00,0x29,0x0C), MAKE_RGB(0x1E,0x3A,0x08), MAKE_RGB(0x2F,0x4B,0x08), \
    MAKE_RGB(0x40,0x5C,0x08), MAKE_RGB(0x51,0x6D,0x08), MAKE_RGB(0x62,0x7E,0x08), MAKE_RGB(0x73,0x8F,0x0D), \
    MAKE_RGB(0x84,0xA0,0x1E), MAKE_RGB(0x95,0xB1,0x2F), MAKE_RGB(0xA6,0xC2,0x40), MAKE_RGB(0xB7,0xD3,0x51), \
    MAKE_RGB(0xC8,0xE4,0x62), MAKE_RGB(0xD9,0xF5,0x73), MAKE_RGB(0xEA,0xFF,0x82), MAKE_RGB(0xFB,0xFF,0x8E   )

define NTSC_ORANGE_GREEN
    MAKE_RGB(0x1B,0x07,0x00), MAKE_RGB(0x2C,0x18,0x00), MAKE_RGB(0x3D,0x29,0x00), MAKE_RGB(0x4E,0x3A,0x00), \
    MAKE_RGB(0x5F,0x4B,0x00), MAKE_RGB(0x70,0x5C,0x00), MAKE_RGB(0x81,0x6D,0x00), MAKE_RGB(0x92,0x7E,0x09), \
    MAKE_RGB(0xA3,0x8F,0x1A), MAKE_RGB(0xB4,0xA0,0x2B), MAKE_RGB(0xC5,0xB1,0x3C), MAKE_RGB(0xD6,0xC2,0x4D), \
    MAKE_RGB(0xE7,0xD3,0x5E), MAKE_RGB(0xF8,0xE4,0x6F), MAKE_RGB(0xFF,0xF5,0x83), MAKE_RGB(0xFF,0xFF,0x97   )

define NTSC_LIGHT_ORANGE
    MAKE_RGB(0x33,0x00,0x00), MAKE_RGB(0x44,0x05,0x00), MAKE_RGB(0x55,0x16,0x00), MAKE_RGB(0x66,0x27,0x00), \
    MAKE_RGB(0x77,0x38,0x00), MAKE_RGB(0x88,0x49,0x00), MAKE_RGB(0x99,0x5A,0x0D), MAKE_RGB(0xAA,0x6B,0x1E), \
    MAKE_RGB(0xBB,0x7C,0x2F), MAKE_RGB(0xCC,0x8D,0x40), MAKE_RGB(0xDD,0x9E,0x51), MAKE_RGB(0xEE,0xAF,0x62), \
    MAKE_RGB(0xFF,0xC0,0x73), MAKE_RGB(0xFF,0xD1,0x89), MAKE_RGB(0xFF,0xE2,0x9F), MAKE_RGB(0xFF,0xF3,0xB5   )
***************************************************************************/

/* Initialise the palette */
void a7800_state::palette_init()
{
	palette_set_colors(machine(), 0, a7800_palette, ARRAY_LENGTH(a7800_palette));
}


PALETTE_INIT_MEMBER(a7800_state,a7800p)
{
	palette_set_colors(machine(), 0, a7800p_palette, ARRAY_LENGTH(a7800p_palette));
}


/***************************************************************************
    MACHINE DRIVERS
***************************************************************************/

static MACHINE_CONFIG_START( a7800_ntsc, a7800_state )
	/* basic machine hardware */
	MCFG_CPU_ADD("maincpu", M6502, A7800_NTSC_Y1/8) /* 1.79 MHz (switches to 1.19 MHz on TIA or RIOT access) */
	MCFG_CPU_PROGRAM_MAP(a7800_mem)
	MCFG_TIMER_DRIVER_ADD_SCANLINE("scantimer", a7800_state, a7800_interrupt, "screen", 0, 1)


	/* video hardware */
	MCFG_SCREEN_ADD("screen", RASTER)
		MCFG_SCREEN_RAW_PARAMS( 7159090, 456, 0, 320, 262, 26, 26 + 192 + 30 )
	MCFG_SCREEN_UPDATE_DRIVER(a7800_state, screen_update_a7800)

	MCFG_PALETTE_LENGTH(ARRAY_LENGTH(a7800_palette))


	/* sound hardware */
	MCFG_SPEAKER_STANDARD_MONO("mono")
	MCFG_SOUND_TIA_ADD("tia", 31400)
	MCFG_SOUND_ROUTE(ALL_OUTPUTS, "mono", 1.00)
	MCFG_POKEY_ADD("pokey", A7800_NTSC_Y1/8)
	MCFG_SOUND_ROUTE(ALL_OUTPUTS, "mono", 1.00)

	/* devices */
	MCFG_RIOT6532_ADD("riot", A7800_NTSC_Y1/12, a7800_r6532_interface)

	MCFG_CARTSLOT_ADD("cart")
	MCFG_CARTSLOT_EXTENSION_LIST("bin,a78")
	MCFG_CARTSLOT_NOT_MANDATORY
	MCFG_CARTSLOT_LOAD(a7800_state,a7800_cart)
	MCFG_CARTSLOT_PARTIALHASH(a7800_partialhash)
	MCFG_CARTSLOT_INTERFACE("a7800_cart")

	/* software lists */
	MCFG_SOFTWARE_LIST_ADD("cart_list","a7800")
	MCFG_SOFTWARE_LIST_FILTER("cart_list","NTSC")
MACHINE_CONFIG_END


static MACHINE_CONFIG_DERIVED( a7800_pal, a7800_ntsc )

	/* basic machine hardware */
	MCFG_CPU_MODIFY("maincpu")
	MCFG_CPU_CLOCK(CLK_PAL)
//  MCFG_TIMER_ADD_SCANLINE("scantimer", a7800_interrupt, "screen", 0, 1)

	MCFG_SCREEN_MODIFY( "screen" )
		MCFG_SCREEN_RAW_PARAMS( 7093788, 456, 0, 320, 312, 34, 34 + 228 + 30 )
	MCFG_PALETTE_INIT_OVERRIDE(a7800_state, a7800p )

	/* sound hardware */
	MCFG_DEVICE_REMOVE("pokey")
	MCFG_POKEY_ADD("pokey", CLK_PAL)
	MCFG_SOUND_ROUTE(ALL_OUTPUTS, "mono", 1.00)

	/* devices */
	MCFG_DEVICE_REMOVE("riot")
	MCFG_RIOT6532_ADD("riot", 3546894/3, a7800_r6532_interface)

	/* software lists */
	MCFG_DEVICE_REMOVE("cart_list")
	MCFG_SOFTWARE_LIST_ADD("cart_list","a7800")
	MCFG_SOFTWARE_LIST_FILTER("cart_list","PAL")
MACHINE_CONFIG_END


/***************************************************************************
    ROM DEFINITIONS
***************************************************************************/

ROM_START( a7800 )
	ROM_REGION(0x40000, "maincpu", 0)
	ROM_FILL(0x0000, 0x40000, 0xff)
	ROM_SYSTEM_BIOS( 0, "a7800", "Atari 7800" )
	ROMX_LOAD("7800.u7", 0xf000, 0x1000, CRC(5d13730c) SHA1(d9d134bb6b36907c615a594cc7688f7bfcef5b43), ROM_BIOS(1))
	ROM_SYSTEM_BIOS( 1, "a7800pr", "Atari 7800 (prototype with Asteroids)" )
	ROMX_LOAD("c300558-001a.u7", 0xc000, 0x4000, CRC(a0e10edf) SHA1(14584b1eafe9721804782d4b1ac3a4a7313e455f), ROM_BIOS(2))
ROM_END

ROM_START( a7800p )
	ROM_REGION(0x40000, "maincpu", 0)
	ROM_FILL(0x0000, 0x40000, 0xff)
	ROM_LOAD("7800pal.rom", 0xc000, 0x4000, CRC(d5b61170) SHA1(5a140136a16d1d83e4ff32a19409ca376a8df874))
ROM_END


/***************************************************************************
    GAME DRIVERS
***************************************************************************/

/*    YEAR  NAME      PARENT    COMPAT  MACHINE     INPUT     INIT          COMPANY   FULLNAME */
CONS( 1986, a7800,    0,        0,      a7800_ntsc, a7800, a7800_state,    a7800_ntsc,  "Atari",  "Atari 7800 (NTSC)" , 0)
CONS( 1986, a7800p,   a7800,    0,      a7800_pal,  a7800, a7800_state,    a7800_pal,   "Atari",  "Atari 7800 (PAL)" , 0)
