/*
 * Argus Client Software.  Tools to read, analyze and manage Argus data.
 * Copyright (c) 2000-2003 QoSient, LLC
 * All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
 *
 * ra  - Read Argus
 *       This program read argus output streams, either through a socket,
 *       a piped stream, or in a file, filters and optionally writes the
 *       output to a file, its stdout or prints the binary records to
 *       stdout in ASCII.
 */


#include <argus_client.h>
#include <signal.h>
#include <ctype.h>

extern int ArgusSrcUserDataLen;
extern int ArgusDstUserDataLen;

int RaInitialized = 0;


void
ArgusClientInit ()
{
   if (!(RaInitialized)) {
      (void) signal (SIGHUP,  (void (*)(int)) RaParseComplete);
      (void) signal (SIGTERM, (void (*)(int)) RaParseComplete);
      (void) signal (SIGQUIT, (void (*)(int)) RaParseComplete);
      (void) signal (SIGINT,  (void (*)(int)) RaParseComplete);

      RaInitialized++;
   }
}


void
RaParseComplete (int sig)
{
   long long total_count, total_dst_count, total_src_count;
   long long total_bytes, total_dst_bytes, total_src_bytes;

   if ((sig >= 0) && aflag) {
      putchar ('\n');

      total_src_count = tcp_src_count + udp_src_count + icmp_src_count +
         ip_src_count + arp_src_count + nonip_src_count;
      total_dst_count = tcp_dst_count + udp_dst_count + icmp_dst_count +
         ip_dst_count + arp_dst_count + nonip_dst_count;
      total_count = total_dst_count + total_src_count;

      total_src_bytes = tcp_src_bytes + udp_src_bytes + icmp_src_bytes +
         ip_src_bytes + arp_src_bytes + nonip_src_bytes;
      total_dst_bytes = tcp_dst_bytes + udp_dst_bytes + icmp_dst_bytes +
         ip_dst_bytes + arp_dst_bytes + nonip_dst_bytes;
      total_bytes = total_dst_bytes + total_src_bytes;

      if (total_count) {
         printf ("             total_pkts         src_pkts         dst_pkts      total_bytes        src_bytes        dst_bytes\n");
         printf ("tcp    %16lld %16lld %16lld %16lld %16lld %16lld\n",
            tcp_dst_count + tcp_src_count, tcp_src_count, tcp_dst_count,
            tcp_dst_bytes + tcp_src_bytes, tcp_src_bytes, tcp_dst_bytes);

         printf ("udp    %16lld %16lld %16lld %16lld %16lld %16lld\n",
            udp_dst_count + udp_src_count, udp_src_count, udp_dst_count,
            udp_dst_bytes + udp_src_bytes, udp_src_bytes, udp_dst_bytes);

         printf ("icmp   %16lld %16lld %16lld %16lld %16lld %16lld\n",
            icmp_dst_count + icmp_src_count, icmp_src_count, icmp_dst_count,
            icmp_dst_bytes + icmp_src_bytes, icmp_src_bytes, icmp_dst_bytes);

         printf ("ip     %16lld %16lld %16lld %16lld %16lld %16lld\n",
            ip_dst_count + ip_src_count, ip_src_count, ip_dst_count,
            ip_dst_bytes + ip_src_bytes, ip_src_bytes, ip_dst_bytes);

         printf ("arp    %16lld %16lld %16lld %16lld %16lld %16lld\n",
            arp_dst_count + arp_src_count, arp_src_count, arp_dst_count,
            arp_dst_bytes + arp_src_bytes, arp_src_bytes, arp_dst_bytes);

         printf ("non-ip %16lld %16lld %16lld %16lld %16lld %16lld\n",
            nonip_dst_count + nonip_src_count, nonip_src_count, nonip_dst_count,
            nonip_dst_bytes + nonip_src_bytes, nonip_src_bytes, nonip_dst_bytes);

         printf ("sum    %16lld %16lld %16lld %16lld %16lld %16lld",
            total_count, total_src_count, total_dst_count,
            total_bytes, total_src_bytes, total_dst_bytes);

         printf ("\n\n");

      } else
         printf ("No data seen.\n");

      fflush (stdout);
   }

   if (sig == SIGINT)
      _exit(0);
}


void
ArgusClientTimeout ()
{
#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusClientTimeout()\n");
#endif
}

void
parse_arg (int argc, char**argv)
{}

void
usage ()
{
   extern char version[];

   fprintf (stderr, "Ra Version %s\n", version);
   fprintf (stderr, "usage: %s \n", ArgusProgramName);
   fprintf (stderr, "usage: %s [options] -S remoteServer  [- filter-expression]\n", ArgusProgramName);
   fprintf (stderr, "usage: %s [options] -r argusDataFile [- filter-expression]\n\n", ArgusProgramName);

   fprintf (stderr, "options: -a                 print record summaries on termination.\n");
   fprintf (stderr, "         -A                 print application bytes.\n");
   fprintf (stderr, "         -b                 dump packet-matching code.\n");
   fprintf (stderr, "         -C                 treat the remote source as a Cisco Netflow source.\n");
   fprintf (stderr, "         -d <bytes>         print number of <bytes> from user data capture buffer.\n");
   fprintf (stderr, "                   format:  num | s<num> | d<num> | s<num>:d<num>\n");
#if defined (ARGUSDEBUG)
   fprintf (stderr, "         -D <level>         specify debug level\n");
#endif
   fprintf (stderr, "         -e <encode>        convert user data using <encode> method.\n");
   fprintf (stderr, "                            Supported types are <Ascii> and <Encode64>.\n");
   fprintf (stderr, "         -E <file>          write records that are rejected by the filter into <file>\n");
   fprintf (stderr, "         -F <conffile>      read configuration from <conffile>.\n");
   fprintf (stderr, "         -h                 print help.\n");
   fprintf (stderr, "         -n                 don't convert numbers to names.\n");
   fprintf (stderr, "         -p <digits>        print fractional time with <digits> precision.\n");
   fprintf (stderr, "         -q                 quiet mode. don't print record outputs.\n");
   fprintf (stderr, "         -r <file>          read argus data <file>. '-' denotes stdin.\n");
   fprintf (stderr, "         -R                 print out response data when availabile.\n");
   fprintf (stderr, "         -s [-][+[#]]field  specify fields to print.\n");
   fprintf (stderr, "                   fields:  startime, lasttime, count, dur, avgdur,\n");
   fprintf (stderr, "                            saddr, daddr, proto, sport, dport, ipid,\n");
   fprintf (stderr, "                            stos, dtos, sttl, dttl, bytes, sbytes, dbytes,\n");
   fprintf (stderr, "                            pkts, spkts, dpkts, load, loss, rate,\n");
   fprintf (stderr, "                            srcid, ind, mac, dir, jitter, status, user,\n");
   fprintf (stderr, "                            win, trans, seq, vlan, mpls.\n");
   fprintf (stderr, "         -S <host[:port]>   specify remote argus <host> and optional port number.\n");
   fprintf (stderr, "         -t <timerange>     specify <timerange> for reading records.\n");
   fprintf (stderr, "                   format:  timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                            timeSpecification: [[[yyyy/]mm/]dd.]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                                 [yyyy/]mm/dd\n");
   fprintf (stderr, "                                                 -%%d{yMdhms}\n");
   fprintf (stderr, "         -T <secs>          attach to remote server for T seconds.\n");
   fprintf (stderr, "         -u                 print time in Unix time format.\n");
#ifdef ARGUS_SASL
   fprintf (stderr, "         -U <user/auth>     specify <user/auth> authentication information.\n");
#endif
   fprintf (stderr, "         -w <file>          write output to <file>. '-' denotes stdout.\n");
   fprintf (stderr, "         -z                 print Argus TCP state changes.\n");
   fprintf (stderr, "         -Z <s|d|b>         print actual TCP flag values.<'s'rc | 'd'st | 'b'oth>\n");
   exit(1);
}

int RaLabelCounter = 0;

void
RaProcessRecord (struct ArgusRecord *argus)
{
   if (Lflag && !(qflag)) {
      if (RaLabel == NULL)
         RaLabel = RaGenerateLabel(argus);

      if (!(RaLabelCounter++ % Lflag)) {
         printf ("%s\n", RaLabel);
         fflush (stdout);
      }

      if (Lflag < 0)
         Lflag = 0;
   }

   if (argus->ahdr.type & ARGUS_MAR)
      RaProcessManRecord (argus);

   else {
      switch (argus->ahdr.status & 0xFFFF) {
         case ETHERTYPE_IP:
            switch (argus->argus_far.flow.ip_flow.ip_p) {
               case IPPROTO_TCP:
                  RaProcessTCPRecord (argus);
                  break;

               case IPPROTO_UDP:
                  RaProcessUDPRecord (argus);
                  break;

               case IPPROTO_ICMP:
                  RaProcessICMPRecord (argus);
                  break;

               default:
                  RaProcessIPRecord (argus);
                  break;
            }
            break;

         case ETHERTYPE_ARP:
         case ETHERTYPE_REVARP:
            RaProcessARPRecord (argus);
            break;

         default:
            RaProcessNonIPRecord (argus);
            break;
      }
   }

   if (Nflag > 0) {
      if (--Nflag == 0) {
         RaParseComplete(SIGINT);
      }
   }
}


void
RaProcessManRecord (struct ArgusRecord *argus)
{
   char *str = NULL;

   if (!(qflag || wfile)) {
      str = get_man_string (argus);
      (void) printf ("%s", str);
      printf ("\n");
      fflush (stdout);
   }
}


void
RaProcessFragRecord (struct ArgusRecord *argus)
{
}


void
RaProcessTCPRecord (struct ArgusRecord *argus)
{
   if (aflag) {
      tcp_src_count += argus->argus_far.src.count;
      tcp_dst_count += argus->argus_far.dst.count;

      if (Aflag) {
         tcp_src_bytes += argus->argus_far.src.appbytes;
         tcp_dst_bytes += argus->argus_far.dst.appbytes;
      } else {
         tcp_src_bytes += argus->argus_far.src.bytes;
         tcp_dst_bytes += argus->argus_far.dst.bytes;
      }
   }

   if (!qflag) {
      printf ("%s", get_tcp_string (argus));
      printf ("\n");
   }

   fflush(stdout);
}


void
RaProcessICMPRecord (struct ArgusRecord *argus)
{
   if (aflag) {
      icmp_src_count += argus->argus_far.src.count;
      icmp_dst_count += argus->argus_far.dst.count;

      if (Aflag) {
         icmp_src_bytes += argus->argus_far.src.appbytes;
         icmp_dst_bytes += argus->argus_far.dst.appbytes;
      } else {
         icmp_src_bytes += argus->argus_far.src.bytes;
         icmp_dst_bytes += argus->argus_far.dst.bytes;
      }
   }

   if (!qflag) {
      printf ("%s", get_icmp_string (argus));
      printf ("\n");
   }

   fflush(stdout);
}


void
RaProcessUDPRecord (struct ArgusRecord *argus)
{
   if (aflag) {
      udp_src_count += argus->argus_far.src.count;
      udp_dst_count += argus->argus_far.dst.count;

      if (Aflag) {
         udp_src_bytes += argus->argus_far.src.appbytes;
         udp_dst_bytes += argus->argus_far.dst.appbytes;
      } else {
         udp_src_bytes += argus->argus_far.src.bytes;
         udp_dst_bytes += argus->argus_far.dst.bytes;
      }
   }

   if (!qflag) {
      printf ("%s", get_ip_string (argus));
      printf ("\n");
   }
   fflush(stdout);
}


void
RaProcessIPRecord (struct ArgusRecord *argus)
{
   if (aflag) {
      ip_src_count += argus->argus_far.src.count;
      ip_dst_count += argus->argus_far.dst.count;

      if (Aflag) {
         ip_src_bytes += argus->argus_far.src.appbytes;
         ip_dst_bytes += argus->argus_far.dst.appbytes;
      } else {
         ip_src_bytes += argus->argus_far.src.bytes;
         ip_dst_bytes += argus->argus_far.dst.bytes;
      }
   }

   if (!qflag) {
      printf ("%s", get_ip_string (argus));
      printf ("\n");
   }
   fflush(stdout);
}


void
RaProcessARPRecord (struct ArgusRecord *argus)
{
   if (aflag) {
      arp_src_count += argus->argus_far.src.count;
      arp_dst_count += argus->argus_far.dst.count;

      if (Aflag) {
         arp_src_bytes += argus->argus_far.src.appbytes;
         arp_dst_bytes += argus->argus_far.dst.appbytes;
      } else {
         arp_src_bytes += argus->argus_far.src.bytes;
         arp_dst_bytes += argus->argus_far.dst.bytes;
      }
   }

   if (!qflag) {
      printf ("%s", get_arp_string (argus));
      printf ("\n");
   }
   fflush(stdout);
}


void
RaProcessNonIPRecord (struct ArgusRecord *argus)
{
   if (aflag) {
      nonip_src_count += argus->argus_far.src.count;
      nonip_dst_count += argus->argus_far.dst.count;

      if (Aflag) {
         nonip_src_bytes += argus->argus_far.src.appbytes;
         nonip_dst_bytes += argus->argus_far.dst.appbytes;
      } else {
         nonip_src_bytes += argus->argus_far.src.bytes;
         nonip_dst_bytes += argus->argus_far.dst.bytes;
      }
   }

   if (!qflag) {
      printf ("%s", get_nonip_string (argus));
      printf ("\n");
   }
   fflush(stdout);
}

int RaSendArgusRecord(struct ArgusRecordStore *argus) {return 0;}

