/****************************************************************
**
** Attal : Lords of Doom
**
** displayLord.cpp
** show infos about Lord
**
** Version : $Id: displayLord.cpp,v 1.34 2008/05/24 16:04:35 lusum Exp $
**
** Author(s) : Pascal Audoux - Sardi Carlo
**
** Date : 24/08/2000
**
** Licence :
**	This program is free software; you can redistribute it and/or modify
**   	it under the terms of the GNU General Public License as published by
**     	the Free Software Foundation; either version 2, or (at your option)
**      any later version.
**
**	This program is distributed in the hope that it will be useful,
** 	but WITHOUT ANY WARRANTY; without even the implied warranty of
**	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**	GNU General Public License for more details.
**
****************************************************************/

#include "displayLord.h"

 
// generic include files

// include files for QT
#include <QHBoxLayout>
#include <QLabel>
#include <QLayout>
#include <QMessageBox>
#include <QPixmap>
#include <QPushButton>
#include <QSignalMapper>
#include <QStatusBar>
#include <QTabWidget>
#include <QVBoxLayout>

// application specific includes
#include "conf.h"

#include "libCommon/log.h"
#include "libCommon/dataTheme.h"
#include "libCommon/genericLord.h"
#include "libCommon/attalSocket.h"

#include "libClient/attalButton.h"
#include "libClient/displayTechnics.h"
#include "libClient/graphicalArtefact.h"
#include "libClient/gui.h"
#include "libClient/imageTheme.h"
#include "libClient/lord.h"
#include "libClient/player.h"
#include "libClient/unitExchange.h"
#include "libClient/widget.h"

//
// ----- DisplayLord -----
//

DisplayLord::DisplayLord( Player * player, QWidget * parent, char * /*name*/ )
//	:QDialog( parent, name, true, WStyle_Customize | WStyle_NoBorder ),
	:QDialog( parent, Qt::Dialog ),
	 _player( player )
{
	_socket = 0;
	QVBoxLayout * layout1 = new QVBoxLayout( this );
	layout1->addSpacing( 5 );

	QHBoxLayout * layout2 = new QHBoxLayout();
	layout2->addSpacing( 5 );

	QVBoxLayout * layout3 = new QVBoxLayout();

	QHBoxLayout * layout4 = new QHBoxLayout();
	_dispGeneral = new DisplayGeneral( player, this );
	layout4->addWidget( _dispGeneral, 1 );
	layout4->addSpacing( 5 );
	QVBoxLayout * layout5 = new QVBoxLayout();
	_dispArtefact = new DisplayArtefacts( player, this );
	layout5->addWidget( _dispArtefact, 1 );
	layout5->addSpacing( 50 + 5 );

	layout4->addLayout( layout5, 1 );

	layout3->addLayout( layout4, 1 );
	_dispUnit = new DisplayUnit( player, this );
	layout3->addWidget( _dispUnit );

	layout2->addLayout( layout3, 1 );

	layout2->addSpacing( 5 );

	_dispList = new DisplayListPanel( player, this );


	layout2->addWidget( _dispList );

	layout1->addLayout( layout2, 1 );
	layout1->addSpacing( 5 );
	QStatusBar * bar = new QStatusBar( this );
	bar->showMessage( tr("Info Lord") );
	bar->setFixedHeight( bar->sizeHint().height() );
	bar->setSizeGripEnabled( false );
	layout1->addWidget( bar );
	
	layout1->activate();
	
	setFixedSize( 800, 600 );
	connect( _dispList, SIGNAL( sig_quit() ), SLOT( accept() ) );
	connect( _dispList, SIGNAL( sig_lord( int ) ), SLOT( slot_lord( int ) ) );
	connect( parent, SIGNAL( sig_updateWidget() ), SLOT( reupdate() ) );
}

DisplayLord::~DisplayLord()
{

}

void DisplayLord::initSocket( AttalSocket * socket )
{
	_socket = socket;
	_dispUnit->initSocket( socket );
}

void DisplayLord::slot_lord( int num )
{
	_player->setSelectedLord( _player->getLord( num ) );
	reupdate();
}

void DisplayLord::show()
{
	reinit();
	QDialog::show();
}

void DisplayLord::reinit()
{
	_dispList->reinit();
	_dispUnit->reinit();
	_dispArtefact->reinit();
	_dispGeneral->reinit();
}

void DisplayLord::reupdate()
{
	_dispList->reupdate();
	_dispUnit->reupdate();
	_dispArtefact->reupdate();
	_dispGeneral->reupdate();	
}

//
// ----- DisplayGeneral -----
//

DisplayGeneral::DisplayGeneral( Player * player, QWidget * parent, const char * /* name */ )
	: QFrame( parent )
{
	_player = player;
	
	_photo = new Icon( this );
	_photo->move( 10, 10 );
	
	_title = new QLabel( this );
	_title->setAlignment( Qt::AlignCenter );
	_title->move( 90, 20 );
	
	Icon * ic1 = new Icon( this );
	ic1->move( 10, 140 );
	ic1->setPixmap( QPixmap( *ImageTheme.getWidgetPixmap( ICO_ATTACK ) ) );
	Icon * ic2 = new Icon( this );
	ic2->move( 80, 140 );
	ic2->setPixmap( QPixmap( *ImageTheme.getWidgetPixmap( ICO_DEFENSE ) ) );
	Icon * ic3 = new Icon( this );
	ic3->move( 150, 140 );
	ic3->setPixmap( QPixmap( *ImageTheme.getWidgetPixmap( ICO_POWER ) ) );
	Icon * ic4 = new Icon( this );
	ic4->move( 220, 140 );
	ic4->setPixmap( QPixmap( *ImageTheme.getWidgetPixmap( ICO_KNOWLEDGE ) ) );
	
	QLabel * titre[4];
	for( int i = 0; i < 4; i++ ) {
		titre[i] = new QLabel( this );
		titre[i]->setAlignment( Qt::AlignCenter );
		titre[i]->setFixedSize( 60, 20 );
		titre[i]->move( 10 + (i*70), 120 );
	}
	titre[0]->setText( tr("Attack") );
	titre[1]->setText( tr("Defense") );
	titre[2]->setText( tr("Power") );
	titre[3]->setText( tr("Knowledge") );
	
	_labAttack = new QLabel( this );
	_labAttack->setAlignment( Qt::AlignCenter );
	_labAttack->setFixedSize( 60, 20 );
	_labAttack->move( 10, 200 );
	
	_labDefense = new QLabel( this );
	_labDefense->setAlignment( Qt::AlignCenter );
	_labDefense->setFixedSize( 60, 20 );
	_labDefense->move( 80, 200 );
	
	_labPower = new QLabel( this );
	_labPower->setAlignment( Qt::AlignCenter );
	_labPower->setFixedSize( 60, 20 );
	_labPower->move( 150, 200 );
	
	_labKnowledge = new QLabel( this );
	_labKnowledge->setAlignment( Qt::AlignCenter );
	_labKnowledge->setFixedSize( 60, 20 );
	_labKnowledge->move( 220, 200 );		
	
	_specialty = new InfoLabel( this );
	_specialty->move( 10, 240 );
	_specialty->setText( tr("Specialty\nNone") );
	
	_experience = new InfoLabel( this );
	_experience->move( 10, 300 );
	
	_spellPoints = new InfoLabel( this );
	_spellPoints->move( 160, 300 );

	reinit();
}

void DisplayGeneral::reinit()
{
	Lord * aLord = (Lord*)_player->getSelectedLord();
	if( _player->getSelectedLordPixmap() ) {
		_photo->setPixmap( *_player->getSelectedLordPixmap() );
	}

	if( aLord ) {
		QString title;
		title.sprintf( "Lord %s\nLevel %d of %s", qPrintable( aLord->getName() ),
				aLord->getCharac( LEVEL ), qPrintable( aLord->getCategoryName() ) );
		_title->setText( title );	
		FIXEDSIZE( _title );

		_labAttack->setText( QString::number( aLord->getCharac( ATTACK ) ) );
		_labDefense->setText( QString::number( aLord->getCharac( DEFENSE ) ) );
		_labPower->setText( QString::number( aLord->getCharac( POWER ) ) );
		_labKnowledge->setText( QString::number( aLord->getCharac( KNOWLEDGE ) ) );

		_specialty->setText( tr("Specialty\nNone") );
		QString temp;
		temp.sprintf( "Experience\n%d", aLord->getCharac( EXPERIENCE ) );
		_experience->setText( temp );
		temp.sprintf( "Spell Points\n%d/%d", aLord->getCharac( TECHNICPOINT ),
				aLord->getCharac( MAXTECHNICPOINT ) );
		_spellPoints->setText( temp );
	}
}

//
// ----- DisplayUnit -----
//

DisplayUnit::DisplayUnit( Player * player, QWidget * parent, const char * /* name */ )
	: QFrame( parent )
{
	_unit = -1;
	_exch = false;
	_player = player;
	_unitExchange = 0;
	_socket = 0;
	int i;
	
	QSignalMapper * sigmap = new QSignalMapper( this );
	
	for( i = 0; i < MAX_UNIT; i++ ) {
		_buttonUnit[i] = new Icon( this );
		_buttonUnit[i]->move( 30 + (i*60), 10 );
		_numUnit[i] = new QLabel( this );
		_numUnit[i]->setFixedSize( 60, 20 );
		_numUnit[i]->setAlignment( Qt::AlignCenter );
		_numUnit[i]->move( 30 + (i*60), 70 );
		sigmap->setMapping( _buttonUnit[i], i );
		connect( _buttonUnit[i], SIGNAL( sig_clicked() ), sigmap, SLOT( map() ) );
	}	
			
	for( i = 0; i < 4; i++ ) {
		_dispo[i] = new AttalButton( this );
		_dispo[i]->setFixedSize( 50, 40 );
	}
	_dispo[0]->move( 480, 5 );
	_dispo[1]->move( 480, 55 );
	_dispo[2]->move( 560, 5 );
	_dispo[3]->move( 560, 55 );
			
	_butExchange = _dispo[2];
	_butExchange->setEnabled( false );
	_butExchange->setPixmap( QPixmap( *ImageTheme.getWidgetPixmap( ICO_EXCHANGE ) ) );
	
	setFixedSize( 640, 100 );
	reinit();
	
	connect( sigmap, SIGNAL( mapped( int ) ), SLOT( slot_unitClicked( int ) ) );
	connect( _butExchange, SIGNAL( clicked() ), SLOT( slot_exchange() ) );
	connect( _dispo[3], SIGNAL( clicked() ), SLOT( slot_technic() ) );
}

void DisplayUnit::reinit()
{
	GenericLord * lord = _player->getSelectedLord();
	QString text;

	if(lord){
		for( int i = 0; i < MAX_UNIT; i++ ) {
			text = "";
			if( lord->getUnit( i ) ) {
				_buttonUnit[i]->setPixmap( ImageTheme.getPhotoCreature( lord->getUnit( i ) ) );
				text = lord->getUnit(i)->getUnitDescription();
				_buttonUnit[i]->setPopupText( text  );
				_numUnit[i]->setText( QString::number( lord->getUnit( i )->getNumber() ) );
			} else {
				_buttonUnit[i]->clear();
				_numUnit[i]->clear();
			}
		}
	}
}

void DisplayUnit::slot_exchange()
{
	_exch = true;
}

void DisplayUnit::slot_technic()
{
	DisplayTechnics dialog;
	dialog.exec();
}

void DisplayUnit::quitExchange()
{
	deselectUnit();
	_exch = false;
}

void DisplayUnit::selectUnit( int num )
{
	_unit = num;
	_butExchange->setEnabled( true );
}

void DisplayUnit::deselectUnit()
{
	_unit = -1;
	_butExchange->setEnabled( false );
}

void DisplayUnit::exchangeUnit( int num )
{	
	GenericLord * lord = _player->getSelectedLord();

	if( _unit == num ) {
		/// XXX: something to do ?
	} else if (lord) {
		if( lord->getUnit( num ) ) {
			if( lord->getUnit( _unit )->getCreature() == lord->getUnit( num )->getCreature() ) {
				if( ! _unitExchange ) {
					_unitExchange = new UnitExchange( this );
				}
				_unitExchange->setMin( 0 );
				_unitExchange->setMax( lord->getUnit( _unit )->getNumber() + lord->getUnit( num )->getNumber() );
				_unitExchange->setUnits( lord->getUnit( _unit ), lord->getUnit( num ) );
				if( _unitExchange->exec() ) {
				}
			}
		} else {
			if( ! _unitExchange ) {
				_unitExchange = new UnitExchange( this );
			}
			_unitExchange->setMin( 0 );
			_unitExchange->setMax( lord->getUnit( _unit )->getNumber() );
			_unitExchange->setUnits( lord->getUnit( _unit ), 0 );
			if( _unitExchange->exec() ) {
			}
		}	
	}
	quitExchange();	
}

void DisplayUnit::slot_unitClicked( int num )
{
	GenericLord * lord = _player->getSelectedLord();

	if(lord){
		if( _exch ) {
			exchangeUnit( num );
		} else {
			if( _unit == num ) {
				QMessageBox msb( tr("Unit"), tr("Do you want destroy this unit ?"), QMessageBox::Warning, QMessageBox::Yes | QMessageBox::Default, QMessageBox::No | QMessageBox::Escape, 0, this );
				if ( msb.exec() == QMessageBox::Yes ) {
					_socket->sendLordUnit( lord, num , 0);
				}
				reinit();
				/// XXX: show unit charact
				deselectUnit();
				return;
			}

			if( _unit == -1 ) {
				if( lord->getUnit( num ) ) {
					selectUnit( num );
				}
			} else {
				if(_socket){
					_socket->sendExchangeUnit( lord, _unit, lord, num );
				}
				deselectUnit();
			}
		}
		reinit();
	}
}

//
// ----- DisplayListPanel -----
//

DisplayListPanel::DisplayListPanel( Player * player, QWidget * parent, const char * /* name */ )
	: QFrame( parent )
{
	_player = player;
	
  	setFrameStyle( QFrame::Panel | QFrame::Sunken );
	setLineWidth( 1 );
	setFixedWidth( 60 );

	QLabel * emblem = new QLabel( this );
	emblem->setFixedSize( 58, 58 );
	emblem->setPixmap( * ImageTheme.getFlag( 0 ) ); /// XXX: TODO change emblem according to player team
	
	_listLord = new DisplayListLord( _player, this );

	AttalButton * butOk = new AttalButton( this, AttalButton::BT_OK );

	QVBoxLayout * layout = new QVBoxLayout( this );
	layout->setAlignment( Qt::AlignHCenter );
	layout->addWidget( emblem );
	layout->addSpacing( 5 );
	layout->addWidget( _listLord, 1 );
	//layout->addStretch( 1 );
	layout->addSpacing( 5 );
	layout->addWidget( butOk );
	layout->addSpacing( 5 );
	layout->setAlignment( butOk, Qt::AlignHCenter );
	layout->activate();

	connect( butOk, SIGNAL( clicked() ), SIGNAL( sig_quit() ) );
	connect( _listLord, SIGNAL( sig_lord( int ) ), SIGNAL( sig_lord( int ) ) );
}

void DisplayListPanel::reinit()
{
	_listLord->reinit();
	update();
}

void DisplayListPanel::reupdate()
{
	_listLord->reupdate();
	update();
}

//
// ----- DisplayListLord -----
//

DisplayListLord::DisplayListLord( Player * player, QWidget * parent, const char * /* name */ )
	: QWidget( parent )
{
	_player = player;
	_sigmap = new QSignalMapper( this );
	setFixedWidth( 55 );
	connect( _sigmap, SIGNAL( mapped( int ) ), SIGNAL( sig_lord( int ) ) );
}

DisplayListLord::~DisplayListLord()
{
	while( ! _listBut.isEmpty() ) {
		delete _listBut.takeFirst();
	}
}

void DisplayListLord::reupdate()
{
	/// XXX: add border to selected unit
}

void DisplayListLord::reinit()
{
	AttalButton * button;

	while( ! _listBut.isEmpty() ) {
		button = _listBut.takeFirst();
		_sigmap->removeMappings( button );
		delete button;
	}
	
	for( uint i = 0; i < _player->numLord(); i++ ) {
		button = new AttalButton( this, AttalButton::BT_LORD );
		button->move( 3, 5 + (50 * i) );
		button->setLord( _player->getLord( i )->getId() );
		button->show();
		_sigmap->setMapping( button, i );
		connect( button, SIGNAL( clicked() ), _sigmap, SLOT( map() ) );
		_listBut.append( button );
	}
	
	setMinimumHeight( _player->numLord() * 50 );
}

//
// ----- DisplayLordTab -----
//

DisplayLordTab::DisplayLordTab( Player * player, AttalSocket * socket, QWidget * parent, char * /* name */)
: 	QDialog( parent ),
	_player( player )
{
	_socket = socket;
	
	QWidget * panel = new QWidget( this );
	panel->setFixedHeight( 80 );

	_photo = new Icon( panel );
	_photo->move( 10, 10 );
	
	_title = new QLabel( panel );
	_title->setAlignment( Qt::AlignCenter );
	_title->move( 90, 20 );
		
	_tabGeneral = new DisplayLordTabGeneral( player );
	_tabUnits = new DisplayLordTabUnits( player, socket );
	_tabArtefacts = new DisplayLordTabArtefacts( player );
	_tabTechnics = new DisplayLordTabTechnics( player );

	_tabs = new QTabWidget( this );
	_tabs->addTab( _tabGeneral, tr( "General" ) );
	_tabs->addTab( _tabUnits, tr( "Units" ) );
	_tabs->addTab( _tabArtefacts, tr( "Artefacts" ) );
	_tabs->addTab( _tabTechnics, tr( "Technics" ) );

	_listPanel = new DisplayListPanel( player, this );

	QVBoxLayout * layV1 = new QVBoxLayout();
	layV1->addWidget( panel );
	layV1->addWidget( _tabs, 1 );

	QHBoxLayout * layout = new QHBoxLayout( this );
	layout->setMargin( 5 );
	layout->setSpacing( 5 );
	layout->addLayout( layV1, 1 );
	layout->addWidget( _listPanel );
	layout->activate();

	connect( _listPanel, SIGNAL( sig_lord( int ) ), SLOT( slot_lord( int ) ) );
	connect( _listPanel, SIGNAL( sig_quit() ), SLOT( accept() ) );
}


void DisplayLordTab::exec()
{
	reinit();
	QDialog::exec();
}

void DisplayLordTab::reinit()
{
	Lord * aLord = ( Lord * )_player->getSelectedLord();
	if( _player->getSelectedLordPixmap() ) {
		_photo->setPixmap( *_player->getSelectedLordPixmap() );
	}

	if( aLord ) {
		QString title;
		title.sprintf( "Lord %s\nLevel %d of %s", qPrintable( aLord->getName() ),
				aLord->getCharac( LEVEL ), qPrintable( aLord->getCategoryName() ) );
		_title->setText( title );	
		FIXEDSIZE( _title );
	}

	_tabGeneral->reinit();
	_tabUnits->reinit();
	_tabArtefacts->reinit();
	_tabTechnics->reinit();
	_listPanel->reinit();
}

void DisplayLordTab::reupdate()
{
	_tabGeneral->reinit();
	_tabUnits->reinit();
	_tabArtefacts->reinit();
	_tabTechnics->reinit();

	_listPanel->reupdate();
}


void DisplayLordTab::slot_lord( int num )
{
	_player->setSelectedLord( _player->getLord( num ) );
	reinit();
}

//
// ----- DisplayLordTabGeneral -----
//

DisplayLordTabGeneral::DisplayLordTabGeneral( Player * player, QWidget * parent, const char * /* name*/ )
: QWidget( parent )
{
	_player = player;
	
	_specialty = new InfoWidget( this );
	_specialty->setIcon( QPixmap(  *ImageTheme.getWidgetPixmap( ICO_SPECIALITY ) ));
	_experience = new InfoWidget( this );
	_experience->setIcon( QPixmap(  *ImageTheme.getWidgetPixmap( ICO_EXPERIENCE ) ));
	_luck = new InfoWidget( this );
	_luck->setIcon( QPixmap(  *ImageTheme.getWidgetPixmap( ICO_LUCK ) ));
	_morale = new InfoWidget( this );
	_morale->setIcon( QPixmap(  *ImageTheme.getWidgetPixmap( ICO_MORALE ) ));
	_attack = new InfoWidget( this );
	_attack->setIcon( QPixmap( *ImageTheme.getWidgetPixmap( ICO_ATTACK ) ) );
	_defense = new InfoWidget( this );
	_defense->setIcon( QPixmap(  *ImageTheme.getWidgetPixmap( ICO_DEFENSE ) ) );
	_charisma = new InfoWidget( this );
	_charisma->setIcon( QPixmap(  *ImageTheme.getWidgetPixmap( ICO_CHARISM ) ));
	_knowledge = new InfoWidget( this );
	_knowledge->setIcon( QPixmap( *ImageTheme.getWidgetPixmap( ICO_KNOWLEDGE ) ) );
	_power = new InfoWidget( this );
	_power->setIcon( QPixmap( *ImageTheme.getWidgetPixmap( ICO_POWER ) ) );
	_mana = new InfoWidget( this );
	_mana->setIcon( QPixmap(  *ImageTheme.getWidgetPixmap( ICO_MANA ) ));
	_move = new InfoWidget( this );
	_move->setIcon( QPixmap(  *ImageTheme.getWidgetPixmap( ICO_MOVE ) ));
	_vision = new InfoWidget( this );
	_vision->setIcon( QPixmap(  *ImageTheme.getWidgetPixmap( ICO_DEFENSE ) ) );

	QGridLayout * layout = new QGridLayout( this );
	layout->setColumnStretch( 0, 1 );
	layout->setColumnStretch( 1, 1 );
	layout->addWidget( _specialty, 0, 0 );
	layout->addWidget( _experience, 0, 1 );
	layout->addWidget( _luck, 1, 0 );
	layout->addWidget( _morale, 1, 1 );
	layout->addWidget( _attack, 2, 0 );
	layout->addWidget( _defense, 2, 1 );
	layout->addWidget( _charisma, 3, 0 );
	layout->addWidget( _knowledge, 3, 1 );
	layout->addWidget( _power, 4, 0 );
	layout->addWidget( _mana, 4, 1 );
	layout->addWidget( _move, 5, 0 );
	layout->addWidget( _vision, 5, 1 );
	layout->setRowStretch( 6, 1 );
	layout->activate();	
}

QString DisplayLordTabGeneral::textCharac( GenericLord * lord, LordCharac charac )
{
	return getCharacName( charac ) + ": " + QString::number( lord->getCharac( charac ) );
}

void DisplayLordTabGeneral::reinit()
{
	GenericLord * lord = _player->getSelectedLord();

	if( lord ) {
		//GenericLordModel * model = DataTheme.lords.at( lord->getId() );
	
		_specialty->setText( tr("No specialty") );
		int nextLevel = DataTheme.lordExperience.getLevel( lord->getCharac( LEVEL )  + 1 );
		_experience->setText( textCharac( lord, EXPERIENCE ) + "\n( next " + QString::number(nextLevel) + " )" );
		_luck->setText( textCharac( lord, LUCK ) );
		_morale->setText( textCharac( lord, MORALE ) );
		_attack->setText( textCharac( lord, ATTACK ) );
		_defense->setText( textCharac( lord, DEFENSE ) );
		_charisma->setText( textCharac( lord, CHARISMA ) );
		_power->setText( textCharac( lord, POWER ) );
		_knowledge->setText( textCharac( lord, KNOWLEDGE ) );
		_mana->setText( textCharac( lord, TECHNICPOINT ) + " / " + QString::number( lord->getCharac( MAXTECHNICPOINT ) ) );
		_move->setText( textCharac( lord, MOVE ) + " / " + QString::number( lord->getCharac( MAXMOVE ) ) );
		_vision->setText( textCharac( lord, VISION ) );
	}
}

//
// -----  DisplayLordTabUnits -----
//

DisplayLordTabUnits::DisplayLordTabUnits( Player * player, AttalSocket * socket, QWidget * parent, const char * /* name */ )
: QWidget( parent )
{
	_player = player;
	_socket = socket;
	_select = -1;
	_exchange = false;
	_unitExchange = 0;
	InfoWidget * unit;

	_butExchange = new AttalButton( this );
	_butExchange->setEnabled( false );
	_butExchange->setText( tr( "Split unit" ) );

	QSignalMapper * sigmap = new QSignalMapper( this );
	
	QVBoxLayout * layout = new QVBoxLayout( this );
	layout->setSpacing( 5 );
	layout->setMargin( 5 );
	layout->addWidget( _butExchange );

	for( uint i = 0; i < MAX_UNIT; i++ ) {
		unit = new InfoWidget( this );
		_units.append( unit );
		layout->addWidget( unit );
		sigmap->setMapping( unit, i );
		connect( unit, SIGNAL( sig_clicked() ), sigmap, SLOT( map() ) );
	}

	layout->addStretch();
	layout->activate();

	connect( sigmap, SIGNAL( mapped( int ) ), SLOT( slot_unitClicked( int ) ) );
	connect( _butExchange, SIGNAL( clicked() ), SLOT( slot_exchange() ) );
}

void DisplayLordTabUnits::reinit()
{
	GenericLord * lord = _player->getSelectedLord();
	GenericFightUnit * unit;
	
	if( lord ) {
		for( uint i = 0; i < MAX_UNIT; i++ ) {
			unit = lord->getUnit( i );
			if( unit ) {
				_units.at( i )->setUnit( unit );
			} else {
				_units.at( i )->clear();
			}
		}
	}
}

void DisplayLordTabUnits::slot_exchange()
{
	_exchange = true;
	_butExchange->setText( tr( "Now click on a free cell or a unit of same type" ) );
}

void DisplayLordTabUnits::quitExchange()
{
	deselectUnit();
	_exchange = false;
	_butExchange->setText( tr( "Split unit" ) );
}

void DisplayLordTabUnits::selectUnit( int num )
{
	_select = num;
	_butExchange->setEnabled( true );
}

void DisplayLordTabUnits::deselectUnit()
{
	_select = -1;
	_butExchange->setEnabled( false );
}

void DisplayLordTabUnits::exchangeUnit( int num )
{	
	GenericLord * lord = _player->getSelectedLord();

	if( _select == num ) {
		/// nothing to do
	} else if (lord) {
		if( lord->getUnit( num ) ) {
			if( lord->getUnit( _select )->getCreature() == lord->getUnit( num )->getCreature() ) {
				if( ! _unitExchange ) {
					_unitExchange = new UnitExchange( this );
				}
				_unitExchange->setMin( 0 );
				_unitExchange->setMax( lord->getUnit( _select )->getNumber() + lord->getUnit( num )->getNumber() );
				_unitExchange->setUnits( lord->getUnit( _select ), lord->getUnit( num ) );
				if( _unitExchange->exec() ) {
					_socket->sendExchangeUnitSplit( lord, _select, _unitExchange->getLeftValue() , num, _unitExchange->getRightValue() );
				}
			}
		} else {
			if( ! _unitExchange ) {
				_unitExchange = new UnitExchange( this );
			}
			_unitExchange->setMin( 0 );
			_unitExchange->setMax( lord->getUnit( _select )->getNumber() );
			_unitExchange->setUnits( lord->getUnit( _select ), 0 );
			if( _unitExchange->exec() ) {
				_socket->sendExchangeUnitSplit( lord, _select, _unitExchange->getLeftValue() , num, _unitExchange->getRightValue() );
			}
		}	
	}
	quitExchange();	
}

void DisplayLordTabUnits::slot_unitClicked( int num )
{
	GenericLord * lord = _player->getSelectedLord();

	if( lord ) {
		if( _exchange ) {
			exchangeUnit( num );
		} else {
			if( _select == num ) {
				QMessageBox msb( tr("Unit"), tr("Do you want destroy this unit ?"), QMessageBox::Warning, QMessageBox::Yes | QMessageBox::Default, QMessageBox::No | QMessageBox::Escape, 0, this );
				if ( msb.exec() == QMessageBox::Yes ) {
					_socket->sendLordUnit( lord, num , 0 );
				}
				reinit();
				/// XXX: show unit charact
				deselectUnit();
				return;
			}

			if( _select == -1 ) {
				if( lord->getUnit( num ) ) {
					selectUnit( num );
				}
			} else {
				if( _socket ) {
					_socket->sendExchangeUnit( lord, _select, lord, num );
				}
				deselectUnit();
			}
		}
		reinit();
	}
}

//
// ----- DisplayLordTabTechnics -----
//

DisplayLordTabTechnics::DisplayLordTabTechnics( Player * player, QWidget * parent, const char * /* name */ )
: QWidget( parent )
{
	_player = player;

	QLabel * label = new QLabel( this );
	label->setText( tr( "Not yet implemented" ) );
	label->move( 5, 5 );
	FIXEDSIZE( label );
}

void DisplayLordTabTechnics::reinit()
{

}

//
// ----- InfoWidget -----
//

InfoWidget::InfoWidget( QWidget * parent, const char * /* name */ )
: QFrame( parent )
{
	setFrameStyle( QFrame::Panel | QFrame::Sunken );
	setLineWidth( 1 );
	setMidLineWidth( 1 );

	_photo = new Icon( this );
	_label = new Label( this );

	QHBoxLayout * layout = new QHBoxLayout( this );
	layout->setSpacing( 5 );
	layout->setMargin( 5 );
	layout->addWidget( _photo );
	layout->addWidget( _label, 1 );
	layout->activate();

	connect( _photo, SIGNAL( sig_clicked() ), SIGNAL( sig_clicked() ) );
}

void InfoWidget::setIcon( const QPixmap & pixmap )
{
	_photo->setPixmap( pixmap ); 
}

void InfoWidget::clear()
{
	_photo->clear();
	_label->clear(); 
}


void InfoWidget::setText( const QString & text )
{
	_label->setText( text );
}

void InfoWidget::setPopup( const QString & text )
{
	_label->setPopupText( text  );
}

void InfoWidget::setUnit( GenericFightUnit * unit )
{
	if( unit ) {
		_photo->setPixmap( ImageTheme.getPhotoCreature( unit )  );
		QString text = unit->getUnitDescription();
		_label->setPopupText( text  );
		text.sprintf( "%ld %s", unit->getNumber(), unit->getCreature()->getName().toLatin1().constData() );
		_label->setText( text );
	} else {
		_photo->clear();
		_label->clear();
	}
}







