import logging

from twisted.internet.defer import fail, succeed, inlineCallbacks
from txaws.ec2.model import IPPermission, SecurityGroup

from juju.errors import ProviderInteractionError
from juju.lib.testing import TestCase
from juju.machine import ProviderMachine
from juju.providers.ec2.securitygroup import (
    open_provider_port, close_provider_port, get_provider_opened_ports,
    destroy_environment_security_group)
from juju.providers.ec2.tests.common import EC2TestMixin


class EC2PortMgmtTest(EC2TestMixin, TestCase):

    @inlineCallbacks
    def test_open_provider_port(self):
        """Verify open port op will use the correct EC2 API."""
        log = self.capture_logging("juju.ec2", level=logging.DEBUG)
        machine = ProviderMachine("i-foobar", "x1.example.com")
        self.ec2.authorize_security_group(
            "juju-moon-machine-1", ip_protocol="tcp", from_port="80",
            to_port="80", cidr_ip="0.0.0.0/0")
        self.mocker.result(succeed(True))
        self.mocker.replay()

        provider = self.get_provider()
        yield open_provider_port(provider, machine, "machine-1", 80, "tcp")
        self.assertIn(
            "Opened 80/tcp on provider machine 'i-foobar'",
            log.getvalue())

    @inlineCallbacks
    def test_close_provider_port(self):
        """Verify close port op will use the correct EC2 API."""
        log = self.capture_logging("juju.ec2", level=logging.DEBUG)
        machine = ProviderMachine("i-foobar", "x1.example.com")
        self.ec2.revoke_security_group(
            "juju-moon-machine-1", ip_protocol="tcp", from_port="80",
            to_port="80", cidr_ip="0.0.0.0/0")
        self.mocker.result(succeed(True))
        self.mocker.replay()

        provider = self.get_provider()
        yield close_provider_port(provider, machine, "machine-1", 80, "tcp")
        self.assertIn(
            "Closed 80/tcp on provider machine 'i-foobar'",
            log.getvalue())

    @inlineCallbacks
    def test_get_provider_opened_ports(self):
        """Verify correct parse of IP perms from describe_security_group."""
        self.ec2.describe_security_groups("juju-moon-machine-1")
        self.mocker.result(succeed([
                    SecurityGroup(
                        "juju-%s-machine-1" % self.env_name,
                        "a security group name",
                        ips=[
                            IPPermission("udp", "53", "53", "0.0.0.0/0"),
                            IPPermission("tcp", "80", "80", "0.0.0.0/0"),
                            # The range 8080-8082 will be ignored
                            IPPermission("tcp", "8080", "8082", "0.0.0.0/0"),
                            # Ignore permissions that are not 0.0.0.0/0
                            IPPermission("tcp", "443", "443", "10.1.2.3")
                            ])]))
        self.mocker.replay()

        provider = self.get_provider()
        machine = ProviderMachine(
            "i-foobar", "x1.example.com")
        opened_ports = yield get_provider_opened_ports(
            provider, machine, "machine-1")
        self.assertEqual(opened_ports, set([(53, "udp"), (80, "tcp")]))

    @inlineCallbacks
    def test_open_provider_port_unknown_instance(self):
        """Verify open port op will use the correct EC2 API."""
        machine = ProviderMachine("i-foobar", "x1.example.com")
        self.ec2.authorize_security_group(
            "juju-moon-machine-1", ip_protocol="tcp", from_port="80",
            to_port="80", cidr_ip="0.0.0.0/0")
        self.mocker.result(fail(self.get_ec2_error("i-foobar")))
        self.mocker.replay()

        provider = self.get_provider()
        ex = yield self.assertFailure(
            open_provider_port(provider, machine, "machine-1", 80, "tcp"),
            ProviderInteractionError)
        self.assertEqual(
            str(ex),
            "Unexpected EC2Error opening 80/tcp on machine i-foobar: "
            "The instance ID 'i-foobar' does not exist")

    @inlineCallbacks
    def test_close_provider_port_unknown_instance(self):
        """Verify open port op will use the correct EC2 API."""
        machine = ProviderMachine("i-foobar", "x1.example.com")
        self.ec2.revoke_security_group(
            "juju-moon-machine-1", ip_protocol="tcp", from_port="80",
            to_port="80", cidr_ip="0.0.0.0/0")
        self.mocker.result(fail(self.get_ec2_error("i-foobar")))
        self.mocker.replay()

        provider = self.get_provider()
        ex = yield self.assertFailure(
            close_provider_port(provider, machine, "machine-1", 80, "tcp"),
            ProviderInteractionError)
        self.assertEqual(
            str(ex),
            "Unexpected EC2Error closing 80/tcp on machine i-foobar: "
            "The instance ID 'i-foobar' does not exist")

    @inlineCallbacks
    def test_get_provider_opened_ports_unknown_instance(self):
        """Verify open port op will use the correct EC2 API."""
        self.ec2.describe_security_groups("juju-moon-machine-1")
        self.mocker.result(fail(self.get_ec2_error("i-foobar")))
        self.mocker.replay()

        provider = self.get_provider()
        machine = ProviderMachine("i-foobar", "x1.example.com")
        ex = yield self.assertFailure(
            get_provider_opened_ports(provider, machine, "machine-1"),
            ProviderInteractionError)
        self.assertEqual(
            str(ex),
            "Unexpected EC2Error getting open ports on machine i-foobar: "
            "The instance ID 'i-foobar' does not exist")


class EC2RemoveGroupsTest(EC2TestMixin, TestCase):

    @inlineCallbacks
    def test_destroy_environment_security_group(self):
        """Verify the deletion of the security group for the environment"""
        self.ec2.delete_security_group("juju-moon")
        self.mocker.result(succeed(True))
        self.mocker.replay()

        provider = self.get_provider()
        destroyed = yield destroy_environment_security_group(provider)
        self.assertTrue(destroyed)

    @inlineCallbacks
    def test_destroy_environment_security_group_missing(self):
        """Verify ignores errors in deleting the env security group"""
        log = self.capture_logging(level=logging.DEBUG)
        self.ec2.delete_security_group("juju-moon")
        self.mocker.result(fail(
                self.get_ec2_error(
                    "juju-moon",
                    format="The security group %r does not exist"
                    )))
        self.mocker.replay()

        provider = self.get_provider()
        destroyed = yield destroy_environment_security_group(provider)
        self.assertFalse(destroyed)
        self.assertIn(
            "Ignoring EC2 error when attempting to delete group "
            "juju-moon: Error Message: The security group "
            "'juju-moon' does not exist",
            log.getvalue())
