"""Handling of the credentials needed to authenticate with the OpenStack api

Supports several different sets of credentials different auth modes need:
* 'legacy' is built into nova and deprecated in favour of using keystone
* 'keypair' works with the HP public cloud implemention of keystone
* 'userpass' is the way keystone seems to want to do authentication generally
"""

import os


class OpenStackCredentials(object):
    """Encapsulation of credentials used to authenticate with OpenStack"""

    _config_vars = {
            'auth-url': ("OS_AUTH_URL", "NOVA_URL"),
            'username': ("OS_USERNAME", "NOVA_USERNAME"),
            'password': ("OS_PASSWORD", "NOVA_PASSWORD"),
            # HP exposes both a numeric id and a name for tenants, passed back
            # as tenantId and tenantName. Use the name only for simplicity.
            'project-name': ("OS_TENANT_NAME", "NOVA_PROJECT_NAME",
                             "NOVA_PROJECT_ID"),
            'region': ("OS_REGION_NAME", "NOVA_REGION_NAME", "NOVA_REGION"),
            # The key variables don't seem to have modern OS_ prefixed aliases
            'access-key': ("NOVA_API_KEY",),
            'secret-key': ("EC2_SECRET_KEY", "AWS_SECRET_ACCESS_KEY"),
            # A usable mode can normally be guessed, but may be configured
            'auth-mode': (),
        }

    # Really, legacy auth could pass in the project id and keystone doesn't
    # require it, but this is what the client expects for now.
    _modes = {
        'userpass': ('username', 'password', 'project-name'),
        'rax': ('username', 'password', 'project-name'),
        'keypair': ('access-key', 'secret-key', 'project-name'),
        'legacy': ('username', 'access-key'),
        }

    _version_to_mode = {
        "v2.0": 'userpass',
        "v1.1": 'legacy',
        "v1.0": 'legacy',
        }

    def __init__(self, creds_dict):
        url = creds_dict.get("auth-url")
        if not url:
            raise ValueError("Missing config 'auth-url' for OpenStack api")
        mode = creds_dict.get("auth-mode")
        if mode is None:
            mode = self._guess_auth_mode(url)
        elif mode not in self._modes:
            # The juju.environment.config layer should raise a pretty error
            raise ValueError("Unknown 'auth-mode' value %r" % (self.mode,))
        missing_keys = [key for key in self._modes[mode]
            if not creds_dict.get(key)]
        if missing_keys:
            raise ValueError("Missing config %s required for %s auth" % (
                ", ".join(map(repr, missing_keys)), mode))
        self.url = url
        self.mode = mode
        for key in self._config_vars:
            if key not in ("auth-url", "auth-mode"):
                setattr(self, key.replace("-", "_"), creds_dict.get(key))

    @classmethod
    def _guess_auth_mode(cls, url):
        """Pick a mode based on the version at the end of `url` given"""
        final_part = url.rstrip("/").rsplit("/", 1)[-1]
        try:
            return cls._version_to_mode[final_part]
        except KeyError:
            raise ValueError(
                "Missing config 'auth-mode' as unknown version"
                " in 'auth-url' given: " + url)

    @classmethod
    def _get(cls, config, key):
        """Retrieve `key` from `config` if present or in matching envvars"""
        val = config.get(key)
        if val is None:
            for env_key in cls._config_vars[key]:

                val = os.environ.get(env_key)
                if val:
                    return val
        return val

    @classmethod
    def from_environment(cls, config):
        """Create credentials from `config` falling back to environment"""
        return cls(dict((k, cls._get(config, k)) for k in cls._config_vars))

    def set_config_defaults(self, data):
        """Populate `data` with these credentials where not already set"""
        for key in self._config_vars:
            if key not in data:
                val = getattr(self, key.replace("auth-", "").replace("-", "_"))
                if val is not None:
                    data[key] = val
