/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the libgltf project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#include "RenderScene.h"

#include "Common.h"
#include "Shaders.h"
#include "TimeFunction.h"

#include <glm/gtx/transform.hpp>
#include <glm/gtc/matrix_inverse.hpp>
#include <glm/gtc/type_ptr.hpp>
#include <glm/gtc/quaternion.hpp>
#include <glm/gtx/quaternion.hpp>

#include <algorithm>
#include <vector>
#include <cerrno>
#include <cmath>
#include <iostream>

using namespace std;

namespace libgltf
{
namespace {

    static int CheckGLError()
    {
        int retCode = LIBGLTF_SUCCESS;

        GLenum glErr = glGetError();
        while (glErr != GL_NO_ERROR)
        {
            std::string sError;
            switch ( glErr )
            {
                case GL_INVALID_ENUM:
                    sError = "invalid enumerant";
                    break;
                case GL_INVALID_VALUE:
                    sError = "invalid value";
                    break;
                case GL_INVALID_OPERATION:
                    sError = "invalid operation";
                    break;
                case GL_STACK_OVERFLOW:
                    sError = "stack overflow";
                    break;
                case GL_STACK_UNDERFLOW:
                    sError = "stack underflow";
                    break;
                case GL_OUT_OF_MEMORY:
                    sError = "out of memory";
                    break;
                case GL_INVALID_FRAMEBUFFER_OPERATION:
                    sError = "invalid framebuffer operation";
                    break;
                default:
                    break;
            }
            if (!sError.empty())
                std::cerr << "OpenGL Error: " << sError.c_str() << std::endl;
            else
                std::cerr << "OpenGL Error, Error Code: " << glErr << std::endl;
            retCode = LIBGLTF_MEMORY_ERROR;
            glErr = glGetError();
        }
        return retCode;
    }
    static int createOpenglBuffer(GLuint * buffers, GLenum target,
        GLsizeiptr size, const GLvoid * pData,  GLenum usage)
    {
        int errCode;
        glGenBuffers(1, buffers);
        errCode = CheckGLError();
        if(errCode != LIBGLTF_SUCCESS )
        {
            return errCode;
        }
        glBindBuffer(target, *buffers);
        errCode = CheckGLError();
        if(errCode != LIBGLTF_SUCCESS )
        {
            return errCode;
        }
        glBufferData(target, size, NULL, usage);
        errCode = CheckGLError();
        if(errCode != LIBGLTF_SUCCESS )
        {
            return errCode;
        }
        int nBufferSize = 0;
        glGetBufferParameteriv(target, GL_BUFFER_SIZE, &nBufferSize);
        if ( nBufferSize != size )
        {
            std::cerr << "Create OpenGL buffer successfully, but size of this buffer is not correct." << std::endl;
            return LIBGLTF_MEMORY_ERROR;
        }
        glBufferSubData(target, 0, size, pData);
        errCode = CheckGLError();
        if(errCode != LIBGLTF_SUCCESS )
        {
            return errCode;
        }
        return LIBGLTF_SUCCESS;
}
    }
RenderWithFBO::RenderWithFBO()
    : mFboProId(0)
    , mFboId(0)
    , mRboId(0)
    , mTexId(0)
    , mShotTexId(0)
    , mRenderTexId(0)
    , mInverseFboId(0)
    , mInverseRboId(0)
    , mInverseTexId(0)
    , mMSAAFboId(0)
    , mMSAARboId(0)
    , mMSAATexId(0)
    , mVertexBuf(0)
    , mTexCoordBuf(0)
{
}

void RenderWithFBO::inverseBitMap(int width, int height)
{
    GLuint proId = loadFboShader(INVERSEVERT, INVERSEFRAG);
    GLuint texCoordBuf = 0;
    GLuint vertexBuf = 0;
    GLfloat coordVertices[] =
    {
        0.0f, 1.0f,
        1.0f, 1.0f,
        1.0f, 0.0f,
        0.0f, 0.0f,
    };
    GLfloat squareVertices[] =
    {
        -1.0f, -1.0f, -1.0,
         1.0f, -1.0f, -1.0,
         1.0f,  1.0f, -1.0,
        -1.0f,  1.0f, -1.0
    };
    setBufferForFbo(texCoordBuf, vertexBuf, coordVertices,
                    sizeof(coordVertices), squareVertices,
                    sizeof(squareVertices));
    createAndBindInverseFBO(width, height);
    glViewport(0, 0, width, height);
    inverseTexture(proId, texCoordBuf, vertexBuf);
}

void RenderWithFBO::releaseBitMapFBO()
{
    glDeleteFramebuffers(1, &mInverseFboId);
    glDeleteRenderbuffers(1, &mInverseRboId);
    glDeleteTextures(1, &mInverseTexId);

}

void RenderWithFBO::releaseBitmapTexture()
{
    glDeleteTextures(1, &mShotTexId);
}

void RenderWithFBO::releaseMSAAFBO()
{
    if( mMSAAFboId != 0 )
    {
        glDeleteFramebuffers(1, &mMSAAFboId);
        glDeleteRenderbuffers(1, &mMSAARboId);
        glDeleteTextures(1, &mMSAATexId);
    }
}
GLuint RenderWithFBO::loadFboShader(const char* vShader,
                                    const char* fShader)
{
    ShaderProgram shaderPro;
    GLuint progId = glCreateProgram();
    if(false == shaderPro.loadShader(progId, vShader,
                                     strlen(vShader), GL_VERTEX_SHADER))
    {
        return 0;
    }
    if(false == shaderPro.loadShader(progId, fShader,
                                     strlen(fShader), GL_FRAGMENT_SHADER))
    {
        return 0;
    }
    return progId;
}

void RenderWithFBO::setBufferForFbo(GLuint& texCoordBuf, GLuint& vertexBuf,
                                    GLfloat* pCoord, GLuint numCoord,
                                    GLfloat* pSquare, GLuint numSquare)
{
    createOpenglBuffer(&texCoordBuf, GL_ARRAY_BUFFER, numCoord, pCoord,
        GL_STATIC_DRAW);
    glBindBuffer(GL_ARRAY_BUFFER, 0);

    createOpenglBuffer(&vertexBuf, GL_ARRAY_BUFFER, numSquare, pSquare,
        GL_STATIC_DRAW);
    glBindBuffer(GL_ARRAY_BUFFER, 0);
}

void RenderWithFBO::createAndBindInverseFBO(int width, int height)
{
    glBindFramebuffer(GL_FRAMEBUFFER, 0);
    createRenderObj(width, height, mInverseRboId);
    createTextureObj(width, height, mInverseTexId);
    createFrameBufferObj(mInverseFboId, mInverseTexId, mInverseRboId);
    glBindFramebuffer(GL_FRAMEBUFFER, mInverseFboId);
}

void RenderWithFBO::createBitmapTexture(int width, int height)
{
    std::vector<unsigned char> buf(width*height*3);
    glBindFramebuffer(GL_FRAMEBUFFER, mFboId);
    glReadPixels(0, 0, width, height, GL_BGR, GL_UNSIGNED_BYTE, &buf[0]);

    glGenTextures(1, &mShotTexId);
    glBindTexture(GL_TEXTURE_2D, mShotTexId);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, width, height, 0, GL_BGR,
        GL_UNSIGNED_BYTE, &buf[0]);
    glBindTexture(GL_TEXTURE_2D, 0);
}

void RenderWithFBO::inverseTexture(GLuint proId, GLuint texCoordBuf,
                                   GLuint vertexBuf)
{
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    GLuint texCoordId;
    GLuint vertexId;
    {
        GLint iTmp = glGetAttribLocation(proId, "texCoord");
        if( iTmp == -1 )
            return;
        texCoordId = static_cast<GLuint>(iTmp);

        iTmp = glGetAttribLocation(proId, "vPosition");
        if( iTmp == -1 )
            return;
        vertexId = static_cast<GLuint>(iTmp);
    }

    GLint textureId = glGetUniformLocation(proId, "RenderTex");
    if( textureId == -1 )
        return;

    glUseProgram(proId);
    glEnableVertexAttribArray(vertexId);
    glBindBuffer(GL_ARRAY_BUFFER, vertexBuf);
    glVertexAttribPointer(vertexId, 3, GL_FLOAT, GL_FALSE, 0, 0);
    glEnableVertexAttribArray(texCoordId);
    glBindBuffer(GL_ARRAY_BUFFER, texCoordBuf);
    glVertexAttribPointer(texCoordId, 2, GL_FLOAT, GL_FALSE, 0, 0);
    glActiveTexture(GL_TEXTURE0);
    glBindTexture(GL_TEXTURE_2D, mShotTexId);
    glUniform1i(textureId, 0);
    glDrawArrays(GL_QUADS, 0, 4);
    glDisableVertexAttribArray(vertexId);
    glDisableVertexAttribArray(texCoordId);
    glActiveTexture(GL_TEXTURE0);
    glBindTexture(GL_TEXTURE_2D, 0);
    glUseProgram(0);
}

void RenderWithFBO::createRenderObj(int width, int height, GLuint& rboId)
{
    glGenRenderbuffers(1, &rboId);
    glBindRenderbuffer(GL_RENDERBUFFER, rboId);
    glRenderbufferStorage(GL_RENDERBUFFER, GL_DEPTH_COMPONENT, width, height);
    glBindRenderbuffer(GL_RENDERBUFFER, 0);
}

void RenderWithFBO::createTextureObj(int width, int height, GLuint& texId)
{
    glGenRenderbuffers(1, &texId);
    glBindRenderbuffer(GL_RENDERBUFFER, texId);
    glRenderbufferStorage(GL_RENDERBUFFER, GL_RGB, width, height);
    glBindRenderbuffer(GL_RENDERBUFFER, 0);

    glGenTextures(1, &mRenderTexId);
    glBindTexture(GL_TEXTURE_2D, mRenderTexId);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
}

int RenderWithFBO::createFrameBufferObj(GLuint& fboId, GLuint texId,
                                        GLuint rboId)
{
    GLenum status;
    glGenFramebuffers(1, &fboId);
    status = glCheckFramebufferStatus(GL_FRAMEBUFFER);
    if (GL_FRAMEBUFFER_COMPLETE != status)
    {
        return LIBGLTF_CREATE_FBO_ERROR;
    }
    glBindFramebuffer(GL_FRAMEBUFFER, fboId);

    glBindRenderbuffer(GL_RENDERBUFFER, texId);
    glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0,
        GL_RENDERBUFFER, texId);
    status = glCheckFramebufferStatus(GL_FRAMEBUFFER);
    if (GL_FRAMEBUFFER_COMPLETE != status)
    {
        return LIBGLTF_CREATE_FBO_ERROR;
    }
    glBindRenderbuffer(GL_RENDERBUFFER, 0);

    glBindRenderbuffer(GL_RENDERBUFFER, rboId);
    glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT,
                              GL_RENDERBUFFER, rboId);
    status = glCheckFramebufferStatus(GL_FRAMEBUFFER);
    if (GL_FRAMEBUFFER_COMPLETE != status)
    {
        return LIBGLTF_CREATE_FBO_ERROR;
    }
    glBindRenderbuffer(GL_RENDERBUFFER, 0);
    glBindFramebuffer(GL_FRAMEBUFFER, 0);
    return LIBGLTF_SUCCESS;
}

int RenderWithFBO::createMultiSampleTextureFrameBufObj(GLuint& fboId,
                                                       GLuint& texId,
                                                       GLuint&  rboId,
                                                       int width,
                                                       int height)
{
    GLenum status;
    glGenFramebuffers(1, &fboId);
    status = glCheckFramebufferStatus(GL_FRAMEBUFFER);
    if (GL_FRAMEBUFFER_COMPLETE != status)
    {
        return LIBGLTF_CREATE_FBO_ERROR;
    }
    glBindFramebuffer(GL_FRAMEBUFFER, fboId);
    //create MSAA texture
    glGenRenderbuffers(1, &texId);
    glBindRenderbuffer(GL_RENDERBUFFER, texId);
    glRenderbufferStorageMultisample(GL_RENDERBUFFER, 4,  GL_RGB, width, height);
    glBindRenderbuffer(GL_RENDERBUFFER, 0);
    glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0,
                              GL_RENDERBUFFER, texId);
    status = glCheckFramebufferStatus(GL_FRAMEBUFFER);
    if (GL_FRAMEBUFFER_COMPLETE != status)
    {
        return LIBGLTF_CREATE_FBO_ERROR;
    }
    glGenRenderbuffers(1, &rboId);
    glBindRenderbuffer(GL_RENDERBUFFER, rboId);
    glRenderbufferStorageMultisample(GL_RENDERBUFFER, 4, GL_DEPTH_COMPONENT24,
                                     width, height);
    glBindRenderbuffer(GL_RENDERBUFFER, 0);
    glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT,
                              GL_RENDERBUFFER, rboId);
    status = glCheckFramebufferStatus(GL_FRAMEBUFFER);
    if (GL_FRAMEBUFFER_COMPLETE != status)
    {
        return LIBGLTF_CREATE_FBO_ERROR;
    }
    return LIBGLTF_SUCCESS;
}

int RenderWithFBO::renderFbo(int srcWidth, int srcHeight)
{
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    glBindFramebuffer(GL_FRAMEBUFFER, mFboId);
    glBindTexture(GL_TEXTURE_2D, mRenderTexId);
    glCopyTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, 0, 0, srcWidth, srcHeight, 0);
    glBindTexture(GL_TEXTURE_2D, 0);
    renderFboTexture();
    return LIBGLTF_SUCCESS;
}

int RenderWithFBO::renderFboTexture()
{
    glBindFramebuffer(GL_FRAMEBUFFER, 0);
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    glUseProgram(mFboProId);

    GLuint texCoordId;
    GLuint vertexId;
    {
        GLint iTmp = glGetAttribLocation(mFboProId, "texCoord");
        if( iTmp == -1 )
            return LIBGLTF_UNKNOWN_ERROR;
        texCoordId = static_cast<GLuint>(iTmp);

        iTmp = glGetAttribLocation(mFboProId, "vPosition");
        if( iTmp == -1 )
            return LIBGLTF_UNKNOWN_ERROR;
        vertexId = static_cast<GLuint>(iTmp);
    }

    GLint textureId = glGetUniformLocation(mFboProId, "RenderTex");
    if( textureId == -1 )
        return LIBGLTF_UNKNOWN_ERROR;

    glEnableVertexAttribArray(vertexId);
    glBindBuffer(GL_ARRAY_BUFFER, mVertexBuf);
    glVertexAttribPointer(vertexId, 2, GL_FLOAT, GL_FALSE, 0, 0);

    glEnableVertexAttribArray(texCoordId);
    glBindBuffer(GL_ARRAY_BUFFER, mTexCoordBuf);
    glVertexAttribPointer(texCoordId, 2, GL_FLOAT, GL_FALSE, 0, 0);

    glBindTexture(GL_TEXTURE_2D, mRenderTexId);
    glUniform1i(textureId, 0);
    glDrawArrays(GL_QUADS, 0, 4);
    glDisableVertexAttribArray(texCoordId);
    glDisableVertexAttribArray(vertexId);
    return LIBGLTF_SUCCESS;
}

int RenderWithFBO::createAndBindFbo(int width, int height, bool isUseMSAA)
{
    if (0 != mFboId)
    {
        return LIBGLTF_SUCCESS;
    }

    mFboProId = loadFboShader(FBOVERT, FBOFRAG);
    if(mFboProId == 0)
    {
        return LIBGLTF_SHADER_ERROR;
    }
    createRenderObj(width, height, mRboId);
    createTextureObj(width, height, mTexId);
    int status = createFrameBufferObj(mFboId, mTexId, mRboId);
    if (LIBGLTF_SUCCESS != status)
    {
        return status;
    }
    if (isUseMSAA)
    {
        status = createMultiSampleTextureFrameBufObj(mMSAAFboId, mMSAATexId,
            mMSAARboId, width, height);
        if (LIBGLTF_SUCCESS != status)
        {
            return status;
        }
    }
    GLfloat coordVertices[] =
    {
        0.0f, 0.0f,
        1.0f, 0.0f,
        1.0f, 1.0f,
        0.0f, 1.0f,
    };
    GLfloat squareVertices[] =
    {
        -1.0f, -1.0f,
         1.0f, -1.0f,
         1.0f,  1.0f,
        -1.0f,  1.0f,
    };
    setBufferForFbo(mTexCoordBuf, mVertexBuf, coordVertices,
                    sizeof(coordVertices), squareVertices,
                    sizeof(squareVertices));
    return LIBGLTF_SUCCESS;
}

void RenderWithFBO::releaseFbo()
{
    if (0 != mFboId)
    {
        glDeleteFramebuffers(1, &mFboId);
        glDeleteRenderbuffers(1, &mRboId);
        glDeleteTextures(1, &mTexId);
        mFboId = 0;
    }
    if (0 != mShotTexId)
    {
        glDeleteTextures(1, &mShotTexId);
    }
    if (0 != mRenderTexId)
    {
        glDeleteTextures(1, &mRenderTexId);
    }
}

RenderPrimitive::RenderPrimitive()
    : mIndicesDataType(DataType_UNKNOW)
    , mpMaterial(0)
    , pNode(0)
    , mVerterCount(0)
    , mIndicesCount(0)
    , mVertexBuffer(0)
    , mNormalBuffer(0)
    , mTexCoordBuffer(0)
    , mJointBuffer(0)
    , mWeightBuffer(0)
    , mIndicesBuffer(0)
    , mVertexBufferData(0)
{
}

RenderPrimitive::~RenderPrimitive()
{
    glDeleteBuffers(1, &mVertexBuffer);
    glDeleteBuffers(1, &mNormalBuffer);
    glDeleteBuffers(1, &mTexCoordBuffer);
    glDeleteBuffers(1, &mJointBuffer);
    glDeleteBuffers(1, &mWeightBuffer);
    glDeleteBuffers(1, &mIndicesBuffer);
}

const Node* RenderPrimitive::getNode() const
{
    return pNode;
}

void RenderPrimitive::setNode(Node* node)
{
    pNode = node;
}

const Material* RenderPrimitive::getMaterial() const
{
    return mpMaterial;
}

void RenderPrimitive::setMaterial(Material* pMaterial)
{
    mpMaterial = pMaterial;
}

unsigned int RenderPrimitive::getVerterCount() const
{
    return mVerterCount;
}

void RenderPrimitive::setVerterCount(unsigned int count)
{
    mVerterCount = count;
}

unsigned int RenderPrimitive::getIndicesCount() const
{
    return mIndicesCount;
}

void RenderPrimitive::setIndicesCount(unsigned int count)
{
    mIndicesCount = count;
}

DataType RenderPrimitive::getIndicesDataType() const
{
    return mIndicesDataType;
}

void RenderPrimitive::setIndicesDataType(DataType type)
{
    mIndicesDataType = type;
}

void RenderPrimitive::getPrimitiveBoundary(glm::vec3* vertexMax,
    glm::vec3* vertexMin)
{
    const glm::mat4& modelMatrix = pNode->getGlobalMatrix();
    const glm::vec3* pVertices = reinterpret_cast<const glm::vec3*>(mVertexBufferData);
    for (size_t i = 0; i < mVerterCount; ++i)
    {
        glm::vec4 tmpPos = glm::vec4(pVertices[i],1.0f);
        glm::vec4 globalPos = modelMatrix*tmpPos;
        vertexMax->x = max(globalPos.x,vertexMax->x);
        vertexMax->y = max(globalPos.y,vertexMax->y);
        vertexMax->z = max(globalPos.z,vertexMax->z);
        vertexMin->x = min(globalPos.x,vertexMin->x);
        vertexMin->y = min(globalPos.y,vertexMin->y);
        vertexMin->z = min(globalPos.z,vertexMin->z);
    }
}

void RenderPrimitive::setVertexBufferData(const char* srcBuf)
{
    mVertexBufferData = srcBuf;
}

unsigned int RenderPrimitive::getVertexBuffer() const
{
    return mVertexBuffer;
}

void RenderPrimitive::setVertexBuffer(unsigned int bufferId)
{
    mVertexBuffer = bufferId;
}

unsigned int RenderPrimitive::getNormalBuffer() const
{
    return mNormalBuffer;
}

void RenderPrimitive::setNormalBuffer(unsigned int bufferId)
{
    mNormalBuffer = bufferId;
}

unsigned int RenderPrimitive::getTexCoordBuffer() const
{
    return mTexCoordBuffer;
}

void RenderPrimitive::setTexCoordBuffer(unsigned int bufferId)
{
    mTexCoordBuffer = bufferId;
}

unsigned int RenderPrimitive::getJointBuffer() const
{
    return mJointBuffer;
}

void RenderPrimitive::setJointBuffer(unsigned int bufferId)
{
    mJointBuffer = bufferId;
}

unsigned int RenderPrimitive::getWeightBuffer() const
{
    return mWeightBuffer;
}

void RenderPrimitive::setWeightBuffer(unsigned int bufferId)
{
    mWeightBuffer = bufferId;
}

unsigned int RenderPrimitive::getIndicesBuffer() const
{
    return mIndicesBuffer;
}

void RenderPrimitive::setIndicesBuffer(unsigned int bufferId)
{
    mIndicesBuffer = bufferId;
}

RenderShader::RenderShader()
    : mPrimitiveVec()
    , mpTechnique(0)
{
}

RenderShader::~RenderShader()
{
    for (unsigned int i = 0, size = mPrimitiveVec.size();
         i < size; ++i)
    {
        delete mPrimitiveVec[i];
    }
    mPrimitiveVec.clear();
}

Technique* RenderShader::getTechnique()
{
    return mpTechnique;
}

void RenderShader::setTechnique(Technique* pTechnique)
{
    mpTechnique = pTechnique;
}

void RenderShader::pushRenderPrim(RenderPrimitive* p)
{
    (mPrimitiveVec).push_back(p);
}

RenderPrimitive* RenderShader::getRenderPrim(unsigned int index)
{
    if (mPrimitiveVec.size() <= index)
    {
        return 0;
    }
    return mPrimitiveVec[index];
}

unsigned int RenderShader::getRenderPrimSize() const
{
    return mPrimitiveVec.size();
}

RenderScene::RenderScene()
    : maCamera()
    , cCamera(0)
    , vCameraIndex()
    , mOrbitInitViewMatrix(0.0)
    , mWalkthroughInitViewMatrix(0.0)
    , bAerialView(false)
    , previousUpdateCameraTime(0.0)
    , flyInfo(1.0f)
    , flyTime(0.0)
    , bFlyCamera(false)
    , bAnimation(true)
    , pLight(0)
    , mAnimationPlay(false)
    , mAnimationLoop(true)
    , mCurrentTime(0)
    , mLastPlaying(0)
    , mUpdateTimeOut(0)
    , mDuration(0)
    , mShaderVec()
    , pScene(0)
    , mLoadJson()
    , mBindBufferMap()
    , mShaderProgram()
    , mCurrentViewport()
    , fbo()
    , mEnableTransparency(false)
    , mEnableRotation(true)
    , mLastModelView(glm::mat4(0.0))
    , bIsTimeAvailable(false)
    , mViewMatrixBeforePatrol(glm::mat4(0.0))
    , mIsCameraAnimationRunning(true)
    , mIsCameraAnimationRunningBeforePatrol(true)
    , pFPSCounter(0)
    , mIsFPSCounterPrinted(false)
    , mIsUseMSAA(false)
#if LOAD_ONCE
    , mCurrentImage()
    , mCurrentTextNumber(0)
#endif
{
}

RenderScene::~RenderScene()
{
    delete pFPSCounter;

    delete pLight;

    for (unsigned int i = 0, size = mShaderVec.size();
         i < size; ++i)
    {
         delete mShaderVec[i];
    }
    mShaderVec.clear();
    mBindBufferMap.clear();
    delete pScene;
}

int RenderScene::loadScene(const std::vector<glTFFile>& inputFiles)
{
    try
    {
        int iStatus = mLoadJson.parseScene(inputFiles);
        return iStatus;
    }
    catch (boost::property_tree::ptree_error& e)
    {
        std::cerr << e.what() << std::endl;
        return LIBGLTF_PARSE_JSON_ERROR;
    }
    catch (boost::exception const&)
    {
        std::cerr << "unknown boost error" << std::endl;
        return LIBGLTF_UNKNOWN_BOOST_ERROR;
    }
}

bool RenderScene::initScene(const std::string& jsonfile, std::vector<glTFFile>& o_glTFFiles)
{
    if (jsonfile.empty())
    {
        return false;
    }
    if (!mLoadJson.parseJsonFile(jsonfile))
    {
        return false;
    }
    mLoadJson.getFileNamesInJson(o_glTFFiles);
    pScene = new Scene();
    mLoadJson.setScene(pScene);
    return true;
}

unsigned int RenderScene::bindAttribute(const Attribute* pAttr)
{
    unsigned int bufferId;
    createOpenglBuffer(&bufferId, GL_ARRAY_BUFFER,
        pAttr->getDataCount() * pAttr->getByteStride(),
        pAttr->getAttributeData(), GL_STATIC_DRAW);
    return bufferId;
}

unsigned int RenderScene::bindIndices(const Attribute* pAttr)
{
    unsigned int bufferId;
    createOpenglBuffer(&bufferId, GL_ELEMENT_ARRAY_BUFFER,
        pAttr->getDataCount() * pAttr->getByteStride(),
        pAttr->getAttributeData(), GL_STATIC_DRAW);
    return bufferId;
}

void RenderScene::constructShader()
{
    for (unsigned int i = 0, size = pScene->getTechSize();
         i < size; ++i)
    {
        RenderShader* pRenderShader = new RenderShader();
        pRenderShader->setTechnique(pScene->getTechnique(i));
        mShaderVec.push_back(pRenderShader);
    }
}

void RenderScene::constructMesh(const std::string& meshName, Node* pNode)
{
    const Mesh* pMesh = pScene->findMesh(meshName);
    for (unsigned int i = 0, size = pMesh->getPrimitiveVecSize();
         i < size; ++i)
    {
        constructPrimitive(pMesh->getPrimitiveVec(i), pNode);
    }
}

void RenderScene::constructPrimitive(const Primitives* pPrimitive,
                                     Node* pNode)
{
    const std::string& materialIndex = pPrimitive->getMaterialIndex();
    Material* pMaterial = pScene->findMaterial(materialIndex);
    if (0 != pMaterial)
    {
        for (unsigned int i = 0, size = mShaderVec.size();
             i < size; ++i)
        {
            RenderShader* pRenderShader = mShaderVec[i];
            Technique* pTechnique = pRenderShader->getTechnique();
            if (pMaterial->getTechniqueId() == pTechnique->getTechId())
            {
                RenderPrimitive* pRenderPrimitive = new RenderPrimitive();
                bindAttributeBuffer(pPrimitive, pRenderPrimitive);
                pRenderPrimitive->setMaterial(pMaterial);
                pRenderPrimitive->setNode(pNode);

                pRenderShader->pushRenderPrim(pRenderPrimitive);
                break;
            }
        }
    }
}

void RenderScene::bindAttributeBuffer(const Primitives* pPrimitive,
                                      RenderPrimitive* pRP)
{
    Attribute* pAttr;
    BindBufferInfo bindBufferInfo;
    std::map<std::string, BindBufferInfo>::const_iterator it;

    const std::string& posId = pPrimitive->getAttributeIndex("POSITION");
    it = mBindBufferMap.find(posId);
    if (mBindBufferMap.end() == it)
    {
        pAttr = pScene->findAttribute(posId);
        if (0 != pAttr)
        {
            bindBufferInfo.mBufferId = bindAttribute(pAttr);
            bindBufferInfo.mDataCount = pAttr->getDataCount();
            bindBufferInfo.mBufferLen = pAttr->getDataCount()
                                        * pAttr->getByteStride();
            bindBufferInfo.mSrcData = pAttr->getAttributeData();
            pRP->setVertexBuffer(bindBufferInfo.mBufferId);
            pRP->setVerterCount(bindBufferInfo.mDataCount);
            pRP->setVertexBufferData(bindBufferInfo.mSrcData);
            mBindBufferMap.insert(
                std::map<std::string, BindBufferInfo>::value_type
                (posId, bindBufferInfo));
        }
    }
    else
    {
        pRP->setVertexBuffer(it->second.mBufferId);
        pRP->setVerterCount(it->second.mDataCount);
        pRP->setVertexBufferData(it->second.mSrcData);
    }

    const std::string& NorId = pPrimitive->getAttributeIndex("NORMAL");
    it = mBindBufferMap.find(NorId);
    if (mBindBufferMap.end() == it)
    {
        pAttr = pScene->findAttribute(NorId);
        if (0 != pAttr)
        {
            bindBufferInfo.mBufferId = bindAttribute(pAttr);
            pRP->setNormalBuffer(bindBufferInfo.mBufferId);
            mBindBufferMap.insert(
                std::map<std::string, BindBufferInfo>::value_type
                (NorId, bindBufferInfo));
        }
    }
    else
    {
        pRP->setNormalBuffer(it->second.mBufferId);
    }

    const std::string& TexId = pPrimitive->getAttributeIndex("TEXCOORD_0");
    it = mBindBufferMap.find(TexId);
    if (mBindBufferMap.end() == it)
    {
        Attribute* pMutableAttr = pScene->findAttribute(TexId);
        if (0 != pMutableAttr)
        {
            float* pData = reinterpret_cast<float*>(pMutableAttr->getAttributeData());
            for (unsigned int i = 0, size = pMutableAttr->getDataCount();
                 i < size; i++)
            {
                pData[i * 2 + 1] = 1 - pData[i * 2 + 1];
            }
            bindBufferInfo.mBufferId = bindAttribute(pMutableAttr);
            pRP->setTexCoordBuffer(bindBufferInfo.mBufferId);
            mBindBufferMap.insert(
                std::map<std::string, BindBufferInfo>::value_type
                (TexId, bindBufferInfo));
        }
    }
    else
    {
        pRP->setTexCoordBuffer(it->second.mBufferId);
    }

    pAttr = pScene->findAttribute(pPrimitive->getAttributeIndex("JOINT"));
    if (0 != pAttr)
    {
        pRP->setJointBuffer(bindAttribute(pAttr));
    }

    pAttr = pScene->findAttribute(pPrimitive->getAttributeIndex("WEIGHT"));
    if (0 != pAttr)
    {
        pRP->setWeightBuffer(bindAttribute(pAttr));
    }

    pAttr = pScene->findAttribute(pPrimitive->getIndicesIndex());
    if (0 != pAttr)
    {
        pRP->setIndicesBuffer(bindIndices(pAttr));
        pRP->setIndicesCount(pAttr->getDataCount());
        pRP->setIndicesDataType(pAttr->getDataType());
    }
}

void RenderScene::getCameraIndex(Node* pNode)
{
    for (unsigned int i = 0, size = pNode->getChildNodeSize();
         i < size; ++i)
    {
        Node* pChild = pNode->getChildNode(i);
        if (!pChild->getCameraIndex().empty())
        {
            vCameraIndex.push_back(pChild->getCameraIndex());
        }
        getCameraIndex(pChild);
    }
}
void RenderScene::initOpengl()
{
    glClearColor(1.0f, 1.0f, 1.0f, 1.0f);
    glClear(GL_COLOR_BUFFER_BIT);
    glEnable(GL_TEXTURE_2D);
    glEnable(GL_DEPTH_TEST);
    glEnable(GL_LIGHTING);
    glEnable(GL_LIGHT0);
    glDepthFunc(GL_LESS);
}

void RenderScene::createDefaultCamera()
{
    glm::vec3 vDelta = pScene->getVertexMax() - pScene->getVertexMin();
    glm::vec3 vHalf =  pScene->getVertexMin() + (vDelta / 2.0f);

    float fPos = std::sqrt(vDelta.x * vDelta.x
                    + vDelta.y * vDelta.y
                    + vDelta.z * vDelta.z);

    glm::vec3 vEye;
    vEye = glm::vec3(0, vHalf.y, 1.5 * fPos);
    maCamera.setViewMatrix(glm::lookAt(vEye, vHalf, glm::vec3(0.0f, 1.0f, 0.0f)));
    mOrbitInitViewMatrix = maCamera.getViewMatrix();
    maCamera.flength = fPos;
    maCamera.vModelCenterPos = vHalf;

    getCameraIndex(pScene->getRootNode());
    if (vCameraIndex.size() > 0)
    {
        std::vector<std::string>::iterator it;
        it = vCameraIndex.begin();
        cCamera = pScene->findCamera(*it);
    }

    float fovy = 37.8492f;
    float aspect = 1.5f;
    float zNear = 1.0f;
    float zFar = 500000.0f;

    if (pScene->getUseCameraInJson())
    {
        fovy = cCamera->getXFov();
        aspect = cCamera->getAspectRatio();
        zNear = cCamera->getNear();
        zFar = cCamera->getFar();
        Node* pNode = cCamera->getCameraNode();
        maCamera.setViewMatrix(glm::inverse(pNode->getGlobalMatrix()));
    }
    if (fPos * 6 > zFar)
    {
        zFar =fPos * 6;
    }
    maCamera.setPerspective(glm::perspective(fovy, aspect, zNear, zFar));
    mWalkthroughInitViewMatrix = maCamera.getViewMatrix();
}

void RenderScene::initNodeTree(Node* pNode, const glm::mat4& matrix,
                               bool parentJointFlag, bool updateFlag)
{
    glm::mat4 globalMatrix;
    bool currJointFlag = pNode->getJointFlag();
    if (!parentJointFlag && currJointFlag)
    {
        globalMatrix = pNode->getLocalMatrix();
    }
    else
    {
        globalMatrix = matrix * pNode->getLocalMatrix();
    }

    if (!pNode->getMatrixFlag())
    {
        updateFlag = true;
    }
    pNode->setUpdateFlag(updateFlag);

    //if (pNode->getSkinIndex().empty())
    {
        pNode->setGlobalMatrix(globalMatrix);
        if( cCamera && pNode == cCamera->getCameraNode() )
        {
            maCamera.setViewMatrix(glm::inverse(globalMatrix));
            mWalkthroughInitViewMatrix = maCamera.getViewMatrix();
        }
    }

    for (unsigned int i = 0, nSize = pNode->getChildNodeSize();
         i < nSize; ++i)
    {
        initNodeTree(pNode->getChildNode(i), globalMatrix,
                     currJointFlag, updateFlag);
    }
}

int RenderScene::initRender(const std::vector<glTFFile>& inputFiles)
{
    if( epoxy_gl_version() < 30 )
    {
        return LIBGLTF_UNKNOWN_ERROR;
    }

    initOpengl();

    int iResult = loadScene(inputFiles);
    if (iResult != LIBGLTF_SUCCESS)
    {
        return iResult;
    }

    Node* pRootNode = pScene->getRootNode();

    constructShader();
    initNodeTree(pRootNode, pRootNode->getGlobalMatrix(), false, false);

    for (unsigned int i = 0, nSize = pScene->getNodeSize();
         i < nSize; ++i)
    {
        Node* pNode = pScene->getNode(i);

        if (0 != pScene->getAnimationCount())
        {
            const std::string& nodeName = pNode->getNodeName();
            pNode->setAnimPoint(pScene->findAnimation(nodeName));
        }

        const std::string& skinIndex = pNode->getSkinIndex();
        if (!skinIndex.empty())
        {
            Node* pPareBone = 0;
            Node* pBone = 0;
            pPareBone = findNodeByName(pRootNode, pNode->getSkeleIndex());
            for (unsigned int j = 0, size = pScene->getSkinSize();
                 j < size; ++j)
            {
                Skin* pSkin = pScene->getSkin(j);
                if (pSkin->getSkinName() == skinIndex)
                {
                    pNode->setSkinPoint(pSkin);
                    for (unsigned int k = 0, jointCount = pSkin->getBoneIdSize();
                         k < jointCount; k++)
                    {
                        pBone = findNodeByJoint(pPareBone, pSkin->getBoneId(k));
                        pNode->pushBoneNode(pBone);
                    }
                    break;
                }
                continue;
            }
        }

        if (pNode->hasMesh())
        {
            for (unsigned int j = 0, size = pNode->getMeshIndexSize();
                 j < size; ++j)
            {
                constructMesh(pNode->getMeshIndex(j), pNode);
            }
        }
    }
    setModelBoundaryValue();
    createDefaultCamera();
    pScene->clearAttributeMap();

    mDuration = pScene->getDuration();
    return LIBGLTF_SUCCESS;
}

void RenderScene::initFPS()
{
    if (0 != pFPSCounter)
    {
        delete pFPSCounter;
    }
    pFPSCounter = new FPSCounter(&mShaderProgram);
    mIsFPSCounterPrinted = true;
}

void RenderScene::endFPS()
{
    mIsFPSCounterPrinted = false ;
}

void RenderScene::renderPrimitive(RenderPrimitive* pPrimitive,
                                  unsigned int progId)
{
    upLoadMatrixInfo(progId, pPrimitive);
    upLoadUniform(progId, pPrimitive);
    upLoadAttribute(progId, pPrimitive);
    upLoadAnimation(progId, pPrimitive);
    drawTriangle(pPrimitive);
}

void RenderScene::upLoadMatrixInfo(unsigned int progId,
                                   RenderPrimitive* pPrimitive)
{
    const Node* pNode = pPrimitive->getNode();
    const glm::mat4& tempMatrix = pNode->getGlobalMatrix();
    const glm::mat4& lookMatrix = maCamera.getViewMatrix();

    {
        mShaderProgram.setUniform(progId, "u_modelViewMatrix", lookMatrix * tempMatrix);
        mShaderProgram.setUniform(progId, "u_normalMatrix",
            glm::mat3(lookMatrix) *
            glm::transpose(glm::inverse(glm::mat3(tempMatrix))));
        mShaderProgram.setUniform(progId, "u_projectionMatrix",
                                  maCamera.getPerspective());
        mShaderProgram.setUniform(progId, "M", tempMatrix);
        mShaderProgram.setUniform(progId, "V", lookMatrix);
    }
}

bool RenderScene::upLoadTechPropertyTransparent()
{
    glEnable(GL_BLEND);
    glBlendEquation(GL_FUNC_ADD);
    glBlendFuncSeparate(GL_SRC_COLOR, GL_ONE_MINUS_DST_COLOR, GL_ZERO, GL_SRC_COLOR);
    glDepthMask(GL_TRUE);
    glEnable(GL_CULL_FACE);
    glEnable(GL_DEPTH_TEST);
    return true;
}

void RenderScene::upLoadTechPropertyOfJsonFile(Technique* pTech)
{
    if (1 == pTech->getTechState().blendEnable )
    {
        glEnable(GL_BLEND);
    }
    else
    {
        glDisable(GL_BLEND);
    }

    if (0 != pTech->getTechState().blendEquation)
    {
        glBlendEquation(pTech->getTechState().blendEquation);
        glBlendFunc(pTech->getTechState().blendFuncSfactor,
                    pTech->getTechState().blendFuncDfactor);
    }

    if (1 == pTech->getTechState().cullFaceEnable)
    {
        glEnable(GL_CULL_FACE);
    }
    else
    {
        glDisable(GL_CULL_FACE);
    }

    if (1 == pTech->getTechState().depthMask)
    {
        glDepthMask(GL_TRUE);
    }
    else
    {
        glDepthMask(GL_FALSE);
    }

    if (1 == pTech->getTechState().depthTestEnable)
    {
        glEnable(GL_DEPTH_TEST);
    }
    else
    {
        glDisable(GL_DEPTH_TEST);
    }
}

void RenderScene::upLoadTechInfo(unsigned int progId, Technique* pTech)
{
    if (mEnableTransparency)
    {
        upLoadTechPropertyTransparent();
    }
    else
    {
        upLoadTechPropertyOfJsonFile(pTech);
    }

    const std::vector<techLight*>& vecLight = pTech->poptLight();
    std::vector<techLight*>::const_iterator it = vecLight.begin();
    std::vector<techLight*>::const_iterator itEnd = vecLight.end();
    for (; it != itEnd; ++it)
    {
        if ((*it)->mSource.empty())
        {
            switch ((*it)->type)
            {
            case DataType_FLOAT:
                mShaderProgram.setUniform(progId, (*it)->mName.c_str(),
                    (*it)->floatValue);
                break;
            case DataType_FLOAT_VEC3:
                mShaderProgram.setUniform(progId, (*it)->mName.c_str(),
                    (*it)->vecValue);
                break;
            case DataType_UNKNOW:
            case DataType_BYTE:
            case DataType_UNSIGNED_BYTE:
            case DataType_SHORT:
            case DataType_UNSIGNED_SHORT:
            case DataType_INT:
            case DataType_UNSIGNED_INT:
            case DataType_ARRAY_BUFFER:
            case DataType_ELEMENT_ARRAY_BUFFER:
            case DataType_ARRAY_BUFFER_BINDING:
            case DataType_ELEMENT_ARRAY_BUFFER_BINDING:
            case DataType_FLOAT_VEC2:
            case DataType_FLOAT_VEC4:
            case DataType_INT_VEC2:
            case DataType_INT_VEC3:
            case DataType_INT_VEC4:
            case DataType_BOOL:
            case DataType_BOOL_VEC2:
            case DataType_BOOL_VEC3:
            case DataType_BOOL_VEC4:
            case DataType_FLOAT_MAT2:
            case DataType_FLOAT_MAT3:
            case DataType_FLOAT_MAT4:
            case DataType_SAMPLER_2D:
            case DataType_SAMPLER_CUBE:
            default:
                break;
            }
        }
        else
        {
            const Node* pNode = pScene->findLightNodeMap((*it)->mSource);
            mShaderProgram.setUniform(progId, (*it)->mName.c_str(),
                                      maCamera.getViewMatrix() *
                                      pNode->getGlobalMatrix());
        }
    }
}

void RenderScene::upLoadAnimation(unsigned int progId,
                                  RenderPrimitive* pPrimitive)
{
    const Node* pNode = pPrimitive->getNode();
    const Skin* pSkin = pNode->getSkinPoint();
    if (0 == pSkin)
    {
        return;
    }

    unsigned int count = pSkin->getBindMatrixCount();
    std::vector<glm::mat4> vTempMatrix(pSkin->getBindMatrix(), pSkin->getBindMatrix() + count);

    const Node* pBone;
    for (unsigned int i = 0; i < vTempMatrix.size(); ++i)
    {
        pBone = pNode->getBoneNode(i);
        if (0 != pBone)
            vTempMatrix[i] = pBone->getGlobalMatrix() * vTempMatrix[i];
    }

    glUniformMatrix4fv(glGetUniformLocation(progId, "u_jointMat"),
                       count, false, glm::value_ptr(vTempMatrix[0]));
}

void RenderScene::updateAnimInfo(Node* pNode)
{
    const Animation* pAnimation = pNode->getAnimPoint();
    if (0 == pAnimation)
    {
        return;
    }

    double time = fmod(mCurrentTime, pAnimation->getDuration());
    glm::mat4 localMatrix = pAnimation->findTimeValue(time);
    if (pAnimation->getChannelBits() == ROTATE_CHANNEL)
    {
        localMatrix = pNode->getTranslate() * localMatrix * pNode->getScale();
    }
    pNode->setLocalMatrix(localMatrix);
}

void RenderScene::updateNodeMatrix(Node* pNode, const glm::mat4& matrix,
                                   bool jointFlag)
{
    glm::mat4 globalMatrix;
    bool flag = pNode->getJointFlag();

    if (pNode->getUpdateFlag())
    {
        updateAnimInfo(pNode);
        if (!jointFlag && flag)
        {
            globalMatrix = pNode->getLocalMatrix();
        }
        else
        {
            globalMatrix = matrix * pNode->getLocalMatrix();
        }

        if (pNode->getSkinIndex().empty())
        {
            if(pNode->getCameraIndex().empty() || mIsCameraAnimationRunning )
            {
                if( cCamera && pNode == cCamera->getCameraNode() && !bAerialView )
                {
                    for (unsigned int i = 0; i < 4; i++)
                    {
                        for (unsigned int j = 0; j < 4; j++)
                        {
                            if (std::abs(pNode->getGlobalMatrix()[i][j] - globalMatrix[i][j]) > 0.01)
                            {
                                maCamera.setViewMatrix(glm::inverse(globalMatrix));
                                mWalkthroughInitViewMatrix = maCamera.getViewMatrix();
                                break;
                            }
                        }
                    }
                }
                pNode->setGlobalMatrix(globalMatrix);
            }
        }
    }
    else
    {
        globalMatrix = pNode->getGlobalMatrix();
    }

    for (unsigned int i = 0, nSize = pNode->getChildNodeSize();
         i < nSize; ++i)
    {
        updateNodeMatrix(pNode->getChildNode(i), globalMatrix, flag);
    }
}

void RenderScene::updateFlyCamera()
{
    static bool startFly = false;
    if (flyTime <= 0)
    {
        bFlyCamera = false;
        startFly = false;
        return;
    }
    glm::mat4 viewMatrix = maCamera.getViewMatrix();
    if (!startFly)
    {
        startFly = true;
        previousUpdateCameraTime = libgltf::time::getCurrentTime();
        maCamera.setViewMatrix(viewMatrix);
        return;
    }
    double currentUpdateCameraTime = libgltf::time::getCurrentTime();
    double timeDifference = libgltf::time::diffTime(currentUpdateCameraTime,
                                        previousUpdateCameraTime) * 1000 * 1000;
    previousUpdateCameraTime = currentUpdateCameraTime;
    flyTime -= timeDifference;
    viewMatrix += flyInfo * (float)timeDifference;
    if (pScene->getUseCameraInJson())
    {
        Node* pNode = cCamera->getCameraNode();
        glm::mat4 tmp = glm::inverse(viewMatrix);
        pNode->setGlobalMatrix(tmp);
    }
    else
    {
        maCamera.setViewMatrix(viewMatrix);
    }
}

void RenderScene::upLoadUniform(unsigned int progId,
                                RenderPrimitive* pPrimitive)
{
    int textureCount = 0;
    const Material* pMaterial = pPrimitive->getMaterial();
    for (unsigned int i = 0, size = pMaterial->getMaterialProperSize();
         i < size; ++i)
    {
        const MaterialProperty* pProper = pMaterial->getMaterialProper(i);
        switch (pProper->getDataType())
        {
        case DataType_FLOAT:
            mShaderProgram.setUniform(progId, pProper->getPropertyName().c_str(),
                                      *(reinterpret_cast<const float*>(pProper->getPropertyData())));
            break;
        case DataType_FLOAT_VEC2:
            mShaderProgram.setUniform(progId, pProper->getPropertyName().c_str(),
                                      *(reinterpret_cast<const glm::vec2*>(pProper->getPropertyData())));
            break;
        case DataType_FLOAT_VEC3:
            mShaderProgram.setUniform(progId, pProper->getPropertyName().c_str(),
                                      *(reinterpret_cast<const glm::vec3*>(pProper->getPropertyData())));
            break;
        case DataType_FLOAT_VEC4:
            mShaderProgram.setUniform(progId, pProper->getPropertyName().c_str(),
                                      *(reinterpret_cast<const glm::vec4*>(pProper->getPropertyData())));
            break;
        case DataType_FLOAT_MAT3:
            mShaderProgram.setUniform(progId, pProper->getPropertyName().c_str(),
                                      *(reinterpret_cast<const glm::mat3*>(pProper->getPropertyData())));
            break;
        case DataType_FLOAT_MAT4:
            mShaderProgram.setUniform(progId, pProper->getPropertyName().c_str(),
                                      *(reinterpret_cast<const glm::mat4*>(pProper->getPropertyData())));
            break;
        case DataType_SAMPLER_2D:
#if LOAD_ONCE
            if (mCurrentImage == pProper->getImagePath() &&
                textureCount == mCurrentTextNumber)
            {
                break;
            }
            mCurrentImage = pProper->getImagePath();
            mCurrentTextNumber = textureCount;
#endif
            mShaderProgram.setUniform(progId,
                                      pProper->getPropertyName().c_str(),
                                      textureCount);
            pScene->findTexture(
                pProper->getImagePath())->bindTexture(textureCount);
            textureCount++;
            break;
        case DataType_UNKNOW:
        case DataType_BYTE:
        case DataType_UNSIGNED_BYTE:
        case DataType_SHORT:
        case DataType_UNSIGNED_SHORT:
        case DataType_INT:
        case DataType_UNSIGNED_INT:
        case DataType_ARRAY_BUFFER:
        case DataType_ELEMENT_ARRAY_BUFFER:
        case DataType_ARRAY_BUFFER_BINDING:
        case DataType_ELEMENT_ARRAY_BUFFER_BINDING:
        case DataType_INT_VEC2:
        case DataType_INT_VEC3:
        case DataType_INT_VEC4:
        case DataType_BOOL:
        case DataType_BOOL_VEC2:
        case DataType_BOOL_VEC3:
        case DataType_BOOL_VEC4:
        case DataType_FLOAT_MAT2:
        case DataType_SAMPLER_CUBE:
        default:
            break;
        }
    }
}

void RenderScene::upLoadAttribute(unsigned int progId,
                                  RenderPrimitive* pPrimitive)
{
    int tmpId;
    if ((tmpId = glGetAttribLocation(progId, "a_position")) != -1 &&
        0 != pPrimitive->getVertexBuffer())
    {
        glEnableVertexAttribArray(tmpId);
        glBindBuffer(GL_ARRAY_BUFFER, pPrimitive->getVertexBuffer());
        glVertexAttribPointer(tmpId, 3, GL_FLOAT, GL_FALSE, 0, 0);
    }
    if ((tmpId = glGetAttribLocation(progId, "a_normal")) != -1 &&
        0 != pPrimitive->getNormalBuffer())
    {
        glEnableVertexAttribArray(tmpId);
        glBindBuffer(GL_ARRAY_BUFFER, pPrimitive->getNormalBuffer());
        glVertexAttribPointer(tmpId, 3, GL_FLOAT, GL_FALSE, 0, 0);
    }
    if ((tmpId = glGetAttribLocation(progId, "a_texcoord0")) != -1 &&
        0 != pPrimitive->getTexCoordBuffer())
    {
        glEnableVertexAttribArray(tmpId);
        glBindBuffer(GL_ARRAY_BUFFER, pPrimitive->getTexCoordBuffer());
        glVertexAttribPointer(tmpId, 2, GL_FLOAT, GL_FALSE, 0, 0);
    }
    if ((tmpId = glGetAttribLocation(progId, "a_weight")) != -1 &&
        0 != pPrimitive->getWeightBuffer())
    {
        glEnableVertexAttribArray(tmpId);
        glBindBuffer(GL_ARRAY_BUFFER, pPrimitive->getWeightBuffer());
        glVertexAttribPointer(tmpId, 4, GL_FLOAT, GL_FALSE, 0, 0);
    }
    if ((tmpId = glGetAttribLocation(progId, "a_joint")) != -1 &&
        0 != pPrimitive->getJointBuffer())
    {
        glEnableVertexAttribArray(tmpId);
        glBindBuffer(GL_ARRAY_BUFFER, pPrimitive->getJointBuffer());
        glVertexAttribPointer(tmpId, 4, GL_FLOAT, GL_FALSE, 0, 0);
    }
}

void RenderScene::drawTriangle(RenderPrimitive* pPrimitive)
{
    if (pPrimitive->getIndicesCount() > 0)
    {
        glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, pPrimitive->getIndicesBuffer());
        glDrawElements(GL_TRIANGLES, static_cast<GLsizei>(pPrimitive->getIndicesCount()),
                       pPrimitive->getIndicesDataType(), 0);
    }
    else
    {
        glDrawArrays(GL_TRIANGLES, 0, static_cast<GLsizei>(pPrimitive->getVerterCount()));
    }
}

int RenderScene::prepareRender(glTFViewport* pViewport)
{
    if (0 != pFPSCounter)
    {
        pFPSCounter->timeStamp();
    }
    {
        int status = initSSAAFrameBuf(pViewport);
        if (LIBGLTF_SUCCESS != status)
            return status;
    }
    return 0;
}

int RenderScene::initSSAAFrameBuf(glTFViewport* pViewport)
{
    if (0 == pViewport->width)
        return LIBGLTF_INVALID_SIZE;

    // When viewport changes we need to release fbo and create a new one
    if( mCurrentViewport.x != pViewport->x || mCurrentViewport.y != pViewport->y ||
    mCurrentViewport.width != pViewport->width || mCurrentViewport.height != pViewport->height )
    {
        fbo.releaseFbo();
        mCurrentViewport = *pViewport;
    }

    int width  = SSAA * pViewport->width;
    int height = SSAA * pViewport->height;

    int status = fbo.createAndBindFbo(width, height, mIsUseMSAA);
    if (LIBGLTF_SUCCESS != status)
        return status;
    if (mIsUseMSAA)
    {
        glBindFramebuffer(GL_FRAMEBUFFER, fbo.getMSAAFboId());
    }
    else
    {
        glBindFramebuffer(GL_FRAMEBUFFER, fbo.getFboId());
    }
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    glViewport(pViewport->x, pViewport->y, width, height);
    return LIBGLTF_SUCCESS;
}

void RenderScene::render()
{
    setTimeForAnim();
    realRender();
}

void RenderScene::realRender()
{
    maCamera.updateViewMatrix();
    if (mCurrentTime >= mUpdateTimeOut || !bIsTimeAvailable)
    {
        if (mAnimationPlay)
        {
            Node* pRootNode = pScene->getRootNode();
            updateNodeMatrix(pRootNode, pRootNode->getGlobalMatrix(), false);
        }
        mUpdateTimeOut = mCurrentTime;
    }
    if (bFlyCamera)
    {
        updateFlyCamera();
    }

    for (unsigned int i = 0, size = mShaderVec.size();
         i < size; ++i)
    {
        renderShader(mShaderVec[i]);
    }
    {
        glActiveTexture(GL_TEXTURE0);
        glBindTexture(GL_TEXTURE_2D, 0);
        mCurrentImage = "";
        mCurrentTextNumber = -1;
    }
}

void RenderScene::renderShader(RenderShader* pRenderShader)
{
    Technique* pTechnique = pRenderShader->getTechnique();

    if (!pTechnique->useTechnique())
    {
        return;
    }

    unsigned int progId = pTechnique->getProgramId();

    {
        upLoadTechInfo(progId, pTechnique);
    }

    for (unsigned int i = 0, size = pRenderShader->getRenderPrimSize();
         i < size; ++i)
    {
        renderPrimitive(pRenderShader->getRenderPrim(i), progId);
    }
}

int RenderScene::completeRender()
{
    if (0 != pFPSCounter && mIsFPSCounterPrinted)
    {
        pFPSCounter->printFPS(&mCurrentViewport);
    }

    {
        glBindFramebuffer(GL_FRAMEBUFFER, 0);

        int width  = SSAA * mCurrentViewport.width;
        int height = SSAA * mCurrentViewport.height;

    if (mIsUseMSAA)
    {
        glBindFramebuffer(GL_READ_FRAMEBUFFER, fbo.getMSAAFboId());
        GLenum status = glCheckFramebufferStatus(GL_READ_FRAMEBUFFER);
        if (status != GL_FRAMEBUFFER_COMPLETE)
        {
            return LIBGLTF_BIND_FBO_ERROR;
        }
        glBindFramebuffer(GL_DRAW_FRAMEBUFFER, fbo.getFboId());
        status = glCheckFramebufferStatus(GL_DRAW_FRAMEBUFFER);
        if (status != GL_FRAMEBUFFER_COMPLETE)
        {
            return LIBGLTF_BIND_FBO_ERROR;
        }
        glBlitFramebuffer(0, 0 ,width, height, 0, 0,width ,height,
                          GL_COLOR_BUFFER_BIT, GL_LINEAR);
        glBindFramebuffer(GL_READ_FRAMEBUFFER,0);
        glBindFramebuffer(GL_DRAW_FRAMEBUFFER,0);
    }
        glViewport(mCurrentViewport.x, mCurrentViewport.y,
                   mCurrentViewport.width, mCurrentViewport.height);
        fbo.renderFbo(width, height);
    }
    return LIBGLTF_SUCCESS;
}

void RenderScene::releaseRender()
{
    fbo.releaseFbo();
    if (mIsUseMSAA)
    {
        fbo.releaseMSAAFBO();
    }
    return;
}

CPhysicalCamera& RenderScene::getCamera()
{
    return maCamera;
}

Node* RenderScene::findNodeByName(Node* pPareNode,
                                  const std::string& nodeId)
{
    if (0 == pPareNode)
        return 0;
    if (pPareNode->getNodeName() == nodeId)
        return pPareNode;

    Node* pNode = 0;
    for (unsigned int i = 0, size = pPareNode->getChildNodeSize();
         i < size; ++i)
    {
        pNode = findNodeByName(pPareNode->getChildNode(i), nodeId);
        if (0 != pNode)
            break;
    }
    return pNode;
}

Node* RenderScene::findNodeByJoint(Node* pPareNode,
                                   const std::string& jointId)
{
    if (0 == pPareNode || !pPareNode->getJointFlag())
        return 0;
    if (pPareNode->getJointId() == jointId)
        return pPareNode;

    Node* pNode = 0;
    for (unsigned int i = 0, size = pPareNode->getChildNodeSize();
         i < size; ++i)
    {
        pNode = findNodeByJoint(pPareNode->getChildNode(i), jointId);
        if (0 != pNode)
            break;
    }
    return pNode;
}

int RenderScene::prepareRenderBitmap(glTFViewport* pViewport)
{
    int status = initSSAAFrameBuf(pViewport);
    if (LIBGLTF_SUCCESS != status)
    {
        return status;
    }
    startAnimation();
    glEnable(GL_DEPTH_TEST);
    return LIBGLTF_SUCCESS;
}

void RenderScene::renderBitmap(double time)
{
    mCurrentTime = time;
    realRender();
}

void RenderScene::setBitZoom(unsigned char* Dstbuffer,
                             unsigned char* Srcbuffer,
                             glTFViewport* pViewport,
                             int bufferDepth)
{
    int width  = SSAA * pViewport->width;
    int height = SSAA * pViewport->height;

    int dstLineByte = pViewport->width * bufferDepth;
    int srcLineByte = width * bufferDepth;
    for (int i = 0; i < pViewport->height; i++)
    {
        float fy = (float)((i + 0.5) * SSAA - 0.5);
        int sy =(int)std::floor(fy);
        fy -= (float)sy;
        sy = min(sy, (int)(height - 2));
        sy = max(0, sy);
        short cbufy[2];
        cbufy[0] = (short)((1.0f - fy) * 2048);
        cbufy[1] = (short)(2048 - cbufy[0]);
        for (int j = 0; j < pViewport->width; j++)
        {
            float fx = (float)((j + 0.5) * SSAA - 0.5);
            int sx = (int)std::floor(fx);
            fx -= (float)sx;
            if (sx >= width - 1)
            {
                fx = 0;
                sx = width - 2;
            }
            short cbufx[2];
            cbufx[0] = (short)((1.0f - fx) * 2048);
            cbufx[1] = (short)(2048 - cbufx[0]);
            for (int k = 0; k < bufferDepth; k++)
            {
                *(Dstbuffer + i * dstLineByte + bufferDepth * j + k) = (
                    *(Srcbuffer + sy * srcLineByte + bufferDepth * sx + k)
                    * cbufx[0] * cbufy[0] +
                    *(Srcbuffer + (sy + 1) * srcLineByte + bufferDepth * sx + k)
                    * cbufx[0] * cbufy[1] +
                    *(Srcbuffer + sy * srcLineByte + bufferDepth * (sx + 1) + k)
                    * cbufx[1] * cbufy[0] +
                    *(Srcbuffer + (sy + 1) * srcLineByte + bufferDepth
                    *(sx + 1) + k) * cbufx[1] * cbufy[1]) >> 22;
            }
        }
    }
}


int RenderScene::completeRenderBitmap(glTFViewport* pViewport,
                                       unsigned char* buffer, GLenum format)
{
    int width  = SSAA * pViewport->width;
    int height = SSAA * pViewport->height;

    glBindFramebuffer(GL_FRAMEBUFFER, 0);

    if (mIsUseMSAA)
    {
        glBindFramebuffer(GL_READ_FRAMEBUFFER, fbo.getMSAAFboId());
        GLenum status = glCheckFramebufferStatus(GL_READ_FRAMEBUFFER);
        if (status != GL_FRAMEBUFFER_COMPLETE)
        {
            return LIBGLTF_BIND_FBO_ERROR;
        }
        glBindFramebuffer(GL_DRAW_FRAMEBUFFER, fbo.getFboId());
        status = glCheckFramebufferStatus(GL_DRAW_FRAMEBUFFER);
        if (status != GL_FRAMEBUFFER_COMPLETE)
        {
            return LIBGLTF_BIND_FBO_ERROR;
        }
        glBlitFramebuffer(0, 0 ,width, height, 0, 0,width ,height,
            GL_COLOR_BUFFER_BIT, GL_LINEAR);
        glBindFramebuffer(GL_READ_FRAMEBUFFER,0);
        glBindFramebuffer(GL_DRAW_FRAMEBUFFER,0);
    }

    glDisable(GL_DEPTH_TEST);
    glViewport(pViewport->x, pViewport->y, width, height);
    fbo.renderFbo(width, height);
    fbo.createBitmapTexture(width, height);
    fbo.inverseBitMap(width, height);
    if (GL_FRAMEBUFFER_COMPLETE != glCheckFramebufferStatus(GL_FRAMEBUFFER))
    {
        return LIBGLTF_BIND_FBO_ERROR;
    }
    int bufferDepth;
    if (format == GL_RGB || format == GL_BGR)
        bufferDepth = 3;
    else if (format == GL_RGBA || format == GL_BGRA)
        bufferDepth = 4;
    else
        return LIBGLTF_UNKNOWN_ERROR;
    int imageSize = width * height * bufferDepth;

    unsigned char *pbuffer = new unsigned char[imageSize];
    glReadPixels(0, 0, width, height, format,
                 GL_UNSIGNED_BYTE, pbuffer);
    setBitZoom(buffer, pbuffer, pViewport, bufferDepth);
    if (pbuffer)
    {
        delete[] pbuffer;
        pbuffer = 0;
    }
    fbo.releaseBitMapFBO();
    fbo.releaseBitmapTexture();
    return LIBGLTF_SUCCESS;
}

void RenderScene::getCameraPos(glm::vec3* Eye, glm::vec3* view,
                               glm::vec3* up)
{
    maCamera.getCameraPosVectors(Eye, view, up);
}

glm::vec3* RenderScene::getModelCenterPos()
{
    return &maCamera.vModelCenterPos;
}

void RenderScene::enableRotation()
{
    mEnableRotation = true;
}

void RenderScene::disableRotation()
{
    mEnableRotation = false;
}

bool RenderScene::isRotationEnabled() const
{
    return mEnableRotation;
}

void RenderScene::enableTransparency()
{
    mEnableTransparency = true;
}

void RenderScene::disableTransparency()
{
    mEnableTransparency = false;
}

double RenderScene::getModelSize() const
{
    return maCamera.flength;
}

void RenderScene::startAnimation()
{
    mAnimationPlay = true;
    mCurrentTime = 0;
    mUpdateTimeOut = 0;
    bAnimation = true;
}

void RenderScene::stopAnimation()
{
    if (pScene->getSkinSize() != 0)
    {
        mAnimationPlay = false;
        bAnimation = false;
    }
}

void RenderScene::stopAerialView()
{
    bAerialView = false;
    if (!pScene->getUseCameraInJson() && vCameraIndex.size() > 0)
    {
        pScene->setUseCameraInJson(true);
    }
    maCamera.setAerialView(false);
}

void RenderScene::startAerialView()
{
    bAerialView = true;
    if (pScene->getUseCameraInJson())
    {
        pScene->setUseCameraInJson(false);
    }
    maCamera.setAerialView(true);
    glm::vec3 vOrigEye;
    glm::vec3 vUp;
    maCamera.getCameraPosVectors(&vOrigEye, 0, &vUp);
    maCamera.setViewMatrix(glm::lookAt(vOrigEye, maCamera.vModelCenterPos, vUp));
}

bool RenderScene::isInAerialView() const
{
    return bAerialView;
}

void RenderScene::setAnimTime(double time)
{
    mCurrentTime = time;
    mUpdateTimeOut = time;
}

double RenderScene::getAnimTime() const
{
    errno = 0;
    double time = fmod(mCurrentTime, mDuration);
    return errno == EDOM ? 0.0 : time;
}

void RenderScene::setAnimLoop(bool loop)
{
    mAnimationLoop = loop;
}

bool RenderScene::getAnimLoop() const
{
    return mAnimationLoop;
}

double RenderScene::getAnimDuration() const
{
    return mDuration;
}

bool RenderScene::isAnimPlay() const
{
    return mAnimationPlay;
}

void RenderScene::resumeAnim()
{
    bAnimation = true;
    mAnimationPlay = true;
}

void RenderScene::setTimeForAnim()
{
    double timeNow = libgltf::time::getCurrentTime();
    if (mAnimationPlay)
    {
        if (bIsTimeAvailable)
        {
            mCurrentTime += libgltf::time::diffTime(timeNow,mLastPlaying);
        }
        else
        {
            bIsTimeAvailable = true;
        }
        mLastPlaying = timeNow;
    }
    else
    {
        bIsTimeAvailable = false;
    }
    if (!mAnimationLoop && mCurrentTime > mDuration)
    {
        stopAnimation();
        setAnimTime(0.0);
    }
}

bool RenderScene::renderFlyCamera(const glm::mat4& glPosViewMatrix, double time)
{
    if (bFlyCamera)
    {
        return false;
    }
    flyInfo = glPosViewMatrix - maCamera.getViewMatrix();
    if ( time < 0.0001 )
    {
        maCamera.setViewMatrix(glPosViewMatrix);
        return true;
    }
    bFlyCamera = true;
    flyTime = time * 1000 * 1000;
    flyInfo = flyInfo / (float)flyTime;
    return true;
}

void RenderScene::setViewMatrix(glm::mat4 viewMatrix)
{
    maCamera.setViewMatrix(viewMatrix);
}

void RenderScene::resetViewMatrix()
{
    if (bAerialView)
        maCamera.setViewMatrix(mOrbitInitViewMatrix);
    else
        maCamera.setViewMatrix(mWalkthroughInitViewMatrix);
}

const glm::mat4& RenderScene::getViewMatrix() const
{
    return maCamera.getViewMatrix();
}

void RenderScene::startPatrol()
{
    mViewMatrixBeforePatrol = maCamera.getViewMatrix();
    mIsCameraAnimationRunningBeforePatrol = mIsCameraAnimationRunning;
    mIsCameraAnimationRunning = false;
}

void RenderScene::endPatrol()
{
    if (pScene->getUseCameraInJson())
    {
        Node* pNode = cCamera->getCameraNode();
        glm::mat4 tmpView = glm::inverse(mViewMatrixBeforePatrol);
        pNode->setGlobalMatrix(tmpView);
        mIsCameraAnimationRunning = mIsCameraAnimationRunningBeforePatrol;
        mIsCameraAnimationRunningBeforePatrol = true;
    }
    else
    {
        maCamera.setViewMatrix(mWalkthroughInitViewMatrix);
    }
    flyTime = 0;
}

void RenderScene::enableMSAA()
{
    mIsUseMSAA = true;
}

void RenderScene::disableMSAA()
{
    mIsUseMSAA = false;
}
void RenderScene::setModelBoundaryValue()
{
    unsigned int shaderSize = mShaderVec.size();
    glm::vec3 vertexMax = glm::vec3(-FLT_MAX,-FLT_MAX,-FLT_MAX);
    glm::vec3 vertexMin = glm::vec3(FLT_MAX,FLT_MAX,FLT_MAX);
    for (unsigned int i = 0;i < shaderSize; ++i)
    {
        RenderShader* pRenderShader = mShaderVec[i];
        unsigned int primitiveSize = pRenderShader->getRenderPrimSize();
        for (unsigned int j = 0;j < primitiveSize; ++j)
        {
            RenderPrimitive* pPrimitive = pRenderShader->getRenderPrim(j);
            if (NULL != pPrimitive)
            {
                pPrimitive->getPrimitiveBoundary(&vertexMax,&vertexMin);
            }
        }
    }
    pScene->setVertexMax(vertexMax[0],vertexMax[1],vertexMax[2]);
    pScene->setVertexMin(vertexMin[0],vertexMin[1],vertexMin[2]);
}
} // namespace libgltf

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
