/************************************************************************/
/*                                                                      */
/*    vspline - a set of generic tools for creation and evaluation      */
/*              of uniform b-splines                                    */
/*                                                                      */
/*            Copyright 2015, 2016 by Kay F. Jahnke                     */
/*                                                                      */
/*    Permission is hereby granted, free of charge, to any person       */
/*    obtaining a copy of this software and associated documentation    */
/*    files (the "Software"), to deal in the Software without           */
/*    restriction, including without limitation the rights to use,      */
/*    copy, modify, merge, publish, distribute, sublicense, and/or      */
/*    sell copies of the Software, and to permit persons to whom the    */
/*    Software is furnished to do so, subject to the following          */
/*    conditions:                                                       */
/*                                                                      */
/*    The above copyright notice and this permission notice shall be    */
/*    included in all copies or substantial portions of the             */
/*    Software.                                                         */
/*                                                                      */
/*    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND    */
/*    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES   */
/*    OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND          */
/*    NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT       */
/*    HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,      */
/*    WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING      */
/*    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR     */
/*    OTHER DEALINGS IN THE SOFTWARE.                                   */
/*                                                                      */
/************************************************************************/

/// \file eval.cc
///
/// \brief simple demonstration of creation and evaluation of a b-spline
///
/// takes a set of knot point values from cin, calculates a 1D b-spline
/// over them, and evaluates it at coordinates taken from cin.
/// The output shows how the coordinate is split into integral and real
/// part and the result of evaluating the spline at this point.
/// Note how the coordinate is not checked for validity. when accessing
/// the spline outside the defined range, the coefficient array will
/// receive an out-of-bounds access, which may result in a seg fault.
/// eval warns you when receiving invalid arguments, but proceeds regardless.
/// To see how out-of-range incoming coordinates can be handled to avoid
/// out-of-bounds access to the coefficients, see use_map.cc
///
/// compile: clang++ -std=c++11 -o eval -pthread eval.cc

#include <vspline/vspline.h>
#include <iomanip>

using namespace std ;
using namespace vigra ;
using namespace vspline ;

int main ( int argc , char * argv[] )
{
  // get the spline degree and boundary conditions from the console

  cout << "enter spline degree: " ;
  int spline_degree ;
  cin >> spline_degree ;
  
  int bci = -1 ;
  bc_code bc ;
  
  while ( bci < 1 || bci > 4 )
  {
    cout << "choose boundary condition" << endl ;
    cout << "1) MIRROR" << endl ;
    cout << "2) PERIODIC" << endl ;
    cout << "3) REFLECT" << endl ;
    cout << "4) NATURAL" << endl ;
    cin >> bci ;
  }
  
  switch ( bci )
  {
    case 1 :
      bc = MIRROR ;
      break ;
    case 2 :
      bc = PERIODIC ;
      break ;
    case 3 :
      bc = REFLECT ;
      break ;
    case 4 :
      bc = NATURAL ;
      break ;
  }
  
  // put the BC code into a TinyVector
  
  TinyVector < bc_code , 1 > bcv ( bc ) ;

  TinyVector < int , 1 > deriv_spec ( 0 ) ;
  
  // obtain knot point values

  double v ;
  std::vector<double> dv ;
  cout << "enter knot point values (end with EOF)" << endl ;
  while ( cin >> v )
    dv.push_back ( v ) ;

  cin.clear() ;
  
  // put the size into a TinyVector
  
  TinyVector < int , 1 > shape ( dv.size() ) ;
  
  // fix the type for the bspline object
  
  typedef bspline < double , 1 > spline_type ;
  spline_type bsp ( shape , spline_degree , bcv ) ; // , EXPLICIT ) ;
  cout << "created bspline object:" << endl << bsp << endl ;

  // fill the data into the spline's 'core' area
  
  for ( size_t i = 0 ; i < dv.size() ; i++ )
    bsp.core[i] = dv[i] ;

  // prefilter the data
  
  bsp.prefilter() ;
  
  cout << fixed << showpoint << setprecision(12) ;
  cout << "spline coefficients (with frame)" << endl ;
  for ( auto& coeff : bsp.container )
    cout << " " << coeff << endl ;

  auto ev = vspline::make_safe_evaluator
            < spline_type , double > ( bsp ) ;

  int ic ;
  double rc ;
  double res ;

  cout << "enter coordinates to evaluate (end with EOF)" << endl ;
  while ( ! cin.eof() )
  {
    // get a coordinate
    
    cin >> v ;
    if ( v < bsp.lower_limit(0) || v > bsp.upper_limit(0) )
    {
      std::cout << "warning: " << v
                << " is outside the spline's defined range."
                << std::endl ;
      std::cout << "using automatic folding to process this value."
                << std::endl ;
    }

    // evaluate the spline at this location

    ev.eval ( v , res ) ;

    cout << v << " -> " << res << endl ;

#ifdef USE_VC
    
    auto vev = vspline::make_safe_evaluator
               < spline_type , double , 4 > ( bsp ) ;

    typedef typename decltype(vev)::in_ele_v vtype ;
    vtype vv(v) ;
    vtype vres ;
    vev.eval ( vv , vres ) ;
    
    cout << vv << " -> " << vres << endl ;

#endif
  }
}
