/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/*
 * This file is part of the libe-book project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#include "FB2BlockContext.h"
#include "FB2Collector.h"
#include "FB2MetadataContext.h"
#include "FB2TableContext.h"
#include "FB2TextContext.h"
#include "FB2Token.h"

namespace libebook
{

FB2BodyContext::FB2BodyContext(FB2ParserContext *const parentContext, FB2Collector *const collector, const bool collectHeadings)
  : FB2NodeContextBase(parentContext, collector)
  , m_collectHeadings(collectHeadings)
  , m_lang()
{
}

FB2XMLParserContext *FB2BodyContext::element(const FB2TokenData &name, const FB2TokenData &ns)
{
  if (FB2Token::NS_FICTIONBOOK == getFB2TokenID(ns))
  {
    switch (getFB2TokenID(name))
    {
    case FB2Token::image :
      return new FB2ImageContext(this);
    case FB2Token::title :
    {
      FB2BlockFormat format;
      if (m_collectHeadings)
        format.headingLevel = 1;
      return new FB2TitleContext(this, format);
    }
    case FB2Token::epigraph :
      return new FB2EpigraphContext(this, FB2BlockFormat());
    case FB2Token::section :
      if (m_collectHeadings)
        return new FB2SectionContext(this, 1, m_lang);
      else
        return new FB2SectionContext(this, m_lang);
    default :
      break;
    }
  }

  return new FB2SkipElementContext(this);
}

void FB2BodyContext::startOfElement()
{
  getCollector()->openPageSpan();
}

void FB2BodyContext::endOfElement()
{
  getCollector()->closePageSpan();
}

void FB2BodyContext::attribute(const FB2TokenData &name, const FB2TokenData *ns, const char *const value)
{
  if ((FB2Token::NS_XML == getFB2TokenID(ns)) && (FB2Token::lang == getFB2TokenID(name)))
    m_lang = value;
}

FB2CiteContext::FB2CiteContext(FB2ParserContext *parentContext, const FB2BlockFormat &format)
  : FB2BlockFormatContextBase(parentContext, format)
{
}

FB2XMLParserContext *FB2CiteContext::element(const FB2TokenData &name, const FB2TokenData &ns)
{
  if (FB2Token::NS_FICTIONBOOK == getFB2TokenID(ns))
  {
    switch (getFB2TokenID(name))
    {
    case FB2Token::p :
      return new FB2PContext(this, getBlockFormat());
    case FB2Token::poem :
      return new FB2PoemContext(this, getBlockFormat());
    case FB2Token::empty_line :
      return new FB2EmptyLineContext(this);
    case FB2Token::subtitle :
      return new FB2SubtitleContext(this, getBlockFormat());
    case FB2Token::table :
      return new FB2TableContext(this, getBlockFormat());
    default :
      break;
    }
  }

  return new FB2SkipElementContext(this);
}

void FB2CiteContext::endOfElement()
{
}

void FB2CiteContext::attribute(const FB2TokenData &name, const FB2TokenData *ns, const char *value)
{
  if ((FB2_NO_NAMESPACE(ns)) && (FB2Token::id == getFB2TokenID(name)))
    getCollector()->defineID(value);
  else if ((FB2Token::NS_XML == getFB2TokenID(ns)) && (FB2Token::lang == getFB2TokenID(name)))
    setLang(value);
}

FB2EmptyLineContext::FB2EmptyLineContext(FB2ParserContext *parentContext)
  : FB2DataContextBase(parentContext)
{
}

void FB2EmptyLineContext::endOfElement()
{
  getCollector()->openParagraph(FB2BlockFormat());
  getCollector()->closeParagraph();
}

void FB2EmptyLineContext::attribute(const FB2TokenData &, const FB2TokenData *, const char *)
{
}

FB2EpigraphContext::FB2EpigraphContext(FB2ParserContext *const parentContext, const FB2BlockFormat &format)
  : FB2BlockFormatContextBase(parentContext, format)
{
}

FB2XMLParserContext *FB2EpigraphContext::element(const FB2TokenData &name, const FB2TokenData &ns)
{
  if (FB2Token::NS_FICTIONBOOK == getFB2TokenID(ns))
  {
    switch (getFB2TokenID(name))
    {
    case FB2Token::p :
      return new FB2PContext(this, getBlockFormat());
    case FB2Token::poem :
      return new FB2PoemContext(this, getBlockFormat());
    case FB2Token::cite :
      return new FB2CiteContext(this, getBlockFormat());
    case FB2Token::empty_line :
      return new FB2EmptyLineContext(this);
    case FB2Token::text_author :
      return new FB2TextAuthorContext(this, getBlockFormat());
    default :
      break;
    }
  }

  return new FB2SkipElementContext(this);
}

void FB2EpigraphContext::endOfElement()
{
}

void FB2EpigraphContext::attribute(const FB2TokenData &name, const FB2TokenData *ns, const char *value)
{
  if ((FB2_NO_NAMESPACE(ns)) && (FB2Token::id == getFB2TokenID(name)))
    getCollector()->defineID(value);
}

FB2ImageContext::FB2ImageContext(FB2ParserContext *parentContext)
  : FB2NodeContextBase(parentContext)
  , m_href()
  , m_altText()
  , m_valid(true)
{
}

FB2XMLParserContext *FB2ImageContext::element(const FB2TokenData &name, const FB2TokenData &ns)
{
  // TODO: implement me
  if (FB2Token::NS_FICTIONBOOK == getFB2TokenID(ns))
  {
    switch (getFB2TokenID(name))
    {
    default :
      break;
    }
  }

  return new FB2SkipElementContext(this);
}

void FB2ImageContext::endOfElement()
{
  if (m_valid && ('#' != m_href[0]))
    m_valid = false;

  if (m_valid)
  {
    try
    {
      getCollector()->insertBitmap(m_href.substr(1).c_str());
    }
    catch (...)
    {
      m_valid = false;
    }
  }

  if (!m_valid)
  {
    // The image is not internal or it is broken. Use alternative text.
    // TODO: maybe extend later to support external URLs?
    getCollector()->openParagraph(FB2BlockFormat());
    getCollector()->openSpan(FB2Style(FB2BlockFormat()));
    const std::string altText = std::string("[Image") + (m_altText.empty() ? "" : ": ") + m_altText + "]";
    getCollector()->insertText(altText.c_str());
    getCollector()->closeSpan();
    getCollector()->closeParagraph();
  }
}

void FB2ImageContext::attribute(const FB2TokenData &name, const FB2TokenData *ns, const char *value)
{
  if (FB2_NO_NAMESPACE(ns))
  {
    switch (getFB2TokenID(name))
    {
    case FB2Token::alt :
      m_altText = value;
      break;
    case FB2Token::id :
      getCollector()->defineID(value);
      break;
    case FB2Token::title :
      // ignore
      break;
    default :
      break;
    }
  }
  else if (FB2Token::NS_XLINK == getFB2TokenID(ns))
  {
    switch (getFB2TokenID(name))
    {
    case FB2Token::href :
      m_href = value;
      break;
    case FB2Token::type :
      m_valid = FB2Token::simple == getFB2TokenID(value);
      break;
    default :
      break;
    }
  }
}

FB2PoemContext::FB2PoemContext(FB2ParserContext *parentContext, const FB2BlockFormat &format)
  : FB2BlockFormatContextBase(parentContext, format)
{
}

FB2XMLParserContext *FB2PoemContext::element(const FB2TokenData &name, const FB2TokenData &ns)
{
  if (FB2Token::NS_FICTIONBOOK == getFB2TokenID(ns))
  {
    switch (getFB2TokenID(name))
    {
    case FB2Token::title :
      return new FB2TitleContext(this, getBlockFormat());
    case FB2Token::epigraph :
      return new FB2EpigraphContext(this, getBlockFormat());
    case FB2Token::stanza :
      return new FB2StanzaContext(this, getBlockFormat());
    case FB2Token::text_author :
      return new FB2TextAuthorContext(this, getBlockFormat());
    case FB2Token::date :
      // ignore
      break;
    default :
      break;
    }
  }

  return new FB2SkipElementContext(this);
}

void FB2PoemContext::startOfElement()
{
  getCollector()->openParagraph(FB2BlockFormat());
  getCollector()->closeParagraph();
}

void FB2PoemContext::endOfElement()
{
}

void FB2PoemContext::attribute(const FB2TokenData &name, const FB2TokenData *ns, const char *value)
{
  if ((FB2_NO_NAMESPACE(ns)) && (FB2Token::id == getFB2TokenID(name)))
    getCollector()->defineID(value);
  else if ((FB2Token::NS_XML == getFB2TokenID(ns)) && (FB2Token::lang == getFB2TokenID(name)))
    setLang(value);
}

FB2SectionContext::FB2SectionContext(FB2ParserContext *const parentContext, const boost::optional<std::string> &lang)
  : FB2NodeContextBase(parentContext)
  , m_level()
  , m_lang(lang)
{
}

FB2SectionContext::FB2SectionContext(FB2ParserContext *const parentContext, const uint8_t level, const boost::optional<std::string> &lang)
  : FB2NodeContextBase(parentContext)
  , m_level(level)
  , m_lang(lang)
{
}

FB2XMLParserContext *FB2SectionContext::element(const FB2TokenData &name, const FB2TokenData &ns)
{
  if (FB2Token::NS_FICTIONBOOK == getFB2TokenID(ns))
  {
    switch (getFB2TokenID(name))
    {
    case FB2Token::title :
    {
      FB2BlockFormat format;
      if (bool(m_level))
        format.headingLevel = get(m_level) + 1;
      if (bool(m_lang))
        format.lang = get(m_lang);
      return new FB2TitleContext(this, format);
    }
    case FB2Token::epigraph :
      return new FB2EpigraphContext(this, makeBlockFormat());
    case FB2Token::image :
      return new FB2ImageContext(this);
    case FB2Token::annotation :
      return new FB2AnnotationContext(this, makeBlockFormat());
    case FB2Token::section :
      if (bool(m_level))
        return new FB2SectionContext(this, get(m_level) + 1, m_lang);
      else
        return new FB2SectionContext(this, m_lang);
    case FB2Token::p :
      return new FB2PContext(this, makeBlockFormat());
    case FB2Token::poem :
      return new FB2PoemContext(this, makeBlockFormat());
    case FB2Token::subtitle :
      return new FB2SubtitleContext(this, makeBlockFormat());
    case FB2Token::cite :
      return new FB2CiteContext(this, makeBlockFormat());
    case FB2Token::empty_line :
      return new FB2EmptyLineContext(this);
    case FB2Token::table :
      return new FB2TableContext(this, makeBlockFormat());
    default :
      break;
    }
  }

  return new FB2SkipElementContext(this);
}

void FB2SectionContext::endOfElement()
{
  // TODO: implement me
}

void FB2SectionContext::attribute(const FB2TokenData &name, const FB2TokenData *ns, const char *value)
{
  if (FB2_NO_NAMESPACE(ns) && (FB2Token::id == getFB2TokenID(name)))
    getCollector()->defineID(value);
  else if ((FB2Token::NS_XML == getFB2TokenID(ns)) && (FB2Token::lang == getFB2TokenID(name)))
    m_lang = value;
}

FB2BlockFormat FB2SectionContext::makeBlockFormat() const
{
  FB2BlockFormat format;

  if (bool(m_lang))
    format.lang = get(m_lang);

  return format;
}

FB2StanzaContext::FB2StanzaContext(FB2ParserContext *parentContext, const FB2BlockFormat &format)
  : FB2BlockFormatContextBase(parentContext, format)
{
  getBlockFormat().stanza = true;
}

FB2XMLParserContext *FB2StanzaContext::element(const FB2TokenData &name, const FB2TokenData &ns)
{
  if (FB2Token::NS_FICTIONBOOK == getFB2TokenID(ns))
  {
    switch (getFB2TokenID(name))
    {
    case FB2Token::title :
      return new FB2TitleContext(this, getBlockFormat());
    case FB2Token::subtitle :
      return new FB2SubtitleContext(this, getBlockFormat());
    case FB2Token::v :
      return new FB2VContext(this, getBlockFormat());
    default :
      break;
    }
  }

  return new FB2SkipElementContext(this);
}

void FB2StanzaContext::endOfElement()
{
  getCollector()->openParagraph(FB2BlockFormat());
  getCollector()->closeParagraph();
}

void FB2StanzaContext::attribute(const FB2TokenData &name, const FB2TokenData *ns, const char *const value)
{
  if ((FB2Token::NS_XML == getFB2TokenID(ns)) && (FB2Token::lang == getFB2TokenID(name)))
    setLang(value);
}

FB2TitleContext::FB2TitleContext(FB2ParserContext *const parentContext, const FB2BlockFormat &format)
  : FB2BlockFormatContextBase(parentContext, format)
{
  getBlockFormat().title = true;
}

FB2XMLParserContext *FB2TitleContext::element(const FB2TokenData &name, const FB2TokenData &ns)
{
  if (FB2Token::NS_FICTIONBOOK == getFB2TokenID(ns))
  {
    switch (getFB2TokenID(name))
    {
    case FB2Token::p :
      return new FB2PContext(this, getBlockFormat());
    case FB2Token::empty_line :
      return new FB2EmptyLineContext(this);
    default :
      break;
    }
  }

  return new FB2SkipElementContext(this);
}

void FB2TitleContext::endOfElement()
{
  // TODO: implement me
}

void FB2TitleContext::attribute(const FB2TokenData &name, const FB2TokenData *ns, const char *value)
{
  if (FB2_NO_NAMESPACE(ns) && (FB2Token::id == getFB2TokenID(name)))
    getCollector()->defineID(value);
  else if ((FB2Token::NS_XML == getFB2TokenID(ns)) && (FB2Token::lang == getFB2TokenID(name)))
    setLang(value);
}

}

/* vim:set shiftwidth=2 softtabstop=2 expandtab: */
