/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the libgltf project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#include "Texture.h"

namespace libgltf
{

Texture::Texture()
    : mTexture(0)
{
}

Texture::~Texture()
{
    deleteTexture();
}

void Texture::createFromData(unsigned char* bData, int a_iWidth, int a_iHeight,
                             GLenum format)
{

    glGenTextures(1, &mTexture);
    glBindTexture(GL_TEXTURE_2D, mTexture);

    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);

    if (format == GL_RGBA || format == GL_BGRA)
    {
        glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, a_iWidth, a_iHeight, 0, format,
                     GL_UNSIGNED_BYTE, bData);
    }

    else if (format == GL_RGB || format == GL_BGR)
    {
        glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, a_iWidth, a_iHeight, 0, format,
                     GL_UNSIGNED_BYTE, bData);
    }
    else if (format == GL_DEPTH_COMPONENT)
    {
        glTexImage2D(GL_TEXTURE_2D, 0, GL_DEPTH_COMPONENT, a_iWidth, a_iHeight, 0, format,
                     GL_UNSIGNED_BYTE, bData);
    }
}

bool Texture::loadTexture2DFromBuffer(unsigned char* buffer, int width,
                                      int height)
{
    GLenum format = GL_RGBA;

    createFromData(buffer, width, height, format);

    return true;
}

void Texture::setTextureParameter(GLenum parameter, GLint value)
{
    glTexParameteri(GL_TEXTURE_2D, parameter, value);
}

void Texture::setFiltering(int a_tfMagnification, int a_tfMinification)
{
    glBindTexture(GL_TEXTURE_2D, mTexture);

    if (a_tfMagnification == TEXTURE_FILTER_MAG_NEAREST)
    {
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    }
    else if (a_tfMagnification == TEXTURE_FILTER_MAG_BILINEAR)
    {
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    }

    if (a_tfMinification == TEXTURE_FILTER_MIN_NEAREST)
    {
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    }
    else if (a_tfMinification == TEXTURE_FILTER_MIN_BILINEAR)
    {
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    }
    else if (a_tfMinification == TEXTURE_FILTER_MIN_NEAREST_MIPMAP)
    {
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER,
                            GL_NEAREST_MIPMAP_NEAREST);
    }
    else if (a_tfMinification == TEXTURE_FILTER_MIN_BILINEAR_MIPMAP)
    {
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER,
                            GL_LINEAR_MIPMAP_NEAREST);
    }
    else if (a_tfMinification == TEXTURE_FILTER_MIN_TRILINEAR)
    {
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER,
                            GL_LINEAR_MIPMAP_LINEAR);
    }
}

void Texture::bindTexture(int iTextureUnit)
{
    glActiveTexture(GL_TEXTURE0 + iTextureUnit);
    glBindTexture(GL_TEXTURE_2D, mTexture);
}

void Texture::deleteTexture()
{
    glDeleteTextures(1, &mTexture);
}

} // namespace libgltf

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
