<?php
/**
 * @author EOLE
 * @date Novembre 2011
 * @file importLDAP.content.php
 * @version 1.0
 * @copyright GNU Public License, see the LICENSE.txt file
 * @brief script d'import OpenLDAP
 * Script destiné à importer / mettre à jour les données du cdc à partir de l'annuaire ldap
 * - Ajout ou mise à jour des utilisateurs (table user), en cas de mise à jour, le champ enabled n'est pas modifié
 * - Ajout ou mise à jour des enregistrements de la table member_of_level, la mise à jour porte sur le champ manager
 * - Création si besoin des enregistrements dans les tables level, discipline, teacher, responsible, school_report
 */

/* Commande sql de vidage de la base pour test
    delete from responsible;
    delete from member_of_level;
    delete from level;
    delete from teacher;
    delete from discipline;
    delete from school_report;
    delete from user
 */
/* Créer le compte admin
INSERT INTO user(login,password,enabled,admin) VALUES ("admin","ce272597bd050f938eefe3323edcc370",1,1)
 */
/* Commande pour mettre tous les password à "o"
UPDATE user SET password = 'ce272597bd050f938eefe3323edcc370'
 */
echo "<h2>Import LDAP</h2>";
if (!isset($_POST["submitted"])) {
    // Formulaire pour valider / confirmer les variables de connexion au ldap
    // @FIXME Vérifier l'attribut action du formulaire auto-appelant, possibilité d'utiliser $_SERVER["REQUEST_URI"] ou #
?>
    <script type="text/javascript">
    $().ready(function() {
        // Interception de la validation du formulaire pour afficher l'image d'attente
        $("#form_import").submit( function() {
            $("#form_p input").hide();
            $("#form_p img").show();
            return true;
        } );
    } );
    </script>
   <h3>Vous êtes sur le point d'importer les données d'un serveur EOLE Scribe vers CDC.</h3>
    <form id="form_import" method="POST" action="#">
      <p id="form_p">
        <img src='./images/ajax-loader.gif' style="display: none;" />
        <input type="submit" name="submitted" value="Lancer l'import LDAP">
      </p>
    </form>
<?php
} else {
    // Lancement du script d'import
    $compteur_level = 0;
    $compteur_discipline = 0;

    $ldc = ldap_connect(__LDAP_HOST, __LDAP_PORT) or die("Impossible de se connecter au serveur LDAP");
    ldap_set_option($ldc, LDAP_OPT_PROTOCOL_VERSION, 3);
    ldap_bind($ldc, __LDAP_USER, __LDAP_PASS) or die("LDAP : Authentification échouée");

    $champs = array("uid", "typeadmin", "sn", "givenname", "mail", "datenaissance", "divcod", "eleve");
    $login_ldap=array(); // Tableau des login ldap qui servira à vérifier s'il n'y a pas eu de suppression de user et si oui on les désactive en base (user.enabled=false)

    // Recherche des élèves
    $compteur_ajout = 0;
    $compteur_maj = 0;
    $lds = ldap_search($ldc, __LDAP_ELEVE_BRANCHE.",".__LDAP_BASE_DN, "sn=*", $champs);
    $ldge = ldap_get_entries ($ldc, $lds);

    for ($n=0 ; $n < $ldge["count"] ; $n++){
        //AjouterEleve($login, $admin, $profil, $mail, $nom, $prenom, $datenaissance, $classe)
        AjouterEleve($ldge[$n]["uid"][0], $ldge[$n]["mail"][0], $ldge[$n]["sn"][0], $ldge[$n]["givenname"][0], $ldge[$n]["datenaissance"][0], $ldge[$n]["divcod"][0]);
        array_push($login_ldap, $ldge[$n]["uid"][0]);
    }
    echo "<h4>Import des élèves... OK : $compteur_ajout ajout(s) et $compteur_maj mise(s) à jour.</h4>";

    // Recherche des responsables
    $compteur_ajout = 0;
    $compteur_maj = 0;
    $lds = ldap_search($ldc, __LDAP_RESP_BRANCHE.",".__LDAP_BASE_DN, "sn=*", $champs);
    $ldge = ldap_get_entries ($ldc, $lds);

    for ($n=0 ; $n < $ldge["count"] ; $n++){
        //AjouterResponsable($login, $mail, $nom, $prenom, $datenaissance, $eleves)
        AjouterResponsable($ldge[$n]["uid"][0], $ldge[$n]["mail"][0], $ldge[$n]["sn"][0], $ldge[$n]["givenname"][0], $ldge[$n]["datenaissance"][0], $ldge[$n]["eleve"]);
        array_push($login_ldap, $ldge[$n]["uid"][0]);
    }
    echo "<h4>Import des responsables... OK : $compteur_ajout ajout(s) et $compteur_maj mise(s) à jour.</h4>";

    // Recherche des personnels
    $compteur_ajout = 0;
    $compteur_maj = 0;
    $lds = ldap_search($ldc, __LDAP_PERS_BRANCHE.",".__LDAP_BASE_DN, "sn=*", $champs);
    $ldge = ldap_get_entries ($ldc, $lds);

    for ($n=0 ; $n < $ldge["count"] ; $n++){
        $uid = $ldge[$n]["uid"][0];
        array_push($login_ldap, $uid);

        // Recherche des matieres du prof
        $matieres = array();
        $lds_mat = ldap_search($ldc, __LDAP_GROUP_BRANCHE.",".__LDAP_BASE_DN, "(&(type=Matiere)(memberuid=".$uid."))", array("cn"));
        $ldge_mat = ldap_get_entries ($ldc, $lds_mat);
        for ($n_mat=0 ; $n_mat < $ldge_mat["count"] ; $n_mat++){
            array_push($matieres, $ldge_mat[$n_mat]["cn"][0]);
        }

        // Recherche des équipes pédagogiques (=classes) du prof
        $classes = array();
        $lds_cla = ldap_search($ldc, __LDAP_GROUP_BRANCHE.",".__LDAP_BASE_DN, "(&(type=Equipe)(memberuid=".$uid."))", array("cn"));
        $ldge_cla = ldap_get_entries ($ldc, $lds_cla);
        for ($n_cla=0 ; $n_cla < $ldge_cla["count"] ; $n_cla++){
            array_push($classes, $ldge_cla[$n_cla]["cn"][0]);
        }

        //AjouterPersonnel($login, $admin, $mail, $nom, $prenom, $datenaissance, $classegere, $matieres, $classes)
        AjouterPersonnel($ldge[$n]["uid"][0], $ldge[$n]["typeadmin"][0], $ldge[$n]["mail"][0], $ldge[$n]["sn"][0], $ldge[$n]["givenname"][0], $ldge[$n]["datenaissance"][0], $ldge[$n]["divcod"], $matieres, $classes);
    }
    echo "<h4>Import des personnels... OK : $compteur_ajout ajout(s) et $compteur_maj mise(s) à jour.</h4>";
    echo "<h4>Import des classes... OK : $compteur_level nouvelle(s).</h4>";
    echo "<h4>Import des matières... OK : $compteur_discipline nouvelle(s).</h4>";

    ldap_close ($ldc);

    $compteur_desactivation = 0;
    DesactiverUser($login_ldap);
    echo "<h4>Désactivation des comptes supprimés... OK : $compteur_desactivation désactivation(s).</h4>";
    echo "<h4>FIN</h4>";
}
// ======== Fin du script global ===========================================

/**
 * @brief Ajouter un personnel
 * @param $login login
 * @param $admin booléen déterminant si la personne est admin ou non
 * @param $mail mail
 * @param $nom nom
 * @param $prenom prénom
 * @param $datenaissance date de naissance
 * @param $classegere tableau de classe dont ce personnel est prof principal
 * @param $matieres tableau des matieres enseignées par le prof
 * @param $classes tableau des classes où le prof enseigne
 */
function AjouterPersonnel($login, $admin, $mail, $nom, $prenom, $datenaissance, $classegere, $matieres, $classes) {
    $enabled = __ENABLED_DEFAULT;
    // Calcul du profil
    switch ($admin) {
    case "0":
        $profil = 30; // valeur pour un prof
        break;
    case "1":
        $profil = 40; // valeur pour l'admin
        break;
    case "2":
        $profil = 40; // valeur pour un prof principal
        break;
    case "3":
        $profil = 40; // valeur pour un personnel administratif
        break;
    default :
        $profil = 10; // valeur la plus basse
        break;
    }
    $admin=($login=="admin")?1:0; // seul le compte admin est admin (!)

    if (is_array($classegere)){
        array_shift($classegere); // pour virer le 1er élément qui est un count
    }

    $id_user = AddOrUpdateUser($login,$nom,$prenom,$mail,$datenaissance,$profil,$enabled,$admin);

    $prefixe = "profs-";                                    // Pour les profs, les classes ou "équipes pédagogiques" sont préfixées par "profs-"
    foreach ($classes as $equipe_pedago) {
        $classe = str_replace($prefixe, "", $equipe_pedago);// on vire le préfixe
        if ($classe != $equipe_pedago) {
            $id_level = AddLevel($classe);                  // Ajout de la classe
            if (($id_user != -1) && ($id_level != -1)) {
                if ((is_array($classegere)) && (in_array($classe, $classegere))) {
                    // Ici le prof est prof principal de cette classe
                    AddOrUpdateMemberOfLevel($id_user, $id_level, true);// Ajout de la correpsondance prof /classe
                } else {
                    AddOrUpdateMemberOfLevel($id_user, $id_level);      // Ajout de la correpsondance prof /classe
                }
            }
        }
    }

    foreach ($matieres as $matiere) {
        $id_discipline = AddDiscipline($matiere);       // Ajout de la matiere
        if (($id_user != -1) && ($id_discipline != -1)) {
            AddTeacher($id_discipline, $id_user);       // Ajout de la relation prof / matiere
        }
    }
}

/**
 * @brief Ajouter un élève
 * @param $login login
 * @param $mail mail
 * @param $nom nom
 * @param $prenom prénom
 * @param $datenaissance date de naissance
 * @param $classe classe de l'élève
 */
function AjouterEleve($login, $mail, $nom, $prenom, $datenaissance, $classe) {
    $profil = 10; // valeur pour tous les élèves
    $admin = False;
    $enabled = __ENABLED_DEFAULT;

    $id_user = AddOrUpdateUser($login,$nom,$prenom,$mail,$datenaissance,$profil,$enabled,$admin); // Ajout / mise à jour de l'élève
    $id_level = AddLevel($classe);                 // Ajout de la classe
    if ($id_user != -1) {
        AddSchoolReport($id_user);                 // Ajout du carnet
        if ($id_level != -1) {
            AddOrUpdateMemberOfLevel($id_user, $id_level); // Ajout de la correpsondance élève /classe
        }
    }
}

/**
 * @brief Ajouter un responsable
 * @param $login login
 * @param $mail mail
 * @param $nom nom
 * @param $prenom prénom
 * @param $eleves tableau des élèves dont ce responsable a la charge
 */
function AjouterResponsable($login, $mail, $nom, $prenom, $datenaissance, $eleves) {
    $profil = 20 ; // valeur pour tous les responsables
    $admin = False;
    $enabled = __ENABLED_DEFAULT;

    $id_user = AddOrUpdateUser($login,$nom,$prenom,$mail,$datenaissance,$profil,$enabled,$admin); // Ajout / mise à jour du responsable
    if (($id_user != -1) && (is_array($eleves))){
        array_shift($eleves);                           // pour virer le 1er élément qui est un count
        foreach ($eleves as $login_eleve) {             // parcours de tous les élèves dont ce user est responsable
            $id_student = GetIdUser($login_eleve);      // récupération de l'id_user de l'élève
            if ($id_student != -1) {
                AddResponsible($id_student, $id_user);   // Ajout de la relation dans la table responsible
            }
        }
    }
}

/**
 * @brief Désactiver les user non présent dans le ldap
 * @param $login_ldap tableau de tous les logins du serveur ldap
 * Parcoure tous les user en base et vérifie s'ils sont présent dans le tableau $login_ldap
 * S'ils sont absent alors ils sont désactivés en base (user.enabled=false)
 */
function DesactiverUser($login_ldap) {
    $db = Connection::getInstance();
    if(!$db) {
        throw new MyException (ERR_CONNECT_DB);
    }
    // récupération des login
    $select = "SELECT login,enabled FROM user";
    $stmt_select = $db->prepare($select);
    if ($stmt_select === null) {
        throw new MyException ('DesactiverUser: erreur de préparation à la requête');
    }
    if ($stmt_select->execute() === false) {
        $stmt_select->errorInfo();
        if (DEBUG==True)
            Logger::log('DesactiverUser: echec du select');
        return -1;
    }
    $result = $stmt_select->fetchAll();

    foreach ($result as $user) {
        if (($user["enabled"]!=0) && (!in_array($user["login"], $login_ldap))) {
            DisableUser($user["login"]);
        }
    }
}

// ======== Requetes ===========================================

/**
 * @brief Pour un login, passe le champ user.enabled à false
 * @param $login login
 * @return true si ok et false sinon
 */
function DisableUser($login) {
    global $compteur_desactivation;

    $db = Connection::getInstance();
    if(!$db) {
        throw new MyException (ERR_CONNECT_DB);
    }
    $sql = "UPDATE user SET enabled=:enabled WHERE login=:login";
    if (DEBUG==True)
        Logger::log('DisableUser: désactivation du compte : '.$login);
    $stmt = $db->prepare($sql);
    if ($stmt === null) {
        throw new MyException ('DisableUser: erreur de préparation à la requête');
    }
    $enabled = false;
    $stmt->bindParam('login', $login, PDO::PARAM_STR);
    $stmt->bindParam('enabled', $enabled, PDO::PARAM_BOOL);
    if ($stmt->execute() === false) {
        $err = $stmt->errorInfo();
        Logger::log("DisableUser: erreur d'update pour ".$login.": ".$err[2]);
        return false;
    }
    $compteur_desactivation++;
    return true;
}

/**
 * @brief Insert ou met à jour si besoin un élément dans la table user
 * Recherche le user par son login
 * Si aucune correspondance alors création du user
 * Si user existant alors comparaison des champs du compte pour voir si une mise à jour est nécessaire
 * En cas de mise à jour, le champ enabled n'est pas modifié
 * @param $login login
 * @param $lastname nom
 * @param $firstname prénom
 * @param $email mail
 * @param $birthdate date de naissance
 * @param $profile profil
 * @param $enabled booléen déterminant si le compte est actif ou non
 * @param $admin booléen déterminant si la personne est admin ou non
 * @return id_user créé/trouvé ou -1 si erreur de requete
 */
function AddOrUpdateUser($login,$lastname,$firstname,$email,$birthdate,$profile,$enabled,$admin) {
    global $compteur_ajout, $compteur_maj;

    $db = Connection::getInstance();
    if(!$db) {
        throw new MyException (ERR_CONNECT_DB);
    }
    // récupération des informations de l'utilisateur par rapport au login
    $select = "SELECT * FROM user WHERE login = :login";
    $stmt_select = $db->prepare($select);
    if ($stmt_select === null) {
        throw new MyException ('AddOrUpdateUser: erreur de préparation à la requête');
    }
    $stmt_select->bindParam('login', $login, PDO::PARAM_STR);
    if ($stmt_select->execute() === false) {
        $stmt_select->errorInfo();
        if (DEBUG==True)
            Logger::log('AddOrUpdateUser: echec du select pour : '.$login);
        return -1;
    }
    $result = $stmt_select->fetch(PDO::FETCH_ASSOC);
    $count = $stmt_select->rowCount();
    if($count == 0){
        // le login n'existe pas donc insertion
        $sql = "INSERT INTO user (login,lastname,firstname,email,birth_date,id_profile,enabled,admin) VALUES (:login,:lastname,:firstname,:email,:birthdate,:profile,:enabled,:admin)";
        if (DEBUG==True)
            Logger::log('AddOrUpdateUser: insertion d\'un nouvel utilisateur : '.$login);
        $stmt = $db->prepare($sql);
        if ($stmt === null) {
            throw new MyException ('AddOrUpdateUser: erreur de préparation à la requête');
        }
        $stmt->bindParam('login', $login, PDO::PARAM_STR);
        $stmt->bindParam('lastname', $lastname, PDO::PARAM_STR);
        $stmt->bindParam('firstname', $firstname, PDO::PARAM_STR);
        $stmt->bindParam('email', $email, PDO::PARAM_STR);
        $birthdate = ($birthdate == NULL)?"":$birthdate;
        $stmt->bindParam('birthdate', $birthdate, PDO::PARAM_STR);
        $stmt->bindParam('profile', $profile, PDO::PARAM_INT);
        $stmt->bindParam('enabled', $enabled, PDO::PARAM_BOOL);
        $stmt->bindParam('admin', $admin, PDO::PARAM_BOOL);
        if ($stmt->execute() === false) {
            $err = $stmt->errorInfo();
            Logger::log("AddOrUpdateUser: erreur d'insertion pour ".$login.": ".$err[2]);
            return -1;
        }else{
            // Récupération de l'id
            $stmt_select->bindParam('login', $login, PDO::PARAM_STR);
            if ($stmt_select->execute() === false) {
                $stmt_select->errorInfo();
                return -1;
            }
            $result = $stmt_select->fetch(PDO::FETCH_ASSOC);
            $compteur_ajout++;
            return $result["id_user"];
        }

    } elseif ($count == 1) {
        // le login existe donc comparaison des valeurs et si différence update sinon rien

        // Pour comparer les dates il faut les remettre au même format (ldap = "Ymd" et mysql = "Y-m-d")
        if (!is_null($birthdate)){
            $datedenaissance = preg_replace("/([0-9]{4})([0-9]{2})([0-9]{2})/", "$1-$2-$3", $birthdate);
        } else {
            $datedenaissance = "0000-00-00"; // FIXE : valeur de la date de naissance par défaut dans mysql si "" est envoyé
        }

        if (($lastname!=$result["lastname"]) || ($firstname!=$result["firstname"]) || ($email!=$result["email"]) || ($datedenaissance!=$result["birth_date"]) || ($profile!=$result["id_profile"]) || ($admin!=$result["admin"])) {
            $sql = "UPDATE user SET lastname=:lastname,firstname=:firstname,email=:email,birth_date=:birthdate,id_profile=:profile,admin=:admin WHERE login=:login";
            if (DEBUG==True)
                Logger::log('AddOrUpdateUser: update d\'un utilisateur connu : '.$login);
            $stmt = $db->prepare($sql);
            if ($stmt === null) {
                throw new MyException ('AddOrUpdateUser: erreur de préparation à la requête');
            }
            $stmt->bindParam('login', $login, PDO::PARAM_STR);
            $stmt->bindParam('lastname', $lastname, PDO::PARAM_STR);
            $stmt->bindParam('firstname', $firstname, PDO::PARAM_STR);
            $stmt->bindParam('email', $email, PDO::PARAM_STR);
            $stmt->bindParam('birthdate', $birthdate, PDO::PARAM_STR);
            $stmt->bindParam('profile', $profile, PDO::PARAM_INT);
            $stmt->bindParam('admin', $admin, PDO::PARAM_BOOL);
            if ($stmt->execute() === false) {
                $err = $stmt->errorInfo();
                Logger::log("AddOrUpdateUser: erreur d'update pour ".$login.": ".$err[2]);
                return -1;
            }
            $compteur_maj++;
            return $result["id_user"];
        } else {
            if (DEBUG==True)
                Logger::log('AddOrUpdateUser: user existant : '.$login);
            return $result["id_user"];
        }
    } else {
        // Ici il y a plus de 1 occurence donc problème
        throw new MyException (ERR_SELECT_NB_ROW);
    }
    // si on arrive ici c'est qu'il y a eu un problème
    return -1;
}

/**
 * @brief Insert si besoin un élément dans la table level
 * @param $classe intitulé de la classe
 * @return id_level ou -1 si problème de requete
 */
function AddLevel($classe) {
    global $compteur_level;

    $db = Connection::getInstance();
    if(!$db) {
        throw new MyException (ERR_CONNECT_DB);
    }
    // récupération des informations de l'utilisateur par rapport au login
    $select = "SELECT * FROM level WHERE name = :name";
    $stmt_select = $db->prepare($select);
    if ($stmt_select === null) {
        throw new MyException ('AddLevel: erreur de préparation à la requête');
    }
    $stmt_select->bindParam('name', $classe, PDO::PARAM_STR);
    if ($stmt_select->execute() === false) {
        $stmt_select->errorInfo();
        return -1;
    }
    $result = $stmt_select->fetch(PDO::FETCH_ASSOC);
    $count = $stmt_select->rowCount();
    if($count == 0){
        // la classe n'existe pas donc insertion
        $sql = "INSERT INTO level (name) VALUES (:name)";
        if (DEBUG==True)
            Logger::log('AddLevel: insertion d\'une nouvelle classe : '.$classe);
        $stmt = $db->prepare($sql);
        if ($stmt === null) {
            throw new MyException ('AddLevel: erreur de préparation à la requête');
        }
        $stmt->bindParam('name', $classe, PDO::PARAM_STR);
        if ($stmt->execute() === false) {
            $err = $stmt->errorInfo();
            Logger::log("AddLevel: erreur d'insertion pour ".$classe.": ".$err[2]);
            return -1;
        }
        // Récupération de l'id
        $stmt_select->bindParam('name', $classe, PDO::PARAM_STR);
        if ($stmt_select->execute() === false) {
            $stmt_select->errorInfo();
            return -1;
        }
        $result = $stmt_select->fetch(PDO::FETCH_ASSOC);
        $compteur_level++;
        return $result["id_level"];
    } elseif ($count == 1) {
        // la classe existe donc on ne fait rien
        if (DEBUG==True)
            Logger::log('AddLevel: level existant : '.$classe);
        return $result["id_level"];
    } else {
        // Ici il y a plus de 1 occurence donc problème
        throw new MyException (ERR_SELECT_NB_ROW);
    }
    // si on arrive ici c'est qu'il y a eu un problème
    return -1;
}

/**
 * @brief Insert ou met à jour si besoin un élément dans la table member_of_level
 * @param $id_user id user
 * @param $id_level id level
 * @param $manager booléen déterminant si la personne est prof principal de la classe ou non
 * @return 0 si ok ou -1 si problème de requete
 */
function AddOrUpdateMemberOfLevel($id_user, $id_level, $manager=false) {
    $db = Connection::getInstance();
    if(!$db) {
        throw new MyException (ERR_CONNECT_DB);
    }
    // récupération des données
    $select = "SELECT * FROM member_of_level WHERE id_user = :id_user AND id_level = :id_level";
    $stmt_select = $db->prepare($select);
    if ($stmt_select === null) {
        throw new MyException ('AddOrUpdateMemberOfLevel: erreur de préparation à la requête');
    }
    $stmt_select->bindParam('id_user', $id_user, PDO::PARAM_INT);
    $stmt_select->bindParam('id_level', $id_level, PDO::PARAM_INT);
    if ($stmt_select->execute() === false) {
        $stmt_select->errorInfo();
        return -1;
    }
    $result = $stmt_select->fetch(PDO::FETCH_ASSOC);
    $count = $stmt_select->rowCount();
    if($count == 0){
        // la classe n'existe pas donc insertion
        $sql = "INSERT INTO member_of_level (id_user,id_level,manager) VALUES (:id_user,:id_level,:manager)";
        if (DEBUG==True)
            Logger::log('AddOrUpdateMemberOfLevel: insertion d\'un nouveau member_of_level : id_user = '.$id_user.' et id_level = '.$id_level);
        $stmt = $db->prepare($sql);
        if ($stmt === null) {
            throw new MyException ('AddOrUpdateMemberOfLevel: erreur de préparation à la requête');
        }
        $stmt->bindParam('id_user', $id_user, PDO::PARAM_INT);
        $stmt->bindParam('id_level', $id_level, PDO::PARAM_INT);
        $stmt->bindParam('manager', $manager, PDO::PARAM_BOOL);
        if ($stmt->execute() === false) {
            $err = $stmt->errorInfo();
            Logger::log("AddOrUpdateMemberOfLevel: erreur d'insertion pour id_user = ".$id_user." et id_level = ".$id_level." : ".$err[2]);
            return -1;
        }
        return 0;
    } elseif ($count == 1) {
        // la classe existe donc on regarde si le champ manager a changé
        if ($manager!=$result["manager"]) {
            $sql = "UPDATE member_of_level SET manager=:manager WHERE id_user = :id_user AND id_level = :id_level";
            if (DEBUG==True)
                Logger::log('AddOrUpdateMemberOfLevel: update d\'un member_of_level connu : '.$login);
            $stmt = $db->prepare($sql);
            if ($stmt === null) {
                throw new MyException ('AddOrUpdateMemberOfLevel: erreur de préparation à la requête');
            }
            $stmt->bindParam('id_user', $id_user, PDO::PARAM_INT);
            $stmt->bindParam('id_level', $id_level, PDO::PARAM_INT);
            $stmt->bindParam('manager', $manager, PDO::PARAM_BOOL);
            if ($stmt->execute() === false) {
                $err = $stmt->errorInfo();
                Logger::log("AddOrUpdateMemberOfLevel: erreur d'update pour id_user = ".$id_user." et id_level = ".$id_level.": ".$err[2]);
                return -1;
            }
            return 0;
        } else {
            if (DEBUG==True)
                Logger::log('AddOrUpdateMemberOfLevel: member_of_level existant : id_user = '.$id_user.' et id_level = '.$id_level);
            return 0;
        }
    } else {
        // Ici il y a plus de 1 occurence donc problème
        throw new MyException (ERR_SELECT_NB_ROW);
    }
    // si on arrive ici c'est qu'il y a eu un problème
    return -1;
}

/**
 * @brief Insert si besoin un élément dans la table school_report
 * @param $id_user id user
 * @return id_report ou -1 si problème de requete
 */
function AddSchoolReport($id_user) {
    $db = Connection::getInstance();
    if(!$db) {
        throw new MyException (ERR_CONNECT_DB);
    }
    // récupération du carnet s'il existe
    $select = "SELECT * FROM school_report WHERE id_user = :id_user";
    $stmt_select = $db->prepare($select);
    if ($stmt_select === null) {
        throw new MyException ('AddSchoolReport: erreur de préparation à la requête');
    }
    $stmt_select->bindParam('id_user', $id_user, PDO::PARAM_INT);
    if ($stmt_select->execute() === false) {
        $stmt_select->errorInfo();
        return -1;
    }
    $result = $stmt_select->fetch(PDO::FETCH_ASSOC);
    $count = $stmt_select->rowCount();
    if($count == 0){
        // le carnet n'existe pas donc insertion
        $sql = "INSERT INTO school_report (id_user) VALUES (:id_user)";
        if (DEBUG==True)
            Logger::log('AddSchoolReport: insertion d\'un nouveau carnet : '.$id_user);
        $stmt = $db->prepare($sql);
        if ($stmt === null) {
            throw new MyException ('AddSchoolReport: erreur de préparation à la requête');
        }
        $stmt->bindParam('id_user', $id_user, PDO::PARAM_INT);
        if ($stmt->execute() === false) {
            $err = $stmt->errorInfo();
            Logger::log("AddSchoolReport: erreur d'insertion pour ".$id_user.": ".$err[2]);
            return -1;
        }
        // Récupération de l'id
        $stmt_select->bindParam('id_user', $id_user, PDO::PARAM_INT);
        if ($stmt_select->execute() === false) {
            $stmt_select->errorInfo();
            return -1;
        }
        $result = $stmt_select->fetch(PDO::FETCH_ASSOC);
        return $result["id_report"];
    } elseif ($count == 1) {
        // le carnet existe donc on ne fait rien
        if (DEBUG==True)
            Logger::log('AddSchoolReport: carnet existant : '.$id_user);
        return $result["id_report"];
    } else {
        // Ici il y a plus de 1 occurence donc problème
        throw new MyException (ERR_SELECT_NB_ROW);
    }
    // si on arrive ici c'est qu'il y a eu un problème
    return -1;
}

/**
 * @brief Insert si besoin un élément dans la table responsible
 * @param $id_student id user de l'élève
 * @param $id_responsible id user du responsable
 * @return 0 si ok ou -1 si problème de requete
 */
function AddResponsible($id_student, $id_responsible) {
    $db = Connection::getInstance();
    if(!$db) {
        throw new MyException (ERR_CONNECT_DB);
    }
    // récupération des données
    $select = "SELECT * FROM responsible WHERE id_student = :id_student AND id_responsible = :id_responsible";
    $stmt_select = $db->prepare($select);
    if ($stmt_select === null) {
        throw new MyException ('AddResponsible: erreur de préparation à la requête');
    }
    $stmt_select->bindParam('id_student', $id_student, PDO::PARAM_INT);
    $stmt_select->bindParam('id_responsible', $id_responsible, PDO::PARAM_INT);
    if ($stmt_select->execute() === false) {
        $stmt_select->errorInfo();
        return -1;
    }
    $result = $stmt_select->fetch(PDO::FETCH_ASSOC);
    $count = $stmt_select->rowCount();
    if($count == 0){
        // l'enregistrement n'existe pas donc insertion
        $sql = "INSERT INTO responsible (id_student, id_responsible) VALUES (:id_student,:id_responsible)";
        if (DEBUG==True)
            Logger::log('AddResponsible: insertion d\'un responsible : id_student = '.$id_student.' et id_responsible = '.$id_responsible);
        $stmt = $db->prepare($sql);
        if ($stmt === null) {
            throw new MyException ('AddResponsible: erreur de préparation à la requête');
        }
        $stmt->bindParam('id_student', $id_student, PDO::PARAM_INT);
        $stmt->bindParam('id_responsible', $id_responsible, PDO::PARAM_INT);
        if ($stmt->execute() === false) {
            $err = $stmt->errorInfo();
            Logger::log("AddResponsible: erreur d'insertion pour id_student = ".$id_student." et id_responsible = ".$id_responsible." : ".$err[2]);
            return -1;
        }
        return 0;
    } elseif ($count == 1) {
        // l'enregistrement existe donc on ne fait rien
        if (DEBUG==True)
            Logger::log('AddResponsible: responsible existant : id_student = '.$id_student.' et id_responsible = '.$id_responsible);
        return 0;
    } else {
        // Ici il y a plus de 1 occurence donc problème
        throw new MyException (ERR_SELECT_NB_ROW);
    }
    // si on arrive ici c'est qu'il y a eu un problème
    return -1;
}

/**
 * @brief Insert si besoin un élément dans la table discipline
 * @param $matiere matière
 * @return id_discipline ou -1 si problème de requete
 */
function AddDiscipline($matiere) {
    global $compteur_discipline;

    $db = Connection::getInstance();
    if(!$db) {
        throw new MyException (ERR_CONNECT_DB);
    }
    // récupération des informations de la discipline par rapport au nom
    $select = "SELECT * FROM discipline WHERE name = :name";
    $stmt_select = $db->prepare($select);
    if ($stmt_select === null) {
        throw new MyException ('AddDiscipline: erreur de préparation à la requête');
    }
    $stmt_select->bindParam('name', $matiere, PDO::PARAM_STR);
    if ($stmt_select->execute() === false) {
        $stmt_select->errorInfo();
        return -1;
    }
    $result = $stmt_select->fetch(PDO::FETCH_ASSOC);
    $count = $stmt_select->rowCount();
    if($count == 0){
        // la matiere n'existe pas donc insertion
        $sql = "INSERT INTO discipline (name) VALUES (:name)";
        if (DEBUG==True)
            Logger::log('AddDiscipline: insertion d\'une nouvelle matiere : '.$matiere);
        $stmt = $db->prepare($sql);
        if ($stmt === null) {
            throw new MyException ('AddDiscipline: erreur de préparation à la requête');
        }
        $stmt->bindParam('name', $matiere, PDO::PARAM_STR);
        if ($stmt->execute() === false) {
            $err = $stmt->errorInfo();
            Logger::log("AddDiscipline: erreur d'insertion pour ".$matiere.": ".$err[2]);
            return -1;
        }
        // Récupération de l'id
        $stmt_select->bindParam('name', $matiere, PDO::PARAM_STR);
        if ($stmt_select->execute() === false) {
            $stmt_select->errorInfo();
            return -1;
        }
        $result = $stmt_select->fetch(PDO::FETCH_ASSOC);
        $compteur_discipline++;
        return $result["id_discipline"];
    } elseif ($count == 1) {
        // la matiere existe donc on ne fait rien
        if (DEBUG==True)
            Logger::log('AddDiscipline: discipline existante : '.$matiere);
        return $result["id_discipline"];
    } else {
        // Ici il y a plus de 1 occurence donc problème
        throw new MyException (ERR_SELECT_NB_ROW);
    }
    // si on arrive ici c'est qu'il y a eu un problème
    return -1;
}

/**
 * @brief Insert si besoin un élément dans la table teacher
 * @param $id_discipline id discipline
 * @param $id_user id user
 * @return 0 si ok ou -1 si problème de requete
 */
function AddTeacher($id_discipline, $id_user) {
    $db = Connection::getInstance();
    if(!$db) {
        throw new MyException (ERR_CONNECT_DB);
    }
    // récupération des données
    $select = "SELECT * FROM teacher WHERE id_discipline = :id_discipline AND id_user = :id_user";
    $stmt_select = $db->prepare($select);
    if ($stmt_select === null) {
        throw new MyException ('AddTeacher: erreur de préparation à la requête');
    }
    $stmt_select->bindParam('id_discipline', $id_discipline, PDO::PARAM_INT);
    $stmt_select->bindParam('id_user', $id_user, PDO::PARAM_INT);
    if ($stmt_select->execute() === false) {
        $stmt_select->errorInfo();
        return -1;
    }
    $result = $stmt_select->fetch(PDO::FETCH_ASSOC);
    $count = $stmt_select->rowCount();
    if($count == 0){
        // la classe n'existe pas donc insertion
        $sql = "INSERT INTO teacher (id_discipline,id_user) VALUES (:id_discipline,:id_user)";
        if (DEBUG==True)
            Logger::log('AddTeacher: insertion d\'un nouveau teacher : id_discipline = '.$id_discipline.' et id_user = '.$id_user);
        $stmt = $db->prepare($sql);
        if ($stmt === null) {
            throw new MyException ('AddTeacher: erreur de préparation à la requête');
        }
        $stmt->bindParam('id_discipline', $id_discipline, PDO::PARAM_INT);
        $stmt->bindParam('id_user', $id_user, PDO::PARAM_INT);
        if ($stmt->execute() === false) {
            $err = $stmt->errorInfo();
            Logger::log("AddTeacher: erreur d'insertion pour id_discipline = ".$id_discipline." et id_user = ".$id_user." : ".$err[2]);
            return -1;
        }
        return 0;
    } elseif ($count == 1) {
        // la classe existe donc on ne fait rien
        if (DEBUG==True)
            Logger::log('AddTeacher: teacher existant : id_discipline = '.$id_discipline.' et id_user = '.$id_user);
        return 0;
    } else {
        // Ici il y a plus de 1 occurence donc problème
        throw new MyException (ERR_SELECT_NB_ROW);
    }
    // si on arrive ici c'est qu'il y a eu un problème
    return -1;
}

/**
 * @brief Retourne le id_user d'un user à partir de son login
 * @param $login login
 * @return id_user si ok ou -1 si problème de requete
 */
function GetIdUser($login) {
    $db = Connection::getInstance();
    if(!$db) {
        throw new MyException (ERR_CONNECT_DB);
    }
    $select = "SELECT id_user FROM user WHERE login = :login";
    $stmt_select = $db->prepare($select);
    if ($stmt_select === null) {
        throw new MyException ('GetIdUser: erreur de préparation à la requête');
    }
    $stmt_select->bindParam('login', $login, PDO::PARAM_STR);
    if ($stmt_select->execute() === false) {
        $stmt_select->errorInfo();
        if (DEBUG==True)
            Logger::log('GetIdUser: echec du select pour : '.$login);
        return -1;
    }
    $result = $stmt_select->fetch(PDO::FETCH_NUM);
    return $result[0];
}

?>
