<?php
/**
 * @author CETIAD Dijon
 * @date Septembre 2011
 * @file Responsible.class.php
 * @version 1.0
 * @copyright GNU Public License, see the LICENSE.txt file
 * @brief Classe regroupant les méthodes utilisés par les responsables
 */

class Responsible extends Student
{

    /**
     * @brief le constructeur permet de loguer l'instanciation d'un responsable
     */
    public function __construct()
    {
        if (DEBUG==True)
            Logger::log('création d\'un nouveau Responsible');
    }

    /**
     * @brief identifier les personnes dont l'utilisateur est responsable
     * @param (int) id du responsable
     * @return (array) les id user et le nom
     */
    public function allUnderCare($id_user) {
        if (DEBUG==True)
            Logger::log('passage dans allUnderCare');
        $db = Connection::getInstance();
        if(!$db) {
            throw new MyException (ERR_CONNECT_DB);
        }
        $sql = 'SELECT id_student,firstname,lastname FROM responsible JOIN user ON responsible.id_student = user.id_user WHERE responsible.id_responsible = :id_user';
        if (DEBUG==True)
            Logger::log('nombre de personne sous la responsabilité de l\'utilisateur : '.$sql);
        $stmt = $db->prepare($sql);
        if ($stmt === null) {
            throw new MyException ('erreur de préparation à la requête');
        }
        $stmt->bindParam('id_user', $id_user, PDO::PARAM_INT);
        if ($stmt->execute() === false) {
            $stmt->errorInfo();
        }
        $result = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $count = $stmt->rowCount();
        if (DEBUG == True)
            Logger::log('nombre de ligne renvoyées : '.$count);
        return($result);
    }

    /**
     * @brief retourne la liste des personnes faisant partie d'une classe selon le niveau d'autorisation
     * @param (int) id de la classe
     * @param (int) id du profil
     * @return (array) les id des utilisateurs de la classe
     * @public
     */
    public function selectAllMemberLevel($id_level,$id_profile) {
        if (DEBUG==True) {
            Logger::log('passage dans selectAllMemberLevel : Responsible.class.php');
        }
        $db = Connection::getInstance();
        if(!$db) {
            throw new MyException (ERR_CONNECT_DB);
        }
        $sql = 'SELECT user.id_user,firstname,lastname, discipline.name as matiere
                FROM user
                JOIN member_of_level ON member_of_level.id_user = user.id_user
                LEFT JOIN teacher ON teacher.id_user = user.id_user
                LEFT JOIN discipline ON discipline.id_discipline = teacher.id_discipline
                WHERE user.id_profile = :id_profile
                AND member_of_level.id_level = :id_level';
        if (DEBUG==True)
            Logger::log('selection de toutes les personnes d\'une classe : '.$sql);
        $stmt = $db->prepare($sql);
        if ($stmt === null) {
            throw new MyException ('erreur de préparation à la requête');
        }
        $stmt->bindParam('id_profile', $id_profile, PDO::PARAM_INT);
        $stmt->bindParam('id_level', $id_level, PDO::PARAM_INT);
        if ($stmt->execute() === false) {
            $stmt->errorInfo();
        }
        $result = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $count = $stmt->rowCount();
        if (DEBUG == True)
            Logger::log('nombre de ligne renvoyées : '.$count);
        return($result);
    }

    /**
     * @brief retourne la liste des classes auquel l'utilisateur appartient
     * @param (int) id de l'utilisateur
     * @return (array) les id et nom des différentes classes
     * @static
     * @fixme méthode au bon endroit ?
     */
    static function selectAllInLevel($idUserUnderCare) {
        if (DEBUG==True)
            Logger::log('passage dans selectAllInLevel');
        $db = Connection::getInstance();
        if(!$db) {
            throw new MyException (ERR_CONNECT_DB);
        }
        $sql = 'SELECT level.id_level,level.name FROM level JOIN member_of_level ON level.id_level = member_of_level.id_level WHERE member_of_level.id_user = :idUserUnderCare';
        if (DEBUG==True)
            Logger::log('selection de toutes les classes d\'une personnes : '.$sql);
        $stmt = $db->prepare($sql);
        if ($stmt === null) {
            throw new MyException ('erreur de préparation à la requête');
        }
        $stmt->bindParam('idUserUnderCare', $idUserUnderCare, PDO::PARAM_INT);
        if ($stmt->execute() === false) {
            $stmt->errorInfo();
        }
        $result = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $count = $stmt->rowCount();
        if (DEBUG == True)
            Logger::log('nombre de ligne renvoyées : '.$count);
        return($result);
    }

    /**
     * @brief on va chercher les infos de l'utilisateur
     * @param (int) id du destinataire
     * @return (array) le nom et le prénom du destinataire
     * @public
     */
    public function selectOnRecipient($idRecipient)
    {
        if (DEBUG==True)
            Logger::log('passage dans selectOnRecipient');
        $db = Connection::getInstance();
        if(!$db) {
            throw new MyException (ERR_CONNECT_DB);
        }
        $sql = 'SELECT firstname,lastname FROM user WHERE id_user= :idRecipient';
        if (DEBUG==True)
            Logger::log('selection des infos d\'un destinataire : '.$sql);
        $stmt = $db->prepare($sql);
        if ($stmt === null) {
            throw new MyException ('erreur de préparation à la requête');
        }
        $stmt->bindParam('idRecipient', $idRecipient, PDO::PARAM_INT);
        if ($stmt->execute() === false) {
            $stmt->errorInfo();
        }
        $result = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $count = $stmt->rowCount();
        if (DEBUG == True)
            Logger::log('nombre de ligne renvoyées : '.$count);
        if($count == 0)
            throw new MyException (ERR_SELECT_NB_NULL);
        if($count > 1)
            throw new MyException (ERR_SELECT_NB_ROW);
        else
            return $result;
    }

    /**
     * @bried Obtenir les informations utilisateurs
     * @param Id de l'utilisateur en cours
     * @return Tableau contenant les informations utilisateurs
     * @todo l'id en paramètre n'est pas souhaitable
     */
    public function getUserInfos($idUser) {
        Logger::log('passage dans getUserInfos '.$idUser);
        $db = Connection::getInstance();
        if(!$db) {
            throw new MyException (ERR_CONNECT_DB);
        }

        // récupération des informations de l'utilisateur par rapport au login
        $sql = "SELECT firstname,lastname,login,email,birth_date as birthdate,DATE_FORMAT(birth_date,'%Y') as year,
                DATE_FORMAT(birth_date,'%m') as month,DATE_FORMAT(birth_date,'%d') as day
                FROM user
                WHERE user.id_user = :idUser";
        $stmt = $db->prepare($sql);
        if ($stmt === null) {
            throw new MyException ('erreur de préparation à la requête');
        }
        $stmt->bindParam('idUser', $idUser, PDO::PARAM_INT);
        if ($stmt->execute() === false) {
            $stmt->errorInfo();
        }
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        $count = $stmt->rowCount();
        if (DEBUG==True)
            Logger::log('selectionner tous de l\'utilisateur');
            Logger::log('nombre de lignes renvoyées : '.$count);
        if($count == 0){
            // le login n'existe pas
            return False;
        } elseif ($count > 1){
            throw new MyException (ERR_SELECT_NB_ROW);
        } elseif ($count == 1) {
            // récupération des élèves à charge
            $sql_elev = "SELECT firstname,lastname,level.name as classe
                        FROM user
                        JOIN member_of_level ON user.id_user = member_of_level.id_user
                        JOIN level ON level.id_level = member_of_level.id_level
                        JOIN responsible ON user.id_user = responsible.id_student
                        WHERE id_responsible = :idUser";
            $stmt_elev = $db->prepare($sql_elev);
            if ($stmt_elev === null) {
                throw new MyException ('erreur de préparation à la requête');
            }
            $stmt_elev->bindParam('idUser', $idUser, PDO::PARAM_INT);
            if ($stmt_elev->execute() === false) {
                $stmt_elev->errorInfo();
            }
            $result_elev = $stmt_elev->fetchAll();
            $count_elev = $stmt_elev->rowCount();
            if($count_elev != 0){
                // des élèves ont été trouvés ajout au tableau de résultat
                $result["eleves_a_charge"]=$result_elev;
            } else {
                // on ajoute quand même un tableau vide
                $result["eleves_a_charge"]=array();
            }


            return $result;
        } else {
            return False;
        }
    }

    /**
     * @brief Retourne les personnes affectées dans une classe dont l'utilisateur est responsable
     * @param (int) id du responsable
     * @return (array) les id user et le nom
     */
    public function allUnderCareInLevel($id_user) {
        if (DEBUG==True)
            Logger::log('passage dans allUnderCareInLevel');
        $db = Connection::getInstance();
        if(!$db) {
            throw new MyException (ERR_CONNECT_DB);
        }
        $sql = 'SELECT DISTINCT id_student,firstname,lastname, level.name FROM responsible JOIN user ON responsible.id_student = user.id_user join member_of_level ON member_of_level.id_user = user.id_user join level on member_of_level.id_level = level.id_level WHERE responsible.id_responsible = :id_user';
        if (DEBUG==True)
            Logger::log('nombre de personne appartenant à une classe sous la responsabilité de l\'utilisateur : '.$sql);
        $stmt = $db->prepare($sql);
        if ($stmt === null) {
            throw new MyException ('erreur de préparation à la requête');
        }
        $stmt->bindParam('id_user', $id_user, PDO::PARAM_INT);
        if ($stmt->execute() === false) {
            $stmt->errorInfo();
        }
        $result = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $count = $stmt->rowCount();
        if (DEBUG == True)
            Logger::log('nombre de ligne renvoyées : '.$count);
        return($result);
    }

    /**
     * @brief Retourne les personnes n'ayant pas été affectées dans une classe dont l'utilisateur est responsable
     * @param (int) id du responsable
     * @return (array) les id user et le nom
     */
    public function allUnderCareNoLevel($id_user) {
        if (DEBUG==True)
            Logger::log('passage dans allUnderCareInLevel');
        $db = Connection::getInstance();
        if(!$db) {
            throw new MyException (ERR_CONNECT_DB);
        }
        $sql = 'SELECT DISTINCT id_student, firstname, lastname FROM user LEFT JOIN member_of_level ON user.id_user = member_of_level.id_user LEFT JOIN level ON member_of_level.id_level = level.id_level join responsible ON responsible.id_student = user.id_user WHERE member_of_level.id_level IS NULL AND responsible.id_responsible = :id_user';
        if (DEBUG==True)
            Logger::log('nombre de personne appartenant à une classe sous la responsabilité de l\'utilisateur : '.$sql);
        $stmt = $db->prepare($sql);
        if ($stmt === null) {
            throw new MyException ('erreur de préparation à la requête');
        }
        $stmt->bindParam('id_user', $id_user, PDO::PARAM_INT);
        if ($stmt->execute() === false) {
            $stmt->errorInfo();
        }
        $result = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $count = $stmt->rowCount();
        if (DEBUG == True)
            Logger::log('nombre de ligne renvoyées : '.$count);
        return($result);
    }


}
?>
