#!/bin/bash
# chargement des parametres
if [ -e param.sh ]; then
	. ./param.sh
fi

# conf java
export CLASSPATH=$CLASSPATH:/usr/share/java/saxonb.jar
# timestamp
DATE=`date '+%Y%m%d'`
TIME=`date '+%H%M%S'`
# identite ENT GAR
SCRIPTVERSION="974-20210402-001"
DEBUTDATE=`date +'%A %d %B %Y'`
DEBUTTIME=`date +%k:%M:%S`
STATUTGLOBAL="OK"
DEFAULTXSD="1.5.4"

MODE=${MODE:-standalone}

# Vérification de la présence des applications xmlstarlet, java
xmlstarlet --version >/dev/null 2>&1
[ $? -ne 0 ] && echo "verifier que xmlstarlet est installé" && exit 1

java  -version >/dev/null 2>&1
[ $? -ne 0 ] && echo "verifier que java (default-jre)  est installé" && exit 1

[ ! -e /usr/share/java/saxonb.jar ] && echo "vérifier que libsaxonb-java est installé" && exit 1

# fichiers AAF sources v1 - remplacés en v2 par des collections directement dans les xsl (c'est pourquoi lors des transformations la source déclarée en ligne de commande n'est plus le fichier source (puisqu'ils sont multiples) mais le fichier de transformation lui meme : c'est dans ce dernier que les collections de fichiers sources sont déclarés.
# Récupération des paramètres par défaut
loadConf() {
	CONF="$1"
	[ ! -e $CONF ] && echo "Configuration $CONF introuvable" && exit 1
	ANONYMAT=`xmlstarlet sel -t -m "//GAR-CONFIG/ANONYMAT" -v . -n $CONF`
	GARPATH=`xmlstarlet sel -t -m "//GAR-CONFIG/GARPATH" -v . -n $CONF`
	AAFPATH=`xmlstarlet sel -t -m "//GAR-CONFIG/AAFPATH" -v . -n $CONF`
	TMPPATH=`xmlstarlet sel -t -m "//GAR-CONFIG/TMPPATH" -v . -n $CONF`
	BASHPATH=`xmlstarlet sel -t -m "//GAR-CONFIG/BASHPATH" -v . -n $CONF`
	VERSXSD=`xmlstarlet sel -t -m "//GAR-CONFIG/VERSION" -v . -n $CONF`
	etabs=`xmlstarlet sel -t -m "//GAR-CONFIG/etabs" -v . -n $CONF`
	eleves=`xmlstarlet sel -t -m "//GAR-CONFIG/eleves" -v . -n $CONF`
	personnels=`xmlstarlet sel -t -m "//GAR-CONFIG/personnels" -v . -n $CONF`
	VERSION="$VERSXSD"
	# Version par défaut si non préciser
	[ -z "$VERSION" ] && VERSION="$DEFAULTXSD"
	XSLPATH="xsl/"
}

if [ ! -e "config.xml" ]; then
	cp config.xml.sample config.xml
fi

# Config par défaut
loadConf config.xml

# Ce fichier de config sera utilisé ( modifier par param de lignes de commandes)
CONFIG=".running-config.xml"

# option --save permettra de préciser de sauvegarder la conf passer en parametres dans le config.xml
# config.xml qui sera utilisé la prochaine fois avec les paramètres passés en ligne de commandes
SAVE=0

# option --thread permettra de lancer les traitements en //
THREAD=0

# option --check permettra d'indiquer de passer la validation xsl
CHECK=0

# MOTIF de début pour fichiers AAF (par ex ${MOTIF}_Complet_*_Eleve_*.xml)
MOTIF=""

# SEND : envoi de l'archive sur le dépot ( Non (0) par défaut )
SEND=0

# NEW_CONF : Possibilité de préciser un autre fichier de config --conf=<path>
NEW_CONF=""

# Fonction permettant de mettre à jour le scripts et les xsl
doUpdate() {
	DIR=$(dirname "$(readlink -f "$0")")
	wget --header="Content-Type: application/x-www-form-urlencoded" --post-data "token=$TOKEN"  $WEBSERVICE/api/gar/${MODE}/setup/aaf2gar.tar.gz  -O /tmp/aaf2gar.tar.gz -o /dev/null
	[ $? -ne 0 ] && echo "impossible de récupérer l'archive" && exit 1
	cd $DIR
	tar -zxvf /tmp/aaf2gar.tar.gz >/dev/null 2>&1
	chmod +x aaf2gar.sh
	chmod +x param.sh

	# Maj des sources xsl
	NEW=$(cat aaf2gar.sh | egrep "^SCRIPTVERSION=" | cut -d \"  -f2)
	if [ "$NEW" = "$SCRIPTVERSION" ]; then
		echo "Script aaf2gar et xsl à jour"
	else
		echo "Mise à jour de la version aaf2gar [$SCRIPTVERSION] vers [$NEW]"
	fi

	# Maj de la version de la version de la grammaire
	wget --header="Content-Type: application/x-www-form-urlencoded" --post-data "token=$TOKEN"  $WEBSERVICE/api/gar/${MODE}/setup/aaf2gar-config.xml  -O config.xml  -o /dev/null
	NEW=`xmlstarlet sel -t -m "//GAR-CONFIG/VERSION" -v . -n config.xml`
	if [ "$NEW" = "$VERSION" ]; then
		echo "Grammaire à jour"
	else
		echo "Mise à jour de la grammaire [$VERSION] vers [$NEW]"
	fi

	exit
}

# Aide sur les options possibles
usage() {
	me=${0##*/}
	DATE=`date '+%Y%m%d'`
	echo "aaf2gar version=$SCRIPTVERSION"
	echo "-------------------------------------------------------------------------------------------------------------------------------------"
	echo "Usage: ${me} [options]"
    echo "-h                : afficher l'aide"
    echo "--aaf-path=<path> : <path> est le chemin où chercher les fichiers AAF  (par defaut ${AAFPATH})"
	echo "                    verifie si il existe dans ce dossier un dossier ${DATE} et l'utilise le cas échéant"
	echo "--gar-path=<path> : <path> est le dossier de destination des fichiers GAR,"
	echo "                     ce dernier sera créé si inexistant, et le contenu sera vidé si le dossier existe (par defaut ${GARPATH})"
	echo "--tmp-path=<path> : <path> est le dossier de destination des fichiers de transformation temporaires GAR, "
	echo "                     ce dernier sera créé si inexistant, et le contenu sera vidé si le dossier existe (par defaut ${TMPPATH})"
	echo "--motif=<motif>   : <motif> est le motif à utiliser pour matcher les fichiers AAF, il modifiera comme suit les expressions "
	echo "                     <motif>_Complet_*_Eleve_*.xml, <motif>_Complet_*_PersEducNat_*.xml,"
	echo "                     <motif>_Complet_*_EtabEducNat_*.xml, <motif>_Complet_*_MefEducNat_*.xml, <motif>_Complet_*_MatiereEducNat_*.xml"
	echo "--save            	: Sauvegarde dans le fichier config.xml les nouveaux paramètres renseignés en ligne de commande"
	echo "--xsd=<xsd version>   : Version du xsd à utiliser ($DEFAULTXSD par defaut)"
	echo "--thread          : Lance les traitements en mode multi-process pour accélerer le temps de traitement"
	echo "--check           : Demande la vérifation xsl des fichiers GAR"
	echo "--config=<fichier>: Indique d'utiliser un autre <fichier> de configuration (par défaut config.xml)"
	# Si URL WEBSERVICE on affiche les options pour utiliser WebService
	if [ ! -z "$WEBSERVICE" ]; then
		echo "--send            : Dépose l'archive sur le dépot sftp ( envoi a eConnect via WebService, qui le déposera sur le dépôt GAR)"
	    echo "--self-update     : Met à jour les fichiers xsl et les scripts avec la dernière version présente sur le serveur eConnect"
	fi
	if [ -z "$ENT" ]; then
		echo "--code=<code projet ent> : Code du projet ENT"
	fi
	echo "-------------------------------------------------------------------------------------------------------------------------------------"
	echo ""
}

# Possibilité de surcharger en ligne de commandes les chemins
# Passage via paramètre
for i in "$@"
do
case $i in
    -a=*|--aaf-path=*)
    AAFPATH="${i#*=}"
    ;;
    -g=*|--gar-path=*)
    GARPATH="${i#*=}"
    ;;
    -t=*|--tmp-path=*)
    TMPPATH="${i#*=}"
    ;;
	--motif=*)
    MOTIF="${i#*=}"
    ;;
	--code=*)
    [ -z "$ENT" ] && CODE="${i#*=}" 
    ;;
	--version=*)
    VERSXSD="${i#*=}"
    VERSION="${i#*=}"
    ;;
    --default)
    DEFAULT=YES
    ;;
    --save)
    SAVE=1
    ;;
    --thread)
    THREAD=1
    ;;
	--send)
    SEND=1
    ;;
    --check)
    CHECK=1
    ;;
	--config=*)
    NEW_CONF="${i#*=}"
	[ ! -e $NEW_CONF ] && echo "$NEW_CONF introuvable" && exit 1
	cp  $NEW_CONF ${CONFIG}
	loadConf $NEW_CONF
    ;;
	--self-update)
    doUpdate
	exit
    ;;
	-h|--help|?)
    usage
	exit
    ;;
    *)
      # unknown option
    ;;
esac
done

# Config actuelle de la conf dans la running-config
# Si pas de new conf definie en paramètre
[ -z "$NEW_CONF" ] && cp config.xml ${CONFIG}

# Ajout d'un slash terminal
addSlashToDirectory() {
 STR=$1
 length=${#STR}
 last_char=${STR:length-1:1}
 [[ $last_char != "/" ]] && STR="$STR/"; :
 # remplacement du ~ par chemin absolue
 echo $STR | sed -e "s|~|${HOME}|"
}

# Edit la config dans le XML
editConfig() {
 PARAM="$1"
 VALUE=$(eval "echo \$$PARAM")
 [ "$2" = "dir" ] && VALUE=$(addSlashToDirectory $VALUE)
 xmlstarlet ed -u "//GAR-CONFIG/$PARAM" -v "$VALUE" ${CONFIG} > /tmp/gar-config.xml
 cp /tmp/gar-config.xml ${CONFIG}
 # Refixe la variable
 eval "$1=$VALUE"
}

debut() {
  last=$(date +"%s")
  echo -n "$1 "
}

fin() {
  now=$(date +"%s")
  delta=$(expr $now - $last)
  echo $delta"s"
}


# Modification des paramètres de la configuration
editConfig AAFPATH "dir"
editConfig GARPATH "dir"
editConfig TMPPATH "dir"

# Motif redefini
if [ ! -z "$MOTIF" ]; then
  eleves="${MOTIF}_Complet_*_Eleve_*.xml"              && editConfig eleves
  etabs="${MOTIF}_Complet_*_EtabEducNat_*.xml"         && editConfig etabs
  matieres="${MOTIF}_Complet_*_MatiereEducNat_*.xml"   && editConfig matieres
  mefs="${MOTIF}_Complet_*_MefEducNat_*.xml"           && editConfig mefs
  personnels="${MOTIF}_Complet_*_PersEducNat_*.xml"    && editConfig personnels
fi

# Version redefini 
editConfig VERSION

# Pas de code ENT trouvé dans params.sh 
# et un code defini en ligne de commande
# on va mettre a jour le config.xml et params.sh
if [ -z "$ENT" -a ! -z "$CODE" ]; then
	DIR=$(dirname "$(readlink -f "$0")")
	ENT="$CODE"
	uuid=$(echo "${CODE}eleve" | md5sum | awk '{print $1}')
	ELVUUID=$( echo $uuid | cut -c1-8 )"-"$( echo $uuid | cut -c9-12 )"-"$( echo $uuid | cut -c13-16 )"-"$( echo $uuid | cut -c17-20 )
	uuid=$(echo "${CODE}enseignant" | md5sum | awk '{print $1}')
	ENSUUID=$( echo $uuid | cut -c1-8 )"-"$( echo $uuid | cut -c9-12 )"-"$( echo $uuid | cut -c13-16 )"-"$( echo $uuid | cut -c17-20 )
	echo "#!/bin/bash
ENT='$ENT'
WEBSERVICE='$WEBSERVICE'
TOKEN='$TOKEN'
MODE='$MODE'
" > $DIR/param.sh
	chmod +x $DIR/param.sh
	editConfig ELVUUID
	editConfig ENSUUID
fi

# Code ENT obligatoire
[ -z "$ENT" ] && echo "Veuillez préciser le code ENT du projet --code=<code du projet> " && exit 1

# Dossier date du jour existe dans dossier AAF
# On va utiliser ce chemin
DATE=`date '+%Y%m%d'`
if [ -d ${AAFPATH}${DATE} ]; then
  xmlstarlet ed -u "//GAR-CONFIG/AAFPATH" -v "${AAFPATH}${DATE}/" ${CONFIG} > /tmp/gar-config.xml
  cp /tmp/gar-config.xml ${CONFIG}
  AAFPATH="${AAFPATH}${DATE}/"

else
  cp ${CONFIG}        ${XSLPATH}/config.xml
fi


cp ficAlimMENESR.dtd ${AAFPATH}
cp ${CONFIG}        ${XSLPATH}/config.xml
echo "--------------------------------------------------------"
echo "[ ENT: ${ENT} | Mode : ${MODE} | Anonymat : ${ANONYMAT} | Thread: ${THREAD} ]"
echo "Source  : ${AAFPATH}"
echo "Dest.   : ${GARPATH}"
echo "Version script : ${SCRIPTVERSION}" 
echo "Version XSD    : ${VERSION}"
echo "--------------------------------------------------------"

# On demande a sauvegarder la config passée par ligne de commande ?
[ $SAVE -eq 1 ] && cp ${CONFIG} config.xml

# Verification du dossier AAF
[ ! -d $AAFPATH ] && echo " Le dossier AAF $AAFPATH n'éxiste pas " && exit 2
[ -z "$GARPATH" ] && GARPATH="/tmp/GAR-XML"
[ -z "$TMPPATH" ] && TMPPATH="/tmp/GAR-TMP"


# Création du dossier de destination du GAR si inexistant et suppression contenu si  existant
[ ! -d $GARPATH ] && mkdir -p $GARPATH
/bin/rm -rf $GARPATH/*

# Création du dossier temporaire  et suppression contenu si  existant
[ ! -d $TMPPATH ] && mkdir -p $TMPPATH
/bin/rm -rf $TMPPATH/*

# Mode THREAD => utilisation de python pour paralléliser les traitements
if [ $THREAD -eq 1 ] ; then
	python threads/doAll.py ${TMPPATH} ${XSLPATH} && STATUTGLOBAL='OK' || STATUTGLOBAL="KO"
else

	# transformation : Etablissements : 1 - Mefs sans tri
	debut "Mefs sans tri "
	java net.sf.saxon.Transform -o:${TMPPATH}GAR_TMP_MEFS_unsorted.xml -s:${XSLPATH}GAR-ENT-Mef.xsl -xsl:${XSLPATH}GAR-ENT-Mef.xsl \
		&& STATUTEXTRACTMEFS='OK' \
		|| STATUTEXTRACTMEFS='KO' STATUTGLOBAL="KO"
	TIMEEXTRACTMEFS=`date '+%k:%M:%S'`
	fin

	# transformation : Etablissements : 2 - Matieres sans tri
	debut "Matieres sans tri "
	java net.sf.saxon.Transform -o:${TMPPATH}GAR_TMP_Matieres_unsorted.xml -s:${XSLPATH}GAR-ENT-Matiere.xsl -xsl:${XSLPATH}GAR-ENT-Matiere.xsl \
		&& STATUTEXTRACTMATIERES='OK' \
		|| STATUTEXTRACTMATIERES='KO' STATUTGLOBAL="KO"
	TIMEEXTRACTMATIERES=`date '+%k:%M:%S'`
	fin

	# transformation : Etablissements : 3 - Tri des Mefs
	debut "Mefs tri "
	java net.sf.saxon.Transform -o:${TMPPATH}GAR_TMP_MEFS_sorted.xml -s:${TMPPATH}GAR_TMP_MEFS_unsorted.xml -xsl:${XSLPATH}GAR-ENT-Mef-TRI.xsl \
		&& STATUTTRIMEFS='OK' \
		|| STATUTTRIMEFS='KO' STATUTGLOBAL="KO"
	TIMETRIMEFS=`date '+%k:%M:%S'`
	fin

	# transformation : Etablissements : 4 - Tri des Matieres
	debut "Matieres tri"
	java net.sf.saxon.Transform -o:${TMPPATH}GAR_TMP_Matieres_sorted.xml -s:${TMPPATH}GAR_TMP_Matieres_unsorted.xml -xsl:${XSLPATH}GAR-ENT-Matiere-TRI.xsl \
		&& STATUTTRIMATIERES='OK' \
		|| STATUTTRIMATIERES='KO' STATUTGLOBAL="KO"
	TIMETRIMATIERES=`date '+%k:%M:%S'`
	fin

	# transformation : Etablissements : 5 - Etape finale
	# /!\ le nom temporaire prédictible du fichier des étabs est une nécessité pour le traitement des groupes et le controle de coherence des MEFS déclarés chez les élèves et les enseignants.
	debut "Etab"
	java net.sf.saxon.Transform -o:${TMPPATH}GAR_TMP_Etab.xml -s:${XSLPATH}GAR-ENT-Etab.xsl -xsl:${XSLPATH}GAR-ENT-Etab.xsl \
		&& STATUTETABLISSEMENTS='OK' \
		|| STATUTETABLISSEMENTS='KO' STATUTGLOBAL="KO"
	TIMEETABLISSEMENTS=`date '+%k:%M:%S'`
	fin

	# transformation : Eleves
	debut "Eleves"
	java net.sf.saxon.Transform -o:${TMPPATH}Source_Eleves.xml -s:${XSLPATH}GAR-ENT-Eleve.xsl -xsl:${XSLPATH}GAR-ENT-Eleve.xsl \
		&& STATUTELEVES='OK' \
		|| STATUTELEVES='KO' STATUTGLOBAL="KO"
	TIMEELEVES=`date '+%k:%M:%S'`
	fin

	# transformation : Enseignants : 1 - PersonMefs sans tri
	debut "PersonMefs sans tri"
	java net.sf.saxon.Transform -o:${TMPPATH}GAR_TMP_PersonMef_unsorted.xml -s:${XSLPATH}GAR-ENT-Enseignant-PersonMEF-noTRI.xsl -xsl:${XSLPATH}GAR-ENT-Enseignant-PersonMEF-noTRI.xsl \
		&& STATUTEXTRACTPERSONMEFS='OK' \
		|| STATUTEXTRACTPERSONMEFS='KO' STATUTGLOBAL="KO"
	TIMEEXTRACTPERSONMEFS=`date '+%k:%M:%S'`
	fin

	# transformation : Enseignants : 2 - Tri des PersonMefs
	debut "Tri des PersonMefs"
	java net.sf.saxon.Transform -o:${TMPPATH}GAR_TMP_PersonMef_sorted.xml -s:${TMPPATH}GAR_TMP_PersonMef_unsorted.xml -xsl:${XSLPATH}GAR-ENT-Enseignant-PersonMEF-TRI.xsl \
		&& STATUTTRIPERSONMEFS='OK' \
		|| STATUTTRIPERSONMEFS='KO' STATUTGLOBAL="KO"
	TIMETRIPERSONMEFS=`date '+%k:%M:%S'`
	fin

	# transformation : Enseignants : 3 - Etape finale
	debut "Enseignants"
	java net.sf.saxon.Transform -o:${TMPPATH}Source_Enseignants.xml -s:${XSLPATH}GAR-ENT-Enseignant.xsl -xsl:${XSLPATH}GAR-ENT-Enseignant.xsl \
		&& STATUTENSEIGNANTS='OK' \
		|| STATUTENSEIGNANTS='KO' STATUTGLOBAL="KO"
	TIMEENSEIGNANTS=`date '+%k:%M:%S'`
	fin


	# transformation : Groupes : 1 : extraction des donnees AAF sans tri
	debut "Groupes sans tri"
	java net.sf.saxon.Transform -o:${TMPPATH}GAR_TMP_Groupe_unsorted.xml -s:${XSLPATH}GAR-ENT-Groupes-noTRI.xsl -xsl:${XSLPATH}GAR-ENT-Groupes-noTRI.xsl \
		&& STATUTEXTRACTGROUPES='OK' \
		|| STATUTEXTRACTGROUPES='KO' STATUTGLOBAL="KO"
	TIMEEXTRACTGROUPES=`date '+%k:%M:%S'`
	fin

	# transformation : Groupes : 2 : tri des donnees extraites
	debut "Groupes tri"
	java net.sf.saxon.Transform -o:${TMPPATH}GAR_TMP_Groupe_sorted.xml -s:${TMPPATH}GAR_TMP_Groupe_unsorted.xml -xsl:${XSLPATH}GAR-ENT-Groupes-TRI.xsl \
		&& STATUTTRIGROUPES='OK' \
		|| STATUTTRIGROUPES='KO' STATUTGLOBAL="KO"
	TIMETRIGROUPES=`date '+%k:%M:%S'`
	fin

	# transformation : Groupes : 3 : deduplication des donnees et construction schema GAR
	debut "Groupes"
	java net.sf.saxon.Transform -o:${TMPPATH}Source_Groupes.xml -s:${TMPPATH}GAR_TMP_Groupe_sorted.xml -xsl:${XSLPATH}GAR-ENT-Groupe.xsl \
		&& STATUTGROUPES='OK' \
		|| STATUTGROUPES='KO' STATUTGLOBAL="KO"
	TIMEGROUPES=`date '+%k:%M:%S'`
	fin
fi

# finalisation nom du fichier etab
# /!\ le nom temporaire prédictible du fichier des étabs est une nécessité pour le traitement des groupes
mv ${TMPPATH}GAR_TMP_Etab.xml ${TMPPATH}Source_Etabs.xml

# Vérification qu'il y a vraiment des données, avant de découper et de les envoyer sur le dépots
nbEtabs=$(grep "<men:GAREtab>" ${TMPPATH}Source_Etabs.xml | wc -l)
[ $nbEtabs -eq 0 ] && echo "Aucun établissement trouvé, vérifier que $AAFPATH contient bien des AAF établissement avec le motif $etabs" && exit 1

nbEleves=$(grep "<men:GAREleve>" ${TMPPATH}Source_Eleves.xml | wc -l)
[ $nbEleves -lt 10 ] && echo "Moins de 1O élèves trouvés, vérifier que $AAFPATH contient bien des AAF élève avec le motif $eleves" && exit 1

nbEnseignants=$(grep "<men:GAREnseignant>" ${TMPPATH}Source_Enseignants.xml  | wc -l)
[ $nbEnseignants -lt 10 ] && echo "Moins de 1O enseignants trouvés, vérifier que $AAFPATH contient bien des AAF PersEducNat avec le motif $personnels" && exit 1

# transformation : Responsables d'affectation
# extraction des DIR :
java net.sf.saxon.Transform -o:${TMPPATH}Source_RespAffs_DIR.xml -s:${XSLPATH}GAR-ENT-RespAff.xsl -xsl:${XSLPATH}GAR-ENT-RespAff.xsl \
    && STATUTEXTRACTAAF='OK' \
    || STATUTEXTRACTAAF='KO' STATUTGLOBAL="KO"
TIMEEXTRACTAAF=`date '+%k:%M:%S'`

# Réception des RespAff NON DIR déclarés dans econnect :
if [ ! -z "$WEBSERVICE" ]; then
    curl ${WEBSERVICE}/api/gar/${MODE}/respAff?anonymat=${ANONYMAT} -F token=`echo -n ${TOKEN}` > ${TMPPATH}Source_RespAffs_nonDIR.xml
    # Controle du contenu du fichier RespAff NON DIR : fichier vide ou non
    if [ -s ${TMPPATH}Source_RespAffs_nonDIR.xml ]; then
        # fichier non vide
        STATUTEXTRACTWS='OK'
      else
        # fichier vide
        STATUTEXTRACTWS='KO'
        STATUTGLOBAL='KO'
    fi
    TIMEEXTRACTWS=`date '+%k:%M:%S'`

    # Fusion des Source_RespAff_*.xml
    # Regroupement des RespAff DIR et Non DIR :
    java net.sf.saxon.Transform -o:${TMPPATH}Source_RespAffs_doublons.xml -s:${XSLPATH}GAR-ENT-RespAff_merge.xsl -xsl:${XSLPATH}GAR-ENT-RespAff_merge.xsl \
        && STATUTRESPAFF='OK' \
        || STATUTRESPAFF='KO' STATUTGLOBAL="KO" 
    TIMERESPAFF=`date '+%k:%M:%S'`
	
	# correctif #28336 Arnaud Grossir
	# Dédoublonnage des DIR (ramenés via parsing des xml + ceux cochés dans econnect)
    if [ "${STATUTRESPAFF}"=="OK" ]
    then
        java net.sf.saxon.Transform -o:${TMPPATH}Source_RespAffs.xml -s:${TMPPATH}Source_RespAffs_doublons.xml -xsl:${XSLPATH}GAR-ENT-RespAff_deduplicate.xsl \
             && STATUTRESPAFF='OK' \
            || STATUTRESPAFF='KO' STATUTGLOBAL="KO" 
        TIMERESPAFF=`date '+%k:%M:%S'`
    fi	
	
else
	# Pas de webService  fichier des respAff = RespAffs_DIR 
	cp ${TMPPATH}Source_RespAffs_DIR.xml ${TMPPATH}Source_RespAffs.xml
	TIMEEXTRACTWS=`date '+%k:%M:%S'`
	TIMERESPAFF=`date '+%k:%M:%S'`
fi

CHECKELEVES="nd"
TIMECHECKELEVES=`date '+%k:%M:%S'`
CHECKENSEIGNANTS="nd"
TIMECHECKENSEIGNANTS=`date '+%k:%M:%S'`
CHECKETABS="nd"
TIMECHECKETABS=`date '+%k:%M:%S'`
CHECKGROUPES="nd"
TIMECHECKGROUPES=`date '+%k:%M:%S'`
CHECKRESPAFF="nd"
TIMECHECKRESPAFF=`date '+%k:%M:%S'`

# A partir de la , on force le mode thread
# pour paralléliser les traitements de xmlstarlet et le split
THREAD=1

if [ $CHECK -eq 1 ]
then
    if [ $THREAD -eq 0 ]
	then
		debut "Check"
		# bonus on teste la validité des fichiers :
		xmlstarlet val -e -s GAR-ENT-${VERSXSD}.xsd ${TMPPATH}Source_Eleves.xml 2>>${TMPPATH}Validation_Eleves.txt \
		&& CHECKELEVES='OK' \
		|| CHECKELEVES='KO' STATUTGLOBAL="KO"
		TIMECHECKELEVES=`date '+%k:%M:%S'`

		xmlstarlet val -e -s GAR-ENT-${VERSXSD}.xsd ${TMPPATH}Source_Enseignants.xml 2>>${TMPPATH}Validation_Enseignants.txt \
		&& CHECKENSEIGNANTS='OK' \
		|| CHECKENSEIGNANTS='KO' STATUTGLOBAL="KO"
		TIMECHECKENSEIGNANTS=`date '+%k:%M:%S'`

		xmlstarlet val -e -s GAR-ENT-${VERSXSD}.xsd ${TMPPATH}Source_Etabs.xml 2>>${TMPPATH}Validation_Etabs.txt \
		&& CHECKETABS='OK' \
		|| CHECKETABS='KO' STATUTGLOBAL="KO"
		TIMECHECKETABS=`date '+%k:%M:%S'`

		xmlstarlet val -e -s GAR-ENT-${VERSXSD}.xsd ${TMPPATH}Source_Groupes.xml 2>>${TMPPATH}Validation_Groupes.txt \
		&& CHECKGROUPES='OK' \
		|| CHECKGROUPES='KO' STATUTGLOBAL="KO"
		TIMECHECKGROUPES=`date '+%k:%M:%S'`

		xmlstarlet val -e -s GAR-ENT-${VERSXSD}.xsd ${TMPPATH}Source_RespAffs.xml 2>>${TMPPATH}Validation_RespAff.txt \
		&& CHECKRESPAFF='OK' \
		|| CHECKRESPAFF='KO' STATUTGLOBAL="KO"
		TIMECHECKRESPAFF=`date '+%k:%M:%S'`
		fin
    fi
fi

# découpage en portions de 10000 noeuds par fichier
if [ $THREAD -eq 0 ]; then
	debut "Split"
	java net.sf.saxon.Transform -ext:on -o:${TMPPATH}null.xml -s:${TMPPATH}Source_Eleves.xml -xsl:${XSLPATH}split_GAR-ENT-Eleve.xsl basename=${ENT}_GAR-ENT_Complet_${DATE}_${TIME}_Eleve \
		&& SPLITELEVES='OK' \
		|| SPLITELEVES='KO' STATUTGLOBAL="KO"
		TIMESPLITELEVES=`date '+%k:%M:%S'`
	java net.sf.saxon.Transform -ext:on -o:${TMPPATH}null.xml -s:${TMPPATH}Source_Enseignants.xml -xsl:${XSLPATH}split_GAR-ENT-Enseignant.xsl basename=${ENT}_GAR-ENT_Complet_${DATE}_${TIME}_Enseignant \
		&& SPLITENSEIGNANTS='OK' \
		|| SPLITENSEIGNANTS='KO' STATUTGLOBAL="KO"
		TIMESPLITENSEIGNANTS=`date '+%k:%M:%S'`
	java net.sf.saxon.Transform -ext:on -o:${TMPPATH}null.xml -s:${TMPPATH}Source_Etabs.xml -xsl:${XSLPATH}split_GAR-ENT-Etab.xsl basename=${ENT}_GAR-ENT_Complet_${DATE}_${TIME}_Etab \
		&& SPLITETABS='OK' \
		|| SPLITETABS='KO' STATUTGLOBAL="KO"
		TIMESPLITETABS=`date '+%k:%M:%S'`
	java net.sf.saxon.Transform -ext:on -o:${TMPPATH}null.xml -s:${TMPPATH}Source_Groupes.xml -xsl:${XSLPATH}split_GAR-ENT-Groupe.xsl basename=${ENT}_GAR-ENT_Complet_${DATE}_${TIME}_Groupe \
		&& SPLITGROUPES='OK' \
		|| SPLITGROUPES='KO' STATUTGLOBAL="KO"
		TIMESPLITGROUPES=`date '+%k:%M:%S'`
	java net.sf.saxon.Transform -ext:on -o:${TMPPATH}null.xml -s:${TMPPATH}Source_RespAffs.xml -xsl:${XSLPATH}split_GAR-ENT-RespAff.xsl basename=${ENT}_GAR-ENT_Complet_${DATE}_${TIME}_RespAff \
		&& SPLITRESPAFF='OK' \
		|| SPLITRESPAFF='KO' STATUTGLOBAL="KO"
		TIMESPLITRESPAFF=`date '+%k:%M:%S'`
	fin
else
	python threads/doSplit.py  ${TMPPATH} ${XSLPATH} ${ENT} ${DATE}_${TIME} && SPLITGLOBAL='OK' || SPLITGLOBAL='KO' STATUTGLOBAL='KO'
fi

# CHECK demandé et mode Thread
# On va lancer en // la validation sur tous les xml
if [ $CHECK -eq 1 ]
then
    if [ $THREAD -eq 1 ]
	then
		python threads/doCheck.py ${GARPATH} ${XSLPATH}
	fi
fi


cd ${GARPATH}

# construction de l'archive
tar -czvf ${ENT}_GAR-ENT_Complet_${DATE}_${TIME}.tar.gz ${ENT}_GAR-ENT_Complet_${DATE}_${TIME}_*.xml \
	&& ARCHIVE='OK' \
	|| ARCHIVE='KO' STATUTGLOBAL="KO"
	TIMEARCHIVE=`date '+%k:%M:%S'`

ARCHIVEMD5=`md5sum ${ENT}_GAR-ENT_Complet_${DATE}_${TIME}.tar.gz | awk '{ print $1 }'`

FINDATE=`date +'%A %d %B %Y'`
FINTIME=`date +%k:%M:%S`



cat > ${GARPATH}${ENT}_GAR-ENT_Complet_${DATE}_${TIME}.xml <<_EOF_
<?xml version="1.0" encoding="UTF-8"?>
<CompteRendu>
<STATUT>${STATUTGLOBAL}</STATUT>
<VERSION>${SCRIPTVERSION}</VERSION>
<XSD>${VERSXSD}</XSD>
<DEBUT>
	<DATE>${DEBUTDATE}</DATE>
	<TIMER>${DEBUTTIME}</TIMER>
</DEBUT>
<GAR-ELEVES>
	<STATUT>${STATUTELEVES}</STATUT>
	<TIMER>${TIMEELEVES}</TIMER>
</GAR-ELEVES>
<ENSEIGNANTS>
	<EXTRACTPERSONMEFS>
		<STATUT>${STATUTEXTRACTPERSONMEFS}</STATUT>
		<TIMER>${TIMEEXTRACTPERSONMEFS}</TIMER>
	</EXTRACTPERSONMEFS>
	<TRIPERSONMEFS>
		<STATUT>${STATUTTRIPERSONMEFS}</STATUT>
		<TIMER>${TIMETRIPERSONMEFS}</TIMER>
	</TRIPERSONMEFS>
	<GAR-ENSEIGNANTS>
		<STATUT>${STATUTENSEIGNANTS}</STATUT>
		<TIMER>${TIMEENSEIGNANTS}</TIMER>
	</GAR-ENSEIGNANTS>
</ENSEIGNANTS>
<ETABLISSEMENTS>
	<EXTRACTMEFS>
		<STATUT>${STATUTEXTRACTMEFS}</STATUT>
		<TIMER>${TIMEEXTRACTMEFS}</TIMER>
	</EXTRACTMEFS>
	<EXTRACTMATIERES>
		<STATUT>${STATUTEXTRACTMATIERES}</STATUT>
		<TIMER>${TIMEEXTRACTMATIERES}</TIMER>
	</EXTRACTMATIERES>
	<TRIMEFS>
		<STATUT>${STATUTTRIMEFS}</STATUT>
		<TIMER>${TIMETRIMEFS}</TIMER>
	</TRIMEFS>
	<TRIMATIERES>
		<STATUT>${STATUTTRIMATIERES}</STATUT>
		<TIMER>${TIMETRIMATIERES}</TIMER>
	</TRIMATIERES>
	<GAR-ETABLISSEMENTS>
		<STATUT>${STATUTETABLISSEMENTS}</STATUT>
		<TIMER>${TIMEETABLISSEMENTS}</TIMER>
	</GAR-ETABLISSEMENTS>
</ETABLISSEMENTS>
<GROUPES>
	<EXTRACTGROUPES>
		<STATUT>${STATUTEXTRACTGROUPES}</STATUT>
		<TIMER>${TIMEEXTRACTGROUPES}</TIMER>
	</EXTRACTGROUPES>
	<TRIGROUPES>
		<STATUT>${STATUTTRIGROUPES}</STATUT>
		<TIMER>${TIMETRIGROUPES}</TIMER>
	</TRIGROUPES>
	<GAR-GROUPES>
		<STATUT>${STATUTGROUPES}</STATUT>
		<TIMER>${TIMEGROUPES}</TIMER>
	</GAR-GROUPES>
</GROUPES>
<RESPAFF>
	<EXTRACTAAF>
		<STATUT>${STATUTEXTRACTAAF}</STATUT>
		<TIMER>${TIMEEXTRACTAAF}</TIMER>
	</EXTRACTAAF>
	<EXTRACTWS>
		<STATUT>${STATUTEXTRACTWS}</STATUT>
		<TIMER>${TIMEEXTRACTWS}</TIMER>
	</EXTRACTWS>
	<GAR-RESPAFF>
		<STATUT>${STATUTRESPAFF}</STATUT>
		<TIMER>${TIMERESPAFF}</TIMER>
	</GAR-RESPAFF>
</RESPAFF>
<CHECK>
        <CHECK-ELEVES>
                <STATUT>${CHECKELEVES}</STATUT>
                <TIMER>${TIMECHECKELEVES}</TIMER>
        </CHECK-ELEVES>
        <CHECK-ENSEIGNANTS>
                <STATUT>${CHECKENSEIGNANTS}</STATUT>
                <TIMER>${TIMECHECKENSEIGNANTS}</TIMER>
        </CHECK-ENSEIGNANTS>
        <CHECK-ETABS>
                <STATUT>${CHECKETABS}</STATUT>
                <TIMER>${TIMECHECKETABS}</TIMER>
        </CHECK-ETABS>
        <CHECK-GROUPES>
                <STATUT>${CHECKGROUPES}</STATUT>
                <TIMER>${TIMECHECKGROUPES}</TIMER>
        </CHECK-GROUPES>
        <CHECK-RESPAFF>
                <STATUT>${CHECKRESPAFF}</STATUT>
                <TIMER>${TIMECHECKRESPAFF}</TIMER>
        </CHECK-RESPAFF>		
</CHECK>
<SPLIT>
	<SPLIT-ELEVES>
		<STATUT>${SPLITELEVES}</STATUT>
		<TIMER>${TIMESPLITELEVES}</TIMER>
	</SPLIT-ELEVES>
	<SPLIT-ENSEIGNANTS>
		<STATUT>${SPLITENSEIGNANTS}</STATUT>
		<TIMER>${TIMESPLITENSEIGNANTS}</TIMER>
	</SPLIT-ENSEIGNANTS>
	<SPLIT-ETABS>
		<STATUT>${SPLITETABS}</STATUT>
		<TIMER>${TIMESPLITETABS}</TIMER>
	</SPLIT-ETABS>
	<SPLIT-GROUPES>
		<STATUT>${SPLITGROUPES}</STATUT>
		<TIMER>${TIMESPLITGROUPES}</TIMER>
	</SPLIT-GROUPES>
	<SPLIT-RESPAFF>
		<STATUT>${SPLITRESPAFF}</STATUT>
		<TIMER>${TIMESPLITRESPAFF}</TIMER>
	</SPLIT-RESPAFF>
</SPLIT>
<ARCHIVE>
	<MD5>${ARCHIVEMD5}</MD5>
	<STATUT>${ARCHIVE}</STATUT>
	<TIMER>${TIMEARCHIVE}</TIMER>
</ARCHIVE>
<FIN>
	<DATE>${FINDATE}</DATE>
	<TIMER>${FINTIME}</TIMER>
</FIN>
</CompteRendu>
_EOF_


if [ ! -z "$WEBSERVICE" ]; then
	echo -e "${TURQUOISE_SOMBRE}------------------------------> DEBUT <---------------------------------${NORMAL}"

	# curl ${WEBSERVICE}/api/gar/[test|prod]/archive \
	#    > -F token=${TOKEN} \
	#    > -F archive=@W${ENT}_GAR-ENT_Complet_${DATE}_${TIME}.tar.gz
	echo -e "${TURQUOISE}curl ${WEBSERVICE}/api/gar/${MODE}/archive \\"
	echo -e " -F token=${TOKEN} \\"
	echo -e " -F md5=${ARCHIVEMD5} \\"
	echo -e " -F archive=@${GARPATH}${ENT}_GAR-ENT_Complet_${DATE}_${TIME}.tar.gz \\"
	echo -e " -F log=@${GARPATH}${ENT}_GAR-ENT_Complet_${DATE}_${TIME}.xml"

	echo -e "${TURQUOISE_SOMBRE}------------------------------> DEBUT <---------------------------------${NORMAL}"

	if [ $SEND -eq 1 ]; then
	echo "Envoi sur le dépot "
	curl ${WEBSERVICE}/api/gar/${MODE}/archive \
	-F token=${TOKEN} \
	-F md5=${ARCHIVEMD5} \
	-F archive=@${GARPATH}${ENT}_GAR-ENT_Complet_${DATE}_${TIME}.tar.gz \
	-F log=@${GARPATH}${ENT}_GAR-ENT_Complet_${DATE}_${TIME}.xml
	[ $? -ne 0 ] && "STATUTGLOBAL" = "KO"
	fi
else
	echo ""
	echo -e "${TURQUOISE_SOMBRE}------------------------------> DEBUT <---------------------------------${NORMAL}"
	echo -e "Archive: ${GARPATH}${ENT}_GAR-ENT_Complet_${DATE}_${TIME}.tar.gz "
	echo -e "MD5    : ${ARCHIVEMD5}"
	echo -e "${TURQUOISE_SOMBRE}------------------------------> DEBUT <---------------------------------${NORMAL}"
fi


if [ "${STATUTGLOBAL}" = 'OK' ]
    then
        exit 0
    else
        exit 1
fi
