<?php

/**
 * ICS.php
 * =======
 * Use this class to create an .ics file.
 *
 * Usage
 * -----
 * Basic usage - generate ics file contents (see below for available properties):
 *   $ics = new ICS($props);
 *   $ics_file_contents = $ics->to_string();
 *
 * Setting properties after instantiation
 *   $ics = new ICS();
 *   $ics->set('summary', 'My awesome event');
 *
 * You can also set multiple properties at the same time by using an array:
 *   $ics->set(array(
 *     'dtstart' => 'now + 30 minutes',
 *     'dtend' => 'now + 1 hour'
 *   ));
 *
 * Available properties
 * --------------------
 * description
 *   String description of the event.
 * dtend
 *   A date/time stamp designating the end of the event. You can use either a
 *   DateTime object or a PHP datetime format string (e.g. "now + 1 hour").
 * dtstart
 *   A date/time stamp designating the start of the event. You can use either a
 *   DateTime object or a PHP datetime format string (e.g. "now + 1 hour").
 * location
 *   String address or description of the location of the event.
 * summary
 *   String short summary of the event - usually used as the title.
 * url
 *   A url to attach to the the event. Make sure to add the protocol (http://
 *   or https://).
 */

class ICS {
  const DT_FORMAT = 'Ymd\THis';
  const DT_FORMATDAY = 'Ymd';

  protected $properties = array();
  private $fgdebug;
  private $available_properties = array(
    'description',
    'dtend',
    'dtstart',
    'location',
    'summary',
    'url',
    'allday',
    'UID'
  );

  public function __construct($props) {
    $this->set($props);
  }

  public function debug($fgdebug = false) {
    $this->fgdebug=$fgdebug;
  }
  public function set($key, $val = false) {
    if (is_array($key)) {
      foreach ($key as $k => $v) {
        if(is_null($v)||$v=="") unset( $this->properties[$k]);
        else $this->set($k, $v);
      }
    } else {
      if (in_array($key, $this->available_properties)) {
        $this->properties[$key] = $this->sanitize_val($val, $key,$this->properties["allday"]);
      }
    }
  }

  public function writeheader() {
    // Build ICS properties - add header
    $header = array(
      'BEGIN:VCALENDAR',
      'VERSION:2.0',
      'PRODID:-//hacksw/handcal//NONSGML v1.0//EN',
      'CALSCALE:GREGORIAN',
      'X-WR-TIMEZONE:Europe/Paris'
    );

    return $this->exportformat($header);
  }

  public function writefooter() {
    // Build ICS properties - add footer
    $footer[] = 'END:VCALENDAR';

    return $this->exportformat($footer);
  }

  public function writeevent() {
    $event[]='BEGIN:VEVENT';

    // Build ICS properties - add header
    $props = array();
    foreach($this->properties as $k => $v) {
      if($k === 'url')
        $props[strtoupper($k . ';VALUE=URI')] = $v;
      elseif(($k=='dtstart'||$k=='dtend')&&$this->properties['allday'])
        $props[strtoupper($k . ';VALUE=DATE')] = $v;
      else
      $props[strtoupper($k)] = $v;
    }

    // Set some default values
    $props['DTSTAMP'] = $this->format_timestamp('now');
    if(!isset($props['UID'])) $props['UID'] = uniqid();

    // Append properties
    foreach ($props as $k => $v) {
      $event[] = "$k:$v";
    }

    // Build ICS properties - add footer
    $event[] = 'END:VEVENT';

    return $this->exportformat($event);
  }

  private function sanitize_val($val, $key = false, $allday) {
    switch($key) {
      case 'dtstamp':
        $val = $this->format_timestamp($val);
        break;

      case 'dtend':
      case 'dtstart':
        if($allday)
          $val = $this->format_daystamp($val);
        else
          $val = $this->format_timestamp($val);
        break;

      default:
        $val = $this->escape_string($val);
    }

    return $val;
  }

  private function format_daystamp($timestamp) {
    $dt = new DateTime($timestamp, new DateTimeZone('Europe/Paris'));
    return $dt->format(self::DT_FORMATDAY);
  }

  private function format_timestamp($timestamp) {
    $dt = new DateTime($timestamp, new DateTimeZone('Europe/Paris'));
    return $dt->format(self::DT_FORMAT);
  }

  private function escape_string($str) {
    return preg_replace('/([\,;])/','\\\$1', $str);
  }

  private function exportformat($string) {
    return implode(($this->fgdebug?"<br>":"\r\n"), $string).($this->fgdebug?"<br>":"\r\n");
  }
}
