<?php

namespace App\Controller;

use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Form\FormError;
use Symfony\Component\ExpressionLanguage\SyntaxError;
use Symfony\Component\Finder\Finder;
use Symfony\Component\Filesystem\Exception\IOExceptionInterface;
use Symfony\Component\Filesystem\Filesystem;

use App\Service\JanusService;
use Symfony\Component\Yaml\Yaml;
use App\DependencyInjection\JanusExtension;
use App\Form\JanusType;

class JanusController extends AbstractController
{
    private $janus;
    
    public function __construct(JanusService $janus) { $this->janus = $janus; }
        
    public function view(Request $request)
    {
        // Initialisation du mode test
        $this->janus->initTest();

        // Récupération de la config
        $config=$this->janus->getConfig();
        $testuid=$config["testuid"];
        $attributescache=$config["attributescache"];
        $ressourcescache=$config["ressourcescache"];
       
        $customers=$this->janus->getCustomers();
        $providers=$this->janus->getProviders();
        $attributes=$this->janus->getAttributes();
        $functions=$this->janus->getFunctions();

        $attributesrender=$this->janus->getAttributesrender();
        $rules=$this->janus->getRules();
        $ressourcestype=$config["ressourcestype"];
        $ressources=$this->janus->getRessources();

        // Configuration en cours ?
        $updated=$this->janus->isUpdated();
        
        return $this->render('Janus/view.html.twig',[
            "useheader"         => true,
            "usesidebar"        => true,

            "updated"           => $updated,
            "config"            => $config,

            "testuid"           => $testuid,
            "attributescache"   => $attributescache,
            "ressourcescache"   => $ressourcescache,

            "customers"         => $customers,
            "providers"         => $providers,
            "attributes"        => $attributes,
            "functions"         => $functions,
            "rules"             => $rules,
            "attributesrender"  => $attributesrender,
            "ressourcestype"    => $ressourcestype,
            "ressources"        => $ressources,
        ]);
    }

    public function order(Request $request)
    {
        // S'assurer que c'est un appel ajax
        if (!$request->isXmlHttpRequest()) {
            return new JsonResponse(array('message' => 'Interdit'), 400);
        }

        // Récupération des paramètres
        $tbparam=explode("-",$request->request->get('type'));
        $type=$tbparam[0];
        $order=$request->request->get('order');

        // Initialisation du mode test
        $this->janus->initTest();        
        $config=$this->janus->getConfig();

        switch($type) {
            case "provider":
                $config["providersorder"]=array_merge(["sso"],$order);        
            break;
            case "rule":
                $config["rulesorder"]=$order;        
            break;       
            case "attribute":
                $config["attributesrender"][$tbparam[1]]=$order;        
            break;   
            case "type":
                $config["ressourcestype"]=$order;        
            break;
            
            // order de ressources
            default:
                $type=str_replace("ressource-","",$type);
                $index=0;
                foreach($order as $name) {
                    $index++;
                    $config["ressources"][$type][$name]["order"]=$index;
                }
            break;
        }

        // Update
        $updated=$this->janus->updateConfig($config);  
        
        return new Response($updated);
    }

    public function getForm($type,$subtype,&$help,&$keys,&$deletable) {
        switch($type) {
            case "testuid": 
                $help="Indiquez un uid qui sera automatiquement tester avant de pouvoir appliquer votre nouvelle configuration";
                $deletable=true;
                $keys=[
                    [
                        "id"=>"name",
                        "type"=>"string",
                        "required"=>true,
                        "updatable"=>false,
                        "value"=>null,
                    ],[
                        "id"=>"customer",
                        "type"=>"customer",
                        "required"=>true,
                        "updatable"=>true,
                        "value"=>null,
                    ],[
                        "id"=>"ssoattributes",
                        "type"=>"ssoattributes",
                        "required"=>false,
                        "updatable"=>true,
                        "value"=>null,
                    ]
                ];
            break;

            case "customers": 
                $help="Clé d'accès du client";
                $deletable=true;
                $keys=[
                    [
                        "id"=>"name",
                        "type"=>"string",
                        "required"=>true,
                        "updatable"=>false,
                        "value"=>null,
                    ],[
                        "id"=>"apikey",
                        "type"=>"string",
                        "required"=>true,
                        "updatable"=>true,
                        "value"=>null,
                    ]
                ];
            break;

            case "attributescache": 
                $help="Interval de temps entre chaque refresh de cache des attributs utilisateur en secondes";
                $deletable=false;
                $keys=[
                    [
                        "id"=>"attributescache",
                        "type"=>"integer",
                        "required"=>true,
                        "updatable"=>true,
                        "value"=>null,
                    ]
                ];
            break;

            case "ressourcescache": 
                $help="Interval de temps entre chaque refresh de cache des ressources utilisateur en secondes";
                $deletable=false;
                $keys=[
                    [
                        "id"=>"ressourcescache",
                        "type"=>"integer",
                        "required"=>true,
                        "updatable"=>true,
                        "value"=>null,
                    ]
                ];
            break;

            case "providers":
                $help="Remplissez l'ensemble des informations qui permettront de dialoguer avec votre provider d'attributs";
                $deletable=true;
                $keys=[
                    [
                        "id"=>"name",
                        "type"=>"string",
                        "required"=>true,
                        "updatable"=>false,
                        "value"=>null,
                    ],[
                        "id"=>"type",
                        "type"=>"typeprovider",
                        "required"=>true,
                        "updatable"=>false,
                        "value"=>null,
                    ],[
                        "id"=>"host",
                        "type"=>"string",
                        "required"=>false,
                        "updatable"=>true,
                        "value"=>null,
                    ],[
                        "id"=>"port",
                        "type"=>"integer",
                        "required"=>false,
                        "updatable"=>true,
                        "value"=> null,
                    ],[
                        "id"=>"user",
                        "type"=>"string",
                        "required"=>false,
                        "updatable"=>true,
                        "value"=>null,
                    ],[
                        "id"=>"password",
                        "type"=>"string",
                        "required"=>false,
                        "updatable"=>true,
                        "value"=>null,
                    ],[
                        "id"=>"basedn",
                        "type"=>"string",
                        "required"=>false,
                        "updatable"=>true,
                        "value"=>null,
                    ],[
                        "id"=>"dbname",
                        "type"=>"string",
                        "required"=>false,
                        "updatable"=>true,
                        "value"=>null,
                    ],[
                        "id"=>"userquery",
                        "type"=>"string",
                        "required"=>false,
                        "updatable"=>true,
                        "value"=>null,
                    ],[
                        "id"=>"groupquery",
                        "type"=>"string",
                        "required"=>false,
                        "updatable"=>true,
                        "value"=>null,
                    ],[
                        "id"=>"groupname",
                        "type"=>"string",
                        "required"=>false,
                        "updatable"=>true,
                        "value"=>null,
                    ],[
                        "id"=>"url",
                        "type"=>"string",
                        "required"=>false,
                        "updatable"=>true,
                        "value"=>null,
                    ],[
                        "id"=>"key",
                        "type"=>"string",
                        "required"=>false,
                        "updatable"=>true,
                        "value"=>null,
                    ],[
                        "id"=>"getuser",
                        "type"=>"boolean",
                        "required"=>false,
                        "updatable"=>true,
                        "value"=>null,
                    ],[
                        "id"=>"getgroups",
                        "type"=>"boolean",
                        "required"=>false,
                        "updatable"=>true,
                        "value"=>null,
                    ],[
                        "id"=>"getressources",
                        "type"=>"getressources",
                        "required"=>false,
                        "updatable"=>true,
                        "value"=>null,
                    ]
                ];
            break;

            case "functionsubmit":
                $help="";
                $deletable=true;
                $keys=[
                    [
                        "id"=>"name",
                        "type"=>"string",
                        "required"=>true,
                        "updatable"=>false,
                        "value"=>null,
                    ]
                ];
            break;

            case "function":
                $help="";
                $deletable=true;
                $keys=[
                    [
                        "id"=>"name",
                        "type"=>"code",
                        "required"=>true,
                        "updatable"=>true,
                        "value"=>null,
                    ]
                ];
            break;

            case "attributes":
                $help="Définir un attribut de votre provider\nname = identifiant janus\nfrom = identifiant provider\ntype = type de données\ndefaut = valeur par défaut si non trouvé";
                $deletable=true;
                $keys=[
                    [
                        "id"=>"name",
                        "type"=>"string",
                        "required"=>true,
                        "updatable"=>false,
                        "value"=>null,
                    ],[
                        "id"=>"from",
                        "type"=>"string",
                        "required"=>true,
                        "updatable"=>true,
                        "value"=>null,
                    ],[
                        "id"=>"type",
                        "type"=>"typeattribute",
                        "required"=>true,
                        "updatable"=>true,
                        "value"=>null,
                    ],[
                        "id"=>"default",
                        "type"=>"string",
                        "required"=>false,
                        "updatable"=>true,
                        "value"=>null,
                    ]
                ];
            break;

            case "rulesattribute":
                $help="Définir une règle d'attribut";
                $deletable=true;
                $keys=[
                    [
                        "id"=>"name",
                        "type"=>"string",
                        "required"=>true,
                        "updatable"=>false,
                        "value"=>null,
                    ],[
                        "id"=>"to",
                        "type"=>"string",
                        "required"=>true,
                        "updatable"=>false,
                        "value"=>null,
                    ],[
                        "id"=>"if",
                        "type"=>"string",
                        "required"=>true,
                        "updatable"=>true,
                        "value"=>null,
                    ],[
                        "id"=>"then",
                        "type"=>"string",
                        "required"=>true,
                        "updatable"=>true,
                        "value"=>null,
                    ],[
                        "id"=>"else",
                        "type"=>"string",
                        "required"=>false,
                        "updatable"=>true,
                        "value"=>null,
                    ]
                ];
            break; 
            

            case "rulesgroup":
                $help="Définir une règle d'attribut";
                $deletable=true;
                $keys=[
                    [
                        "id"=>"name",
                        "type"=>"string",
                        "required"=>true,
                        "updatable"=>false,
                        "value"=>null,
                    ],[
                        "id"=>"if",
                        "type"=>"string",
                        "required"=>true,
                        "updatable"=>true,
                        "value"=>null,
                    ],[
                        "id"=>"then",
                        "type"=>"thengroup",
                        "required"=>true,
                        "updatable"=>true,
                        "value"=>"add",
                    ],[
                        "id"=>"to",
                        "type"=>"string",
                        "required"=>true,
                        "updatable"=>true,
                        "value"=>null,
                    ]
                ];
            break;   
            
            case "attributesrender":
                $help="Définir un attribut de restituer au client";
                $deletable=true;
                $keys=[
                    [
                        "id"=>"name",
                        "type"=>"choice",
                        "required"=>true,
                        "updatable"=>false,
                        "value"=>null,
                        "choice"=>$this->janus->getFreeattributes($subtype)
                    ]
                ];
            break;            
        }
    }

    public function submit($type, Request $request) {
        // Initialisation du mode test
        $this->janus->initTest();  
        $config=$this->janus->getConfig();

        // subtype ?
        $tmp=explode("-",$type);
        $type=$tmp[0];
        $subtype=(isset($tmp[1])?$tmp[1]:null);
        
        // Construction du formulaire en fonction du type souhaité
        $this->getForm($type,$subtype,$help,$keys,$deletable);


        $data=[];
        $data["type"]=$type;
        $data["help"]=$help;
        foreach($keys as $key) {
            $data[$key["id"]]=$key["value"];
        }

        $form = $this->createForm(JanusType::class,$data,[
            "mode"=>"submit",
            "type"=>$type,
            "keys"=>$keys,
            "config"=>$config,
        ]);
        
        
        // Récupération des data du formulaire
        $form->handleRequest($request);
        
        // Controle de cohérance
        $data = $form->getData();  
        $this->getErrorForm($type,$form,$request,$data,"submit");

        // Sur validation
        if ($form->get('submit')->isClicked() && $form->isValid()) {  
            switch($type) {
                case "testuid": 
                    $name=$data["name"];
                    unset($data["type"]);
                    unset($data["name"]);
                    unset($data["help"]);

                    $config[$type][$name]=$data;
                break;

                case "customers":
                    $name=$data["name"];
                    $config[$type][$name]=$data["apikey"];

                    $config["attributesrender"][$name]=["uid","groups"];
                break;

                case "providers":
                    switch($data["type"]) {
                        case "bdd":
                            $tmp = [
                                "type" => $data["type"],
                                "host" => $data["host"],
                                "port" => $data["port"],
                                "user" => $data["user"],
                                "password" => $data["password"],
                                "dbname" => $data["dbname"],
                                "userquery" => $data["userquery"],
                                "groupquery" => $data["groupquery"],
                                "groupname" => $data["groupname"],
                            ];
                        break;

                        case "ldap":
                            $tmp = [
                                "type" => $data["type"],
                                "host" => $data["host"],
                                "port" => $data["port"],
                                "user" => $data["user"],
                                "password" => $data["password"],
                                "basedn" => $data["basedn"],
                                "userquery" => $data["userquery"],
                                "groupquery" => $data["groupquery"],
                                "groupname" => $data["groupname"],
                            ];
                        break;

                        case "ninegate":
                            $tmp = [
                                "type" => $data["type"],
                                "url" => $data["url"],
                                "key" => $data["key"],
                                "getuser" => $data["getuser"],
                                "getgroups" => $data["getgroups"],
                                "getressources" => $data["getressources"],
                            ];
                        break;                                                   
                    }

                    $config["providers"][$data["name"]]=$tmp;
                    array_push($config["providersorder"],$data["name"]);
                break;

                case "attributes":
                    $name=$data["name"];
                    unset($data["name"]);
                    unset($data["help"]);
                    if(!array_key_exists($subtype,$config[$type])) $config[$type][$subtype]=[];

                    $config[$type][$subtype][$name]=$data;                
                break;

                case "rulesattribute":
                    $name=$data["name"];
                    $data["for"]="attributes";
                    unset($data["name"]);
                    unset($data["help"]);

                    $config["rules"][$name]=$data;
                    array_push($config["rulesorder"],$name);
                break;

                case "rulesgroup":
                    $name=$data["name"];
                    $data["for"]="groups";
                    unset($data["name"]);
                    unset($data["help"]);

                    $config["rules"][$name]=$data;
                    array_push($config["rulesorder"],$name);
                break;

                case "attributesrender":
                    $name=$data["name"];
                    unset($data["name"]);
                    unset($data["help"]);
                    if(!array_key_exists($subtype,$config[$type])) $config[$type][$subtype]=[];

                    if(!in_array($name,$config[$type][$subtype])) array_push($config[$type][$subtype],$name);
                break;                
            }

            // Update
            $updated=$this->janus->updateConfig($config);              
            return $this->redirectToRoute("app_janus_view");
        }

        // Affichage du formulaire
        return $this->render('Janus/edit.html.twig', [
            'useheader'         => true,
            'usesidebar'        => true,       
            'mode'              => "submit",
            'type'              => $type,
            'id'                => null,
            'deletable'         => false,
            'form'              => $form->createView()
        ]);        
    }

    public function update($type, $id, Request $request) {
        // Initialisation du mode test
        $this->janus->initTest();  
        $config=$this->janus->getConfig();

        // subtype ?
        $tmp=explode("-",$type);
        $type=$tmp[0];
        $subtype=(isset($tmp[1])?$tmp[1]:null);
        
        // Impossible de modifier un provider sso
        if($type=="providers"&&$id=="sso")
            return $this->redirectToRoute("app_janus_view");

        // Construction du formulaire en fonction du type souhaité
        $this->getForm($type,$subtype,$help,$keys,$deletable);


        $data=[];
        $data["type"]=$type;
        $data["help"]=$help;
        $renametype=$type;
        if($renametype=="rulesattribute") $renametype="rules";
        if($renametype=="rulesgroup") $renametype="rules";

        if(is_array($config[$renametype])&&array_key_exists($id,$config[$renametype])) {
            $data["name"]=$id;
        }

        foreach($keys as $key) {
            $configtype=($subtype==null?$config[$renametype]:$config[$renametype][$subtype]);

            if(is_array($configtype)&&array_key_exists($id,$configtype)) {
                if(is_array($configtype[$id])) {
                    if(array_key_exists($key["id"],$configtype[$id])) 
                        $data[$key["id"]]=$configtype[$id][$key["id"]];
                    elseif($key["id"]=="name")
                        $data["name"]=$id;
                    else
                        $data[$key["id"]]=null;                        
                }
                else {
                    if($key["id"]=="name")
                        $data["name"]=$id;
                    else
                        $data[$key["id"]]=$configtype[$id];
                }
            }
            else
                $data[$key["id"]]=$id;
            
            if($data[$key["id"]]=="") $data[$key["id"]]=null;
        }

        $form = $this->createForm(JanusType::class,$data,[
            "mode"=>"update",
            "type"=>$type,
            "keys"=>$keys,
            "config"=>$config,
        ]);
        
        
        // Récupération des data du formulaire
        $form->handleRequest($request);
        
        // Controle de cohérance
        $data = $form->getData();  

        $this->getErrorForm($type,$form,$request,$data,"update");

        // Sur validation
        if ($form->get('submit')->isClicked() && $form->isValid()) {  
            switch($type) {
                case "testuid":
                    $name=$data["name"];
                    unset($data["type"]);
                    unset($data["name"]);
                    unset($data["help"]);
                    $config[$type][$name]=$data;
                break;

                case "attributescache": 
                    case "ressourcescache": 
                        $config[$type]=$data[$type];
                    break;
                    
                case "customers":
                    $name=$data["name"];
                    $config[$type][$name]=$data["apikey"];
                break;

                case "providers":
                    switch($data["type"]) {
                        case "bdd":
                            $tmp = [
                                "type" => $data["type"],
                                "host" => $data["host"],
                                "port" => $data["port"],
                                "user" => $data["user"],
                                "password" => $data["password"],
                                "dbnaattributme" => $data["dbname"],
                                "userquery" => $data["userquery"],
                                "groupquery" => $data["groupquery"],
                                "groupname" => $data["groupname"],
                            ];
                        break;

                        case "ldap":
                            $tmp = [
                                "type" => $data["type"],
                                "host" => $data["host"],
                                "port" => $data["port"],
                                "user" => $data["user"],
                                "password" => $data["password"],
                                "basedn" => $data["basedn"],
                                "userquery" => $data["userquery"],
                                "groupquery" => $data["groupquery"],
                                "groupname" => $data["groupname"],
                            ];
                        break;

                        case "ninegate":
                            $tmp = [
                                "type" => $data["type"],
                                "url" => $data["url"],
                                "key" => $data["key"],
                                "getuser" => $data["getuser"],
                                "getgroups" => $data["getgroups"],
                                "getressources" => $data["getressources"],
                            ];
                        break;                                                   
                    }
                    
                    $config["providers"][$data["name"]]=$tmp;
                break;


                case "attributes":
                    $name=$data["name"];
                    unset($data["name"]);
                    unset($data["help"]);
                    if(!is_array($config[$type][$subtype])) $config[$type][$subtype]=[];
                    $config[$type][$subtype][$name]=$data;                
                break;      
                
                case "rulesattribute":
                    $name=$data["name"];
                    $data["for"]="attributes";
                    unset($data["name"]);
                    unset($data["help"]);
                    if(is_null($data["else"])) unset($data["else"]);

                    $config["rules"][$name]=$data;
                break;   

                case "rulesgroup":
                    $name=$data["name"];
                    $data["for"]="groups";
                    unset($data["name"]);
                    unset($data["help"]);

                    $config["rules"][$name]=$data;
                break;                                
            }

            // Update
            $updated=$this->janus->updateConfig($config);              
            return $this->redirectToRoute("app_janus_view");
        }

        // Affichage du formulaire
        return $this->render('Janus/edit.html.twig', [
            'useheader'         => true,
            'usesidebar'        => true,       
            'mode'              => "update",
            'type'              => $type,
            'subtype'           => $subtype,
            'id'                => $id,
            'deletable'         => $deletable,
            'form'              => $form->createView()
        ]);        
    }

    public function delete($type, $id, Request $request) {
        // Initialisation du mode test
        $this->janus->initTest();  
        $config=$this->janus->getConfig();

        // subtype ?
        $tmp=explode("-",$type);
        $type=$tmp[0];
        $subtype=(isset($tmp[1])?$tmp[1]:null);

        // Impossible de supprimer un provider sso
        if($type=="providers"&&$id=="sso")
            return $this->redirectToRoute("app_janus_view");

        // Construction du formulaire en fonction du type souhaité
        $this->getForm($type,$subtype,$help,$keys,$deletable);

        // Charger les datas
        $data=[];
        $data["type"]=$type;
        $data["help"]=$help;
        $renametype=$type;
        if($renametype=="rulesattribute") $renametype="rules";
        if($renametype=="rulesgroup") $renametype="rules";

        if(is_array($config[$renametype])&&array_key_exists($id,$config[$renametype])) {
            $data["name"]=$id;
        }

        foreach($keys as $key) {
            $configtype=($subtype==null?$config[$renametype]:$config[$renametype][$subtype]);

            if(is_array($configtype)&&array_key_exists($id,$configtype)) {
                if(is_array($configtype[$id])) {
                    if(array_key_exists($key["id"],$configtype[$id])) 
                        $data[$key["id"]]=$configtype[$id][$key["id"]];
                    elseif($key["id"]=="name")
                        $data["name"]=$id;
                    else
                        $data[$key["id"]]=null;                        
                }
                else {
                    if($key["id"]=="name")
                        $data["name"]=$id;
                    else
                        $data[$key["id"]]=$configtype[$id];
                }
            }
            else
                $data[$key["id"]]=$id;
            
            if($data[$key["id"]]=="") $data[$key["id"]]=null;
        }

        $form = $this->createForm(JanusType::class,$data,[
            "mode"=>"update",
            "type"=>$type,
            "keys"=>$keys,
            "config"=>$config,
        ]);

        // Récupération des data du formulaire
        $form->handleRequest($request);

        // Controle de cohérance
        $data = $form->getData();  
        $haveerror=$this->getErrorForm($type,$form,$request,$data,"delete",$deletable);
       
        // Sur validation
        if (!$haveerror) {  
            switch($type) {
                case "testuid":
                    unset($config[$type][$data["name"]]);
                break;

                case "customers":
                    unset($config[$type][$data["name"]]);
                    unset($config["attributesrender"][$data["name"]]);
                break;

                case "providers":
                    unset($config[$type][$data["name"]]);
                    unset($config["providersorder"][$data["name"]]);
                break;

                case "attributes":
                    unset($config[$type][$subtype][$data["name"]]);

                    // Ne jamais enlever des rendus uid et groups
                    if($data["name"]!="uid"&&$data["name"]!="groups") {                    
                        // Y a-t-il d'autre provider qui utilise ce name
                        $used=false;
                        foreach($config[$type] as $provider) {
                            if(array_key_exists($data["name"],$provider)) {
                                $used=true;
                            }
                        }

                        // Y a-t-il un rule qui utilise ce name
                        foreach($config["rules"] as $key=>$rule) {
                            if($rule["for"]=="attributes") {
                                if($data["name"]==$rule["to"]) $used=true;
                            }
                        }

                        // Si non utilisé on le supprime de l'ensemble des rendus
                        if(!$used) {
                            foreach($config["attributesrender"] as $customer => $param) {
                                if (($key = array_search($data["name"], $param)) !== false) {
                                    unset($config["attributesrender"][$customer][$key]);
                                }                            
                            }
                        }
                    }
                break;                

                case "rulesattribute":
                    unset($config["rules"][$data["name"]]);
                    if (($key = array_search($data["name"], $config["rulesorder"])) !== false) {
                        unset($config["rulesorder"][$key]);
                    }
                    
                    // Ne jamais enlever des rendus uid et groups
                    if($data["to"]!="uid"&&$data["to"]!="groups") {
                        // Y a-t-il un provider qui utilise ce name
                        $used=false;
                        foreach($config["attributes"] as $provider) {
                            if(array_key_exists($data["to"],$provider)) {
                                $used=true;
                            }
                        }

                        // Y a-t-il d'autre rule qui utilise ce name
                        foreach($config["rules"] as $key=>$rule) {
                            if($rule["for"]=="attributes") {
                                if($data["to"]==$rule["to"]) $used=true;
                            }
                        }

                        // Si non utilisé on le supprime de l'ensemble des rendus
                        if(!$used) {
                            foreach($config["attributesrender"] as $customer => $param) {
                                if (($key = array_search($data["to"], $param)) !== false) {
                                    unset($config["attributesrender"][$customer][$key]);
                                }                            
                            }
                        }
                    }
                break;                

                case "rulesgroup":
                    unset($config["rules"][$data["name"]]);
                    if (($key = array_search($data["name"], $config["rulesorder"])) !== false) {
                        unset($config["rulesorder"][$key]);
                    }
                break;                 
            }

            // Update
            $updated=$this->janus->updateConfig($config);     
            return $this->redirectToRoute("app_janus_view");
        }
        
        return $this->redirectToRoute("app_janus_view_update",["type"=>$type,"id"=>$id]);
    }

    public function submitfct(Request $request) {
        // Initialisation du mode test
        $this->janus->initTest();  
        $config=$this->janus->getConfig();

        // Construction du formulaire en fonction du type souhaité
        $this->getForm("functionsubmit","",$help,$keys,$deletable);

        // Ouverture du fichier test
        $data=[];
        $data["type"]="function";
        $data["help"]=$help;
        $data["name"]="myfunction.php";

        $form = $this->createForm(JanusType::class,$data,[
            "mode"=>"submit",
            "type"=>"function",
            "keys"=>$keys,
            "config"=>$config,
        ]);

        // Récupération des data du formulaire
        $form->handleRequest($request);
        $data = $form->getData();  
        $this->getErrorForm("functionsubmit",$form,$request,$data,"submit");

        // Sur validation
        if ($form->get('submit')->isClicked() && $form->isValid()) {  
            $name=$data["name"];
            $filesystem = new Filesystem();
            $filesystem->mkdir("/var/www/html/janus/config/janus/functions/add");
            $filesystem->copy("/var/www/html/janus/scripts/janusfunction.php","/var/www/html/janus/config/janus/functions/add/".$name,true);
            return $this->redirectToRoute("app_janus_view_updatefunction",["id"=>$name]);
        }

        // Affichage du formulaire
        return $this->render('Janus/edit.html.twig', [
            'useheader'         => true,
            'usesidebar'        => true,       
            'mode'              => "submit",
            'type'              => "function",
            'subtype'           => "",
            'id'                => null,
            'deletable'         => false,
            'form'              => $form->createView()
        ]);      
    }

    public function updatefct($id, Request $request) {
        // Initialisation du mode test
        $this->janus->initTest();  
        $config=$this->janus->getConfig();

        // Construction du formulaire en fonction du type souhaité
        $this->getForm("function","",$help,$keys,$deletable);
        $reload=false;

        // Déterminer le path du fichier à modifier
        $filesystem = new Filesystem();
        if($filesystem->exists('/var/www/html/janus/config/janus/functions/add/'.$id))
            $path='/var/www/html/janus/config/janus/functions/add/'.$id;
        else {
            // On prend la version eole si la version janus n'existe pas
            if(!$filesystem->exists('/var/www/html/janus/config/janus/functions/'.$id)||$filesystem->exists('/var/www/html/janus/config/janus/functions/del/'.$id))
                $path='/var/www/html/janus/config/eole/functions/'.$id;
            else
                $path='/var/www/html/janus/config/janus/functions/'.$id;
        }
        
        // Si la fonction EOLE existe on ne peut pas supprimer mais on peut remettre la version d'origine
        if($filesystem->exists('/var/www/html/janus/config/eole/functions/'.$id)) {
            $deletable=false;
            $reload=($filesystem->exists('/var/www/html/janus/config/janus/functions/'.$id)||$filesystem->exists('/var/www/html/janus/config/janus/functions/add/'.$id));
        }

        // Ouverture du fichier test
        $data=[];
        $data["type"]="function";
        $data["help"]=$help;
        $data["name"]=file_get_contents($path);

        $form = $this->createForm(JanusType::class,$data,[
            "mode"=>"update",
            "type"=>"function",
            "keys"=>$keys,
            "config"=>$config,
        ]);

        // Récupération des data du formulaire
        $form->handleRequest($request);
        $data = $form->getData();  

        // Sur validation
        if ($form->get('submit')->isClicked() && $form->isValid()) {  
            // On place le fichier
            $name=$data["name"];
            $filesystem->mkdir('/var/www/html/janus/config/janus/functions/add');
            file_put_contents("/var/www/html/janus/config/janus/functions/add/".$id,$name);

            // Au cas il ne fait plus partie des fichiers à supprimer vu que l'on vient de le modifier
            $filesystem->remove('/var/www/html/janus/config/janus/functions/del/'.$id);
            
            // Retour à console
            return $this->redirectToRoute("app_janus_view");
        }

        // Affichage du formulaire
        return $this->render('Janus/function.html.twig', [
            'useheader'         => true,
            'usesidebar'        => true,       
            'mode'              => "update",
            'type'              => "function",
            'subtype'           => "",
            'id'                => $id,
            'deletable'         => $deletable,
            'reload'            => $reload,
            'form'              => $form->createView()
        ]);      
    }

    public function deletefct($id, Request $request) {
        // Initialisation du mode test
        $this->janus->initTest();  
        $config=$this->janus->getConfig();

        // Initialisation du mode test
        $this->janus->initTest();  
        $config=$this->janus->getConfig();

        // Construction du formulaire en fonction du type souhaité
        $this->getForm("function","",$help,$keys,$deletable);
        $reload=false;

        // Déterminer le path du fichier à modifier
        $filesystem = new Filesystem();
        if($filesystem->exists('/var/www/html/janus/config/janus/functions/add/'.$id))
            $path='/var/www/html/janus/config/janus/functions/add/'.$id;
        else {
            // On prend la version eole si la version janus n'existe pas
            if(!$filesystem->exists('/var/www/html/janus/config/janus/functions/'.$id))
                $path='/var/www/html/janus/config/eole/functions/'.$id;
            else
                $path='/var/www/html/janus/config/janus/functions/'.$id;
        }

        // Si la fonction EOLE existe on ne peut pas supprimer mais on peut remettre la version d'origine
        if($filesystem->exists('/var/www/html/janus/config/eole/functions/'.$id)) {
            $deletable=false;
            $reload=($filesystem->exists('/var/www/html/janus/config/janus/functions/'.$id)||$filesystem->exists('/var/www/html/janus/config/janus/functions/add/'.$id));
        }

        $filesystem->mkdir('/var/www/html/janus/config/janus/functions/del');
        $filesystem->touch('/var/www/html/janus/config/janus/functions/del/'.$id);

        return $this->redirectToRoute("app_janus_view"); 
    }        

    public function getErrorForm($type,$form,$request,$data,$mode,$deletable=null) {
        // Initialisation du mode test
        $this->janus->initTest();  
        $config=$this->janus->getConfig();
        $haveerror=false;

        if ($mode=="delete") {
            if(!$deletable) 
                $form->addError(new FormError("Suppression du paramètre interdit"));
            
            switch($type) {
                case "customers":
                    // Impossible si le customers est associé à un testuid
                    foreach($config["testuid"] as $testuid) {
                        if($testuid["customer"]==$data["name"])
                            $form->addError(new FormError("Suppression impossible ce customer est associé à un testuid"));        
                    }
                    if($data["name"]=="envole")
                        $form->addError(new FormError("Le customer envole n'est pas supprimable"));        
                break;
            }

            $errors = $form->getErrors();
            $this->get('session')->getFlashBag()->clear();
            foreach( $errors as $error ) {
                $request->getSession()->getFlashBag()->add("error", $error->getMessage());
                $haveerror=true;
            }    
        }

        if ($form->get('submit')->isClicked() && $mode=="submit") {
            switch($type) {
                case "testuid": 
                case "customers": 
                    // On s'assure que l'identifiant n'est pas déjà présent dans la configuration
                    $name=$data["name"];
                    if(array_key_exists($name,$config[$type]))
                        $form->addError(new FormError("$name est déjà utilisé comme identifiant de $type"));
                break;

                case "providers": 
                    // On s'assure que l'identifiant n'est pas déjà présent dans la configuration
                    $name=$data["name"];
                    if(array_key_exists($name,$config[$type]))
                        $form->addError(new FormError("Un provider utilise déjà ce nom"));
                break;

                case "rulesattribute":
                    // La condition if doit etre entre paranthèse
                    $if=$data["if"];
                    if (stripos($if, '(') === false) 
                        $form->addError(new FormError("La condition if doit être entre parenthèses"));
                    elseif(substr($if, -1)!=")")
                        $form->addError(new FormError("La condition if doit être entre parenthèses"));
                break;

                case "rulesgroup":
                    // La condition if doit etre entre paranthèse
                    $if=$data["if"];
                    if (stripos($if, '(') === false) 
                        $form->addError(new FormError("La condition if doit être entre parenthèses"));
                    elseif(substr($if, -1)!=")")
                        $form->addError(new FormError("La condition if doit être entre parenthèses"));
                break;

                case "functionsubmit":
                    $name=$data["name"];

                    // On s'assure que le nom de fichier ne contient pas des caractères speciaux
                    $string = preg_replace('~[^a-zA-Z0-9._-]~', '', $name);
                    if($string!=$name)
                        $form->addError(new FormError('Caractères interdit dans votre nom de fonction'));

                    // On s'assure que le nom de fichier se termine bien par .php
                    if(substr($name, -4)!=".php")
                        $form->addError(new FormError('Votre nom de fonction doit être un .php'));

                    // On s'assure que le fichier n'existe pas déjà dans les fonctions déjà existante
                    $filesystem = new Filesystem();
                    if(!$filesystem->exists('/var/www/html/janus/config/janus/functions/del/'.$name)) {
                        if($filesystem->exists('/var/www/html/janus/config/janus/functions/add/'.$name)||$filesystem->exists('/var/www/html/janus/config/janus/functions/'.$name))
                            $form->addError(new FormError("Cette fonction existe déjà"));
                    }

                    if($filesystem->exists('/var/www/html/janus/config/eole/functions/'.$name))
                        $form->addError(new FormError("Cette fonction existe déjà"));
                break;
            }
        }

        if ($form->get('submit')->isClicked() && ($mode=="submit"||$mode=="update")) {
            switch($type) {
                case "testuid":
                    foreach($data["ssoattributes"] as $key=>$value) {
                        if(empty($key))
                            $form->addError(new FormError("Les clés SSO sont obligatoires"));
                    }
                break;

                case "providers": 
                    switch($data["type"]) {
                        case "bdd":
                            // Validation des champs
                            if(empty($data["host"])||empty($data["user"])||empty($data["password"])||empty($data["dbname"]))
                                $form->addError(new FormError("Valeurs obligatoire pour un provider de type bdd : host - user - password - dbname"));

                            if(empty($data["userquery"])&&empty($data["groupquery"]))
                                $form->addError(new FormError("Un provider bdd doit avoir au minimum une valeur dans : userquery et/ou groupquery"));
                            
                            if(!empty($data["userquery"])&&stripos($data["userquery"],":uid")===false)
                                $form->addError(new FormError("Le parametre userquery doit comporter le mot clé :uid pour fonctionner"));

                            if(!empty($data["groupquery"])&&stripos($data["groupquery"],":uid")===false)
                                $form->addError(new FormError("Le parametre groupquery doit comporter le mot clé :uid pour fonctionner"));       

                            if(!empty($data["groupquery"])&&empty($data["groupname"]))
                                $form->addError(new FormError("Le parametre groupname est obligatoire si vous précisez un groupquery, il représente le champ qui doit être récupérer comme nom de groupe"));                                  

                            // Tester la connexion
                            if(!$this->janus->testConnexion($data))
                                $form->addError(new FormError("Connexion à la base de données impossible"));                                  
                        break;

                        case "ldap":
                            // Validation des champs
                            if(empty($data["host"])||empty($data["port"])||empty($data["user"])||empty($data["password"])||empty($data["basedn"]))
                                $form->addError(new FormError("Valeurs obligatoire pour un provider de type ldap : host - port - user - password - basedn"));

                            if(empty($data["userquery"])&&empty($data["groupquery"]))
                                $form->addError(new FormError("Un provider bdd doit avoir au minimum une valeur dans : userquery et/ou groupquery"));
                            
                            if(!empty($data["userquery"])&&stripos($data["userquery"],":uid")===false)
                                $form->addError(new FormError("Le parametre userquery doit comporter le mot clé :uid pour fonctionner"));

                            if(!empty($data["groupquery"])&&stripos($data["groupquery"],":uid")===false)
                                $form->addError(new FormError("Le parametre groupquery doit comporter le mot clé :uid pour fonctionner"));       

                            if(!empty($data["groupquery"])&&empty($data["groupname"]))
                                $form->addError(new FormError("Le parametre groupname est obligatoire si vous précisez un groupquery, il représente le champ qui doit être récupérer comme nom de groupe"));                                  

                            // Tester la connexion
                            if(!$this->janus->testConnexion($data))
                                $form->addError(new FormError("Connexion à l'annuaire impossible"));                                  
                        break;        
                        
                        case "ninegate":
                            // Validation des champs
                            if(empty($data["url"])||empty($data["key"]))
                                $form->addError(new FormError("Valeurs obligatoire pour un provider de type ninegate : url- key"));

                            if(!$data["getuser"]&&!$data["getgroups"]&&empty($data["getressources"]))
                                $form->addError(new FormError("Un provider ninegate doit avoir au minimum une valeur dans : getuser et/ou getgroups et/ou getressources"));
                            
                            // Tester la connexion
                            if(!$this->janus->testConnexion($data))
                                $form->addError(new FormError("Connexion à ninegate impossible"));                                  
                        break;                         
                    }
                break;

                case "rulesattribute":
                    // La condition if doit etre entre paranthèse
                    $if=$data["if"];
                    if (stripos($if, '(') === false) 
                        $form->addError(new FormError("La condition if doit être entre parenthèses"));
                    elseif(substr($if, -1)!=")")
                        $form->addError(new FormError("La condition if doit être entre parenthèses"));
                break;

                case "rulesgroup":
                    // La condition if doit etre entre paranthèse
                    $if=$data["if"];
                    if (stripos($if, '(') === false) 
                        $form->addError(new FormError("La condition if doit être entre parenthèses"));
                    elseif(substr($if, -1)!=")")
                        $form->addError(new FormError("La condition if doit être entre parenthèses"));
                break;
            }
        }

        if ($form->get('submit')->isClicked() && !$form->isValid()) {
            $this->get('session')->getFlashBag()->clear();

            $errors = $form->getErrors();
            foreach( $errors as $error ) {
                $request->getSession()->getFlashBag()->add("error", $error->getMessage());
                $haveerror=true;
            }
        }

        return $haveerror;
    } 

    public function test(Request $request)
    {
        // Initialisation du mode test
        $this->janus->initTest();  
        $config=$this->janus->getConfig();

        // Placer l'indicadeur isvalid à false
        $this->janus->updateIsvalid(false);

        // Récupérer les uid à tester avant de pouvoir valider
        $testuid=$config["testuid"];

        // Tableau de retour
        $errors=[];
        $attributes=[];
        $ressources=[];
        
        // Tester la structure de la configuration
        $janusextension = new JanusExtension();
        $prepare=[];
        $prepare["janus"]=$config;      
        try{          
            $janusextension->test($prepare);
        } catch (\Exception $e) {
            if(!array_key_exists("Structure de la Configuration",$errors)) $errors["Structure de la Configuration"]=[];
            array_push($errors["Structure de la Configuration"],["message"=>$e->getMessage(),"trace"=>null]);
        }
    

        // tester l'ensemble des uid
        foreach($testuid as $uid=>$param) {
            try{
                $attributes[$uid]=$this->janus->getUserattributes($param["customer"],$uid,true,$param["ssoattributes"],true); 
            } catch (SyntaxError $e) {
                if(!array_key_exists($uid,$errors)) $errors[$uid]=[];
                array_push($errors[$uid],["message"=>$e->getMessage(),"trace"=>$e->getTraceAsString()]);
            } catch (\Exception $e) {
                if(!array_key_exists($uid,$errors)) $errors[$uid]=[];
                array_push($errors[$uid],["message"=>$e->getMessage(),"trace"=>$e->getTraceAsString()]);
            }
            
            try{
                $ressources[$uid]=$this->janus->getUserressources($param["customer"],$uid,true,"all",$param["ssoattributes"],true);    
            } catch (SyntaxError $e) {
                if(!array_key_exists($uid,$errors)) $errors[$uid]=[];
                array_push($errors[$uid],["message"=>$e->getMessage(),"trace"=>$e->getTraceAsString()]);
            } catch (\Exception $e) {
                if(!array_key_exists($uid,$errors)) $errors[$uid]=[];
                array_push($errors[$uid],["message"=>$e->getMessage(),"trace"=>$e->getTraceAsString()]);
            }              
        }
       
        // Placer l'indicadeur isvalid à true si pas d'erreurs
        if(empty($errors)) $this->janus->updateIsvalid(true);

        return $this->render('Janus/test.html.twig',[
            "useheader"         => true,
            "usesidebar"        => true,

            "errors"            => $errors,

            "testuid"           => $testuid,
            "attributes"        => $attributes,
            "ressources"        => $ressources,
        ]);
    }

    public function reloadjanus(Request $request)
    {
        // Initialisation du mode test
        $this->janus->initTest();  
        $this->janus->endTest();  
        
        return $this->redirectToRoute("app_janus_view");
    }

    public function updatejanus(Request $request)
    {
        // Initialisation du mode test
        $this->janus->initTest();  
        if($this->janus->isValid()) {
            $config=$this->janus->getConfig();
            $prepare=[];
            $prepare["janus"]=$config;

            // Ecriture du fichier de configuration
            $yaml = Yaml::dump($prepare, 5, 4, Yaml::PARSE_OBJECT_FOR_MAP);
            $dir=$this->getParameter("kernel.project_dir");
            file_put_contents($dir.'/config/janus/janus.yaml', $yaml);

            // On passe l'ensemble des functions add en réel
            $dir=$this->getParameter('kernel.project_dir');
            $dir.="/config/janus/functions/add";
            $filesystem = new Filesystem();
            if($filesystem->exists($dir)) {
                $finder = new Finder();
                $finder->in($dir)->files()->name('*.php')->depth('== 0');;
                foreach (iterator_to_array($finder) as $file) {
                    $pathsource=$file->getPathname();
                    $pathdestination=str_replace("/add/","/",$pathsource);
                    $filesystem->copy($pathsource,$pathdestination,true);
                }
            }  

            // On supprime les fichiers présent dans del
            $dir=$this->getParameter('kernel.project_dir');
            $dir.="/config/janus/functions/del";
            $filesystem = new Filesystem();
            if($filesystem->exists($dir)) {
                $finder = new Finder();
                $finder->in($dir)->files()->name('*.php')->depth('== 0');;
                foreach (iterator_to_array($finder) as $file) {
                    $pathdestination=str_replace("/del/","/",$file->getPathname());
                    $filesystem->remove($pathdestination);
                }
            }              

            // Fin du marqueur test
            $this->janus->endTest();  
        }

        return $this->redirectToRoute("app_janus_view");
    }
}
