<?php

namespace App\Entity;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Bridge\Doctrine\Validator\Constraints\UniqueEntity;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * User
 * 
 * @ORM\Entity(repositoryClass="App\Repository\UserRepository")
 * @ORM\Table(name="user",uniqueConstraints={@ORM\UniqueConstraint(name="username", columns={"username"})})
 * @UniqueEntity("username", message="Ce nom d'utilisateur existe dèja")
 */

class User implements UserInterface, \Serializable
{
    /**
     * @var int
     *
     * @ORM\Column(name="id", type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private $id;

    /**
     * @var string
     *
     * @ORM\Column(name="username", type="string", length=255)
     */
    private $username;

    /**
     * @var string
     *
     * @ORM\Column(name="password", type="string", length=255)
     */
    private $password;

    /**
     * @var array
     *
     * @ORM\Column(name="roles", type="array", length=255)
     */
    private $roles = array();

    /**
     * @ORM\Column(name="salt", type="string", length=255)
     */
    private $salt = '';

    /**
      * @ORM\Column(type="string", length=200, nullable=true, options={"default" : 0})
     */
    private $avatar;

    /**
     * @ORM\Column(type="string", length=250, nullable=true)
     */
    private $apikey;

    /**
    * @ORM\OneToMany(targetEntity="Usercache", mappedBy="user", cascade={"persist"}, orphanRemoval=false)
    */    
    protected $usercaches;    

    public function __construct()
    {
        $this->groups = new ArrayCollection();
        $this->usercaches = new ArrayCollection();
    }

    public function getUsername(): ?string
    {
        return $this->username;
    }

    public function getSalt(): ?string
    {
        return $this->salt;
    }

    public function setPassword($password): self
    {
        if($password!=$this->password&&$password!=""&&!is_null($password)){
            $this->salt = uniqid(mt_rand(), true);
            $hash = "{SSHA}" . base64_encode(pack("H*", sha1($password . $this->salt)) . $this->salt);

            $this->password = $hash;
        }

        return $this;
    }

    public function getPassword(): ?string
    {
        return $this->password;
    }

    public function getRoles(): ?array
    {
        return $this->roles;
    }

    public function hasRole(string $role): ?bool
    {
        return in_array($role,$this->roles);
    }

    public function eraseCredentials()
    {
    }

    public function serialize()
    {
        return serialize(array(
            $this->id,
            $this->username,
            $this->password,
            $this->salt,
        ));
    }

    public function unserialize($serialized)
    {
        list (
            $this->id,
            $this->username,
            $this->password,
            $this->salt
        ) = unserialize($serialized, array('allowed_classes' => false));
    }


    public function setId(int $id): self
    {
        $this->id = $id;

        return $this;
    }

    public function getId(): ?int
    {
        return $this->id;
    }

    public function setUsername(string $username): self
    {
        $this->username = $username;

        return $this;
    }


    public function setRoles(array $roles): self
    {
        $this->roles = $roles;

        return $this;
    }

    public function setSalt(string $salt): self
    {
        $this->salt = $salt;

        return $this;
    }

    public function getAvatar(): ?string
    {
        if($this->avatar)
            return $this->avatar;
        else
            return "noavatar.png";
    }

    public function setAvatar(?string $avatar): self
    {
        $this->avatar = $avatar;

        return $this;
    }

    public function getApikey(): ?string
    {
        return $this->apikey;
    }

    public function setApikey(?string $apikey): self
    {
        $this->apikey = $apikey;

        return $this;
    }

    /**
     * @return Collection|Usercache[]
     */
    public function getUsercaches(): Collection
    {
        return $this->usercaches;
    }

    public function addUsercach(Usercache $usercach): self
    {
        if (!$this->usercaches->contains($usercach)) {
            $this->usercaches[] = $usercach;
            $usercach->setUser($this);
        }

        return $this;
    }

    public function removeUsercach(Usercache $usercach): self
    {
        if ($this->usercaches->contains($usercach)) {
            $this->usercaches->removeElement($usercach);
            // set the owning side to null (unless already changed)
            if ($usercach->getUser() === $this) {
                $usercach->setUser(null);
            }
        }

        return $this;
    }
}
