/**
 * <pre>
 * Listens to HELP_COMMAND and displays notifications.
 * Provides interactive tutorial for first time users.
 * </pre>
 *
 * @constructor
 * @param {mindmaps.EventBus} eventBus
 * @param {mindmaps.commandRegistry} commandRegistry
 */
mindmaps.HelpController = function(eventBus, commandRegistry) {

  /**
   * Prepare tutorial guiders.
   */
  function setupInteractiveMode() {
    if (isTutorialDone()) {
      console.debug("skipping tutorial");
      return;
    }

    var notifications = [];
    var interactiveMode = true;

    // start tutorial after a short delay
    eventBus.once(mindmaps.Event.DOCUMENT_OPENED, function() {
      setTimeout(start, 1000);
    });

    function closeAllNotifications() {
      notifications.forEach(function(n) {
        n.close();
      });
    }

    var helpMain, helpRoot;
    function start() {
      helpMain = new mindmaps.Notification(
          "#toolbar",
          {
            position : "bottomMiddle",
            maxWidth : 550,
            title : "Bienvenue dans mindmaps",
            content : "Bonjour, on dirait que vous êtes nouveau ici ! Ces bulles d'information "
                + "vont vous guider à travers l'application. Ou elles ne le feront pas si vous décidez de <a class='skip-tutorial link'>passer le tutorial<a/>."
          });
      notifications.push(helpMain);
      helpMain.$().find("a.skip-tutorial").click(function() {
        interactiveMode = false;
        closeAllNotifications();
        tutorialDone();
      });
      setTimeout(theRoot, 2000);
    }

    function theRoot() {
      if (isTutorialDone())
        return;

      helpRoot = new mindmaps.Notification(
          ".node-caption.root",
          {
            position : "bottomMiddle",
            closeButton : true,
            maxWidth : 350,
            title : "C'est ici qu'on commence : votre idée principale",
            content : "Double-cliquez sur l'idée pour changer sa légende. Ce sera le sujet principal de votre carte mentale."
          });
      notifications.push(helpRoot);

      eventBus.once(mindmaps.Event.NODE_TEXT_CAPTION_CHANGED, function() {
        helpRoot.close();
        setTimeout(theNub, 900);
      });
    }

    function theNub() {
      if (isTutorialDone())
        return;

      var helpNub = new mindmaps.Notification(
          ".node-caption.root",
          {
            position : "bottomMiddle",
            closeButton : true,
            maxWidth : 350,
            padding : 20,
            title : "Créer de nouvelles idées",
            content : "Il est maintenant temps de construire votre carte mentale.<br/> Placez votre souris sur l'idée , cliquez puis déplacez"
                + " le <span style='color:red'>cercle rouge</span> en dehors de l'idée. Vous venez de créer une nouvelle branche."
          });
      notifications.push(helpNub);
      eventBus.once(mindmaps.Event.NODE_CREATED, function() {
        helpMain.close();
        helpNub.close();
        setTimeout(newNode, 900);
      });
    }

    function newNode() {
      if (isTutorialDone())
        return;

      var helpNewNode = new mindmaps.Notification(
          ".node-container.root > .node-container:first",
          {
            position : "bottomMiddle",
            closeButton : true,
            maxWidth : 350,
            title : "Cotre première branche",
            content : "Bien! C'est facile, non ? Ce cercle rouge est votre outil le plus important. Maintenant, vous pouvez déplacer votre idée"
                + " librement ou double-cliquer pour changer son texte."
          });
      notifications.push(helpNewNode);
      setTimeout(inspector, 2000);

      eventBus.once(mindmaps.Event.NODE_MOVED, function() {
        helpNewNode.close();
        setTimeout(navigate, 0);
        setTimeout(toolbar, 15000);
        setTimeout(menu, 10000);
        setTimeout(tutorialDone, 20000);
      });
    }

    function navigate() {
      if (isTutorialDone())
        return;

      var helpNavigate = new mindmaps.Notification(
          ".float-panel:has(#navigator)",
          {
            position : "bottomRight",
            closeButton : true,
            maxWidth : 350,
            expires : 10000,
            title : "Navigation",
            content : "Vous pouvez cliquer et bouger le fond de carte pour vous déplacer. Utilisez la molette de la souris ou le curseur pour zommer/dézoomer."
          });
      notifications.push(helpNavigate);
    }

    function inspector() {
      if (isTutorialDone())
        return;

      var helpInspector = new mindmaps.Notification(
          "#inspector",
          {
            position : "leftBottom",
            closeButton : true,
            maxWidth : 350,
            padding : 20,
            title : "Vous n'aimez pas les couleurs ?",
            content : "Utilisez ces boutons de contrôme pour changer l'apparence de vos idées. "
                + "Essayez de cliquer sur l'icône dans le coin en haut à droite pour minimiser ce panneau."
          });
      notifications.push(helpInspector);
    }

    function toolbar() {
      if (isTutorialDone())
        return;

      var helpToolbar = new mindmaps.Notification(
          "#toolbar .buttons-left",
          {
            position : "bottomLeft",
            closeButton : true,
            maxWidth : 350,
            padding : 20,
            title : "La barre d'outils",
            content : "Ces boutons font ce qu'ils disent. Vous pouvez les utilisez ou travailler avec les raccourcis claviers. "
                + "Placez votre souris sur ces boutons pour voir les combinaisons de touches."
          });
      notifications.push(helpToolbar);
    }

    function menu() {
      if (isTutorialDone())
        return;

      var helpMenu = new mindmaps.Notification(
          "#toolbar .buttons-right",
          {
            position : "leftTop",
            closeButton : true,
            maxWidth : 350,
            title : "Sauvegader votre travail",
            content : "Le bouton à droite ouvre un menu où vous pouvez saugarder votre travail ou commencer à travailler "
                + "sur un autre si vous le désirez."
          });
      notifications.push(helpMenu);
    }

    function isTutorialDone() {
      return mindmaps.LocalStorage.get("mindmaps.tutorial.done") == 1;
    }

    function tutorialDone() {
      mindmaps.LocalStorage.put("mindmaps.tutorial.done", 1);
    }

  }

  /**
   * Prepares notfications to show for help command.
   */
  function setupHelpButton() {
    var command = commandRegistry.get(mindmaps.HelpCommand);
    command.setHandler(showHelp);

    var notifications = [];
    function showHelp() {
      // true if atleast one notifications is still on screen
      var displaying = notifications.some(function(noti) {
        return noti.isVisible();
      });

      // hide notifications if visible
      if (displaying) {
        notifications.forEach(function(noti) {
          noti.close();
        });
        notifications.length = 0;
        return;
      }

      // show notifications
      var helpRoot = new mindmaps.Notification(
          ".node-caption.root",
          {
            position : "bottomLeft",
            closeButton : true,
            maxWidth : 350,
            title : "Voici votre idée centrale",
            content : "Double-cliquez sur un idée pour éditer son texte. Placez votre souris sur "
                + "une idée et déplacez le bouton rouge pour en créer une nouvelle."
          });

      var helpNavigator = new mindmaps.Notification(
          "#navigator",
          {
            position : "leftTop",
            closeButton : true,
            maxWidth : 350,
            padding : 20,
            title : "Voici le navigateur",
            content : "Use this panel to get an overview of your map. "
                + "Vous pouvez naviguer dedans en déplaçant le rectangle rouge ou changer le niveau de zoom en cliquant sur les loupes."
          });

      var helpInspector = new mindmaps.Notification(
          "#inspector",
          {
            position : "leftTop",
            closeButton : true,
            maxWidth : 350,
            padding : 20,
            title : "Voici l'inspecteur",
            content : "Utilisez ces options pour changer l'apparence de vos idées. "
                + "Essayer de cliquez sur l'icône en haut a droite pour minimiser ce panneau."
          });

      var helpToolbar = new mindmaps.Notification(
          "#toolbar .buttons-left",
          {
            position : "bottomLeft",
            closeButton : true,
            maxWidth : 350,
            title : "Voici la barre d'outils",
            content : "Ces boutons font ce qu'ils disent. Vous pouvez les utiliser ou travailler avec les raccourcis clavier. "
                + "Placez votre souris sur ces boutons pour voir les combinaisons de touches."
          });

      notifications.push(helpRoot, helpNavigator, helpInspector,
          helpToolbar);
    }
  }

  setupInteractiveMode();
  setupHelpButton();
};
