<?php

namespace App\Event;

use App\Entity\User;
use App\Entity\UserGroup;

use Doctrine\Common\EventSubscriber;
use Doctrine\ORM\Event\LifecycleEventArgs;
use Doctrine\ORM\Event\PreUpdateEventArgs;
use Doctrine\ORM\EntityManager;
use Ramsey\Uuid\Uuid;
use App\Service\LdapService;

class SyncUser implements EventSubscriber
{
    protected $container;
    protected $em;
    protected $shouldSync;
    protected $shouldSyncpwd;
    protected $baseUser;
    protected $ldap;
    protected $ldap_username;

    public function __construct($container, EntityManager $em, LdapService $ldap, $ldap_username) {
        $this->container = $container;
        $this->em = $em;
        $this->ldap = $ldap;
        $this->ldap_username = $ldap_username;

        $this->shouldSync = true;
        $this->shouldSyncpwd = true;
    }

    public function getSubscribedEvents()
    {
        return array(
            'postPersist',
            'preUpdate',
            'postUpdate',
            'preRemove'
        );
    }

    public function preUpdate(PreUpdateEventArgs $args) {
        $entity = $args->getEntity();

        if(!($entity instanceof User)) return;

        // Synchronisation uniquement si changement de valeur
        $this->shouldSync = $args->hasChangedField('username') ||
                            $args->hasChangedField('firstname') ||
                            $args->hasChangedField('lastname') ||
                            $args->hasChangedField('email') ||
                            $args->hasChangedField('role') ||
                            $args->hasChangedField('avatar') ||
                            $args->hasChangedField('niveau01') ||
                            $args->hasChangedField('niveau02') ||
                            $args->hasChangedField('siren') ||
                            $args->hasChangedField('siret') ||
                            $args->hasChangedField('labelniveau01') ||
                            $args->hasChangedField('labelniveau02') ||
                            $args->hasChangedField('authlevel') ||
                            $args->hasChangedField('usualname') ||
                            $args->hasChangedField('telephonenumber') ||
                            $args->hasChangedField('postaladress') ||
                            $args->hasChangedField('givensname') ||
                            $args->hasChangedField('birthdate') ||
                            $args->hasChangedField('gender') ||
                            $args->hasChangedField('job') ||
                            $args->hasChangedField('position') ||
                            $args->hasChangedField('belongingpopulation') ||
                            $args->hasChangedField('birthcountry') ||
                            $args->hasChangedField('birthplace')
                            ;

        $this->shouldSyncpwd = $args->hasChangedField('password');

        $entity->setSiren($entity->getNiveau01()->getSiren());
        $entity->setLabelniveau01($entity->getNiveau01()->getLabel());
        if($entity->getNiveau02()!==null) {
            $entity->setSiret($entity->getNiveau02()->getSiret());
            $entity->setLabelniveau02($entity->getNiveau02()->getLabel());
        }
        else {
            $entity->setSiret(null);
            $entity->setLabelniveau02(null);
        }
    }

    public function postUpdate(LifecycleEventArgs $args)
    {
        $entity = $args->getEntity();


        // On met à jour/créé la fiche de l'utilisateur dans l'annuaire
        if ($entity instanceof User && $this->shouldSync) {
            $this->upsertUser($entity);
        }

        // On met à jour/créé le password de l'utilisateur dans l'annuaire
        if ($entity instanceof User && $this->shouldSyncpwd) {
            $this->upsertUserpwd($entity);
        }        
    }

    public function postPersist(LifecycleEventArgs $args)
    {
        $entity = $args->getEntity();

        // On créait une fiche pour l'usager dans l'annuaire
        if ($entity instanceof User) {
            $this->upsertUser($entity);
            $this->upsertUserpwd($entity);
        }
    }

    public function preRemove(LifecycleEventArgs $args)
    {
        $entity = $args->getEntity();

        if ($entity instanceof User) {
            $this->removeUser($entity);
        }
    }

    public function removeUser($user) {
        // Synchonisation LDAP
        if($this->ldap->isEnabled()) {
            // On recherche l'utilisateur dans l'annuaire
            $criteria = '('.$this->ldap_username.'='.$user->getUsername().')';
            $subbranch=$this->baseUser;
            $results = $this->ldap->search($criteria, array($this->ldap_username), $subbranch);

            if(count($results)) {
                $this->ldap->deleteUser($user);
            }
        }
        
        // Synchronisation Onlyoffice
        /*
        $only = $this->container->get('cadoles.core.service.only');
        if($only->isEnabled()) {
            $only->delUser($user);
        } 
        */       
    }

    public function upsertUser($user, $force = false)
    {
        // Synchonisation LDAP
        if($this->ldap->isEnabled()) {
            // On recherche l'utilisateur dans l'annuaire
            $criteria = '('.$this->ldap_username.'='.$user->getUsername().')';
            $subbranch=$this->baseUser;
            $results = $this->ldap->search($criteria, array($this->ldap_username), $subbranch);

            // Mise à jour si elle existe
            if(count($results) > 0) {
                $this->ldap->modifyUser($user);
            }
            // Sinon création de la fiche
            else {
                $this->ldap->addUser($user);
            }

            $this->ldap->addGroupUser($user);
        }

        // Synchronisation Onlyoffice
        /*
        $only = $this->container->get('cadoles.core.service.only');
        if($only->isEnabled()) {
            $only->syncUser($user);
        }
        */

        // On ajoute l'utilisateur au groupe 'Tout le Monde' quoi qu'il arrive
        $fgall=$this->em->getRepository('App\Entity\Group')->findBy(array('fgall'=>true));
        if($fgall) {
            $data=$this->em->getRepository('App\Entity\UserGroup')->findBy(array('user'=>$user,'group'=>$fgall[0]));
            if(!$data) {
                $key = Uuid::uuid4();
                $data=new UserGroup();
                $data->setUser($user);
                $data->setGroup($fgall[0]);
                $data->setKeyvalue($key);
                $data->setRolegroup(0);
                $this->em->persist($data);
                $this->em->flush();            
            }
        }
    }

    public function upsertUserpwd($user, $force = false)
    {
        // Synchonisation LDAP
        if($this->ldap->isEnabled()) {
            // On recherche l'utilisateur dans l'annuaire
            $criteria = '('.$this->ldap_username.'='.$user->getUsername().')';
            $subbranch=$this->baseUser;
            $results = $this->ldap->search($criteria, array($this->ldap_username), $subbranch);

            // Mise à jour si elle existe
            if(count($results) > 0) {
                $this->ldap->modifyUserpwd($user);
            }
        }

        // On vide le password temporaire AD
        $user->setPasswordad("");
        $this->em->flush();
    }

    public function getBaseUser() {
        return $this->baseUser;
    }

    public function setBaseUser($baseUser) {
        $this->baseUser = $baseUser;
        return $this;
    }       
}
