<?php
/**
* LDAP
*
*/

class LDAP
{
	/**
	* identifiant de serveur LDAP pour connexion anonyme
	* @var false|integer
	*/
	private $cid = false;

	/**
	* identifiant de connexion anonyme
	* @var false|integer
	*/
	private $bid = false;

	/**
	* identifiant de serveur LDAP pour connexion utilisateur
	* @var false|integer
	*/
	private $cidU = false;

	/**
	* identifiant de connexion utilisateur
	* @var false|integer
	*/
	private $bidU = false;

	/**
	* identifiant de serveur LDAP pour connexion administrateur
	* @var false|integer
	*/
	private $cidA = false;

	/**
	* identifiant de connexion administrateur
	* @var false|integer
	*/
	private $bidA = false;

	/**
	* dn du dernier utilisateur connecté
	* @var false|string
	*/
	private $dnUser = false;

	/**
	* serveur LDAP
	* @var string
	*/
	private $host;
    /**
     * port LDAP
     * @var number
     */
    private $port;

	/**
	* dn racine
	* @var string
	*/
	private $basedn;

	/**
	* version du protocole LDAP
	* @var false|integer
	*/
	private $protocoleVersion = false;

	/**
	* branche des Etablissements
	* @var string
	*/
	private $brancheEtab = '';

	/**
	* branches autorisées pour l'authentification
	* @var array
	*/
	private $branches = array();

	/**
	* listes autorisées pour l'authentification
	* @var array
	*/
	private $brancheListes = array();

	/**
	* uid d'utilisateurs autorisés pour l'authentification
	* @var array
	*/
	private $privilegies = array();

	/**
	* login et pass de l'utilisateur ayant des droits d'écriture dans l'annuaire
	* @var array
	*/
	private $admin = array();

	/**
	* login et pass de l'utilisateur ayant des droits de recherche
	* @var array
	*/
	private $recherche = array();

	/**
	* taille maximum du résultat d'une recherche
	* @var integer
	*/
	private $tailleMaxRecherche = 100;

	/**
	* temps maximum d'une recherche en secondes
	* @var integer
	*/
	private $tempsMaxRecherche = 100;

	/**
	* dn de la branche de l'utilisateur connecté
	* @var false|string
	*/
	private $brancheUser = false;

	/**
	* dn de la première liste où est présent l'utilisateur connecté
	* @var false|string
	*/
	private $listeUser = false;

	/**
	* l'utilisateur est un privilégié
	* @var bool
	*/
	private $privilegieUser = false;


	/**
	* initialisation des variables de la classe
	* @param array $param tableau de paramètres
	*/
	public function __construct(&$param)
	{
		//récupération des paramètres
		$this->host 	= $param['host'];
        $this->port     = $param['port'];
		$this->basedn 	= $param['basedn'];

		if(isset($param['protocoleVersion']) && $param['protocoleVersion'])
			$this->protocoleVersion = $param['protocoleVersion'];

		if(isset($param['brancheEtab']) && $param['brancheEtab'])
			$this->brancheEtab = $param['brancheEtab'];

		if(isset($param['branches']) && $param['branches'])
		{
			if(is_array($param['branches']))
				$this->branches = $param['branches'];
			else
				$this->branches[0] = $param['branches'];
		}

		if(isset($param['branchesListes']) && $param['branchesListes'])
		{
			if(is_array($param['branchesListes']))
				$this->brancheListes = $param['branchesListes'];
			else
				$this->brancheListes[0] = $param['branchesListes'];
		}

		if(isset($param['privilegies']) && $param['privilegies'])
		{
			if(is_array($param['privilegies']))
				$this->privilegies = $param['privilegies'];
			else
				$this->privilegies[0] = $param['privilegies'];
		}

		if(isset($param['admin']) && $param['admin'])
			$this->admin = $param['admin'];

		if(isset($param['recherche']) && $param['recherche'])
			$this->recherche = $param['recherche'];

		if(isset($param['tailleMaxRecherche']) && $param['tailleMaxRecherche'])
			$this->tailleMaxRecherche = $param['tailleMaxRecherche'];

		if(isset($param['tempsMaxRecherche']) && $param['tempsMaxRecherche'])
			$this->tempsMaxRecherche = $param['tempsMaxRecherche'];
	}

	/**
	* destructeur
	* déconnexions si non réalisées
	*/
	public function __destruct()
	{
		// On vérifie que les connexions ont bien été fermées
		if($this->cid)
			$this->deconnect();
		if($this->cidU)
			$this->deconnectUser();
		if($this->cidA)
			$this->deconnectAdmin();
	}

	/**
	* connexion (bind) anonyme
	*/
	private function connect()
	{
		try
		{
			$this->cid = ldap_connect($this->host);//, $this->port);

			if($this->protocoleVersion)
				ldap_set_option($this->cid, LDAP_OPT_PROTOCOL_VERSION, 3);

			$this->bid = ldap_bind($this->cid);

			return true;
		}
		catch(Exception $e)
		{
			return false;
		}
	}

	/**
	* connexion d'un utilisateur
	* @param string $user
	* @param string $pass
	* @param bool $recherche
	* @param bool $erreur
	*/
	private function connectUser($user,$pass,$recherche=false,$erreur=true)
	{
		$this->cidU = ldap_connect($this->host, $this->port);

		if($this->protocoleVersion)
			ldap_set_option($this->cidU, LDAP_OPT_PROTOCOL_VERSION, 3);

		if($recherche)
		{
			$dn = $this->uidExiste($user);
		}
		else
		{
			$this->dnUser = $this->uidExiste($user);
			$dn = $this->dnUser;
		}

		if($dn)
		{
			try
			{
				//BUG FIX (juin 2009) :
				//sur Scribe 2.2 (pas sur 2.0), la fonction ldap_bind() pose problème avec un $pass erroné
				//variable $this->bidU mise à true en cas de succès même si elle devait recevoir un entier...
				if(@ldap_bind($this->cidU,$dn,$pass)) $this->bidU = true;
				else $this->bidU = false;
				return $this->bidU;
			}
			catch(Exception $e)
			{
				//si le mot de passe est incorrect, warning (level 2) à attraper
				$this->bidU = false;

				if(ldap_errno($this->cidU) == 49 && $erreur)
					trigger_error('pass de connexion LDAP invalide...', E_USER_ERROR);
				elseif(ldap_errno($this->cidU) != 49)
					trigger_error('erreur LDAP n° '.ldap_errno($this->cidU).' ...', E_USER_ERROR);

			}
		}
		elseif(!$erreur)
		{
			$this->bidU = false;
		}
		else
			trigger_error('login de connexion LDAP inconnu...', E_USER_ERROR);
	}

	/**
	* connexion d'un administrateur
	* @param bool $uid indique que le user admin est un uid et non un "cn=dirmanager"
	*/
	private function connectAdmin($uid=false)
	{
		$this->cidA = ldap_connect($this->host, $this->port);

		if($this->protocoleVersion)
			ldap_set_option($this->cidA, LDAP_OPT_PROTOCOL_VERSION, 3);

		if($uid)
		{
			$dn = $this->uidExiste($this->admin['user']);
			if($dn)
				$user = $dn;
			else
				trigger_error('login de connexion LDAP inconnu...', E_USER_ERROR);
		}
		else
			$user = $this->admin['user'];

		$this->bidA = ldap_bind($this->cidA,$user,$this->admin['pass']);
	}

	/**
	* déconnexion d'une connexion anonyme
	*/
	private function deconnect()
	{
		ldap_unbind($this->cid);
		$this->cid = false;
		$this->bid = false;
	}

	/**
	* déconnexion d'un utilisateur
	*/
	private function deconnectUser()
	{
		ldap_unbind($this->cidU);
		$this->cidU = false;
		$this->bidU = false;
	}

	/**
	* déconnexion d'un utilisateur
	*/
	private function deconnectAdmin()
	{
		ldap_unbind($this->cidA);
		$this->cidA = false;
		$this->bidA = false;
	}

	/**
	* test connexion (bind) anonyme
	*/
	public function testConnectAnonyme()
	{
		try
		{
			$cid = ldap_connect($this->host, $this->port);

			if($this->protocoleVersion)
				ldap_set_option($cid, LDAP_OPT_PROTOCOL_VERSION, 3);

			$bid = ldap_bind($cid);

			return true;
		}
		catch(Exception $e)
		{
			return false;
		}
	}

	/**
	* Fonction de test de presence du DN
	* @param string $dn dn de recherche des données
	* @return false|true retourne vrai si trouvé
	*/
	//public function testDN($dn,$recherche,$user=false,$pass=false)
	public function presenceDN($dn,$user=false,$pass=false)
	{
		// variables pour le test de recherche.
		$recherche = array('uid');
		$filtre='(uid=test)';

		// Connexion en mode anonyme ou avec user et pass ou avec user et pass de recherche
		// recherche des données
		if($user && $pass)
		{
			$this->connectUser($user,$pass);
			$lecid = $this->cidU;
		}
		elseif($this->recherche && $this->recherche['user'] && $this->recherche['pass'])
		{
			$this->connectUser($this->recherche['user'],$this->recherche['pass'],true);
			$lecid = $this->cidU;
		}
		else
		{
			$this->connect();
			$lecid = $this->cid;
		}

		$sr = false;
		try
		{
			$sr = ldap_search($lecid,$dn,$filtre,$recherche,0,$this->tailleMaxRecherche,$this->tempsMaxRecherche);
			return true;
		}
		catch(Exception $e)
		{
			return false;
		}

		if($user && $pass || $this->recherche && $this->recherche['user'] && $this->recherche['pass'])
			$this->deconnectUser();
		else
			$this->deconnect();
	}

	/**
	* authentification d'un utilisateur
	* renvoie false ou true
	* @param string $user
	* @param string $pass
	* @return false|integer
	*/
	public function authentification($user,$pass)
	{
		if(!empty($user) && !empty($pass))
		{
			//indique si l'utilisateur a été trouvé dans une des branches autorisées
			$autorise = false;

			//privilégié ?
			if(!empty($this->privilegies) && in_array(strtolower($user),$this->privilegies))
			{
				$autorise = true;
				$this->privilegieUser = true;
			}

			//dans une des branches ?
			if(!$autorise && !empty($this->branches) && $this->uidDansBranches($user,$this->branches))
				$autorise = true;

			//dans une des listes ?
			if(!$autorise && !empty($this->brancheListes) && $this->uidDansListes($user,$this->brancheListes))
				$autorise = true;

			//si autorisé on le connecte
			if($autorise)
			{
				$this->connectUser($user,$pass,false,false);
				$lebid = $this->bidU;
				$this->deconnectUser();
				return $lebid;
			}
			else
				return false;
		}
		else
			return false;
	}

	/**
	* existence de l'uid dans le base dn
	* @param string $uid
	* @return false|string
	*/
	private function uidExiste($uid)
	{
		//connexion anonyme
		$this->connect();
		$sr  = ldap_search($this->cid,$this->basedn,"uid=$uid",array('uid','dn'));
		$res = ldap_get_entries($this->cid, $sr);
		$this->deconnect();
		if($res['count'] == 0)
			return false;
		else
			return $res[0]['dn'];
	}

	/**
	* récupération d'informations diverses sur l'utilisateur connecté
	* @return false|array
	*/
	public function infosUser()
	{
		// Connexion en mode anonyme ou avec user et pass de recherche
		if($this->recherche && $this->recherche['user'] && $this->recherche['pass'])
		{
			$this->connectUser($this->recherche['user'],$this->recherche['pass'],true);
			$lecid = $this->cidU;
		}
		else
		{
			$this->connect();
			$lecid = $this->cid;
		}

		$sr  = ldap_search($lecid,$this->dnUser,'(objectclass=*)',array('cn','rne'));
		$res = ldap_get_entries($lecid, $sr);

		//déconnexion
		if($this->recherche && $this->recherche['user'] && $this->recherche['pass'])
			$this->deconnectUser();
		else
			$this->deconnect();

		if($res['count'] == 0)
			return false;
		else
		{
			$infos['cn'] 	 = $res[0]['cn'][0];
			if(isset($res[0]['rne']) && isset($res[0]['rne'][0]))
			{
				$infos['rne'] = $res[0]['rne'][0];
			}
			else
			{
				$infos['rne'] = '';
			}
			$infos['dn'] 	 = $this->dnUser;
			$infos['profil'] = $this->listeUser;
			return $infos;
		}
	}

	/**
	* récupération du nom complet de l'établissement connecté
	* @return string
	*/
	public function nomEtab()
	{
		// Connexion en mode anonyme ou avec user et pass de recherche
		if($this->recherche && $this->recherche['user'] && $this->recherche['pass'])
		{
			$this->connectUser($this->recherche['user'],$this->recherche['pass'],true);
			$lecid = $this->cidU;
		}
		else
		{
			$this->connect();
			$lecid = $this->cid;
		}

		$sr = ldap_search($lecid,$this->dnUser,'(objectclass=*)',array('cn'));
		$res = ldap_get_entries($lecid, $sr);

		//déconnexion
		if($this->recherche && $this->recherche['user'] && $this->recherche['pass'])
			$this->deconnectUser();
		else
			$this->deconnect();

		if($res['count'] == 0)
			return 'Etablissement sans nom...';
		else
		{
			return $res[0]['cn'][0];
		}
	}

	/**
	* donne le dn de l'utilisateur connecté
	* @return string
	*/
	public function donneDN()
	{
		return $this->dnUser;
	}

	/**
	* indique si un dn appartient à une branche
	* @param string $dn
	* @param string $branche
	* @return bool
	*/
	public function dnDansBranche($dn,$branche)
	{
		$taille = strlen($branche);
		return !strcasecmp(substr($dn,-$taille,$taille),$branche);
	}

	/**
	* indique si un dn appartient à une liste
	* @param string $dn
	* @param string $liste
	* @return false|integer
	*/
	public function dnDansListe($dn,$liste)
	{
		// Connexion en mode anonyme ou avec user et pass de recherche
		if($this->recherche && $this->recherche['user'] && $this->recherche['pass'])
		{
			$this->connectUser($this->recherche['user'],$this->recherche['pass'],true);
			$lecid = $this->cidU;
		}
		else
		{
			$this->connect();
			$lecid = $this->cid;
		}

		$sr    = ldap_search($lecid,$liste.','.$this->basedn,'uniquemember='.$dn,array('uniquemember'));
		$count = ldap_count_entries($lecid,$sr);

		//déconnexion
		if($this->recherche && $this->recherche['user'] && $this->recherche['pass'])
			$this->deconnectUser();
		else
			$this->deconnect();

		return $count;
	}

	/**
	* indique si un uid appartient à un tableau de branches
	* @param string $uid
	* @param array $branches
	* @return bool
	*/
	public function uidDansBranches($uid,&$branches)
	{
		$result = false;

		// Connexion en mode anonyme ou avec user et pass de recherche
		if($this->recherche && $this->recherche['user'] && $this->recherche['pass'])
		{
			$this->connectUser($this->recherche['user'],$this->recherche['pass'],true);
			$lecid = $this->cidU;
		}
		else
		{
			$this->connect();
			$lecid = $this->cid;
		}

		foreach($branches as $value)
		{
			if($value!='')
			{
				//$sr    = ldap_search($lecid,$value.','.$this->basedn,'uid='.$uid,array('uid'));
				$sr    = ldap_search($lecid,$this->basedn,'uid='.$uid,array('uid'));
				$count = ldap_count_entries($lecid,$sr);
				if($count)
				{
					$ldapinfo = ldap_get_entries($lecid, $sr);
					//$this->brancheUser = $value.','.$this->basedn;
					$this->brancheUser = $ldapinfo[0]["dn"];
					
					$result = true;
					break;
				}
			}
		}

		//déconnexion
		if($this->recherche && $this->recherche['user'] && $this->recherche['pass'])
			$this->deconnectUser();
		else
			$this->deconnect();

		return $result;
	}

	/**
	* renvoie le dn de la branche de l'utilisateur connecté
	* @return string
	*/
	public function donneBrancheUser()
	{
		return $this->brancheUser;
	}

	/**
	* indique si un uid appartient à un tableau de listes
	* @param string $uid
	* @param array $listes
	* @return bool
	*/
	public function uidDansListes($uid,&$listes)
	{
		$result = false;

		// Connexion en mode anonyme ou avec user et pass de recherche
		if($this->recherche && $this->recherche['user'] && $this->recherche['pass'])
		{
			$this->connectUser($this->recherche['user'],$this->recherche['pass'],true);
			$lecid = $this->cidU;
		}
		else
		{
			$this->connect();
			$lecid = $this->cid;
		}

		foreach($listes as $value)
		{
			if($value!='')
			{
				$sr = ldap_search($lecid,$value.','.$this->basedn,'uniquemember=uid='.$uid.',*',array('uniquemember'));
				$count = ldap_count_entries($lecid,$sr);
				if($count)
				{
					$this->listeUser = $value.','.$this->basedn;
					$result = true;
					break;
				}
			}
		}

		//déconnexion
		if($this->recherche && $this->recherche['user'] && $this->recherche['pass'])
			$this->deconnectUser();
		else
			$this->deconnect();

		return $result;
	}

	/**
	* renvoie le dn de la liste de l'utilisateur connecté
	* @return string
	*/
	public function donneListeUser()
	{
		return $this->listeUser;
	}

	/**
	* utlisateur connecté ou dn en particulier est un établissement ?
	* @param false|string $dn
	* @return bool
	*/
	public function estEtab($dn=false)
	{
		if(!$dn)
			$dn = $this->dnUser;
		return $this->dnDansBranche($dn,$this->brancheEtab.','.$this->basedn);
	}

	/**
	* utlisateur connecté est un privilégié ?
	* @return bool
	*/
	public function estPrivilegie()
	{
		return $this->privilegieUser;
	}

	/**
	* Fonction de recherche d'informations dans une partie de l'annuaire
	* @param string $dn dn de recherche des données
	* @param array $recherche données à rechercher
	* @param string $filtre filtre de recherche
	* @param string $tri critère de tri pour le résultat
	* @param false|string $user utilisateur spécifique pour la connexion au LDAP
	* @param false|string $pass mot de passe associé à l'utilisateur spécifique
	* @param false|integer $tailleMaxRetour nombre de résultats maximum à renvoyer
	* @return array tableau avec le résultat de recherche
	*/
	public function rechercher($dn,$recherche,$filtre='(objectclass=*)',$tri='dn',$user=false,$pass=false,$tailleMaxRetour=false)
	{
		// Connexion en mode anonyme ou avec user et pass ou avec user et pass de recherche
		// recherche des données
		if($user && $pass)
		{
			$this->connectUser($user,$pass);
			$lecid = $this->cidU;
		}
		elseif($this->recherche && $this->recherche['user'] && $this->recherche['pass'])
		{
			$this->connectUser($this->recherche['user'],$this->recherche['pass'],true);
			$lecid = $this->cidU;
		}
		else
		{
			$this->connect();
			$lecid = $this->cid;
		}

		$sr = false;
		try
		{
			$sr = ldap_search($lecid,$dn,$filtre,$recherche,0,$this->tailleMaxRecherche,$this->tempsMaxRecherche);
		}
		catch(Exception $e)
		{
			if(ldap_errno($lecid) == 4 || ldap_errno($lecid) == 11)
				$sr = false;
			else
				trigger_error('erreur LDAP n° '.ldap_errno($lecid).' ...', E_USER_ERROR);
		}

		if($sr !== false)
		{
			ldap_sort($lecid,$sr,$tri);
			$res = ldap_get_entries($lecid, $sr);
		}

		if($user && $pass || $this->recherche && $this->recherche['user'] && $this->recherche['pass'])
			$this->deconnectUser();
		else
			$this->deconnect();

		// Analyse des données récupérées
		if($sr == false || $res['count'] == 0)
			return false;
		else
		{
			$infos = array();
			for($j=0;$j<$res['count'];$j++)
			{
				if($tailleMaxRetour && $j==$tailleMaxRetour)
					break;
				$infos[$j] = array();
				for($i=0;$i<count($recherche);$i++)
					if(isset($res[$j][$recherche[$i]][0]))
					{
						if($res[$j][$recherche[$i]]['count']<=1)
							$infos[$j][$recherche[$i]] = trim($res[$j][$recherche[$i]][0]);
						else
						{
							$infos[$j][$recherche[$i]] = array();
							for($k=0;$k<$res[$j][$recherche[$i]]['count'];$k++)
								array_push($infos[$j][$recherche[$i]],trim($res[$j][$recherche[$i]][$k]));
						}
					}
					else
						$infos[$j][$recherche[$i]] = '';
				$infos[$j]['dn'] = trim($res[$j]['dn']);
			}
			return $infos;
		}
	}

	/**
	* Fonction de modification d'informations pour un utilisateur dans l'annuaire
	* @param string $dn dn de l'utilisateur à modifier
	* @param array $nouveau nouvelles données
	* @param string $user
	* @param string $pass
	* @param bool $uid
	* @return bool
	*/
	public function modifierDonneesUtilisateur($dn,&$nouveau,$user=false,$pass=false,$uid=false)
	{
		if($user && $pass)
		{
			$this->connectUser($user,$pass);
			$lecid = $this->cidU;
		}
		else
		{
			$this->connectAdmin($uid);
			$lecid = $this->cidA;
		}
		$result = ldap_mod_replace($lecid,$dn,$nouveau);

		if($user && $pass)
			$this->deconnectUser();
		else
			$this->deconnectAdmin();

		return $result;
	}

	/**
	* Fonction d'insertion d'informations pour un utilisateur dans l'annuaire
	* @param string $dn dn de l'utilisateur à modifier
	* @param array $nouveau nouvelles données
	* @param bool $uid
	* @return bool
	*/
	public function ajouterDonneesUtilisateur($dn,&$nouveau,$uid=false)
	{
		$this->connectAdmin($uid);
		$result = ldap_mod_add($this->cidA,$dn,$nouveau);
		$this->deconnectAdmin();

		return $result;
	}

	/**
	* Fonction d'insertion d'un nouvel utilisateur dans l'annuaire
	* @param string $dn dn de l'utilisateur à ajouter
	* @param array $nouveau nouvelles données
	* @param bool $uid
	* @return bool
	*/
	public function ajouterUtilisateur($dn,&$nouveau,$uid=false)
	{
		$this->connectAdmin($uid);
		$result = ldap_add($this->cidA,$dn,$nouveau);
		$this->deconnectAdmin();
		return $result;
	}

	/**
	* Fonction de suppression d'informations pour un utilisateur dans l'annuaire
	* @param string $dn dn de l'utilisateur à modifier
	* @param array $donnees données à supprimer
	* @param bool $uid
	* @return bool
	*/
	public function supprimerDonneesUtilisateur($dn,&$donnees,$uid=false)
	{
		$this->connectAdmin($uid);
		$result = ldap_mod_del($this->cidA,$dn,$donnees);
		$this->deconnectAdmin();
		return $result;
	}

	/**
	* Fonction de suppression d'un utilisateur dans l'annuaire
	* @param string $dn dn de l'utilisateur à supprimer
	* @param bool $uid
	* @return bool
	*/
	public function supprimerUtilisateur($dn,$uid=false)
	{
		$this->connectAdmin($uid);
		$result = ldap_delete($this->cidA,$dn);
		$this->deconnectAdmin();
		return $result;
	}

	/**
	* Fonction de réinitialisation du mot de passe d'un utilisateur dans l'annuaire
	* @param string $dn dn de l'utilisateur à réinitialiser
	* @param bool $uid
	* @return bool
	*/
	public function reinit($dn,$uid=false)
	{
		$this->connectAdmin($uid);
		$sr = ldap_search($this->cidA,$dn,"(objectclass=*)",array("employeenumber"));
		$res = ldap_get_entries($this->cidA,$sr);
		if($res["count"] != 0)
		{
			$donnees['userpassword'][0] = $res[0]["employeenumber"][0];
			$result = ldap_mod_replace($this->cidA,$dn,$donnees);
		}
		else
			$result = false;
		$this->deconnectAdmin();
		return $result;
	}
}

?>
