<?php
/**
 * @version $Id$
 * @author Thomas Crespin <thomas.crespin@sesamath.net>
 * @copyright Thomas Crespin 2009-2018
 *
 * ****************************************************************************************************
 * SACoche <https://sacoche.sesamath.net> - Suivi d’Acquisitions de Compétences
 * © Thomas Crespin pour Sésamath <https://www.sesamath.net> - Tous droits réservés.
 * Logiciel placé sous la licence libre Affero GPL 3 <https://www.gnu.org/licenses/agpl-3.0.html>.
 * ****************************************************************************************************
 *
 * Ce fichier est une partie de SACoche.
 *
 * SACoche est un logiciel libre ; vous pouvez le redistribuer ou le modifier suivant les termes 
 * de la “GNU Affero General Public License” telle que publiée par la Free Software Foundation :
 * soit la version 3 de cette licence, soit (à votre gré) toute version ultérieure.
 *
 * SACoche est distribué dans l’espoir qu’il vous sera utile, mais SANS AUCUNE GARANTIE :
 * sans même la garantie implicite de COMMERCIALISABILITÉ ni d’ADÉQUATION À UN OBJECTIF PARTICULIER.
 * Consultez la Licence Publique Générale GNU Affero pour plus de détails.
 *
 * Vous devriez avoir reçu une copie de la Licence Publique Générale GNU Affero avec SACoche ;
 * si ce n’est pas le cas, consultez : <http://www.gnu.org/licenses/>.
 *
 */
 
// Extension de classe qui étend DB (pour permettre l’autoload)

// Ces méthodes ne concernent qu’une base STRUCTURE.
// Ces méthodes ne concernent que les élèves (et les parents).

class DB_STRUCTURE_ELEVE
{

/**
 * compter_demandes_evaluation
 *
 * @param int  $eleve_id
 * @return array
 */
public static function DB_compter_demandes_evaluation($eleve_id)
{
  $DB_SQL = 'SELECT demande_statut, COUNT(demande_id) AS nombre '
          . 'FROM sacoche_demande '
          . 'WHERE eleve_id=:eleve_id '
          . 'GROUP BY demande_statut ';
  $DB_VAR = array(
    ':eleve_id' => $eleve_id,
  );
  return DB::queryTab(SACOCHE_STRUCTURE_BD_NAME , $DB_SQL , $DB_VAR , TRUE , TRUE);
}

/**
 * Récupérer les informations relatives à un devoir
 *
 * Pas réussi à récupérer en même temps la liste des profs associés en écriture au devoir car
 * une jointure sur LEFT JOIN sacoche_jointure_devoir_prof USING (devoir_id)
 * pour avoir GROUP_CONCAT(prof_id SEPARATOR ",") AS partage_id_listing
 * avec GROUP BY devoir_id
 * et la condition AND ( ( prof_id IS NULL ) OR ( jointure_droit != :jointure_droit_non ) )
 * fonctionne si aucun prof n’est associé à l’éval
 * ou si au moins un prof y est associé en saisie
 * mais pas si tous les profs associés l’y sont uniquement en visualisation !!!
 * (dans ce cas la requête retourne un tableau vide...)
 *
 * @param int   $devoir_id
 * @return array
 */
public static function DB_recuperer_devoir_infos($devoir_id)
{
  $DB_SQL = 'SELECT proprio_id, devoir_date, devoir_info, devoir_visible_date, devoir_autoeval_date, '
          . 'devoir_autoeval_date >= NOW() AS devoir_autoeval_en_cours '
          . 'FROM sacoche_devoir '
          . 'WHERE devoir_id=:devoir_id ';
  $DB_VAR = array(
    ':devoir_id' => $devoir_id,
  );
  return DB::queryRow(SACOCHE_STRUCTURE_BD_NAME , $DB_SQL , $DB_VAR);
}

/**
 * En complément de DB_recuperer_devoir_infos()
 *
 * @param int   $devoir_id
 * @return array
 */
public static function DB_lister_devoir_profs_droit_saisie($devoir_id)
{
  $DB_SQL = 'SELECT prof_id '
          . 'FROM sacoche_jointure_devoir_prof '
          . 'WHERE devoir_id = :devoir_id AND jointure_droit != :jointure_droit_non ';
  $DB_VAR = array(
    ':devoir_id'          => $devoir_id,
    ':jointure_droit_non' => 'voir',
  );
  return DB::queryCol(SACOCHE_STRUCTURE_BD_NAME , $DB_SQL , $DB_VAR );
}

/**
 * Retourner les résultats pour un élève, pour des items donnés
 *
 * @param int    $eleve_id
 * @param string $liste_item_id   id des items séparés par des virgules
 * @param string $user_profil_type
 * @return array
 */
public static function DB_lister_result_eleve_items( $eleve_id , $liste_item_id , $user_profil_type )
{
  // Cette fonction peut être appelée avec un autre profil.
  $sql_view = ( ($user_profil_type=='eleve') || ($user_profil_type=='parent') ) ? 'AND ( saisie_visible_date IS NULL OR saisie_visible_date <= NOW() ) ' : '' ;
  $DB_SQL = 'SELECT item_id, saisie_note AS note '
          . 'FROM sacoche_saisie '
          . 'LEFT JOIN sacoche_devoir USING (devoir_id) ' // Pas d’INNER JOIN pour les évals des années antérieures
          . 'WHERE eleve_id=:eleve_id AND item_id IN('.$liste_item_id.') AND saisie_note!="PA" '.$sql_view
          . 'AND ( devoir_diagnostic = 0 OR devoir_diagnostic IS NULL ) ' // Pas d’évaluations diagnostiques ici
          . 'ORDER BY saisie_date ASC, devoir_id ASC '; // ordre sur devoir_id ajouté à cause des items évalués plusieurs fois le même jour
  $DB_VAR = array(
    ':eleve_id' => $eleve_id,
  );
  return DB::queryTab(SACOCHE_STRUCTURE_BD_NAME , $DB_SQL , $DB_VAR);
}

/**
 * Lister les évaluations concernant la classe ou les groupes d’un élève sur une période donnée
 *
 * @param int    $eleve_id
 * @param int    $prof_id
 * @param string $date_debut_sql
 * @param string $date_fin_sql
 * @param string $user_profil_type
 * @return array
 */
public static function DB_lister_devoirs_eleve( $eleve_id , $prof_id , $date_debut_sql , $date_fin_sql , $user_profil_type )
{
  // Récupérer classe et groupes de l’élève
  $DB_SQL = 'SELECT eleve_classe_id, GROUP_CONCAT(groupe_id SEPARATOR ",") AS eleve_groupes_id '
          . 'FROM sacoche_user '
          . 'LEFT JOIN sacoche_jointure_user_groupe USING (user_id) '
          . 'WHERE user_id=:user_id '
          . 'GROUP BY user_id ';
  $DB_VAR = array(
    ':user_id' => $eleve_id,
  );
  $DB_ROW = DB::queryRow(SACOCHE_STRUCTURE_BD_NAME , $DB_SQL , $DB_VAR);
  if( !$DB_ROW['eleve_classe_id'] && !$DB_ROW['eleve_groupes_id'] )
  {
    return NULL;
  }
  else
  {
    $virgule = ( $DB_ROW['eleve_classe_id'] && $DB_ROW['eleve_groupes_id'] ) ? ',' : '' ;
    $listing_groupes = $DB_ROW['eleve_classe_id'].$virgule.$DB_ROW['eleve_groupes_id'];
    // Cette fonction peut être appelée avec un autre profil.
    $is_eleve_ou_parent = ( ($user_profil_type=='eleve') || ($user_profil_type=='parent') ) ? TRUE : FALSE ;
    $sql_view   = ($is_eleve_ou_parent) ? 'AND ( devoir_visible_date IS NULL OR devoir_visible_date <= NOW() ) ' : '' ;
    $join_prof  = ($prof_id) ? 'LEFT JOIN sacoche_jointure_devoir_prof ON ( sacoche_devoir.devoir_id = sacoche_jointure_devoir_prof.devoir_id) ' : '' ;
    $where_prof = ($prof_id) ? 'AND ( sacoche_devoir.proprio_id=:proprio_id OR sacoche_jointure_devoir_prof.prof_id=:prof_id ) ' : '' ;
    $join_expe  = ($is_eleve_ou_parent)
                ? 'LEFT JOIN sacoche_referentiel_item USING (item_id) '
                . 'LEFT JOIN sacoche_referentiel_theme USING (theme_id) '
                . 'LEFT JOIN sacoche_referentiel_domaine USING (domaine_id) '
                . 'LEFT JOIN sacoche_matiere USING (matiere_id) '
                : '';
    $where_expe = ($is_eleve_ou_parent) ? 'AND matiere_experimentale = 0 ' : '' ;
    $DB_SQL = 'SELECT sacoche_devoir.* , '
            . 'devoir_autoeval_date >= NOW() AS devoir_autoeval_en_cours , '
            . 'sacoche_user.user_genre AS prof_genre , sacoche_user.user_nom AS prof_nom , sacoche_user.user_prenom AS prof_prenom , '
            . 'jointure_texte , jointure_audio , jointure_doc_sujet , jointure_doc_corrige , jointure_doc_copie , '
            . 'COUNT(DISTINCT sacoche_jointure_devoir_item.item_id) AS items_nombre '
            . 'FROM sacoche_devoir '
            . 'LEFT JOIN sacoche_jointure_devoir_eleve ON ( sacoche_devoir.devoir_id=sacoche_jointure_devoir_eleve.devoir_id AND sacoche_jointure_devoir_eleve.eleve_id=:eleve_id ) '
            . 'LEFT JOIN sacoche_user ON sacoche_devoir.proprio_id=sacoche_user.user_id '
            . 'LEFT JOIN sacoche_jointure_devoir_item ON ( sacoche_devoir.devoir_id = sacoche_jointure_devoir_item.devoir_id ) '
            . $join_prof
            . $join_expe
            . 'WHERE groupe_id IN('.$listing_groupes.') '.$where_prof.$where_expe.'AND devoir_date>="'.$date_debut_sql.'" AND devoir_date<="'.$date_fin_sql.'" '.$sql_view
            . 'GROUP BY sacoche_devoir.devoir_id '
            . 'ORDER BY devoir_date DESC, sacoche_devoir.devoir_id DESC '; // ordre sur devoir_id ajouté pour conserver une logique à l’affichage en cas de plusieurs devoirs effectués le même jour
    $DB_VAR = array(
      ':eleve_id'   => $eleve_id,
      ':proprio_id' => $prof_id,
      ':prof_id'    => $prof_id,
    );
    return DB::queryTab(SACOCHE_STRUCTURE_BD_NAME , $DB_SQL , $DB_VAR);
  }
}

/**
 * lister_nb_saisies_par_evaluation
 *
 * @param int      $eleve_id
 * @param string   $listing_devoir_id   soit une chaine de devoir (string), soit un devoir unique (int)
 * @param string   $user_profil_type
 * @return array   tableau (devoir_id;saisies_nombre)
 */
public static function DB_lister_nb_saisies_par_evaluation( $eleve_id , $listing_devoir_id , $user_profil_type )
{
  $is_eleve_ou_parent = ( ($user_profil_type=='eleve') || ($user_profil_type=='parent') ) ? TRUE : FALSE ;
  $join_expe  = ($is_eleve_ou_parent)
              ? 'LEFT JOIN sacoche_referentiel_item USING (item_id) '
              . 'LEFT JOIN sacoche_referentiel_theme USING (theme_id) '
              . 'LEFT JOIN sacoche_referentiel_domaine USING (domaine_id) '
              . 'LEFT JOIN sacoche_matiere USING (matiere_id) '
              : '';
  $where_expe = ($is_eleve_ou_parent) ? 'AND matiere_experimentale = 0 ' : '' ;
  $DB_SQL = 'SELECT COUNT(saisie_note) AS saisies_nombre, devoir_id '
          . 'FROM sacoche_saisie '
          . $join_expe
          . 'WHERE devoir_id IN('.$listing_devoir_id.') AND eleve_id=:eleve_id '
          . 'AND saisie_note!="PA" '
          . $where_expe
          . 'GROUP BY devoir_id ';
  $DB_VAR = array(
    ':eleve_id'          => $eleve_id,
    ':listing_devoir_id' => $listing_devoir_id,
  );
  return DB::queryTab(SACOCHE_STRUCTURE_BD_NAME , $DB_SQL , $DB_VAR);
}

/**
 * Lister les évaluations concernant un élève donné sur les derniers jours
 *
 * @param int    $eleve_id
 * @param int    $nb_jours
 * @return array
 */
public static function DB_lister_derniers_devoirs_eleve_avec_notes_saisies( $eleve_id , $nb_jours )
{
  $sql_view = 'AND ( devoir_visible_date IS NULL OR devoir_visible_date <= NOW() ) '; // Cette fonction n’est appelée qu’avec un profil élève ou parent
  $DB_SQL = 'SELECT devoir_id , devoir_date , devoir_info , sacoche_user.user_genre AS prof_genre , sacoche_user.user_nom AS prof_nom , sacoche_user.user_prenom AS prof_prenom '
          . 'FROM sacoche_saisie '
          . 'LEFT JOIN sacoche_devoir USING (devoir_id) '
          . 'LEFT JOIN sacoche_user ON sacoche_devoir.proprio_id=sacoche_user.user_id '
          . 'LEFT JOIN sacoche_referentiel_item USING (item_id) '
          . 'LEFT JOIN sacoche_referentiel_theme USING (theme_id) '
          . 'LEFT JOIN sacoche_referentiel_domaine USING (domaine_id) '
          . 'LEFT JOIN sacoche_matiere USING (matiere_id) '
          . 'WHERE eleve_id=:eleve_id AND saisie_note!="PA" '
          . 'AND devoir_date > DATE_SUB( NOW() , INTERVAL :nb_jours DAY ) '.$sql_view
          . 'AND ( devoir_diagnostic = 0 OR devoir_diagnostic IS NULL ) ' // Pas d’évaluations diagnostiques ici
          . 'AND matiere_experimentale = 0 ' // Pas de matière expérimentale ici
          . 'GROUP BY devoir_id '
          . 'ORDER BY devoir_date DESC, devoir_id DESC '; // ordre sur devoir_id ajouté pour conserver une logique à l’affichage en cas de plusieurs devoirs effectués le même jour
  $DB_VAR = array(
    ':eleve_id' => $eleve_id,
    ':nb_jours' => $nb_jours,
  );
  return DB::queryTab(SACOCHE_STRUCTURE_BD_NAME , $DB_SQL , $DB_VAR);
}

/**
 * Lister les évaluations concernant un élève donné comportant une auto-évaluation en cours
 *
 * @param int    $eleve_id
 * @param int    $classe_id   id de la classe de l’élève ; en effet sacoche_jointure_user_groupe ne contient que les liens aux groupes, donc il faut tester aussi la classe
 * @return array
 */
public static function DB_lister_devoirs_eleve_avec_autoevaluation_en_cours( $eleve_id , $classe_id )
{
  $sql_view = 'AND ( devoir_visible_date IS NULL OR devoir_visible_date <= NOW() ) '; // Cette fonction n’est appelée qu’avec un profil élève ou parent
  $where_classe = ($classe_id) ? 'sacoche_devoir.groupe_id='.$classe_id.' OR ' : '';
  $DB_SQL = 'SELECT devoir_id , devoir_date , devoir_info , sacoche_user.user_genre AS prof_genre , sacoche_user.user_nom AS prof_nom , sacoche_user.user_prenom AS prof_prenom '
          . 'FROM  sacoche_devoir '
          . 'LEFT JOIN sacoche_jointure_user_groupe USING (groupe_id) '
          . 'LEFT JOIN sacoche_user ON sacoche_devoir.proprio_id=sacoche_user.user_id '
          . 'WHERE ('.$where_classe.'sacoche_jointure_user_groupe.user_id=:eleve_id) '
          . 'AND devoir_autoeval_date IS NOT NULL AND devoir_autoeval_date >= NOW() '.$sql_view
          . 'GROUP BY devoir_id '
          . 'ORDER BY devoir_date DESC, devoir_id DESC '; // ordre sur devoir_id ajouté pour conserver une logique à l’affichage en cas de plusieurs devoirs effectués le même jour
  $DB_VAR = array(
    ':eleve_id' => $eleve_id,
  );
  return DB::queryTab(SACOCHE_STRUCTURE_BD_NAME , $DB_SQL , $DB_VAR);
}

/**
 * Lister les évaluations à venir concernant un élève donné
 *
 * @param int    $eleve_id
 * @param int    $classe_id   id de la classe de l’élève ; en effet sacoche_jointure_user_groupe ne contient que les liens aux groupes, donc il faut tester aussi la classe
 * @return array
 */
public static function DB_lister_prochains_devoirs_eleve( $eleve_id , $classe_id )
{
  $sql_view = 'AND ( devoir_visible_date IS NULL OR devoir_visible_date <= NOW() ) '; // Cette fonction n’est appelée qu’avec un profil élève ou parent
  $where_classe = ($classe_id) ? 'sacoche_devoir.groupe_id='.$classe_id.' OR ' : '';
  $DB_SQL = 'SELECT devoir_id , devoir_date , devoir_info , sacoche_user.user_genre AS prof_genre , sacoche_user.user_nom AS prof_nom , sacoche_user.user_prenom AS prof_prenom '
          . 'FROM  sacoche_devoir '
          . 'LEFT JOIN sacoche_jointure_user_groupe USING (groupe_id) '
          . 'LEFT JOIN sacoche_user ON sacoche_devoir.proprio_id=sacoche_user.user_id '
          . 'LEFT JOIN sacoche_jointure_devoir_item USING (devoir_id) '
          . 'LEFT JOIN sacoche_referentiel_item USING (item_id) '
          . 'LEFT JOIN sacoche_referentiel_theme USING (theme_id) '
          . 'LEFT JOIN sacoche_referentiel_domaine USING (domaine_id) '
          . 'LEFT JOIN sacoche_matiere USING (matiere_id) '
          . 'WHERE ('.$where_classe.'sacoche_jointure_user_groupe.user_id=:eleve_id) '
          . 'AND devoir_date > NOW() '.$sql_view
          . 'AND matiere_experimentale = 0 ' // Pas de matière expérimentale ici
          . 'GROUP BY devoir_id '
          . 'ORDER BY devoir_date ASC, devoir_id DESC '; // ordre sur devoir_id ajouté pour conserver une logique à l’affichage en cas de plusieurs devoirs effectués le même jour
  $DB_VAR = array(
    ':eleve_id' => $eleve_id,
  );
  return DB::queryTab(SACOCHE_STRUCTURE_BD_NAME , $DB_SQL , $DB_VAR);
}

/**
 * Lister les résultats concernant un élève sur les derniers jours
 *
 * @param int    $eleve_id
 * @param int    $nb_jours
 * @return array
 */
public static function DB_lister_derniers_resultats_eleve( $eleve_id , $nb_jours )
{
  $sql_view = 'AND ( saisie_visible_date IS NULL OR saisie_visible_date <= NOW() ) '; // Cette fonction n’est appelée qu’avec un profil élève ou parent
  $DB_SQL = 'SELECT item_id , item_nom , saisie_date , saisie_note , '
          . 'CONCAT(niveau_ref,".",domaine_code,theme_ordre,item_ordre) AS ref_auto , '
          . 'CONCAT(domaine_ref,theme_ref,item_ref) AS ref_perso , '
          . 'matiere_id , niveau_id , matiere_nom, matiere_ref '
          . 'FROM sacoche_saisie '
          . 'LEFT JOIN sacoche_devoir USING (devoir_id) ' // Pas d’INNER JOIN pour les évals des années antérieures
          . 'LEFT JOIN sacoche_referentiel_item USING (item_id) '
          . 'LEFT JOIN sacoche_referentiel_theme USING (theme_id) '
          . 'LEFT JOIN sacoche_referentiel_domaine USING (domaine_id) '
          . 'LEFT JOIN sacoche_matiere USING (matiere_id) '
          . 'LEFT JOIN sacoche_niveau USING (niveau_id) '
          . 'WHERE eleve_id=:eleve_id '
          . 'AND saisie_date > DATE_SUB( NOW() , INTERVAL :nb_jours DAY ) '.$sql_view
          . 'AND ( devoir_diagnostic = 0 OR devoir_diagnostic IS NULL ) ' // Pas d’évaluations diagnostiques ici
          . 'AND matiere_experimentale = 0 ' // Pas de matière expérimentale ici
          // Pas de 'GROUP BY item_id ' car le regroupement est effectué avant le tri par date
          . 'ORDER BY saisie_date DESC, devoir_id DESC '; // ordre sur devoir_id ajouté pour conserver une logique à l’affichage en cas de plusieurs devoirs effectués le même jour
  $DB_VAR = array(
    ':eleve_id' => $eleve_id,
    ':nb_jours' => $nb_jours,
  );
  return DB::queryTab(SACOCHE_STRUCTURE_BD_NAME , $DB_SQL , $DB_VAR , TRUE); // TRUE permet d’avoir item_id en clef et, pour un item qui ressortirait plusieurs fois, d’avoir la dernière saisie en [item_id][0]
}

/**
 * Retourner les items d’un devoir et des informations supplémentaires ; les clefs du tableau sont les item_id car on en a besoin
 *
 * @param int    $devoir_id
 * @param string $user_profil_type
 * @return array
 */
public static function DB_lister_items_devoir_avec_infos_pour_eleves( $devoir_id , $user_profil_type )
{
  $where_expe = ( ($user_profil_type=='eleve') || ($user_profil_type=='parent') ) ? 'AND matiere_experimentale = 0 ' : '' ;
  $DB_SQL = 'SELECT item_id, item_nom, COUNT(sacoche_jointure_referentiel_socle.item_id) AS s2016_nb, '
          . 'item_cart, item_comm, item_lien, '
          . 'matiere_id, matiere_nb_demandes, matiere_ref , '
          . 'CONCAT(niveau_ref,".",domaine_code,theme_ordre,item_ordre) AS ref_auto , '
          . 'CONCAT(domaine_ref,theme_ref,item_ref) AS ref_perso , '
          . 'referentiel_calcul_methode, referentiel_calcul_limite, referentiel_calcul_retroactif '
          . 'FROM sacoche_jointure_devoir_item '
          . 'LEFT JOIN sacoche_referentiel_item USING (item_id) '
          . 'LEFT JOIN sacoche_referentiel_theme USING (theme_id) '
          . 'LEFT JOIN sacoche_referentiel_domaine USING (domaine_id) '
          . 'LEFT JOIN sacoche_niveau USING (niveau_id) '
          . 'LEFT JOIN sacoche_matiere USING (matiere_id) '
          . 'LEFT JOIN sacoche_referentiel USING (matiere_id,niveau_id) '
          . 'LEFT JOIN sacoche_jointure_referentiel_socle USING (item_id) '
          . 'WHERE devoir_id=:devoir_id '.$where_expe
          . 'GROUP BY sacoche_jointure_devoir_item.item_id '
          . 'ORDER BY jointure_ordre ASC, matiere_ref ASC, niveau_ordre ASC, domaine_ordre ASC, theme_ordre ASC, item_ordre ASC';
  $DB_VAR = array(
    ':devoir_id' => $devoir_id,
  );
  return DB::queryTab(SACOCHE_STRUCTURE_BD_NAME , $DB_SQL , $DB_VAR , TRUE);
}

/**
 * Retourner les notes obtenues à un élève à un devoir
 *
 * @param int   $devoir_id
 * @param int   $eleve_id
 * @param string $user_profil_type
 * @param bool  $with_marqueurs   // Avec ou sans les marqueurs de demandes d’évaluations
 * @return array
 */
public static function DB_lister_saisies_devoir_eleve( $devoir_id , $eleve_id , $user_profil_type , $with_marqueurs )
{
  // Cette fonction peut être appelée avec un autre profil.
  $sql_view = ( ($user_profil_type=='eleve') || ($user_profil_type=='parent') ) ? 'AND ( saisie_visible_date IS NULL OR saisie_visible_date <= NOW() ) ' : '' ;
  $req_view = ($with_marqueurs) ? '' : 'AND saisie_note!="PA" ' ;
  $DB_SQL = 'SELECT item_id, saisie_note '
          . 'FROM sacoche_saisie '
          . 'WHERE devoir_id=:devoir_id AND eleve_id=:eleve_id '.$sql_view.$req_view;
  $DB_VAR = array(
    ':devoir_id' => $devoir_id,
    ':eleve_id'  => $eleve_id,
  );
  return DB::queryTab(SACOCHE_STRUCTURE_BD_NAME , $DB_SQL , $DB_VAR);
}

/**
 * lister_crcn_saisies
 *
 * @param int   $eleve_id
 * @param int   $nb_jours
 * @return array
 */
public static function DB_lister_dernieres_competences_numeriques_saisies( $eleve_id , $nb_jours )
{
  $DB_SQL = 'SELECT eleve_id, prof_id, crcn_saisie_date, COUNT(crcn_competence_id) AS competences_nombre, '
          . 'user_nom AS prof_nom, user_prenom AS prof_prenom, user_genre AS prof_genre '
          . 'FROM sacoche_crcn_saisie '
          . 'LEFT JOIN sacoche_user ON sacoche_crcn_saisie.prof_id = sacoche_user.user_id '
          . 'WHERE eleve_id=:eleve_id AND crcn_niveau_numero IS NOT NULL '
          . 'AND crcn_saisie_date > DATE_SUB( NOW() , INTERVAL :nb_jours DAY ) '
          . 'GROUP BY crcn_saisie_date, prof_id '
          . 'ORDER BY crcn_saisie_date DESC ';
  $DB_VAR = array(
    ':eleve_id' => $eleve_id,
    ':nb_jours' => $nb_jours,
  );
  return DB::queryTab(SACOCHE_STRUCTURE_BD_NAME , $DB_SQL , $DB_VAR);
}

}
?>