# -*- mode: python; coding: utf-8 -*-

import os

class NoEoleDebsumsReportError(Exception):
    """No EOLE debsums report found"""
    pass

class EoleDebsumsReportParseError(Exception):
    """Problem parsing EOLE debsums report"""
    pass

class EoleDebsumsReport(object):
    """An EOLE report of debsums invocation"""
    def __init__(self, contents=None, filename=None):
        """
        @param contents: the content of the report
        @type contents: C{str}
        @param filename: name of the report file
        @type filename: C{str}
        """
        self._report = {}
        self._filename = filename

        # Check that either contents or filename is passed (but not both)
        if (not filename and not contents) or (filename and contents):
            raise Exception("Either filename or contents must be passed")

        if filename and not os.access(self._filename, os.F_OK):
            raise NoEoleDebsumsReportError, "Report file {0} not found".format(self._filename)

        if contents:
            self._contents = contents[:]
        else:
            self._read()

        self._parse()

    def _read(self):
        """Load the content of a report file."""
        with file(self._filename) as f:
            self._contents = f.read()

    def _parse(self):
        """Parse an EOLE debsums report file."""
        for line in self._contents.splitlines():
            info = line.rstrip('\n').split(': ')
            if len(info) != 2:
                raise EoleDebsumsReportParseError, "Error parsing line '{0}'".format(line)
            self.add_file(*info)

    def __getitem__(self, item):
        return self._report[item]

    def __setitem__(self, item, value):
        self._report[item] = value

    def __iter__(self):
        return self._report.__iter__()

    def items(self):
        return self._report.items()

    def iteritems(self):
        return self._report.iteritems()

    def add_file(self, package=None, filename=None):
        """
        Add a modified file name for a package.

        @param package: Name of the package containing the modified file
        @type package: C{str}
        @param filename: Name of the modified file
        @type filename: C{str}
        """
        if package not in self._report:
            self._report[package] = set([filename])
        else:
            self._report[package].add(filename)
