# -*- coding: utf-8 -*-
#
##########################################################################
# eoleflask.loader -- loading flask applications
# Copyright © 2013 Pôle de compétences EOLE <eole@ac-dijon.fr>
#
# License CeCILL:
#  * in french: http://www.cecill.info/licences/Licence_CeCILL_V2-fr.html
#  * in english http://www.cecill.info/licences/Licence_CeCILL_V2-en.html
##########################################################################

"""Load flask applications

This module define the function :func:`get_apps_list` to build a dictionary
of ``mount_point``:``module`` for use by :mode:`werkzeug.wsgi`
dispatcher.

"""

import logging

from os import walk
from os.path import join, isdir
import json
from eoleflask.flask_key import get_secret_key
from eoleflask.util import APPS_CONF_DIR, APPS_ENABLE_DIR
from pyeole.log import init_logging

_LOG_DIR = '/var/log/eoleflask'
"""Directory to store log files.

"""

_STATIC_BASE = '/usr/share/eole/flask'
"""Base directory where static files are stored per application
"""

log = init_logging(name='eoleflask', level='info',
                   filename=join(_LOG_DIR, 'eoleflask.log'))

class NoApplicationError(StandardError):
    """Unable to load some applications

    """
    pass


def load_app(configfile=None, conf_dir=APPS_ENABLE_DIR):
    """Load a flask application and its configuration
     in global namespace

    :param configfile: configuration file
    :type configfile: `str`
    :param conf_dir: directory for configuration files
    :type conf_dir: `str`
    :return: loaded flask app object
    :rtype: `flask.app`

    """
    if configfile is not None:
        with open(join(conf_dir,configfile), 'r') as f:
            # Load the config for the current app
            try:
                config = json.loads(f.read().decode('utf-8'))
            except StandardError, err:
                log.error("Unable to load json config file %s" % f.name,
                          exc_info=True)
                return False

	if not config.has_key('APPNAME'):
            log.error('Missing "APPNAME" key in config')
            return False

	module = config['APPNAME']
	# Try to import the current app
        try:
            m = __import__(module)
        except StandardError, err:
            log.error("Unable to load module '%s': %s" % (module, err),
                      exc_info=True)
            return False

        # Where to look for resources
        if 'INSTANCE_PATH' not in config:
            config['INSTANCE_PATH'] = join(_STATIC_BASE,
                                           config['MOUNT_POINT'].lstrip('/'),
                                           'resources')

        # Prefix of URL for static files
        if 'STATIC_URL_PATH' not in config:
            config['STATIC_URL_PATH'] = '/static'

        # Where to lookup static files
        if 'STATIC_FOLDER' not in config:
            config['STATIC_FOLDER'] = join(_STATIC_BASE,
                                           config['MOUNT_POINT'].lstrip('/'),
                                           'static')
        elif config['STATIC_FOLDER'] != '/static':
            m.app.add_url_rule(config['STATIC_FOLDER'] + '/<path:filename>',
                               endpoint='static',
                               view_func=m.app.send_static_file)

        # Where to lookup templates
        if 'TEMPLATE_FOLDER' not in config:
            config['TEMPLATE_FOLDER'] = join(_STATIC_BASE,
                                             config['MOUNT_POINT'].lstrip('/'),
                                             'templates')

        #Needed to avoid propagation of exception
        config['DEBUG'] = False
        config['PROPAGATE_EXCEPTIONS'] = False
        config['SECRET_KEY'] = get_secret_key(config['APPNAME'])

        m.app.root_path = config['INSTANCE_PATH']

        m.app.config.root_path = conf_dir

        m.app.instance_path = config['INSTANCE_PATH']

        m.app.static_url_path = config['STATIC_URL_PATH']

        m.app.static_folder = config['STATIC_FOLDER']

        m.app.template_folder = config['TEMPLATE_FOLDER']

        m.app.config.update(config)

        # Add logger handler
        logfile = join(_LOG_DIR, '{0}.log'.format(config['APPNAME']))

        msg_formatter = u'%(asctime)s: %(name)s - %(message)s'
        date_formatter = u''
        formatter = logging.Formatter(msg_formatter, date_formatter)

        file_handler = logging.FileHandler(filename=logfile)
        file_handler.setFormatter(formatter)

        m.app.logger.addHandler(file_handler)
        if 'LOG_LEVEL' in config.keys():
            # Default log level is WARNING
            m.app.logger.setLevel(getattr(logging, config['LOG_LEVEL'].upper()))

        m.app.logger.debug(m.app.config['TEMPLATE_FOLDER'])
        # return the app
        return m.app

def get_apps_list(apps_dir=None):
    """Load all flask applications and configurations

    :param apps_dir: subdirectory for additional app config files
    :type apps_dir:'str'
    :return: dict of mount points and associated apps
    :rtype: dict
    :raise NoApplicationError: if no application is loaded.

    """
    apps = {}
    conf_dirs = [APPS_ENABLE_DIR]
    if apps_dir:
        # try to load additional applications from specified dir
        apps_dir = join(APPS_CONF_DIR, apps_dir, 'enabled')
        if not isdir(apps_dir):
            raise NoApplicationError(u'invalid configuration directory: {0}'.format(apps_dir))
        conf_dirs.append(apps_dir)
    for apps_conf_dir in conf_dirs:
        for root, dirs, filenames in walk(apps_conf_dir):
            for name in filenames:
                if not name.endswith('.conf'):
                    continue
                app = load_app(name, apps_conf_dir)
                if app:
                    # populate the array of apps
                    apps[app.config['MOUNT_POINT']] = app

    if len(apps) == 0:
        raise NoApplicationError('No application loaded')

    return apps

